/************************************************************************/
/*                                                                      */
/*    vspline - a set of generic tools for creation and evaluation      */
/*              of uniform b-splines                                    */
/*                                                                      */
/*            Copyright 2015 - 2017 by Kay F. Jahnke                    */
/*                                                                      */
/*    The git repository for this software is at                        */
/*                                                                      */
/*    https://bitbucket.org/kfj/vspline                                 */
/*                                                                      */
/*    Please direct questions, bug reports, and contributions to        */
/*                                                                      */
/*    kfjahnke+vspline@gmail.com                                        */
/*                                                                      */
/*    Permission is hereby granted, free of charge, to any person       */
/*    obtaining a copy of this software and associated documentation    */
/*    files (the "Software"), to deal in the Software without           */
/*    restriction, including without limitation the rights to use,      */
/*    copy, modify, merge, publish, distribute, sublicense, and/or      */
/*    sell copies of the Software, and to permit persons to whom the    */
/*    Software is furnished to do so, subject to the following          */
/*    conditions:                                                       */
/*                                                                      */
/*    The above copyright notice and this permission notice shall be    */
/*    included in all copies or substantial portions of the             */
/*    Software.                                                         */
/*                                                                      */
/*    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND    */
/*    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES   */
/*    OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND          */
/*    NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT       */
/*    HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,      */
/*    WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING      */
/*    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR     */
/*    OTHER DEALINGS IN THE SOFTWARE.                                   */
/*                                                                      */
/************************************************************************/

/*! \file unary_functor.h

    \brief interface definition for unary functors

    vspline's evaluation and remapping code relies on a unary functor template
    which is used as the base for vspline::evaluator and also constitutes the
    type of object accepted by most of the functions in transform.h.

    This template produces functors which are meant to yield a single output
    for a single input, where both the input and output types may be single
    types or vigra::TinyVectors, and their elementary types may be vectorized.
    The functors are expected to provide methods named eval() which are capable
    of performing the required functionality. These eval routines take both
    their input and output by reference - the input is taken by const &, and the
    output as plain &. The result type of the eval routines is void. While
    such unary functors can be hand-coded, the class template 'unary_functor'
    provides services to create such functors in a uniform way, with a specifc
    system of associated types and some convenience code. Using unary_functor
    is meant to facilitate the creation of the unary functors used in vspline.
    
    Using unary_functor generates objects which can be easily combined into
    more complex unary functors, a typical use would be to 'chain' two
    unary_functors, see class template 'chain_type' below, which also provides
    an example for the use of unary_functor.
    
    class unary_functor takes three template arguments:
    
    - the argument type, IN
    
    - the result type, OUT
    
    - the number of fundamentals in a vector, _vsize
    
    When using Vc, the vectorized argument and result type are deduced from
    IN, OUT and _vsize by querying vspline::vector_traits.
    
    So where is eval() or operator()? Not in class unary_functor. The actual
    functionality is provided by the derived class. There is deliberately no
    code concerning evaluation in class unary_functor. My initial implementation
    had pure virtual functions to define the interface for evaluation, but this
    required explicitly providing the overloads in the derived class. Simply
    omitting any reference to evaluation allows the derived class to accomplish
    evaluation with a template if the code is syntactically the same for vectorized
    and unvectorized operation. To users of concrete functors inheriting from
    unary_functor this makes no difference. The only drawback is that it's not
    possible to perform evaluation via a base class pointer or reference. But
    this is best avoided anyway because it degrades performance. If the need arises
    to have several unary_functors with the same template signature share a common
    type, there's a mechanism to make the internals opaque by 'grokking'.
    grokking provides a wrapper around a unary_functor which hides it's type,
    vspline::grok_type directly inherits from unary_functor and the only template
    arguments are IN, OUT and _vsize. This hurts performance a little - just as
    calling via a base class pointer/reference would, but the code is outside
    class unary_functor and therefore only activated when needed.
    
    Class vspline::evaluator is itself coded as a vspline::unary_functor and can
    serve as another example for the use of the code in this file.
*/

#ifndef VSPLINE_UNARY_FUNCTOR_H
#define VSPLINE_UNARY_FUNCTOR_H

#include <functional>
#include <vspline/common.h>

namespace vspline {

/// we derive all vspline::unary_functors from this empty class, to have
/// a common base type for all of them. This enables us to easily check if
/// a type is a vspline::unary_functor without having to wrangle with
/// unary_functor's template arguments.
 
template < int _vsize >
struct unary_functor_tag { } ;

/// class unary_functor provides a functor object which offers a system
/// of types for concrete unary functors derived from it. If Vc isn't used,
/// this is trivial, but with Vc in use, we get vectorized types derived from
/// plain IN and OUT via query of vspline::vector_traits.
///
/// class unary_functor does not provide anything callable, this is left to
/// the concrete functors inheriting from unary_functor. It is expected, though,
/// that the derived classes provide evaluation capability, either as a
/// method template or as (overloaded) method(s) 'eval'. eval is to be coded
/// as taking it's argument as a const&, and writing it's result to it's
/// second argument, to which it receives a reference. eval's return type
/// is void.
///
/// Why not lay down an interface with a pure virtual function eval()
/// which derived classes would need to override? Suppose you had, in
/// unary_functor,
///
/// virtual void eval ( const in_type & in , out_type & out ) = 0 ;
///
/// Then, in a derived class, you'd have to provide an override with this
/// signature. Initially, this seems reasonable enough, but if you want to
/// implement eval() as a member function template in the derived class, you
/// still would have to provide the override (calling an instatiated version
/// of your template), because your template won't be recognized as a viable
/// way to override the pure virtual base class member function. Since
/// providing eval as a template is common (oftentimes vectorized and
/// unvectorized code are the same) I've decided against having virtual eval
/// routines, to avoid the need for explicitly overriding them in derived
/// classes which provide eval() as a template.
///
/// How about providing operator() in unary_functor? operator() would need to
/// delegate to eval(), but eval is not known in unary_functor; only classes
/// derived from unary_functor provide eval(). We could only call the derived
/// class' eval if it were virtual and declared in unary_functor itself. Since
/// this has been ruled out, we can't have operator() either. Initially this
/// looks like an annoying ommision: we have a functor and it doesn't provide
/// operator()? But it turns out that
///
/// - most of the time we don't use objects derived from unary_functor directly
///
/// - and if we do and need function call syntax, we can use vspline::callable
///   or vspline::grok which produce objects with operator().
///
/// finally, if you define your own functors derived from unary_functor, nothing
/// is stopping you from supplying operator() for your derived classes, if you
/// don't want wrap them in callable types vspline provides.
///
/// With no virtual member functions, class unary_functor becomes very simple,
/// which is desirable from a design standpoint, and also makes unary_functors
/// smaller, avoiding the creation of the virtual function table.

template < typename IN ,    // argument or input type
           typename OUT ,   // result type
           int _vsize = vspline::vector_traits < IN > :: size
         >
struct unary_functor
: public unary_functor_tag < _vsize >
{
  // number of elements in simdized data. For code not using Vc, and also for
  // cases where vectorization isn't possible, vsize will be 1 and the vectorized
  // types will 'collapse' to the unvectorized types.

  enum { vsize = _vsize } ;

  // number of dimensions. This may well be different for IN and OUT.
  // note how we rely on vigra's ExpandElementResult mechanism to inspect IN and OUT.

  enum { dim_in = vigra::ExpandElementResult < IN > :: size } ;
  enum { dim_out = vigra::ExpandElementResult < OUT > :: size } ;

  // typedefs for incoming (argument) and outgoing (result) type. These two types
  // are non-vectorized types, like vigra::TinyVector < float , 2 >. Since such types
  // consist of elements of the same type, the corresponding vectorized type can be
  // easily automatically determined.
  
  typedef IN in_type ;
  typedef OUT out_type ;
  
  // elementary types of same. we rely on vigra's ExpandElementResult mechanism
  // to provide these types.
  
  typedef typename vigra::ExpandElementResult < IN > :: type in_ele_type ;
  typedef typename vigra::ExpandElementResult < OUT > :: type out_ele_type ;
  
  // for vectorized operation, we need a few extra typedefs. I use a _v
  // suffix instead of the _type suffix above to indicate vectorized types.
  // If Vc is not in use, the _v types simply collapse to the unvectorized
  // types, having them does no harm, but it's not safe to assume that,
  // for example, in_v and in_type are in fact different types.

  /// a simdized type of the elementary type of result_type,
  /// which is used for coefficients and results. this is fixed via
  /// the traits class vector_traits (in common.h). Note how we derive
  /// this type using vsize from the template argument, not what
  /// vspline::vector_traits deems appropriate for ele_type - though
  /// both numbers will be the same in most cases.
  
  typedef typename vector_traits < IN , vsize > :: ele_v in_ele_v ;
  typedef typename vector_traits < OUT , vsize > :: ele_v out_ele_v ;
  
  /// vectorized in_type and out_type. vspline::vector_traits supplies these
  /// types so that multidimensional/multichannel data come as vigra::TinyVectors,
  /// while 'singular' data won't be made into TinyVectors of one element.
  
  typedef typename vector_traits < IN , vsize > :: type in_v ;
  typedef typename vector_traits < OUT , vsize > :: type out_v ;

  /// vsize wide vector of ints, used for gather/scatter indexes
  
  typedef typename vector_traits < int , vsize > :: ele_v ic_v ;

  // out_type_of checks if it's template argument is the same type as
  // in_type. If this is so, the corresponding result type is out_type.
  // If not, the input must be vectorized and the corresponding vectorized
  // result type is queried from vspline::vector_traits
  
  template < class argument_type >
  using out_type_of =
    typename std::conditional
    < std::is_same < argument_type , in_type > :: value ,
      out_type ,
      typename vspline::vector_traits < out_type , vsize > :: type
    > :: type ;
} ;

/// class callable_type provides a wrapper for vspline::unary_functors
/// which provides operator(). class unary_functor, as it is currently
/// implemented, cannot provide operator(), since the eval routines
/// are only defined in the concrete functors which inherit from it.
/// To be able to provide operator() in class unary_functor, I tried
/// an implementation which pulled in the eval functionality as a
/// policy. This worked, but the code was verbose and convoluted,
/// so I abandoned it. This is my new way of providing a callable.
/// If the callable is needed, it provides, and the code for
/// class unary_functor is much simpler.
/// to obtain the return type O for a given argument type I,
/// we use class unary_functor's out_typ_of mechanism.
/// This class is typical for a class derived from unary_functor:
/// It inherits from unary_functor. It defines it's in_type, out_type,
/// vsize and base_type, then proceeds to define it's evaluation
/// code and other capabilities it may provide.

template < typename inner_type >
struct callable_type
: public vspline::unary_functor < typename inner_type::in_type ,
                                  typename inner_type::out_type ,
                                  inner_type::vsize >
{
  // definition of in_type, out_type, vsize and base_type
  
  typedef typename inner_type::in_type in_type ;
  typedef typename inner_type::out_type out_type ;
  enum { vsize = inner_type::vsize } ;
  typedef vspline::unary_functor
          < in_type , out_type , vsize > base_type ;
  
  // the wrapped, 'inner' functor
          
  const inner_type inner ;

  // the constructor initializes the inner functor
  
  callable_type ( const inner_type _inner )
  : inner ( _inner )
  { } ;

  // we set up a member function template 'eval' which delegates to
  // the inner type's eval.
  
  template < class I , class O >
  void eval ( const I & i , O & o ) const
  {
    inner.eval ( i , o ) ;
  }

  // here comes the implementation of operator(). We provide a
  // member function template to handle all parameter signatures
  // uniformly. Note how the return type is derived from the
  // argument type by using unary_functor's out_type_of mechanism.
  
  template < class I ,
             class O = typename base_type::template out_type_of<I> >
  O operator() ( const I & in ) const
  {
    O result ;
    inner.eval ( in , result ) ;
    return result ;
  }
} ;

/// callable() is a factory function wrapping a vspline::unary_functor
/// in a vspline::callable_type which provides operator() overloads
/// processing vectorized and unvectorized arguments.

template < class inner_type >
vspline::callable_type < inner_type >
callable ( const inner_type & inner )
{
  return vspline::callable_type < inner_type > ( inner ) ;
}

/// class chain_type is a helper class to pass one unary functor's result
/// as argument to another one. We rely on T1 and T2 to provide a few of the
/// standard types used in unary functors. Typically, T1 and T2 will both be
/// vspline::unary_functors, but the type requirements could also be fulfilled
/// manually.

template < typename T1 ,
           typename T2 >
struct chain_type
: public vspline::unary_functor < typename T1::in_type ,
                                  typename T2::out_type ,
                                  T1::vsize >
{
  // definition of in_type, out_type, vsize and base_type
  
  typedef typename T1::in_type in_type ;
  typedef typename T2::out_type out_type ;
  enum { vsize = T1::vsize } ;
  typedef vspline::unary_functor
          < in_type , out_type , vsize > base_type ;
  
  // we require both functors to share the same vectorization width
                  
  static_assert ( T1::vsize == T2::vsize ,
                  "can only chain unary_functors with the same vector width" ) ;

  // hold the two functors by value

  const T1 t1 ;
  const T2 t2 ;
  
  // the constructor initializes them

  chain_type ( const T1 & _t1 , const T2 & _t2 )
  : t1 ( _t1 ) ,
    t2 ( _t2 )
    { } ;

  // the actual eval needs a bit of trickery to determine the type of
  // the intermediate type from the type of the first argument.

  template < typename argument_type ,
             typename result_type >
  void eval ( const argument_type & argument ,
                      result_type & result ) const
  {
    typedef typename T1::template out_type_of < argument_type >
                     intermediate_type ;
    
    intermediate_type intermediate ;

    t1.eval ( argument , intermediate ) ; // evaluate first functor into it

    t2.eval ( intermediate , result ) ;   // feed it as input to second functor
  }

} ;

/// chain is a factory function yielding the result of chaining
/// two unary_functors.

template < class T1 , class T2 >
vspline::chain_type < T1 , T2 >
chain ( const T1 & t1 , const T2 & t2 )
{
  return vspline::chain_type < T1 , T2 > ( t1 , t2 ) ;
}

/// using operator overloading, we can exploit operator+'s semantics
/// to chain several unary functors. We need to specifically enable
/// this for types derived from unary_functor_tag to avoid a catch-all
/// situation.

template < typename T1 ,
           typename T2 ,
           typename std::enable_if
           < std::is_base_of
             < vspline::unary_functor_tag < T2::vsize > ,
               T1
             > :: value ,
             int
           > :: type = 0 ,
           typename std::enable_if
           < std::is_base_of
             < vspline::unary_functor_tag < T1::vsize > ,
               T2
             > :: value ,
             int
           > :: type = 0
         >
vspline::chain_type < T1 , T2 >
operator+ ( const T1 & t1 , const T2 & t2 )
{
  return vspline::chain ( t1 , t2 ) ;
}

/// sometimes, vectorized code for a vspline::unary_functor is not at hand
/// for some specific evaluation. class broadcast_type can broadcast unvectorized
/// evaluation code, so that vectorized data can be procesed with this code,
/// albeit less efficiently.

template < class inner_type , int _vsize >
struct broadcast_type
: public vspline::unary_functor < typename inner_type::in_type ,
                                  typename inner_type::out_type ,
                                  _vsize >
{
  // definition of in_type, out_type, vsize and base_type
  
  typedef typename inner_type::in_type in_type ;
  typedef typename inner_type::out_type out_type ;
  enum { vsize = _vsize } ;
  typedef vspline::unary_functor
          < in_type , out_type , vsize > base_type ;
  
  const inner_type inner ;
  
  broadcast_type ( const inner_type & _inner )
  : inner ( _inner )
  { } ;
  
  /// single-value evaluation simply delegates to inner

  void eval ( const in_type & in , out_type & out ) const
  {
    inner.eval ( in , out ) ;
  }
  
  // vector_traits for in_type and out_type
  
  typedef typename vspline::vector_traits < in_type , _vsize > iv_traits ;
  typedef typename vspline::vector_traits < out_type , _vsize > ov_traits ;
  
  // now we implement the actual broadcast

  /// vectorized evaluation will match this template:
  
  template < class in_v , class out_v>
  void eval ( const in_v & in ,
                    out_v & out ) const
  {
    // we want TinyVectors even if there is only one channel.
    // this way we can iterate over the channels.

    typedef typename iv_traits::nd_ele_v iv_type ;
    typedef typename ov_traits::nd_ele_v ov_type ;
    
    // we instantiate an iv_type by 'wrapping' the incoming in_v, which
    // produces a TinyVector with one element for types which aren't
    // already TinyVectors.
    
    iv_type iv ( wrap ( in ) ) ;
    
    // we have an ov_type handy for the output
    
    ov_type ov ;
    
    // now the broadcast:
    
    // we want a definite TinyVector for unvectorized input and output types
    // so that we can iterate over them, even if they have only one element
    
    typename iv_traits::nd_ele_type i ;
    typename ov_traits::nd_ele_type o ;
    
    // at the same time, we want to be able to feed these data to eval,
    // which does *not* like TinyVectors with one element.

    in_type  & iref ( reinterpret_cast < in_type& >  ( i ) ) ;
    out_type & oref ( reinterpret_cast < out_type& > ( o ) ) ;
    
    for ( int e = 0 ; e < _vsize ; e++ )
    {
      // extract the eth input value from the simdized input
      
      for ( int d = 0 ; d < iv_traits::dimension ; d++ )
        i[d] = iv[d][e] ;
      
      // process it with eval, passing the eval-compatible references
      
      inner.eval ( iref , oref ) ;
      
      // now distribute eval's result to the SIMD output
      
      for ( int d = 0 ; d < ov_traits::dimension ; d++ )
        ov[d][e] = o[d] ;
    }
    
    // finally, assign ov to out. We use unwrap() here, in case ov is a
    // TinyVector with one element.
    
    out = unwrap ( ov ) ;
  }

} ;

/// type of a std::function for unvectorized evaluation:

template < class IN , class OUT >
using eval_type = std::function < void ( const IN & , OUT & ) > ;

/// helper class hold_type holds a single-element evaluation function

template < class IN , class OUT >
struct hold_type
: public vspline::unary_functor < IN , OUT , 1 >
{
  const eval_type < IN , OUT > eval ;
  
  hold_type ( eval_type < IN , OUT > _eval )
  : eval ( _eval )
  { } ;
} ;

/// factory function to create a broadcast_type from another vspline::unary_functor
/// This will pick the other functor's unvectorized eval routine and broadcast it,
/// The vectorized eval routine of the other functor (if present) is ignored.

template < class inner_type , int _vsize >
broadcast_type < inner_type , _vsize > broadcast ( const inner_type & inner )
{
  return broadcast_type < inner_type , _vsize > ( inner ) ;
}

/// factory function to create a broadcast_type from a std::function
/// implementing the unvectorized evaluation.
/// to broadcast a single-value evaluation function, we package it
/// in a hold_type, which broadcast can handle.

template < class IN , class OUT , int _vsize >
broadcast_type < hold_type < IN , OUT > , _vsize >
broadcast ( eval_type < IN , OUT > _eval )
{
  return broadcast_type < hold_type < IN , OUT > , _vsize >
    ( hold_type < IN , OUT > ( _eval ) ) ;
}

/// eval_wrap is a helper function template, wrapping an 'ordinary'
/// function which returns some value given some input in a void function
/// taking input as const reference and writing output to a reference,
/// which is the signature used for evaluation in vspline::unary_functors.

template < class IN , class OUT >
std::function < void ( const IN& , OUT& ) >
eval_wrap ( std::function < OUT ( const IN& ) > f )
{
  return [f] ( const IN& in , OUT& out ) { out = f ( in ) ; } ;
}

/// class grok_type is a helper class wrapping a vspline::unary_functor
/// so that it's type becomes opaque - a technique called 'type erasure',
/// here applied to vspline::unary_functors with their specific
/// capability of providing both vectorized and unvectorized operation
/// in one common object.
///
/// While 'grokking' a unary_functor may degrade performance slightly,
/// the resulting type is less complex, and when working on complex
/// constructs involving several unary_functors, it can be helpful to
/// wrap the whole bunch into a grok_type for some time to make compiler
/// messages more palatable. I even suspect that the resulting functor,
/// which simply delegates to two std::functions, may optimize better at
/// times than a more complex functor in the grokkee.
///
/// Performance aside, 'grokking' a vspline::unary_functor produces a
/// simple, consistent type that can hold *any* unary_functor with the
/// given input and output type(s), so it allows to hold and use a
/// variety of (intrinsically differently typed) functors at runtime
/// via a common handle which is a vspline::unary_functor itself and
/// can be passed to the transform-type routines. With unary_functors
/// being first-class, copyable objects, this also makes it possible
/// to pass around unary_functors between different TUs where user
/// code can provide new functors at will which can simply be used
/// without having to recompile to make their type known, at the cost
/// of a call through a std::function.
///
/// grok_type also provides a convenient way to introduce functors into
/// vspline. Since the functionality is implemented with std::functions,
/// we allow direct initialization of these std::functions on top of
/// 'grokking' the capabilities of another unary_functor via lambda
/// expressions. 'Ordinary' functions can also be grokked.
///
/// For grok_type objects where _vsize is greater 1, there are
/// constructor overloads taking only a single function. These
/// constructors broadcast the unvectorized function to process
/// vectorized data, providing a quick way to produce code which
/// runs with vector data, albeit less efficiently than true vector
/// code.
///
/// finally, for convenience, grok_type also provides operator(), just
/// like vspline::callable, to use the grok_type object with function
/// call syntax, and it also provides the common 'eval' routine(s), just
/// like any other unary_functor.

template < typename IN ,    // argument or input type
           typename OUT ,   // result type
           int _vsize = vspline::vector_traits < IN > :: size
         >
struct grok_type
: public vspline::unary_functor < IN , OUT , _vsize >
{
  // if Vc is in use, we provide code for this template class.
  // otherwise it's just empty - only the specialization for
  // _vsize == 1 will be used, see below

#ifdef USE_VC
  
  typedef vspline::unary_functor < IN , OUT , _vsize > base_type ;
  using base_type::vsize ;
  using typename base_type::in_type ;
  using typename base_type::out_type ;
  
  typedef std::function < void ( const in_type & , out_type & ) > eval_type ;
  typedef std::function < out_type ( const in_type & ) > call_type ;

  eval_type _ev ;

  // here, we are certain to have vsize > 1: we have a specialization
  // for class grok_type with vsize == 1 below. We derive in_v, and out_v,
  // the data types for vetorized evaluation.

  typedef typename vspline::vector_traits
                   < in_type , vsize > :: type in_v ;
                   
  typedef typename vspline::vector_traits
                   < out_type , vsize > :: type out_v ;
  
  // given these types, we can define the types for the std::function
  // we will use to wrap the grokkee's evaluation code in.

  typedef std::function < void ( const in_v & , out_v & ) > v_eval_type ;
  
  // this is the class member holding the std::function:

  v_eval_type _v_ev ;

  // we also define a std::function type using 'normal' call/return syntax

  typedef std::function < out_v ( const in_v & ) > v_call_type ;

  /// we provide a default constructor so we can create an empty
  /// grok_type and assign to it later. Calling the empty grok_type's
  /// eval will result in an exception.

  grok_type() { } ;
  
  /// direct initialization of the internal evaluation functions
  /// this overload, with two arguments, specifies the unvectorized
  /// and the vectorized evaluation function explicitly.
  
  grok_type ( const eval_type & fev ,
              const v_eval_type & vfev )
  : _ev ( fev ) ,
    _v_ev ( vfev )
  { } ;
  
  /// constructor taking a call_type and a v_call_type,
  /// initializing the two std::functions _ev and _v_ev
  /// with wrappers around these arguments which provide
  /// the 'standard' vspline evaluation functor signature

  grok_type ( call_type f , v_call_type vf )
  : _ev ( eval_wrap ( f ) )
  , _v_ev ( eval_wrap ( vf ) )
  { } ;
    
  /// constructor from 'grokkee' using lambda expressions
  /// to initialize the std::functions _ev and _v_ev.
  /// we enable this if grokkee_type is a vspline::unary_functor

  template < class grokkee_type ,
             typename std::enable_if
              < std::is_base_of
                < vspline::unary_functor_tag < vsize > ,
                  grokkee_type
                > :: value ,
                int
              > :: type = 0
            >
  grok_type ( grokkee_type grokkee )
  : _ev ( [ grokkee ] ( const IN & in , OUT & out )
            { grokkee.eval ( in , out ) ; } )
  , _v_ev ( [ grokkee ] ( const in_v & in , out_v & out )
            { grokkee.eval ( in , out ) ; } )
  { } ;
    
  /// constructor taking only an unvectorized evaluation function.
  /// this function is broadcast, providing evaluation of SIMD types
  /// with non-vector code, which is less efficient.

  grok_type ( const eval_type & fev )
  : _ev ( fev ) ,
    _v_ev ( [ fev ] ( const in_v & in , out_v & out )
            { vspline::broadcast < IN , OUT , vsize > (fev)
                       .eval ( in , out ) ; } )
  { } ;
  
  /// constructor taking only one call_type, which is also broadcast,
  /// since the call_type std::function is wrapped to provide a
  /// std::function with vspline's standard evaluation functor signature
  /// and the result is fed to the single-argument functor above.

  grok_type ( const call_type & f )
  : grok_type ( eval_wrap ( f ) )
  { } ;
  
  /// unvectorized evaluation. This is delegated to _ev.

  void eval ( const IN & i , OUT & o ) const
  {
    _ev ( i , o ) ;
  }
  
  /// vectorized evaluation function template
  /// the eval overload above will catch calls with (in_type, out_type)
  /// while this overload template will catch vectorized evaluations.

  template < class in_v , class out_v >
  void eval ( const in_v & i , out_v & o ) const
  {
    _v_ev ( unwrap ( i ) , o ) ;
  }
  
  /// since grok_type is meant as a hold-all, type-erased shell for
  /// unary_functors, we also equip it with operator(), to save users
  /// from having to wrap it in a vspline::callable if they want to
  /// use function-call syntax with it. Since here the call is done
  /// via the std::functions, performance may be worse than what a
  /// 'straight' vspline::callable delivers.

  template < class I ,
             class O = typename base_type::template out_type_of<I> >
  O operator() ( const I & in ) const
  {
    O result ;
    eval ( in , result ) ;
    return result ;
  }
  
#endif

} ;

/// specialization of grok_type for _vsize == 1
/// this is the only possible specialization if Vc is not in use.
/// here we don't use _v_ev but only the unvectorized evaluation.

template < typename IN ,    // argument or input type
           typename OUT     // result type
         >
struct grok_type < IN , OUT , 1 >
: public vspline::unary_functor < IN , OUT , 1 >
{
  typedef vspline::unary_functor < IN , OUT , 1 > base_type ;
  using base_type::vsize ;  
  using typename base_type::in_type ;
  using typename base_type::out_type ;
  
  typedef std::function < void ( const in_type & , out_type & ) > eval_type ;
  typedef std::function < out_type ( const in_type & ) > call_type ;

  eval_type _ev ;

  grok_type() { } ;
  
  template < class grokkee_type ,
             typename std::enable_if
              < std::is_base_of
                < vspline::unary_functor_tag < 1 > ,
                  grokkee_type
                > :: value ,
                int
              > :: type = 0
           >
  grok_type ( grokkee_type grokkee )
  : _ev ( [ grokkee ] ( const IN & in , OUT & out )
            { grokkee.eval ( in , out ) ; } )
  { } ;
  
  grok_type ( const eval_type & fev )
  : _ev ( fev )
  { } ;
  
  grok_type ( call_type f )
  : _ev ( eval_wrap ( f ) )
  { } ;
    
  void eval ( const IN & i , OUT & o ) const
  {
    _ev ( i , o ) ;
  }
  
  template < class I ,
             class O = typename base_type::template out_type_of<I> >
  O operator() ( const I & in ) const
  {
    O result ;
    eval ( in , result ) ;
    return result ;
  }
} ;

/// grok() is the corresponding factory function, wrapping grokkee
/// in a vspline::grok_type.

template < class grokkee_type >
vspline::grok_type < typename grokkee_type::in_type ,
                     typename grokkee_type::out_type ,
                     grokkee_type::vsize >
grok ( const grokkee_type & grokkee )
{
  return vspline::grok_type < typename grokkee_type::in_type ,
                              typename grokkee_type::out_type ,
                              grokkee_type::vsize >
                  ( grokkee ) ;
}

} ; // end of namespace vspline

#endif // VSPLINE_UNARY_FUNCTOR_H

