/* ====================================================================
 * Copyright (c) 2003-2008, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "config.h"
#include "CopyDialog.h"
#include "RevisionWidget.h"
#include "ExternProvider.h"
#include "Settings.h"
#include "commands/CopyParam.h"
#include "sublib/Gui.h"
#include "sublib/ExternButton.h"
#include "sublib/settings/LayoutSettings.h"
#include "util/max.h"

// qt
#include <QtGui/QLayout>
#include <QtGui/QGroupBox>
#include <QtGui/QPushButton>
#include <QtGui/QLabel>
#include <QtGui/QComboBox>
#include <QtGui/QFileDialog>
#include <QtGui/QToolTip>
#include <QtGui/QRadioButton>
#include <QtGui/QButtonGroup>
#include <QtGui/QDialogButtonBox>


enum CopyDialog_CopyType {
  cptBranch,  ///< copy to branches
  cptTag,     ///< copy to tags
  cptCustom   ///< copy to user specified path
};

CopyDialog::CopyDialog( ExternProvider* p, QWidget *parent )
: super( parent, NULL, true ), _p(p)
{
  setName( "CopyDialog" );  
  setCaption( _q("subcommander:copy") );

  QVBoxLayout *vbl = new QVBoxLayout(this);
  vbl->setContentsMargins(5,5,5,5);
  vbl->setSpacing(5);
  {
    QGroupBox* gb = new QGroupBox(this);
    gb->setTitle( _q("copy options: ") );
    gb->setFlat(true);
    vbl->addWidget(gb);
    {
      QGridLayout* gl = new QGridLayout(gb);
      int left, top, right, bottom;
      gl->getContentsMargins(&left,&top,&right,&bottom);
      gl->setContentsMargins(0,top,0,bottom);
      {
        int row = 0;
        { // source
          QLabel* l  = new QLabel(gb);
          _src       = new QComboBox(gb);
          _srcBrowse = new ExternButton(gb);
          _srcBrowse->setEnabled(false);
          
          l->setBuddy(_srcBrowse);
          l->setText( _q("&source:") );
          _src->setEditable(true);
          _src->setAutoCompletion(true);
          
          gl->addWidget(l,row,0,1,2);
          row++;
          gl->addWidget(_src,row,0); 
          gl->addWidget(_srcBrowse,row,1); 
          
          connect( _src, SIGNAL(activated(const QString&)), SLOT(checkOk(const QString&)) );
          connect( _src, SIGNAL(highlighted(const QString&)), SLOT(checkOk(const QString&)) );
          connect( _src, SIGNAL(textChanged(const QString&)), SLOT(checkOk(const QString&)) );
          
          QToolTip::add( _src, _q("the source file/folder you like to copy") );
        }
        {
          row++;
          _srcRev = new RevisionWidget(true,"SND","HBCP",0,gb);
          _srcRev->setTitle( _q("select source revision: ") );
          gl->addWidget(_srcRev,row,0);
        }
        {
          row++;
          QLabel* l  = new QLabel(this);
          _dstRoot   = new QComboBox(this);
          _dstBrowse = new ExternButton(this);
          _dstBrowse->setEnabled(false);
          
          l->setBuddy(_dstBrowse);
          l->setText( _q("&destination root:") );
          //l->setFixedWidth( l->sizeHint().width() );
          _dstRoot->setEditable(true);
          _dstRoot->setAutoCompletion(true);
          _dstRoot->setInsertPolicy(QComboBox::InsertAtBottom);
          _dstRoot->setEnabled(false);
          
          gl->addWidget(l,row,0,1,2);

          row++;
          QHBoxLayout* rl = new QHBoxLayout;
          gl->addLayout(rl,row,0);
          {
            QRadioButton* b = new QRadioButton( _q("branch"), gb);
            QRadioButton* t = new QRadioButton( _q("tag"),    gb);
            QRadioButton* c = new QRadioButton( _q("custom"), gb);
            rl->addWidget(b);
            rl->addWidget(t);
            rl->addWidget(c);
            rl->addStretch(1);

            QButtonGroup* bg = new QButtonGroup(this);
            bg->addButton(b,cptBranch);
            bg->addButton(t,cptTag);
            bg->addButton(c,cptCustom);
            
            b->setChecked(true);
            
            connect( bg, SIGNAL(buttonClicked(int)), this, SLOT(copyTypeChanged(int)) );
          }
          
          row++;
          gl->addWidget(_dstRoot,row,0); 
          gl->addWidget(_dstBrowse,row,1); 
          
          connect( _dstRoot, SIGNAL(activated(const QString&)), SLOT(checkOk(const QString&)) );
          connect( _dstRoot, SIGNAL(highlighted(const QString&)), SLOT(checkOk(const QString&)) );
          connect( _dstRoot, SIGNAL(textChanged(const QString&)), SLOT(checkOk(const QString&)) );
          connect( _dstBrowse, SIGNAL(clicked()), SLOT(repExtern()) );
          
          QToolTip::add( _dstRoot,
                        _q("the destination root folder where you like to copy the source to") );          
        }
        {
          row++;
          QLabel* l = new QLabel(this);
          _dstName  = new QComboBox(this);
          
          l->setBuddy(_dstName);
          l->setText( _q("&destination name:") );
          l->setFixedWidth( l->sizeHint().width() );
          _dstName->setEditable(true);
          _dstName->setAutoCompletion(true);
          _dstName->setFocus();
          
          gl->addWidget(l,row,0,1,2);
          row++;
          gl->addWidget(_dstName,row,0); 
          
          connect( _dstName, SIGNAL(activated(const QString&)), SLOT(checkOk(const QString&)) );
          connect( _dstName, SIGNAL(highlighted(const QString&)), SLOT(checkOk(const QString&)) );
          connect( _dstName, SIGNAL(textChanged(const QString&)), SLOT(checkOk(const QString&)) );
          
          //connect( pb, SIGNAL(clicked()), SLOT(selectWcPath()) );
          
          QToolTip::add( _dstName, _q("the name of the source in the destination root folder") );
        }
      }
    }
    
    QDialogButtonBox* db = new QDialogButtonBox(this);
    db->setContentsMargins(5,5,5,5);
    vbl->addWidget(db);
    {
      int minWidth = 0;

      _ok = new QPushButton(this);
      _ok->setEnabled(false);
      _ok->setText( _q("&Ok") );
      _ok->setDefault(true);
      db->addButton( _ok, QDialogButtonBox::AcceptRole );
      minWidth = std::max( minWidth, _ok->sizeHint().width() );
      
      QPushButton* ca = new QPushButton(this);
      ca->setText( _q("&Cancel") );
      db->addButton( ca, QDialogButtonBox::RejectRole );
      minWidth = std::max( minWidth, ca->sizeHint().width() );

      _ok->setMinimumWidth(minWidth);
      ca->setMinimumWidth(minWidth);

      connect( _ok, SIGNAL(clicked()), SLOT(accept()) );
      connect( ca,  SIGNAL(clicked()), SLOT(reject()) );
    }
  }

  Settings s;
  resize( s.layout().getSize( name(), QSize(350,100) ) );
  setMaximumHeight( sizeHint().height() );
}

CopyDialog::~CopyDialog()
{
  Settings s;
  s.layout().setSize( name(), geometry().size() );
}

void CopyDialog::selectWcPath()
{
#if 0
  QString sel = QFileDialog::getExistingDirectory( _wc->currentText(), this, "",
    "select wc target path..." );

  if( ! sel.isNull() )
  {
    _wc->insertItem( sel, 0 );
  }
#endif
}

void CopyDialog::setSrcPathOrUrl( const QString& url )
{
  _src->insertItem( url, 0 );
}

QString CopyDialog::getSrcPathOrUrl()
{
  return _src->currentText();
}

void CopyDialog::addDstPathOrUrl( const QString& url )
{
  _dstRoot->insertItem( url );
}

QString CopyDialog::getDstPathOrUrl()
{
  return _dstRoot->currentText();
}

void CopyDialog::setBranchesUrl( const QString& url )
{
  _branches = url;
}

void CopyDialog::setTagsUrl( const QString& url )
{
  _tags = url;
}

svn::Revision* CopyDialog::getRevision()
{
  return _srcRev->getRevision();
}

void CopyDialog::enableSrc( bool enable )
{
  _src->setEnabled(enable);
  _srcBrowse->setEnabled(enable);
}

void CopyDialog::enableDst( bool enable )
{
  _dstRoot->setEnabled(enable);
  _dstBrowse->setEnabled(enable);
  
  if(!enable)
    copyTypeChanged(cptBranch);
}

void CopyDialog::checkOk( const QString& text )
{
  QString src     = _src->currentText();
  QString dstRoot = _dstRoot->currentText();
  QString dstName = _dstName->currentText();

  if( ! src.isEmpty() && ! dstRoot.isEmpty() && ! dstName.isEmpty() )
  {
    _ok->setEnabled(true);
  }
  else
  {
    _ok->setEnabled(false);
  }
}

QString CopyDialog::getTargetPathOrUrl()
{
  return _dstRoot->currentText() + "/" + _dstName->currentText();
}

void CopyDialog::repExtern()
{
  sc::String res;

  if( _p->selectUrl( this, sc::String(_dstRoot->currentText().utf8()), res, ExternProvider::Dir ) )
  {
    _dstRoot->setCurrentText(QString::fromUtf8(res));
  }
}

void CopyDialog::copyTypeChanged(int copyType )
{
  switch (copyType) {
    case cptBranch:
    {
      _dstRoot->setEnabled(false);
      _dstBrowse->setEnabled(false);
      _dstRoot->setEditText(_branches);
      break;
    }
    case cptTag:
    {
      _dstRoot->setEditText(_tags);
      _dstBrowse->setEnabled(false);
      _dstRoot->setEnabled(false);
      break;
    }
    case cptCustom:
    {
      _dstRoot->setEnabled(true);
      _dstBrowse->setEnabled(true);
      break;
    }
    default:
      break;
  }
}

