//filter-slfi-file2filter.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2012-2019
 *
 *  This file is part of roard a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include <roaraudio.h>
#include <libroarlight/libroarlight.h>

#define MAX_STACKSIZE 16

static int __cb_event_add(struct roar_slfi_inst * inst, void * userdata, uint8_t event) {
 (void)inst;
 return roar_slfi_event_add(userdata, event);
}

static struct roar_slfi_inst * __parse_filter(char * p) {
 struct roar_slfi_inst * ret;
 struct roar_keyval * para = NULL;
 ssize_t paralen = -1;
 char * args;
 int err;

 args = strstr(p, " ");
 if ( args != NULL ) {
  *args = 0;
  args++;
  // strip spaces:
  for (; *p == ' '; p++);

  paralen = roar_keyval_split(&para, args, NULL, NULL, 1);
 }

 ret = roar_slfi_new(p, 1, para, paralen);
 err = roar_error;

 if ( para != NULL )
  roar_mm_free(para);

 roar_err_set(err);
 return ret;
}

static struct roar_slfi_inst * __parse_vio(struct roar_vio_calls * vio) {
 struct roar_slfi_inst * stack[MAX_STACKSIZE] = {NULL};
 size_t stackpointer = 0;
 size_t i;
 char buf[1024];
 char * p;
 int err = ROAR_ERROR_NONE;
 char * end;

 // create root node.
 stack[stackpointer] = roar_slfi_new("chain", 1, NULL, -1);
 if ( stack[stackpointer] == NULL )
  return NULL;
 stackpointer++;

 while ((p = roar_vio_fgets(vio, buf, sizeof(buf))) != NULL) {
  // strip spaces:
  for (; *p == ' '; p++);

  // strip newlines.
  for (end = p; *end; end++) {
   if ( *end == '\n' || *end == '\r' ) {
    *end = 0;
    break;
   }
  }
  // skip comments:
  if ( *p == '#' )
   continue;
  if ( *p == '{' ) {
   if ( stackpointer == (MAX_STACKSIZE-1) ) {
    err = ROAR_ERROR_NOSPC;
    break;
   }
   stackpointer++;
   stack[stackpointer] = NULL;
  } else if ( *p == '}' ) {
   if ( stackpointer == 1 ) {
    err = ROAR_ERROR_ILLSEQ;
    break;
   }
   if ( stack[stackpointer] != NULL ) {
    roar_slfi_unref(stack[stackpointer]);
    stack[stackpointer] = NULL;
   }
   stackpointer--;
  } else {
   if ( stack[stackpointer] != NULL ) {
    roar_slfi_unref(stack[stackpointer]);
    stack[stackpointer] = NULL;
   }
   stack[stackpointer] = __parse_filter(p);
   if ( stack[stackpointer] == NULL ) {
    err = roar_error;
    break;
   }
   if ( roar_slfi_ctl(stack[stackpointer-1], ROAR_SLFI_CMD_PUSH, stack[stackpointer]) == -1 ) {
    err = roar_error;
    break;
   }
  }
 }

 for (i = stackpointer; i; i--)
  roar_slfi_unref(stack[i]);

 if (err != ROAR_ERROR_NONE ) {
  roar_slfi_unref(stack[0]);
  stack[0] = NULL;
 }

 roar_err_set(err);
 return stack[0];
}

static struct roar_slfi_inst * __parse_file(const char * file) {
 struct roar_vio_calls vio;
 struct roar_slfi_inst * ret;
 int err;

 if ( file == NULL ) {
  roar_err_set(ROAR_ERROR_INVAL);
  return NULL;
 }

 if ( roar_vio_open_dstr_simple(&vio, file, ROAR_VIOF_READ) == -1 )
  return NULL;

 ret = __parse_vio(&vio);
 err = roar_error;

 roar_vio_close(&vio);

 roar_err_set(err);
 return ret;
}

static int __init(struct roar_slfi_inst * inst, const struct roar_keyval * para, ssize_t paralen) {
 const struct roar_keyval * kv;
 const char * filename = NULL;
 ssize_t i;

 for (i = 0; i < paralen; i++) {
  kv = &(para[i]);
  if ( kv->key == NULL || kv->value == NULL )
   continue;
  if ( !strcmp(kv->key, "file") ) {
   filename = kv->value;
  } else {
   ROAR_WARN("__init(*): Unknown parameter: %s", kv->key);
  }
 }

 inst->userdata = __parse_file(filename);
 if ( inst->userdata == NULL )
  return -1;

 roar_slfi_cb_set_event_add(inst->userdata, __cb_event_add, inst);

 return 0;
}

static int __uninit(struct roar_slfi_inst * inst) {
 roar_slfi_unref(inst->userdata);
 inst->userdata = NULL;
 return 0;
}

static int __update(struct roar_slfi_inst * inst, uint8_t * universe, ssize_t size_of_universe, int32_t usecspassed, const uint8_t * event, size_t eventlen) {
 return roar_slfi_update(inst->userdata, universe, size_of_universe, usecspassed, event, eventlen);
}

static const struct roar_slfi_filter filter[1] = {
 {
  .name = "file2filter",
  .description = "SLFI filter structure builder",
  .flags = ROAR_SLFI_FLAG_ON_UPDATE,
  .init = __init,
  .uninit = __uninit,
  .update = __update,
  .ctl = NULL
 }
};

ROAR_DL_PLUGIN_REG_SLFI(filter);

// This is the plugin control block.
ROAR_DL_PLUGIN_START(filter_slfi_file2filter) {
 // Here we set the name and vendor of our plugin.
 // If you have no Vendor ID you need to use ROAR_DL_PLUGIN_META_PRODUCT_NV().
 ROAR_DL_PLUGIN_META_PRODUCT_NIV("filter-slfi-file2filter", ROAR_VID_ROARAUDIO, ROAR_VNAME_ROARAUDIO);

 // This sets the version of your plugin.
 ROAR_DL_PLUGIN_META_VERSION(ROAR_VERSION_STRING);

 // This sets the license of your plugin.
 // If there is no tag for the license you use you can just
 // use ROAR_DL_PLUGIN_META_LICENSE().
 ROAR_DL_PLUGIN_META_LICENSE_TAG(GPLv3_0);

 // This sets the author and contact infos.
 // There are several other macros to do this with other parameters.
 // See ROAR_DL_PLUGIN_META_CONTACT*() in the header or documentation.
 ROAR_DL_PLUGIN_META_CONTACT_FLNE("Philipp", "Schafft", "ph3-der-loewe", "lion@lion.leolix.org");

 // This sets the description for your plugin.
 ROAR_DL_PLUGIN_META_DESC("This plugin loads SLFI filter structures based on information stored in a file.");

 // Load filters.
 ROAR_DL_PLUGIN_REG_FNFUNC(ROAR_DL_FN_FILTER);

// This is the end of the control block.
} ROAR_DL_PLUGIN_END

//ll
