#ifndef _RHEOLEF_FIELD_EXPR_V2_NONLINEAR_TERMINAL_H
#define _RHEOLEF_FIELD_EXPR_V2_NONLINEAR_TERMINAL_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// terminals (leaves) for the non-linear expression tree
//
// 0) utilities
// 1) class-function
//    1.1) base class for the class-function family
//    1.2) general function or class-function
//    1.3) normal to a surface
//    1.4) h_local
//    1.5) penalty
// 2) field and such
//    2.1) base class for the field family
//    2.2) field
//    2.3) grad of a field
//    2.4)  div of a field
//    2.5) curl of a field
//    2.6) jump of a field
// 3) convected field, as compose(uh,X) where X is a characteristic
//
#include "rheolef/field_expr_v2_linear.h"
#include "rheolef/basis_on_pointset.h"
#include "rheolef/field_expr_v2_utilities.h"
#include "rheolef/test.h"
#include "rheolef/characteristic.h"

#ifdef TODO
#include "rheolef/field_evaluate.h"
#endif // TODO


namespace rheolef { 
// -------------------------------------------------------------------
// 0. utilities
// -------------------------------------------------------------------
// 0.1. field_nonlinear_expr_v2, a type concept for nonlinear expression
// -------------------------------------------------------------------
namespace details {

// Define a trait type for detecting field expression valid arguments
template<class Expr, class Sfinae = void> struct is_field_expr_v2_nonlinear_arg : std::false_type {};
template<class Expr>                      struct is_field_expr_v2_nonlinear_arg <Expr,
  typename std::enable_if<
    is_field_expr_v2_linear_arg<Expr>::value
  >::type
> : std::true_type {};

// -------------------------------------------------------------------
// 0.2. terminal wrapper for exprs
// -------------------------------------------------------------------
template <class Expr, class Sfinae = void>
struct field_expr_v2_nonlinear_terminal_wrapper_traits
{ 
  // catch-all case: non-terminals are not wrapped
  typedef Expr   type;
};

} // namespace details
// ---------------------------------------------------------------------------
// 1. field-function
// ---------------------------------------------------------------------------
// 1.1. base class for the field-function family
// ---------------------------------------------------------------------------
namespace details {

template<class T>
class field_expr_v2_nonlinear_terminal_function_base_rep {
public:
// typedefs:

  typedef geo_element::size_type                                size_type;
  typedef rheo_default_memory_model                             memory_type; // TODO: deduce it
  typedef T                                                     scalar_type;
  typedef T                                                     float_type;

// allocators:

  field_expr_v2_nonlinear_terminal_function_base_rep ();
  field_expr_v2_nonlinear_terminal_function_base_rep (const field_expr_v2_nonlinear_terminal_function_base_rep<T>&);

// accessors:

  bool initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const;
  bool initialize (const space_basic<float_type,memory_type>& Xh) const;
  void evaluate_initialize (const geo_element& K) const;
  const geo_element& get_side (const geo_element& K, const side_information_type& sid) const;

// data:
protected:
  mutable geo_basic<float_type,memory_type> _omega;
  mutable basis_on_pointset<float_type>     _piola_ops;
  mutable std::vector<size_type>            _dis_inod; // working data:
};
// ---------------------------------------------------------------------------
// 1.2) general function or class-function
// ---------------------------------------------------------------------------
template<class Function>
class field_expr_v2_nonlinear_terminal_function_rep 
 : field_expr_v2_nonlinear_terminal_function_base_rep
     <typename float_traits<typename details::function_traits<Function>::result_type>::type>
{
public:
// typedefs:

  typedef geo_element::size_type                                size_type;
  typedef rheo_default_memory_model                             memory_type; // TODO: how to deduce it ?
  typedef typename details::function_traits<Function>::copiable_type  function_type;
  typedef typename details::function_traits<Function>::result_type   result_type;
  typedef typename details::function_traits<Function>::template arg<0>::type argument_type;
  typedef typename scalar_traits<result_type>::type             scalar_type;
  typedef typename  float_traits<result_type>::type             float_type;
  typedef result_type                                           value_type;
  typedef field_expr_v2_nonlinear_terminal_function_base_rep<float_type>     base;

// alocators:

  field_expr_v2_nonlinear_terminal_function_rep(const Function& f) : base(), _f(f) {}

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::valued_tag_traits<result_type>::value;

  space_constant::valued_type valued_tag() const {
      return space_constant::valued_tag_traits<result_type>::value; }

  bool initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& quad) const {
    return base::initialize (omega, quad);
  }
  bool initialize (const space_basic<float_type,memory_type>& Xh) const {
    return base::initialize (Xh);
  }
  bool evaluate (const geo_element& K, std::vector<result_type>& value) const {
    base::evaluate_initialize (K);
    reference_element hat_K = K.variant();
    value.resize (base::_piola_ops.size(hat_K));
    size_type q = 0;
    for (typename std::vector<result_type>::iterator
	  iter = value.begin(),
	  last = value.end(); iter != last; ++iter, ++q) {
      point_basic<float_type> xq = piola_transformation (base::_omega, base::_piola_ops, K, base::_dis_inod, q);
      *iter = _f (xq);
    }
    return true;
  }
  void evaluate_on_side (const geo_element& L, const side_information_type& sid, std::vector<result_type>& value) const {
    evaluate (base::get_side(L,sid), value);
  }
  template<class ValueType>
  bool valued_check() const {
    static const bool status = is_equal<ValueType,result_type>::value;
    check_macro (status, "unexpected result_type");
    return status;
  }
// data:
protected:
  function_type  _f;
};
template<class Function>
class field_expr_v2_nonlinear_terminal_function : public smart_pointer<field_expr_v2_nonlinear_terminal_function_rep<Function> >
{
public:
// typedefs:

  typedef field_expr_v2_nonlinear_terminal_function_rep<Function> rep;
  typedef smart_pointer<rep>                    base;
  typedef typename rep::size_type               size_type;
  typedef typename rep::memory_type             memory_type;
  typedef typename rep::result_type             result_type;
  typedef typename rep::argument_type           argument_type;
  typedef typename rep::value_type              value_type;
  typedef typename rep::scalar_type             scalar_type;
  typedef typename rep::float_type              float_type;
  static const space_constant::valued_type valued_hint = rep::valued_hint;

// alocators:

  explicit field_expr_v2_nonlinear_terminal_function (const Function& f) 
    : base(new_macro(rep(f))) {}

  template<class TrueFunction, 
           class Sfinae = typename std::enable_if<std::is_function<TrueFunction>::value, TrueFunction>::type>
  explicit field_expr_v2_nonlinear_terminal_function (TrueFunction f) 
    : base(new_macro(rep(std::ptr_fun(f)))) {} 

  template <class Constant, 
            class Sfinae = typename std::enable_if <details::is_field_expr_v2_constant<Constant>::value, Constant>::type>
  explicit field_expr_v2_nonlinear_terminal_function (const Constant& c) 
    : base(new_macro(rep(f_constant<point_basic<float_type>,result_type>(c)))) {}

#ifdef TO_CLEAN
  template <class FieldFunctor, class Result>
  explicit field_expr_v2_nonlinear_terminal_function (const field_functor<FieldFunctor,Result>& f)
    : base(new_macro(rep(f.get_ref()))) {}
#endif // TO_CLEAN

// accessors:

  space_constant::valued_type valued_tag() const { return base::data().valued_tag(); }

  bool initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const {
  	return base::data().initialize(omega,hat_x);
  }
  bool initialize (const space_basic<float_type,memory_type>& Xh) const {
  	return base::data().initialize(Xh);
  }
  bool evaluate (const geo_element& K, std::vector<result_type>& value) const {
  	return base::data().evaluate (K, value); }
  void evaluate_on_side (const geo_element& L, const side_information_type& sid, std::vector<result_type>& value) const {
  	base::data().evaluate_on_side (L, sid, value); }
  template<class ValueType>
  bool valued_check() const {
  	return base::data().template valued_check<ValueType>(); }
};
// concept::
template<class F> struct is_field_expr_v2_nonlinear_arg     <field_expr_v2_nonlinear_terminal_function<F> > : std::true_type {};
template<class F> struct is_field_expr_v2_nonlinear_arg     <F,
  typename std::enable_if<
    is_field_function<F>::value
  >::type
> : std::true_type {};

// wrapper:
template <class Expr>
struct field_expr_v2_nonlinear_terminal_wrapper_traits <Expr,
  typename std::enable_if<
    is_field_function<Expr>::value
  >::type
>
{
  typedef field_expr_v2_nonlinear_terminal_function<Expr>   type;
};
// wrapper for Expr=constant (used by nary compose to wrap constant args)
template <class Expr>
struct field_expr_v2_nonlinear_terminal_wrapper_traits <Expr,
  typename std::enable_if<
    is_field_expr_v2_constant<Expr>::value
  >::type
>
{
  typedef typename promote<Expr,Float>::type                 float_type; // promote int to Float, at least
  typedef f_constant<point_basic<float_type>,Expr>           fun_t;
  typedef field_expr_v2_nonlinear_terminal_function<fun_t>   type;
};
// ---------------------------------------------------------------------------
// 1.3) normal to a surface
// ---------------------------------------------------------------------------
// the special class-function, used in nonlinear field expressions:
template<class T>
struct normal_pseudo_function : std::unary_function <point_basic<T>, point_basic<T> > {
  point_basic<T> operator() (const point_basic<T>&) const {
    fatal_macro ("special normal() class-function should not be directly evaluated");
    return point_basic<T>();
  }
};
template<class T>
class field_expr_v2_nonlinear_terminal_function_rep<normal_pseudo_function<T> >
 : field_expr_v2_nonlinear_terminal_function_base_rep<T> {
public:
// typedefs:

  typedef field_expr_v2_nonlinear_terminal_function_base_rep<T> base;
  typedef geo_element::size_type                                size_type;
  typedef rheo_default_memory_model                             memory_type; // TODO: deduce it
  typedef normal_pseudo_function<T>                             function_type;
  typedef point_basic<T>                                        result_type;
  typedef point_basic<T>                                        argument_type;
  typedef T                                                     scalar_type;
  typedef T                                                     float_type;
  typedef result_type                                           value_type;

// alocators:

  field_expr_v2_nonlinear_terminal_function_rep (const function_type&);
  field_expr_v2_nonlinear_terminal_function_rep (const field_expr_v2_nonlinear_terminal_function_rep<normal_pseudo_function<T> >&);

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::vector;

  space_constant::valued_type valued_tag() const { return space_constant::vector; }

  bool initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const;
  bool initialize (const space_basic<float_type,memory_type>& Xh) const;

  bool evaluate         (const geo_element& K,                                   std::vector<point_basic<T> >& value) const;
  void evaluate_on_side (const geo_element& L, const side_information_type& sid, std::vector<point_basic<T> >& value) const;

  template<class ValueType>
  bool valued_check() const {
    static const bool status = details::is_equal<ValueType,result_type>::value;
    check_macro (status, "unexpected result_type");
    return status;
  }
};

} // namespace details

// the normal() pseudo-function
template<class T>
inline
details::field_expr_v2_nonlinear_terminal_function <details::normal_pseudo_function<T> >
normal_basic()
{
  return details::field_expr_v2_nonlinear_terminal_function
	   <details::normal_pseudo_function<T> >
           (details::normal_pseudo_function<T>());
}
inline
details::field_expr_v2_nonlinear_terminal_function <details::normal_pseudo_function<Float> >
normal()
{
  return normal_basic<Float>();
}
// ---------------------------------------------------------------------------
// 1.4) h_local
// ---------------------------------------------------------------------------
namespace details {

// the special class-function, used in nonlinear field expressions:
template<class T>
struct h_local_pseudo_function : std::unary_function <point_basic<T>, T> {
  T operator() (const point_basic<T>&) const {
    fatal_macro ("special h_local() class-function should not be directly evaluated");
    return 0;
  }
};

template<class T>
class field_expr_v2_nonlinear_terminal_function_rep<h_local_pseudo_function<T> >
 : field_expr_v2_nonlinear_terminal_function_base_rep<T> {
public:
// typedefs:

  typedef field_expr_v2_nonlinear_terminal_function_base_rep<T> base;
  typedef geo_element::size_type                                size_type;
  typedef rheo_default_memory_model                             memory_type; // TODO: deduce it
  typedef h_local_pseudo_function<T>                            function_type;
  typedef T                                                     result_type;
  typedef point_basic<T>                                        argument_type;
  typedef T                                                     scalar_type;
  typedef T                                                     float_type;
  typedef result_type                                           value_type;

// alocators:

  field_expr_v2_nonlinear_terminal_function_rep (const function_type&);
  field_expr_v2_nonlinear_terminal_function_rep (const field_expr_v2_nonlinear_terminal_function_rep<h_local_pseudo_function<T> >&);

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::scalar;

  space_constant::valued_type valued_tag() const { return space_constant::scalar; }

  bool initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const;
  bool initialize (const space_basic<float_type,memory_type>& Xh) const;

  bool evaluate         (const geo_element& K,                                   std::vector<result_type>& value) const;
  void evaluate_on_side (const geo_element& L, const side_information_type& sid, std::vector<result_type>& value) const;

  template<class ValueType>
  bool valued_check() const {
    static const bool status = details::is_equal<ValueType,result_type>::value;
    check_macro (status, "unexpected result_type");
    return status;
  }
};

} // namespace details

// the h_local() pseudo-function
template<class T>
inline
details::field_expr_v2_nonlinear_terminal_function <details::h_local_pseudo_function<T> >
h_local_basic()
{
  return details::field_expr_v2_nonlinear_terminal_function
	   <details::h_local_pseudo_function<T> >
           (details::h_local_pseudo_function<T>());
}
inline
details::field_expr_v2_nonlinear_terminal_function <details::h_local_pseudo_function<Float> >
h_local()
{
  return h_local_basic<Float>();
}
// ---------------------------------------------------------------------------
// 1.5) penalty
// ---------------------------------------------------------------------------
// the special class-function, used in nonlinear field expressions:
namespace details {

template<class T>
struct penalty_pseudo_function : std::unary_function <point_basic<T>, T> {
  T operator() (const point_basic<T>&) const {
    fatal_macro ("special penalty() class-function should not be directly evaluated");
    return 0;
  }
};

template<class T>
class field_expr_v2_nonlinear_terminal_function_rep<penalty_pseudo_function<T> >
 : field_expr_v2_nonlinear_terminal_function_base_rep<T> {
public:
// typedefs:

  typedef field_expr_v2_nonlinear_terminal_function_base_rep<T> base;
  typedef geo_element::size_type                                size_type;
  typedef rheo_default_memory_model                             memory_type; // TODO: deduce it
  typedef penalty_pseudo_function<T>                            function_type;
  typedef T                                                     result_type;
  typedef point_basic<T>                                        argument_type;
  typedef T                                                     scalar_type;
  typedef T                                                     float_type;
  typedef result_type                                           value_type;

// alocators:

  field_expr_v2_nonlinear_terminal_function_rep (const function_type&);
  field_expr_v2_nonlinear_terminal_function_rep (const field_expr_v2_nonlinear_terminal_function_rep<penalty_pseudo_function<T> >&);

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::scalar;

  space_constant::valued_type valued_tag() const { return space_constant::scalar; }

  bool initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const;
  bool initialize (const space_basic<float_type,memory_type>& Xh) const;

  bool evaluate         (const geo_element& K,                                   std::vector<result_type>& value) const;
  void evaluate_on_side (const geo_element& L, const side_information_type& sid, std::vector<result_type>& value) const;

  template<class ValueType>
  bool valued_check() const {
    static const bool status = details::is_equal<ValueType,result_type>::value;
    check_macro (status, "unexpected result_type");
    return status;
  }
protected:
// internal:
  T evaluate_measure  (const geo_element& K) const;
  void evaluate_internal (const geo_element& K, const geo_element& L, std::vector<result_type>& value) const;
};

} // namespace details

// the penalty() pseudo-function
template<class T>
inline
details::field_expr_v2_nonlinear_terminal_function <details::penalty_pseudo_function<T> >
penalty_basic()
{
  return details::field_expr_v2_nonlinear_terminal_function
	   <details::penalty_pseudo_function<T> >
           (details::penalty_pseudo_function<T>());
}
inline
details::field_expr_v2_nonlinear_terminal_function <details::penalty_pseudo_function<Float> >
penalty()
{
  return penalty_basic<Float>();
}

// ---------------------------------------------------------------------------
// 2) field and such
// ---------------------------------------------------------------------------
// ---------------------------------------------------------------------------
// 2.1) base class for the field family
// ---------------------------------------------------------------------------
namespace details {

template<class T, class M>
class field_expr_v2_nonlinear_terminal_field_base_rep {
public:
// typedefs:

  typedef geo_element::size_type          size_type;

// alocators:

#ifdef TO_CLEAN
  field_expr_v2_nonlinear_terminal_field_base_rep (const field_basic<T,M>& uh)
    : _uh(uh),   _omega(), _bops(), _use_dom2bgd(false), _use_bgd2dom(false), 
      _have_dg_on_sides(false), _tilde_L(), _dis_idof() {}
#endif // TO_CLEAN

  template <class Expr,
            class Sfinae = typename std::enable_if<details::is_field_expr_v2_linear_arg<Expr>::value>::type>
  explicit field_expr_v2_nonlinear_terminal_field_base_rep (const Expr& expr) 
    : _uh(expr), _omega(), _bops(), _use_dom2bgd(false), _use_bgd2dom(false), 
      _have_dg_on_sides(false), _tilde_L(), _dis_idof() {}

// accessors:

  const geo_basic<T,M>& get_geo() const { return _omega; }
  const geo_element& get_side (const geo_element& K, const side_information_type& sid) const;

  bool initialize (const geo_basic<T,M>& omega, const quadrature<T>& quad) const;
  bool initialize (const space_basic<T,M>& Xh) const;
  void evaluate_initialize (const geo_element& K) const;
  void evaluate_on_side_initialize (const geo_element& L, const side_information_type& sid) const;

// data:
//protected:
  field_basic<T,M>                 _uh;
  mutable geo_basic<T,M>           _omega;
  mutable basis_on_pointset<T>     _bops;
  mutable bool                     _use_dom2bgd;
  mutable bool                     _use_bgd2dom;
  mutable bool                     _have_dg_on_sides;
  mutable reference_element        _tilde_L;
// working data:
  mutable std::vector<size_type>   _dis_idof;
};
// ---------------------------------------------------------------------------
// 2.2) field 
// ---------------------------------------------------------------------------
template<class T, class M>
class field_expr_v2_nonlinear_terminal_field_rep : public field_expr_v2_nonlinear_terminal_field_base_rep<T,M> {
public:
// typedefs:

  typedef field_expr_v2_nonlinear_terminal_field_base_rep<T,M> base;
  typedef geo_element::size_type          size_type;
  typedef M                               memory_type;
  typedef undeterminated_basic<T>         result_type;
  typedef result_type                     value_type;
  typedef T                               scalar_type;
  typedef typename float_traits<T>::type  float_type;

// alocators:

#ifdef TO_CLEAN
  field_expr_v2_nonlinear_terminal_field_rep (const field_basic<T,M>& uh) 
   : base(uh) {}
#endif // TO_CLEAN

  template <class Expr,
            class Sfinae = typename std::enable_if<details::is_field_expr_v2_linear_arg<Expr>::value>::type>
  explicit field_expr_v2_nonlinear_terminal_field_rep (const Expr& expr) 
    : base(expr) {}

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::last_valued;

  space_constant::valued_type valued_tag() const { return base::_uh.valued_tag(); }

  const geo_basic<T,M>& get_geo() const { return base::get_geo(); }
  bool initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& quad) const;
  bool initialize (const space_basic<float_type,memory_type>& Xh) const;
  template<class ValueType>
  bool evaluate (const geo_element& K, std::vector<ValueType>& value) const;
  template<class ValueType>
  void evaluate_on_side (const geo_element& L, const side_information_type& sid, std::vector<ValueType>& value) const;
  template<class ValueType>
  bool valued_check() const {
    space_constant::valued_type valued_tag = space_constant::valued_tag_traits<ValueType>::value;
    bool status = (base::_uh.valued_tag() == valued_tag) ||
                  (base::_uh.valued_tag() == space_constant::unsymmetric_tensor &&
                   valued_tag             == space_constant::tensor);
    check_macro (status,
           "unexpected "<< base::_uh.valued()
        << "-valued field while a " << space_constant::valued_name(valued_tag)
        << "-valued one is expected in expression");
    return status;
  }
};
template<class T, class M>
class field_expr_v2_nonlinear_terminal_field : public smart_pointer<field_expr_v2_nonlinear_terminal_field_rep<T,M> >
{
public:
// typedefs:

  typedef field_expr_v2_nonlinear_terminal_field_rep<T,M>    rep;
  typedef smart_pointer<rep>                    base;
  typedef typename rep::size_type               size_type;
  typedef typename rep::memory_type             memory_type;
  typedef typename rep::result_type             result_type;
  typedef typename rep::float_type              float_type;
  typedef typename rep::scalar_type             scalar_type;
  typedef typename rep::value_type              value_type;
  static const space_constant::valued_type valued_hint = rep::valued_hint;

// alocators:

  template <class Expr,
            class Sfinae = typename std::enable_if<details::is_field_expr_v2_linear_arg<Expr>::value>::type>
  explicit field_expr_v2_nonlinear_terminal_field (const Expr& expr) 
    : base(new_macro(rep(expr))) {} 

// accessors:

  space_constant::valued_type valued_tag() const { return base::data().valued_tag(); }

  const geo_basic<T,M>& get_geo() const { return base::data().get_geo(); }

  bool initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const {
  	return base::data().initialize(omega,hat_x);
  }
  bool initialize (const space_basic<float_type,memory_type>& Xh) const {
  	return base::data().initialize(Xh);
  }
  template<class ValueType>
  bool valued_check() const {
  	return base::data().template valued_check<ValueType>();
  }
  template<class ValueType>
  bool evaluate (const geo_element& K, std::vector<ValueType>& value) const {
  	return base::data().evaluate(K,value);
  } 
  template<class ValueType>
  void evaluate_on_side (const geo_element& L, const side_information_type& sid, std::vector<ValueType>& value) const {
  	base::data().evaluate_on_side (L, sid, value); }
};
// concept::
template<class T, class M> struct is_field_expr_v2_nonlinear_arg     <field_expr_v2_nonlinear_terminal_field<T,M> > : std::true_type {};

// wrapper:
template <class Expr>
struct field_expr_v2_nonlinear_terminal_wrapper_traits <Expr,
  typename std::enable_if<
    is_field_expr_v2_linear_arg<Expr>::value
  >::type
>
{
  typedef typename Expr::scalar_type T;
  typedef typename Expr::memory_type M;
  typedef field_expr_v2_nonlinear_terminal_field<T,M>   type;
};
// ---------------------------------------------------------------------------
// 2.3) grad of a field
// ---------------------------------------------------------------------------
template<class T, class M>
class field_expr_v2_nonlinear_terminal_field_grad_rep : public field_expr_v2_nonlinear_terminal_field_base_rep<T,M> {
public:
// typedefs:

  typedef field_expr_v2_nonlinear_terminal_field_base_rep<T,M> base;
  typedef geo_element::size_type          size_type;
  typedef M                               memory_type;
  typedef undeterminated_basic<T>         result_type;
  typedef result_type                     value_type;
  typedef T                               scalar_type;
  typedef typename float_traits<T>::type  float_type;

// allocators:

  template <class Expr,
            class Sfinae = typename std::enable_if<details::is_field_expr_v2_linear_arg<Expr>::value>::type>
  explicit field_expr_v2_nonlinear_terminal_field_grad_rep (const Expr& expr, const grad_option_type& opt) 
    : base(expr), 
      _opt(opt), 
      _u_test(base::_uh.get_space())
    {}

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::last_valued;

  space_constant::valued_type valued_tag() const;

  bool initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& quad) const;
  bool initialize (const space_basic<float_type,memory_type>& Xh) const;
  template<class ValueType>
  bool evaluate (const geo_element& K, std::vector<ValueType>& value) const;
  template<class ValueType>
  bool valued_check() const {
	_u_test.template grad_valued_check<ValueType>(); 
	return true;
  }
protected:
// data:
  grad_option_type          _opt;
  test_basic<T,M,vf_tag_01> _u_test;
};

template<class T, class M>
class field_expr_v2_nonlinear_terminal_field_grad : public smart_pointer<field_expr_v2_nonlinear_terminal_field_grad_rep<T,M> >
{
public:
// typedefs:

  typedef field_expr_v2_nonlinear_terminal_field_grad_rep<T,M>    rep;
  typedef smart_pointer<rep>                    base;
  typedef typename rep::size_type               size_type;
  typedef typename rep::memory_type             memory_type;
  typedef typename rep::result_type             result_type;
  typedef typename rep::float_type              float_type;
  typedef typename rep::scalar_type             scalar_type;
  typedef typename rep::value_type              value_type;
  static const space_constant::valued_type valued_hint = rep::valued_hint;

// alocators:

  template <class Expr,
            class Sfinae = typename std::enable_if<details::is_field_expr_v2_linear_arg<Expr>::value>::type>
  explicit field_expr_v2_nonlinear_terminal_field_grad (const Expr& expr, const grad_option_type& opt = grad_option_type()) 
    : base(new_macro(rep(expr,opt))) {} 

// accessors:

  space_constant::valued_type valued_tag() const { return base::data().valued_tag(); }

  template<class Result>
  bool evaluate (const geo_element& K, std::vector<Result>& value) const {
  	return base::data().evaluate(K,value); }

  bool initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const {
  	return base::data().initialize(omega,hat_x);
  }
  bool initialize (const space_basic<float_type,memory_type>& Xh) const {
  	return base::data().initialize(Xh);
  }
  template<class ValueType>
  bool valued_check() const {
  	return base::data().template valued_check<ValueType>();
  }
};
template<class T, class M> struct is_field_expr_v2_nonlinear_arg     <field_expr_v2_nonlinear_terminal_field_grad<T,M> > : std::true_type {};

} // namespace details 

// grad(uh)
template<class Expr>
inline
typename
std::enable_if<
  details::is_field_expr_v2_linear_arg<Expr>::value
 ,details::field_expr_v2_nonlinear_terminal_field_grad<
    typename Expr::scalar_type
   ,typename Expr::memory_type
  >
>::type
grad (const Expr& expr)
{
  typedef typename Expr::scalar_type T;
  typedef typename Expr::memory_type M;
  return details::field_expr_v2_nonlinear_terminal_field_grad<T,M> (expr);
}
// grad_s(uh)
template<class Expr>
inline
typename
std::enable_if<
  details::is_field_expr_v2_linear_arg<Expr>::value
 ,details::field_expr_v2_nonlinear_terminal_field_grad<
    typename Expr::scalar_type
   ,typename Expr::memory_type
  >
>::type
grad_s (const Expr& expr)
{
  typedef typename Expr::scalar_type T;
  typedef typename Expr::memory_type M;
  static details::grad_option_type   opt;
  opt.surfacic = true;
  return details::field_expr_v2_nonlinear_terminal_field_grad<T,M> (expr, opt);
}
// grad_h(uh)
template<class Expr>
inline
typename
std::enable_if<
  details::is_field_expr_v2_linear_arg<Expr>::value
 ,details::field_expr_v2_nonlinear_terminal_field_grad<
    typename Expr::scalar_type
   ,typename Expr::memory_type
  >
>::type
grad_h (const Expr& expr)
{
  typedef typename Expr::scalar_type T;
  typedef typename Expr::memory_type M;
  static details::grad_option_type   opt;
  opt.broken = true;
  return details::field_expr_v2_nonlinear_terminal_field_grad<T,M> (expr, opt);
}
// D(uh)
template<class Expr>
inline
typename
std::enable_if<
  details::is_field_expr_v2_linear_arg<Expr>::value
 ,details::field_expr_v2_nonlinear_terminal_field_grad<
    typename Expr::scalar_type
   ,typename Expr::memory_type
  >
>::type
D (const Expr& expr)
{
  typedef typename Expr::scalar_type T;
  typedef typename Expr::memory_type M;
  details::grad_option_type opt;
  opt.symmetrized = true;
  return details::field_expr_v2_nonlinear_terminal_field_grad<T,M> (expr, opt);
}
// Ds(uh)
template<class Expr>
inline
typename
std::enable_if<
  details::is_field_expr_v2_linear_arg<Expr>::value
 ,details::field_expr_v2_nonlinear_terminal_field_grad<
    typename Expr::scalar_type
   ,typename Expr::memory_type
  >
>::type
Ds (const Expr& expr)
{
  typedef typename Expr::scalar_type T;
  typedef typename Expr::memory_type M;
  details::grad_option_type opt;
  opt.symmetrized = true;
  opt.surfacic = true;
  return details::field_expr_v2_nonlinear_terminal_field_grad<T,M> (expr, opt);
}
// Ds(uh)
template<class Expr>
inline
typename
std::enable_if<
  details::is_field_expr_v2_linear_arg<Expr>::value
 ,details::field_expr_v2_nonlinear_terminal_field_grad<
    typename Expr::scalar_type
   ,typename Expr::memory_type
  >
>::type
Dh (const Expr& expr)
{
  typedef typename Expr::scalar_type T;
  typedef typename Expr::memory_type M;
  details::grad_option_type opt;
  opt.symmetrized = true;
  opt.broken = true;
  return details::field_expr_v2_nonlinear_terminal_field_grad<T,M> (expr, opt);
}
// ---------------------------------------------------------------------------
// 2.4) div of a field
// ---------------------------------------------------------------------------
namespace details {

template<class T, class M>
class field_expr_v2_nonlinear_terminal_field_div_rep : public field_expr_v2_nonlinear_terminal_field_base_rep<T,M> {
public:
// typedefs:


  typedef field_expr_v2_nonlinear_terminal_field_base_rep<T,M> base;
  typedef geo_element::size_type          size_type;
  typedef M                               memory_type;
#ifdef TODO
  typedef undeterminated_basic<T>         result_type; // div(tensor) -> vector : TODO
#endif // TODO
  typedef T		                  result_type; // div(vector) -> scalar
  typedef result_type                     value_type;
  typedef T                               scalar_type;
  typedef typename float_traits<T>::type  float_type;

// alocators:

  template <class Expr,
            class Sfinae = typename std::enable_if<details::is_field_expr_v2_linear_arg<Expr>::value>::type>
  explicit field_expr_v2_nonlinear_terminal_field_div_rep (const Expr& expr, const grad_option_type& opt) 
    : base(expr),
      _opt(opt), 
      _u_test(base::_uh.get_space())
    {}

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::last_valued;

  space_constant::valued_type valued_tag() const;

  bool initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& quad) const;
  bool initialize (const space_basic<float_type,memory_type>& Xh) const;
  template<class ValueType>
  bool evaluate (const geo_element& K, std::vector<ValueType>& value) const;

  template<class ValueType>
  void evaluate_on_side (const geo_element& L, const side_information_type& sid, std::vector<ValueType>& value) const {
    base::evaluate_on_side_initialize (L, sid);
    evaluate (L, value);
  }
  template<class ValueType>
  bool valued_check() const { 
	_u_test.template div_valued_check<ValueType>();
	return true;
  }
// data:
  grad_option_type          _opt;
  test_basic<T,M,vf_tag_01> _u_test;
};

template<class T, class M>
class field_expr_v2_nonlinear_terminal_field_div : public smart_pointer<field_expr_v2_nonlinear_terminal_field_div_rep<T,M> >
{
public:
// typedefs:

  typedef field_expr_v2_nonlinear_terminal_field_div_rep<T,M>    rep;
  typedef smart_pointer<rep>                    base;
  typedef typename rep::size_type               size_type;
  typedef typename rep::memory_type             memory_type;
  typedef typename rep::result_type             result_type;
  typedef typename rep::float_type              float_type;
  typedef typename rep::scalar_type             scalar_type;
  typedef typename rep::value_type              value_type;
  static const space_constant::valued_type valued_hint = rep::valued_hint;

// alocators:

  template <class Expr,
            class Sfinae = typename std::enable_if<details::is_field_expr_v2_linear_arg<Expr>::value>::type>
  explicit field_expr_v2_nonlinear_terminal_field_div (const Expr& expr, const grad_option_type& opt = grad_option_type()) 
    : base(new_macro(rep(expr,opt))) {} 

// accessors:

  space_constant::valued_type valued_tag() const { return base::data().valued_tag(); }

  template<class Result>
  bool evaluate (const geo_element& K, std::vector<Result>& value) const {
  	return base::data().evaluate(K,value); }

  bool initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const {
  	return base::data().initialize(omega,hat_x);
  }
  bool initialize (const space_basic<float_type,memory_type>& Xh) const {
  	return base::data().initialize(Xh);
  }
  template<class ValueType>
  void evaluate_on_side (const geo_element& L, const side_information_type& sid, std::vector<ValueType>& value) const {
  	base::data().evaluate_on_side (L, sid, value);
  }
  template<class ValueType>
  bool valued_check() const {
  	return base::data().template valued_check<ValueType>();
  }
};
template<class T, class M> struct is_field_expr_v2_nonlinear_arg     <field_expr_v2_nonlinear_terminal_field_div<T,M> > : std::true_type {};

} // namespace details

// div(uh)
template<class Expr>
inline
typename
std::enable_if<
  details::is_field_expr_v2_linear_arg<Expr>::value
 ,details::field_expr_v2_nonlinear_terminal_field_div<
    typename Expr::scalar_type
   ,typename Expr::memory_type
  >
>::type
div (const Expr& expr)
{
  typedef typename Expr::scalar_type T;
  typedef typename Expr::memory_type M;
  return details::field_expr_v2_nonlinear_terminal_field_div<T,M> (expr);
}
// div_s(uh)
template<class Expr>
inline
typename
std::enable_if<
  details::is_field_expr_v2_linear_arg<Expr>::value
 ,details::field_expr_v2_nonlinear_terminal_field_div<
    typename Expr::scalar_type
   ,typename Expr::memory_type
  >
>::type
div_s (const Expr& expr)
{
  typedef typename Expr::scalar_type T;
  typedef typename Expr::memory_type M;
  static details::grad_option_type   opt;
  opt.surfacic = true;
  return details::field_expr_v2_nonlinear_terminal_field_div<T,M> (expr, opt);
}
// div_h(uh)
template<class Expr>
inline
typename
std::enable_if<
  details::is_field_expr_v2_linear_arg<Expr>::value
 ,details::field_expr_v2_nonlinear_terminal_field_div<
    typename Expr::scalar_type
   ,typename Expr::memory_type
  >
>::type
div_h (const Expr& expr)
{
  typedef typename Expr::scalar_type T;
  typedef typename Expr::memory_type M;
  static details::grad_option_type   opt;
  opt.broken = true;
  return details::field_expr_v2_nonlinear_terminal_field_div<T,M> (expr, opt);
}
// ---------------------------------------------------------------------------
// 2.5) curl of a field
// ---------------------------------------------------------------------------
namespace details {

template<class T, class M>
class field_expr_v2_nonlinear_terminal_field_curl_rep : public field_expr_v2_nonlinear_terminal_field_base_rep<T,M> {
public:
// typedefs:

  typedef field_expr_v2_nonlinear_terminal_field_base_rep<T,M> base;
  typedef geo_element::size_type          size_type;
  typedef M                               memory_type;
  typedef undeterminated_basic<T>         result_type;
  typedef result_type                     value_type;
  typedef T                               scalar_type;
  typedef typename float_traits<T>::type  float_type;

// alocators:

  template <class Expr,
            class Sfinae = typename std::enable_if<details::is_field_expr_v2_linear_arg<Expr>::value>::type>
  explicit field_expr_v2_nonlinear_terminal_field_curl_rep (const Expr& expr, const grad_option_type& opt) 
   : base(expr),
     _opt(opt), 
     _u_test(base::_uh.get_space())
    {}

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::last_valued;

  space_constant::valued_type valued_tag() const;

  bool initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& quad) const;
  bool initialize (const space_basic<float_type,memory_type>& Xh) const;
  template<class ValueType>
  bool evaluate (const geo_element& K, std::vector<ValueType>& value) const;
  template<class ValueType>
  bool valued_check() const { 
	_u_test.template curl_valued_check<ValueType>();
	return true;
  }
// data:
  grad_option_type          _opt;
  test_basic<T,M,vf_tag_01> _u_test;
};

template<class T, class M>
class field_expr_v2_nonlinear_terminal_field_curl : public smart_pointer<field_expr_v2_nonlinear_terminal_field_curl_rep<T,M> >
{
public:
// typedefs:

  typedef field_expr_v2_nonlinear_terminal_field_curl_rep<T,M>    rep;
  typedef smart_pointer<rep>                    base;
  typedef typename rep::size_type               size_type;
  typedef typename rep::memory_type             memory_type;
  typedef typename rep::result_type             result_type;
  typedef typename rep::float_type              float_type;
  typedef typename rep::scalar_type             scalar_type;
  typedef typename rep::value_type              value_type;
  static const space_constant::valued_type valued_hint = rep::valued_hint;

// alocators:

  template <class Expr,
            class Sfinae = typename std::enable_if<details::is_field_expr_v2_linear_arg<Expr>::value>::type>
  explicit field_expr_v2_nonlinear_terminal_field_curl (const Expr& expr, const grad_option_type& opt = grad_option_type()) 
    : base(new_macro(rep(expr,opt))) {} 

// accessors:

  space_constant::valued_type valued_tag() const { return base::data().valued_tag(); }

  template<class Result>
  bool evaluate (const geo_element& K, std::vector<Result>& value) const {
  	return base::data().evaluate(K,value); }

  bool initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const {
  	return base::data().initialize(omega,hat_x);
  }
  bool initialize (const space_basic<float_type,memory_type>& Xh) const {
  	return base::data().initialize(Xh);
  }
  template<class ValueType>
  bool valued_check() const {
  	return base::data().template valued_check<ValueType>();
  }
};
template<class T, class M> struct is_field_expr_v2_nonlinear_arg     <field_expr_v2_nonlinear_terminal_field_curl<T,M> > : std::true_type {};

} // namespace details

// curl(uh)
template<class Expr>
inline
typename
std::enable_if<
  details::is_field_expr_v2_linear_arg<Expr>::value
 ,details::field_expr_v2_nonlinear_terminal_field_curl<
    typename Expr::scalar_type
   ,typename Expr::memory_type
  >
>::type
curl (const Expr& expr)
{
  typedef typename Expr::scalar_type T;
  typedef typename Expr::memory_type M;
  return details::field_expr_v2_nonlinear_terminal_field_curl<T,M> (expr);
}
// TODO: bcurl(uh) = Batchelor curl ?
// TODO: curl_s(uh) curl_h(uh)...?

// ----------------------------------------------------------------------------
// 2.6) jump of a field
// ----------------------------------------------------------------------------
namespace details {

template<class T, class M>
class field_expr_v2_nonlinear_terminal_field_dg_rep {
public:
// typedefs:

  typedef geo_element::size_type          size_type;
  typedef M                               memory_type;
  typedef undeterminated_basic<T>         result_type;
  typedef result_type                     value_type;
  typedef T                               scalar_type;
  typedef typename float_traits<T>::type  float_type;

// alocators:

  template <class Expr,
            class Sfinae = typename std::enable_if<details::is_field_expr_v2_linear_arg<Expr>::value>::type>
  explicit field_expr_v2_nonlinear_terminal_field_dg_rep(const Expr& expr, const float_type& c0, const float_type& c1)
   : _expr0(expr), _expr1(expr),
     _c0(c0), _c1(c1)
   {}

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::last_valued;

  space_constant::valued_type valued_tag() const { return _expr0.valued_tag(); }

  bool initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& quad) const;
  bool initialize (const space_basic<float_type,memory_type>& Xh) const;
  template<class ValueType>
  bool evaluate (const geo_element& K, std::vector<ValueType>& value) const;
  template<class ValueType>
  bool valued_check() const {
    space_constant::valued_type valued_tag = space_constant::valued_tag_traits<ValueType>::value;
    bool status = _expr0.valued_tag() == valued_tag;
    check_macro (status,
 	              "unexpected "<<  space_constant::valued_name(_expr0.valued_tag())
        << "-valued field while a " << space_constant::valued_name(valued_tag)
        << "-valued one is expected in expression");
    return status;
  }
// data:
protected:
  field_expr_v2_nonlinear_terminal_field<T,M> _expr0, _expr1;
  float_type                     _c0,    _c1;
};

template<class T, class M>
class field_expr_v2_nonlinear_terminal_field_dg : public smart_pointer<field_expr_v2_nonlinear_terminal_field_dg_rep<T,M> >
{
public:
// typedefs:

  typedef field_expr_v2_nonlinear_terminal_field_dg_rep<T,M> rep;
  typedef smart_pointer<rep>                    base;
  typedef typename rep::size_type               size_type;
  typedef typename rep::memory_type             memory_type;
  typedef typename rep::result_type             result_type;
  typedef typename rep::float_type              float_type;
  typedef typename rep::scalar_type             scalar_type;
  typedef typename rep::value_type              value_type;
  static const space_constant::valued_type valued_hint = rep::valued_hint;

// alocators:

  template <class Expr,
            class Sfinae = typename std::enable_if<details::is_field_expr_v2_linear_arg<Expr>::value>::type>
  explicit field_expr_v2_nonlinear_terminal_field_dg(const Expr& expr, const float_type& c0, const float_type& c1)
    : base(new_macro(rep(expr,c0,c1))) {} 

// accessors:

  space_constant::valued_type valued_tag() const { return base::data().valued_tag(); }

  bool initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const {
  	return base::data().initialize(omega,hat_x);
  }
  bool initialize (const space_basic<float_type,memory_type>& Xh) const {
  	return base::data().initialize(Xh);
  }
  template<class ValueType>
  bool valued_check() const {
  	return base::data().template valued_check<ValueType>();
  }
  template<class Result>
  bool evaluate (const geo_element& K, std::vector<Result>& value) const {
  	return base::data().evaluate(K,value);
  } 
};
template<class T, class M> struct is_field_expr_v2_nonlinear_arg     <field_expr_v2_nonlinear_terminal_field_dg<T,M> > : std::true_type {};

} // namespace details

#define _RHEOLEF_make_field_expr_v2_nonlinear_terminal_field_dg(op,c0,c1)			\
template<class Expr>					\
inline							\
typename						\
std::enable_if<						\
  details::is_field_expr_v2_linear_arg<Expr>::value	\
 ,details::field_expr_v2_nonlinear_terminal_field_dg<	\
    typename Expr::scalar_type				\
   ,typename Expr::memory_type				\
  >							\
>::type							\
op (const Expr& expr)					\
{							\
  return details::field_expr_v2_nonlinear_terminal_field_dg		\
	  <typename Expr::scalar_type ,typename Expr::memory_type>	\
	  (expr, c0, c1); 						\
}

_RHEOLEF_make_field_expr_v2_nonlinear_terminal_field_dg (jump,    1,    -1)
_RHEOLEF_make_field_expr_v2_nonlinear_terminal_field_dg (average, 0.5,  0.5)
_RHEOLEF_make_field_expr_v2_nonlinear_terminal_field_dg (inner,   1,    0)
_RHEOLEF_make_field_expr_v2_nonlinear_terminal_field_dg (outer,   0,    1)
#undef _RHEOLEF_make_field_expr_v2_nonlinear_terminal_field_dg 


// ---------------------------------------------------------------------------
// 3) convected field, as compose(uh,X) where X is a characteristic
// ---------------------------------------------------------------------------
namespace details {

template<class T, class M>
class field_expr_v2_nonlinear_terminal_field_o_characteristic_rep {
public:
// typedefs:

  typedef geo_element::size_type          size_type;
  typedef M                               memory_type;
  typedef undeterminated_basic<T>         result_type;
  typedef result_type                     value_type;
  typedef T                               scalar_type;
  typedef typename float_traits<T>::type  float_type;

// alocators:

  field_expr_v2_nonlinear_terminal_field_o_characteristic_rep (const field_o_characteristic<T,M>& uoX);
  field_expr_v2_nonlinear_terminal_field_o_characteristic_rep (const field_basic<T,M>& uh, const characteristic_basic<T,M>& X);

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::last_valued;

  space_constant::valued_type valued_tag() const { return _uh.valued_tag(); }

  bool initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const;
  bool initialize (const space_basic<float_type,memory_type>& Xh) const;

  template<class Result>
  bool evaluate (const geo_element& K, std::vector<Result>& value) const;

  template<class ValueType>
  bool valued_check() const {
    space_constant::valued_type valued_tag = space_constant::valued_tag_traits<ValueType>::value;
    bool status = (_uh.valued_tag() == valued_tag);
    check_macro (status, 
        "unexpected "<<_uh.valued()
        << "-valued field while a " << space_constant::valued_name(valued_tag)
        << "-valued one is expected in expression");
    return status;
  }

// internal:
  template<class Result>
  void _check () const;

// data:
  field_basic<T,M>                               _uh;
  characteristic_basic<T,M>                      _X;
  mutable bool                                   _is_initialized;
  mutable quadrature<T>                          _quad;
  mutable basis_on_pointset<T>                   _bops;
  mutable disarray<T,M>                          _uq;
  mutable typename disarray<T,M>::const_iterator _uq_K;
  mutable size_type                              _start_q;
};

template<class T, class M>
class field_expr_v2_nonlinear_terminal_field_o_characteristic : 
  public smart_pointer<field_expr_v2_nonlinear_terminal_field_o_characteristic_rep<T,M> >
{
public:
// typedefs:

  typedef field_expr_v2_nonlinear_terminal_field_o_characteristic_rep<T,M>    rep;
  typedef smart_pointer<rep>                    base;
  typedef typename rep::size_type               size_type;
  typedef typename rep::memory_type             memory_type;
  typedef typename rep::result_type             result_type;
  typedef typename rep::float_type              float_type;
  typedef typename rep::scalar_type             scalar_type;
  typedef typename rep::value_type              value_type;
  static const space_constant::valued_type valued_hint = rep::valued_hint;

// alocators:

  field_expr_v2_nonlinear_terminal_field_o_characteristic (const field_o_characteristic<T,M>& uoX)
    : base(new_macro(rep(uoX))) {} 

  field_expr_v2_nonlinear_terminal_field_o_characteristic (const field_basic<T,M>& uh, const characteristic_basic<T,M>& X)
    : base(new_macro(rep(uh,X))) {} 

// accessors:

  space_constant::valued_type valued_tag() const { return base::data().valued_tag(); }

  bool initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const {
  	return base::data().initialize(omega,hat_x);
  }
  bool initialize (const space_basic<float_type,memory_type>& Xh) const {
  	return base::data().initialize(Xh);
  }
  template<class Result>
  bool evaluate (const geo_element& K, std::vector<Result>& value) const {
  	return base::data().evaluate(K,value); }

  template<class ValueType>
  bool valued_check() const {
  	return base::data().template valued_check<ValueType>();
  }
};
template<class T, class M> struct is_field_expr_v2_nonlinear_arg     <field_expr_v2_nonlinear_terminal_field_o_characteristic<T,M> > : std::true_type {};

} // namespace details

// compose a field with a characteristic
template<class T, class M>
inline
details::field_expr_v2_nonlinear_terminal_field_o_characteristic<T,M>
compose (const field_basic<T,M>& uh, const characteristic_basic<T,M>& X)
{
  return details::field_expr_v2_nonlinear_terminal_field_o_characteristic<T,M> (uh,X);
}

} // namespace rheolef
#endif // _RHEOLEF_FIELD_EXPR_V2_NONLINEAR_TERMINAL_H
