# ifndef _RHEO_FORM_DIAG_H
# define _RHEO_FORM_DIAG_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

/*Class:form_diag
NAME:  @code{form_diag} - diagional form class 
DESCRIPTION:       
  Implement a form using two diagonal matrix.
EXAMPLE:
  Here is the computation of the mass matrix for
  the P1 finite element:
  @example
     geo m("square");
     space V(m,"P1");
     form_diag m(V,"mass");
  @end example
NOTE:
SEE ALSO: class "space", class "basic_diag"
AUTHOR: Pierre.Saramito@imag.fr
DATE:   15 july 1997
METHODS: @form_diag
End:
*/

#include "rheolef/skit.h"
#include "rheolef/space.h"
#include "rheolef/diag.h"
namespace rheolef { 

//<form_diag:  
class form_diag {
public:
// typedefs:

    typedef basic_diag<Float>::size_type size_type;

// allocator/deallocator:
        
    form_diag ();
    form_diag (const space& X);
    form_diag (const space& X, const char* op_name);
    form_diag (const class field& dh);

// accessors:

    const space& get_space() const;
    const geo&   get_geo() const;
    size_type size() const;
    size_type nrow() const;
    size_type ncol() const;

// linear algebra (partial):

    friend form_diag operator * (const form_diag&, const form_diag&);
    friend class field operator * (const form_diag& a, const class field& x);
    friend form_diag operator / (const Float& lambda, const form_diag& m);
    //new
    friend form_diag operator * (const Float& lambda, const form_diag&);
    friend form_diag operator + (const form_diag&, const form_diag&);
    friend form_diag operator - (const form_diag&);
    friend form_diag operator - (const form_diag&, const form_diag&);
    //wen

// input/output:

    friend std::ostream& operator << (std::ostream& s, const form_diag& a);

// data
private :
    space       X_;
public :
    basic_diag<Float> uu;
    basic_diag<Float> bb;
};
//>form_diag:

// ------------ inline'd -----------------------------------

inline
form_diag::form_diag ()
: X_(), uu(), bb()
{
}
inline 
form_diag::form_diag (const space& X)
: X_(X), uu(), bb()
{
    X_.freeze();
}
inline
const space& 
form_diag::get_space() const
{
  return X_;
}
inline
const geo&
form_diag::get_geo() const 
{
    return X_.get_geo();
}
inline
form_diag::size_type
form_diag::size() const 
{
    return get_space().size();
}
inline
form_diag::size_type
form_diag::nrow() const 
{
    return size();
}
inline
form_diag::size_type
form_diag::ncol() const 
{
    return size();
}
}// namespace rheolef
# endif /* _RHEO_FORM_DIAG_H */
