#ifndef _RHEO_QUADRATURE_OPTION_TYPE_H
#define _RHEO_QUADRATURE_OPTION_TYPE_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/smart_pointer.h"
#include "rheolef/reference_element.h"
#include "rheolef/point.h"

namespace rheolef { 

/*Class:quadrature
NAME: @code{quadrature_option_type} - send options to the integrate function
@cindex  quadrature formulae
@clindex quadrature
@clindex reference_element
SYNOPSYS:
@noindent
The @code{quadrature_option_type} class is used to send options to
the @code{integrate} function @pxref{integrate algorithm}.
This class allows to set the family (Gauss, Gauss-Lobatto, etc)
and the polynomial degree that is exactly integrated.
See also the @pxref{quadrature iclass} for more on quadrature formulae.
End:
*/

//<doc:
class quadrature_option_type {
public:
// typedefs:

  typedef size_t size_type;

  typedef enum {
	gauss		= 0,
	gauss_lobatto	= 1,
	gauss_radau	= 2,
	middle_edge	= 3,
	superconvergent	= 4,
	max_family	= 5
  } family_type; // update also family_name[] in quatrature.cc

// allocators:

  quadrature_option_type(
        family_type ft = quadrature_option_type::gauss,
        size_type k = std::numeric_limits<size_type>::max());

  quadrature_option_type (const quadrature_option_type& qopt);
  quadrature_option_type& operator= (const quadrature_option_type& qopt);

// accessors & modifiers:

  size_t         get_order() const;
  family_type    get_family() const;
  std::string    get_family_name() const;
  void set_order (size_t r);
  void set_family (family_type ft);
// data:
protected:
  family_type   _family;
  size_t        _order;
};
//>doc:
// ------------------------------------------------------------
// inlined
// ------------------------------------------------------------
inline 
quadrature_option_type::quadrature_option_type (family_type ft, size_type k)
 : _family(ft),
   _order(k)
{
}
inline
quadrature_option_type::quadrature_option_type (const quadrature_option_type& qopt)
 : _family(qopt._family),
   _order(qopt._order)
{
}
inline
quadrature_option_type&
quadrature_option_type::operator= (const quadrature_option_type& qopt)
{
  _family = qopt._family;
  _order  = qopt._order;
  return *this;
}
inline
quadrature_option_type::size_type
quadrature_option_type::get_order () const
{
    return _order;
}
inline
quadrature_option_type::family_type
quadrature_option_type::get_family () const
{
    return _family;
}
inline
void
quadrature_option_type::set_order (size_t r)
{
    _order = r;
}
inline
void
quadrature_option_type::set_family (family_type ft)
{
    _family = ft;
}

}// namespace rheolef
#endif // _RHEO_QUADRATURE_OPTION_TYPE_H
