% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/irace.R
\name{irace}
\alias{irace}
\title{Execute one run of the Iterated Racing algorithm.}
\usage{
irace(scenario)
}
\arguments{
\item{scenario}{\code{list()}\cr Data structure containing \pkg{irace}
settings. The data structure has to be the one returned by the function
\code{\link[=defaultScenario]{defaultScenario()}} or \code{\link[=readScenario]{readScenario()}}.}
}
\value{
(\code{data.frame})

A data frame with the set of best algorithm configurations found by \pkg{irace}.
The data frame has the following columns:
\itemize{
\item \code{.ID.} : Internal id of the candidate configuration.
\item \verb{Parameter names} : One column per parameter name in \code{parameters}.
\item \code{.PARENT.} : Internal id of the parent candidate configuration.
}

Additionally, this function saves an R data file containing an object called
\code{iraceResults}. The path of the file is indicated in \code{scenario$logFile}.
The \code{iraceResults} object is a list with the following structure:

\describe{

\item{\code{scenario}}{The scenario R object containing the \pkg{irace}
options used for the execution. See \code{\link{defaultScenario}}
for more information. The element \code{scenario$parameters} contains the parameters R object that describes the target algorithm parameters. See
\code{\link{readParameters}}.}

\item{\code{allConfigurations}}{The target algorithm configurations
generated by \pkg{irace}. This object is a data frame, each row is a
candidate configuration, the first column (\code{.ID.}) indicates the
internal identifier of the configuration, the following columns
correspond to the parameter values, each column named as the parameter
name specified in the parameter object. The final column
(\code{.PARENT.})  is the identifier of the configuration from which
model the actual configuration was sampled.}

\item{\code{allElites}}{A list that contains one element per iteration,
each element contains the internal identifier of the elite candidate
configurations of the corresponding iteration (identifiers correspond to
\code{allConfigurations$.ID.}).}

\item{\code{iterationElites}}{A vector containing the best candidate
configuration internal identifier of each iteration. The best
configuration found corresponds to the last one of this vector.}

\item{\code{experiments}}{A matrix with configurations as columns and
instances as rows. Column names correspond to the internal identifier of
the configuration (\code{allConfigurations$.ID.}).}

\item{\code{experimen_log}}{A \code{data.table} with columns \code{iteration},
\code{instance}, \code{configuration}, \code{time}.  This matrix contains the log of all the
experiments that \pkg{irace} performs during its execution.  The
instance column refers to the index of the \code{race_state$instances_log}
data frame. Time is saved ONLY when reported by the \code{targetRunner}.}

\item{\code{softRestart}}{A logical vector that indicates if a soft
restart was performed on each iteration. If \code{FALSE}, then no soft
restart was performed.}

\item{\code{state}}{An environment that contains the state of \pkg{irace}, the
recovery is done using the information contained in this object.}

\item{\code{testing}}{A list that contains the testing results. The
elements of this list are: \code{experiments} a matrix with the testing
experiments of the selected configurations in the same format as the
explained above and \code{seeds} a vector with the seeds used to execute
each experiment.}

}
}
\description{
The function \code{irace} implements the Iterated Racing procedure for parameter
tuning. It receives a configuration scenario and a parameter space to be
tuned, and returns the best configurations found, namely, the elite
configurations obtained from the last iterations. As a first step, it checks
the correctness of \code{scenario} using \code{\link[=checkScenario]{checkScenario()}} and recovers a
previous execution if \code{scenario$recoveryFile} is set. A R data file log of
the execution is created in \code{scenario$logFile}.
}
\details{
The execution of this function is reproducible under some conditions. See
the FAQ section in the \href{https://cran.r-project.org/package=irace/vignettes/irace-package.pdf}{User Guide}.
}
\examples{
\dontrun{
# In general, there are three steps:
scenario <- readScenario(filename = "scenario.txt")
irace(scenario = scenario)
}
#######################################################################
# This example illustrates how to tune the parameters of the simulated
# annealing algorithm (SANN) provided by the optim() function in the
# R base package.  The goal in this example is to optimize instances of
# the following family:
#      f(x) = lambda * f_rastrigin(x) + (1 - lambda) * f_rosenbrock(x)
# where lambda follows a normal distribution whose mean is 0.9 and
# standard deviation is 0.02. f_rastrigin and f_rosenbrock are the
# well-known Rastrigin and Rosenbrock benchmark functions (taken from
# the cmaes package). In this scenario, different instances are given
# by different values of lambda.
#######################################################################
## First we provide an implementation of the functions to be optimized:
f_rosenbrock <- function (x) {
  d <- length(x)
  z <- x + 1
  hz <- z[1L:(d - 1L)]
  tz <- z[2L:d]
  sum(100 * (hz^2 - tz)^2 + (hz - 1)^2)
}
f_rastrigin <- function (x) {
  sum(x * x - 10 * cos(2 * pi * x) + 10)
}

## We generate 20 instances (in this case, weights):
weights <- rnorm(20, mean = 0.9, sd = 0.02)

## On this set of instances, we are interested in optimizing two
## parameters of the SANN algorithm: tmax and temp. We setup the
## parameter space as follows:
parameters_table <- '
  tmax "" i,log (1, 5000)
  temp "" r (0, 100)
  '
## We use the irace function readParameters to read this table:
parameters <- readParameters(text = parameters_table)

## Next, we define the function that will evaluate each candidate
## configuration on a single instance. For simplicity, we restrict to
## three-dimensional functions and we set the maximum number of
## iterations of SANN to 1000.
target_runner <- function(experiment, scenario)
{
    instance <- experiment$instance
    configuration <- experiment$configuration

    D <- 3
    par <- runif(D, min=-1, max=1)
    fn <- function(x) {
      weight <- instance
      return(weight * f_rastrigin(x) + (1 - weight) * f_rosenbrock(x))
    }
    # For reproducible results, we should use the random seed given by
    # experiment$seed to set the random seed of the target algorithm.
    res <- withr::with_seed(experiment$seed,
                     stats::optim(par,fn, method="SANN",
                                  control=list(maxit=1000
                                             , tmax = as.numeric(configuration[["tmax"]])
                                             , temp = as.numeric(configuration[["temp"]])
                                               )))
    ## This list may also contain:
    ## - 'time' if irace is called with 'maxTime'
    ## - 'error' is a string used to report an error
    ## - 'outputRaw' is a string used to report the raw output of calls to
    ##   an external program or function.
    ## - 'call' is a string used to report how target_runner called the
    ##   external program or function.
    return(list(cost = res$value))
}

## We define a configuration scenario by setting targetRunner to the
## function define above, instances to the first 10 random weights, and
## a maximum budget of 'maxExperiments' calls to targetRunner.
scenario <- list(targetRunner = target_runner,
                 instances = weights[1:10],
                 maxExperiments = 500,
                 # Do not create a logFile
                 logFile = "",
                 parameters = parameters)

## We check that the scenario is valid. This will also try to execute
## target_runner.
checkIraceScenario(scenario)

\donttest{
## We are now ready to launch irace. We do it by means of the irace
## function. The function will print information about its
## progress. This may require a few minutes, so it is not run by default.
tuned_confs <- irace(scenario = scenario)

## We can print the best configurations found by irace as follows:
configurations_print(tuned_confs)

## We can evaluate the quality of the best configuration found by
## irace versus the default configuration of the SANN algorithm on
## the other 10 instances previously generated.
test_index <- 11:20
test_seeds <- sample.int(2147483647L, size = length(test_index), replace = TRUE)
test <- function(configuration)
{
  res <- lapply(seq_along(test_index),
                function(x) target_runner(
                              experiment = list(instance = weights[test_index[x]],
                                                seed = test_seeds[x],
                                                configuration = configuration),
                              scenario = scenario))
  return (sapply(res, getElement, name = "cost"))
}
## To do so, first we apply the default configuration of the SANN
## algorithm to these instances:
default <- test(data.frame(tmax=10, temp=10))

## We extract and apply the winning configuration found by irace
## to these instances:
tuned <- test(removeConfigurationsMetaData(tuned_confs[1,]))

## Finally, we can compare using a boxplot the quality obtained with the
## default parametrization of SANN and the quality obtained with the
## best configuration found by irace.
boxplot(list(default = default, tuned = tuned))
}
}
\seealso{
\describe{
\item{\code{\link[=irace_main]{irace_main()}}}{a higher-level interface to \code{\link[=irace]{irace()}}.}
\item{\code{\link[=irace_cmdline]{irace_cmdline()}}}{a command-line interface to \code{\link[=irace]{irace()}}.}
\item{\code{\link[=readScenario]{readScenario()}}}{for reading a configuration scenario from a file.}
\item{\code{\link[=readParameters]{readParameters()}}}{read the target algorithm parameters from a file.}
\item{\code{\link[=defaultScenario]{defaultScenario()}}}{returns the default scenario settings of \pkg{irace}.}
\item{\code{\link[=checkScenario]{checkScenario()}}}{to check that the scenario is valid.}
}
}
\author{
Manuel López-Ibáñez and Jérémie Dubois-Lacoste
}
\concept{running}
