% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/req-body.R
\name{req_body}
\alias{req_body_raw}
\alias{req_body_file}
\alias{req_body_json}
\alias{req_body_json_modify}
\alias{req_body_form}
\alias{req_body_multipart}
\title{Send data in request body}
\usage{
req_body_raw(req, body, type = NULL)

req_body_file(req, path, type = NULL)

req_body_json(
  req,
  data,
  auto_unbox = TRUE,
  digits = 22,
  null = "null",
  type = "application/json",
  ...
)

req_body_json_modify(req, ...)

req_body_form(.req, ..., .multi = c("error", "comma", "pipe", "explode"))

req_body_multipart(.req, ...)
}
\arguments{
\item{req, .req}{A \link{request}.}

\item{body}{A literal string or raw vector to send as body.}

\item{type}{MIME content type. You shouldn't generally need to specify this as
the defaults are usually pretty good, e.g. \code{req_body_file()} will guess it
from the extension of of \code{path}. Will be ignored if you have manually set
a \code{Content-Type} header.}

\item{path}{Path to file to upload.}

\item{data}{Data to include in body.}

\item{auto_unbox}{Should length-1 vectors be automatically "unboxed" to
JSON scalars?}

\item{digits}{How many digits of precision should numbers use in JSON?}

\item{null}{Should \code{NULL} be translated to JSON's null (\code{"null"})
or an empty list (\code{"list"}).}

\item{...}{<\code{\link[rlang:dyn-dots]{dynamic-dots}}> Name-data pairs used to send
data in the body.
\itemize{
\item For \code{req_body_form()}, the values must be strings (or things easily
coerced to strings);
\item For \code{req_body_multipart()} the values must be strings or objects
produced by \code{\link[curl:multipart]{curl::form_file()}}/\code{\link[curl:multipart]{curl::form_data()}}.
\item For \code{req_body_json_modify()}, any simple data made from atomic vectors
and lists.
}

\code{req_body_json()} uses this argument differently; it takes additional
arguments passed on to  \code{\link[jsonlite:fromJSON]{jsonlite::toJSON()}}.}

\item{.multi}{Controls what happens when an element of \code{...} is a vector
containing multiple values:
\itemize{
\item \code{"error"}, the default, throws an error.
\item \code{"comma"}, separates values with a \verb{,}, e.g. \verb{?x=1,2}.
\item \code{"pipe"}, separates values with a \code{|}, e.g. \code{?x=1|2}.
\item \code{"explode"}, turns each element into its own parameter, e.g. \code{?x=1&x=2}.
}

If none of these functions work, you can alternatively supply a function
that takes a character vector and returns a string.}
}
\value{
A modified HTTP \link{request}.
}
\description{
\itemize{
\item \code{req_body_file()} sends a local file.
\item \code{req_body_raw()} sends a string or raw vector.
\item \code{req_body_json()} sends JSON encoded data. Named components of this data
can later be modified with \code{req_body_json_modify()}.
\item \code{req_body_form()} sends form encoded data.
\item \code{req_body_multipart()} creates a multi-part body.
}

Adding a body to a request will automatically switch the method to POST.
}
\examples{
req <- request(example_url()) |>
  req_url_path("/post")

# Most APIs expect small amounts of data in either form or json encoded:
req |>
  req_body_form(x = "A simple text string") |>
  req_dry_run()

req |>
  req_body_json(list(x = "A simple text string")) |>
  req_dry_run()

# For total control over the body, send a string or raw vector
req |>
  req_body_raw("A simple text string") |>
  req_dry_run()

# There are two main ways that APIs expect entire files
path <- tempfile()
writeLines(letters[1:6], path)

# You can send a single file as the body:
req |>
  req_body_file(path) |>
  req_dry_run()

# You can send multiple files, or a mix of files and data
# with multipart encoding
req |>
  req_body_multipart(a = curl::form_file(path), b = "some data") |>
  req_dry_run()
}
