% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayesfactor_parameters.R
\name{bayesfactor_parameters}
\alias{bayesfactor_parameters}
\alias{bayesfactor_pointnull}
\alias{bayesfactor_rope}
\alias{bf_parameters}
\alias{bf_pointnull}
\alias{bf_rope}
\alias{bayesfactor_parameters.numeric}
\alias{bayesfactor_parameters.stanreg}
\alias{bayesfactor_parameters.brmsfit}
\alias{bayesfactor_parameters.blavaan}
\alias{bayesfactor_parameters.data.frame}
\title{Bayes Factors (BF) for a Single Parameter}
\usage{
bayesfactor_parameters(
  posterior,
  prior = NULL,
  direction = "two-sided",
  null = 0,
  verbose = TRUE,
  ...
)

bayesfactor_pointnull(
  posterior,
  prior = NULL,
  direction = "two-sided",
  null = 0,
  verbose = TRUE,
  ...
)

bayesfactor_rope(
  posterior,
  prior = NULL,
  direction = "two-sided",
  null = rope_range(posterior),
  verbose = TRUE,
  ...
)

bf_parameters(
  posterior,
  prior = NULL,
  direction = "two-sided",
  null = 0,
  verbose = TRUE,
  ...
)

bf_pointnull(
  posterior,
  prior = NULL,
  direction = "two-sided",
  null = 0,
  verbose = TRUE,
  ...
)

bf_rope(
  posterior,
  prior = NULL,
  direction = "two-sided",
  null = rope_range(posterior),
  verbose = TRUE,
  ...
)

\method{bayesfactor_parameters}{numeric}(
  posterior,
  prior = NULL,
  direction = "two-sided",
  null = 0,
  verbose = TRUE,
  ...
)

\method{bayesfactor_parameters}{stanreg}(
  posterior,
  prior = NULL,
  direction = "two-sided",
  null = 0,
  verbose = TRUE,
  effects = c("fixed", "random", "all"),
  component = c("conditional", "location", "smooth_terms", "sigma", "zi",
    "zero_inflated", "all"),
  parameters = NULL,
  ...
)

\method{bayesfactor_parameters}{brmsfit}(
  posterior,
  prior = NULL,
  direction = "two-sided",
  null = 0,
  verbose = TRUE,
  effects = c("fixed", "random", "all"),
  component = c("conditional", "location", "smooth_terms", "sigma", "zi",
    "zero_inflated", "all"),
  parameters = NULL,
  ...
)

\method{bayesfactor_parameters}{blavaan}(
  posterior,
  prior = NULL,
  direction = "two-sided",
  null = 0,
  verbose = TRUE,
  ...
)

\method{bayesfactor_parameters}{data.frame}(
  posterior,
  prior = NULL,
  direction = "two-sided",
  null = 0,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{posterior}{A numerical vector, \code{stanreg} / \code{brmsfit} object,
\code{emmGrid} or a data frame - representing a posterior distribution(s)
from (see 'Details').}

\item{prior}{An object representing a prior distribution (see 'Details').}

\item{direction}{Test type (see 'Details'). One of \code{0},
\code{"two-sided"} (default, two tailed), \code{-1}, \code{"left"} (left
tailed) or \code{1}, \code{"right"} (right tailed).}

\item{null}{Value of the null, either a scalar (for point-null) or a range
(for a interval-null).}

\item{verbose}{Toggle off warnings.}

\item{...}{Arguments passed to and from other methods. (Can be used to pass
arguments to internal \code{\link[logspline:logspline]{logspline::logspline()}}.)}

\item{effects}{Should results for fixed effects, random effects or both be
returned? Only applies to mixed models. May be abbreviated.}

\item{component}{Should results for all parameters, parameters for the
conditional model or the zero-inflated part of the model be returned? May
be abbreviated. Only applies to \pkg{brms}-models.}

\item{parameters}{Regular expression pattern that describes the parameters
that should be returned. Meta-parameters (like \code{lp__} or \code{prior_}) are
filtered by default, so only parameters that typically appear in the
\code{summary()} are returned. Use \code{parameters} to select specific parameters
for the output.}
}
\value{
A data frame containing the (log) Bayes factor representing evidence
\emph{against} the null.
}
\description{
This method computes Bayes factors against the null (either a point or an
interval), based on prior and posterior samples of a single parameter. This
Bayes factor indicates the degree by which the mass of the posterior
distribution has shifted further away from or closer to the null value(s)
(relative to the prior distribution), thus indicating if the null value has
become less or more likely given the observed data.
\cr \cr
When the null is an interval, the Bayes factor is computed by comparing the
prior and posterior odds of the parameter falling within or outside the null
interval (Morey & Rouder, 2011; Liao et al., 2020); When the null is a point,
a Savage-Dickey density ratio is computed, which is also an approximation of
a Bayes factor comparing the marginal likelihoods of the model against a
model in which the tested parameter has been restricted to the point null
(Wagenmakers et al., 2010; Heck, 2019).
\cr \cr
Note that the \code{logspline} package is used for estimating densities and
probabilities, and must be installed for the function to work.
\cr \cr
\code{bayesfactor_pointnull()} and \code{bayesfactor_rope()} are wrappers
around \code{bayesfactor_parameters} with different defaults for the null to
be tested against (a point and a range, respectively). Aliases of the main
functions are prefixed with \verb{bf_*}, like \code{bf_parameters()} or
\code{bf_pointnull()}.
\cr \cr
\strong{For more info, in particular on specifying correct priors for factors
with more than 2 levels, see
\href{https://easystats.github.io/bayestestR/articles/bayes_factors.html}{the Bayes factors vignette}.}
}
\details{
This method is used to compute Bayes factors based on prior and posterior
distributions.

\subsection{One-sided & Dividing Tests (setting an order restriction)}{
One sided tests (controlled by \code{direction}) are conducted by restricting
the prior and posterior of the non-null values (the "alternative") to one
side of the null only (\cite{Morey & Wagenmakers, 2014}). For example, if we
have a prior hypothesis that the parameter should be positive, the
alternative will be restricted to the region to the right of the null (point
or interval). For example, for a Bayes factor comparing the "null" of \code{0-0.1}
to the alternative \verb{>0.1}, we would set
\code{bayesfactor_parameters(null = c(0, 0.1), direction = ">")}.
\cr\cr
It is also possible to compute a Bayes factor for \strong{dividing}
hypotheses - that is, for a null and alternative that are complementary,
opposing one-sided hypotheses (\cite{Morey & Wagenmakers, 2014}). For
example, for a Bayes factor comparing the "null" of \verb{<0} to the alternative
\verb{>0}, we would set \code{bayesfactor_parameters(null = c(-Inf, 0))}.
}
}
\note{
There is also a
\href{https://easystats.github.io/see/articles/bayestestR.html}{\code{plot()}-method}
implemented in the
\href{https://easystats.github.io/see/}{\pkg{see}-package}.
}
\section{Setting the correct \code{prior}}{

For the computation of Bayes factors, the model priors must be proper priors
(at the very least they should be \emph{not flat}, and it is preferable that
they be \emph{informative}); As the priors for the alternative get wider, the
likelihood of the null value(s) increases, to the extreme that for completely
flat priors the null is infinitely more favorable than the alternative (this
is called \emph{the Jeffreys-Lindley-Bartlett paradox}). Thus, you should
only ever try (or want) to compute a Bayes factor when you have an informed
prior.
\cr\cr
(Note that by default, \code{brms::brm()} uses flat priors for fixed-effects;
See example below.)
\cr\cr
It is important to provide the correct \code{prior} for meaningful results.
\itemize{
\item When \code{posterior} is a numerical vector, \code{prior} should also be a numerical vector.
\item When \code{posterior} is a \code{data.frame}, \code{prior} should also be a \code{data.frame}, with matching column order.
\item When \code{posterior} is a \code{stanreg} or \code{brmsfit} model: \itemize{
\item \code{prior} can be set to \code{NULL}, in which case prior samples are drawn internally.
\item \code{prior} can also be a model equivalent to \code{posterior} but with samples from the priors \emph{only}. See \code{\link[=unupdate]{unupdate()}}.
\item \strong{Note:} When \code{posterior} is a \code{brmsfit_multiple} model, \code{prior} \strong{must} be provided.
}
\item When \code{posterior} is an \code{emmGrid} object: \itemize{
\item \code{prior} should be the \code{stanreg} or \code{brmsfit} model used to create the \code{emmGrid} objects.
\item \code{prior} can also be an \code{emmGrid} object equivalent to \code{posterior} but created with a model of priors samples \emph{only}.
\item \strong{Note:} When the \code{emmGrid} has undergone any transformations (\code{"log"}, \code{"response"}, etc.), or \code{regrid}ing, then \code{prior} must be an \code{emmGrid} object, as stated above.
}
}
}

\section{Interpreting Bayes Factors}{

A Bayes factor greater than 1 can be interpreted as evidence against the
null, at which one convention is that a Bayes factor greater than 3 can be
considered as "substantial" evidence against the null (and vice versa, a
Bayes factor smaller than 1/3 indicates substantial evidence in favor of the
null-model) (\cite{Wetzels et al. 2011}).
}

\examples{
library(bayestestR)
if (require("logspline")) {
  prior <- distribution_normal(1000, mean = 0, sd = 1)
  posterior <- distribution_normal(1000, mean = .5, sd = .3)
  bayesfactor_parameters(posterior, prior)
}
\dontrun{
# rstanarm models
# ---------------
if (require("rstanarm") && require("emmeans") && require("logspline")) {
  contrasts(sleep$group) <- contr.orthonorm # see vingette
  stan_model <- stan_lmer(extra ~ group + (1 | ID), data = sleep)
  bayesfactor_parameters(stan_model)
  bayesfactor_parameters(stan_model, null = rope_range(stan_model))

  # emmGrid objects
  # ---------------
  group_diff <- pairs(emmeans(stan_model, ~group))
  bayesfactor_parameters(group_diff, prior = stan_model)
}

# brms models
# -----------
if (require("brms")) {
  contrasts(sleep$group) <- contr.orthonorm # see vingette
  my_custom_priors <-
    set_prior("student_t(3, 0, 1)", class = "b") +
    set_prior("student_t(3, 0, 1)", class = "sd", group = "ID")

  brms_model <- brm(extra ~ group + (1 | ID),
    data = sleep,
    prior = my_custom_priors
  )
  bayesfactor_parameters(brms_model)
}
}
}
\references{
\itemize{
\item Wagenmakers, E. J., Lodewyckx, T., Kuriyal, H., and Grasman, R. (2010).
Bayesian hypothesis testing for psychologists: A tutorial on the
Savage-Dickey method. Cognitive psychology, 60(3), 158-189.
\item Heck, D. W. (2019). A caveat on the Savage–Dickey density ratio: The
case of computing Bayes factors for regression parameters. British Journal of
Mathematical and Statistical Psychology, 72(2), 316-333.
\item Morey, R. D., & Wagenmakers, E. J. (2014). Simple relation between
Bayesian order-restricted and point-null hypothesis tests. Statistics &
Probability Letters, 92, 121-124.
\item Morey, R. D., & Rouder, J. N. (2011). Bayes factor approaches for
testing interval null hypotheses. Psychological methods, 16(4), 406.
\item Liao, J. G., Midya, V., & Berg, A. (2020). Connecting and contrasting
the Bayes factor and a modified ROPE procedure for testing interval null
hypotheses. The American Statistician, 1-19.
\item Wetzels, R., Matzke, D., Lee, M. D., Rouder, J. N., Iverson, G. J., and
Wagenmakers, E.-J. (2011). Statistical Evidence in Experimental Psychology:
An Empirical Comparison Using 855 t Tests. Perspectives on Psychological
Science, 6(3), 291–298. \doi{10.1177/1745691611406923}
}
}
\author{
Mattan S. Ben-Shachar
}
