// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "net/third_party/quic/core/crypto/quic_encrypter.h"

#include "net/third_party/quic/core/crypto/aes_128_gcm_12_encrypter.h"
#include "net/third_party/quic/core/crypto/aes_128_gcm_encrypter.h"
#include "net/third_party/quic/core/crypto/aes_256_gcm_encrypter.h"
#include "net/third_party/quic/core/crypto/chacha20_poly1305_encrypter.h"
#include "net/third_party/quic/core/crypto/chacha20_poly1305_tls_encrypter.h"
#include "net/third_party/quic/core/crypto/crypto_protocol.h"
#include "net/third_party/quic/core/crypto/null_encrypter.h"
#include "net/third_party/quic/platform/api/quic_bug_tracker.h"
#include "net/third_party/quic/platform/api/quic_logging.h"
#include "net/third_party/quic/platform/api/quic_ptr_util.h"
#include "third_party/boringssl/src/include/openssl/tls1.h"

namespace quic {

// static
std::unique_ptr<QuicEncrypter> QuicEncrypter::Create(QuicTag algorithm) {
  switch (algorithm) {
    case kAESG:
      return QuicMakeUnique<Aes128Gcm12Encrypter>();
    case kCC20:
      return QuicMakeUnique<ChaCha20Poly1305Encrypter>();
    default:
      QUIC_LOG(FATAL) << "Unsupported algorithm: " << algorithm;
      return nullptr;
  }
}

// static
std::unique_ptr<QuicEncrypter> QuicEncrypter::CreateFromCipherSuite(
    uint32_t cipher_suite) {
  switch (cipher_suite) {
    case TLS1_CK_AES_128_GCM_SHA256:
      return QuicMakeUnique<Aes128GcmEncrypter>();
    case TLS1_CK_AES_256_GCM_SHA384:
      return QuicMakeUnique<Aes256GcmEncrypter>();
    case TLS1_CK_CHACHA20_POLY1305_SHA256:
      return QuicMakeUnique<ChaCha20Poly1305TlsEncrypter>();
    default:
      QUIC_BUG << "TLS cipher suite is unknown to QUIC";
      return nullptr;
  }
}

}  // namespace quic
