//
// ZipArchive.h
//
// $Id: //poco/1.3/Zip/include/Poco/Zip/ZipArchive.h#5 $
//
// Library: Zip
// Package: Zip
// Module:  ZipArchive
//
// Definition of the ZipArchive class.
//
// Copyright (c) 2007, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Zip_ZipArchive_INCLUDED
#define Zip_ZipArchive_INCLUDED


#include "Poco/Zip/Zip.h"
#include "Poco/Zip/ZipLocalFileHeader.h"
#include "Poco/Zip/ZipFileInfo.h"
#include "Poco/Zip/ZipArchiveInfo.h"
#include <istream>
#include <map>


namespace Poco {
namespace Zip {


class ParseCallback;
class Compress;


class Zip_API ZipArchive
	/// A ZipArchive contains information on the content of a zip file
{
public:
	typedef std::map<std::string, ZipLocalFileHeader> FileHeaders;
	typedef std::map<std::string, ZipFileInfo> FileInfos;
	typedef std::map<Poco::UInt16, ZipArchiveInfo> DirectoryInfos;

	ZipArchive(std::istream& in);
		/// Creates the ZipArchive from a file. Note that the in stream will be in state failed after the constructor is finished

	ZipArchive(std::istream& in, ParseCallback& callback);
		/// Creates the ZipArchive from a file or network stream. Note that the in stream will be in state failed after the constructor is finished

	~ZipArchive();
		/// Destroys the ZipArchive.

	FileInfos::const_iterator fileInfoBegin() const;

	FileInfos::const_iterator fileInfoEnd() const;

	FileHeaders::const_iterator findHeader(const std::string& fileName) const;

	FileHeaders::const_iterator headerBegin() const;

	FileHeaders::const_iterator headerEnd() const;

    const std::string& getZipComment() const;

private:
	void parse(std::istream& in, ParseCallback& pc);

	ZipArchive(const FileHeaders& entries, const FileInfos& infos, const DirectoryInfos& dirs	);

private:
	FileHeaders    _entries;
		/// Info generated by parsing the data block of the zip file
	FileInfos      _infos;
		/// Info generated by parsing the directory block of the zip file
	DirectoryInfos _disks;
		/// Stores directory info for all found disks

	friend class Compress;
};


inline ZipArchive::FileInfos::const_iterator ZipArchive::fileInfoBegin() const
{
	return _infos.begin();
}


inline ZipArchive::FileInfos::const_iterator ZipArchive::fileInfoEnd() const
{
	return _infos.end();
}


inline ZipArchive::FileHeaders::const_iterator ZipArchive::findHeader(const std::string& fileName) const
{
	return _entries.find(fileName);
}


inline ZipArchive::FileHeaders::const_iterator ZipArchive::headerBegin() const
{
	return _entries.begin();
}


inline ZipArchive::FileHeaders::const_iterator ZipArchive::headerEnd() const
{
	return _entries.end();
}


} } // namespace Poco::Zip


#endif // Zip_ZipArchive_INCLUDED
