/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment viewport opacity example. Must be run with the PGM_GL_ARGB_VISUAL
 * environment variable set to 1 and in a composited desktop with compiz,
 * xcompmgr, or other.
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

#include <pgm/pgm.h>
#include <math.h>

/* Transform the mapping matrix of the image */
static gboolean
transform_mapping_matrix_cb (gpointer data)
{
  PgmImage *image = PGM_IMAGE (data);
  PgmMat4x4 *mapping = NULL;
  static float factor = 0.0f;
  float scale;

  /* Get the scale factor using a sinusoidal animation */
  scale = (float) ((cos (factor) + 4.0) / 3.0);

  /* Compute the scaling matrix and set the image mapping matrix */
  mapping = pgm_mat4x4_new_translate_from_scalars (0.5f, 0.5f, 0.0f);
  pgm_mat4x4_scale_from_scalars (mapping, scale, scale, 0.0f);
  pgm_mat4x4_translate_from_scalars (mapping, -0.5f, -0.5f, 0.0f);
  pgm_image_set_mapping_matrix (PGM_IMAGE (image), mapping);
  pgm_mat4x4_free (mapping);

  /* Update the factor for the next animation */
  factor += 0.075f;

  return TRUE;
}

/* key-press handler */
static void
keypress_event_cb (PgmViewport *viewport,
                   PgmEventKey *event,
                   gpointer data)
{
  static gboolean fullscreen = FALSE;
  static guchar opacity = 0;

  if (event->type == PGM_KEY_PRESS)
    {
      switch (event->keyval)
        {
          /* Handle quit */
        case PGM_q:
        case PGM_Escape:
          pgm_main_quit ();
          break;

          /* Fullscreen */
        case PGM_f:
          fullscreen = !fullscreen;
          pgm_viewport_set_fullscreen (viewport, fullscreen);
          break;

          /* Increase viewport opacity */
        case PGM_z:
          opacity = MIN (255, opacity + 5);
          pgm_viewport_set_opacity (viewport, opacity);
          break;

          /* Decrease viewport opacity */
        case PGM_a:
          opacity = MAX (0, opacity - 5);
          pgm_viewport_set_opacity (viewport, opacity);
          break;

        default:
          break;
        }
    }
}

/* Entry point */
int
main (int argc,
      char *argv[])
{
  const gchar *img_path = "examples/pictures/star.png";
  PgmViewport *viewport = NULL;
  gulong caps_mask = 0;
  PgmCanvas *canvas;
  PgmDrawable *img;

  /* Init */
  pgm_init (&argc, &argv);

  /* OpenGL viewport creation */
  pgm_viewport_factory_make ("opengl", &viewport);
  if (!viewport)
    {
      g_print ("Cannot create the 'opengl' viewport\n");
      return -1;
    }
  pgm_viewport_set_title (viewport, "Clear color");
  pgm_viewport_set_decorated (viewport, FALSE);
  pgm_viewport_set_opacity (viewport, 0);
  pgm_viewport_set_size (viewport, 400, 400);

  /* Output a message if the clear color is not supported */
  pgm_viewport_get_caps_mask (viewport, &caps_mask);
  if (!(caps_mask & PGM_VIEWPORT_OPACITY))
    g_print ("WARNING: The viewport opacity modification is not supported. "
             "The content of your window will not be composited with "
             "the rest of the desktop. You can enable it setting the "
             "PGM_GL_ARGB_VISUAL environment variable to 1. Note that at the "
             "moment, this feature is only available on Linux with the NVIDIA "
             "proprietary driver.\n\n");

  /* Output key options */
  g_print ("'a' : Decrease viewport opacity\n"
           "'z' : Increase viewport opacity\n");

  /* Image definition */
  img = pgm_image_new_from_file (img_path, 0);
  pgm_drawable_set_size (img, 400, 400);
  pgm_drawable_set_position (img, 0, 0, 0);
  pgm_drawable_set_bg_color (img, 0, 0, 0, 0);
  pgm_drawable_set_fg_color (img, 255, 255, 255, 200);
  pgm_drawable_show (img);

  /* Canvas handling */
  canvas = pgm_canvas_new ();
  pgm_canvas_set_size (canvas, 400, 400);
  pgm_viewport_set_canvas (viewport, canvas);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img);

  /* Main loop */
  g_signal_connect (G_OBJECT (viewport), "key-press-event",
                    G_CALLBACK (keypress_event_cb), NULL);
  g_signal_connect (G_OBJECT (viewport), "delete-event",
                    G_CALLBACK (pgm_main_quit), NULL);
  pgm_viewport_show (viewport);

  /* Let's transform the mapping matrix */
  g_timeout_add (10, transform_mapping_matrix_cb, img);

  pgm_main ();

  /* Deinit */
  gst_object_unref (canvas);
  gst_object_unref (viewport);
  pgm_deinit ();

  return 0;
}
