/*
 * AddCardDialog.java
 *
 * Created on 11. Juni 2001, 23:03
 */
package pauker.program.gui.swing;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dialog;
import java.text.DateFormat;
import javax.swing.JDialog;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import pauker.program.Card;
import pauker.program.CardSide;
import pauker.program.SearchEngine;

/**
 * A dialog for editing Pauker flash cards
 * @author Ronny.Standtke@gmx.net
 */
public class EditCardDialog extends JDialog {

    private Card card;
    private boolean okButtonPressed;
    private SearchEngine searchEngine;

    /**
     * creates a new EditCardDialog
     * @param dialog the parent dialog
     * @param paukerFrame the pauker frame
     * @param card the card to edit
     * @param searchEngine the search engine for similar cards
     */
    public EditCardDialog(Dialog dialog, PaukerFrame paukerFrame,
            Card card, SearchEngine searchEngine) {
        super(dialog, true);
        initDialog(dialog, paukerFrame, card, searchEngine);
    }

    /**
     * creates a new EditCardDialog
     * @param paukerFrame the pauker frame
     * @param card the card to edit
     * @param searchEngine the search engine for similar cards
     */
    public EditCardDialog(
            PaukerFrame paukerFrame, Card card, SearchEngine searchEngine) {
        super(paukerFrame, true);
        initDialog(paukerFrame, paukerFrame, card, searchEngine);
    }

    private void initDialog(Component parent, PaukerFrame paukerFrame,
            Card card, SearchEngine searchEngine) {
        this.card = card;
        this.searchEngine = searchEngine;

        okButtonPressed = false;

        initComponents();

        setTitle(PaukerFrame.STRINGS.getString("Edit_Card"));

        editCardPanel.setPaukerFrame(paukerFrame);
        editCardPanel.setSplitOrientation(paukerFrame.getSplitOrientation());
        editCardPanel.applyWrapSettings();
        editCardPanel.setCard(card);
        editCardPanel.setRepeatByTyping(card.isRepeatedByTyping());

        if (card.isLearned()) {
            long now = System.currentTimeMillis();
            long expirationTime = card.getExpirationTime();
            if (expirationTime < now) {
                expiresLabel.setText(
                        PaukerFrame.STRINGS.getString("Expired_At"));
                expiresLabel.setForeground(Color.red);
                expiresTextField.setForeground(Color.red);
            }
            DateFormat dateFormat = DateFormat.getDateTimeInstance(
                    DateFormat.LONG, DateFormat.SHORT);
            learnedTextField.setText(
                    dateFormat.format(card.getLearnedTimestamp()));
            expiresTextField.setText(dateFormat.format(expirationTime));

        } else {
            remove(expirationPanel);
        }

        getRootPane().setDefaultButton(editOKButton);
        pack();
        setLocationRelativeTo(parent);
        setVisible(true);
    }

    /**
     * returns <code>true</code>, if the OK button was pressed to close the
     * dialog, <code>false</code> otherwise
     * @return <code>true</code>, if the OK button was pressed to close the
     * dialog, <code>false</code> otherwise
     */
    public boolean okPressed() {
        return okButtonPressed;
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Erzeugte Anweisungen ">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        editCardPanel = new pauker.program.gui.swing.EditCardPanel();
        jSeparator1 = new javax.swing.JSeparator();
        expirationPanel = new javax.swing.JPanel();
        learnedLabel = new javax.swing.JLabel();
        learnedTextField = new javax.swing.JTextField();
        expiresLabel = new javax.swing.JLabel();
        expiresTextField = new javax.swing.JTextField();
        jSeparator2 = new javax.swing.JSeparator();
        buttonPanel = new javax.swing.JPanel();
        editOKButton = new javax.swing.JButton();
        editCancelButton = new javax.swing.JButton();

        getContentPane().setLayout(new java.awt.GridBagLayout());

        setDefaultCloseOperation(javax.swing.WindowConstants.DISPOSE_ON_CLOSE);
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowOpened(java.awt.event.WindowEvent evt) {
                formWindowOpened(evt);
            }
            public void windowClosing(java.awt.event.WindowEvent evt) {
                closeDialog(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 0, 5);
        getContentPane().add(editCardPanel, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        getContentPane().add(jSeparator1, gridBagConstraints);

        expirationPanel.setLayout(new java.awt.GridBagLayout());

        learnedLabel.setFont(new java.awt.Font("Dialog", 0, 10));
        java.util.ResourceBundle bundle = java.util.ResourceBundle.getBundle("pauker/Strings"); // NOI18N
        learnedLabel.setText(bundle.getString("Learned_At")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 0, 0);
        expirationPanel.add(learnedLabel, gridBagConstraints);

        learnedTextField.setEditable(false);
        learnedTextField.setFont(new java.awt.Font("Dialog", 0, 10));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 0, 5);
        expirationPanel.add(learnedTextField, gridBagConstraints);

        expiresLabel.setFont(new java.awt.Font("Dialog", 0, 10));
        expiresLabel.setText(bundle.getString("Expires_At")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 0, 0);
        expirationPanel.add(expiresLabel, gridBagConstraints);

        expiresTextField.setEditable(false);
        expiresTextField.setFont(new java.awt.Font("Dialog", 0, 10));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 0, 5);
        expirationPanel.add(expiresTextField, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 0, 0, 0);
        expirationPanel.add(jSeparator2, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        getContentPane().add(expirationPanel, gridBagConstraints);

        buttonPanel.setLayout(new java.awt.GridBagLayout());

        editOKButton.setFont(new java.awt.Font("Dialog", 0, 10));
        editOKButton.setText(bundle.getString("OK")); // NOI18N
        editOKButton.setName("editOKButton");
        editOKButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                editOKButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        buttonPanel.add(editOKButton, gridBagConstraints);

        editCancelButton.setFont(new java.awt.Font("Dialog", 0, 10));
        editCancelButton.setText(bundle.getString("Cancel")); // NOI18N
        editCancelButton.setName("editCancelButton");
        editCancelButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                editCancelButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 0);
        buttonPanel.add(editCancelButton, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 0, 5, 0);
        getContentPane().add(buttonPanel, gridBagConstraints);

    }// </editor-fold>//GEN-END:initComponents
    private void formWindowOpened(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_formWindowOpened
        editCardPanel.setFocus(Card.Element.FRONT_SIDE);
        // Bug#1721717
        // default button is not correctly set
        // !!! setting the default button without SwingUtilities seems not to
        // work with Windows XP and Java 6 !!!
        SwingUtilities.invokeLater(new Runnable() {

            public void run() {
                getRootPane().setDefaultButton(editOKButton);
            }
        });
    }//GEN-LAST:event_formWindowOpened

    private void editCancelButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_editCancelButtonActionPerformed
        setVisible(false);
        dispose();
    }//GEN-LAST:event_editCancelButtonActionPerformed

    private void editOKButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_editOKButtonActionPerformed

        String frontSideText = editCardPanel.getFrontSide();
        String reverseSideText = editCardPanel.getReverseSide();

        if (frontSideText.length() == 0) {
            JOptionPane.showMessageDialog(
                    this, PaukerFrame.STRINGS.getString("FrontSide_Is_Empty"),
                    PaukerFrame.STRINGS.getString("Error"), 
                    JOptionPane.ERROR_MESSAGE);

            editCardPanel.setFocus(Card.Element.FRONT_SIDE);
            return;
        }

        if (reverseSideText.length() == 0) {
            JOptionPane.showMessageDialog(
                    this, PaukerFrame.STRINGS.getString("ReverseSide_Is_Empty"),
                    PaukerFrame.STRINGS.getString("Error"), 
                    JOptionPane.ERROR_MESSAGE);

            editCardPanel.setFocus(Card.Element.REVERSE_SIDE);
            return;
        }

        // set front side
        CardSide frontSide = card.getFrontSide();
        frontSide.setText(frontSideText);
        frontSide.setFont(editCardPanel.getFrontSideFont());
        frontSide.setForegroundColor(editCardPanel.getFrontForegroundColor());
        frontSide.setBackgroundColor(editCardPanel.getFrontBackgroundColor());
        frontSide.setOrientation(
                editCardPanel.getFrontSideComponentOrientation());
        frontSide.setRepeatByTyping(editCardPanel.isRepeatByTyping());

        // set reverse side
        CardSide reverseSide = card.getReverseSide();
        reverseSide.setText(reverseSideText);
        reverseSide.setFont(editCardPanel.getReverseSideFont());
        reverseSide.setForegroundColor(
                editCardPanel.getReverseSideForegroundColor());
        reverseSide.setBackgroundColor(
                editCardPanel.getReverseSideBackgroundColor());
        reverseSide.setOrientation(
                editCardPanel.getReverseSideComponentOrientation());

        searchEngine.index(card);

        okButtonPressed = true;
        setVisible(false);
        dispose();
    }//GEN-LAST:event_editOKButtonActionPerformed

    /** Closes the dialog */
    private void closeDialog(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_closeDialog
        setVisible(false);
        dispose();
    }//GEN-LAST:event_closeDialog
    // Variablendeklaration - nicht modifizieren//GEN-BEGIN:variables
    private javax.swing.JPanel buttonPanel;
    private javax.swing.JButton editCancelButton;
    private pauker.program.gui.swing.EditCardPanel editCardPanel;
    private javax.swing.JButton editOKButton;
    private javax.swing.JPanel expirationPanel;
    private javax.swing.JLabel expiresLabel;
    private javax.swing.JTextField expiresTextField;
    private javax.swing.JSeparator jSeparator1;
    private javax.swing.JSeparator jSeparator2;
    private javax.swing.JLabel learnedLabel;
    private javax.swing.JTextField learnedTextField;
    // Ende der Variablendeklaration//GEN-END:variables
}
