///
// Copyright (C) 2002 - 2004, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "window.h"
#include <fstream>
#include <gtkmm/stock.h>
#include "widget/filesel.h"
#include "widget/programs.h"
#include "widget/usererror.h"
#include "config.h"
#include "printdialog.h"
#include "aboutdialog.h"
#include "propertiesdialog.h"
#include "streamdialog.h"
#include "inspiration.h"
#include "util/warning.h"
#include "util/tempfile.h"
#include "docpropsdialog.h"
#include "docview.h"
#include "document/page.h"
#include "templatepagedialog.h"
#include "clipboard.h"

namespace {
  template<class P> P nonull(P p) {
    if(!p)
      throw std::runtime_error("A pointer was null in " __FILE__);
    return p;
  }
  
  void duplicate_view(const FrameWindow *orig) { new FrameWindow(*orig); }
  
  void new_document(DocumentView *view) {
    DocPropsDialog::instance().create(view);
  }
  
  void print_to_viewer(const DocumentView *view) {
    const DocRef document = nonull(view->get_document());
    std::string tmp_file = Tempfile::find_new_name();
    std::ofstream out(tmp_file.c_str());
    if(!out)
      throw std::runtime_error("Could not open file for printing: "
                               + tmp_file);
    // print all pages:
    document->print
      (out, document->get_first_page_num(),
       document->get_first_page_num() + document->get_num_of_pages() - 1,
       false /* not eps */, true /* include fonts */);
    Programs::open_postscript_file(tmp_file, true /* delete file when done */);
  }
  
  void show_doc_props_dialog(DocumentView *view) {
    DocPropsDialog::instance().modify(view);
  }
  
  void copi(DocumentView *view) {
    Clipboard::copy(nonull(view->get_document()));
  }
  
  void cut(DocumentView *view) {
    Clipboard::cut(nonull(view->get_document()));
  }
  
  void paste(DocumentView *view) {
    Clipboard::paste(nonull(view->get_document()),
                     nonull(view->get_page()));
  }
  
  void delete_selected(DocumentView *view) {
    nonull(view->get_document())->delete_selected();
  }
  
  void group_selected(DocumentView *view) {
    nonull(view->get_page())->group_selected();
  }
  
  void ungroup_selected(DocumentView *view) {
    nonull(view->get_page())->ungroup_selected();
  }
  
  void select_all_frames(DocumentView *view, bool select_not_unselect) {
    nonull(view->get_page())->select_all(select_not_unselect);
  }
  
  void rearrange_selected(DocumentView *view, RearrangeTarget target) {
    nonull(view->get_page())->rearrange_selected(target);
  }
  
  void insert_page(Gtk::Window *parent, DocumentView *view, bool after) {
    DocRef document = nonull(view->get_document());
    int page_num =
      document->get_num_of_pages()
      ? (view->get_current_page_num() + (after ? 1 : 0))
      : document->get_first_page_num();
    if(document->get_template_pages().empty()) {
      document->new_page(page_num);
      view->set_current_page_num(page_num);
    } else {
      new TemplatePageDialog(*parent, *view, page_num);
    }
  }
  
  void toggle_grid(Glib::RefPtr<Gtk::ToggleAction> action,
                   DocumentView *view) {
    view->set_snap(action->get_active() ? snap::GRID : snap::NONE);
  }
  
  void toggle_toolbar(Glib::RefPtr<Gtk::UIManager> uimanager) {
    Gtk::Widget *toolbar = nonull(uimanager->get_widget("/ToolBar"));
    if(toolbar->is_visible())
      toolbar->hide();
    else
      toolbar->show();
  }
  
  void toggle_fullscreen(Glib::RefPtr<Gtk::ToggleAction> action,
                         Gtk::Window *window) {
    if(action->get_active())
      window->fullscreen();
    else
      window->unfullscreen();
  }
  
  void zoom(Zoomer *zoomer, Gtk::BuiltinStockID id) {
    float f = zoomer->get_factor();
    using namespace Gtk;
    if(id == Stock::ZOOM_IN) f = 2.0 * f;
    else if(id == Stock::ZOOM_OUT) f = 0.5 * f;
    else if(id == Stock::ZOOM_100) f = 1.0;
    // rely on Zoomer to limit the values
    zoomer->set_factor(f);
  }
}

void FrameWindow::create_menus() {
  using namespace Gtk;
  using namespace sigc;
  
  /// *** file menu ***
  
  main_group->add(Action::create("FileMenu", "_File"));
  
  main_group->add(Action::create("FileNew", Stock::NEW, "_New ...",
                                 "Create a new document"),
                  bind(sigc::ptr_fun(&new_document), &document_view));
  
  doc_group->add(Action::create("FileNewView", "New Vi_ew",
                                "Create a new window "
                                "for the current document"),
                 bind(sigc::ptr_fun(&duplicate_view), this));
  
  main_group->add(Action::create("FileOpen", Stock::OPEN, "_Open ...",
                                 "Open a document"),
                  mem_fun(*open_dialog, &Filesel::show_all));
  
  doc_group->add(Action::create("FileSave", Stock::SAVE, "_Save ...",
                                "Save document"),
                 mem_fun(*this, &FrameWindow::save));
  
  doc_group->add(Action::create("FileSaveAs", Stock::SAVE_AS),
                 AccelKey("<shift><control>S"),
                 mem_fun(*save_dialog, &Filesel::show_all));
  
  page_group->add(Action::create("FilePrint", Stock::PRINT, "_Print ...",
                                 "Print document"),
                  AccelKey("<control>P"),
                  mem_fun(*print_dialog, &PrintDialog::show_it));
  
  page_group->add(Action::create("FilePrintPreview", Stock::PRINT_PREVIEW,
                                 "Print Pre_view ...",
                                 "Print document to external viewer"),
                  AccelKey("<shift><control>P"),
                  bind(sigc::ptr_fun(&print_to_viewer), &document_view));
  
  doc_group->add(Action::create("FileDocProps", "P_roperties ...",
                                "Document properties"),
                 bind(sigc::ptr_fun(&show_doc_props_dialog), &document_view));
  
  main_group->add(Action::create("FileClose", Stock::CLOSE),
                  mem_fun(*this, &FrameWindow::close));
  
  main_group->add(Action::create("FileQuit", Stock::QUIT),
                  ptr_fun(&FrameWindow::quit));
  
  
  /// *** edit menu ***
  
  main_group->add(Action::create("EditMenu", "_Edit"));
  
  selection_group->add(Action::create("EditCopy", Stock::COPY),
                       bind(sigc::ptr_fun(&copi), &document_view));
  
  selection_group->add(Action::create("EditCut", Stock::CUT),
                       bind(sigc::ptr_fun(&cut), &document_view));
  
  page_group->add(Action::create("EditPaste", Stock::PASTE),
                  bind(sigc::ptr_fun(&paste), &document_view));
  
  selection_group->add(Action::create("EditDelete", Stock::DELETE),
                       AccelKey("Delete"),
                       bind(sigc::ptr_fun(&delete_selected), &document_view));
  
  page_group->add(Action::create("EditSelectAll", "Select _All"),
                  AccelKey("<control>A"),
                  bind(sigc::ptr_fun(&select_all_frames), &document_view,
                       true /* select */));
  
  page_group->add(Action::create("EditUnselectAll", "Dese_lect All"),
                  AccelKey("<shift><control>A"),
                  bind(sigc::ptr_fun(&select_all_frames), &document_view,
                       false /* deselect */));
  
  // *** arrange submenu ***
  
  main_group->add(Action::create("ArrangeMenu", "A_rrange"));
  
  selection_group->add(Action::create("ArrangeGroup", "_Group"),
                       AccelKey("<control>G"),
                       bind(sigc::ptr_fun(group_selected), &document_view));
  
  selection_group->add(Action::create("ArrangeUngroup", "_Ungroup"),
                       AccelKey("<control>U"),
                       bind(sigc::ptr_fun(ungroup_selected), &document_view));
  
  selection_group->add(Action::create("ArrangeTop", Stock::GOTO_TOP,
                                      "Move to _top", "Move to top"),
                       AccelKey("<control>Home"),
                       bind(sigc::ptr_fun(&rearrange_selected),
                            &document_view, TOP));
  
  
  selection_group->add(Action::create("ArrangeUp", Stock::GO_UP,
                                      "Move _up", "Move up"),
                       AccelKey("<control>Page_Up"),
                       bind(sigc::ptr_fun(&rearrange_selected),
                            &document_view, UP));
  
  selection_group->add(Action::create("ArrangeDown", Stock::GO_DOWN,
                                      "Move _down", "Move down"),
                       AccelKey("<control>Page_Down"),
                       bind(sigc::ptr_fun(&rearrange_selected),
                            &document_view, DOWN));
  
  selection_group->add(Action::create("ArrangeBottom", Stock::GOTO_BOTTOM,
                                      "Move to _bottom", "Move to bottom"),
                       AccelKey("<control>End"),
                       bind(sigc::ptr_fun(&rearrange_selected),
                            &document_view, DOWN));
  
  page_group->add(Action::create("EditInsertText", StockID("newframe"),
                                 "Insert Te_xt Frame ...",
                                 "Insert text frame"),
                  mem_fun(*text_frame_dialog, &TextFrameDialog::show_raise));
  
  page_group->add(Action::create("EditInsertImage", StockID("moose"),
                                 "Insert _Image ...", "Insert image"),
                  mem_fun(*import_dialog, &Filesel::show_all));
  
  main_group->add(Action::create("EditPrefs", Stock::PREFERENCES),
                  mem_fun(config, &PFile::dialog_show));
  
  
  // *** page menu ***
  
  main_group->add(Action::create("PageMenu", "_Page"));
  
  page_group->add(Action::create("PageDelete", "_Delete"),
                  mem_fun(document_view, &DocumentView::delete_page));
  
  doc_group->add(Action::create("PageInsertBefore", "Insert _Before ..."),
                 bind(sigc::ptr_fun(&insert_page),
                      this, &document_view, false /* not after */));
  
  // we only need one of these when we don't have any pages
  page_group->add(Action::create("PageInsertAfter", "Insert _After ..."),
                 bind(sigc::ptr_fun(&insert_page),
                      this, &document_view, true /* after */));
  
  // *** view menu ***
  
  main_group->add(Action::create("ViewMenu", "_View"));
  
  main_group->add(ToggleAction::create("ViewToolbar", "_Toolbar", "",
                                       true /* active */),
                  bind(sigc::ptr_fun(toggle_toolbar), uimanager));
  {
    Glib::RefPtr<ToggleAction> action =
      ToggleAction::create("ViewFullScreen", "_Full screen", "",
                           false /* not active */);
    main_group->add(action, AccelKey("F11"),
                    bind(sigc::ptr_fun(toggle_fullscreen), action, this));
  }
  
  main_group->add(Action::create("ZoomIn", Stock::ZOOM_IN), 
                  AccelKey("<ctrl>plus"),
                  bind(sigc::ptr_fun(&zoom), &zoom_factor, Stock::ZOOM_IN));
  main_group->add(Action::create("ZoomOut", Stock::ZOOM_OUT),
                  AccelKey("<ctrl>minus"),
                  bind(sigc::ptr_fun(&zoom), &zoom_factor, Stock::ZOOM_OUT));
  main_group->add(Action::create("Zoom100", Stock::ZOOM_100),
                  AccelKey("<ctrl>0"),
                  bind(sigc::ptr_fun(&zoom), &zoom_factor, Stock::ZOOM_100));
  
  doc_group->add(Action::create("ViewProps", Stock::PROPERTIES,
                                "_Properties", "Show Properties window"),
                 mem_fun(PropertiesDialog::instance(),
                         &PropertiesDialog::show_raise));
  
  doc_group->add(Action::create("ViewStreams", StockID("streams"),
                                "_Streams", "Show Streams window"),
                 mem_fun(StreamDialog::instance(),
                         &StreamDialog::show_raise));
  
  {
    Glib::RefPtr<Gtk::ToggleAction> action =
      ToggleAction::create("ViewSnapGrid", StockID("grid"),
                           "Snap to _grid", "Toggle snap to grid");
    doc_group->add(action,
                   bind(sigc::ptr_fun(&toggle_grid), action, &document_view));
  }
  
  // *** help menu ***
  main_group->add(Action::create("HelpMenu", "_Help"));
  
  main_group->add(Action::create("HelpHelp", Stock::HELP,"_User's guide"),
                  AccelKey("F1"),
                  ptr_fun(&Programs::open_docs));
  
  main_group->add(Action::create("HelpAbout", "_About"),
                  mem_fun(::AboutDialog::instance(), &::AboutDialog::show_all));
  
  main_group->add(Action::create("HelpHomepage", Stock::HOME, "_Homepage"),
                  ptr_fun(&Programs::open_homepage));
  
  main_group->add(Action::create("HelpInspiration", "_Inspiration"),
                  mem_fun(Inspiration::instance(),
                          &Inspiration::show_all));
  
  // Layout the actions in a menubar and toolbar:
  try {
    const Glib::ustring ui_info = 
      "<ui>"
      "  <menubar name='MenuBar'>"
      "    <menu action='FileMenu'>"
      "      <menuitem action='FileNew'/>"
      "      <menuitem action='FileNewView'/>"
      "      <menuitem action='FileOpen'/>"
      "      <separator/>"
      "      <menuitem action='FileSave'/>"
      "      <menuitem action='FileSaveAs'/>"
      "      <separator/>"
      "      <menuitem action='FilePrint'/>"
      "      <menuitem action='FilePrintPreview'/>"
      "      <separator/>"
      "      <menuitem action='FileDocProps'/>"
      "      <separator/>"
      "      <menuitem action='FileClose'/>"
      "      <menuitem action='FileQuit'/>"
      "    </menu>"
      
      "    <menu action='EditMenu'>"
      "      <menuitem action='EditCopy'/>"
      "      <menuitem action='EditCut'/>"
      "      <menuitem action='EditPaste'/>"
      "      <menuitem action='EditDelete'/>"
      "      <separator/>"
      "      <menuitem action='EditSelectAll'/>"
      "      <menuitem action='EditUnselectAll'/>"
      "      <separator/>"
      "      <menu action='ArrangeMenu'>"
      "        <menuitem action='ArrangeGroup'/>"
      "        <menuitem action='ArrangeUngroup'/>"
      "        <separator/>"
      "        <menuitem action='ArrangeTop'/>"
      "        <menuitem action='ArrangeUp'/>"
      "        <menuitem action='ArrangeDown'/>"
      "        <menuitem action='ArrangeBottom'/>"
      "      </menu>"
      "      <separator/>"
      "      <menuitem action='EditInsertText'/>"
      "      <menuitem action='EditInsertImage'/>"
      "      <separator/>"
      "      <menuitem action='EditPrefs'/>"
      "    </menu>"
      
      "    <menu action='PageMenu'>"
      "      <menuitem action='PageDelete'/>"
      "      <separator/>"
      "      <menuitem action='PageInsertBefore'/>"
      "      <menuitem action='PageInsertAfter'/>"
      "    </menu>"
      
      "    <menu action='ViewMenu'>"
      "      <menuitem action='ViewToolbar'/>"
      "      <menuitem action='ViewFullScreen'/>"
      "      <separator/>"
      "      <menuitem action='ZoomIn'/>"
      "      <menuitem action='ZoomOut'/>"
      "      <menuitem action='Zoom100'/>"
      "      <separator/>"
      "      <menuitem action='ViewProps'/>"
      "      <menuitem action='ViewStreams'/>"
      "      <separator/>"
      "      <menuitem action='ViewSnapGrid'/>"
      "    </menu>"
      
      "    <menu action='HelpMenu'>"
      "      <menuitem action='HelpHelp'/>"
      "      <menuitem action='HelpAbout'/>"
      "      <separator/>"
      "      <menuitem action='HelpHomepage'/>"
      "      <menuitem action='HelpInspiration'/>"
      "    </menu>"
      "  </menubar>"
      
      "  <toolbar name='ToolBar'>"
      "    <toolitem action='FileNew'/>"
      "    <separator/>"
      "    <toolitem action='FilePrintPreview'/>"
      "    <separator/>"
      "    <toolitem action='EditInsertText'/>"
      "    <toolitem action='EditInsertImage'/>"
      "    <separator/>"
      "    <toolitem action='ViewProps'/>"
      "    <toolitem action='ViewStreams'/>"
      "    <separator/>"
      "    <toolitem action='ViewSnapGrid'/>"
      "    <separator/>"
      "    <toolitem action='ZoomIn'/>"
      "    <toolitem action='ZoomOut'/>"
      "    <toolitem action='Zoom100'/>"
      "    <separator/>"
      "    <toolitem action='ArrangeTop'/>"
      "    <toolitem action='ArrangeUp'/>"
      "    <toolitem action='ArrangeDown'/>"
      "    <toolitem action='ArrangeBottom'/>"
      "  </toolbar>"
      "</ui>";
    
    uimanager->add_ui_from_string(ui_info);
  }
  catch(const Glib::Error& e) {
    warning << "failed to build menus: " <<  e.what() << std::endl;
  }
}
