///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <scripting/Scripting.h>
#include <core/scene/SceneRoot.h>
#include <core/scene/SceneNode.h>
#include <core/scene/SelectionSet.h>
#include <core/scene/ObjectNode.h>
#include <core/scene/GroupNode.h>
#include <core/scene/material/Material.h>
#include <core/scene/objects/SceneObject.h>
#include <core/scene/objects/geometry/SimpleGeometryObject.h>
#include <core/scene/objects/ModifiedObject.h>
#include <core/scene/objects/Modifier.h>
#include <core/scene/objects/ModifierApplication.h>
#include <core/scene/objects/PipelineFlowState.h>
#include <core/viewport/Viewport.h>
#include "ContainerWrappers.h"

namespace Scripting {

using namespace boost::python;

SceneNode* SelectionSetGetItem(const SelectionSet& set, int index)
{
	if(index < 0) index += set.count();
    if(index >= 0 && index < (int)set.count()) return set.node(index);
    PyErr_SetString(PyExc_IndexError, "Index out of range");
    throw_error_already_set();
	return NULL;
}

class SceneObjectWrapper : public SceneObject, public wrapper<SceneObject> {
public:
	SceneObjectWrapper(bool isLoading) : SceneObject(isLoading) {}
	TimeInterval objectValidity(TimeTicks time) { return this->get_override("ObjectValidity")(time); }
	void renderObject(TimeTicks time, ObjectNode* contextNode, Viewport* vp) { this->get_override("RenderObject")(time, ObjectNode::SmartPtr(contextNode), ptr(vp)); }
	Box3 boundingBox(TimeTicks time, ObjectNode* contextNode) { return this->get_override("BoundingBox")(time, ObjectNode::SmartPtr(contextNode)); }
};

class SimpleGeometryObjectWrapper : public SimpleGeometryObject, public wrapper<SimpleGeometryObject> {
public:
	SimpleGeometryObjectWrapper(bool isLoading) : SimpleGeometryObject(isLoading) {}
	TimeInterval objectValidity(TimeTicks time) {
		try { return this->get_override("ObjectValidity")(time); }
		catch(error_already_set &) { PyErr_Print(); }
	}
	void buildMesh(TimeTicks time, TriMesh& mesh, TimeInterval& meshValidity) {
		try { get_override("BuildMesh")(time, ref(mesh), ref(meshValidity)); }
		catch(error_already_set &) { PyErr_Print(); }
	}
};

void ExportScene()
{
	class_<Material, bases<RefTarget>, intrusive_ptr<Material>, noncopyable>("Material", no_init)
		.add_property("Name", make_function(&Material::name, return_value_policy<copy_const_reference>()), &Material::setName)
	;

	class_<PipelineFlowState, bases<RefMaker> >("PipelineFlowState", no_init)
		.add_property("Result", make_function(&PipelineFlowState::result, return_internal_reference<>()), (void (PipelineFlowState::*)(const SceneObject::SmartPtr&))&PipelineFlowState::setResult)
		.add_property("StateValidity", make_function(&PipelineFlowState::stateValidity, return_value_policy<copy_const_reference>()), &PipelineFlowState::setStateValidity)
	;

	class_<QVector<SceneNode*>, noncopyable>("SceneNodeList", no_init)
		.def(QVector_readonly_indexing_suite< QVector<SceneNode*>, return_internal_reference<> >())
	;

	class_<SceneNode, bases<RefTarget>, intrusive_ptr<SceneNode>, noncopyable>("SceneNode", no_init)
		.add_property("IsRootNode", &SceneNode::isRootNode)
		.add_property("Name", make_function(&SceneNode::name, return_value_policy<copy_const_reference>()), &SceneNode::setName)
		.add_property("ParentNode", make_function(&SceneNode::parentNode, return_internal_reference<>()))
		.add_property("DisplayColor", make_function(&SceneNode::displayColor, return_value_policy<return_by_value>()), &SceneNode::setDisplayColor)
		.def("DeleteNode", &SceneNode::deleteNode)
		.def("AddChild", (void (SceneNode::*)(SceneNode*))&SceneNode::addChild)
		.def("RemoveChild", &SceneNode::removeChild)
		.add_property("Selected", &SceneNode::isSelected, &SceneNode::setSelected)
		.add_property("IsObjectNode", &SceneNode::isObjectNode)
		.add_property("IsGroupNode", &SceneNode::isGroupNode)
		.add_property("Children", make_function(&SceneNode::children, return_internal_reference<>()))
		.add_property("Transformation", make_function(&SceneNode::transformationController, return_internal_reference<>()), (void (SceneNode::*)(const TransformationController::SmartPtr&))&SceneNode::setTransformationController)
	;

	class_<SceneRoot, bases<SceneNode>, intrusive_ptr<SceneRoot>, noncopyable>("SceneRoot", no_init)
		.def("GetNodeByName", &SceneRoot::getNodeByName, return_internal_reference<>())
		.def("__getitem__", &SceneRoot::getNodeByName, return_internal_reference<>())
	;

	class_<GroupNode, bases<SceneNode>, intrusive_ptr<GroupNode>, noncopyable>("GroupNode", init<>())
		.add_property("IsOpen", &GroupNode::isGroupOpen, &GroupNode::setGroupOpen)
	;

	class_<SelectionSet, bases<RefTarget>, intrusive_ptr<SelectionSet>, noncopyable>("SelectionSet")
		.add_property("Count", &SelectionSet::count)
		.def("Contains", &SelectionSet::contains)
		.def("Add", (void (SelectionSet::*)(SceneNode*))&SelectionSet::add)
		.def("Remove", &SelectionSet::remove)
		.def("Clear", &SelectionSet::clear)
		.def("__len__", &SelectionSet::count)
		.def("__getitem__",  &SelectionSetGetItem, return_internal_reference<>())
	;

	class_<SceneObjectWrapper, bases<RefTarget>, intrusive_ptr<SceneObjectWrapper>, noncopyable>("SceneObject", init<bool>())
		.add_property("InputObjectCount", &SceneObject::inputObjectCount)
		.def("GetInputObject", make_function(&SceneObject::inputObject, return_internal_reference<>()))
		.def("CanConvertTo", &SceneObject::canConvertTo)
		.def("ConvertTo", (SceneObject::SmartPtr (SceneObject::*)(PluginClassDescriptor*, TimeTicks))&SceneObject::convertTo)
		.def("ObjectValidity", pure_virtual(&SceneObject::objectValidity))
		.def("RenderObject", pure_virtual(&SceneObject::renderObject))
		.def("BoundingBox", pure_virtual(&SceneObject::boundingBox))
	;

	class_<ObjectNode, bases<SceneNode>, intrusive_ptr<ObjectNode>, noncopyable>("ObjectNode", init< optional<SceneObject*> >())
		.add_property("SceneObject", make_function(&ObjectNode::sceneObject, return_internal_reference<>()), (void (ObjectNode::*)(SceneObject*))&ObjectNode::setSceneObject)
		.add_property("Material", make_function(&ObjectNode::material, return_internal_reference<>()), (void (ObjectNode::*)(Material*))&ObjectNode::setMaterial)
		.def("ApplyModifier", (void (ObjectNode::*)(Modifier*))&ObjectNode::applyModifier)
		.def("EvalPipeline", &ObjectNode::evalPipeline, return_value_policy<copy_const_reference>())
	;

	class_<SimpleGeometryObjectWrapper, bases<SceneObject>, intrusive_ptr<SimpleGeometryObjectWrapper>, noncopyable>("SimpleGeometryObject", init<bool>())
		.def("BuildMesh", pure_virtual(&SimpleGeometryObjectWrapper::buildMesh))
	;

	enum_<EvaluationStatus::EvaluationStatusType>("EvaluationStatusType")
    	.value("SUCCESS", EvaluationStatus::EVALUATION_SUCCESS)
    	.value("WARNING", EvaluationStatus::EVALUATION_WARNING)
    	.value("ERROR", EvaluationStatus::EVALUATION_ERROR)
    ;

	class_<EvaluationStatus>("EvaluationStatus", no_init)
		.add_property("Type", &EvaluationStatus::type)
		.add_property("ShortMessage", make_function(&EvaluationStatus::shortMessage, return_internal_reference<>()))
		.add_property("LongMessage", make_function(&EvaluationStatus::longMessage, return_internal_reference<>()))
	;

	class_<ModifierApplication, bases<RefTarget>, intrusive_ptr<ModifierApplication>, noncopyable>("ModifierApplication", init<Modifier*>())
		.add_property("Modifier", make_function(&ModifierApplication::modifier, return_internal_reference<>()))
		.add_property("ModifiedObject", make_function(&ModifierApplication::modifiedObject, return_internal_reference<>()))
		.add_property("Status", make_function(&ModifierApplication::status, return_internal_reference<>()))
	;

	class_<Modifier, bases<RefTarget>, intrusive_ptr<Modifier>, noncopyable>("Modifier", no_init)
		.add_property("Enabled", &Modifier::isModifierEnabled, &Modifier::setModifierEnabled)
	;

	class_<QVector<ModifierApplication*>, noncopyable>("ModifierApplicationList", no_init)
		.def(QVector_readonly_indexing_suite< QVector<ModifierApplication*>, return_internal_reference<> >())
	;

	class_<ModifiedObject, bases<SceneObject>, intrusive_ptr<ModifiedObject>, noncopyable>("ModifiedObject", init<>())
		.add_property("InputObject", make_function((SceneObject* (ModifiedObject::*)() const)&ModifiedObject::inputObject, return_internal_reference<>()), (void (ModifiedObject::*)(SceneObject*))&ModifiedObject::setInputObject)
		.def("InsertModifier", make_function(&ModifiedObject::insertModifier, return_internal_reference<>()))
		.def("RemoveModifier", &ModifiedObject::removeModifier)
		.add_property("ModifierApplications", make_function(&ModifiedObject::modifierApplications, return_internal_reference<>()))
	;
}

};
