///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __MULTI_FILE_PARSER_H
#define __MULTI_FILE_PARSER_H

#include <core/Core.h>
#include "AbstractFileColumnParser.h"

namespace AtomViz {

/**
 * \brief Abstract base class for atom file format parsers that can
 *        load multiple atoms files as a movie.
 *
 * This abstract base class is meant as basis for specialized file format
 * parsers that should be able to read in multiple files from a directory.
 *
 * Alternatively this base class supports multi-timestep files. These kind
 * of files contain more than one timestep in a single file.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT MultiFileParser : public AbstractFileColumnParser
{
protected:

	/// \brief Constructs a new instance of this class.
	/// \param isLoading Specifies whether the object's data fields will be initialized from the
	///                  data stored in a scene file after the instance has been created.
	MultiFileParser(bool isLoading = false) : AbstractFileColumnParser(isLoading),
		_scanMovieFile(false), _useWildcardFilename(false) {}

public:

	/// \brief Sets whether the parser should scan the directory for multiple data files
	///        using a wild-card pattern.
	/// \sa setWildcardFilename(), useWildcardFilename()
	void setUseWildcardFilename(bool enable) { this->_useWildcardFilename = enable; }

	/// \brief Returns whether the parser scans the directory for multiple data files
	///        using a wild-card pattern.
	/// \sa setWildcardFilename(), setUseWildcardFilename()
	bool useWildcardFilename() const { return this->_useWildcardFilename; }

	/// \brief Sets the wild-card pattern that is used to scan the directory for
	///        multiple atom files.
	///
	/// The wild-card filename should contain the special characters * and % to match
	/// multiple files.
	///
	/// The wild-card filename is only used if the feature has been enabled using setUseWildcardFilename().
	///
	/// \sa setWildcardFilename(), useWildcardFilename(), wildcardFilename()
	void setWildcardFilename(const QString& filename) { this->_wildcardFilename = filename; }

	/// \brief Returns the wild-card pattern used to match multiple atom files.
	/// \sa setWildcardFilename()
	const QString& wildcardFilename() const { return this->_wildcardFilename; }

	/// \brief Sets whether the parser should scan the input file(s) for multiple atomic data sets.
	/// \sa movieFileEnabled()
	void setMovieFileEnabled(bool enable) { this->_scanMovieFile = enable; }

	/// \brief Returns whether the parser looks for multiple atomic data sets in a single input file.
	/// \sa setMovieFileEnabled()
	bool movieFileEnabled() const { return this->_scanMovieFile; }

	// From AtomsFileParser:

	/// \brief Sets the name of the input file for this parser.
	virtual bool setInputFile(const QString& filename);

	/// \brief Scans the input file for multiple time steps before it is actually loaded.
	virtual bool prepareInputFile(bool suppressDialogs = false);

	/// \brief Reads an atomic data set from the input file.
	virtual EvaluationStatus loadAtomsFile(AtomsObject* destination, int movieFrame = 0, bool suppressDialogs = false);

	/// \brief Returns the number of movie frames.
	virtual int numberOfMovieFrames() { return _timeSteps.size(); }

public:

	Q_PROPERTY(bool useWildcardFilename READ useWildcardFilename WRITE setUseWildcardFilename)
	Q_PROPERTY(QString wildcardFilename READ wildcardFilename WRITE setWildcardFilename)
	Q_PROPERTY(bool movieFileEnabled READ movieFileEnabled WRITE setMovieFileEnabled)

protected:

	/// \brief Saves the class' contents to the given stream.
	/// \sa RefTarget::saveToStream()
	virtual void saveToStream(ObjectSaveStream& stream);

	/// \brief Loads the class' contents from the given stream.
	/// \sa RefTarget::loadFromStream()
	virtual void loadFromStream(ObjectLoadStream& stream);

	/// \brief Creates a copy of this object.
	/// \sa RefTarget::Clone()
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper);


	/// \brief Records the storage location of single time step for random access at a later time.
	/// \param filepath The full path to the file that contains the atomic data.
	/// \param byteOffset The byte offset into the file where the time step is stored.
	///                   This value is used to seek to the position in the file when the time step is loaded.
	/// \param lineNumber Optionally sepcifies the line number in the input file where the time step is stored.
	///                   This information only makes sense for text file formats and is only used to determine
	///                   the location of a parsing error when it is shown to the user.
	///
	/// This method must only be called by the implementation of the scanFileForTimeSteps() method.
	void addTimeStep(const QString& filepath, streampos byteOffset = 0, int lineNumber = 1);

	/// \brief Scans an atoms file for the time step frames contained therein.
	/// \param filepath The path of the input file to be scanned.
	/// \param suppressDialogs Specifies whether any dialogs or message boxes shown by the parser should be supressed during preparation.
	///                        This parameter might be set to \a true when the parser is invoked from a script to not interrupt the parsing process.
	/// \return \a false when the operation has been canceled by the user; \a true on success.
	/// \throws Exception on parsing error.
	///
	/// This virtual method must be implemented by sub-classes that support file formats that
	/// can store multiple time steps in a single file. The purpose of this method is to
	/// scan the atoms file for movie frames and record their position in the file using
	/// addTimeStep().
	///
	/// The default implementation of this method calls addTimeStep() just once, assuming that
	/// the input file contains only a single time step.
	///
	/// \sa addTimeStep(), prepareInputFile()
	virtual bool scanFileForTimeSteps(const QString& filepath, bool suppressDialogs = false) {
		addTimeStep(filepath);
		return true;
	}

	/// \brief Parses the atomic data of a single time step.
	/// \param destination Destination for the atomic data. The parser should store all atoms in the data channels of this AtomsObject.
	/// \param movieFrame The index of the movie frame to load (starting at 0).
	/// \param filename The path to the file to load.
	/// \param byteOffset The offset into the file where the movie frame is stored as reported by scanFileForTimeSteps().
	/// \param lineNumber The line in the file where the movie frame is stored as reported by scanFileForTimeSteps().
	/// \param suppressDialogs Specifies whether any dialogs or message boxes shown by the parser should be supressed during loading.
	///                        This parameter might be set to true when the parser is invoked from a script to not interrupt the parsing process.
	/// \return A status object with EvaluationStatus::EVALUATION_ERROR when the operation has been canceled by the user.
	/// \throws Exception on error.
	///
	/// This method must be implemented by sub-classes to do the actual parsing of the atoms file.
	/// It is called by loadAtomsFile().
	virtual EvaluationStatus loadTimeStep(AtomsObject* destination, int movieFrame, const QString& filename, streampos byteOffset, int lineNumber, bool suppressDialogs) = 0;

private:

	/// \brief This structures stores information about one simulation time step.
	struct TimeStep {
		/// The data file that contains the time step data (full path).
		/// This can be either the same for all time steps or unique file for each time step.
		QString filename;
		/// The byte offset into the file where the time step is stored.
		/// This information is only used when loading one big movie file.
		streampos byteOffset;
		/// The line number in the input file where the time step is stored.
		/// This information is only used when loading one big movie file.
		int lineNumber;
		/// The modification time of the input file.
		/// This is used to detect any changes to the input file making the stored byte offset invalid.
		QDateTime lastModificationTime;
	};

	/// Enables scanning of the input file for multiple time steps.
	bool _scanMovieFile;
	/// Indicates that the parser should look for other files in the same directory with a different time step number.
	bool _useWildcardFilename;
	/// The wild-card name that is used to load multiple input files.
	QString _wildcardFilename;
	/// The cached information about the time steps.
	QVector<TimeStep> _timeSteps;

private:
	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(MultiFileParser)
};


};	// End of namespace AtomViz

#endif // __MULTI_FILE_PARSER_H
