/*
 *  Copyright (c) 2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Path.h"

#include <GTLCore/Debug.h>
#include <GTLCore/Macros_p.h>
#include <GTLCore/SharedPointer.h>
#include <GTLCore/Region.h>

using namespace OpenRijn;

struct Path::Private : public GTLCore::SharedPointerData
{
  Private() {
  }
  Private(const Private& _rhs) : GTLCore::SharedPointerData() {
    foreach(Element* elt, _rhs.elements)
    {
      switch(elt->type)
      {
        case Path::MoveToElement:
        case Path::LineToElement:
          elements.push_back(new Element(*elt));
          break;
        case Path::CurveToElement:
          elements.push_back(new CurveElement(*static_cast<CurveElement*>(elt)));
          break;
      }
    }
  }
  std::vector<Element*> elements;
};

Path::Path() : d(new Private)
{
  d->ref();
}

GTL_SHARED_DATA(Path)

void Path::moveTo(const DrawingPoint& pt)
{
  deref();
  Element* elt = new Element;
  elt->type = MoveToElement;
  elt->point = pt;
  d->elements.push_back(elt);
}

void Path::lineTo(const DrawingPoint& pt)
{
  deref();
  Element* elt = new Element;
  elt->type = LineToElement;
  elt->point = pt;
  d->elements.push_back(elt);
}

void Path::curveTo(float x1, float y1, float x2, float y2, const DrawingPoint& pt2)
{
  deref();
  CurveElement* elt = new CurveElement;
  elt->type = CurveToElement;
  elt->point = pt2;
  elt->x1 = x1;
  elt->y1 = y1;
  elt->x2 = x2;
  elt->y2 = y2;
  d->elements.push_back(elt);
}

std::size_t Path::elementCount() const
{
  return d->elements.size();
}

const Path::Element* Path::elementAt(int idx) const
{
  GTL_ASSERT(std::size_t(idx) < d->elements.size());
  return d->elements[idx];
}

inline void includeCoordinate(float x, float y, GTLCore::RegionF& r)
{
  if( x < r.left() ) r.setLeft(x);
  if( y > r.right() ) r.setRight(y);
  if( y < r.top() ) r.setTop(y);
  if( y > r.bottom() ) r.setBottom(y);
}

GTLCore::RegionF Path::boundingRegion() const
{
  if( d->elements.size() == 0 ) return GTLCore::RegionF();
  
  float xs = 0, ys = 0;
  std::size_t idx = 0;
  if( d->elements[0]->type == MoveToElement)
  {
    xs = d->elements[0]->point.x;
    ys = d->elements[0]->point.y;
    ++idx;
  }
  GTLCore::RegionF r(xs, ys, 0, 0);
  
  for(; idx < d->elements.size(); ++idx)
  {
      Element* elt = d->elements[idx];
      switch(elt->type)
      {
        case Path::CurveToElement:
        {
          CurveElement* celt = static_cast<CurveElement*>(elt);
          includeCoordinate(celt->x1, celt->y1, r);
          includeCoordinate(celt->x2, celt->y2, r);
        }
        case Path::MoveToElement:
        case Path::LineToElement:
          includeCoordinate(elt->point.x, elt->point.y, r);
          break;
      }
  }
  
  return r;
}
