/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var array2iterator = require( '@stdlib/array/to-iterator' );
var iterEvery = require( './../lib' );


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.strictEqual( typeof iterEvery, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function throws an error if not provided an iterator', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		5,
		NaN,
		true,
		false,
		null,
		void 0,
		{},
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws a type error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			iterEvery( value );
		};
	}
});

tape( 'if provided an "empty" iterator, the function returns `true`', function test( t ) {
	var bool;
	var arr;

	arr = array2iterator( [] );
	bool = iterEvery( arr );

	t.strictEqual( bool, true, 'returns true' );
	t.end();
});

tape( 'the function returns `true` if all iterated values are truthy', function test( t ) {
	var bool;
	var arr;

	arr = array2iterator( [ 1, 1, 1, 1, 1 ] );
	bool = iterEvery( arr );

	t.strictEqual( bool, true, 'returns true' );
	t.end();
});

tape( 'the function returns `true` if all iterated values are truthy (value+done)', function test( t ) {
	var bool;
	var arr;

	arr = createIterator( [ 1, 1, 1, 1 ] );
	bool = iterEvery( arr );

	t.strictEqual( bool, true, 'returns true' );
	t.end();

	function createIterator( arr ) {
		var len;
		var it;
		var i;

		len = arr.length;
		i = -1;

		it = {};
		it.next = next;

		return it;

		function next() {
			i += 1;
			if ( i < len-1 ) {
				return {
					'value': arr[ i ],
					'done': false
				};
			}
			if ( i === len-1 ) {
				return {
					'value': arr[ i ],
					'done': true
				};
			}
			return {
				'done': true
			};
		}
	}
});

tape( 'the function returns `false` if at least one iterated value is falsy', function test( t ) {
	var bool;
	var arr;

	arr = array2iterator( [ 1, 1, 1, 0 ] );
	bool = iterEvery( arr );

	t.strictEqual( bool, false, 'returns false' );
	t.end();
});

tape( 'the function returns `false` if at least one iterated value is falsy (value+done)', function test( t ) {
	var bool;
	var arr;

	arr = createIterator( [ 1, 1, 1, 0 ] );
	bool = iterEvery( arr );

	t.strictEqual( bool, false, 'returns false' );
	t.end();

	function createIterator( arr ) {
		var len;
		var it;
		var i;

		len = arr.length;
		i = -1;

		it = {};
		it.next = next;

		return it;

		function next() {
			i += 1;
			if ( i < len-1 ) {
				return {
					'value': arr[ i ],
					'done': false
				};
			}
			if ( i === len-1 ) {
				return {
					'value': arr[ i ],
					'done': true
				};
			}
			return {
				'done': true
			};
		}
	}
});
