/* pdu-udp.c

   PDU builder for UDP datagrams

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "pbuild-priv.h"

struct ipv4_pseudo_hdr {
    in_addr_t saddr;
    in_addr_t daddr;
    uint8_t zero;
    uint8_t protocol;
    uint16_t len;
};

struct ipv6_pseudo_hdr {
    ip6_addr_t saddr;
    ip6_addr_t daddr;
    uint32_t len;
    uint16_t zero1;
    uint8_t zero2;
    uint8_t nxt;
};

static const struct pdu_dict protocol_mappings[] = {
    {"dns", (uint16_t []) {53} },
    {"sntp", (uint16_t []) {123} },
    {NULL, NULL}
};

static GHashTable *protos;

static void
build(const GNode *pdu, void *dest)
{
    struct udp_hdr *udp;
    uint16_t u16, *ptr;
    void *field;
    const GNode *next_pdu;

    udp = dest;

    u16 = htons( (field = _pb_pdata(pdu, "src") ) ? num_next(field) : 1073);
    SSVAL(udp, offsetof(struct udp_hdr, uh_sport), u16);

    /* Destination port handling */
    if (  !(field = _pb_pdata(pdu, "dst") )
	&& (next_pdu = pb_nextpdu(pdu) ) ) {
	/*
	 * Choose the destination port number based on the PDU ID of the
	 * next PDU.
	 */
	ptr = g_hash_table_lookup(protos, pb_getname(next_pdu) );
	u16 = ptr ? *ptr : 0;
    } else
	/*
	 * Use whatever was supplied by the user or a default of 53 if nothing
	 * was supplied.
	 */
	u16 = field ? num_next(field) : 53;

    u16 = htons(u16);
    SSVAL(udp, offsetof(struct udp_hdr, uh_dport), u16);
}

static void
postbuild(const GNode *pdu, void *dest, void *prev_pdu_hdr)
{
    struct udp_hdr *udp;
    struct ip_hdr *ip;
    uint16_t cksum, u16;
    void *field;
    size_t payload_len, opts_len;

    udp = dest;

    payload_len = ( (struct node_data *) pdu->data)->_data_pdu.payload_len;
    opts_len = ( (struct node_data *) pdu->data)->_data_pdu.opts_len;

    u16 =  (field = _pb_pdata(pdu, "length") )
	   ? htons(num_next(field) )
	   : htons(sizeof(struct udp_hdr) + payload_len);
    SSVAL(udp, offsetof(struct udp_hdr, uh_ulen), u16);

    /*
     * If we have a numspec for the checksum we force a (most likely bogus)
     * checksum, otherwise we calculate the real checksum.
     */
    if ( (field = _pb_pdata(pdu, "cksum") ) ) {
	cksum = htons(num_next(field) );
    } else {
	/* Calculate checksum */

	/* Must be 0 before calculating cksum! */
	SSVAL(udp, offsetof(struct udp_hdr, uh_sum), 0);

	ip = prev_pdu_hdr;

	if (ip) {
	    /*
	     * PDU definition does include an explicit IP PDU. Get everything
	     * we need for the pseudo-header from there.
	     */

	    if (ip->ip_v == 4) {
		/* IPv4 packet */
		struct ipv4_pseudo_hdr *ph;
		int segment_size;

		segment_size = sizeof(struct udp_hdr) + opts_len + payload_len;

		ph = xmalloc(sizeof(struct ipv4_pseudo_hdr) + segment_size);
		ph->saddr = ip->ip_src;
		ph->daddr = ip->ip_dst;
		ph->zero = 0;
		ph->protocol = ip->ip_p;
		ph->len = htons(segment_size);
		memcpy( (uint8_t *) ph + sizeof(struct ipv4_pseudo_hdr),
		       dest, segment_size);

		cksum = _pb_cksum(ph, sizeof(struct ipv4_pseudo_hdr)
				  + segment_size);

		free(ph);
	    } else {
		/* IPv6 packet */
		struct ipv6_pseudo_hdr *ph;
		int segment_size;
		struct ip6_hdr *ip6;

		ip6 = prev_pdu_hdr;

		segment_size = sizeof(struct tcp_hdr) + opts_len + payload_len;
		ph = xmalloc(sizeof(struct ipv6_pseudo_hdr) + segment_size);

		ph->saddr = ip6->ip6_src;
		ph->daddr = ip6->ip6_dst;
		ph->len = htonl(segment_size);
		ph->zero1 = 0;
		ph->zero2 = 0;
		ph->nxt = IPPROTO_UDP;

		memcpy( (uint8_t *) ph + sizeof(struct ipv6_pseudo_hdr),
		       dest, segment_size);

		cksum = _pb_cksum(ph, sizeof(struct ipv6_pseudo_hdr)
				  + segment_size);

		/*
		 * RFC 2460 states that if the checksum is 0 it must be changed
		 * to 0xffff
		 */
		if (!cksum)
		    cksum = 0xffff;

		free(ph);
	    }
	} else {
	    /*
	     * PDU definition does not include an explicit IP PDU. Get
	     * everything we need for the pseudo-header from user-specified
	     * data and some assumptions (IP proto is 6, and IP version is 4.)
	     */

	    struct ipv4_pseudo_hdr *ph;
	    int segment_size;

	    segment_size = sizeof(struct tcp_hdr) + opts_len + payload_len;

	    ph = xmalloc(sizeof(struct ipv4_pseudo_hdr) + segment_size);
	    ph->saddr = ip_next(_pb_pdata(pdu, "srcip") );
	    ph->daddr = ip_next(_pb_pdata(pdu, "dstip") );
	    ph->zero = 0;
	    ph->protocol = IP_PROTO_UDP;
	    ph->len = htons(segment_size);
	    memcpy( (uint8_t *) ph + sizeof(struct ipv4_pseudo_hdr),
		   dest, segment_size);

	    cksum = _pb_cksum(ph, sizeof(struct ipv4_pseudo_hdr)
		    + segment_size);

	    free(ph);
	}
    }

    SSVAL(udp, offsetof(struct udp_hdr, uh_sum), cksum);
}

#if 0
static void
pdu_udphdr_dumper(pdu_t *p, const char *prefix)
{
    struct udphdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Source port: %s\n", prefix, num_info(hdr_data->src_port) );
    printf("%s    Destination port: %s\n", prefix, num_info(hdr_data->dst_port) );
    printf("%s    UDP checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
}
#endif

static pdu_t pdu_udp = {
    .name = "udp",
    .description = "UDP datagram",
    .documented_in = "RFC 768",
    .len = sizeof(struct udp_hdr),
    .fields = (field_t []) {
	{.name = "src", .type = PDU_FTYPE_NUMTYPE},
	{.name = "dst", .type = PDU_FTYPE_NUMTYPE},
	{.name = "length", .type = PDU_FTYPE_NUMTYPE},
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = "srcip", .type = PDU_FTYPE_IP},
	{.name = "dstip", .type = PDU_FTYPE_IP},
	{.name = NULL}
    },
    .build = &build,
    .postbuild = &postbuild
};

void
_pb_register_udp(void)
{
    /*
     * Create hash table for protocol name to protocol value
     * mappings. This is used by the 1st pass PDU builder.
     */
    protos = _pb_new_hash(protocol_mappings);

    _pb_register_protocol(&pdu_udp);
}
