/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "mediascanner/locale.h"

// Boost C++
#include <boost/locale/conversion.hpp>
#include <boost/locale/generator.hpp>

// Standard Library
#include <iostream>
#include <string>

// Media Scanner
#include "mediascanner/logging.h"

// Standard Library
using std::locale;
using std::string;

namespace mediascanner {

static const logging::Domain kError("error/locale", logging::error());

namespace internal {

void PrintLocaleWarning(const std::locale &locale) {
    const std::string locale_name = locale.name();
    kError("WARNING: Generating Unicode facets for {1} locale. "
           "Please call mediascanner::SetupLocale().") % locale_name;
}

} // namespace internal

bool CheckLocaleFacets(const std::locale &loc) {
    return std::has_facet<boost::locale::converter<char> >(loc)
            && std::has_facet<boost::locale::converter<wchar_t> >(loc);
}

locale SetupLocale(const string &id) {
    const locale new_locale = SetupLocale(locale(), id);
    locale::global(new_locale);
    return new_locale;
}

// Ensure the passed locale has the required Boost facets.
locale SetupLocale(const locale &base, const string &id) {
    static const boost::locale::generator generator;
    return generator.generate(base, id);
}

} // namespace mediascanner
