/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009-2011 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/sysctl.h>
#include <sys/socket.h>
#include <net/if.h>
#include <net/if_mib.h>
#include <string.h>
#include <stdlib.h>
#include "utils.h"

static void assign(struct sa_net_interface* __restrict__ dst, struct ifmibdata* __restrict__ if_data) SA_NONNULL;

__thread int n_mib[6] = { CTL_NET, PF_LINK, NETLINK_GENERIC, IFMIB_IFDATA, 0, IFDATA_GENERAL };
__thread struct ifmibdata* buffer_n;
__thread uint16_t buffer_size_n;
__thread uint16_t number_interfaces; // the actual number of interfaces

int sa_open_net(void) {
	buffer_n = NULL;
	buffer_size_n = 0;
	number_interfaces = 0;
	return 0;
}

int sa_close_net(void) {
	if (buffer_n != NULL)
		free(buffer_n);
	return 0;
}

int sa_count_net_interfaces(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	u_int count;
	size_t len = sizeof count;
	if (sysctlbyname("net.link.generic.system.ifcount", &count, &len, NULL, 0) == -1)
		return ENOSYS;

	*number = count;
	return 0;
}

int sa_reset_net_interfaces() {
	size_t len = sizeof(struct ifmibdata);
	n_mib[4] = 1; // i know, it's 1
	number_interfaces = 0;

	while (1) {
		if (number_interfaces * len >= buffer_size_n) {
			size_t size = (number_interfaces + 1) * len;
			safe_realloc(&buffer_n, &size);
			buffer_size_n = size;
			if (buffer_n == NULL)
				return ENOMEM;
		}

		if (sysctl(n_mib, 6, &buffer_n[number_interfaces], &len, NULL, 0) == -1)
			break;

		++n_mib[4];
		++number_interfaces;
	}

	return 0;
}

int sa_get_net_interfaces_ids(char* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	*written = 0;
	uint16_t i;
	for (i = 0; i < number_interfaces; ++i) {
		if (i == dst_size)
			return ENOMEM;

		strlcpy(&dst[i * SA_NET_INTERFACE_NAME], buffer_n[i].ifmd_name, SA_NET_INTERFACE_NAME);
		++(*written);
	}
	return 0;
}

int sa_get_net_interface(char* name, struct sa_net_interface* dst) {
	if (name == NULL || dst == NULL)
		return EINVAL;

	uint16_t i;
	for (i = 0; i < number_interfaces; ++i) {
		if (strncmp(buffer_n[i].ifmd_name, name, sizeof buffer_n->ifmd_name) != 0)
			continue;

		assign(dst, &buffer_n[i]);
		return 0;
	}
	return ENODEV;
}

int sa_get_net_interfaces(struct sa_net_interface* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	*written = 0;
	uint16_t i;
	for (i = 0; i < number_interfaces; ++i) {
		if (i == dst_size)
			return ENOMEM;

		assign(&dst[i], &buffer_n[i]);
		++(*written);
	}
	return 0;
}

static void assign(struct sa_net_interface* __restrict__ dst, struct ifmibdata* __restrict__ if_data) {
	strlcpy(dst->name, if_data->ifmd_name, sizeof dst->name);
	dst->received_bytes = if_data->ifmd_data.ifi_ibytes;
	dst->received_packets = if_data->ifmd_data.ifi_ipackets;
	dst->received_errors = if_data->ifmd_data.ifi_ierrors;
	dst->received_drop = if_data->ifmd_data.ifi_iqdrops;
	dst->received_multicast = if_data->ifmd_data.ifi_imcasts;
	dst->sent_bytes = if_data->ifmd_data.ifi_obytes;
	dst->sent_packets = if_data->ifmd_data.ifi_opackets;
	dst->sent_errors = if_data->ifmd_data.ifi_oerrors;
	dst->sent_multicast = if_data->ifmd_data.ifi_omcasts;
}
