/* ptypeTable.h
 */
#ifndef OSL_PTYPETABLE_H
#define OSL_PTYPETABLE_H

#include "osl/config.h"
#include "osl/ptype.h"
#include "osl/ptypeTraits.h"
#include "osl/effectContent.h"
#include "osl/direction.h"
#include "osl/square.h"
#include "osl/misc/carray.h"
#include "osl/misc/carray2d.h"
#include "osl/offset32.h"

namespace osl
{
  class PtypeTable
  {
  private:
    CArray<mask_t, PTYPE_SIZE> numMaskLows;
    CArray<int, PTYPE_SIZE> numIndices;
    CArray<const char *, PTYPE_SIZE> names;
    CArray<const char *, PTYPE_SIZE> csaNames;
    CArray<bool, PTYPE_SIZE> betterToPromote;
    CArray<int, PTYPE_SIZE> moveMasks;
    CArray<int, PTYPE_SIZE> indexMins;
    CArray<int, PTYPE_SIZE> indexLimits;

    CArray2d<int, 2, PTYPE_SIZE> canDropLimit;
    // これらの2次元配列は2^nにそろえておいた方が速い．
    CArray2d<EffectContent,PTYPEO_SIZE,Offset32::SIZE> effectTable;
    CArray2d<EffectContent,PTYPEO_SIZE,Offset32::SIZE> effectTableNotLongU;
    CArray2d<unsigned int, 2, SHORT_DIRECTION_SIZE> shortMoveMask;
    
    template<Ptype T> void initPtypeSub(Int2Type<false> isBasic);
    template<Ptype T> void initPtypeSub(Int2Type<true> isBasic);
    template<Ptype T> void initPtype();
  public:
    PtypeTable();
  private:
    void init();
  public:
    unsigned int getShortMoveMask(Player p,PtypeO ptypeo,Direction dir) const
    {
      return shortMoveMask[playerToIndex(p)][static_cast<int>(dir)] &
	(1<<(ptypeo-PTYPEO_MIN));
    }
    mask_t getMaskLow(Ptype ptype) const
    {
      return numMaskLows[ptype];
    }
#if OSL_WORDSIZE == 64
    int getIndex(Ptype) const
    {
      return 0;
    }
#elif OSL_WORDSIZE == 32
    int getIndex(Ptype ptype) const
    {
      return numIndices[ptype];
    }
#endif
    /**
     * 遅くて良い?
     */
    bool hasLongMove(Ptype ptype) const
    {
      return getIndexMin(unpromote(ptype))>=32;
    }
    bool isBetterToPromote(Ptype ptype) const
    {
      return betterToPromote[ptype];
    }
    int getCanDropLimit(Player player,Ptype ptype) const
    {
      assert(isValid(ptype) && !isPromoted(ptype));
      return canDropLimit[playerToIndex(player)][ptype];
    }

  private:
    bool canDropTo(Ptype ptype, Square pos, Int2Type<BLACK>) const
    {
      return pos.y() >= getCanDropLimit(BLACK,ptype);
    }
    bool canDropTo(Ptype ptype, Square pos, Int2Type<WHITE>) const
    {
      return pos.y() <= getCanDropLimit(WHITE,ptype);
    }
  public:
    bool canDropTo(Player pl, Ptype ptype, Square pos) const
    {
      if (pl == BLACK)
	return canDropTo(ptype, pos, Int2Type<BLACK>());
      else
	return canDropTo(ptype, pos, Int2Type<WHITE>());
    }

    const char *getName(Ptype ptype) const
    {
      return names[ptype];
    }
    const char *getCsaName(Ptype ptype) const
    {
      return csaNames[ptype];
    }
    int getMoveMask(Ptype ptype) const
    {
      return moveMasks[ptype];
    }
    int getIndexMin(Ptype ptype) const
    {
      assert(isBasic(ptype));
      return indexMins[ptype];
    }
    int getIndexLimit(Ptype ptype) const
    {
      assert(isBasic(ptype));
      return indexLimits[ptype];
    }
    static int getKingIndex(Player p) 
    {
      assert(isValid(p));
      if (p==BLACK)
	return KingTraits<BLACK>::index;
      else
	return KingTraits<WHITE>::index;
    }
    /** 
     * fromにいるptypeoがtoに利きを持つか?
     * @param ptypeo - 駒の種類
     * @param from - 駒の位置
     * @param to - 利きをチェックするマスの位置
     */
    const EffectContent getEffect(PtypeO ptypeo,Square from, Square to) const
    {
      assert(from.isOnBoard() && to.isOnBoard());
      return getEffect(ptypeo,Offset32(to,from));
    }
    const EffectContent& getEffect(PtypeO ptypeo,Offset32 offset32) const
    {
      assert(isValidPtypeO(ptypeo));
      return effectTable[ptypeo-PTYPEO_MIN][offset32.index()];
    }
  private:
    EffectContent& effect(PtypeO ptypeo,Offset32 offset32)
    {
      assert(isValidPtypeO(ptypeo));
      const int i1 = ptypeo-PTYPEO_MIN;
      const int i2 = offset32.index();
      return effectTable[i1][i2];
    }
  public:
    /** ptypeo が，自分から offset のところに効きを持つか? U除く */
    const EffectContent
    getEffectNotLongU(PtypeO ptypeo, Square from, Square to) const
    {
      assert(isValidPtypeO(ptypeo));
      assert(from.isOnBoard() && to.isOnBoard());
      Offset32 offset32=Offset32(to,from);
      return effectTableNotLongU[ptypeo-PTYPEO_MIN][offset32.index()];
    }
    bool hasUnblockableEffect(PtypeO attacker, Square from, Square to) const
    {
      const EffectContent effect = getEffect(attacker, from, to);
      return effect.hasUnblockableEffect();
    }
  };
  
  extern const PtypeTable Ptype_Table;

} // namespace osl


#endif /* OSL_PTYPETABLE_H */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
