/*
 * This program is released under the Common Public License V1.0
 *
 * You should have received a copy of Common Public License V1.0 along with
 * with this program.
 *
 * Copyright IBM Corp. 2023
 */

#ifndef ECDSA_TEST_H
# define ECDSA_TEST_H

#include <openssl/obj_mac.h>

struct ecdsa_kat_tv {
	unsigned int nid;
	unsigned char *x;
	unsigned char *y;
	unsigned char *k;
	unsigned char *d;
	unsigned char *hash;
	unsigned int hash_len;
	unsigned char *sig;
	unsigned int sig_len;
};

/*
 * Supported curves are:
 *
 *   NID  Name (OpenSSL)         Elliptic Curve       D Length (bytes)
 *   -----------------------------------------------------------------
 *   409  NID_X9_62_prime192v    secp192r1            24
 *   713  NID_secp224r1          secp224r1            28
 *   415  NID_X9_62_prime256v1   secp256r1            32
 *   715  NID_secp384r1          secp384r1            48
 *   716  NID_secp521r1          secp521r1            66
 *   921  NID_brainpoolP160r1    brainpoolP160r1      20
 *   923  NID_brainpoolP192r1    brainpoolP192r1      24
 *   925  NID_brainpoolP224r1    brainpoolP224r1      28
 *   927  NID_brainpoolP256r1    brainpoolP256r1      32
 *   929  NID_brainpoolP320r1    brainpoolP320r1      40
 *   931  NID_brainpoolP384r1    brainpoolP384r1      48
 *   933  NID_brainpoolP512r1    brainpoolP512r1      64
 *
 * Brainpool KATs created with OpenSSL 3.0.2
 */
const struct ecdsa_kat_tv ECDSA_TV[] = {
	{
		.nid = NID_X9_62_prime192v1,
		.x = (unsigned char []) {
			0x07, 0x00, 0x8e, 0xa4, 0x0b, 0x08, 0xdb, 0xe7,
			0x64, 0x32, 0x09, 0x6e, 0x80, 0xa2, 0x49, 0x4c,
			0x94, 0x98, 0x2d, 0x2d, 0x5b, 0xcf, 0x98, 0xe6
		},
		.y = (unsigned char []) {
			0x76, 0xfa, 0xb6, 0x81, 0xd0, 0x0b, 0x41, 0x4e,
			0xa6, 0x36, 0xba, 0x21, 0x5d, 0xe2, 0x6d, 0x98,
			0xc4, 0x1b, 0xd7, 0xf2, 0xe4, 0xd6, 0x54, 0x77
		},
		.k = (unsigned char []) {
			0xcb, 0x0a, 0xbc, 0x70, 0x43, 0xa1, 0x07, 0x83,
			0x68, 0x45, 0x56, 0xfb, 0x12, 0xc4, 0x15, 0x4d,
			0x57, 0xbc, 0x31, 0xa2, 0x89, 0x68, 0x5f, 0x25
		},
		.d = (unsigned char []) {
			0xe1, 0x4f, 0x37, 0xb3, 0xd1, 0x37, 0x4f, 0xf8,
			0xb0, 0x3f, 0x41, 0xb9, 0xb3, 0xfd, 0xd2, 0xf0,
			0xeb, 0xcc, 0xf2, 0x75, 0xd6, 0x60, 0xd7, 0xf3
		},
		.hash = (unsigned char []) {
			0x60, 0x80, 0x79, 0x42, 0x3f, 0x12, 0x42, 0x1d,
			0xe6, 0x16, 0xb7, 0x49, 0x3e, 0xbe, 0x55, 0x1c,
			0xf4, 0xd6, 0x5b, 0x92
		},
		.hash_len = 20,
		.sig = (unsigned char []) {
			0x69, 0x94, 0xd9, 0x62, 0xbd, 0xd0, 0xd7, 0x93,
			0xff, 0xdd, 0xf8, 0x55, 0xec, 0x5b, 0xf2, 0xf9,
			0x1a, 0x96, 0x98, 0xb4, 0x62, 0x58, 0xa6, 0x3e,
			0x02, 0xba, 0x64, 0x65, 0xa2, 0x34, 0x90, 0x37,
			0x44, 0xab, 0x02, 0xbc, 0x85, 0x21, 0x40, 0x5b,
			0x73, 0xcf, 0x5f, 0xc0, 0x0e, 0x1a, 0x9f, 0x41
		},
		.sig_len = 48,
	},
	{
		.nid = NID_secp224r1,
		.x = (unsigned char []) {
			0x60, 0x54, 0x95, 0x75, 0x6e, 0x6e, 0x88, 0xf1,
			0xd0, 0x7a, 0xe5, 0xf9, 0x87, 0x87, 0xaf, 0x9b,
			0x4d, 0xa8, 0xa6, 0x41, 0xd1, 0xa9, 0x49, 0x2a,
			0x12, 0x17, 0x4e, 0xab
		},
		.y = (unsigned char []) {
			0xf5, 0xcc, 0x73, 0x3b, 0x17, 0xde, 0xcc, 0x80,
			0x6e, 0xf1, 0xdf, 0x86, 0x1a, 0x42, 0x50, 0x5d,
			0x0a, 0xf9, 0xef, 0x7c, 0x3d, 0xf3, 0x95, 0x9b,
			0x8d, 0xfc, 0x66, 0x69
		},
		.k = (unsigned char []) {
			0xd9, 0xa5, 0xa7, 0x32, 0x81, 0x17, 0xf4, 0x8b,
			0x4b, 0x8d, 0xd8, 0xc1, 0x7d, 0xae, 0x72, 0x2e,
			0x75, 0x6b, 0x3f, 0xf6, 0x4b, 0xd2, 0x9a, 0x52,
			0x71, 0x37, 0xee, 0xc0
		},
		.d = (unsigned char []) {
			0x16, 0x79, 0x7b, 0x5c, 0x0c, 0x7e, 0xd5, 0x46,
			0x1e, 0x2f, 0xf1, 0xb8, 0x8e, 0x6e, 0xaf, 0xa0,
			0x3c, 0x0f, 0x46, 0xbf, 0x07, 0x20, 0x00, 0xdf,
			0xc8, 0x30, 0xd6, 0x15
		},
		.hash = (unsigned char []) {
			0x07, 0xeb, 0x2a, 0x50, 0xbf, 0x70, 0xee, 0xe8,
			0x74, 0x67, 0x60, 0x06, 0x14, 0xa4, 0x90, 0xe7,
			0x60, 0x04, 0x37, 0xd0, 0x77, 0xec, 0x65, 0x1a,
			0x27, 0xe6, 0x5e, 0x67
		},
		.hash_len = 28,
		.sig = (unsigned char []) {
			0x2f, 0xc2, 0xcf, 0xf8, 0xcd, 0xd4, 0x86, 0x6b,
			0x1d, 0x74, 0xe4, 0x5b, 0x07, 0xd3, 0x33, 0xaf,
			0x46, 0xb7, 0xaf, 0x08, 0x88, 0x04, 0x9d, 0x0f,
			0xdb, 0xc7, 0xb0, 0xd6, 0x8d, 0x9c, 0xc4, 0xc8,
			0xea, 0x93, 0xe0, 0xfd, 0x9d, 0x64, 0x31, 0xb9,
			0xa1, 0xfd, 0x99, 0xb8, 0x8f, 0x28, 0x17, 0x93,
			0x39, 0x63, 0x21, 0xb1, 0x1d, 0xac, 0x41, 0xeb
		},
		.sig_len = 56,
	},
	{
		.nid = NID_X9_62_prime256v1,
		.x = (unsigned char []) {
			0x1c, 0xcb, 0xe9, 0x1c, 0x07, 0x5f, 0xc7, 0xf4,
			0xf0, 0x33, 0xbf, 0xa2, 0x48, 0xdb, 0x8f, 0xcc,
			0xd3, 0x56, 0x5d, 0xe9, 0x4b, 0xbf, 0xb1, 0x2f,
			0x3c, 0x59, 0xff, 0x46, 0xc2, 0x71, 0xbf, 0x83,
		},
		.y = (unsigned char []) {
			0xce, 0x40, 0x14, 0xc6, 0x88, 0x11, 0xf9, 0xa2,
			0x1a, 0x1f, 0xdb, 0x2c, 0x0e, 0x61, 0x13, 0xe0,
			0x6d, 0xb7, 0xca, 0x93, 0xb7, 0x40, 0x4e, 0x78,
			0xdc, 0x7c, 0xcd, 0x5c, 0xa8, 0x9a, 0x4c, 0xa9,
		},
		.k = (unsigned char []) {
			0x94, 0xa1, 0xbb, 0xb1, 0x4b, 0x90, 0x6a, 0x61,
			0xa2, 0x80, 0xf2, 0x45, 0xf9, 0xe9, 0x3c, 0x7f,
			0x3b, 0x4a, 0x62, 0x47, 0x82, 0x4f, 0x5d, 0x33,
			0xb9, 0x67, 0x07, 0x87, 0x64, 0x2a, 0x68, 0xde,
		},
		.d = (unsigned char []) {
			0x51, 0x9b, 0x42, 0x3d, 0x71, 0x5f, 0x8b, 0x58,
			0x1f, 0x4f, 0xa8, 0xee, 0x59, 0xf4, 0x77, 0x1a,
			0x5b, 0x44, 0xc8, 0x13, 0x0b, 0x4e, 0x3e, 0xac,
			0xca, 0x54, 0xa5, 0x6d, 0xda, 0x72, 0xb4, 0x64,
		},
		.hash = (unsigned char []) {
			0x44, 0xac, 0xf6, 0xb7, 0xe3, 0x6c, 0x13, 0x42,
			0xc2, 0xc5, 0x89, 0x72, 0x04, 0xfe, 0x09, 0x50,
			0x4e, 0x1e, 0x2e, 0xfb, 0x1a, 0x90, 0x03, 0x77,
			0xdb, 0xc4, 0xe7, 0xa6, 0xa1, 0x33, 0xec, 0x56
		},
		.hash_len = 32,
		.sig = (unsigned char []) {
			0xf3, 0xac, 0x80, 0x61, 0xb5, 0x14, 0x79, 0x5b,
			0x88, 0x43, 0xe3, 0xd6, 0x62, 0x95, 0x27, 0xed,
			0x2a, 0xfd, 0x6b, 0x1f, 0x6a, 0x55, 0x5a, 0x7a,
			0xca, 0xbb, 0x5e, 0x6f, 0x79, 0xc8, 0xc2, 0xac,
			0x8b, 0xf7, 0x78, 0x19, 0xca, 0x05, 0xa6, 0xb2,
			0x78, 0x6c, 0x76, 0x26, 0x2b, 0xf7, 0x37, 0x1c,
			0xef, 0x97, 0xb2, 0x18, 0xe9, 0x6f, 0x17, 0x5a,
			0x3c, 0xcd, 0xda, 0x2a, 0xcc, 0x05, 0x89, 0x03,
		},
		.sig_len = 64,
	},
	{
		.nid = NID_secp384r1,
		.x = (unsigned char []) {
			0xc2, 0xb4, 0x79, 0x44, 0xfb, 0x5d, 0xe3, 0x42,
			0xd0, 0x32, 0x85, 0x88, 0x01, 0x77, 0xca, 0x5f,
			0x7d, 0x0f, 0x2f, 0xca, 0xd7, 0x67, 0x8c, 0xce,
			0x42, 0x29, 0xd6, 0xe1, 0x93, 0x2f, 0xca, 0xc1,
			0x1b, 0xfc, 0x3c, 0x3e, 0x97, 0xd9, 0x42, 0xa3,
			0xc5, 0x6b, 0xf3, 0x41, 0x23, 0x01, 0x3d, 0xbf,
		},
		.y = (unsigned char []) {
			0x37, 0x25, 0x79, 0x06, 0xa8, 0x22, 0x38, 0x66,
			0xed, 0xa0, 0x74, 0x3c, 0x51, 0x96, 0x16, 0xa7,
			0x6a, 0x75, 0x8a, 0xe5, 0x8a, 0xee, 0x81, 0xc5,
			0xfd, 0x35, 0xfb, 0xf3, 0xa8, 0x55, 0xb7, 0x75,
			0x4a, 0x36, 0xd4, 0xa0, 0x67, 0x2d, 0xf9, 0x5d,
			0x6c, 0x44, 0xa8, 0x1c, 0xf7, 0x62, 0x0c, 0x2d,
		},
		.k = (unsigned char []) {
			0xdc, 0xed, 0xab, 0xf8, 0x59, 0x78, 0xe0, 0x90,
			0xf7, 0x33, 0xc6, 0xe1, 0x66, 0x46, 0xfa, 0x34,
			0xdf, 0x9d, 0xed, 0x6e, 0x5c, 0xe2, 0x8c, 0x66,
			0x76, 0xa0, 0x0f, 0x58, 0xa2, 0x52, 0x83, 0xdb,
			0x88, 0x85, 0xe1, 0x6c, 0xe5, 0xbf, 0x97, 0xf9,
			0x17, 0xc8, 0x1e, 0x1f, 0x25, 0xc9, 0xc7, 0x71,
		},
		.d = (unsigned char []) {
			0x20, 0x1b, 0x43, 0x2d, 0x8d, 0xf1, 0x43, 0x24,
			0x18, 0x2d, 0x62, 0x61, 0xdb, 0x3e, 0x4b, 0x3f,
			0x46, 0xa8, 0x28, 0x44, 0x82, 0xd5, 0x2e, 0x37,
			0x0d, 0xa4, 0x1e, 0x6c, 0xbd, 0xf4, 0x5e, 0xc2,
			0x95, 0x2f, 0x5d, 0xb7, 0xcc, 0xbc, 0xe3, 0xbc,
			0x29, 0x44, 0x9f, 0x4f, 0xb0, 0x80, 0xac, 0x97,
		},
		.hash = (unsigned char []) {
			0x31, 0xa4, 0x52, 0xd6, 0x16, 0x4d, 0x90, 0x4b,
			0xb5, 0x72, 0x4c, 0x87, 0x82, 0x80, 0x23, 0x1e,
			0xae, 0x70, 0x5c, 0x29, 0xce, 0x9d, 0x4b, 0xc7,
			0xd5, 0x8e, 0x02, 0x0e, 0x10, 0x85, 0xf1, 0x7e,
			0xeb, 0xcc, 0x1a, 0x38, 0xf0, 0xed, 0x0b, 0xf2,
			0xb3, 0x44, 0xd8, 0x1f, 0xbd, 0x89, 0x68, 0x25,
		},
		.hash_len = 48,
		.sig = (unsigned char []) {
			0x50, 0x83, 0x5a, 0x92, 0x51, 0xba, 0xd0, 0x08,
			0x10, 0x61, 0x77, 0xef, 0x00, 0x4b, 0x09, 0x1a,
			0x1e, 0x42, 0x35, 0xcd, 0x0d, 0xa8, 0x4f, 0xff,
			0x54, 0x54, 0x2b, 0x0e, 0xd7, 0x55, 0xc1, 0xd6,
			0xf2, 0x51, 0x60, 0x9d, 0x14, 0xec, 0xf1, 0x8f,
			0x9e, 0x1d, 0xdf, 0xe6, 0x9b, 0x94, 0x6e, 0x32,
			0x04, 0x75, 0xf3, 0xd3, 0x0c, 0x64, 0x63, 0xb6,
			0x46, 0xe8, 0xd3, 0xbf, 0x24, 0x55, 0x83, 0x03,
			0x14, 0x61, 0x1c, 0xbd, 0xe4, 0x04, 0xbe, 0x51,
			0x8b, 0x14, 0x46, 0x4f, 0xdb, 0x19, 0x5f, 0xdc,
			0xc9, 0x2e, 0xb2, 0x22, 0xe6, 0x1f, 0x42, 0x6a,
			0x4a, 0x59, 0x2c, 0x00, 0xa6, 0xa8, 0x97, 0x21,
		},
		.sig_len = 96,
	},
	{
		.nid = NID_secp521r1,
		.x = (unsigned char []) {
			0x00, 0x61, 0x38, 0x7f, 0xd6, 0xb9, 0x59, 0x14,
			0xe8, 0x85, 0xf9, 0x12, 0xed, 0xfb, 0xb5, 0xfb,
			0x27, 0x46, 0x55, 0x02, 0x7f, 0x21, 0x6c, 0x40,
			0x91, 0xca, 0x83, 0xe1, 0x93, 0x36, 0x74, 0x0f,
			0xd8, 0x1a, 0xed, 0xfe, 0x04, 0x7f, 0x51, 0xb4,
			0x2b, 0xdf, 0x68, 0x16, 0x11, 0x21, 0x01, 0x3e,
			0x0d, 0x55, 0xb1, 0x17, 0xa1, 0x4e, 0x43, 0x03,
			0xf9, 0x26, 0xc8, 0xde, 0xbb, 0x77, 0xa7, 0xfd,
			0xaa, 0xd1
		},
		.y = (unsigned char []) {
			0x00, 0xe7, 0xd0, 0xc7, 0x5c, 0x38, 0x62, 0x6e,
			0x89, 0x5c, 0xa2, 0x15, 0x26, 0xb9, 0xf9, 0xfd,
			0xf8, 0x4d, 0xce, 0xcb, 0x93, 0xf2, 0xb2, 0x33,
			0x39, 0x05, 0x50, 0xd2, 0xb1, 0x46, 0x3b, 0x7e,
			0xe3, 0xf5, 0x8d, 0xf7, 0x34, 0x64, 0x35, 0xff,
			0x04, 0x34, 0x19, 0x95, 0x83, 0xc9, 0x7c, 0x66,
			0x5a, 0x97, 0xf1, 0x2f, 0x70, 0x6f, 0x23, 0x57,
			0xda, 0x4b, 0x40, 0x28, 0x8d, 0xef, 0x88, 0x8e,
			0x59, 0xe6
		},
		.k = (unsigned char []) {
			0x00, 0x3a, 0xf5, 0xab, 0x6c, 0xaa, 0x29, 0xa6,
			0xde, 0x86, 0xa5, 0xba, 0xb9, 0xaa, 0x83, 0xc3,
			0xb1, 0x6a, 0x17, 0xff, 0xcd, 0x52, 0xb5, 0xc6,
			0x0c, 0x76, 0x9b, 0xe3, 0x05, 0x3c, 0xdd, 0xde,
			0xac, 0x60, 0x81, 0x2d, 0x12, 0xfe, 0xcf, 0x46,
			0xcf, 0xe1, 0xf3, 0xdb, 0x9a, 0xc9, 0xdc, 0xf8,
			0x81, 0xfc, 0xec, 0x3f, 0x0a, 0xa7, 0x33, 0xd4,
			0xec, 0xbb, 0x83, 0xc7, 0x59, 0x3e, 0x86, 0x4c,
			0x6d, 0xf1
		},
		.d = (unsigned char []) {
			0x00, 0xf7, 0x49, 0xd3, 0x27, 0x04, 0xbc, 0x53,
			0x3c, 0xa8, 0x2c, 0xef, 0x0a, 0xcf, 0x10, 0x3d,
			0x8f, 0x4f, 0xba, 0x67, 0xf0, 0x8d, 0x26, 0x78,
			0xe5, 0x15, 0xed, 0x7d, 0xb8, 0x86, 0x26, 0x7f,
			0xfa, 0xf0, 0x2f, 0xab, 0x00, 0x80, 0xdc, 0xa2,
			0x35, 0x9b, 0x72, 0xf5, 0x74, 0xcc, 0xc2, 0x9a,
			0x0f, 0x21, 0x8c, 0x86, 0x55, 0xc0, 0xcc, 0xcf,
			0x9f, 0xee, 0x6c, 0x5e, 0x56, 0x7a, 0xa1, 0x4c,
			0xb9, 0x26
		},
		.hash = (unsigned char []) {
			0x65, 0xf8, 0x34, 0x08, 0x09, 0x22, 0x61, 0xbd,
			0xa5, 0x99, 0x38, 0x9d, 0xf0, 0x33, 0x82, 0xc5,
			0xbe, 0x01, 0xa8, 0x1f, 0xe0, 0x0a, 0x36, 0xf3,
			0xf4, 0xbb, 0x65, 0x41, 0x26, 0x3f, 0x80, 0x16,
			0x27, 0xc4, 0x40, 0xe5, 0x08, 0x09, 0x71, 0x2b,
			0x0c, 0xac, 0xe7, 0xc2, 0x17, 0xe6, 0xe5, 0x05,
			0x1a, 0xf8, 0x1d, 0xe9, 0xbf, 0xec, 0x32, 0x04,
			0xdc, 0xd6, 0x3c, 0x4f, 0x9a, 0x74, 0x10, 0x47,
		},
		.hash_len = 64,
		.sig = (unsigned char []) {
			0x00, 0x4d, 0xe8, 0x26, 0xea, 0x70, 0x4a, 0xd1,
			0x0b, 0xc0, 0xf7, 0x53, 0x8a, 0xf8, 0xa3, 0x84,
			0x3f, 0x28, 0x4f, 0x55, 0xc8, 0xb9, 0x46, 0xaf,
			0x92, 0x35, 0xaf, 0x5a, 0xf7, 0x4f, 0x2b, 0x76,
			0xe0, 0x99, 0xe4, 0xbc, 0x72, 0xfd, 0x79, 0xd2,
			0x8a, 0x38, 0x0f, 0x8d, 0x4b, 0x4c, 0x91, 0x9a,
			0xc2, 0x90, 0xd2, 0x48, 0xc3, 0x79, 0x83, 0xba,
			0x05, 0xae, 0xa4, 0x2e, 0x2d, 0xd7, 0x9f, 0xdd,
			0x33, 0xe8,
			0x00, 0x87, 0x48, 0x8c, 0x85, 0x9a, 0x96, 0xfe,
			0xa2, 0x66, 0xea, 0x13, 0xbf, 0x6d, 0x11, 0x4c,
			0x42, 0x9b, 0x16, 0x3b, 0xe9, 0x7a, 0x57, 0x55,
			0x90, 0x86, 0xed, 0xb6, 0x4a, 0xed, 0x4a, 0x18,
			0x59, 0x4b, 0x46, 0xfb, 0x9e, 0xfc, 0x7f, 0xd2,
			0x5d, 0x8b, 0x2d, 0xe8, 0xf0, 0x9c, 0xa0, 0x58,
			0x7f, 0x54, 0xbd, 0x28, 0x72, 0x99, 0xf4, 0x7b,
			0x2f, 0xf1, 0x24, 0xaa, 0xc5, 0x66, 0xe8, 0xee,
			0x3b, 0x43,
		},
		.sig_len = 132,
	},
	{
		.nid = NID_brainpoolP160r1,
		.x = (unsigned char []) {
			0x48, 0x1f, 0x7c, 0x83, 0x47, 0xda, 0x67, 0x62,
			0xeb, 0xff, 0x60, 0xe4, 0xea, 0xee, 0xdd, 0xdf,
			0x10, 0xfa, 0x77, 0xfd
		},
		.y = (unsigned char []) {
			0xd1, 0x83, 0x39, 0x29, 0x65, 0x9e, 0x8e, 0xbb,
			0x74, 0x86, 0xe4, 0x97, 0x80, 0xf0, 0x1f, 0x74,
			0x09, 0x4d, 0x90, 0x3a
		},
		.k = (unsigned char []) {},
		.d = (unsigned char []) {
			0x6f, 0xa7, 0x33, 0xb5, 0x02, 0x5b, 0x17, 0x9d,
			0x23, 0x62, 0x62, 0x09, 0x70, 0x91, 0x9a, 0x17,
			0x63, 0xfd, 0x94, 0x49
		},
		.hash = (unsigned char []) {
			0x14, 0x8b, 0x0c, 0xd0, 0x5d, 0xb8, 0xb5, 0xb9,
			0x58, 0xdf, 0xaa, 0x59, 0x7e, 0x61, 0x3d, 0x3c,
			0x16, 0x48, 0x87, 0xd3
		},
		.hash_len = 20,
		.sig = (unsigned char []) {
			0x20, 0xD7, 0x54, 0xBF, 0xF9, 0x58, 0x17, 0x4B,
			0x6A, 0x3D, 0x75, 0x3D, 0xFB, 0x26, 0x9F, 0x8F,
			0xA9, 0xEC, 0x8E, 0x41, 0x28, 0x87, 0xF2, 0x63,
			0x7C, 0x19, 0xEF, 0xDF, 0x44, 0xF1, 0x19, 0x8B,
			0xCF, 0x3A, 0xDF, 0x79, 0xCF, 0x5B, 0x63, 0xF8
		},
		.sig_len = 40,
	},
	{
		.nid = NID_brainpoolP192r1,
		.x = (unsigned char []) {
			0x94, 0xf7, 0x63, 0xfa, 0x41, 0x57, 0x59, 0x6a,
			0xcc, 0x91, 0xdf, 0xe7, 0xff, 0x0f, 0xe7, 0x9d,
			0x13, 0xa6, 0x4d, 0xf9, 0xf2, 0x53, 0xc2, 0x09
		},
		.y = (unsigned char []) {
			0x04, 0x94, 0x25, 0x78, 0xa9, 0x7a, 0x65, 0x9a,
			0x41, 0xf9, 0xa8, 0x6e, 0x27, 0x2b, 0x3d, 0x67,
			0xb7, 0xf8, 0xcc, 0x9c, 0xdd, 0x1b, 0x6f, 0xcb
		},
		.k = (unsigned char []) {},
		.d = (unsigned char []) {
			0x3a, 0xa6, 0xb6, 0xe5, 0x03, 0x76, 0x44, 0x2e,
			0xbf, 0x49, 0x7b, 0x4d, 0x9a, 0xc9, 0xf7, 0xa2,
			0x9f, 0x82, 0x02, 0x32, 0x08, 0xee, 0x28, 0x59
		},
		.hash = (unsigned char []) {
			0x14, 0x8b, 0x0c, 0xd0, 0x5d, 0xb8, 0xb5, 0xb9,
			0x58, 0xdf, 0xaa, 0x59, 0x7e, 0x61, 0x3d, 0x3c,
			0x16, 0x48, 0x87, 0xd3
		},
		.hash_len = 20,
		.sig = (unsigned char []) {
			0x28, 0xAE, 0x43, 0xF5, 0xBC, 0x8F, 0x38, 0x70,
			0x98, 0x7C, 0x42, 0xE9, 0x30, 0x05, 0x98, 0x73,
			0x67, 0x18, 0xA1, 0x67, 0x41, 0x56, 0x21, 0x04,
			0x44, 0x35, 0xE8, 0x19, 0xD3, 0x02, 0xC4, 0x55,
			0xD7, 0x88, 0xA6, 0xB7, 0xFC, 0xF2, 0x89, 0x82,
			0x4B, 0xDA, 0x37, 0x24, 0xA2, 0x01, 0xEC, 0x06
		},
		.sig_len = 48,
	},
	{
		.nid = NID_brainpoolP224r1,
		.x = (unsigned char []) {
			0x17, 0x16, 0x7f, 0xe8, 0x17, 0x82, 0xfa, 0xfc,
			0xe0, 0x88, 0xeb, 0xd6, 0x00, 0xea, 0x9b, 0x23,
			0x2a, 0xb6, 0xd7, 0x55, 0x09, 0x13, 0xaf, 0x3d,
			0x68, 0x5f, 0x1e, 0x47
		},
		.y = (unsigned char []) {
			0x71, 0xbd, 0x1d, 0xee, 0x29, 0x64, 0x59, 0x9e,
			0xd0, 0xac, 0x87, 0x19, 0x40, 0x3d, 0x66, 0x82,
			0x1e, 0x48, 0x04, 0x4d, 0xf7, 0x76, 0xf8, 0x4b,
			0xbf, 0x33, 0x22, 0x3d
		},
		.k = (unsigned char []) {},
		.d = (unsigned char []) {
			0x50, 0x76, 0x19, 0xc9, 0x59, 0x15, 0x9c, 0x72,
			0x36, 0x68, 0xdc, 0x19, 0xf6, 0x9f, 0xe1, 0x5e,
			0x54, 0x12, 0x32, 0xc6, 0x50, 0x7f, 0x0b, 0x10,
			0xc0, 0x27, 0x03, 0x02
		},
		.hash = (unsigned char []) {
			0x35, 0x05, 0x63, 0xa5, 0x7b, 0x91, 0x16, 0x83,
			0xbc, 0xf3, 0xbd, 0xd5, 0x8f, 0xe6, 0x64, 0x8a,
			0x12, 0x7a, 0xd7, 0xac, 0x9a, 0x9e, 0x5d, 0x73,
			0xde, 0xfe, 0x24, 0x33,
		},
		.hash_len = 28,
		.sig = (unsigned char []) {
			0x79, 0x23, 0xDA, 0x04, 0x43, 0x3C, 0x6B, 0xDE,
			0x71, 0x0D, 0xC4, 0xB7, 0x29, 0x83, 0xD9, 0xEE,
			0x43, 0xA4, 0x5E, 0xF1, 0xDC, 0x4B, 0x91, 0x0C,
			0xD2, 0x7E, 0xB6, 0x8C, 0xC8, 0x91, 0x73, 0x56,
			0x94, 0x8A, 0x15, 0x56, 0x2B, 0x71, 0xD1, 0x72,
			0xDE, 0x41, 0xB7, 0xA9, 0x46, 0x8D, 0xA8, 0x6C,
			0x25, 0x31, 0x56, 0x37, 0x36, 0xFB, 0x5D, 0xB7
		},
		.sig_len = 56,
	},
	{
		.nid = NID_brainpoolP256r1,
		.x = (unsigned char []) {
			0x17, 0x7f, 0xf4, 0x70, 0xb1, 0x76, 0x38, 0xea,
			0x8e, 0xca, 0xfc, 0x7a, 0xaa, 0x15, 0x94, 0x23,
			0xe4, 0x04, 0x4e, 0x3c, 0xa1, 0xc6, 0x6c, 0x18,
			0x6d, 0xaf, 0xbf, 0xc2, 0x5e, 0xaa, 0x01, 0x8d
		},
		.y = (unsigned char []) {
			0x89, 0x7a, 0xc3, 0x2c, 0xaa, 0xb8, 0x8d, 0x69,
			0xd9, 0xb6, 0x88, 0x13, 0x4c, 0x94, 0x2a, 0x0c,
			0x3b, 0x08, 0xd1, 0x80, 0x4a, 0x31, 0xc7, 0x80,
			0xb3, 0x30, 0x64, 0xc3, 0x91, 0x2c, 0xd0, 0xaf
		},
		.k = (unsigned char []) {},
		.d = (unsigned char []) {
			0x61, 0x63, 0x78, 0x1d, 0x5b, 0xf2, 0x78, 0xde,
			0x0e, 0xb2, 0x29, 0x68, 0x63, 0xbb, 0xfa, 0xaa,
			0xc5, 0x1e, 0x6e, 0xe3, 0x92, 0x75, 0xd9, 0x2d,
			0x03, 0x99, 0xe8, 0x79, 0x25, 0x78, 0x50, 0x65
		},
		.hash = (unsigned char []) {
			0x3b, 0xf0, 0xc4, 0x55, 0xa5, 0xfa, 0x03, 0x4b,
			0xf4, 0xd5, 0xdd, 0x83, 0x8b, 0x5c, 0x82, 0xee,
			0xe0, 0x73, 0xcb, 0xc6, 0xd0, 0xb2, 0xe8, 0x52,
			0x4e, 0x1c, 0x36, 0x9a, 0xd4, 0xbd, 0xa8, 0x6f
		},
		.hash_len = 32,
		.sig = (unsigned char []) {
			0x76, 0x21, 0x86, 0xBF, 0x14, 0xA8, 0xCE, 0x65,
			0x7C, 0x80, 0xB1, 0xFE, 0x7F, 0x8F, 0x96, 0x84,
			0xA6, 0xFB, 0x74, 0xD7, 0xA4, 0xAE, 0xD1, 0x74,
			0xCB, 0x36, 0x7E, 0x05, 0xE1, 0x5E, 0x04, 0x90,
			0x41, 0x59, 0x6B, 0xC7, 0xA2, 0xC6, 0x0A, 0x5A,
			0xCA, 0xAC, 0x79, 0x46, 0xB7, 0x0A, 0xEC, 0x3E,
			0x0F, 0x50, 0x51, 0xA3, 0x01, 0xEF, 0x19, 0x96,
			0xA8, 0xAE, 0x22, 0xAC, 0x10, 0xE9, 0x6E, 0x71
		},
		.sig_len = 64,
	},
	{
		.nid = NID_brainpoolP320r1,
		.x = (unsigned char []) {
			0x81, 0x92, 0x06, 0x3b, 0xa2, 0x3f, 0x62, 0x38,
			0x3a, 0xb1, 0x67, 0x90, 0x5d, 0xd4, 0x61, 0xee,
			0x0a, 0x5e, 0x55, 0x24, 0x9f, 0xe3, 0xf1, 0xb2,
			0x93, 0x0e, 0x24, 0xc2, 0xb6, 0x22, 0x1e, 0x99,
			0xf8, 0x8e, 0xd8, 0xcf, 0x26, 0x96, 0x92, 0xc7
		},
		.y = (unsigned char []) {
			0x98, 0x47, 0x01, 0xff, 0x72, 0xdd, 0x23, 0x36,
			0xf7, 0x7b, 0xd8, 0x19, 0x62, 0x29, 0xaa, 0x8e,
			0x71, 0x68, 0x22, 0xf7, 0x53, 0x4c, 0xf7, 0xf5,
			0xa1, 0x9b, 0x4b, 0x1b, 0x0e, 0x33, 0x9c, 0xa5,
			0x86, 0x54, 0xda, 0x5a, 0xc3, 0x5d, 0x23, 0x72
		},
		.k = (unsigned char []) {},
		.d = (unsigned char []) {
			0x0e, 0x8e, 0x4d, 0x61, 0x86, 0x37, 0x75, 0x2f,
			0x07, 0x1c, 0xc6, 0xec, 0x27, 0xa5, 0x59, 0xf9,
			0x64, 0x0d, 0x3c, 0xdb, 0xe2, 0xd8, 0x1c, 0x4a,
			0x5d, 0x92, 0x53, 0x58, 0x3a, 0x48, 0x6b, 0x39,
			0x7d, 0xc3, 0x44, 0xea, 0xcb, 0x4a, 0x48, 0x17
		},
		.hash = (unsigned char []) {
			0x3b, 0xf0, 0xc4, 0x55, 0xa5, 0xfa, 0x03, 0x4b,
			0xf4, 0xd5, 0xdd, 0x83, 0x8b, 0x5c, 0x82, 0xee,
			0xe0, 0x73, 0xcb, 0xc6, 0xd0, 0xb2, 0xe8, 0x52,
			0x4e, 0x1c, 0x36, 0x9a, 0xd4, 0xbd, 0xa8, 0x6f
		},
		.hash_len = 32,
		.sig = (unsigned char []) {
			0x94, 0x3C, 0x2A, 0x46, 0x3E, 0x57, 0xAC, 0xF7,
			0x3C, 0x12, 0xDD, 0xF1, 0x3A, 0x0B, 0x4F, 0x5D,
			0xFB, 0xF8, 0x40, 0x94, 0x58, 0x04, 0x94, 0x0A,
			0xCB, 0xAD, 0xD7, 0x16, 0xD9, 0xF7, 0x1B, 0x27,
			0xF8, 0x65, 0xA0, 0x82, 0xAA, 0x3D, 0xB7, 0x9B,
			0x18, 0x46, 0xEA, 0x6D, 0x48, 0xBF, 0x91, 0xAD,
			0x2A, 0xFE, 0x72, 0x3B, 0x15, 0xAB, 0x59, 0xFE,
			0x20, 0x30, 0xBE, 0x16, 0x94, 0xCD, 0x2C, 0x50,
			0xF0, 0x09, 0x99, 0x7F, 0x0E, 0x80, 0xFF, 0x7E,
			0x94, 0x1B, 0x29, 0xCE, 0xFD, 0x44, 0x60, 0xC6
		},
		.sig_len = 80,
	},
	{
		.nid = NID_brainpoolP384r1,
		.x = (unsigned char []) {
			0x0d, 0xd2, 0x85, 0x22, 0xcd, 0xcb, 0x49, 0x17,
			0xf8, 0xe6, 0x22, 0x69, 0x74, 0x42, 0x6f, 0xd3,
			0xa3, 0xa8, 0xa6, 0xe8, 0x5e, 0x53, 0x62, 0x43,
			0xe6, 0xc1, 0xf3, 0x46, 0x66, 0xa8, 0xef, 0xb5,
			0xdb, 0xb2, 0xfe, 0x15, 0x42, 0xe3, 0x04, 0x7a,
			0xd9, 0xa3, 0x80, 0x9d, 0x73, 0x5f, 0x7a, 0xeb
		},
		.y = (unsigned char []) {
			0x8b, 0x84, 0x7e, 0x1f, 0x77, 0x8f, 0x6e, 0x2e,
			0x57, 0x6f, 0xb0, 0x50, 0xd0, 0xed, 0x36, 0x8e,
			0xde, 0x1c, 0x95, 0x08, 0x0f, 0xf1, 0x27, 0x96,
			0x64, 0x6e, 0xc5, 0x27, 0xce, 0xa7, 0x3f, 0x46,
			0x2d, 0x4f, 0xec, 0xb6, 0x58, 0x4c, 0x23, 0xfa,
			0xf0, 0x03, 0xbc, 0xf4, 0xcc, 0x1a, 0xc0, 0x9c
			},
		.k = (unsigned char []) {},
		.d = (unsigned char []) {
			0x3c, 0xe8, 0x54, 0xdf, 0xc0, 0x47, 0xcf, 0xf4,
			0xe9, 0xc8, 0xc0, 0x88, 0x2d, 0x2a, 0xf1, 0x17,
			0x0c, 0x85, 0x5a, 0x21, 0x06, 0x12, 0x31, 0x10,
			0x86, 0x1d, 0x58, 0x52, 0xbe, 0x87, 0x39, 0x31,
			0x53, 0xe5, 0xe8, 0x6c, 0x7a, 0x78, 0xde, 0x2e,
			0x74, 0xa1, 0xc0, 0x6f, 0x2e, 0xef, 0x3f, 0xe9
		},
		.hash = (unsigned char []) {
			0x62, 0xce, 0x2c, 0xdc, 0xcb, 0x8e, 0x36, 0xbc,
			0x66, 0x57, 0x46, 0x72, 0x8c, 0x04, 0xe5, 0xa2,
			0xc8, 0x22, 0x36, 0xc9, 0x40, 0x45, 0xc2, 0xaa,
			0xec, 0x15, 0xfc, 0x49, 0x31, 0xac, 0x99, 0x69,
			0x5c, 0xa7, 0xe9, 0x77, 0x57, 0x0e, 0x49, 0x52,
			0xb6, 0x8f, 0x7c, 0x67, 0x7b, 0x29, 0xd3, 0x43
		},
		.hash_len = 48,
		.sig = (unsigned char []) {
			0x7E, 0xD9, 0x8A, 0xCD, 0xF5, 0x2B, 0x17, 0xD3,
			0x0A, 0xE9, 0x64, 0x8C, 0xBD, 0x8D, 0x1D, 0x49,
			0x5E, 0xF0, 0xDA, 0x41, 0x19, 0x47, 0x69, 0x57,
			0x05, 0x26, 0xCC, 0xDE, 0x76, 0xB6, 0xC3, 0xCF,
			0x8C, 0x3B, 0xE8, 0x7B, 0xD9, 0x9F, 0xC7, 0x08,
			0x41, 0xA1, 0x31, 0x07, 0xF3, 0x53, 0x81, 0xA5,
			0x8A, 0x8B, 0x74, 0x10, 0xE0, 0xBE, 0xE9, 0xBE,
			0xD9, 0xD8, 0xB3, 0xF9, 0x27, 0xBA, 0x55, 0x85,
			0xC9, 0x9E, 0x9B, 0xE7, 0x56, 0xF5, 0x41, 0xA1,
			0x86, 0xBC, 0x33, 0xC9, 0xCC, 0x87, 0x3F, 0xBA,
			0xA9, 0x00, 0x21, 0x4E, 0x1A, 0x07, 0x49, 0x48,
			0x67, 0xA9, 0x66, 0xDD, 0x6E, 0x6B, 0xEB, 0xAA
		},
		.sig_len = 96,
	},
	{
		.nid = NID_brainpoolP512r1,
		.x = (unsigned char []) {
			0x37, 0x7c, 0x5d, 0xff, 0xc8, 0x56, 0xf4, 0x08,
			0x6a, 0x03, 0x9e, 0xb2, 0xd0, 0xd5, 0x65, 0xc2,
			0xaa, 0xe0, 0xae, 0xd0, 0x4d, 0x8f, 0x3a, 0x32,
			0x02, 0x68, 0x64, 0x91, 0xb6, 0xd9, 0x44, 0x2d,
			0x77, 0xc2, 0xf6, 0xab, 0x36, 0xe3, 0x4e, 0x75,
			0xa2, 0x66, 0x1a, 0xc6, 0xa9, 0x88, 0x47, 0xa5,
			0x52, 0x3a, 0xd5, 0xa4, 0xe0, 0x5c, 0xb8, 0x90,
			0x57, 0xd5, 0xbd, 0x18, 0x6e, 0xdc, 0xbc, 0xfc
		},
		.y = (unsigned char []) {
			0x9d, 0x1c, 0xdc, 0xc4, 0x19, 0x2b, 0x89, 0x07,
			0x85, 0x11, 0xbe, 0x70, 0x5b, 0x77, 0x2d, 0x2a,
			0x01, 0x8b, 0x5b, 0x3b, 0x94, 0x54, 0x1f, 0xe4,
			0x3c, 0x58, 0x77, 0x0b, 0x75, 0x12, 0x1c, 0xc8,
			0xdc, 0xac, 0xc7, 0x5f, 0x76, 0x1e, 0xd5, 0x07,
			0xae, 0x97, 0xc1, 0xc1, 0xb1, 0x8b, 0x49, 0xc3,
			0x7e, 0xf4, 0x2c, 0xa4, 0x10, 0xf4, 0x74, 0x8c,
			0x08, 0x71, 0x8d, 0x88, 0x72, 0xc5, 0xcd, 0x99
		},
		.k = (unsigned char []) {},
		.d = (unsigned char []) {
			0x13, 0x74, 0x50, 0x7f, 0xea, 0xb8, 0x2d, 0x75,
			0x05, 0x3e, 0x35, 0x37, 0x6d, 0x01, 0x9d, 0xb0,
			0xdd, 0xf4, 0x20, 0x97, 0x81, 0x15, 0xb9, 0x63,
			0x17, 0xd2, 0x90, 0xd5, 0xce, 0x7a, 0x82, 0x39,
			0x02, 0xe8, 0xf2, 0xce, 0xb8, 0x10, 0xe6, 0xa8,
			0x05, 0x96, 0x74, 0x11, 0xde, 0x2e, 0x69, 0xe4,
			0x96, 0xa0, 0x36, 0x31, 0xa6, 0xc7, 0x78, 0x5f,
			0xef, 0xa9, 0x5b, 0x8d, 0x3e, 0x28, 0xc8, 0x45
		},
		.hash = (unsigned char []) {
			0x03, 0x3b, 0x9d, 0x33, 0xce, 0x42, 0xeb, 0x8e,
			0x8a, 0xa2, 0x70, 0xc9, 0xe1, 0xe6, 0xc7, 0xbc,
			0xe3, 0xe7, 0x32, 0xe4, 0xd9, 0x3b, 0xcf, 0x4a,
			0xeb, 0xe4, 0xc6, 0x7c, 0xa5, 0xf6, 0xce, 0xea,
			0x7d, 0x44, 0x71, 0x83, 0x17, 0x91, 0xda, 0x95,
			0x41, 0x71, 0xd7, 0x93, 0x6c, 0x66, 0xdf, 0x3f,
			0x17, 0x4a, 0x67, 0xfb, 0x32, 0xee, 0x3b, 0xd4,
			0xc8, 0x48, 0x70, 0x0b, 0x01, 0x88, 0x8b, 0x0c
		},
		.hash_len = 64,
		.sig = (unsigned char []) {
			0x51, 0xCB, 0xD2, 0xCF, 0x40, 0xA3, 0xDB, 0xAF,
			0x71, 0x89, 0x8E, 0x29, 0x03, 0xF9, 0xA3, 0xF2,
			0x69, 0x20, 0x68, 0x5E, 0xA7, 0xA2, 0xBB, 0x49,
			0xFB, 0x0A, 0xC9, 0x15, 0x85, 0xA5, 0x7D, 0x3A,
			0x58, 0x93, 0x01, 0x8F, 0x8B, 0x20, 0xDA, 0x10,
			0xE9, 0x10, 0x4F, 0x4B, 0x75, 0x3A, 0x34, 0x51,
			0xFD, 0xB5, 0xF4, 0xB7, 0xC4, 0x34, 0x26, 0x47,
			0x37, 0x8D, 0xE4, 0x01, 0xA6, 0x23, 0x56, 0x8D,
			0x8F, 0x11, 0x66, 0xE9, 0x17, 0xE4, 0xBD, 0x4A,
			0xBE, 0xB0, 0x9E, 0x62, 0x92, 0xD1, 0x5F, 0x30,
			0xD3, 0x4B, 0x14, 0x2E, 0x97, 0xEF, 0xBD, 0x7A,
			0xC7, 0x3B, 0x13, 0x86, 0x9E, 0x2A, 0x0F, 0xCB,
			0xCE, 0x6D, 0x61, 0x5F, 0xEB, 0xF1, 0xEA, 0xEB,
			0x52, 0x99, 0xFD, 0x85, 0x0A, 0xFF, 0xE7, 0x46,
			0xA7, 0xA5, 0xC2, 0x8C, 0xDF, 0x4F, 0x2F, 0xBC,
			0x2F, 0xF2, 0x09, 0x14, 0xD4, 0x3C, 0xAB, 0xB9
		},
		.sig_len = 128,
	},
};

const size_t ECDSA_TV_LEN = sizeof(ECDSA_TV) / sizeof(ECDSA_TV[0]);

#endif
