/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques                                                     */
/*                                                                          */
/* file:     heat.c                                                         */
/*                                                                          */
/* description:  solver for parabolic model problem                         */
/*                                                                          */
/*                   u,t - \Delta u = f  in \Omega                          */
/*                                u = g  on \partial \Omega                 */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include <alberta.h>

/*--------------------------------------------------------------------------*/
/*  function for displaying mesh, discrete solution, and/or estimate        */
/*  defined in graphics.c                                                   */
/*--------------------------------------------------------------------------*/
void graphics(MESH *mesh, DOF_REAL_VEC *u_h, REAL (*get_est)(EL *el),
	      REAL (*u)(const REAL_D x));

/*--------------------------------------------------------------------------*/
/* global variables: finite element space, discrete solution, discrete      */
/*                   solution from previous time step, load vector,         */
/*                   system matrix, and structure for adaptive procedure    */
/*--------------------------------------------------------------------------*/

static const FE_SPACE *fe_space;         /* initialized by main()           */
static DOF_REAL_VEC   *u_h = nil;        /* initialized by main()           */
static DOF_REAL_VEC   *u_old = nil;      /* initialized by main()           */
static DOF_REAL_VEC   *f_h = nil;        /* initialized by main()           */
static DOF_MATRIX     *matrix = nil;     /* initialized by main()           */
static ADAPT_INSTAT   *adapt_instat;     /* initialized by main()           */

static REAL theta = 0.5;   /*---  parameter of the time discretization   ---*/
static REAL err_L2  = 0.0; /*---  spatial error in a single time step    ---*/

/*--------------------------------------------------------------------------*/
/* struct heat_leaf_data: structure for storing one REAL value on each      */
/*                          leaf element as LEAF_DATA                       */
/* rw_el_est():  return a pointer to the memory for storing the element     */
/*               estimate (stored as LEAF_DATA), called by heat_est()       */
/* get_el_est(): return the value of the element estimates (from LEAF_DATA),*/
/*               called by adapt_method_stat() and graphics()               */
/*--------------------------------------------------------------------------*/

struct heat_leaf_data
{
  REAL estimate;            /*  one real for the estimate                   */
  REAL est_c;               /*  one real for the coarsening estimate        */
};

static REAL *rw_el_est(EL *el)
{
  if (IS_LEAF_EL(el))
    return(&((struct heat_leaf_data *)LEAF_DATA(el))->estimate);
  else
    return(nil);
}

static REAL get_el_est(EL *el)
{
  if (IS_LEAF_EL(el))
    return(((struct heat_leaf_data *)LEAF_DATA(el))->estimate);
  else
    return(0.0);
}

static REAL *rw_el_estc(EL *el)
{
  if (IS_LEAF_EL(el))
    return(&((struct heat_leaf_data *)LEAF_DATA(el))->est_c);
  else
    return(nil);
}

static REAL get_el_estc(EL *el)
{
  if (IS_LEAF_EL(el))
    return(((struct heat_leaf_data *)LEAF_DATA(el))->est_c);
  else
    return(0.0);
}

static REAL time_est = 0.0;

static REAL get_time_est(MESH *mesh, ADAPT_INSTAT *adapt)
{
  return(time_est);
}

/*--------------------------------------------------------------------------*/
/* For test purposes: exact solution and its gradient (optional)            */
/*--------------------------------------------------------------------------*/

static REAL eval_time_u = 0.0;
static REAL u(const REAL_D x)
{
  return(sin(M_PI*eval_time_u)*exp(-10.0*SCP_DOW(x,x)));
}

static REAL eval_time_u0 = 0.0;
static REAL u0(const REAL_D x)
{
  eval_time_u = eval_time_u0;
  return(u(x));
}

#if 0
static REAL eval_time_grd_u = 0.0;
static const REAL *grd_u(const REAL_D x)
{
  static REAL_D grd;
  REAL          ux = sin(M_PI*eval_time_grd_u)*exp(-10.0*SCP_DOW(x,x));
  int           n;

  for (n = 0;  n < DIM_OF_WORLD; n++)
    grd[n] = -20.0*x[n]*ux;

  return(grd);
}
#endif

/*--------------------------------------------------------------------------*/
/* problem data: right hand side, boundary values                           */
/*--------------------------------------------------------------------------*/

static REAL eval_time_g = 0.0;
static REAL g(const REAL_D x)              /* boundary values, not optional */
{
  eval_time_u = eval_time_g;
  return(u(x));
}

static REAL eval_time_f = 0.0;
static REAL f(const REAL_D x)              /* -Delta u, not optional        */
{
  REAL  r2 = SCP_DOW(x,x), ux  = sin(M_PI*eval_time_f)*exp(-10.0*r2);
  REAL  ut = M_PI*cos(M_PI*eval_time_f)*exp(-10.0*r2);
  return(ut - (400.0*r2 - 20.0*DIM_OF_WORLD)*ux);
}


/*---8<---------------------------------------------------------------------*/
/*---  write error and estimator data to files                           ---*/
/*--------------------------------------------------------------------->8---*/

static void write_statistics(const char *path, ADAPT_INSTAT *adapt, int n_dof,
			     REAL space_est, REAL time_est, REAL err_L2)
{
  static FILE *file_ndof = nil, *file_tau = nil;
  static FILE *file_space_est = nil, *file_time_est = nil, *file_L2_err = nil;
  const char  *name = fe_space->bas_fcts->name;
  REAL        time = adapt->time;
  char        filename[1024];

  if (!file_ndof)
  {
    sprintf(filename, "%s/n_dof-%s.agr", path ? path : ".", name);
    file_ndof = fopen(filename, "w");
  }

  if (!file_tau)
  {
    sprintf(filename, "%s/tau-%s.agr", path ? path : ".", name);
    file_tau       = fopen(filename, "w");
  }

  if (!file_space_est)
  {
    sprintf(filename, "%s/space_est-%s.agr", path ? path : ".", name);
    file_space_est = fopen(filename, "w");
  }

  if (!file_time_est)
  {
    sprintf(filename, "%s/time_est-%s.agr", path ? path : ".", name);
    file_time_est  = fopen(filename, "w");
  }

  if (!file_L2_err)
  {
    sprintf(filename, "%s/L2_err-%s.agr", path ? path : ".", name);
    file_L2_err    = fopen(filename, "w");
  }

  if (file_ndof) 
    fprintf(file_ndof, "%.6le %d\n", time, n_dof);

/*---  don't print zeros, zeros do not allow log display of estimate ---*/
  if (file_space_est)
    fprintf(file_space_est, "%.6le %.6le\n", time, MAX(space_est,1.e-20));

  if (time > adapt->start_time)
  {
    if (file_tau)
    {
      fprintf(file_tau, "%.6le %.6le\n", time, adapt->timestep);
    }
/*---  don't print zeros, zeros do not allow log display of estimate ---*/
    if (file_time_est)
      fprintf(file_time_est, "%.6le %.6le\n", time, MAX(time_est,1.e-20));
  }

/*---  don't print zeros, zeros do not allow log display of error    ---*/
  if (file_L2_err) 
    fprintf(file_L2_err, "%.6le %.6le\n", time, MAX(err_L2,1.e-20));

  if (time >= adapt->end_time)
  {
    if (file_ndof)      fclose(file_ndof);
    if (file_tau)       fclose(file_tau);
    if (file_space_est) fclose(file_space_est);
    if (file_time_est)  fclose(file_time_est);
    if (file_L2_err)    fclose(file_L2_err);
  }
  else
  {
    fflush(nil);
  }

  return;
}

/*---8<---------------------------------------------------------------------*/
/*---   interpolation is solve on the initial grid                       ---*/
/*--------------------------------------------------------------------->8---*/

static void interpol_u0(MESH *mesh)
{
  dof_compress(mesh);
  interpol(u0, u_h);

  return;
}

static void init_timestep(MESH *mesh, ADAPT_INSTAT *adapt)
{
  FUNCNAME("init_timestep");

  INFO(adapt_instat->info,1,
    "---8<---------------------------------------------------\n");
  INFO(adapt_instat->info, 1,"starting new timestep\n");

  dof_copy(u_h, u_old);
  return;
}

static void set_time(MESH *mesh, ADAPT_INSTAT *adapt)
{
  FUNCNAME("set_time");

  INFO(adapt->info,1,
    "---8<---------------------------------------------------\n");
  if (adapt->time == adapt->start_time)
  {
    INFO(adapt->info, 1,"start time: %.4le\n", adapt->time);
  }
  else
  {
    INFO(adapt->info, 1,"timestep for (%.4le %.4le), tau = %.4le\n",
			 adapt->time-adapt->timestep, adapt->time,
			 adapt->timestep);
  }

  eval_time_f = adapt->time - (1 - theta)*adapt->timestep;
  eval_time_g = adapt->time;

  return;
}

static void close_timestep(MESH *mesh, ADAPT_INSTAT *adapt)
{
  FUNCNAME("close_timestep");
  static REAL err_max = 0.0;                     /* max space-time error    */
  static REAL est_max = 0.0;                     /* max space-time estimate */
  static int  write_fe_data = 0, write_stat_data = 0;
  static int  step = 0;
  static char path[256] = "./";

  REAL        space_est = adapt->adapt_space->err_sum;
  REAL        tolerance = adapt->rel_time_error*adapt->tolerance;

  err_max = MAX(err_max, err_L2);
  est_max = MAX(est_max, space_est + time_est);

  INFO(adapt->info,1,
    "---8<---------------------------------------------------\n");

  if (adapt->time == adapt->start_time)
  {
    tolerance = adapt->adapt_initial->tolerance;
    INFO(adapt->info,1,"start time: %.4le\n", adapt->time);
  }
  else
  {
    tolerance += adapt->adapt_space->tolerance;
    INFO(adapt->info,1,"timestep for (%.4le %.4le), tau = %.4le\n",
			adapt->time-adapt->timestep, adapt->time, 
			adapt->timestep);
  }
  INFO(adapt->info,2,"max. est.  = %.4le, tolerance = %.4le\n", 
		      est_max, tolerance);
  INFO(adapt->info,2,"max. error = %.4le, ratio = %.2lf\n", 
		      err_max, err_max/MAX(est_max,1.0e-20));

  if (!step)
  {
    GET_PARAMETER(1, "write finite element data", "%d", &write_fe_data);
    GET_PARAMETER(1, "write statistical data", "%d", &write_stat_data);
    GET_PARAMETER(1, "data path", "%s", path);
  }

/*---8<---------------------------------------------------------------------*/
/*---   write mesh and discrete solution to file for post-processing     ---*/
/*--------------------------------------------------------------------->8---*/

  if (write_fe_data)
  {
    const char *fn;
    
    fn= generate_filename(path, "mesh", step);
    write_mesh_xdr(mesh, fn, adapt->time);
    fn= generate_filename(path, "u_h", step);
    write_dof_real_vec(u_h, fn);
  }

  step++;

/*---8<---------------------------------------------------------------------*/
/*---  write data about estimate, error, time step size, etc.            ---*/
/*--------------------------------------------------------------------->8---*/

  if (write_stat_data)
  {
    int n_dof = fe_space->admin->size_used;
    write_statistics(path, adapt, n_dof, space_est, time_est, err_L2);
  }

  graphics(mesh, u_h, get_el_est, u);

  return;
}


/*--------------------------------------------------------------------------*/
/* build(): assemblage of the linear system: matrix, load vector,           */
/*          boundary values, called by adapt_method_stat()                  */
/*          on the first call initialize u_h, f_h, matrix and information   */
/*          for assembling the system matrix                                */
/*                                                                          */
/* struct op_info: structure for passing information from init_element() to */
/*                 LALt()                                                   */
/* init_element(): initialization on the element; calculates the            */
/*                 coordinates and |det DF_S| used by LALt; passes these    */
/*                 values to LALt via user_data,                            */
/*                 called on each element by update_matrix()                */
/* LALt():         implementation of -Lambda id Lambda^t for -Delta u,      */
/*                 called by update_matrix()                                */
/* c():            implementation of 1/tau*m(,)                             */
/*--------------------------------------------------------------------------*/

struct op_info
{
  REAL_D  Lambda[N_LAMBDA]; /*  the gradient of the barycentric coordinates */
  REAL    det;              /*  |det D F_S|                                 */

  REAL    tau_1;
};

static const REAL (*LALt(const EL_INFO *el_info, const QUAD *quad, 
			 int iq, void *ud))[N_LAMBDA]
{
  struct op_info *info = (struct op_info *)ud;
  int            i, j, k, dim = el_info->mesh->dim;
  static REAL    LALt[N_LAMBDA][N_LAMBDA] = {};

  for (i = 0; i <= dim; i++)
    for (j = i; j <= dim; j++)
    {
      for (LALt[i][j] = k = 0; k < DIM_OF_WORLD; k++)
	LALt[i][j] += info->Lambda[i][k]*info->Lambda[j][k];
      LALt[i][j] *= info->det;
      LALt[j][i] = LALt[i][j];
    }
  return((const REAL (*)[N_LAMBDA]) LALt);
}

static REAL c(const EL_INFO *el_info, const QUAD *quad, int iq, void *ud)
{
  struct op_info *info = (struct op_info *)ud;

  return(info->tau_1*info->det);
}

static void assemble(DOF_REAL_VEC *u_old, DOF_MATRIX *matrix, DOF_REAL_VEC *fh,
		     DOF_REAL_VEC *u_h, REAL theta, REAL tau,
		     REAL (*f)(const REAL_D), REAL (*g)(const REAL_D))
{
  FUNCNAME("assemble");
  static struct op_info *op_info = nil;
  static const REAL     **(*fill_a)(const EL_INFO *, void *) = nil;
  static void           *a_info = nil;
  static const REAL     **(*fill_c)(const EL_INFO *, void *) = nil;
  static void           *c_info = nil;

  static const DOF_ADMIN *admin = nil;
  static int             n;
  static const REAL   *(*get_u_loc)(const EL *, const DOF_REAL_VEC *, REAL *);
  static const DOF    *(*get_dof)(const EL *, const DOF_ADMIN *, DOF *);
  static const S_CHAR *(*get_bound)(const EL_INFO *, S_CHAR *);
  static const QUAD     *quad = nil;

  TRAVERSE_STACK        *stack = get_traverse_stack();
  const EL_INFO         *el_info;
  FLAGS                  fill_flag;
  const REAL           **a_mat, **c_mat;
  REAL                  *f_vec;
  int                    i, j;

  if(!quad)
    quad = get_quadrature(u_h->fe_space->bas_fcts->dim,
			  2*u_h->fe_space->bas_fcts->degree);

/*--------------------------------------------------------------------------*/
/*  init functions for matrix assembling                                    */
/*--------------------------------------------------------------------------*/

  if (admin != u_h->fe_space->admin)
  {
    OPERATOR_INFO          o_info2 = {nil}, o_info0 = {nil};
    const EL_MATRIX_INFO   *matrix_info;
    const BAS_FCTS         *bas_fcts = u_h->fe_space->bas_fcts;

    admin      = u_h->fe_space->admin;

    n = bas_fcts->n_bas_fcts;
    get_dof    = bas_fcts->get_dof_indices;
    get_bound  = bas_fcts->get_bound;
    get_u_loc  = bas_fcts->get_real_vec;

    if (!op_info)
      op_info = MEM_ALLOC(1, struct op_info);

    o_info2.row_fe_space = o_info2.col_fe_space = u_h->fe_space;

    o_info2.quad[2]        = quad;
    o_info2.LALt           = LALt;
    o_info2.LALt_pw_const  = true;
    o_info2.LALt_symmetric = true;
    o_info2.user_data      = op_info;

    matrix_info = fill_matrix_info(&o_info2, nil);
    fill_a = matrix_info->el_matrix_fct;
    a_info = matrix_info->fill_info;

    o_info0.row_fe_space = o_info0.col_fe_space = u_h->fe_space;

    o_info0.quad[0]        = quad;
    o_info0.c              = c;
    o_info0.c_pw_const     = true;
    o_info0.user_data      = op_info;

    matrix_info = fill_matrix_info(&o_info0, nil);
    fill_c = matrix_info->el_matrix_fct;
    c_info = matrix_info->fill_info;
  }

  op_info->tau_1 = 1.0/tau;

/*--------------------------------------------------------------------------*/
/*  and now assemble the matrix and right hand side                         */
/*--------------------------------------------------------------------------*/

  clear_dof_matrix(matrix);
  dof_set(0.0, fh);
  f_vec = fh->vec;

  fill_flag = CALL_LEAF_EL|FILL_COORDS|FILL_BOUND;
  el_info = traverse_first(stack, u_h->fe_space->mesh, -1, fill_flag);
  while (el_info)
  {
    const REAL   *u_old_loc = (*get_u_loc)(el_info->el, u_old, nil);
    const DOF    *dof       = (*get_dof)(el_info->el, admin, nil);
    const S_CHAR *bound     = (*get_bound)(el_info, nil);

/*--------------------------------------------------------------------------*/
/* initialization of values used by LALt and c                              */
/*--------------------------------------------------------------------------*/
    switch(el_info->mesh->dim) {
    case 1:
      op_info->det = el_grd_lambda_1d(el_info, op_info->Lambda);
      break;
#if DIM_OF_WORLD > 1
    case 2:
      op_info->det = el_grd_lambda_2d(el_info, op_info->Lambda);
      break;
#if DIM_OF_WORLD > 2
    case 3:
      op_info->det = el_grd_lambda_3d(el_info, op_info->Lambda);
      break;
#endif
#endif
    default:
      ERROR_EXIT("Illegal dim!\n");
    }

    a_mat = fill_a(el_info, a_info);
    c_mat = fill_c(el_info, c_info);

/*--------------------------------------------------------------------------*/
/*  add theta*a(psi_i,psi_j) + 1/tau*m(4*u^3*psi_i,psi_j)                   */
/*--------------------------------------------------------------------------*/

    if (theta)
    {
      add_element_matrix(matrix, theta, n, n, dof, dof, a_mat, bound);
    }
    add_element_matrix(matrix, 1.0, n, n, dof, dof, c_mat, bound);

/*--------------------------------------------------------------------------*/
/*  f += -(1-theta)*a(u_old,psi_i) + 1/tau*m(u_old,psi_i)                   */
/*--------------------------------------------------------------------------*/

    if (1.0 - theta)
    {
      REAL theta1 = 1.0 - theta;
      for (i = 0; i < n; i++)
      {
	if (bound[i] < DIRICHLET)
	{
	  REAL val = 0.0;
	  for (j = 0; j < n; j++)
	    val += (-theta1*a_mat[i][j] + c_mat[i][j])*u_old_loc[j];
	  f_vec[dof[i]] += val;
	}
      }
    }
    else
    {
      for (i = 0; i < n; i++)
      {
	if (bound[i] < DIRICHLET)
	{
	  REAL val = 0.0;
	  for (j = 0; j < n; j++)
	    val += c_mat[i][j]*u_old_loc[j];
	  f_vec[dof[i]] += val;
	}
      }
    }
    el_info = traverse_next(stack, el_info);
  }

  free_traverse_stack(stack);

  L2scp_fct_bas(f, quad, fh);
  dirichlet_bound(g, fh, u_h, nil);

  return;
}

static void build(MESH *mesh, U_CHAR flag)
{
  FUNCNAME("build");

  dof_compress(mesh);
  
  INFO(adapt_instat->adapt_space->info, 2,
    "%d DOFs for %s\n", fe_space->admin->size_used, fe_space->name);

  assemble(u_old, matrix, f_h, u_h, theta, adapt_instat->timestep, f, g);

  return;
}

/*--------------------------------------------------------------------------*/
/* solve(): solve the linear system, called by adapt_method_stat()          */
/*--------------------------------------------------------------------------*/

static void solve(MESH *mesh)
{
  FUNCNAME("solve");
  static REAL       tol = 1.e-8;
  static int        max_iter = 1000, info = 2, icon = 1, restart = 0;
  static OEM_SOLVER solver = NoSolver;

  if (solver == NoSolver)
  {
    tol = 1.e-8;
    GET_PARAMETER(1, "solver", "%d", &solver);
    GET_PARAMETER(1, "solver tolerance", "%f", &tol);
    GET_PARAMETER(1, "solver precon", "%d", &icon);
    GET_PARAMETER(1, "solver max iteration", "%d", &max_iter);
    GET_PARAMETER(1, "solver info", "%d", &info);
    if (solver == GMRes)
      GET_PARAMETER(1, "solver restart", "%d", &restart);
  }
  oem_solve_s(matrix, f_h, u_h, solver, tol, icon, restart, max_iter, info);

  return;
}

/*--------------------------------------------------------------------------*/
/* Functions for error estimate:                                            */
/* estimate():   calculates error estimate via heat_est()                   */
/*               calculates exact error also (only for test purpose),       */
/*               called by adapt_method_stat()                              */
/*--------------------------------------------------------------------------*/

static REAL r(const EL_INFO *el_info, const QUAD *quad, int iq, REAL t, 
	      REAL uh_iq, const REAL_D grd_uh_iq)
{
  REAL_D      x;
  coord_to_world(el_info, quad->lambda[iq], x);
  eval_time_f = t;
  return(-f(x));
}


static REAL estimate(MESH *mesh, ADAPT_STAT *adapt)
{
  FUNCNAME("estimate");
  static int     degree;
  static REAL    C[4] = {-1.0, 1.0, 1.0, 1.0};
  REAL_DD        A = {{0.0}};
  FLAGS          r_flag = 0;  /* = (INIT_UH|INIT_GRD_UH), if needed by r()  */
  int            n;
  REAL           space_est;

  for (n = 0; n < DIM_OF_WORLD; n++)
    A[n][n] = 1.0;   /* set diagonal of A; all other elements are zero      */

  eval_time_u = adapt_instat->time;

  if (C[0] < 0)
  {
    C[0] = 1.0;
    GET_PARAMETER(1, "estimator C0", "%f", &C[0]);
    GET_PARAMETER(1, "estimator C1", "%f", &C[1]);
    GET_PARAMETER(1, "estimator C2", "%f", &C[2]);
    GET_PARAMETER(1, "estimator C3", "%f", &C[3]);
  }

  degree = 2*u_h->fe_space->bas_fcts->degree;
  time_est = heat_est(u_h, adapt_instat, rw_el_est, rw_el_estc,
		      degree, C, u_old, (const REAL_D *) A, r, r_flag);

  space_est = adapt_instat->adapt_space->err_sum;
  err_L2 = L2_err(u, u_h, nil, 0, nil, nil);

  INFO(adapt_instat->info,2,
    "---8<---------------------------------------------------\n");
  INFO(adapt_instat->info, 2,"time = %.4le with timestep = %.4le\n",
			      adapt_instat->time, adapt_instat->timestep);
  INFO(adapt_instat->info, 2,"estimate   = %.4le, max = %.4le\n", space_est,
			      sqrt(adapt_instat->adapt_space->err_max));
  INFO(adapt_instat->info, 2,"||u-uh||L2 = %.4le, ratio = %.2lf\n", err_L2,
			      err_L2/MAX(space_est,1.e-20));

  return(adapt_instat->adapt_space->err_sum);
}

static REAL est_initial(MESH *mesh, ADAPT_STAT *adapt)
{
  err_L2 = adapt->err_sum = L2_err(u0, u_h, nil, 0, rw_el_est, &adapt->err_max);
  return(adapt->err_sum);
}

/*--------------------------------------------------------------------------*/
/* main program                                                             */
/*--------------------------------------------------------------------------*/

int main(int argc, char **argv)
{
  FUNCNAME("main");
  MACRO_DATA     *data;
  MESH           *mesh;
  const BAS_FCTS *lagrange;
  int             n_refine = 0, k, p = 1, dim;
  char            filename[128];
  REAL            fac = 1.0;

/*--------------------------------------------------------------------------*/
/*  first of all, init parameters of the init file                          */
/*--------------------------------------------------------------------------*/

  init_parameters(0, "INIT/heat.dat");
  for (k = 1; k+1 < argc; k += 2)
    ADD_PARAMETER(0, argv[k], argv[k+1]);

  GET_PARAMETER(1, "mesh dimension", "%d", &dim);
  GET_PARAMETER(1, "macro file name", "%s", filename);
  GET_PARAMETER(1, "global refinements", "%d", &n_refine);
  GET_PARAMETER(1, "parameter theta", "%e", &theta);
  GET_PARAMETER(1, "polynomial degree", "%d", &p);
  
/*--------------------------------------------------------------------------*/
/*  get a mesh, and read the macro triangulation from file                  */
/*--------------------------------------------------------------------------*/
  data = read_macro(filename);

  mesh = GET_MESH(dim, "ALBERTA mesh", data, nil);
  free_macro_data(data);

  init_leaf_data(mesh, sizeof(struct heat_leaf_data), nil, nil);

  lagrange = get_lagrange(mesh->dim, p);
  TEST_EXIT(lagrange, "no lagrange BAS_FCTS\n");

  fe_space = get_fe_space(mesh, lagrange->name, nil, lagrange, false);

  global_refine(mesh, n_refine * dim);

  graphics(mesh, nil, nil, nil);

  matrix = get_dof_matrix("A", fe_space, fe_space);
  f_h    = get_dof_real_vec("f_h", fe_space);
  u_h    = get_dof_real_vec("u_h", fe_space);
  u_h->refine_interpol = fe_space->bas_fcts->real_refine_inter;
  u_h->coarse_restrict = fe_space->bas_fcts->real_coarse_inter;
  u_old  = get_dof_real_vec("u_old", fe_space);
  u_old->refine_interpol = fe_space->bas_fcts->real_refine_inter;
  u_old->coarse_restrict = fe_space->bas_fcts->real_coarse_inter;
  dof_set(0.0, u_h);      /*  initialize u_h  !                          */

/*--------------------------------------------------------------------------*/
/*  init adapt structure and start adaptive method                          */
/*--------------------------------------------------------------------------*/

  adapt_instat = get_adapt_instat(dim, "heat", "adapt", 2, adapt_instat);

/*--------------------------------------------------------------------------*/
/*  adapt time step size to refinement level and polynomial degree          */
/*--------------------------------------------------------------------------*/

  if (theta < 0.5) {
    WARNING("You are using the explicit Euler scheme\n");
    WARNING("Use a sufficiently small time step size!!!\n");
    fac = 1.0e-3;
  }

  if (theta == 0.5)
    adapt_instat->timestep *= fac*pow(2, -(REAL)(p*(n_refine))/2.0);
  else 
    adapt_instat->timestep *= fac*pow(2, -(REAL)(p*(n_refine)));
  MSG("using initial timestep size = %.4le\n", adapt_instat->timestep);

  eval_time_u0 = adapt_instat->start_time;

  adapt_instat->adapt_initial->get_el_est = get_el_est;
  adapt_instat->adapt_initial->estimate = est_initial;
  adapt_instat->adapt_initial->solve = interpol_u0;

  adapt_instat->adapt_space->get_el_est   = get_el_est;
  adapt_instat->adapt_space->get_el_estc  = get_el_estc;
  adapt_instat->adapt_space->estimate = estimate;
  adapt_instat->adapt_space->build_after_coarsen = build;
  adapt_instat->adapt_space->solve = solve;

  adapt_instat->init_timestep  = init_timestep;
  adapt_instat->set_time       = set_time;
  adapt_instat->get_time_est   = get_time_est;
  adapt_instat->close_timestep = close_timestep;

  adapt_method_instat(mesh, adapt_instat);
  
  WAIT_REALLY;
  return(0);
}
