/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     disc_lagrange_2_1d.c                                           */
/*                                                                          */
/* description:  piecewise quadratic discontinuous Lagrange elements in 1d  */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#define N_BAS_1D N_VERTICES_1D+1

static const REAL d_bary2_1d[N_BAS_1D][N_LAMBDA] = {{1.0, 0.0, 0.0, 0.0},
						    {0.0, 1.0, 0.0, 0.0},
						    {0.5, 0.5, 0.0, 0.0}};

/****************************************************************************/
/*  basisfunction at vertex 0                                               */
/****************************************************************************/

static REAL d_phi2v0_1d(const REAL lambda[N_LAMBDA])
{
  return(lambda[0]*(2.0*lambda[0] - 1.0));
}

static const REAL *d_grd_phi2v0_1d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[0] = 4.0*lambda[0] - 1.0;
  return((const REAL *) grd);
}

static const REAL (*d_D2_phi2v0_1d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{4, 0, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 0, 0}};

  return(D2);
}

/****************************************************************************/
/*  basisfunction at vertex 1                                               */
/****************************************************************************/

static REAL d_phi2v1_1d(const REAL lambda[N_LAMBDA])
{
  return(lambda[1]*(2.0*lambda[1] - 1.0));
}

static const REAL *d_grd_phi2v1_1d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[1] = 4.0*lambda[1] - 1.0;
  return((const REAL *) grd);
}

static const REAL (*d_D2_phi2v1_1d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 0},
					      {0, 4, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 0, 0}};

  return(D2);
}

/****************************************************************************/
/*  basisfunction at center                                                 */
/****************************************************************************/

static REAL d_phi2c0_1d(const REAL lambda[N_LAMBDA])
{
  return(4.0*lambda[0]*lambda[1]);
}

static const REAL *d_grd_phi2c0_1d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[0] = 4.0*lambda[1];
  grd[1] = 4.0*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*d_D2_phi2c0_1d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 4, 0, 0},
					      {4, 0, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 0, 0}};
  return(D2);
}

/****************************************************************************/
/*  functions for combining basisfunctions with coefficients                */
/****************************************************************************/

static const DOF *d_get_dof_indices2_1d(const EL *el, const DOF_ADMIN *admin,
					DOF *idof)
{
  static DOF  index_vec[N_BAS_1D];
  int         i, n0, node;
  DOF         *rvec = idof ? idof : index_vec;
  DOF         **dof = el->dof;

  n0 = admin->n0_dof[CENTER];
  node = admin->mesh->node[CENTER];

  for (i = 0; i < N_BAS_1D; i++)
    rvec[i] = dof[node][n0+i];

  return((const DOF *) rvec);
}

static const S_CHAR *d_get_bound2_1d(const EL_INFO *el_info, S_CHAR *bound)
{
  FUNCNAME("d_get_bound2_1d");
  static S_CHAR  bound_vec[N_BAS_1D];
  S_CHAR         *rvec = bound ? bound : bound_vec;
  int            i;

  TEST_FLAG(FILL_BOUND, el_info);

/*--------------------------------------------------------------------*/
/*--- basis functions at vertices                                  ---*/
/*--------------------------------------------------------------------*/

  for (i = 0; i < N_VERTICES_1D; i++)
    rvec[i] = el_info->vertex_bound[i];

/*--------------------------------------------------------------------*/
/*--- basis functions at center                                    ---*/
/*--------------------------------------------------------------------*/

  rvec[i] = INTERIOR;

  return((const S_CHAR *) rvec);
}

static const int *d_get_int_vec2_1d(const EL *el, const DOF_INT_VEC *vec,
				    int *ivec)
{
  FUNCNAME("d_get_int_vec2_1d");
  int           i, n0, node;
  static int    local_vec[N_BAS_1D];
  int           *v = nil, *rvec = ivec ? ivec : local_vec;
  DOF           **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[CENTER];
  node = vec->fe_space->admin->mesh->node[CENTER];

  for (i = 0; i < N_BAS_1D; i++)
    rvec[i] = v[dof[node][n0+i]];

  return((const int *) rvec);
}

static const REAL *d_get_real_vec2_1d(const EL *el, const DOF_REAL_VEC *vec,
				      REAL *Rvec)
{
  FUNCNAME("d_get_real_vec2_1d");
  int            i, n0, node;
  static REAL    local_vec[N_BAS_1D];
  REAL           *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[CENTER];
  node = vec->fe_space->admin->mesh->node[CENTER];

  for (i = 0; i < N_BAS_1D; i++)
    rvec[i] = v[dof[node][n0+i]];

  return((const REAL *) rvec);
}

static const REAL_D *d_get_real_d_vec2_1d(const EL *el, 
					  const DOF_REAL_D_VEC *vec,
					  REAL_D *Rvec)
{
  FUNCNAME("d_get_real_d_vec2_1d");
  int            i, k, n0, node;
  static REAL_D  local_vec[N_BAS_1D];
  REAL_D         *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[CENTER];
  node = vec->fe_space->admin->mesh->node[CENTER];

  for (i = 0; i < N_BAS_1D; i++)
    for (k = 0; k < DIM_OF_WORLD; k++)
      rvec[i][k] = v[dof[node][n0+i]][k];

  return((const REAL_D *) rvec);
}

static const U_CHAR *d_get_uchar_vec2_1d(const EL *el, const DOF_UCHAR_VEC *vec,
					 U_CHAR *uvec)
{
  FUNCNAME("d_get_uchar_vec2_1d");
  int            i, n0, node;
  static U_CHAR  local_vec[N_BAS_1D];
  U_CHAR         *v = nil, *rvec = uvec ? uvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[CENTER];
  node = vec->fe_space->admin->mesh->node[CENTER];

  for (i = 0; i < N_BAS_1D; i++)
    rvec[i] = v[dof[node][n0+i]];

  return((const U_CHAR *) rvec);
}

static const S_CHAR *d_get_schar_vec2_1d(const EL *el, const DOF_SCHAR_VEC *vec,
					 S_CHAR *svec)
{
  FUNCNAME("d_get_schar_vec2_1d");
  int            i, n0, node;
  static S_CHAR  local_vec[N_BAS_1D];
  S_CHAR         *v = nil, *rvec = svec ? svec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[CENTER];
  node = vec->fe_space->admin->mesh->node[CENTER];

  for (i = 0; i < N_BAS_1D; i++)
    rvec[i] = v[dof[node][n0+i]];

  return((const S_CHAR *) rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of scalar functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL(d_,2,1,3)

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of vector functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL_D(d_,2,1,3)


/****************************************************************************/
/*  functions for interpolation/ restriction during refinement/coarsening   */
/****************************************************************************/

static void d_real_refine_inter2_1d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  EL              *el;
  const DOF       *cdof;
  const REAL      *pvec;
  const BAS_FCTS  *bas_fcts = drv->fe_space->bas_fcts;
  const DOF_ADMIN *admin = drv->fe_space->admin;
  REAL            *v = drv->vec;

  if (n < 1) return;

  el = list->el_info.el;
  pvec = (*bas_fcts->get_real_vec)(el, drv, nil);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, first child                           ---*/
/*--------------------------------------------------------------------*/

  cdof = (*bas_fcts->get_dof_indices)(el->child[0], admin, nil);

  v[cdof[0]] = (pvec[0]);
  v[cdof[1]] = (pvec[2]);
  v[cdof[2]] = (0.375*pvec[0] - 0.125*pvec[1] + 0.75*pvec[2]);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, second child                          ---*/
/*--------------------------------------------------------------------*/

  cdof = (*bas_fcts->get_dof_indices)(el->child[1], admin, nil);

  v[cdof[0]] = (pvec[2]);
  v[cdof[1]] = (pvec[1]);
  v[cdof[2]] = (-0.125*pvec[0] + 0.375*pvec[1] + 0.75*pvec[2]);

  return;
}

static void d_real_coarse_inter2_1d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  EL              *el;
  const DOF       *pdof;
  const REAL      *cvec;
  const BAS_FCTS  *bas_fcts = drv->fe_space->bas_fcts;
  const DOF_ADMIN *admin = drv->fe_space->admin;
  REAL            *v = drv->vec;

  if (n < 1) return;

  el = list->el_info.el;
  pdof = (*bas_fcts->get_dof_indices)(el, admin, nil);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, from first child                      ---*/
/*--------------------------------------------------------------------*/

  cvec = (*bas_fcts->get_real_vec)(el->child[0], drv, nil);

  v[pdof[0]] = cvec[0];

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, second child                          ---*/
/*--------------------------------------------------------------------*/

  cvec = (*bas_fcts->get_real_vec)(el->child[1], drv, nil);

  v[pdof[1]] = cvec[1];
  v[pdof[2]] = cvec[0];

  return;
}

static BAS_FCT      *d_phi2_1d[3]     = {d_phi2v0_1d, d_phi2v1_1d, d_phi2c0_1d};
static GRD_BAS_FCT  *d_grd_phi2_1d[3] = {d_grd_phi2v0_1d, d_grd_phi2v1_1d,
					 d_grd_phi2c0_1d};
static D2_BAS_FCT   *d_D2_phi2_1d[3]  = {d_D2_phi2v0_1d, d_D2_phi2v1_1d,
					 d_D2_phi2c0_1d};

static BAS_FCTS  disc_lagrange2_1d = {"disc_lagrange2_1d", 1, N_BAS_1D, 2,
				      {0, N_BAS_1D, 0, 0}, 
				      nil,
				      d_phi2_1d, d_grd_phi2_1d, d_D2_phi2_1d, 
				      d_get_dof_indices2_1d, 
				      d_get_bound2_1d,
				      d_interpol2_1d,
				      d_interpol_d2_1d,
				      d_get_int_vec2_1d,
				      d_get_real_vec2_1d,
				      d_get_real_d_vec2_1d,
				      d_get_uchar_vec2_1d,
				      d_get_schar_vec2_1d,
				      d_real_refine_inter2_1d,
				      d_real_coarse_inter2_1d,
				      nil,
				      nil,
				      nil,
				      nil,
				      d_bary2_1d, };

#undef N_BAS_1D
