package latexDraw.figures;

import java.awt.*;
import java.awt.image.BufferedImage;

import latexDraw.util.LaTeXDrawPoint2D;

/** 
 * This abstract class defines a general model of a grid; it can be used to define
 * a grid or axes for instance.<br>
 * <br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 14/08/06<br>
 * @author Arnaud BLOUIN<br>
 * @since 1.8<br>
 * @version 2.0.0<br>
 */
public abstract class GridShape extends Figure
{
	private static final long serialVersionUID = 1L;
	
	/** If true, the x label will be displayed at the south of the grid. Else at the north */
	protected boolean isXLabelSouth;
	
	/** If true, the y label will be displayed at the west of the grid. Else at the east */
	protected boolean isYLabelWest;
	
	/** The minimum values of the axes */
	protected LaTeXDrawPoint2D gridStart;
	
	/** The maximum values of the axes */
	protected LaTeXDrawPoint2D gridEnd;
	
	/** The position of the SW point of the grid in the drawpanel */
	protected LaTeXDrawPoint2D position;
	
	/** The origin of the grid */
	protected LaTeXDrawPoint2D origin;
	
	/** The current font*/
	protected Font font;
	
	/** The current font metrics of the text */
	protected FontMetrics fontMetrics;
	
	
	/** The value by default of isXLabelSouth */
	public static final boolean DEFAULT_ISXLABELSOUTH = true;
	
	/** The value by default of isYLabelWest */
	public static final boolean DEFAULT_ISYLABELWEST = true;
	
	/** The origin of the grid by default */
	public static final LaTeXDrawPoint2D DEFAULT_ORIGIN = new LaTeXDrawPoint2D(0,0);
	
	/** The minimum points of the grid by default */
	public static final LaTeXDrawPoint2D DEFAULT_GRIDMIN = (LaTeXDrawPoint2D)DEFAULT_ORIGIN.clone();
	
	/** The maximum points of the grid by default */
	public static final LaTeXDrawPoint2D DEFAULT_GRIDMAX = new LaTeXDrawPoint2D(2,2);
	
	
	
	protected GridShape(LaTeXDrawPoint2D pos, boolean increaseMeter)
	{
		super(increaseMeter);
		
		canHaveShadow 		= false;
		canBeFilled 		= false;
		isBordersMovable 	= false;
		isDashableOrDotable = false;
		isDoubleBoundaryable = false;
		isThicknessable 	= false;
		isResizable 		= false;
		position 			= pos;
		isXLabelSouth 		= DEFAULT_ISXLABELSOUTH;
		isYLabelWest  		= DEFAULT_ISYLABELWEST;
		origin         		= (LaTeXDrawPoint2D)DEFAULT_ORIGIN.clone();
		gridStart			= (LaTeXDrawPoint2D)DEFAULT_GRIDMIN.clone();
		gridEnd				= (LaTeXDrawPoint2D)DEFAULT_GRIDMAX.clone();
	}


	
	@Override
	public Object clone() throws CloneNotSupportedException
	{
		GridShape g = (GridShape)super.clone();
		
		g.borders 	= (LaTeXDrawRectangle)borders.clone();
		g.gridEnd 	= (LaTeXDrawPoint2D)gridEnd.clone();
		g.gridStart = (LaTeXDrawPoint2D)gridStart.clone();
		g.origin 	= (LaTeXDrawPoint2D)origin.clone();
		g.position 	= (LaTeXDrawPoint2D)position.clone();
		g.gravityCenter = g.borders.getGravityCenter();
		g.isXLabelSouth = isXLabelSouth;
		g.isYLabelWest  = isYLabelWest;
		g.shape = g.createShape2D();
		
		return g;
	}



	@Override
	public Shape createNonRotatedShape2D()
	{
		return borders.createNonRotatedShape2D();
	}




	@Override
	public Shape createShape2D()
	{
		if(borders==null)
			updateBorders();
		
		return borders.createShape2D();
	}



	public abstract void updateBorders();



	@Override
	public abstract void draw(Graphics2D g, Object antiAlias, Object rendering, 
								Object alphaInter, Object colorRendering);
	

	@Override
	public abstract String getCodePSTricks(DrawBorders drawBorders, float ppc);



	@Override
	public boolean isIn(LaTeXDrawPoint2D pt)
	{
		if(borders==null) return false;
		return borders.isIn(pt);
	}



	@Override
	public boolean isTooSmallToBeRescaled()
	{
		return true;
	}



	@Override
	public void mirrorHorizontal(LaTeXDrawPoint2D orig)
	{
		updateBorders();
		borders.mirrorHorizontal(orig);
		position.setLocation(borders.getTheNWPoint().x, position.y);
		updateBorders(fontMetrics);
	}



	public abstract void updateBorders(FontMetrics fm);



	@Override
	public void mirrorVertical(LaTeXDrawPoint2D orig)
	{
		updateBorders();
		borders.mirrorVertical(orig);
		position.setLocation(position.x, borders.getTheSEPoint().y);
		updateBorders(fontMetrics);
	}



	@Override
	public void onDragged(Point formerPt, Point newPt)
	{
		if(formerPt.equals(newPt)) return;
		
		if(isOnRotation && borders.dSelected!=null)
		{
			borders.onDragged(formerPt, newPt);
			shape = createShape2D();
			rotationAngle = borders.getRotationAngle();
		}
		else
			shift(formerPt, newPt);
		gravityCenter = borders.getGravityCenter();
	}



	@Override
	@Deprecated
	public void rescaleX(double formerX, double newX, double percent, LaTeXDrawRectangle bound) 
	{
		/*
		 * This function is disable
		 */
	}

	
	
	@Override
	@Deprecated
	public void rescaleY(double formerY, double newY, double percent, LaTeXDrawRectangle bound) 
	{
		/*
		 * This function is disable
		 */
	}



	@Override
	@Deprecated
	public void setLastPoint(double x, double y) 
	{
		/*
		 * This function is disable
		 */
	}

	
	@Override
	@Deprecated
	public void setFirstPoint(double x, double y) 
	{
		/*
		 * This function is disable
		 */
	}



	@Override
	public void shift(double shiftX, double shiftY)
	{
		if(position!=null)
		{
			position.x+=shiftX;
			position.y+=shiftY;
			updateBorders(getFontMetrics());
		}	
	}



	@Override
	public void updateShape()
	{
		shape = createShape2D();
	}

	
	
	/**
	 * @param x The x-coordinate to set.
	 */
	public synchronized void setGridEndX(int x)
	{
		if(x>=gridStart.x)
			gridEnd.x = x;
	}

	
	
	/**
	 * @param y The y-coordinate to set.
	 */
	public synchronized void setGridEndY(int y)
	{
		if(y>=gridStart.y)
			gridEnd.y = y;
	}



	/**
	 * @return the isXLabelSouth.
	 */
	public synchronized boolean isXLabelSouth()
	{
		return isXLabelSouth;
	}



	/**
	 * @param isXLabelSouth the isXLabelSouth to set.
	 */
	public synchronized void setXLabelSouth(boolean isXLabelSouth)
	{
		this.isXLabelSouth = isXLabelSouth;
	}




	/**
	 * @return the isYLabelWest.
	 */
	public synchronized boolean isYLabelWest()
	{
		return isYLabelWest;
	}




	/**
	 * @param isYLabelWest the isYLabelWest to set.
	 */
	public synchronized void setYLabelWest(boolean isYLabelWest)
	{
		this.isYLabelWest = isYLabelWest;
	}




	/**
	 * @return the gridStart.
	 */
	public synchronized LaTeXDrawPoint2D getGridStart()
	{
		return gridStart;
	}




	/**
	 * @param gridStart the gridStart to set.
	 */
	public synchronized void setGridStart(LaTeXDrawPoint2D gridStart)
	{
		this.gridStart = gridStart;
	}




	/**
	 * @return the gridEnd.
	 */
	public synchronized LaTeXDrawPoint2D getGridEnd()
	{
		return gridEnd;
	}



	/**
	 * @param gridEnd the gridEnd to set.
	 */
	public synchronized void setGridEnd(LaTeXDrawPoint2D gridEnd)
	{
		this.gridEnd = gridEnd;
	}



	/**
	 * @return the position.
	 */
	public synchronized LaTeXDrawPoint2D getPosition()
	{
		return position;
	}




	/**
	 * @param position the position to set.
	 */
	public synchronized void setPosition(LaTeXDrawPoint2D position)
	{
		this.position = position;
	}




	/**
	 * @return the origin.
	 */
	public synchronized LaTeXDrawPoint2D getOrigin()
	{
		return origin;
	}



	/**
	 * @param origin the origin to set.
	 */
	public synchronized void setOrigin(LaTeXDrawPoint2D origin)
	{
		this.origin = origin;
	}
	
	
	/**
	 * @param y The y-coordinate to set.
	 */
	public synchronized void setGridStartY(int y)
	{
		if(y<=gridEnd.y)
			gridStart.y = y;
	}
	

	
	/**
	 * @param x The x-coordinate to set.
	 */
	public synchronized void setGridStartX(int x)
	{
		if(x<=gridEnd.x)
			gridStart.x = x;
	}
	
	
	
	/**
	 * @param x The X-coordinate to set.
	 */
	public synchronized void setOriginX(int x)
	{
		origin.x = x;
	}

	
	
	/**
	 * @param y The Y-coordinate to set.
	 */
	public synchronized void setOriginY(int y)
	{
		origin.y = y;
	}
	
	
	
	
	/**
	 * @return the fontMetrics
	 */
	public synchronized FontMetrics getFontMetrics()
	{
		return fontMetrics;
	}




	/**
	 * Allows to update the Font and the FontMetrics.
	 */
	public synchronized void updateFonts()
	{
		BufferedImage bufferImage = new BufferedImage(2, 2, BufferedImage.TYPE_INT_RGB);
		Graphics2D g = bufferImage.createGraphics();
		g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
		g.setFont(getFont());
		fontMetrics = g.getFontMetrics();
		
		g.dispose();
		bufferImage.flush();
		g = null;
		bufferImage = null;
	}


	
	/**
	 * @return the font
	 */
	public synchronized Font getFont()
	{
		return font;
	}
		
	
	@Override
	@Deprecated
	public LaTeXDrawPoint2D getLastPoint()
	{
		return null;
	}
}
