/*
 * JetS3t : Java S3 Toolkit
 * Project hosted at http://bitbucket.org/jmurty/jets3t/
 *
 * Copyright 2006-2010 James Murty
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jets3t.service.utils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * A simple container class to store the results generated by
 * {@link FileComparer#buildDiscrepancyLists(Map, Map)}.
 * <p>
 * The following key lists are available:
 * <ul>
 * <li>onlyOnServerKeys: objects with these keys exist only in S3, not on the local computer</li>
 * <li>updatedOnServerKeys: objects with these keys exist in S3 and on the local computer, but the
 *     contents of the objects are different and the S3 version is newer than the local one.</li>
 * <li>onlyOnClientKeys: objects with these keys exist only on the local computer, not in S3</li>
 * <li>updatedOnClientKeys: objects with these keys exist in S3 and on the local computer, but the
 *     contents of the objects are different and the local version is newer than the S3 one.</li>
 * <li>alreadySynchronisedKeys: identical objects with these keys exist in S3 and on the local computer</li>
 * </ul>
 *
 *
 * @author James Murty
 */
public class FileComparerResults {
    public List<String> onlyOnServerKeys, updatedOnServerKeys, updatedOnClientKeys, onlyOnClientKeys,
        alreadySynchronisedKeys, alreadySynchronisedLocalPaths;

    public FileComparerResults(
        List<String> onlyOnServerKeys, List<String> updatedOnServerKeys,
        List<String> updatedOnClientKeys, List<String> onlyOnClientKeys,
        List<String> alreadySynchronisedKeys, List<String> alreadySynchronisedLocalPaths)
    {
        this.onlyOnServerKeys = onlyOnServerKeys;
        this.updatedOnServerKeys = updatedOnServerKeys;
        this.updatedOnClientKeys = updatedOnClientKeys;
        this.onlyOnClientKeys = onlyOnClientKeys;
        this.alreadySynchronisedKeys = alreadySynchronisedKeys;
        this.alreadySynchronisedLocalPaths = alreadySynchronisedLocalPaths;
    }

    public FileComparerResults() {
        this(new ArrayList<String>(), new ArrayList<String>(), new ArrayList<String>(),
            new ArrayList<String>(), new ArrayList<String>(), new ArrayList<String>());
    }

    public void merge(FileComparerResults resultsToAdd) {
        this.updatedOnServerKeys.addAll(resultsToAdd.updatedOnServerKeys);
        this.updatedOnClientKeys.addAll(resultsToAdd.updatedOnClientKeys);
        this.alreadySynchronisedKeys.addAll(resultsToAdd.alreadySynchronisedKeys);
        this.alreadySynchronisedLocalPaths.addAll(resultsToAdd.alreadySynchronisedLocalPaths);

        this.onlyOnServerKeys.addAll(resultsToAdd.onlyOnServerKeys);

        // Only keys present on S3 and no-where else should remain in server keys list.
        onlyOnServerKeys.removeAll(updatedOnServerKeys);
        onlyOnServerKeys.removeAll(updatedOnClientKeys);
        onlyOnServerKeys.removeAll(onlyOnClientKeys);
        onlyOnServerKeys.removeAll(alreadySynchronisedKeys);
        onlyOnServerKeys.removeAll(alreadySynchronisedLocalPaths);

        this.onlyOnClientKeys.addAll(resultsToAdd.onlyOnClientKeys);

        // Only keys present on client and no-where else should remain in client keys list.
        onlyOnClientKeys.removeAll(updatedOnServerKeys);
        onlyOnClientKeys.removeAll(updatedOnClientKeys);
        onlyOnClientKeys.removeAll(onlyOnServerKeys);
        onlyOnClientKeys.removeAll(alreadySynchronisedKeys);
        onlyOnServerKeys.removeAll(alreadySynchronisedLocalPaths);
    }

    public long getCountOfItemsCompared() {
        return this.updatedOnServerKeys.size()
            + this.updatedOnClientKeys.size()
            + this.onlyOnServerKeys.size()
            + this.onlyOnClientKeys.size()
            + this.alreadySynchronisedKeys.size();
    }

    @Override
    public String toString() {
        return "onlyOnServerKeys: " + onlyOnServerKeys
            + ", updatedOnServerKeys: " + updatedOnServerKeys
            + ", updatedOnClientKeys: " + updatedOnClientKeys
            + ", onlyOnClientKeys: " + onlyOnClientKeys
            + ", alreadySynchronisedKeys: " + alreadySynchronisedKeys
            + ", alreadySynchronisedLocalPaths: " + alreadySynchronisedLocalPaths;
    }

}
