/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#ifndef ISCRIPTKIT_H
#define ISCRIPTKIT_H


#include "iarray.h"
#include "icalculator.h"
#include "istring.h"

class iScript;


namespace iScriptKit
{
	//
	//  Calculator with error-reporting type conversion
	//
	class Value;
	class Calculator : public iCalculator<double>
	{

		friend class ::iScript;

	public:

		bool Verify(const iString &expr, int dim, int use, const iString &err);
		void AddVariable(const result_t *var); // scope resolution helper

	protected:

		Calculator(iScript *script);

		virtual const iCalculatorKit::Variable* FindExternalVariable(const iString &name) const;

		iScript *mScript;
	};

	//
	//  Base class for values (variables and parameters)
	//
	class Value
	{

		friend class VariableDeclarationStatement;

	public:

		virtual ~Value();

		const iString& Name() const;
		int Dim() const;
		int Use() const;

		inline bool IsReleased() const { return mIsReleased; }
		inline const Calculator::result_t* CalculatorValue() const { return mValue; }

		virtual iString GetValueAsText() const = 0;
		virtual iString GetTypeAsText() const = 0;
		virtual int GetTypeId() const = 0;

		bool Morph(int dim);

	protected:

		Value(iScript *script, const iString &name, int dim, int use);

		void ReportError(const iString &str);

		Calculator::result_t *mValue;
		bool mIsReleased;
		iScript *mScript;

#ifdef I_DEBUG
	private:

		int mId;
		static int mIdCounter;
#endif
	};


	//
	//  Class for parameters.
	//
	template<class T>
	class Constant : public Value
	{

	public:

		static int UsagePattern();
		static T ConvertToNative(Calculator::number_t v);
		static Calculator::number_t ConvertFromNative(T v);

		Constant(iScript *script, const iString &name, T v);
		Constant(iScript *script, const iString &name, int dim, T *v);
		Constant(iScript *script, const Calculator::result_t *val);  // special direct assignment constructor

		virtual iString GetValueAsText() const;
		virtual iString GetTypeAsText() const;
		virtual int GetTypeId() const;

	protected:

		const iString& GetTypeName() const;
	};

	
	template<class T>
	inline int Constant<T>::UsagePattern()
	{
		return 0;
	}


	template<>
	inline int Constant<bool>::UsagePattern()
	{
		return 1;
	}


	template<>
	inline int Constant<int>::GetTypeId() const
	{
		return 1;
	}


	template<>
	inline int Constant<bool>::GetTypeId() const
	{
		return 2;
	}


	template<>
	inline int Constant<float>::GetTypeId() const
	{
		return 3;
	}


	template<>
	inline int Constant<double>::GetTypeId() const
	{
		return 4;
	}


	template<>
	inline const iString& Constant<int>::GetTypeName() const
	{
		static const iString t("int");
		return t;
	}


	template<>
	inline const iString& Constant<bool>::GetTypeName() const
	{
		static const iString t("bool");
		return t;
	}


	template<>
	inline const iString& Constant<float>::GetTypeName() const
	{
		static const iString t("float");
		return t;
	}


	template<>
	inline const iString& Constant<double>::GetTypeName() const
	{
		static const iString t("double");
		return t;
	}


	//
	//  Assignment operation type
	//
	const short _Direct = 0;
	const short _IncAdd = 1; 
	const short _IncSub = 2; 
	const short _IncMul = 4; 
	const short _IncDiv = 8;
	const short _IncAll = 15;


	//
	//  Class for variables.
	//
	template<class T>
	class Variable : public Constant<T>
	{

	public:

		static short NaturalMask();
		static T Combine(T prev, T v, short at);

		Variable(iScript *script, const iString &name);
		Variable(iScript *script, const iString &name, int dim);

		bool Assign(const Calculator::result_t *v, short at, int index);
		bool Assign(T v, int index);
	};


	template<class T>
	inline short Variable<T>::NaturalMask()
	{
		return _IncAll;
	}


	template<>
	inline short Variable<bool>::NaturalMask()
	{
		return _Direct;
	}


	template<class T>
	inline T Variable<T>::Combine(T prev, T v, short at)
	{
		switch(at)
		{
		case _IncAdd:
			{
				prev += v;
				break;
			}
		case _IncSub:
			{
				prev -= v;
				break;
			}
		case _IncMul:
			{
				prev *= v;
				break;
			}
		case _IncDiv:
			{
				prev /= v;
				break;
			}
		default:
			{
				prev = v;
				break;
			}
		}
		return prev;
	}


	template<>
	inline bool Variable<bool>::Combine(bool, bool v, short)
	{
		return v;
	}


	//
	//  Base class for prototypes.
	//
	class Statement;
	class Prototype
	{

		friend class ::iScript;

	public:

		inline const iString& Command() const { return mCommand; }

	protected:

		Prototype(iScript *script, const iString &command);
		virtual ~Prototype();

		virtual Statement* CreateWorker() const = 0;

		iString mCommand;
		iScript *mScript;

	private:

		//
		//  A prototype cannot be compiled except by its script
		//
		bool Compile(const iString &arg, int line, int pos);
	};


	//
	//  Specific prototypes
	//
	template<class T>
	class Prototype0 : public Prototype
	{

		friend class ::iScript;

	protected:

		virtual Statement* CreateWorker() const;

	private:

		//
		//  To be created by script only
		//
		Prototype0(iScript *script, const iString &command) : Prototype(script,command){}
	};


	//
	//  Specific prototypes
	//
	class VariableDeclarationStatement;
	template<class T, class A>
	class Prototype1 : public Prototype
	{

		friend class ::iScript;
		friend class VariableDeclarationStatement;

	protected:

		virtual Statement* CreateWorker() const;

	private:

		//
		//  To be created by script only
		//
		Prototype1(iScript *script, const iString &command, A arg) : Prototype(script,command), mArgument(arg){}

		A mArgument;
	};


	//
	//  Specific prototypes
	//
	template<class T, class A1, class A2>
	class Prototype2 : public Prototype
	{

		friend class ::iScript;

	protected:

		virtual Statement* CreateWorker() const;

	private:

		//
		//  To be created by script only
		//
		Prototype2(iScript *script, const iString &command, A1 arg1, A2 arg2) : Prototype(script,command), mArgument1(arg1), mArgument2(arg2){}

		A1 mArgument1;
		A2 mArgument2;
	};


	//
	//  Base class for statements.
	//
	class Statement
	{

		friend class ::iScript;
		friend class Prototype;
		friend class GenericFlowControlStatement;

	public:

		virtual ~Statement();

		inline int EntryInCode() const { return mEntryInCode; }
		inline int LineInScript() const { return mLineInScript; }
		inline const iString& Command() const { return mCommand; }
		inline const iString& Argument() const { return mArgument; }

	protected:

		Statement(iScript *script, const iString &command);

		virtual bool CompileBody() = 0;
		virtual bool ExecuteBody() = 0;

		//
		//  Helper functiuons needed here and there
		//
		void ShiftOperandWhiteSpace();
		int ParseOperandForIndex(const iString &str, int pos, int &last) const;
		Value* FindVariable(const iString &name) const;

		Calculator* CreateCalculator() const;

		inline void ReportError(const iString &message) const { this->ReportError(message,mOperand); }
		void ReportError(const iString &message, const iString &context) const;
		void ReportError(const Calculator *c) const;

		bool mAcceptsArrays, mOwnsCalculator;
		int mIndex, mEntryInCode, mLineInScript;
		iString mCommand, mOperand, mIndexExpression;
		iScript *mScript;

	private:

		iString mArgument;
		//
		//  A statement cannot be executed except by its script
		//
		bool Execute();
	};


	//
	//  A simple, argument-less function call statement
	//
	class SimpleStatement : public Statement
	{

	public:

		typedef bool (*FunctionType)(iScript *script);

		SimpleStatement(iScript *script, const iString &command, FunctionType f);

	protected:

		virtual bool CompileBody();
		virtual bool ExecuteBody();

		FunctionType mFun;
	};


	//
	//  A deprecated statement
	//
	class DeprecatedStatement : public Statement
	{

	public:

		DeprecatedStatement(iScript *script, const iString &command, const iString &message);

	protected:

		virtual bool CompileBody();
		virtual bool ExecuteBody();

		const iString mMessage;
	};


	//
	//  A programmable function call statement (can be specialized to implement any command)
	//
	class FunctionCallStatement : public Statement
	{

	public:

		typedef bool (*FunctionType1)(iScript *script, const iString &arg);
		typedef bool (*FunctionType2)(iScript *script, const iString &arg, int index);

	public:

		FunctionCallStatement(iScript *script, const iString &command, FunctionType1 f);
		FunctionCallStatement(iScript *script, const iString &command, FunctionType2 f);

	protected:

		virtual bool CompileBody();
		virtual bool ExecuteBody();

		Calculator* GetIndexCalculator() const;

		FunctionType1 mFun1;
		FunctionType2 mFun2;
		Calculator *mIndexCalculator;
	};


	//
	//  Assignment statement
	//
	class AssignmentStatement : public Statement
	{

	public:

		~AssignmentStatement();

	protected:

		AssignmentStatement(iScript *script, const iString &command, short mask, int dim, int use);

		virtual bool CompileBody();
		virtual bool ExecuteBody();
		virtual bool SetValue(const Calculator::result_t *result, int index) = 0;

		Calculator* GetCalculator() const;
		Calculator* GetIndexCalculator() const;
		bool ParseOperandForAssignmentType();

		int mDim, mUse;
		short mAssignmentType, mAssignmentMask;
		Calculator *mCalculator, *mIndexCalculator;
	};


	//
	//  Variable assignment statement
	//
	template<class T>
	class VariableAssignmentStatement : public AssignmentStatement
	{

	public:

		VariableAssignmentStatement(iScript *script, Variable<T> &var);

	protected:

		virtual bool SetValue(const Calculator::result_t *result, int index);

		Variable<T> &mVar;
	};


	//
	//  Function call assignment statement
	//
	template<class T>
	class FunctionCallAssignmentStatement : public AssignmentStatement
	{

	private:

		typedef bool (*FunctionType1)(iScript *script, short at, T result);
		typedef bool (*FunctionType2)(iScript *script, short at, int num, const T *result, int index);

	public:

		FunctionCallAssignmentStatement(iScript *script, const iString &command, FunctionType1 f, short mask = Variable<T>::NaturalMask());
		FunctionCallAssignmentStatement(iScript *script, const iString &command, FunctionType2 f, short mask = Variable<T>::NaturalMask());

	protected:

		virtual bool SetValue(const Calculator::result_t *result, int index);

		FunctionType1 mFun1;
		FunctionType2 mFun2;
	};


	//
	//  FlowControl types
	//
	enum FlowControlType
	{ 
		_Entry = 1,		// enter a branch
		_Exit = 2		// exit a branch
	};

	enum FlowControlState
	{ 
		_Empty = 0,
		_Jumped = 1,
		_Returning = 2
	};

	//
	//  Generic flow control statement
	//
	class GenericFlowControlStatement : public Statement
	{

	public:

		inline bool IsOfType(FlowControlType type) const { return (mType & type) != 0; }
		inline bool TestState(FlowControlState state) const { return (mState & state) != 0; }

	protected:

		GenericFlowControlStatement(iScript *script, const iString &command, FlowControlType type);

		virtual bool CompileBody();
		virtual bool ExecuteBody();

		virtual bool ChildCompileBody() = 0;
		virtual bool ChildExecuteBody() = 0;

		void Jump();

		FlowControlType mType;
		FlowControlState mState;
		GenericFlowControlStatement* mJumpPoint;
	};


	//
	//  Closing flow control statement (like enddo or endif)
	//
	class ClosingFlowControlStatement : public GenericFlowControlStatement
	{

	public:

		ClosingFlowControlStatement(iScript *script, const iString &command);

	protected:

		virtual bool ChildCompileBody();
		virtual bool ChildExecuteBody();
	};


	//
	//  Swapping flow control statement (like else)
	//
	class SwappingFlowControlStatement : public GenericFlowControlStatement
	{

	public:

		SwappingFlowControlStatement(iScript *script, const iString &command);

	protected:

		virtual bool ChildCompileBody();
		virtual bool ChildExecuteBody();

		bool mAllowInLoops;
		GenericFlowControlStatement* mArrivalPoint;
	};


	//
	//  Conditional flow control statement (like if)
	//
	class ConditionalFlowControlStatement : public GenericFlowControlStatement
	{

	public:

		virtual ~ConditionalFlowControlStatement();

	protected:

		ConditionalFlowControlStatement(iScript *script, const iString &command);

		virtual bool ChildCompileBody();
		virtual bool ChildExecuteBody();
		virtual bool ParseOperandForCondition() = 0;

		Calculator* GetCalculator() const;

		Calculator *mCalculator;
	};


	//
	//  Loop flow control statement (like do or for)
	//
	class LoopFlowControlStatement : public GenericFlowControlStatement
	{

	public:

		virtual ~LoopFlowControlStatement();

	protected:

		LoopFlowControlStatement(iScript *script, const iString &command, bool usesSingleCalculator);

		virtual bool ChildCompileBody();
		virtual bool ChildExecuteBody();
		virtual bool ParseOperandForCount() = 0;
		virtual void OnExecute(){}

		Calculator* GetCalculator(int i) const;

		int mIndex, mCount, mStep;
		iString mFirstExpression, mStepExpression;
		Calculator *mCalculator[3];
		bool mUsesSingleCalculator;
	};


	//
	//  Variable declaration statement
	//
	class VariableDeclarationStatement : public Statement
	{

	protected:

		VariableDeclarationStatement(iScript *script, const iString &command);

		virtual bool CompileBody(); // variables are actually created at compile-time
		virtual bool ExecuteBody();
		virtual Prototype* CreatePrototype(const iString &name, int dim) const = 0;

		template<class T>
		Prototype* NewPrototype(const iString &name, int dim) const;

	private:

		iArray<iString> mVarNames;

		template<class T>
		class PrototypeHelper : public Prototype
		{

		public:

			PrototypeHelper(iScript *script, Variable<T> &var) : Prototype(script,var.Name()), mVar(var){}

		protected:

			virtual Statement* CreateWorker() const { return new VariableAssignmentStatement<T>(this->mScript,this->mVar); }
			
		private:

			Variable<T> &mVar;
		};
	};
};

#endif // ISCRIPTKIT_H

