/*
 * linux-kernel specific functions
 * $Id: ifp_os_linux.c,v 1.2 2005/08/25 04:10:54 jim-campbell Exp $
 *
 * Copyright (C) Geoff Oakham, 2004; <oakhamg@users.sourceforge.net>
 */

#include <linux/libusb.h>

#include "ifp.h"
#include "ifp_os.h"
#include "prim.h"

int ifp_locale_to_utf16(char * ob, int max_o, const char * ib, int max_i)
{
	int i = 0;

	//Warning about possible buffer overrun: here 'n' is the
	//size of the buffer 's', but we have no way of preventing
	//'buf' from being overrun.  (We can tell afterwards, but by
	//then it's too late.)

	i = utf8_mbstowcs((wchar_t *)ob, ib, max_i);
	BUG_ON(i > max_o/sizeof(wchar_t));

	return 0;
}
int ifp_utf16_to_locale(char * ob, int max_o, const char * ib, int max_i)
{
	int i = 0;

	i = utf8_wcstombs(ob, (wchar_t *)ib, max_o);
	BUG_ON(i >= max_o);
	ob[i] = '\0';

	return 0;
}

int ifp_os_sleep(int ms) {
	int i;

	//what about this:
	//interruptible_sleep_on_timeout(wait_queue_head_t *q,long timeout);
	//
	//sleep for 1/1000 seconds
	int timeout = (int)((long)HZ*(long)ms/(long)1000);
	current->state = TASK_INTERRUPTIBLE;
	timeout = schedule_timeout(timeout);
	if (timeout) {
		ifp_err("interuptted while waiting for flush.");
		i = -EINTR;
		return i;
	}
	return 0;

}

int ifp_os_control_send(struct ifp_device * dev, int command, int arg1, int arg2,
	int * r1, int * r2)
{
    struct libusb_device * usbdev = dev->device;
	int pipe;
	u8 ctl[8];
	int c_size;
	int i;
	int loops_left = 20;

	BUG_ON(r1 == NULL);
	c_size = r2 ? 8 : 4;

	pipe = usb_rcvctrlpipe(dev->device,0);

	do {
#if 0
		if (i == -EPIPE) {
			i = usb_clear_halt(dev->device, pipe);
			if (i == -EPIPE) {
				//loop?
				ifp_err_i(i, "failed to clear halt");
			} else if (i) {
				ifp_err_jump(i, out,
					"error attempting to clear halt");
			}
		}
#endif
#if 0
		if (command == 0x07) {
			//ifp_os_sleep(5);
			schedule();
		}
#endif
        i = libusb_control_msg(usbdev, pipe, command, IFP_REQ_TYPE,
			arg1, arg2, ctl, c_size, IFP_TIMEOUT);

		//This is a big hack.  I actually have no clue why EPIPE
		//is occasionally returned here, nor how to correctly handle it.
		//
		//When this happens, I find that reissuing the control message
		//once is usually sufficient.  However, the device occasionally
		//behaves as if it successfully received an extra request.
		//
		//In other words, one of the requests that returned EPIPE
		//succeeded.  (But most of them didn't.)  If I could detect
		//which request succeeded, I could work around this.  However,
		//none of the return values tell me.. I've checked:
		//'endpoint_halted', 'i', r1 and r2.

		if (i == -EPIPE) {
			int rr1, rr2 = 0;
			if (r2) {
				rr2 = ifp_os_le32_to_cpup(ctl + 4);
			}
			rr1 = ifp_os_le32_to_cpup(ctl);

			//ifp_dbg("If you see the message \"error reading block.. I expected x bytes but got less\" there's a corruption or a read error.  What USB chipset are you using anyways?  Mail geoff please!");
			ifp_dbg("got epipe; loop=%d (r1=%d, r2=%d)",
				loops_left, rr1, rr2
				);
			if (loops_left != 20 || command != IFP_FILE_DOWNLOAD) {
				ifp_err("wierd case: command=%02x and loop=%d", command,
					20 + 1 - loops_left);
			}
			if (command == IFP_FILE_DOWNLOAD) {
				dev->download_pipe_errors++;
			}

			if (loops_left == 15) {
				ifp_dbg("waiting for device to be available");
			}

			//ifp_dbg("sleeping for a bit");
			ifp_os_sleep(200);
			loops_left--;

		}
		if (!loops_left) {
			ifp_err_i(i, "timeout attempting to %s ifp control code "
				"the command %0x (%d, %d). ctl[%d]  Returned %d.",
				"sending", command, arg1, arg2, c_size, i);
			return i;
		}
	} while (i == -EPIPE);
	if (i < 0) {
		ifp_err_i(i, "error %s ifp control code the command %0x (%d, %d)."
			" ctl[%d]  Returned %d.",
			"sending", command, arg1, arg2, c_size, i);
		return i;
	} else if (i != c_size) {
		if (command == IFP_FILE_DOWNLOAD) { dev->alt_readcount++; }
		ifp_err_i(i, "warning: unexpected error value.. I expected %d.",
			c_size);
	} else {
		if (command == IFP_FILE_DOWNLOAD) { dev->alt_readcount++; }
		i = 0;
	}

	if (r2) {
		*r2 = ifp_os_le32_to_cpup(ctl + 4);
	}
	*r1 = ifp_os_le32_to_cpup(ctl);

	return i;
}

int ifp_os_push(struct ifp_device * dev, void * p, int n)
{
	int i = 0, actual;
	int pipe;

	pipe = usb_sndbulkpipe(dev->device,(dev->bulk_to & 0x7f));

	i = usb_bulk_msg(dev->device, pipe, p, n, &actual, IFP_TIMEOUT);

	if (i == -EPIPE) {
		ifp_dbg("EPIPE status received");

		i = usb_clear_halt(dev->device, pipe);
		ifp_err_jump(i, out, "failed to clear halt");

		i = usb_bulk_msg(dev->device, pipe, p, n, &actual,IFP_TIMEOUT);
		ifp_err_jump(i, out, "failed to reissue xfer");
	} else if (i < 0) {
		ifp_err_i(i, "usb_bulk_msg failed");
		return i;
	} else if (actual != n) {
		ifp_dbg("usb_bulk_msg could only send %d of %d bytes",
			actual, n);
	}
out:

	return i;
}

int ifp_os_pop(struct ifp_device * dev, void * p, int n)
{
	int i = 0, actual;
	int pipe;
	memset(p, 0, n);

	pipe = usb_rcvbulkpipe(dev->device,(dev->bulk_from & 0x7f));

	i = usb_bulk_msg(dev->device, pipe, p, n, &actual, IFP_TIMEOUT);

	if (i == -EPIPE) {
		ifp_dbg("EPIPE status received");

		i = usb_clear_halt(dev->device, pipe);
		ifp_err_jump(i, out, "failed to clear halt");

		i = usb_bulk_msg(dev->device, pipe, p, n, &actual,IFP_TIMEOUT);
		ifp_err_jump(i, out, "failed to reissue xfer");
	} else if (i < 0) {
		ifp_err_i(i, "usb_bulk_msg failed");
		return i;
	} else {
		if (i) {
			ifp_dbg("(warning) i=%d is positive",i);
		}
		if (actual != n) {
			ifp_dbg("(warning) usb_bulk_msg only received %d of %d bytes",
				actual, n);
			return actual ? actual : -1;
		}
	}
out:

	return i;
}

int ifp_os_init(struct ifp_device * dev, void * device_handle)
{
	int i = 0;
    struct libusb_device * ldev = device_handle;
	struct usb_host_interface * interface;
	int address0, address1;

	dev->model = ldev->descriptor.idProduct;
	dev->device = device_handle;

	interface = &ldev->actconfig->interface[0]->altsetting[0];
	address0 = interface->endpoint[0].desc.bEndpointAddress;
	address1 = interface->endpoint[1].desc.bEndpointAddress;
	if (address0 & 0x0080) {
		BUG_ON(address1 & 0x0080);
		dev->bulk_from = address0;
		dev->bulk_to = address1;
	} else {
		BUG_ON((address1 & 0x0080)==0);
		dev->bulk_from = address1;
		dev->bulk_to = address0;
	}

	return i;
}
int ifp_os_finalize(struct ifp_device * dev)
{
	//release happens somewhere else
	dev->device = NULL;
	return 0;
}

void * ifp_find_device(void) {
        int i;
        struct libusb_device * dev = NULL;
	const int iRiver_Vendor = 0x4102;

	//search for working devices first.
	for (i = 0; i != IFP_PRODUCT_IDs; i++) {
                dev = usb_find_device(iRiver_Vendor, ifp_product_ids[i]);
                if (dev) {
                        ifp_info("iRiverFS: detected a %s series device.",
				ifp_product_strings[i]);
                        return dev;
                }
        }

	//We haven't found a compatible device with the 'manager' firmware.
	//To be helpful to the user, let's check if there are any devices
	//using the alternative 'ums' firmware--the firmware that emulates
	//a USB mass storage device.
	for (i = 0; i != IFP_PRODUCT_IDs; i++) {
                dev = usb_find_device(iRiver_Vendor,
			ifp_product_ids[i] | ifp_product_ums_bit);
                if (dev) {
                        ifp_info("iRiverFS: skipping the %s device with 'UMS' "
				"firmware because the USB mass storage driver "
				"will work better with it.\n",
				ifp_product_strings[i]);
                        return NULL;
                }
        }
        return NULL;
}
IFP_EXPORT(ifp_find_device);

int ifp_release_device(void * dev) {
#if 0
	int i = 0;
	i = usb_reset_device(dev);
	if (i) {
		ifp_err_i(i, "usb_reset() returned an error");
	}
#endif
	usb_put_dev(dev);
	return 0;
}
IFP_EXPORT(ifp_release_device);

#if !defined (__KERNEL__)
# error "makefile error"
#endif // __KERNEL__

