/*
 *  $Id: cpde_synth.c 22340 2019-07-25 10:23:59Z yeti-dn $
 *  Copyright (C) 2019 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libgwyddion/gwythreads.h>
#include <libprocess/stats.h>
#include <libprocess/filters.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwyapp.h>
#include "dimensions.h"
#include "preview.h"
#include "libgwyddion/gwyomp.h"

#define CPDE_SYNTH_RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

#define DECLARE_PRESET(name) \
    static gboolean (cpde_##name)(GwyDataField *dfield, \
                                  gdouble *domain, \
                                  const CpdeSynthArgs *args, \
                                  GTimer *timer, \
                                  gdouble preview_time)

enum {
    PAGE_DIMENSIONS = 0,
    PAGE_GENERATOR  = 1,
    PAGE_NPAGES
};

typedef enum {
    CPDE_TURING_PATTERN = 0,
    CPDE_NPRESETS
} CpdeSynthPresetType;

typedef struct _DomainSynthControls CpdeSynthControls;

typedef struct {
    gint active_page;
    gint seed;
    gboolean randomize;
    gboolean animated;
    CpdeSynthPresetType preset;
    guint niters;
    gdouble height;
    gdouble turing_size;
    gdouble turing_chaos;
} CpdeSynthArgs;

typedef gboolean (*CpdeSynthPresetFunc)(GwyDataField *dfield,
                                        gdouble *domain,
                                        const CpdeSynthArgs *args,
                                        GTimer *timer,
                                        gdouble preview_time);

typedef struct {
    CpdeSynthPresetType type;
    const gchar *name;
    CpdeSynthPresetFunc func;
} CpdeSynthPreset;

struct _DomainSynthControls {
    CpdeSynthArgs *args;
    GwyDimensions *dims;
    GtkWidget *dialog;
    GtkWidget *view;
    GtkWidget *update_now;
    GtkWidget *animated;
    GtkObject *seed;
    GtkWidget *randomize;
    GtkTable *table;
    GtkWidget *preset;
    GtkObject *niters;
    GtkObject *height;
    GtkWidget *height_units;
    GtkWidget *height_init;
    GtkObject *size;
    GtkWidget *size_value;
    GtkWidget *size_units;
    GtkObject *chaos;
    GwyContainer *mydata;
    GwyDataField *surface;
    gdouble pxsize;
    gdouble zscale;
    gboolean in_update;
};

static gboolean   module_register      (void);
static void       cpde_synth           (GwyContainer *data,
                                        GwyRunType run);
static void       run_noninteractive   (CpdeSynthArgs *args,
                                        GwyDimensionArgs *dimsargs,
                                        GwyContainer *data,
                                        GwyDataField *dfield,
                                        gint oldid,
                                        GQuark quark);
static gboolean   cpde_synth_dialog    (CpdeSynthArgs *args,
                                        GwyDimensionArgs *dimsargs,
                                        GwyContainer *data,
                                        GwyDataField *dfield,
                                        gint id);
static void       update_controls      (CpdeSynthControls *controls,
                                        CpdeSynthArgs *args);
static GtkWidget* preset_selector_new  (CpdeSynthControls *controls);
static void       page_switched        (CpdeSynthControls *controls,
                                        GtkNotebookPage *page,
                                        gint pagenum);
static void       preset_type_selected (GtkComboBox *combo,
                                        CpdeSynthControls *controls);
static void       update_values        (CpdeSynthControls *controls);
static void       update_sensitivity   (CpdeSynthControls *controls);
static void       height_init_clicked  (CpdeSynthControls *controls);
static void       cpde_synth_invalidate(CpdeSynthControls *controls);
static void       preview              (CpdeSynthControls *controls);
static gboolean   cpde_synth_do        (CpdeSynthArgs *args,
                                        GwyDataField *dfield,
                                        gdouble preview_time);
static void       init_field_randomly  (GwyDataField *dfield,
                                        guint32 seed);
static void       cpde_synth_load_args (GwyContainer *container,
                                        CpdeSynthArgs *args,
                                        GwyDimensionArgs *dimsargs);
static void       cpde_synth_save_args (GwyContainer *container,
                                        const CpdeSynthArgs *args,
                                        const GwyDimensionArgs *dimsargs);

DECLARE_PRESET(turing_pattern);

#define GWY_SYNTH_CONTROLS CpdeSynthControls
#define GWY_SYNTH_INVALIDATE(controls) cpde_synth_invalidate(controls)

#include "synth.h"

static const CpdeSynthArgs cpde_synth_defaults = {
    PAGE_DIMENSIONS,
    42, TRUE, TRUE,
    CPDE_TURING_PATTERN,
    10000,
    1.0,
    8.0, 0.25,
};

static const GwyDimensionArgs dims_defaults = GWY_DIMENSION_ARGS_INIT;

static const CpdeSynthPreset presets[] = {
    {
        CPDE_TURING_PATTERN,
        N_("Turing pattern"),
        &cpde_turing_pattern,
    },
};

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Generates images by assorted coupled partial "
       "differential equation models."),
    "Yeti <yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti)",
    "2019",
};

GWY_MODULE_QUERY2(module_info, cpde_synth)

static gboolean
module_register(void)
{
    gwy_process_func_register("cpde_synth",
                              (GwyProcessFunc)&cpde_synth,
                              N_("/S_ynthetic/Coupled PD_Es..."),
                              GWY_STOCK_SYNTHETIC_TURING_PATTERN,
                              CPDE_SYNTH_RUN_MODES,
                              0,
                              N_("Generate image by coupled PDEs"));

    return TRUE;
}

static void
cpde_synth(GwyContainer *data, GwyRunType run)
{
    CpdeSynthArgs args;
    GwyDimensionArgs dimsargs;
    GwyDataField *dfield;
    GQuark quark;
    gint id;

    g_return_if_fail(run & CPDE_SYNTH_RUN_MODES);
    cpde_synth_load_args(gwy_app_settings_get(), &args, &dimsargs);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     GWY_APP_DATA_FIELD_KEY, &quark,
                                     0);

    if (run == GWY_RUN_IMMEDIATE
        || cpde_synth_dialog(&args, &dimsargs, data, dfield, id)) {
        run_noninteractive(&args, &dimsargs, data, dfield, id, quark);
    }

    gwy_dimensions_free_args(&dimsargs);
}

static void
run_noninteractive(CpdeSynthArgs *args,
                   GwyDimensionArgs *dimsargs,
                   GwyContainer *data,
                   GwyDataField *dfield,
                   gint oldid,
                   GQuark quark)
{
    GwySIUnit *siunit;
    gboolean replace = dimsargs->replace && dfield;
    gboolean add = dimsargs->add && dfield;
    gint newid = -1;
    gboolean ok, wait_enabled;

    if (args->randomize)
        args->seed = g_random_int() & 0x7fffffff;

    if (add) {
        dfield = gwy_data_field_duplicate(dfield);
        gwy_data_field_renormalize(dfield, -0.5, 0.5);
    }
    else if (replace) {
        dfield = gwy_data_field_new_alike(dfield, TRUE);
        init_field_randomly(dfield, args->seed);
    }
    else {
        gdouble mag = pow10(dimsargs->xypow10) * dimsargs->measure;
        dfield = gwy_data_field_new(dimsargs->xres, dimsargs->yres,
                                    mag*dimsargs->xres, mag*dimsargs->yres,
                                    TRUE);
        init_field_randomly(dfield, args->seed);

        siunit = gwy_data_field_get_si_unit_xy(dfield);
        gwy_si_unit_set_from_string(siunit, dimsargs->xyunits);

        siunit = gwy_data_field_get_si_unit_z(dfield);
        gwy_si_unit_set_from_string(siunit, dimsargs->zunits);
    }

    wait_enabled = gwy_app_wait_get_enabled();
    if (wait_enabled && args->animated) {
        set_up_wait_data_field_preview(dfield, data, oldid);
    }
    gwy_app_wait_start(gwy_app_find_window_for_channel(data, oldid),
                       _("Initializing..."));
    ok = cpde_synth_do(args, dfield, wait_enabled ? 1.25 : 0.0);
    gwy_app_wait_finish();

    if (!ok) {
        g_object_unref(dfield);
        return;
    }

    gwy_data_field_renormalize(dfield,
                               pow10(dimsargs->zpow10) * args->height, 0.0);

    if (replace) {
        gwy_app_undo_qcheckpointv(data, 1, &quark);
        gwy_container_set_object(data, gwy_app_get_data_key_for_id(oldid),
                                 dfield);
        gwy_app_channel_log_add_proc(data, oldid, oldid);
        g_object_unref(dfield);
        return;
    }

    if (data) {
        newid = gwy_app_data_browser_add_data_field(dfield, data, TRUE);
    }
    else {
        data = gwy_container_new();
        newid = 0;
        gwy_container_set_object(data, gwy_app_get_data_key_for_id(newid),
                                 dfield);
        gwy_app_data_browser_add(data);
        gwy_app_data_browser_reset_visibility(data,
                                              GWY_VISIBILITY_RESET_SHOW_ALL);
        g_object_unref(data);
    }

    if (oldid != -1) {
        gwy_app_sync_data_items(data, data, oldid, newid, FALSE,
                                GWY_DATA_ITEM_GRADIENT,
                                0);
    }
    gwy_app_set_data_field_title(data, newid, _("Generated"));
    gwy_app_channel_log_add_proc(data, add ? oldid : -1, newid);
    g_object_unref(dfield);
}

static gboolean
cpde_synth_dialog(CpdeSynthArgs *args,
                  GwyDimensionArgs *dimsargs,
                  GwyContainer *data,
                  GwyDataField *dfield_template,
                  gint id)
{
    GtkWidget *dialog, *table, *vbox, *hbox, *notebook;
    CpdeSynthControls controls;
    GwyDataField *dfield;
    gboolean finished;
    gint row, response;

    gwy_clear(&controls, 1);
    controls.in_update = TRUE;
    controls.args = args;
    controls.pxsize = 1.0;
    dialog = gtk_dialog_new_with_buttons(_("Coupled PDEs"),
                                         NULL, 0,
                                         _("_Reset"), RESPONSE_RESET,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                         GTK_STOCK_OK, GTK_RESPONSE_OK,
                                         NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialog), GWY_HELP_DEFAULT);
    controls.dialog = dialog;

    hbox = gtk_hbox_new(FALSE, 2);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox,
                       FALSE, FALSE, 4);

    vbox = gtk_vbox_new(FALSE, 4);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 4);

    controls.mydata = gwy_container_new();
    dfield = gwy_data_field_new(PREVIEW_SIZE, PREVIEW_SIZE,
                                dimsargs->measure*PREVIEW_SIZE,
                                dimsargs->measure*PREVIEW_SIZE,
                                TRUE);
    gwy_container_set_object_by_name(controls.mydata, "/0/data", dfield);

    if (dfield_template) {
        gdouble min, max;

        gwy_data_field_get_min_max(dfield_template, &min, &max);
        gwy_app_sync_data_items(data, controls.mydata, id, 0, FALSE,
                                GWY_DATA_ITEM_PALETTE,
                                0);
        controls.surface = gwy_synth_surface_for_preview(dfield_template,
                                                         PREVIEW_SIZE);
        controls.zscale = max - min;
    }
    controls.view = create_preview(controls.mydata, 0, PREVIEW_SIZE,
                                   FALSE);
    gtk_box_pack_start(GTK_BOX(vbox), controls.view, FALSE, FALSE, 0);

    gtk_box_pack_start(GTK_BOX(vbox),
                       gwy_synth_progressive_preview_new(&controls,
                                                         &controls.update_now,
                                                         &controls.animated,
                                                         &args->animated),
                       FALSE, FALSE, 0);
    g_signal_connect_swapped(controls.update_now, "clicked",
                             G_CALLBACK(preview), &controls);

    gtk_box_pack_start(GTK_BOX(vbox),
                       gwy_synth_random_seed_new(&controls,
                                                 &controls.seed, &args->seed),
                       FALSE, FALSE, 0);

    controls.randomize = gwy_synth_randomize_new(&args->randomize);
    gtk_box_pack_start(GTK_BOX(vbox), controls.randomize, FALSE, FALSE, 0);

    notebook = gtk_notebook_new();
    gtk_box_pack_start(GTK_BOX(hbox), notebook, TRUE, TRUE, 4);
    g_signal_connect_swapped(notebook, "switch-page",
                             G_CALLBACK(page_switched), &controls);

    controls.dims = gwy_dimensions_new(dimsargs, dfield_template);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
                             gwy_dimensions_get_widget(controls.dims),
                             gtk_label_new(_("Dimensions")));

    table = gtk_table_new(5 + (dfield_template ? 1 : 0), 3, FALSE);
    controls.table = GTK_TABLE(table);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table,
                             gtk_label_new(_("Generator")));
    row = 0;

    gtk_table_attach(GTK_TABLE(table),
                     gwy_label_new_header(_("Simulation Parameters")),
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.preset = preset_selector_new(&controls);
    gwy_table_attach_adjbar(table, row++, _("_Pattern:"), NULL,
                            GTK_OBJECT(controls.preset),
                            GWY_HSCALE_WIDGET_NO_EXPAND);

    controls.size = gtk_adjustment_new(args->turing_size,
                                       1.0, 100.0, 0.1, 10.0, 0);
    row = gwy_synth_attach_lateral(&controls, row, controls.size,
                                   &args->turing_size,
                                   _("_Size:"), GWY_HSCALE_SQRT, NULL,
                                   &controls.size_value, &controls.size_units);

    controls.chaos = gtk_adjustment_new(args->turing_chaos,
                                        0.0, 1.0, 0.001, 0.1, 0);
    g_object_set_data(G_OBJECT(controls.chaos), "target", &args->turing_chaos);
    gwy_table_attach_adjbar(table, row++, _("Degree of _chaos:"), NULL,
                            GTK_OBJECT(controls.chaos), GWY_HSCALE_LINEAR);
    g_signal_connect_swapped(controls.chaos, "value-changed",
                             G_CALLBACK(gwy_synth_double_changed), &controls);

    controls.niters = gtk_adjustment_new(args->niters, 1, 200000, 1, 10, 0);
    g_object_set_data(G_OBJECT(controls.niters), "target", &args->niters);
    gwy_table_attach_adjbar(table, row++, _("_Number of iterations:"), NULL,
                            GTK_OBJECT(controls.niters), GWY_HSCALE_SQRT);
    g_signal_connect_swapped(controls.niters, "value-changed",
                             G_CALLBACK(gwy_synth_int_changed), &controls);

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    gtk_table_attach(GTK_TABLE(table),
                     gwy_label_new_header(_("Output Options")),
                     0, 3, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    row = gwy_synth_attach_height(&controls, row,
                                  &controls.height, &args->height,
                                  _("_Height:"), NULL, &controls.height_units);

    if (dfield_template) {
        controls.height_init
            = gtk_button_new_with_mnemonic(_("_Like Current Image"));
        g_signal_connect_swapped(controls.height_init, "clicked",
                                 G_CALLBACK(height_init_clicked), &controls);
        gtk_table_attach(GTK_TABLE(table), controls.height_init,
                         0, 2, row, row+1, GTK_FILL, 0, 0, 0);
        row++;
    }

    gtk_widget_show_all(dialog);
    controls.in_update = FALSE;
    /* Must be done when widgets are shown, see GtkNotebook docs */
    gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), args->active_page);
    update_values(&controls);
    update_sensitivity(&controls);

    finished = FALSE;
    while (!finished) {
        response = gtk_dialog_run(GTK_DIALOG(dialog));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            case GTK_RESPONSE_OK:
            gtk_widget_destroy(dialog);
            case GTK_RESPONSE_NONE:
            finished = TRUE;
            break;

            case RESPONSE_RESET:
            {
                gint temp2 = args->active_page;
                *args = cpde_synth_defaults;
                args->active_page = temp2;
            }
            controls.in_update = TRUE;
            update_controls(&controls, args);
            controls.in_update = FALSE;
            break;

            default:
            g_assert_not_reached();
            break;
        }
    }

    cpde_synth_save_args(gwy_app_settings_get(), args, dimsargs);

    g_object_unref(controls.mydata);
    GWY_OBJECT_UNREF(controls.surface);
    gwy_dimensions_free(controls.dims);

    return response == GTK_RESPONSE_OK;
}

static GtkWidget*
preset_selector_new(CpdeSynthControls *controls)
{
    GtkWidget *combo;
    GwyEnum *model;
    guint n, i;

    n = G_N_ELEMENTS(presets);
    model = g_new(GwyEnum, n);
    for (i = 0; i < n; i++) {
        model[i].value = presets[i].type;
        model[i].name = presets[i].name;
    }

    combo = gwy_enum_combo_box_new(model, n,
                                   G_CALLBACK(preset_type_selected), controls,
                                   controls->args->preset, TRUE);
    g_object_weak_ref(G_OBJECT(combo), (GWeakNotify)g_free, model);

    return combo;
}

static void
update_controls(CpdeSynthControls *controls,
                CpdeSynthArgs *args)
{
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->seed), args->seed);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->randomize),
                                 args->randomize);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->animated),
                                 args->animated);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->niters), args->niters);
    gwy_enum_combo_box_set_active(GTK_COMBO_BOX(controls->preset),
                                  args->preset);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->size),
                             args->turing_size);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->chaos),
                             args->turing_chaos);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->height), args->height);
}

static void
preset_type_selected(GtkComboBox *combo, CpdeSynthControls *controls)
{
    controls->args->preset = gwy_enum_combo_box_get_active(combo);
    /* TODO anything? */
}

static void
page_switched(CpdeSynthControls *controls,
              G_GNUC_UNUSED GtkNotebookPage *page,
              gint pagenum)
{
    if (controls->in_update)
        return;

    controls->args->active_page = pagenum;
    if (pagenum == PAGE_GENERATOR)
        update_values(controls);
}

static void
update_values(CpdeSynthControls *controls)
{
    GwyDimensions *dims = controls->dims;

    controls->pxsize = dims->args->measure * pow10(dims->args->xypow10);
    if (controls->height_units)
        gtk_label_set_markup(GTK_LABEL(controls->height_units),
                             dims->zvf->units);
    gtk_label_set_markup(GTK_LABEL(controls->size_units),
                         dims->xyvf->units);

    gwy_synth_update_lateral(controls, GTK_ADJUSTMENT(controls->size));
}

static void
height_init_clicked(CpdeSynthControls *controls)
{
    gdouble mag = pow10(controls->dims->args->zpow10);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->height),
                             controls->zscale/mag);
}

static void
cpde_synth_invalidate(G_GNUC_UNUSED CpdeSynthControls *controls)
{
}

static void
update_sensitivity(G_GNUC_UNUSED CpdeSynthControls *controls)
{
}

static void
preview(CpdeSynthControls *controls)
{
    CpdeSynthArgs *args = controls->args;
    GwyDataField *dfield;

    dfield = GWY_DATA_FIELD(gwy_container_get_object_by_name(controls->mydata,
                                                             "/0/data"));

    if (controls->dims->args->add && controls->surface) {
        gwy_data_field_copy(controls->surface, dfield, FALSE);
        gwy_data_field_renormalize(dfield, -0.5, 0.5);
    }
    else
        init_field_randomly(dfield, args->seed);

    gwy_app_wait_start(GTK_WINDOW(controls->dialog), _("Initializing..."));
    cpde_synth_do(args, dfield, 1.25);
    gwy_app_wait_finish();
}

static void
copy_domain_to_data_field(GwyDataField *dfield,
                          gdouble *domain,
                          guint which)
{
    gint xres, yres, i, n;
    gdouble *d;

    g_assert(which < 4);
    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    n = xres*yres;
    d = gwy_data_field_get_data(dfield);
    for (i = 0; i < n; i++)
        d[i] = domain[i + n*which];

    gwy_data_field_data_changed(dfield);
}

static gboolean
cpde_synth_do(CpdeSynthArgs *args,
              GwyDataField *dfield,
              gdouble preview_time)
{
    gint xres, yres, i;
    gdouble *d, *domain;
    CpdeSynthPresetFunc preset_func;
    GTimer *timer;
    gboolean ok = FALSE;

    if (!args->animated)
        preview_time = 0.0;

    preset_func = presets[args->preset].func;
    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    d = gwy_data_field_get_data(dfield);
    domain = g_new(gdouble, 5*xres*yres);
    for (i = 0; i < xres*yres; i++)
        domain[i + xres*yres] = domain[i] = d[i] - 0.5;

    timer = g_timer_new();
    gwy_synth_update_progress(NULL, 0, 0, 0);
    if (!gwy_app_wait_set_message(_("Running computation...")))
        goto fail;

    if ((ok = preset_func(dfield, domain, args, timer, preview_time)))
        copy_domain_to_data_field(dfield, domain, 0);

fail:
    g_free(domain);
    g_timer_destroy(timer);

    return ok;
}

/*
 * Funny nonlinear function.  It has following properties
 * - odd
 * - large positive derivative at 0
 * - maximum at some positive value
 * - zero at some larger value
 * - then negative, but not too much
 */
static inline gdouble
funny_func(gdouble x)
{
    return x/(1.0 + 0.01*x*x) - 0.01*x;
}

/* Mixed rectangular-diagonal Laplacian. */
static inline gdouble
laplacian8(const gdouble *rowm, const gdouble *row, const gdouble *rowp,
           guint jm, guint j, guint jp)
{
    return (rowm[j] + row[jm] + row[jp] + rowp[j]
            + 0.25*(rowm[jm] + rowm[jp] + rowp[jm] + rowp[jp])
            - 5.0*row[j]);
}

static inline gdouble
smooth8(const gdouble *rowm, const gdouble *row, const gdouble *rowp,
        guint jm, guint j, guint jp)
{
    return (row[j]
            + 0.125*(rowm[j] + row[jm] + row[jp] + rowp[j])
            + 0.03125*(rowm[jm] + rowm[jp] + rowp[jm] + rowp[jp]))/1.625;
}

static gdouble
checker_smooth(gint xres, gint yres, gdouble *r, gdouble *tmp)
{
    gdouble rr = 0.0;
    gint i;

#ifdef _OPENMP
#pragma omp parallel for if (gwy_threads_are_enabled()) default(none) \
            reduction(+:rr) \
            shared(r,tmp,xres,yres) \
            private(i)
#endif
    for (i = 0; i < yres; i++) {
        gint ix = i*xres;
        gint ixp = ((i + 1) % yres)*xres;
        gint ixm = ((i + yres-1) % yres)*xres;
        gdouble t;
        gint j;

        t = smooth8(r + ixm, r + ix, r + ixp, xres-1, 0, 1);
        tmp[ix] = t;
        rr += t*t;

        for (j = 1; j < xres-1; j++) {
            t = smooth8(r + ixm, r + ix, r + ixp, j-1, j, j+1);
            tmp[ix + j] = t;
            rr += t*t;
        }

        t = smooth8(r + ixm, r + ix, r + ixp, xres-2, xres-1, 0);
        tmp[ix + xres-1] = t;
        rr += t*t;
    }

    gwy_assign(r, tmp, xres*yres);

    return rr;
}

static void
do_iter_turing(gint xres, gint yres, gdouble *domain,
               const gdouble *params, gdouble size)
{
    gdouble realdt, cr0, cr1, rr0, rr1;
    gint i, n = xres*yres;

    cr0 = cr1 = 0.0;
#ifdef _OPENMP
#pragma omp parallel for if (gwy_threads_are_enabled()) default(none) \
            reduction(+:cr0,cr1) \
            shared(domain,xres,yres,n,params,size) \
            private(i)
#endif
    for (i = 0; i < yres; i++) {
        gdouble *c0 = domain;
        gdouble *c1 = domain + n;
        gdouble *r0 = domain + 2*n;
        gdouble *r1 = domain + 3*n;
        const gdouble p = params[0];
        const gdouble q = params[1];
        const gdouble p0 = params[2];
        const gdouble q0 = params[3];
        gdouble h = params[4]/size;
        gdouble mu0h = 0.00001/h/h;
        gdouble mu1h = 0.0001/h/h;
        gint ix = i*xres;
        gint ixp = ((i + 1) % yres)*xres;
        gint ixm = ((i + yres-1) % yres)*xres;
        gdouble cx0, cx1, c0lap, c1lap;
        gint j;

        cx0 = c0[ix];
        cx1 = c1[ix];
        c0lap = laplacian8(c0 + ixm, c0 + ix, c0 + ixp, xres-1, 0, 1);
        c1lap = laplacian8(c1 + ixm, c1 + ix, c1 + ixp, xres-1, 0, 1);
        r0[ix] = q0*funny_func(cx0) + q*cx1 + mu0h*c0lap;
        r1[ix] = p0*funny_func(cx1) + p*cx0 + mu1h*c1lap;
        cr0 += cx0*cx0;
        cr1 += cx1*cx1;

        for (j = 1; j < xres-1; j++) {
            cx0 = c0[ix + j];
            cx1 = c1[ix + j];
            c0lap = laplacian8(c0 + ixm, c0 + ix, c0 + ixp, j-1, j, j+1);
            c1lap = laplacian8(c1 + ixm, c1 + ix, c1 + ixp, j-1, j, j+1);
            r0[ix + j] = q0*funny_func(cx0) + q*cx1 + mu0h*c0lap;
            r1[ix + j] = p0*funny_func(cx1) + p*cx0 + mu1h*c1lap;
            cr0 += cx0*cx0;
            cr1 += cx1*cx1;
        }

        cx0 = c0[ix + xres-1];
        cx1 = c1[ix + xres-1];
        c0lap = laplacian8(c0 + ixm, c0 + ix, c0 + ixp, xres-2, xres-1, 0);
        c1lap = laplacian8(c1 + ixm, c1 + ix, c1 + ixp, xres-2, xres-1, 0);
        r0[ix + xres-1] = q0*funny_func(cx0) + q*cx1 + mu0h*c0lap;
        r1[ix + xres-1] = p0*funny_func(cx1) + p*cx0 + mu1h*c1lap;
        cr0 += cx0*cx0;
        cr1 += cx1*cx1;
    }

    rr0 = checker_smooth(xres, yres, domain + 2*n, domain + 4*n);
    rr0 = sqrt(cr0/rr0);
    rr1 = checker_smooth(xres, yres, domain + 3*n, domain + 4*n);
    rr1 = sqrt(cr1/rr1);
    realdt = 0.5*MIN(rr0, rr1);
    for (i = 0; i < 2*n; i++)
        domain[i] += realdt*domain[2*n + i];
}

static gboolean
cpde_turing_pattern(GwyDataField *dfield, gdouble *domain,
                    const CpdeSynthArgs *args,
                    GTimer *timer, gdouble preview_time)
{
    GwySynthUpdateType update;
    gint xres, yres;
    guint i, niters = args->niters;
    gdouble params[5];

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);

    params[0] = 1.12;
    params[1] = -1.4;
    params[2] = -1.10 - 0.9*args->turing_chaos;
    params[3] = 0.75 + 0.5*args->turing_chaos;
    params[4] = G_PI/(138.0 - 18.0*args->turing_chaos);

    for (i = 0; i < niters; i++) {
        do_iter_turing(xres, yres, domain, params, args->turing_size);
        if (i % 20 == 0) {
            update = gwy_synth_update_progress(timer, preview_time, i, niters);
            if (update == GWY_SYNTH_UPDATE_CANCELLED)
                return FALSE;
            if (update == GWY_SYNTH_UPDATE_DO_PREVIEW)
                copy_domain_to_data_field(dfield, domain, 0);
        }
    }

    return TRUE;
}

static void
init_field_randomly(GwyDataField *dfield, guint32 seed)
{
    gdouble *d = gwy_data_field_get_data(dfield);
    gint xres = gwy_data_field_get_xres(dfield);
    gint yres = gwy_data_field_get_yres(dfield);
    GRand *rng = g_rand_new();
    gint n = xres*yres, k;

    g_rand_set_seed(rng, seed);
    for (k = 0; k < n; k++)
        d[k] = g_rand_double(rng);

    g_rand_free(rng);
}

static const gchar prefix[]           = "/module/cpde_synth";
static const gchar active_page_key[]  = "/module/cpde_synth/active_page";
static const gchar animated_key[]     = "/module/cpde_synth/animated";
static const gchar height_key[]       = "/module/cpde_synth/height";
static const gchar niters_key[]       = "/module/cpde_synth/niters";
static const gchar preset_key[]       = "/module/cpde_synth/preset";
static const gchar randomize_key[]    = "/module/cpde_synth/randomize";
static const gchar turing_size_key[]  = "/module/cpde_synth/turing/size";
static const gchar turing_chaos_key[] = "/module/cpde_synth/turing/chaos";
static const gchar seed_key[]         = "/module/cpde_synth/seed";

static void
cpde_synth_sanitize_args(CpdeSynthArgs *args)
{
    args->active_page = CLAMP(args->active_page,
                              PAGE_DIMENSIONS, PAGE_NPAGES-1);
    args->seed = MAX(0, args->seed);
    args->randomize = !!args->randomize;
    args->animated = !!args->animated;
    args->niters = MIN(args->niters, 200000);
    args->preset = MIN(args->preset, CPDE_NPRESETS-1);
    args->turing_size = CLAMP(args->turing_size, 2.2, 100.0);
    args->turing_chaos = CLAMP(args->turing_chaos, 0.0, 1.0);
    args->height = CLAMP(args->height, 0.001, 10000.0);
}

static void
cpde_synth_load_args(GwyContainer *container,
                     CpdeSynthArgs *args,
                     GwyDimensionArgs *dimsargs)
{
    *args = cpde_synth_defaults;

    gwy_container_gis_int32_by_name(container, active_page_key,
                                    &args->active_page);
    gwy_container_gis_int32_by_name(container, seed_key, &args->seed);
    gwy_container_gis_boolean_by_name(container, randomize_key,
                                      &args->randomize);
    gwy_container_gis_boolean_by_name(container, animated_key,
                                      &args->animated);
    gwy_container_gis_int32_by_name(container, niters_key, &args->niters);
    gwy_container_gis_enum_by_name(container, preset_key, &args->preset);
    gwy_container_gis_double_by_name(container, turing_size_key,
                                     &args->turing_size);
    gwy_container_gis_double_by_name(container, turing_chaos_key,
                                     &args->turing_chaos);
    gwy_container_gis_double_by_name(container, height_key, &args->height);
    cpde_synth_sanitize_args(args);

    gwy_clear(dimsargs, 1);
    gwy_dimensions_copy_args(&dims_defaults, dimsargs);
    gwy_dimensions_load_args(dimsargs, container, prefix);
}

static void
cpde_synth_save_args(GwyContainer *container,
                     const CpdeSynthArgs *args,
                     const GwyDimensionArgs *dimsargs)
{
    gwy_container_set_int32_by_name(container, active_page_key,
                                    args->active_page);
    gwy_container_set_int32_by_name(container, seed_key, args->seed);
    gwy_container_set_boolean_by_name(container, randomize_key,
                                      args->randomize);
    gwy_container_set_boolean_by_name(container, animated_key,
                                      args->animated);
    gwy_container_set_int32_by_name(container, niters_key, args->niters);
    gwy_container_set_enum_by_name(container, preset_key, args->preset);
    gwy_container_set_double_by_name(container, turing_size_key,
                                     args->turing_size);
    gwy_container_set_double_by_name(container, turing_chaos_key,
                                     args->turing_chaos);
    gwy_container_set_double_by_name(container, height_key, args->height);

    gwy_dimensions_save_args(dimsargs, container, prefix);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
