------------------------------------------------------------------------------
--  Thin Ada95 binding to OCI (Oracle Call Interface)                    --
--  Copyright (C) 2000-2007 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: demo.adb,v 1.7 2007/10/22 14:02:27 vagul Exp $

--  Example of Logon and statement execution.

with
   Ada.Text_IO,
   OCI.Thick.OCIDate_Var,
   OCI.Thick.OCINumber_Var,
   OCI.Thick.String_Bounded,
   OCI.Thick.Servers,
   OCI.Thick.Connections,
   OCI.Thick.Statements,
   OCI.Thick.Number_Functions,
   OCI.Thick.Strings;

procedure Demo is
   use OCI.Thick;
   use Ada.Text_IO;

   Server : Servers.Server := Servers.Attach ("");
   --  Using the OCI.Thick.Servers package.
   --  Creating server attachment.
   --  Empty name mean local server.

   Connect : Connections.Connection := Connections.Logon (
      DB => Server,
      User => "scott",
      Password => "tiger");
   --  Using the OCI.Thick.Connections package.
   --  Logon to the database and creating database connection

   Easy_Connect : Connections.Connection := Connections.Logon
                                              (Server_Name => "",
                                               User        => "scott",
                                               Password    => "tiger");
   --  We can create connection without previous server attached,
   --  Just spetsify the server name instead of server.

   Stmt_Text : String := "select "
       & "OBJECT_NAME, " --  VARCHAR2(128)
       & "OBJECT_TYPE, " --  VARCHAR2(18)
       & "OBJECT_ID, "   --  NUMBER
       & "CREATED "      --  DATE
       & "from user_objects where STATUS = :STATUS";

   Stmt_OCI : Statements.Statement := Statements.Prepare (Stmt_Text);
   --  Using the OCI.Thick.Statements package.
   --  Prepearing statement.
   --  OCI is prepearing statements without any communication with a server.

   Stmt : Statements.Statement := Statements.Prepare (Connect, Stmt_Text);
   --  Thick binding allow you to emulate ODBC like behavior of prepare.
   --  It looks like we are using connection for prepearing statement,
   --  but we are just remember Connection inside of Statement for the
   --  execute later.

   --  Define variables for fetch from cursor.

   Object_Name : Strings.Variable;             --  Unlimited length string.
   Object_Type : String_Bounded.Variable (18); --  Fixed length string.
   Object_Id   : OCINumber_Var.Variable;       --  Number
   Created     : OCIDate_Var.Variable;         --  Date

   Status_OCI  : Strings.Variable;

   Status      : Strings.Variable;
   --  Bind variable for Status parameter.

begin
   --  We can see the server version information.

   Put_Line (Connections.Server_Version (Connect));
   New_Line;

   Strings.Bind (Stmt_OCI, Status_OCI, "Status");

   Strings.Bind (Stmt, Status, "Status");
   Strings.Set_Value (Status, "VALID");

   --  Statements prepeared without connection have to be executed only
   --  with connection.

   Statements.Execute (Easy_Connect, Stmt_OCI);

   --  Statements prepeared with connection should be executed
   --  without connection.

   Statements.Execute (Stmt);

   Strings.Define        (Stmt, Object_Name, 1);
   String_Bounded.Define (Stmt, Object_Type, 2);
   OCINumber_Var.Define  (Stmt, Object_Id, 3);
   OCIDate_Var.Define    (Stmt, Created, 4);

   while Statements.Fetch (Stmt) loop
      Put (Strings.Value (Object_Name));

      Set_Col (24);
      Put (String_Bounded.Value (Object_Type));

      Set_Col (32);
      Put_Line (Number_Functions.To_String
             (OCINumber_Var.Value (Object_Id), "9999999"));

   end loop;
end Demo;
