------------------------------------------------------------------------------
--  Ada95 Interface to Oracle RDBMS                                         --
--  Copyright (C) 2006-2008 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-db.adb,v 1.35 2008/07/03 06:22:48 vagul Exp $

with
   Ada.Characters.Handling,
   Ada.Exceptions,
   Ada.Strings.Fixed,
   Ada.Strings.Maps,
   Ada.Unchecked_Conversion,
   Ada.Unchecked_Deallocation,
   Interfaces.C.Strings,
   OCI.Thread;

--  with Ada.Text_IO;
--  with System.Address_Image;

package body OCI.Thick.DB is

   --  procedure Print (Item : in String) renames Ada.Text_IO.Put_Line;

   use type SWord;
   use type Ub4;
   use type OCIHandle;

   Dummy_Connect : Connection;

   use Ada.Strings.Maps;

   package Dates renames Date.Internal;

   type Context_Type is record
      In_Data  : access procedure
                          (Data      :    out Data_Holder;
                           Position  : in     Positive;
                           Iteration : in     Positive);
      Out_Data : access procedure
                          (Data      : in Data_Holder;
                           Position  : in Positive;
                           Iteration : in Positive);
      Holder    : Data_Holder;
      Buffer    : aliased Buffer_Type;
      Alen      : aliased Ub4;  -- Output length
      Ind       : aliased Sb2;  -- Output null indicator
      Rcode     : aliased Ub2;  -- Output return code
      Piece     : Ub1; -- Output piece
      Position  : Natural := 0; -- Output position
      Iteration : Natural := 0; -- Output iteration
      Kind      : Data_Type; -- Output data type
      Loc       : Lobs.Locator; -- Output locator
      Output    : Boolean := False;
      Str_Low   : Natural := 0;
      Connect   : Connection; --  to be able to create lob locator.
      Defines   : Define_Array_Access;
      Error     : Ada.Exceptions.Exception_Occurrence; -- from callback
      Iters     : Natural; -- To check the number of iterations in callbacks
   end record;

   To_Data_Type : constant array (Lobs.Lob_Type) of Data_Type
     := (Lobs.Bin => Type_Bin_Lob, Lobs.File => Type_Bin_Lob,
         Lobs.Char => Type_Char_Lob, Lobs.NChar => Type_Char_Lob);

   subtype Lob_Subtype is Data_Type range Type_Char_Lob .. Type_Bin_Lob;

   To_Lob_Type : constant array (Lob_Subtype) of Lobs.Lob_Type
     := (Type_Char_Lob => Lobs.Char, Type_Bin_Lob => Lobs.Bin);

   Dos_Fix_CR : constant Character_Mapping := To_Mapping ("" & ASCII.CR, " ");

   Size_Of : constant array (Data_Type) of Sb4
     := (Type_Integer    => Sb4 (Integer'Size / System.Storage_Unit),
         Type_Long_Float => Sb4 (Long_Float'Size / System.Storage_Unit),
         Type_String     => Sb4 (Sb2'Last),
         Type_Number     => Sb4 (OCINumber'Size / System.Storage_Unit),
         Type_Date       => Sb4 (Dates.Date_Time'Size / System.Storage_Unit),
         Type_Char_Lob   => Sb4 (OCILobLocator'Size / System.Storage_Unit),
         Type_Bin_Lob    => Sb4 (OCILobLocator'Size / System.Storage_Unit));

   Same_Indicator : constant Sb2 := 16#ADA#;

   function To_Statement_Type is
      new Ada.Unchecked_Conversion (Ub2, Statement_Type);

   function To_OCI is new Ada.Unchecked_Conversion (Data_Type, Ub2);

   procedure Reset_Bind_Iter (Binds : in Bind_Array_Access);
   --  Procedure to reset internal iteration checker in binded variables.
   --  It is necessary to detect Oracle bugs reseting iteration counter.

   function Is_Lob (Kind : Data_Type) return Boolean;
   pragma Inline (Is_Lob);

   function Callback_In_Bind
     (Ictxp  : in     Bind_Access;
      Bindp  : in     OCIBind;
      Iter   : in     Ub4;
      Index  : in     Ub4;
      Bufpp  : access DVoid;
      Alenp  : in     A_Ub4;
      Piecep : in     A_Ub1;
      Indpp  : access A_Sb2) return SWord;
   pragma Convention (C, Callback_In_Bind);

   function Callback_Out_Bind
     (Octxp   : in     Bind_Access;
      Bindp   : in     OCIBind;
      Iter    : in     Ub4;
      Index   : in     Ub4;
      Bufpp   : access DVoid;
      Alenpp  : access A_Ub4;
      Piecep  : in     A_Ub1;
      Indpp   : access A_Sb2;
      Rcodepp : access A_Ub2) return SWord;
   pragma Convention (C, Callback_Out_Bind);

   procedure Callback_Out
     (PC        : in     Context_Access;
      Kind      : in     Data_Type;
      Iteration : in     Positive;
      Position  : in     Positive;
      Loc       : in     Lobs.Locator; -- only for lob types.
      Bufp      :    out DVoid;
      Alenp     : in out A_Ub4;
      Piece     : in out Ub1;
      Indp      : in out A_Sb2;
      Rcodep    : in out A_Ub2;
      Result    :    out SWord);

   function Callback_Define
     (Octxp   : in     Define_Access;
      Defnp   : in     OCIDefine;
      Iter    : in     Ub4;
      Bufpp   : access DVoid;
      Alenpp  : access A_Ub4;
      Piecep  : in     A_Ub1;
      Indpp   : access A_Sb2;
      Rcodepp : access A_Ub2) return  SWord;
   pragma Convention (C, Callback_Define);

   procedure Execute_Internal
     (Stmt     : in out Statement;
      In_Data  : access procedure
                         (Data      :    out Data_Holder;
                          Position  : in     Positive;
                          Iteration : in     Positive);
      Out_Data : access procedure
                         (Data      : in Data_Holder;
                          Position  : in Positive;
                          Iteration : in Positive);
      Count             : in     Natural;
      Raise_Exception   : in out Boolean;
      Commit_On_Success : in     Boolean);

   procedure Execute_Internal
     (Stmt              : in out Statement;
      Data              : in out Container_Type'Class;
      Raise_Exception   : in out Boolean;
      Commit_On_Success : in     Boolean);

   procedure Out_Flush (PC : in Context_Access; Last : in Boolean);
   --  Common code for Execute Bind output and for Fetch

   function Get_Attr_Ub2 is new Get_Attr_G (Ub2);
   function Get_Attr_Ub4 is new Get_Attr_G (Ub4);

   ----------
   -- Bind --
   ----------

   procedure Bind
     (Stmt : in out Statement; Kind : in Data_Type; Position : in Positive) is
   begin
      Check_Error
        (OCIBindByPos
           (Stmtp    => OCIStmt (Stmt.Handle),
            Bindpp   => Stmt.Binds (Position).Bind'Access,
            Errhp    => OCI.Thread.Error,
            Position => Ub4 (Position),
            Valuep   => System.Null_Address,
            Value_Sz => Size_Of (Kind),
            Dty      => To_OCI (Kind),
            Indp     => null,
            Mode     => OCI_DATA_AT_EXEC));

      Check_Error (OCIBindDynamic
        (Bindp  => Stmt.Binds (Position).Bind,
         Errhp  => OCI.Thread.Error,
         Ictxp  => Stmt.Binds (Position)'Address,
         Icbfp  => Callback_In_Bind'Address,
         Octxp  => Stmt.Binds (Position)'Address,
         Ocbfp  => Callback_Out_Bind'Address));

      Stmt.Binds (Position).Kind := Kind;
      Stmt.Binds (Position).Position := Position;
      Stmt.Binds (Position).Context  := Stmt.Context;
   end Bind;

   procedure Bind
     (Stmt : in out Statement; Kind : in Data_Type; Name : in String) is
   begin
      for J in Stmt.Binds'Range loop
         --  Do not need indexed search because bind should be only once for
         --  statement and there is not very much bind variables.

         if Bind_Names."="
              (Stmt.Binds (J).Name, Ada.Characters.Handling.To_Upper (Name))
         then
            Check_Error
              (OCIBindByName
                 (Stmtp       => OCIStmt (Stmt.Handle),
                  Bindpp      => Stmt.Binds (J).Bind'Access,
                  Errhp       => OCI.Thread.Error,
                  Placeholder => C.To_C (Name),
                  Placeh_Len  => Name'Length,
                  Valuep      => System.Null_Address,
                  Value_Sz    => Size_Of (Kind),
                  Dty         => To_OCI (Kind),
                  Indp        => null,
                  Mode        => OCI_DATA_AT_EXEC));

            Check_Error (OCIBindDynamic
              (Bindp => Stmt.Binds (J).Bind,
               Errhp => OCI.Thread.Error,
               Ictxp => Stmt.Binds (J)'Address,
               Icbfp => Callback_In_Bind'Address,
               Octxp => Stmt.Binds (J)'Address,
               Ocbfp => Callback_Out_Bind'Address));

            Stmt.Binds (J).Kind     := Kind;
            Stmt.Binds (J).Position := J;
            Stmt.Binds (J).Context  := Stmt.Context;

            return;
         end if;
      end loop;

      raise Lib_Error with "Bind variable '" & Name & "' not found";
   end Bind;

   ----------------
   -- Bind_Count --
   ----------------

   function Bind_Count (Stmt : in Statement) return Natural is
   begin
      if Stmt.Binds = null then
         return 0;
      else
         return Stmt.Binds'Length;
      end if;
   end Bind_Count;

   ---------------
   -- Bind_Name --
   ---------------

   function Bind_Name
     (Stmt : in Statement; Position : Positive) return String is
   begin
      return Bind_Names.To_String (Stmt.Binds (Position).Name);
   end Bind_Name;

   ---------------------
   -- Callback_Define --
   ---------------------

   function Callback_Define
     (Octxp   : in     Define_Access;
      Defnp   : in     OCIDefine;
      Iter    : in     Ub4;
      Bufpp   : access DVoid;
      Alenpp  : access A_Ub4;
      Piecep  : in     A_Ub1;
      Indpp   : access A_Sb2;
      Rcodepp : access A_Ub2) return SWord
   is
      pragma Unreferenced (Defnp);
      Result : SWord;
   begin
      Callback_Out
        (PC        => Octxp.Context,
         Iteration => Positive (Iter + 1),
         Position  => Octxp.Position,
         Kind      => Octxp.Kind,
         Loc       => Octxp.Loc,
         Bufp      => Bufpp.all,
         Alenp     => Alenpp.all,
         Piece     => Piecep.all,
         Indp      => Indpp.all,
         Rcodep    => Rcodepp.all,
         Result    => Result);

      return Result;
   end Callback_Define;

   ----------------------
   -- Callback_In_Bind --
   ----------------------

   function Callback_In_Bind
     (Ictxp  : in     Bind_Access;
      Bindp  : in     OCIBind;
      Iter   : in     Ub4;
      Index  : in     Ub4;
      Bufpp  : access DVoid;
      Alenp  : in     A_Ub4;
      Piecep : in     A_Ub1;
      Indpp  : access A_Sb2) return SWord
   is
      pragma Unreferenced (Index);
      use type Ub1;

      PC : Context_Access := Ictxp.Context;
      Iteration : constant Positive := Positive (Iter + 1);

      procedure In_String_Piece;

      ---------------------
      -- In_String_Piece --
      ---------------------

      procedure In_String_Piece is
         use type Lib.C.size_t;

         Done : Boolean;
         Last : Lib.C.size_t;
      begin
         if Piecep.all = OCI_FIRST_PIECE then
            PC.Str_Low := 1;

            PC.Holder := Null_Data;

            if PC.In_Data /= null then
               PC.In_Data (PC.Holder, Ictxp.Position, Iteration => Iteration);
            end if;

            if Is_Null (PC.Holder) then
               PC.Buffer.Char (1) := Lib.C.nul;
               Alenp.all := 1;

               Piecep.all := OCI_ONE_PIECE;

               PC.Ind := Null_Indicator;

               return;
            else
               PC.Ind := Not_Null_Indicator;
            end if;
         end if;

         C_Slice
           (PC.Holder,
            Low      => PC.Str_Low,
            Item     => PC.Buffer.Char (1 .. Char_Buffer_Size - 1),
            Last     => Last,
            Done     => Done);

         if Done then
            PC.Buffer.Char (Last + 1) := Lib.C.nul;
            Alenp.all := Ub4 (Last + 1);

            if PC.Str_Low = 1 then
               Piecep.all := OCI_ONE_PIECE;
            else
               Piecep.all := OCI_LAST_PIECE;
            end if;

         else
            Alenp.all := Ub4 (Last);

            if PC.Str_Low = 1 then
               Piecep.all := OCI_FIRST_PIECE;
            else
               Piecep.all := OCI_NEXT_PIECE;
            end if;
         end if;

         PC.Str_Low := PC.Str_Low + Natural (Last);
      end In_String_Piece;

   begin
      if Iteration > PC.Iters then
         raise Program_Error with
           "Iteration" & Natural'Image (Iteration)
           & " more then " & Natural'Image (PC.Iters);
      end if;

      Out_Flush (Ictxp.Context, Last => True);

      if Bindp /= Ictxp.Bind then
         raise Program_Error;
      end if;

      Indpp.all := PC.Ind'Access;
      Bufpp.all := PC.Buffer'Address;

      if Ictxp.Kind = Type_String then
         In_String_Piece;

      else
         PC.Holder := Null_Data;

         if PC.In_Data /= null then
            PC.In_Data (PC.Holder, Ictxp.Position, Iteration);
         end if;

         Alenp.all := Ub4 (Size_Of (Ictxp.Kind));

         Piecep.all := OCI_ONE_PIECE;

         if Is_Null (PC.Holder) then
            PC.Ind := Null_Indicator;
         else
            PC.Ind := Not_Null_Indicator;

            case Ictxp.Kind is
            when Type_String     => raise Program_Error;
            when Type_Integer    => PC.Buffer.Int  := Value (PC.Holder);
            when Type_Long_Float => PC.Buffer.Flt  := Value (PC.Holder);
            when Type_Number     => PC.Buffer.Numb := Value (PC.Holder);
            when Type_Date =>
               PC.Buffer.Dat := Dates.To_Oracle (Value (PC.Holder));
            when Type_Char_Lob | Type_Bin_Lob =>
               --  Save in value for further out value.
               Ictxp.Loc := Value (PC.Holder);
               Bufpp.all := Handle (Ictxp.Loc);
            end case;
         end if;

         if Is_Lob (Ictxp.Kind) then
            if Is_Null (PC.Holder) or else Bufpp.all = Empty_Handle then
               Ictxp.Loc
                 := Lobs.Create
                      (Ictxp.Context.Connect, To_Lob_Type (Ictxp.Kind));
               Bufpp.all := Handle (Ictxp.Loc);
            end if;

            if not Lobs.Is_Init (Ictxp.Loc) then
               PC.Ind := Null_Indicator;
            end if;
         end if;
      end if;

      if Piecep.all = OCI_FIRST_PIECE or else Piecep.all = OCI_ONE_PIECE then
         Ictxp.Iter := Ictxp.Iter + 1;
         Ictxp.Was  := True;

         if Ictxp.Iter /= Iteration then
            raise Program_Error with
              "Unexpected iteration sequence in input"
              & Natural'Image (Ictxp.Iter) & Natural'Image (Iteration);
         end if;

      end if;

      return OCI_CONTINUE;
   exception
      when E : others =>
         Ada.Exceptions.Save_Occurrence (PC.Error, E);
         return OCI_ERROR;
   end Callback_In_Bind;

   ------------------
   -- Callback_Out --
   ------------------

   procedure Callback_Out
     (PC        : in     Context_Access;
      Kind      : in     Data_Type;
      Iteration : in     Positive;
      Position  : in     Positive;
      Loc       : in     Lobs.Locator;
      Bufp      :    out DVoid;
      Alenp     : in out A_Ub4;
      Piece     : in out Ub1;
      Indp      : in out A_Sb2;
      Rcodep    : in out A_Ub2;
      Result    :    out SWord) is
   begin
      Out_Flush
        (PC   => PC,
         Last => PC.Position /= Position or else PC.Iteration /= Iteration);

      if Kind = Type_String then
         PC.Alen := PC.Buffer.Char'Length;
         PC.Buffer.Char (1) := C.nul;
      else
         PC.Alen := Ub4 (Size_Of (Kind));
         Piece   := OCI_ONE_PIECE;
      end if;

      Alenp  := PC.Alen'Access;
      Indp   := PC.Ind'Access;
      Rcodep := PC.Rcode'Access;
      PC.Ind := Same_Indicator;

      if Is_Lob (Kind) then
         Bufp := Handle (Loc);

         if Bufp = Empty_Handle then
            PC.Loc := Lobs.Create (PC.Connect, To_Lob_Type (Kind));
            Bufp := Handle (PC.Loc);
         else
            PC.Loc := Loc;
         end if;
      else
         Bufp := PC.Buffer'Address;
      end if;

      PC.Position  := Position;
      PC.Iteration := Iteration;
      PC.Piece     := Piece;
      PC.Output    := True;
      PC.Kind      := Kind;

      Result := OCI_CONTINUE;
   exception
      when E : others =>
         Ada.Exceptions.Save_Occurrence (PC.Error, E);
         Result := OCI_ERROR;
   end Callback_Out;

   -----------------------
   -- Callback_Out_Bind --
   -----------------------

   function Callback_Out_Bind
     (Octxp   : in     Bind_Access;
      Bindp   : in     OCIBind;
      Iter    : in     Ub4;
      Index   : in     Ub4;
      Bufpp   : access DVoid;
      Alenpp  : access A_Ub4;
      Piecep  : in     A_Ub1;
      Indpp   : access A_Sb2;
      Rcodepp : access A_Ub2) return SWord
   is
      use type Ub1;

      pragma Unreferenced (Bindp, Index);

      Iteration : constant Positive := Positive (Iter + 1);
      Result    : SWord;

   begin
      if Piecep.all = OCI_FIRST_PIECE or else Piecep.all = OCI_ONE_PIECE then
         if Octxp.Was then
            Octxp.Was := False;
         else
            Octxp.Iter := Octxp.Iter + 1;
         end if;

         if Octxp.Iter /= Iteration then
            raise Program_Error with
              "Unexpected iteration sequence in output"
              & Natural'Image (Octxp.Iter) & Natural'Image (Iteration);
         end if;
      end if;

      Callback_Out
        (PC        => Octxp.Context,
         Kind      => Octxp.Kind,
         Iteration => Iteration,
         Position  => Octxp.Position,
         Loc       => Octxp.Loc,
         Bufp      => Bufpp.all,
         Alenp     => Alenpp.all,
         Piece     => Piecep.all,
         Indp      => Indpp.all,
         Rcodep    => Rcodepp.all,
         Result    => Result);

      return Result;
   end Callback_Out_Bind;

   ------------
   -- Cancel --
   ------------

   procedure Cancel (Stmt : in Statement) is
   begin
      Check_Error (OCIStmtFetch
        (OCIStmt (Stmt.Handle), Thread.Error, Nrows => 0));
   end Cancel;

   ------------------
   -- Column_Index --
   ------------------

   function Column_Index
     (Stmt : in Statement; Name : in String) return Positive is
   begin
      if Stmt.Context.Defines = null then
         Stmt.Context.Defines
           := new Define_Array (1 .. Number_Of_Columns (Stmt));
      end if;

      for J in Stmt.Context.Defines'Range loop
         --  Do not need indexed search because define should be only once for
         --  statement and there is not very much define variables.

         if Column_Name (Stmt, J)
            = Ada.Characters.Handling.To_Upper (Name)
         then
            return J;
         end if;
      end loop;

      raise Constraint_Error with "Column '" & Name & "' not found.";
   end Column_Index;

   -----------------
   -- Column_Name --
   -----------------

   function Column_Name
     (Stmt : in Statement; Position : in Positive) return String is
   begin
      if Stmt.Context.Defines (Position).Param
         = OCIParam (System.Null_Address)
      then
         Check_Error (OCIParamGet
           (Hndlp   => Stmt.Handle,
            Htype   => OCI_HTYPE_STMT,
            Errhp   => Thread.Error,
            Parmdpp => Stmt.Context.Defines (Position).Param'Access,
            Pos     => Ub4 (Position)));
      end if;

      return Get_Attr
               (OCIHandle (Stmt.Context.Defines (Position).Param),
                OCI_DTYPE_PARAM,
                OCI_ATTR_NAME);
   end Column_Name;

   ------------
   -- Define --
   ------------

   procedure Define
     (Stmt : in out Statement; Kind : in Data_Type; Position : in Positive) is
   begin
      if Stmt.Context.Defines = null then
         Stmt.Context.Defines
           := new Define_Array (1 .. Number_Of_Columns (Stmt));
      end if;

      Check_Error (OCIDefineByPos
        (Stmtp    => OCIStmt (Stmt.Handle),
         Defnpp   => Stmt.Context.Defines (Position).Define'Access,
         Errhp    => OCI.Thread.Error,
         Position => Ub4 (Position),
         Value    => System.Null_Address,
         Value_Sz => Size_Of (Kind),
         Dty      => To_OCI (Kind),
         Indp     => null,
         Mode     => OCI_DYNAMIC_FETCH));

      Check_Error (OCIDefineDynamic
        (Defnp => Stmt.Context.Defines (Position).Define,
         Errhp => OCI.Thread.Error,
         Octxp => Stmt.Context.Defines (Position)'Address,
         Ocbfp => Callback_Define'Address));

      Stmt.Context.Defines (Position).Kind     := Kind;
      Stmt.Context.Defines (Position).Position := Position;
      Stmt.Context.Defines (Position).Context  := Stmt.Context;
   end Define;

   procedure Define
     (Stmt : in out Statement; Kind : in Data_Type; Name : in String) is
   begin
      Define (Stmt, Kind, Column_Index (Stmt, Name));
   end Define;

   procedure Define
     (Stmt : in out Statement; Loc : in Lobs.Locator; Position : in Positive)
   is
   begin
      Define (Stmt, To_Data_Type (Lobs.Get_Lob_Type (Loc)), Position);
      Stmt.Context.Defines (Position).Loc := Loc;
   end Define;

   procedure Define
     (Stmt : in out Statement; Loc : in Lobs.Locator; Name : in String) is
   begin
      Define (Stmt, Loc, Column_Index (Stmt, Name));
   end Define;

   -------------
   -- Defined --
   -------------

   function Defined (Stmt : in Statement) return Boolean is
   begin
      return Stmt.Context /= null and then Stmt.Context.Defines /= null;
   end Defined;

   --------------
   -- Describe --
   --------------

   procedure Describe (Connect : in Connection; Stmt : in out Statement) is
   begin
      Stmt.Connect := Connect;
      Describe (Stmt);
   end Describe;

   --------------
   -- Describe --
   --------------

   procedure Describe (Stmt : in out Statement) is
   begin
      Check_Error (OCIStmtExecute
        (OCISvcCtx (Handle (Stmt.Connect)),
         OCIStmt (Stmt.Handle),
         Thread.Error,
         0,
         Mode => OCI_DESCRIBE_ONLY));

      Stmt.Described := True;
   end Describe;

   --------------
   -- Describe --
   --------------

   procedure Describe (Stmt : in out Statement; Success : out Boolean) is
   begin
      Success := OCIStmtExecute
                   (OCISvcCtx (Handle (Stmt.Connect)),
                    OCIStmt (Stmt.Handle),
                    Thread.Error,
                    0,
                    Mode => OCI_DESCRIBE_ONLY) = OCI_SUCCESS;

      if Success then
         Stmt.Described := True;
      end if;
   end Describe;

   ---------------
   -- Described --
   ---------------

   function Described (Stmt : in Statement) return Boolean is
   begin
      return Stmt.Described;
   end Described;

   -------------
   -- Destroy --
   -------------

   procedure Destroy (Object : in out Statement) is
      procedure Free is
        new Ada.Unchecked_Deallocation (Bind_Array, Bind_Array_Access);
      procedure Free is
        new Ada.Unchecked_Deallocation (Define_Array, Define_Array_Access);
      procedure Free is
        new Ada.Unchecked_Deallocation (Context_Type, Context_Access);
      Ctx : Context_Access renames Object.Context;
   begin
      if Object.Handle /= Empty_Handle then
         if Ctx /= null then
            if Ctx.Defines /= null then
               for J in Ctx.Defines'Range loop
                  if Ctx.Defines (J).Param /= OCIParam (Empty_Handle) then
                     Check_Error
                       (OCIDescriptorFree
                          (Descp => OCIHandle (Ctx.Defines (J).Param),
                           Dtype => OCI_DTYPE_PARAM),
                        Raise_Exception => False);
                  end if;
               end loop;

               Free (Ctx.Defines);
            end if;

            Free (Object.Context);
         end if;

         OCI.Thick.Free (Object.Handle, OCI_HTYPE_STMT);
         Free (Object.Binds);
      end if;

   end Destroy;

   -------------
   -- Execute --
   -------------

   procedure Execute
     (Connect     : in     Connection;
      Stmt        : in out Statement;
      Auto_Commit : in     Boolean := False) is
   begin
      Stmt.Connect := Connect;
      Execute (Stmt, Auto_Commit => Auto_Commit);
   end Execute;

   procedure Execute
     (Stmt    : in out Statement;
      In_Data : access procedure
                         (Data      :    out Data_Holder;
                          Position  : in     Positive;
                          Iteration : in     Positive);
      Out_Data : access procedure
                         (Data      : in Data_Holder;
                          Position  : in Positive;
                          Iteration : in Positive);
      Count       : in     Natural := 1;
      Auto_Commit : in Boolean := False)
   is
      Raise_Exception : Boolean := True;
   begin
      Execute_Internal
        (Stmt              => Stmt,
         In_Data           => In_Data,
         Out_Data          => Out_Data,
         Count             => Count,
         Raise_Exception   => Raise_Exception,
         Commit_On_Success => Auto_Commit);
   end Execute;

   procedure Execute
     (Stmt     : in out Statement;
      In_Data  : access procedure
                         (Data      :    out Data_Holder;
                          Position  : in     Positive;
                          Iteration : in     Positive);
      Out_Data : access procedure
                         (Data      : in Data_Holder;
                          Position  : in Positive;
                          Iteration : in Positive);
      Success     :    out Boolean;
      Count       : in     Natural := 1;
      Auto_Commit : in Boolean := False)
   is
      Raise_Exception : Boolean := False;
   begin
      Execute_Internal
        (Stmt              => Stmt,
         In_Data           => In_Data,
         Out_Data          => Out_Data,
         Count             => Count,
         Raise_Exception   => Raise_Exception,
         Commit_On_Success => Auto_Commit);
      Success := not Raise_Exception;
   end Execute;

   procedure Execute
     (Connect     : in     Connection;
      Stmt        : in out Statement;
      Data        : in out Container_Type'Class;
      Auto_Commit : in Boolean := False) is
   begin
      Stmt.Connect := Connect;
      Execute (Stmt, Data, Auto_Commit => Auto_Commit);
   end Execute;

   procedure Execute
     (Stmt : in out Statement; Auto_Commit : in Boolean := False)
   is
      Raise_Exception : Boolean := True;
      Data : Container_Type;
   begin
      Execute_Internal
        (Stmt, Data, Raise_Exception, Commit_On_Success => Auto_Commit);
   end Execute;

   procedure Execute
     (Stmt        : in out Statement;
      Data        : in out Container_Type'Class;
      Auto_Commit : in     Boolean := False)
   is
      Raise_Exception : Boolean := True;
   begin
      Execute_Internal
        (Stmt, Data, Raise_Exception, Commit_On_Success => Auto_Commit);
   end Execute;

   procedure Execute
     (Stmt        : in out Statement;
      Data        : in out Container_Type'Class;
      Success     :    out Boolean;
      Auto_Commit : in     Boolean := False)
   is
      Raise_Exception : Boolean := False;
   begin
      Execute_Internal
        (Stmt, Data, Raise_Exception, Commit_On_Success => Auto_Commit);
      Success := not Raise_Exception;
   end Execute;

   procedure Execute
     (Stmt        : in out Statement;
      Success     :    out Boolean;
      Auto_Commit : in     Boolean := False)
   is
      Raise_Exception : Boolean := False;
      Data : Container_Type;
   begin
      Execute_Internal
        (Stmt, Data,
         Raise_Exception   => Raise_Exception,
         Commit_On_Success => Auto_Commit);
      Success := not Raise_Exception;
   end Execute;

   -------------
   -- Execute --
   -------------

   function Execute
     (Connect     : in Connection;
      Code        : in String;
      Auto_Commit : in Boolean := False) return Statement
   is
      Result : Statement := Prepare (Connect, Code);
   begin
      Execute (Result, Auto_Commit => Auto_Commit);
      return Result;
   end Execute;

   procedure Execute
     (Connect     : in Connection;
      Code        : in String;
      Auto_Commit : in Boolean := False)
   is
      Dummy : Statement := Prepare (Connect, Code);
   begin
      Execute (Dummy, Auto_Commit => Auto_Commit);
   end Execute;

   ----------------------
   -- Execute_Internal --
   ----------------------

   procedure Execute_Internal
     (Stmt              : in out Statement;
      Data              : in out Container_Type'Class;
      Raise_Exception   : in out Boolean;
      Commit_On_Success : in     Boolean)
   is
      procedure In_Data
        (Holder    :    out Data_Holder;
         Position  : in     Positive;
         Iteration : in     Positive);

      procedure Out_Data
        (Holder    : in Data_Holder;
         Position  : in Positive;
         Iteration : in Positive);

      -------------
      -- In_Data --
      -------------

      procedure In_Data
        (Holder    :    out Data_Holder;
         Position  : in     Positive;
         Iteration : in     Positive) is
      begin
         Get
           (Container => Data,
            Item      => Holder,
            Position  => Position,
            Iteration => Iteration);
      end In_Data;

      --------------
      -- Out_Data --
      --------------

      procedure Out_Data
        (Holder    : in Data_Holder;
         Position  : in Positive;
         Iteration : in Positive) is
      begin
         Set
           (Container => Data,
            Item      => Holder,
            Position  => Position,
            Iteration => Iteration);
      end Out_Data;

   begin
      Execute_Internal
        (Stmt              => Stmt,
         In_Data           => In_Data'Access,
         Out_Data          => Out_Data'Access,
         Count             => Length (Data),
         Raise_Exception   => Raise_Exception,
         Commit_On_Success => Commit_On_Success);
   end Execute_Internal;

   procedure Execute_Internal
     (Stmt    : in out Statement;
      In_Data : access procedure
                         (Data      :    out Data_Holder;
                          Position  : in     Positive;
                          Iteration : in     Positive);
      Out_Data : access procedure
                         (Data      : in Data_Holder;
                          Position  : in Positive;
                          Iteration : in Positive);
      Count             : in     Natural;
      Raise_Exception   : in out Boolean;
      Commit_On_Success : in     Boolean)
   is
      use Ada.Exceptions;

      Select_Stmt : constant Boolean := Type_Of_Statement (Stmt) = Stmt_Select;

      Commit_Mode : constant array (Boolean) of Ub4
        := (False => OCI_DEFAULT, True => OCI_COMMIT_ON_SUCCESS);

      Rc : SWord;

      Iters : constant Ub4 :=
        Ub4'Max (Boolean'Pos (not Select_Stmt),
                 Boolean'Pos (not Select_Stmt) * Ub4 (Count));

   begin
      if Stmt.Context /= null then
         Stmt.Context.In_Data  := In_Data;
         Stmt.Context.Out_Data := Out_Data;
         Stmt.Context.Connect  := Stmt.Connect;
         Stmt.Context.Iters    := Natural (Iters);
         Save_Occurrence (Stmt.Context.Error, Null_Occurrence);
         Reset_Bind_Iter (Stmt.Binds);
      end if;

      Rc := OCIStmtExecute
        (OCISvcCtx (Handle (Stmt.Connect)),
         OCIStmt (Stmt.Handle),
         Thread.Error,
         Iters => Iters,
         Mode  => Commit_Mode (Commit_On_Success));

      if Stmt.Context /= null then
         if Exception_Identity (Stmt.Context.Error) /= Null_Id then
            Reraise_Occurrence (Stmt.Context.Error);
         end if;

         if not Select_Stmt then
            Out_Flush (Stmt.Context, Last => True);
         end if;
      end if;

      Stmt.Executing := Rc = OCI_STILL_EXECUTING;

      if Rc = OCI_SUCCESS then
         Stmt.Executed   := True;
         Stmt.Described  := True;
         Raise_Exception := False;

      elsif Stmt.Executing then
         Raise_Exception := False;

      elsif Raise_Exception then
         Check_Error (Rc);

      else
         Raise_Exception := True;
      end if;

   end Execute_Internal;

   --------------
   -- Executed --
   --------------

   function Executed (Stmt : in Statement) return Boolean is
   begin
      return Stmt.Executed;
   end Executed;

   -----------
   -- Fetch --
   -----------

   function Fetch
     (Stmt    : in     Statement;
      Process : access procedure
                         (Data      : in Data_Holder;
                          Position  : in Positive;
                          Iteration : in Positive);
      Count   : in     Positive := 1) return Boolean
   is
      use Ada.Exceptions;
      RC : SWord;
   begin
      Stmt.Context.Out_Data := Process;
      Stmt.Context.Connect  := Stmt.Connect;

      Save_Occurrence (Stmt.Context.Error, Null_Occurrence);

      RC := OCIStmtFetch
              (OCIStmt (Stmt.Handle), Thread.Error, Nrows => Ub4 (Count));

      if Exception_Identity (Stmt.Context.Error) /= Null_Id then
         Reraise_Occurrence (Stmt.Context.Error);
      end if;

      Out_Flush (Stmt.Context, Last => True);

      case RC is
         when OCI_NO_DATA   => return False;
         when OCI_SUCCESS   => return True;
         when others => Check_Error (RC);
      end case;

      --  We should not be there.

      raise Program_Error;
   end Fetch;

   procedure Fetch
     (Stmt  : in     Statement;
      Data  :    out Container_Type'Class;
      Count : in     Positive)
   is

      procedure Process
        (Holder : in Data_Holder; Position, Index : in Positive);

      -------------
      -- Process --
      -------------

      procedure Process
        (Holder : in Data_Holder; Position, Index : in Positive) is
      begin
         Set (Data, Holder, Position, Index);
      end Process;

   begin
      Data.Clear;

      if Fetch (Stmt, Process'Access, Count) then
         null;
      end if;
   end Fetch;

   --------------------
   -- Get_Connection --
   --------------------

   function Get_Connection (Stmt : in Statement) return Connection is
   begin
      return Stmt.Connect;
   end Get_Connection;

   ------------------
   -- Is_Executing --
   ------------------

   function Is_Executing (Stmt : in Statement) return Boolean is
   begin
      return Stmt.Executing;
   end Is_Executing;

   function Is_Lob (Kind : Data_Type) return Boolean is
   begin
      return Kind = Type_Char_Lob or else Kind = Type_Bin_Lob;
   end Is_Lob;

   -----------------------
   -- Number_Of_Columns --
   -----------------------

   function Number_Of_Columns (Stmt : in Statement) return Natural is
   begin
      return Natural
               (Get_Attr_Ub4
                  (Stmt.Handle, OCI_HTYPE_STMT, OCI_ATTR_PARAM_COUNT));
   end Number_Of_Columns;

   ---------------
   -- Out_Flush --
   ---------------

   procedure Out_Flush (PC : in Context_Access; Last : in Boolean) is
      use type Sb2;
      use type C.char;

      function String_Buffer return String;
      pragma Inline (String_Buffer);

      function String_Buffer return String is
         use C;
      begin
         return To_Ada
                  (PC.Buffer.Char (1 .. size_t (PC.Alen)),
                   Trim_Nul => PC.Buffer.Char (size_t (PC.Alen)) = nul);
      end String_Buffer;

   begin
      if PC.Output then
         PC.Output := False;

         if PC.Ind = Same_Indicator
            and then not (PC.Kind = Type_String
                          and then PC.Buffer.Char (1) /= C.nul)
         then
            --  Not touched indicator mean absent output data.
            --  Not documented oracle feature or bug.

            return;
         end if;
      else
         return;
      end if;

      if PC.Ind = Null_Indicator then
         PC.Holder := Null_Data;

         if PC.Out_Data /= null then
            PC.Out_Data (PC.Holder, PC.Position, PC.Iteration);
         end if;

      else
         case PC.Kind is
         when Type_String =>
            case PC.Piece is
            when OCI_FIRST_PIECE | OCI_ONE_PIECE =>
               --  In bind "one" does not mean absent "next"
               --  In define "First" does not mean present "next".
               --  It is because of week design of parameter Piece in output
               --  user callbacks.

               if PC.Alen = 0 then
                  --  Strange OCI behavior,
                  --  empty string with not null indicator.

                  PC.Holder := Null_Data;
               else
                  PC.Holder := To_Data (String_Buffer);
               end if;

            when OCI_NEXT_PIECE => Append (PC.Holder, String_Buffer);
            when others => raise Constraint_Error with PC.Piece'Img;
            end case;

         when Type_Integer    => PC.Holder := To_Data (PC.Buffer.Int);
         when Type_Long_Float => PC.Holder := To_Data (PC.Buffer.Flt);
         when Type_Number     => PC.Holder := To_Data (PC.Buffer.Numb);
         when Type_Date => PC.Holder := To_Data (Dates.To_Ada (PC.Buffer.Dat));
         when Type_Char_Lob | Type_Bin_Lob => PC.Holder := To_Data (PC.Loc);
         end case;

         if Last and PC.Out_Data /= null then
            PC.Out_Data (PC.Holder, PC.Position, PC.Iteration);
         end if;

      end if;
   end Out_Flush;

   ------------------------
   -- Parse_Error_Offset --
   ------------------------

   function Parse_Error_Offset (Stmt : in Statement) return Natural is
   begin
      return Natural
               (Get_Attr_Ub2
                  (Stmt.Handle, OCI_HTYPE_STMT, OCI_ATTR_PARSE_ERROR_OFFSET));
   end Parse_Error_Offset;

   -------------
   -- Prepare --
   -------------

   function Prepare
     (Connect : in Connection; Code : in String) return Statement
   is
      Stmt : Statement;

      procedure Set_Bind_Names;

      --------------------
      -- Set_Bind_Names --
      --------------------

      procedure Set_Bind_Names is
         Names  : aliased Vector_Of_OraText (1 .. 256);
         Lens   : aliased Vector_Of_Ub1     (Names'Range);
         INames : aliased Vector_Of_OraText (Names'Range);
         ILens  : aliased Vector_Of_Ub1     (Names'Range);
         Dups   : aliased Vector_Of_Ub1     (Names'Range);
         Binds  : aliased Vector_Of_OCIBind (Names'Range);
         Found  : aliased Sb4;

         RC : constant SWord
           := OCIStmtGetBindInfo
                (Stmtp    => OCIStmt (Stmt.Handle),
                 Errhp    => Thread.Error,
                 Size     => Names'Length,
                 Startloc => 1,
                 Found    => Found'Unchecked_Access,
                 Bvnp     => Names  (Names'First)'Unchecked_Access,
                 Bvnl     => Lens   (Lens'First)'Unchecked_Access,
                 Invp     => INames (INames'First)'Unchecked_Access,
                 Inpl     => ILens  (ILens'First)'Unchecked_Access,
                 Dupl     => Dups   (Dups'First)'Unchecked_Access,
                 Hndl     => Binds  (Binds'First)'Unchecked_Access);
         use Interfaces.C.Strings;
         use type C.size_t;
         use type Ub1;
      begin
         if RC = OCI_NO_DATA then
            return;
         end if;

         Check_Error (RC);

         declare
            Binds : Bind_Array (1 .. Integer (Found));
            Count : Natural := 0;
         begin
            for J in 1 .. Integer (Found) loop
               if Dups (J) = 0 then
                  Count := Count + 1;

                  Bind_Names.Set_Bounded_String
                    (Binds (Count).Name,
                     C.To_Ada (Value (Names (J), C.size_t (Lens (J))), False));
               end if;
            end loop;

            Stmt.Binds := new Bind_Array'(Binds (1 .. Count));
         end;
      end Set_Bind_Names;

   begin
      Stmt.Handle := Alloc_Handle (Thread.Environment, OCI_HTYPE_STMT);

      Check_Error
        (OCIStmtPrepare
           (OCIStmt (Stmt.Handle),
            Thread.Error,
            C.To_C (Ada.Strings.Fixed.Translate (Code, Dos_Fix_CR)),
            Code'Length));

      Set_Bind_Names;

      Stmt.Connect := Connect;

      if Type_Of_Statement (Stmt) = Stmt_Select or else Stmt.Binds /= null then
         Stmt.Context := new Context_Type;
      end if;

      return Stmt;
   end Prepare;

   function Prepare (Code : in String) return Statement is
   begin
      return Prepare (Dummy_Connect, Code);
   end Prepare;

   ---------------------
   -- Reset_Bind_Iter --
   ---------------------

   procedure Reset_Bind_Iter (Binds : in Bind_Array_Access) is
   begin
      if Binds /= null then
         for J in Binds'Range loop
            Binds (J).Iter := 0;
            Binds (J).Was  := False;
         end loop;
      end if;
   end Reset_Bind_Iter;

   --------------------
   -- Rows_Processed --
   --------------------

   function Rows_Processed (Stmt : in Statement) return Natural is
   begin
      return Natural (Get_Attr_Ub4 (Stmt.Handle,
                                    OCI_HTYPE_STMT,
                                    OCI_ATTR_ROW_COUNT));
   end Rows_Processed;

   ------------------
   -- Set_Blocking --
   ------------------

   procedure Set_Blocking (Item : in out Statement; Mode : in Boolean) is
   begin
      Connections.Set_Blocking (Item.Connect, Mode);
   end Set_Blocking;

   --------------------
   -- Set_Connection --
   --------------------

   procedure Set_Connection
     (Stmt    : in out Statement;
      Connect : in     Connection) is
   begin
      Stmt.Connect := Connect;
   end Set_Connection;

   -----------------------
   -- Type_Of_Statement --
   -----------------------

   function Type_Of_Statement (Stmt : in Statement) return Statement_Type is
   begin
      return To_Statement_Type
               (Get_Attr_Ub2
                  (Stmt.Handle, OCI_HTYPE_STMT, OCI_ATTR_STMT_TYPE));
   end Type_Of_Statement;

end OCI.Thick.DB;
