------------------------------------------------------------------------------
--  Thin Ada95 binding to OCI (Oracle Call Interface)                    --
--  Copyright (C) 2000-2006 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-date.adb,v 1.16 2008/02/11 08:52:33 vagul Exp $

with OCI.Lib;
with OCI.Thread;
with Interfaces.C;

package body OCI.Thick.Date is

   ---------
   -- "<" --
   ---------

   function "<" (Left, Right : OCIDate) return Boolean is
   begin
      return Compare (Left, Right) = -1;
   end "<";

   ----------
   -- "<=" --
   ----------

   function "<=" (Left, Right : OCIDate) return Boolean is
   begin
      return Compare (Left, Right) /= 1;
   end "<=";

   ---------
   -- ">" --
   ---------

   function ">" (Left, Right : OCIDate) return Boolean is
   begin
      return Compare (Left, Right) = 1;
   end ">";

   ----------
   -- "=>" --
   ----------

   function ">=" (Left, Right : OCIDate) return Boolean is
   begin
      return Compare (Left, Right) /= -1;
   end ">=";

   -------------
   -- Compare --
   -------------

   function Compare (Left, Right : OCIDate) return Compare_Result is
      Result : aliased SWord;
   begin
      Check_Error (OCIDateCompare
        (Err => Thread.Error,
         Date1 => Left,
         Date2 => Right,
         Result => Result'Access));
      return Compare_Result (Result);
   end Compare;

   -----------
   -- Equal --
   -----------

   function Equal (Left, Right : OCIDate) return Boolean is
   begin
      return Compare (Left, Right) = 0;
   end Equal;

   -------------
   -- SysDate --
   -------------

   function SysDate return OCIDate is
      Result : aliased OCIDate;
   begin
      Check_Error
        (OCIDateSysDate (Err => Thread.Error, Date => Result'Access));
      return Result;
   end SysDate;

   ------------
   -- To_Ada --
   ------------

   function To_Ada (DT : in OCIDate) return Ada.Calendar.Time is
   begin
      return Ada.Calendar.Time_Of
         (Year    => Integer (DT.OCIDateYYYY),
          Month   => Integer (DT.OCIDateMM),
          Day     => Integer (DT.OCIDateDD),
          Seconds => Duration
                       (Natural (DT.OCIDateTime.OCITimeSS)
                        + 60 * (Natural (DT.OCIDateTime.OCITimeMI)
                                + 60 * Natural (DT.OCIDateTime.OCITimeHH))));
   end To_Ada;

   -------------
   -- To_Date --
   -------------

   function To_Date (Item : String; Format : String) return OCIDate is
      Result : aliased OCIDate;
   begin
      Check_Error (OCIDateFromText
        (Err          => Thread.Error,
         Date_Str     => C.To_C (Item),
         d_Str_Length => Item'Length,
         Fmt          => C.To_C (Format),
         Fmt_Length   => Format'Length,
         Lang_Name    => CStr.Null_Ptr,
         Lang_Length  => 0,
         Date         => Result'Access));
      return Result;
   end To_Date;

   ------------
   -- To_OCI --
   ------------

   function To_OCI (DT : in Ada.Calendar.Time) return OCIDate is
      use Ada.Calendar;

      Year    : Year_Number;
      Month   : Month_Number;
      Day     : Day_Number;
      Seconds : Day_Duration;
      Secs    : Natural;
   begin
      Split (DT, Year, Month, Day, Seconds);

      if Seconds = 0.0 then
         Secs := 0;
      else
         Secs := Natural (Seconds - 0.5);
      end if;

      return (OCIDateYYYY => Sb2 (Year),
              OCIDateMM   => Ub1 (Month),
              OCIDateDD   => Ub1 (Day),
              OCIDateTime => (OCITimeHH => Ub1 (Secs / 3600),
                              OCITimeMI => Ub1 (Secs / 60 rem 60),
                              OCITimeSS => Ub1 (Secs rem 60)));
   end To_OCI;

   ---------------
   -- To_String --
   ---------------

   function To_String
     (From   : in OCIDate;
      Format : in String := "YYYY-MM-DD HH24:MI:SS") return String
   is
      use Interfaces.C;
      use type Ub4;
      Buff : aliased Text := (0 .. Format'Length + 64 => C.nul);
      Len  : aliased Ub4 := Buff'Length - 1;
   begin
      Check_Error (OCIDateToText
        (Err         => Thread.Error,
         Date        => From,
         Fmt         => C.To_C (Format),
         Fmt_Length  => Format'Length,
         Lang_Name   => CStr.Null_Ptr,
         Lang_Length => 0,
         Buf_Size    => Len'Access,
         Buf         => CStr.To_Chars_Ptr (Buff'Unchecked_Access)));
      return C.To_Ada (Buff, False) (1 .. Integer (Len));
   end To_String;

end OCI.Thick.Date;
