/* mpn_trialdiv -- find small factors of an mpn number using trial division.

   Contributed to the GNU project by Torbjorn Granlund.

   THE FUNCTION IN THIS FILE IS INTERNAL WITH A MUTABLE INTERFACE.  IT IS ONLY
   SAFE TO REACH IT THROUGH DOCUMENTED INTERFACES.  IN FACT, IT IS ALMOST
   GUARANTEED THAT IT WILL CHANGE OR DISAPPEAR IN A FUTURE GNU MP RELEASE.

Copyright 2009 Free Software Foundation, Inc.

This file is part of the GNU MP Library.

The GNU MP Library is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 3 of the License, or (at your
option) any later version.

The GNU MP Library is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
License for more details.

You should have received a copy of the GNU Lesser General Public License
along with the GNU MP Library.	If not, see http://www.gnu.org/licenses/.  */

/*
   Fast, division-free trial division for GMP.

   This function will find the first (smallest) factor represented in
   trialdivtab.h.  It does not stop the factoring effort just because it has
   reached some sensible limit, such as the square root of the input number.

   The caller can limit the factoring effort by passing NPRIMES.  The function
   well then divide to *at least* that limit.  A position which only
   mpn_trialdiv can make sense of is returned in the WHERE parameter.  It can
   be used for restarting the factoring effort; the first call should pass 0
   here.
*/

#include "gmp.h"
#include "gmp-impl.h"

struct gmp_primes_dtab {
  mp_limb_t binv;
  mp_limb_t lim;
};

struct gmp_primes_ptab {
  mp_limb_t ppp;	/* primes, multiplied together */
  mp_limb_t cps[7];	/* ppp values pre-computed for mpn_mod_1s_4p */
  unsigned int idx:24;	/* index of  first primes in dtab */
  unsigned int np :8;	/* number of primes related to this entry */
};

#define P(p,inv,lim) {inv,lim}

#include "trialdivtab.h"

#define PTAB_LINES (sizeof (gmp_primes_ptab) / sizeof (gmp_primes_ptab[0]))

/* Attempt to find a factor of T using trial division.
   Input: A non-negative number T.
   Output: non-zero if we found a factor, zero otherwise.  To get the actual
   prime factor, compute the mod B inverse of the return value.  */
/* FIXME: We could optimize out one of the outer loop conditions if we
   had a final ptab entry with a huge nd field.  */
mp_limb_t
mpn_trialdiv (mp_srcptr tp, mp_size_t tn, mp_size_t nprimes, int *where)
{
  mp_limb_t ppp;
  mp_limb_t *cps;
  struct gmp_primes_dtab *dp;
  long i, j, idx, np;
  mp_limb_t r, q;

  ASSERT (tn >= 1);

  for (i = *where; i < PTAB_LINES; i++)
    {
      ppp = gmp_primes_ptab[i].ppp;
      cps = gmp_primes_ptab[i].cps;

#if __GNU_MP_VERSION == 4 && __GNU_MP_VERSION_MINOR < 4
      if (tn < 4)
	r = mpn_mod_1 (tp, tn, ppp); /* FIXME */
      else
#endif
	r = mpn_mod_1s_4p (tp, tn, ppp << cps[1], cps);

      idx = gmp_primes_ptab[i].idx;
      np = gmp_primes_ptab[i].np;

      /* Check divisibility by individual primes.  */
      dp = &gmp_primes_dtab[idx] + np;
      for (j = -np; j < 0; j++)
	{
	  q = r * dp[j].binv;
	  if (q <= dp[j].lim)
	    {
	      *where = i;
	      return dp[j].binv;
	    }
	}

      nprimes -= np;
      if (nprimes <= 0)
	return 0;
    }
  return 0;
}
