/*
    Copyright 2020 Daniel Schultz

    This file is part of FLINT.

    FLINT is free software: you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 2.1 of the License, or
    (at your option) any later version.  See <https://www.gnu.org/licenses/>.
*/

#include "nmod_mpoly_factor.h"
#include "profiler.h"


slong check_omega(slong om, const nmod_mpoly_t p, const char ** vars, const nmod_mpoly_ctx_t ctx)
{
    slong i;
    nmod_mpoly_factor_t g;
    nmod_mpoly_t q;
    fmpz_t omega;
    timeit_t timer;

    fmpz_init(omega);
    nmod_mpoly_factor_init(g, ctx);
    nmod_mpoly_init(q, ctx);

    timeit_start(timer);
    nmod_mpoly_factor(g, p, ctx);
    timeit_stop(timer);

    fmpz_zero(omega);
    for (i = 0; i < g->num; i++)
        fmpz_add(omega, omega, g->exp + i);

    if (fmpz_cmp_si(omega, om) < 0)
    {
        flint_printf("factorization has wrong number of factors\n");
        flint_abort();
    }

    if (!nmod_mpoly_factor_expand(q, g, ctx) ||
        !nmod_mpoly_equal(q, p, ctx))
    {
        flint_printf("factorization does not match\n");
        flint_abort();
    }

    nmod_mpoly_clear(q, ctx);
    nmod_mpoly_factor_clear(g, ctx);
    fmpz_clear(omega);

    return timer->wall;
}


int main(int argc, char *argv[])
{
    slong i, k, time, total_time;

    flint_printf("\n------ bivariate powers ------\n");
    total_time = 0;
    {
        nmod_mpoly_ctx_t ctx;
        nmod_mpoly_t a, b, c;
        const char * vars[] = {"x", "y", "z", "t"};

        nmod_mpoly_ctx_init(ctx, 4, ORD_LEX, UWORD(4611686018427388073));
        nmod_mpoly_init(a, ctx);
        nmod_mpoly_init(b, ctx);
        nmod_mpoly_init(c, ctx);

        for (i = 1; i <= 100; i++)
        {
            nmod_mpoly_set_str_pretty(a, "x+1", vars, ctx);
            nmod_mpoly_set_str_pretty(b, "y+2", vars, ctx);
            nmod_mpoly_set_str_pretty(c, "1+x+y+z*0+t*0", vars, ctx);
            nmod_mpoly_pow_ui(c, c, i, ctx);
            nmod_mpoly_add(a, a, c, ctx);
            nmod_mpoly_add(b, b, c, ctx);
            nmod_mpoly_mul(a, a, b, ctx);

            time = check_omega(2, a, vars, ctx);
            flint_printf("power %wd: %wd\n", i, time);
            total_time += time;
        }

        nmod_mpoly_clear(a, ctx);
        nmod_mpoly_clear(b, ctx);
        nmod_mpoly_clear(c, ctx);
        nmod_mpoly_ctx_clear(ctx);
    }
    flint_printf("total_time: %wd\n", total_time);


    flint_printf("\n------ characteristic 43051 ------\n");
    total_time = 0;
    {
        nmod_mpoly_ctx_t ctx;
        nmod_mpoly_t a;
        const char * vars[] = {"x","y","z","u","v","w","s","t"};

        nmod_mpoly_ctx_init(ctx, 8, ORD_LEX, 43051);
        nmod_mpoly_init(a, ctx);

        flint_printf(" #1: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a,
            "(13269*y^128+9940*x^2*y^126+38886*y^126+22127*x^4*y^124+7922*x^2*y^124+25434*y^124+26172*x^6*y^122+20019*x^4*y^122+710*x^2*y^122+39701*y^122+30041*x^8*y^120+14420*x^6*y^120+18592*x^4*y^120"
            "+28677*x^2*y^120+1803*y^120+38458*x^10*y^118+20323*x^8*y^118+8330*x^6*y^118+30874*x^4*y^118+29200*x^2*y^118+29220*y^118+33741*x^12*y^116+35378*x^10*y^116+13296*x^8*y^116+33509*x^6*y^116+15132*x^4*y^116"
            "+16915*x^2*y^116+36779*y^116+12066*x^14*y^114+35681*x^12*y^114+38852*x^10*y^114+32818*x^8*y^114+34141*x^6*y^114+29281*x^4*y^114+17106*x^2*y^114+3974*y^114+7189*x^16*y^112+38835*x^14*y^112+2669*x^12*y^112"
            "+13844*x^10*y^112+27827*x^8*y^112+4543*x^6*y^112+22980*x^4*y^112+14802*x^2*y^112+7911*y^112+8742*x^18*y^110+17449*x^16*y^110+17848*x^14*y^110+11408*x^12*y^110+15204*x^10*y^110+38475*x^8*y^110+22477*x^6*y^110"
            "+10864*x^4*y^110+7567*x^2*y^110+2699*y^110+15389*x^20*y^108+28996*x^18*y^108+33417*x^16*y^108+9037*x^14*y^108+37403*x^12*y^108+14179*x^10*y^108+22500*x^8*y^108+30910*x^6*y^108+6007*x^4*y^108+3533*x^2*y^108"
            "+13784*y^108+3379*x^22*y^106+9097*x^20*y^106+8875*x^18*y^106+23815*x^16*y^106+26019*x^14*y^106+12197*x^12*y^106+33524*x^10*y^106+26645*x^8*y^106+12525*x^6*y^106+4936*x^4*y^106+16621*x^2*y^106+6518*y^106"
            "+34806*x^24*y^104+27532*x^22*y^104+15481*x^20*y^104+42694*x^18*y^104+9716*x^16*y^104+1534*x^14*y^104+3364*x^12*y^104+26521*x^10*y^104+8018*x^8*y^104+21019*x^6*y^104+23939*x^4*y^104+37877*x^2*y^104+1982*y^104"
            "+39670*x^26*y^102+7080*x^24*y^102+38717*x^22*y^102+22559*x^20*y^102+17582*x^18*y^102+10978*x^16*y^102+17609*x^14*y^102+23514*x^12*y^102+7142*x^10*y^102+18372*x^8*y^102+24156*x^6*y^102+31836*x^4*y^102"
            "+7956*x^2*y^102+5676*y^102+20087*x^28*y^100+42143*x^26*y^100+35489*x^24*y^100+3622*x^22*y^100+12450*x^20*y^100+11884*x^18*y^100+3686*x^16*y^100+8795*x^14*y^100+16044*x^12*y^100+18886*x^10*y^100"
            "+22394*x^8*y^100+29781*x^6*y^100+12108*x^4*y^100+16344*x^2*y^100+40582*y^100+41751*x^30*y^98+13657*x^28*y^98+38882*x^26*y^98+11472*x^24*y^98+15202*x^22*y^98+22179*x^20*y^98+26651*x^18*y^98+18181*x^16*y^98"
            "+37427*x^14*y^98+39266*x^12*y^98+41153*x^10*y^98+589*x^8*y^98+41441*x^6*y^98+11572*x^4*y^98+36978*x^2*y^98+34026*y^98+28554*x^32*y^96+3506*x^30*y^96+14744*x^28*y^96+12003*x^26*y^96+5792*x^24*y^96"
            "+5472*x^22*y^96+9862*x^20*y^96+16472*x^18*y^96+23916*x^16*y^96+37473*x^14*y^96+1570*x^12*y^96+31816*x^10*y^96+1632*x^8*y^96+18727*x^6*y^96+42747*x^4*y^96+39545*x^2*y^96+20847*y^96+23258*x^34*y^94"
            "+14891*x^32*y^94+22624*x^30*y^94+19054*x^28*y^94+24085*x^26*y^94+1173*x^24*y^94+1725*x^22*y^94+38084*x^20*y^94+9415*x^18*y^94+27090*x^16*y^94+20682*x^14*y^94+35221*x^12*y^94+27233*x^10*y^94+21631*x^8*y^94"
            "+36526*x^6*y^94+30049*x^4*y^94+37434*x^2*y^94+9317*y^94+4301*x^36*y^92+24353*x^34*y^92+38244*x^32*y^92+40102*x^30*y^92+10779*x^28*y^92+2406*x^26*y^92+29521*x^24*y^92+35520*x^22*y^92+24174*x^20*y^92"
            "+31341*x^18*y^92+31252*x^16*y^92+36897*x^14*y^92+38817*x^12*y^92+11280*x^10*y^92+11765*x^8*y^92+11426*x^6*y^92+17193*x^4*y^92+5249*x^2*y^92+3786*y^92+2062*x^38*y^90+16466*x^36*y^90+33854*x^34*y^90"
            "+18164*x^32*y^90+41008*x^30*y^90+8560*x^28*y^90+6013*x^26*y^90+28230*x^24*y^90+30742*x^22*y^90+766*x^20*y^90+4589*x^18*y^90+4721*x^16*y^90+23669*x^14*y^90+42632*x^12*y^90+38975*x^10*y^90+4290*x^8*y^90"
            "+39169*x^6*y^90+42415*x^4*y^90+39842*x^2*y^90+6303*y^90+28643*x^40*y^88+26752*x^38*y^88+27117*x^36*y^88+19904*x^34*y^88+31532*x^32*y^88+36366*x^30*y^88+17029*x^28*y^88+12013*x^26*y^88+392*x^24*y^88"
            "+28586*x^22*y^88+10892*x^20*y^88+27171*x^18*y^88+751*x^16*y^88+23150*x^14*y^88+8943*x^12*y^88+16890*x^10*y^88+13828*x^8*y^88+13590*x^6*y^88+30307*x^4*y^88+23580*x^2*y^88+35840*y^88+13720*x^42*y^86"
            "+4201*x^40*y^86+29883*x^38*y^86+39011*x^36*y^86+16134*x^34*y^86+2827*x^32*y^86+34105*x^30*y^86+18083*x^28*y^86+1128*x^26*y^86+42071*x^24*y^86+8728*x^22*y^86+32551*x^20*y^86+6477*x^18*y^86+35594*x^16*y^86"
            "+3430*x^14*y^86+34152*x^12*y^86+34763*x^10*y^86+23526*x^8*y^86+36433*x^6*y^86+40790*x^4*y^86+40902*x^2*y^86+1260*y^86+28351*x^44*y^84+35086*x^42*y^84+21750*x^40*y^84+29763*x^38*y^84+23105*x^36*y^84"
            "+5133*x^34*y^84+25378*x^32*y^84+7204*x^30*y^84+599*x^28*y^84+10893*x^26*y^84+20630*x^24*y^84+38664*x^22*y^84+3311*x^20*y^84+39756*x^18*y^84+29977*x^16*y^84+12988*x^14*y^84+32133*x^12*y^84+7480*x^10*y^84"
            "+38030*x^8*y^84+24896*x^6*y^84+23322*x^4*y^84+527*x^2*y^84+7194*y^84+14689*x^46*y^82+27358*x^44*y^82+39746*x^42*y^82+41624*x^40*y^82+9843*x^38*y^82+37032*x^36*y^82+34955*x^34*y^82+41680*x^32*y^82"
            "+38649*x^30*y^82+873*x^28*y^82+19101*x^26*y^82+5510*x^24*y^82+42854*x^22*y^82+29836*x^20*y^82+5166*x^18*y^82+35976*x^16*y^82+41226*x^14*y^82+15451*x^12*y^82+25699*x^10*y^82+36727*x^8*y^82+41747*x^6*y^82"
            "+15122*x^4*y^82+33256*x^2*y^82+11865*y^82+1261*x^48*y^80+41191*x^46*y^80+23857*x^44*y^80+24554*x^42*y^80+249*x^40*y^80+9784*x^38*y^80+30001*x^36*y^80+5892*x^34*y^80+3102*x^32*y^80+21979*x^30*y^80"
            "+12432*x^28*y^80+5754*x^26*y^80+35541*x^24*y^80+6519*x^22*y^80+27304*x^20*y^80+15388*x^18*y^80+18735*x^16*y^80+9247*x^14*y^80+38058*x^12*y^80+30223*x^10*y^80+23680*x^8*y^80+10792*x^6*y^80+2037*x^4*y^80"
            "+31430*x^2*y^80+37487*y^80+40821*x^50*y^78+29556*x^48*y^78+21181*x^46*y^78+433*x^44*y^78+13497*x^42*y^78+16557*x^40*y^78+3344*x^38*y^78+31070*x^36*y^78+7191*x^34*y^78+34074*x^32*y^78+11389*x^30*y^78"
            "+34808*x^28*y^78+40597*x^26*y^78+18785*x^24*y^78+29152*x^22*y^78+13545*x^20*y^78+15043*x^18*y^78+35323*x^16*y^78+6749*x^14*y^78+10723*x^12*y^78+6215*x^10*y^78+13219*x^8*y^78+2403*x^6*y^78+27159*x^4*y^78"
            "+6739*x^2*y^78+39482*y^78+24936*x^52*y^76+39579*x^50*y^76+18109*x^48*y^76+9200*x^46*y^76+21917*x^44*y^76+21512*x^42*y^76+3494*x^40*y^76+30316*x^38*y^76+27578*x^36*y^76+16055*x^34*y^76+17444*x^32*y^76"
            "+30974*x^30*y^76+29826*x^28*y^76+20885*x^26*y^76+6088*x^24*y^76+39051*x^22*y^76+42106*x^20*y^76+633*x^18*y^76+40532*x^16*y^76+9605*x^14*y^76+12151*x^12*y^76+19166*x^10*y^76+32793*x^8*y^76+26505*x^6*y^76"
            "+17150*x^4*y^76+42179*x^2*y^76+3463*y^76+35354*x^54*y^74+1784*x^52*y^74+39386*x^50*y^74+24457*x^48*y^74+7039*x^46*y^74+5638*x^44*y^74+28496*x^42*y^74+11840*x^40*y^74+8367*x^38*y^74+14337*x^36*y^74"
            "+41239*x^34*y^74+31896*x^32*y^74+16415*x^30*y^74+4437*x^28*y^74+37810*x^26*y^74+21840*x^24*y^74+30056*x^22*y^74+20859*x^20*y^74+15006*x^18*y^74+12615*x^16*y^74+7097*x^14*y^74+39697*x^12*y^74+15179*x^10*y^74"
            "+9091*x^8*y^74+24597*x^6*y^74+33813*x^4*y^74+16431*x^2*y^74+39442*y^74+21994*x^56*y^72+23221*x^54*y^72+4777*x^52*y^72+21109*x^50*y^72+29308*x^48*y^72+37585*x^46*y^72+35642*x^44*y^72+33195*x^42*y^72"
            "+16961*x^40*y^72+5696*x^38*y^72+10891*x^36*y^72+23260*x^34*y^72+21439*x^32*y^72+15920*x^30*y^72+11650*x^28*y^72+17637*x^26*y^72+14778*x^24*y^72+22650*x^22*y^72+40164*x^20*y^72+13178*x^18*y^72+31708*x^16*y^72"
            "+8834*x^14*y^72+2983*x^12*y^72+39918*x^10*y^72+34849*x^8*y^72+23260*x^6*y^72+40205*x^4*y^72+6976*x^2*y^72+2094*y^72+1146*x^58*y^70+24413*x^56*y^70+12873*x^54*y^70+25581*x^52*y^70+20753*x^50*y^70"
            "+24596*x^48*y^70+38206*x^46*y^70+35617*x^44*y^70+24374*x^42*y^70+15768*x^40*y^70+1180*x^38*y^70+18543*x^36*y^70+31526*x^34*y^70+18439*x^32*y^70+5449*x^30*y^70+31649*x^28*y^70+6921*x^26*y^70+40896*x^24*y^70"
            "+17814*x^22*y^70+38174*x^20*y^70+4280*x^18*y^70+10545*x^16*y^70+14409*x^14*y^70+9825*x^12*y^70+11439*x^10*y^70+696*x^8*y^70+9148*x^6*y^70+9556*x^4*y^70+37894*x^2*y^70+28384*y^70+11524*x^60*y^68"
            "+31287*x^58*y^68+36708*x^56*y^68+8833*x^54*y^68+32388*x^52*y^68+34835*x^50*y^68+24759*x^48*y^68+19158*x^46*y^68+37727*x^44*y^68+17225*x^42*y^68+22448*x^40*y^68+29257*x^38*y^68+3050*x^36*y^68+26991*x^34*y^68"
            "+42260*x^32*y^68+28008*x^30*y^68+19951*x^28*y^68+10241*x^26*y^68+4701*x^24*y^68+42846*x^22*y^68+32709*x^20*y^68+24142*x^18*y^68+17844*x^16*y^68+33978*x^14*y^68+24093*x^12*y^68+7345*x^10*y^68+24056*x^8*y^68"
            "+1137*x^6*y^68+24621*x^4*y^68+20000*x^2*y^68+8368*y^68+19698*x^62*y^66+17428*x^60*y^66+18656*x^58*y^66+12098*x^56*y^66+33898*x^54*y^66+36282*x^52*y^66+3982*x^50*y^66+34954*x^48*y^66+25276*x^46*y^66"
            "+7124*x^44*y^66+2548*x^42*y^66+34243*x^40*y^66+33396*x^38*y^66+1802*x^36*y^66+20924*x^34*y^66+22330*x^32*y^66+37722*x^30*y^66+27253*x^28*y^66+20318*x^26*y^66+6915*x^24*y^66+29259*x^22*y^66+40765*x^20*y^66"
            "+40850*x^18*y^66+6350*x^16*y^66+14602*x^14*y^66+16563*x^12*y^66+28333*x^10*y^66+37615*x^8*y^66+34203*x^6*y^66+15083*x^4*y^66+29973*x^2*y^66+11182*y^66+16668*x^64*y^64+17076*x^62*y^64+23108*x^60*y^64"
            "+16076*x^58*y^64+21715*x^56*y^64+42365*x^54*y^64+14679*x^52*y^64+2435*x^50*y^64+4323*x^48*y^64+24661*x^46*y^64+5920*x^44*y^64+30586*x^42*y^64+29527*x^40*y^64+9929*x^38*y^64+40914*x^36*y^64+10823*x^34*y^64"
            "+28152*x^32*y^64+33222*x^30*y^64+33030*x^28*y^64+21067*x^26*y^64+29242*x^24*y^64+24720*x^22*y^64+33707*x^20*y^64+21129*x^18*y^64+6904*x^16*y^64+25010*x^14*y^64+18601*x^12*y^64+15843*x^10*y^64+4486*x^8*y^64"
            "+30374*x^6*y^64+22083*x^4*y^64+31695*x^2*y^64+22493*y^64+12276*x^66*y^62+24005*x^64*y^62+1458*x^62*y^62+32140*x^60*y^62+30310*x^58*y^62+6565*x^56*y^62+42796*x^54*y^62+30875*x^52*y^62+9859*x^50*y^62"
            "+5671*x^48*y^62+41619*x^46*y^62+16726*x^44*y^62+14558*x^42*y^62+38081*x^40*y^62+39519*x^38*y^62+23184*x^36*y^62+7717*x^34*y^62+36359*x^32*y^62+13753*x^30*y^62+15058*x^28*y^62+17577*x^26*y^62+7552*x^24*y^62"
            "+40018*x^22*y^62+18031*x^20*y^62+1277*x^18*y^62+19526*x^16*y^62+14797*x^14*y^62+17118*x^12*y^62+22507*x^10*y^62+32407*x^8*y^62+4565*x^6*y^62+3746*x^4*y^62+41973*x^2*y^62+38408*y^62+41341*x^68*y^60"
            "+15439*x^66*y^60+38824*x^64*y^60+29213*x^62*y^60+22553*x^60*y^60+35030*x^58*y^60+5673*x^56*y^60+3965*x^54*y^60+23647*x^52*y^60+15672*x^50*y^60+23949*x^48*y^60+37541*x^46*y^60+34126*x^44*y^60+35653*x^42*y^60"
            "+30103*x^40*y^60+3398*x^38*y^60+21058*x^36*y^60+4178*x^34*y^60+36146*x^32*y^60+7207*x^30*y^60+12375*x^28*y^60+2010*x^26*y^60+1614*x^24*y^60+6990*x^22*y^60+39480*x^20*y^60+5586*x^18*y^60+24150*x^16*y^60"
            "+20509*x^14*y^60+23325*x^12*y^60+37591*x^10*y^60+18376*x^8*y^60+135*x^6*y^60+38976*x^4*y^60+2479*x^2*y^60+5132*y^60+35081*x^70*y^58+10055*x^68*y^58+31883*x^66*y^58+23638*x^64*y^58+111*x^62*y^58"
            "+36014*x^60*y^58+14815*x^58*y^58+32603*x^56*y^58+14793*x^54*y^58+15900*x^52*y^58+8066*x^50*y^58+9231*x^48*y^58+1433*x^46*y^58+8801*x^44*y^58+34927*x^42*y^58+12648*x^40*y^58+17794*x^38*y^58+4717*x^36*y^58"
            "+22750*x^34*y^58+39069*x^32*y^58+32550*x^30*y^58+12566*x^28*y^58+28916*x^26*y^58+3222*x^24*y^58+18972*x^22*y^58+16638*x^20*y^58+10451*x^18*y^58+33459*x^16*y^58+20790*x^14*y^58+25238*x^12*y^58+21942*x^10*y^58"
            "+25742*x^8*y^58+38731*x^6*y^58+6279*x^4*y^58+33448*x^2*y^58+40665*y^58+1992*x^72*y^56+22602*x^70*y^56+43019*x^68*y^56+38997*x^66*y^56+11795*x^64*y^56+6799*x^62*y^56+21300*x^60*y^56+4250*x^58*y^56"
            "+1391*x^56*y^56+35236*x^54*y^56+13580*x^52*y^56+32097*x^50*y^56+9259*x^48*y^56+5648*x^46*y^56+25912*x^44*y^56+26802*x^42*y^56+3076*x^40*y^56+1568*x^38*y^56+14097*x^36*y^56+6182*x^34*y^56+42121*x^32*y^56"
            "+33024*x^30*y^56+31955*x^28*y^56+6931*x^26*y^56+7828*x^24*y^56+15697*x^22*y^56+40832*x^20*y^56+18829*x^18*y^56+22533*x^16*y^56+40039*x^14*y^56+20856*x^12*y^56+7369*x^10*y^56+38738*x^8*y^56+16962*x^6*y^56"
            "+6555*x^4*y^56+6155*x^2*y^56+17337*y^56+25063*x^74*y^54+7849*x^72*y^54+3993*x^70*y^54+33525*x^68*y^54+35796*x^66*y^54+27258*x^64*y^54+32803*x^62*y^54+21301*x^60*y^54+16149*x^58*y^54+14072*x^56*y^54"
            "+32225*x^54*y^54+42102*x^52*y^54+21056*x^50*y^54+21135*x^48*y^54+21510*x^46*y^54+21638*x^44*y^54+1767*x^42*y^54+40617*x^40*y^54+12037*x^38*y^54+7724*x^36*y^54+13452*x^34*y^54+17124*x^32*y^54+10586*x^30*y^54"
            "+12990*x^28*y^54+14528*x^26*y^54+40503*x^24*y^54+17442*x^22*y^54+30693*x^20*y^54+29523*x^18*y^54+5902*x^16*y^54+41843*x^14*y^54+27627*x^12*y^54+16147*x^10*y^54+13582*x^8*y^54+27791*x^6*y^54+27196*x^4*y^54"
            "+24489*x^2*y^54+16314*y^54+38813*x^76*y^52+37642*x^74*y^52+997*x^72*y^52+8437*x^70*y^52+28275*x^68*y^52+37920*x^66*y^52+39022*x^64*y^52+21600*x^62*y^52+28446*x^60*y^52+12594*x^58*y^52+30021*x^56*y^52"
            "+963*x^54*y^52+27416*x^52*y^52+22971*x^50*y^52+461*x^48*y^52+11039*x^46*y^52+17409*x^44*y^52+6722*x^42*y^52+12652*x^40*y^52+24340*x^38*y^52+26213*x^36*y^52+18368*x^34*y^52+15457*x^32*y^52+42555*x^30*y^52"
            "+13509*x^28*y^52+5437*x^26*y^52+432*x^24*y^52+25772*x^22*y^52+26984*x^20*y^52+26485*x^18*y^52+12322*x^16*y^52+16373*x^14*y^52+27311*x^12*y^52+1147*x^10*y^52+7118*x^8*y^52+24230*x^6*y^52+29138*x^4*y^52"
            "+7764*x^2*y^52+12887*y^52+36780*x^78*y^50+32809*x^76*y^50+35028*x^74*y^50+34763*x^72*y^50+2179*x^70*y^50+21534*x^68*y^50+8824*x^66*y^50+15959*x^64*y^50+40301*x^62*y^50+13376*x^60*y^50+30565*x^58*y^50"
            "+31475*x^56*y^50+30495*x^54*y^50+16418*x^52*y^50+5053*x^50*y^50+26234*x^48*y^50+12901*x^46*y^50+29730*x^44*y^50+40763*x^42*y^50+42343*x^40*y^50+42509*x^38*y^50+35211*x^36*y^50+41383*x^34*y^50+4838*x^32*y^50"
            "+17793*x^30*y^50+16851*x^28*y^50+2502*x^26*y^50+27544*x^24*y^50+25660*x^22*y^50+30680*x^20*y^50+2403*x^18*y^50+2899*x^16*y^50+3400*x^14*y^50+11532*x^12*y^50+22755*x^10*y^50+9158*x^8*y^50+34876*x^6*y^50"
            "+18182*x^4*y^50+6935*x^2*y^50+30723*y^50+41480*x^80*y^48+18665*x^78*y^48+22404*x^76*y^48+39684*x^74*y^48+42481*x^72*y^48+40986*x^70*y^48+9440*x^68*y^48+5311*x^66*y^48+38442*x^64*y^48+26692*x^62*y^48"
            "+7739*x^60*y^48+36291*x^58*y^48+11040*x^56*y^48+15778*x^54*y^48+3419*x^52*y^48+7692*x^50*y^48+22548*x^48*y^48+3458*x^46*y^48+22218*x^44*y^48+8138*x^42*y^48+29129*x^40*y^48+37396*x^38*y^48+41999*x^36*y^48"
            "+22450*x^34*y^48+35776*x^32*y^48+16663*x^30*y^48+30596*x^28*y^48+5572*x^26*y^48+11292*x^24*y^48+22640*x^22*y^48+3895*x^20*y^48+23764*x^18*y^48+5658*x^16*y^48+1770*x^14*y^48+14594*x^12*y^48+35696*x^10*y^48"
            "+28740*x^8*y^48+39807*x^6*y^48+21201*x^4*y^48+32788*x^2*y^48+467*y^48+28277*x^82*y^46+21388*x^80*y^46+29022*x^78*y^46+10823*x^76*y^46+16670*x^74*y^46+18291*x^72*y^46+30573*x^70*y^46+27194*x^68*y^46"
            "+36056*x^66*y^46+8701*x^64*y^46+21702*x^62*y^46+1608*x^60*y^46+1153*x^58*y^46+32066*x^56*y^46+5275*x^54*y^46+26654*x^52*y^46+40862*x^50*y^46+11931*x^48*y^46+40390*x^46*y^46+39493*x^44*y^46+13443*x^42*y^46"
            "+40146*x^40*y^46+33850*x^38*y^46+11769*x^36*y^46+16720*x^34*y^46+38288*x^32*y^46+31694*x^30*y^46+6762*x^28*y^46+20344*x^26*y^46+39827*x^24*y^46+7122*x^22*y^46+24426*x^20*y^46+22137*x^18*y^46+20010*x^16*y^46"
            "+6158*x^14*y^46+42761*x^12*y^46+34882*x^10*y^46+21773*x^8*y^46+4728*x^6*y^46+23276*x^4*y^46+13858*x^2*y^46+19550*y^46+18042*x^84*y^44+18424*x^82*y^44+30517*x^80*y^44+27229*x^78*y^44+10937*x^76*y^44"
            "+18269*x^74*y^44+12745*x^72*y^44+23122*x^70*y^44+15015*x^68*y^44+22397*x^66*y^44+33999*x^64*y^44+38038*x^62*y^44+28303*x^60*y^44+10625*x^58*y^44+18795*x^56*y^44+39147*x^54*y^44+14803*x^52*y^44+34315*x^50*y^44"
            "+27419*x^48*y^44+15314*x^46*y^44+36436*x^44*y^44+3411*x^42*y^44+28836*x^40*y^44+12758*x^38*y^44+31308*x^36*y^44+34208*x^34*y^44+12621*x^32*y^44+1299*x^30*y^44+39496*x^28*y^44+35319*x^26*y^44+16066*x^24*y^44"
            "+1968*x^22*y^44+17711*x^20*y^44+24536*x^18*y^44+24805*x^16*y^44+563*x^14*y^44+22542*x^12*y^44+6718*x^10*y^44+16265*x^8*y^44+2024*x^6*y^44+24659*x^4*y^44+19696*x^2*y^44+4535*y^44+10504*x^86*y^42"
            "+37860*x^84*y^42+32406*x^82*y^42+14292*x^80*y^42+12380*x^78*y^42+15078*x^76*y^42+9167*x^74*y^42+38278*x^72*y^42+3597*x^70*y^42+38756*x^68*y^42+36343*x^66*y^42+24110*x^64*y^42+19712*x^62*y^42+17022*x^60*y^42"
            "+18779*x^58*y^42+18561*x^56*y^42+32615*x^54*y^42+14108*x^52*y^42+31567*x^50*y^42+9235*x^48*y^42+8428*x^46*y^42+17151*x^44*y^42+14630*x^42*y^42+31408*x^40*y^42+15386*x^38*y^42+26901*x^36*y^42+19188*x^34*y^42"
            "+35660*x^32*y^42+18112*x^30*y^42+9989*x^28*y^42+42510*x^26*y^42+4515*x^24*y^42+16339*x^22*y^42+24628*x^20*y^42+41504*x^18*y^42+42523*x^16*y^42+18954*x^14*y^42+18903*x^12*y^42+36965*x^10*y^42+9151*x^8*y^42"
            "+38003*x^6*y^42+40263*x^4*y^42+24391*x^2*y^42+14740*y^42+31188*x^88*y^40+21316*x^86*y^40+27934*x^84*y^40+31960*x^82*y^40+778*x^80*y^40+21473*x^78*y^40+20012*x^76*y^40+28203*x^74*y^40+31068*x^72*y^40"
            "+19311*x^70*y^40+33559*x^68*y^40+19931*x^66*y^40+16403*x^64*y^40+3283*x^62*y^40+11978*x^60*y^40+35948*x^58*y^40+17011*x^56*y^40+16383*x^54*y^40+27811*x^52*y^40+2656*x^50*y^40+31268*x^48*y^40+33075*x^46*y^40"
            "+24069*x^44*y^40+24003*x^42*y^40+9713*x^40*y^40+17333*x^38*y^40+19700*x^36*y^40+24090*x^34*y^40+37365*x^32*y^40+2711*x^30*y^40+29136*x^28*y^40+16831*x^26*y^40+10347*x^24*y^40+22176*x^22*y^40+29371*x^20*y^40"
            "+36081*x^18*y^40+5113*x^16*y^40+32126*x^14*y^40+2028*x^12*y^40+30741*x^10*y^40+33024*x^8*y^40+10398*x^6*y^40+27455*x^4*y^40+34902*x^2*y^40+30512*y^40+29849*x^90*y^38+41133*x^88*y^38+27716*x^86*y^38"
            "+138*x^84*y^38+32679*x^82*y^38+2747*x^80*y^38+33487*x^78*y^38+13823*x^76*y^38+39475*x^74*y^38+3497*x^72*y^38+41243*x^70*y^38+39880*x^68*y^38+12411*x^66*y^38+34154*x^64*y^38+5074*x^62*y^38+22161*x^60*y^38"
            "+39319*x^58*y^38+25549*x^56*y^38+15968*x^54*y^38+8305*x^52*y^38+41426*x^50*y^38+35134*x^48*y^38+36521*x^46*y^38+14182*x^44*y^38+30536*x^42*y^38+8423*x^40*y^38+5474*x^38*y^38+31565*x^36*y^38+4711*x^34*y^38"
            "+43038*x^32*y^38+38258*x^30*y^38+42573*x^28*y^38+8200*x^26*y^38+11618*x^24*y^38+18823*x^22*y^38+10215*x^20*y^38+36187*x^18*y^38+42142*x^16*y^38+37681*x^14*y^38+42931*x^12*y^38+38124*x^10*y^38+17536*x^8*y^38"
            "+35925*x^6*y^38+8998*x^4*y^38+27564*x^2*y^38+16004*y^38+12442*x^92*y^36+29872*x^90*y^36+38715*x^88*y^36+5344*x^86*y^36+40687*x^84*y^36+15441*x^82*y^36+942*x^80*y^36+34058*x^78*y^36+12928*x^76*y^36"
            "+9450*x^74*y^36+10515*x^72*y^36+9330*x^70*y^36+15511*x^68*y^36+31114*x^66*y^36+42745*x^64*y^36+14424*x^62*y^36+7069*x^60*y^36+1769*x^58*y^36+13742*x^56*y^36+303*x^54*y^36+24852*x^52*y^36+38114*x^50*y^36"
            "+42462*x^48*y^36+16431*x^46*y^36+22962*x^44*y^36+14290*x^42*y^36+12347*x^40*y^36+17425*x^38*y^36+11366*x^36*y^36+27289*x^34*y^36+5298*x^32*y^36+28347*x^30*y^36+23967*x^28*y^36+20786*x^26*y^36+1208*x^24*y^36"
            "+19388*x^22*y^36+1056*x^20*y^36+21187*x^18*y^36+10954*x^16*y^36+5538*x^14*y^36+28512*x^12*y^36+13434*x^10*y^36+31202*x^8*y^36+36133*x^6*y^36+10807*x^4*y^36+24627*x^2*y^36+18242*y^36+32303*x^94*y^34"
            "+34804*x^92*y^34+12595*x^90*y^34+466*x^88*y^34+25693*x^86*y^34+30639*x^84*y^34+39699*x^82*y^34+38419*x^80*y^34+6841*x^78*y^34+39044*x^76*y^34+31469*x^74*y^34+14907*x^72*y^34+18833*x^70*y^34+6608*x^68*y^34"
            "+17902*x^66*y^34+8702*x^64*y^34+15290*x^62*y^34+24784*x^60*y^34+11200*x^58*y^34+4515*x^56*y^34+18866*x^54*y^34+39522*x^52*y^34+11216*x^50*y^34+30197*x^48*y^34+38665*x^46*y^34+24942*x^44*y^34+2599*x^42*y^34"
            "+28470*x^40*y^34+39133*x^38*y^34+7643*x^36*y^34+41850*x^34*y^34+37617*x^32*y^34+34835*x^30*y^34+36766*x^28*y^34+27850*x^26*y^34+30279*x^24*y^34+38347*x^22*y^34+31850*x^20*y^34+36012*x^18*y^34+1895*x^16*y^34"
            "+17845*x^14*y^34+23484*x^12*y^34+40104*x^10*y^34+13404*x^8*y^34+40238*x^6*y^34+6260*x^4*y^34+41873*x^2*y^34+21135*y^34+30296*x^96*y^32+1003*x^94*y^32+30294*x^92*y^32+21119*x^90*y^32+1811*x^88*y^32"
            "+39110*x^86*y^32+1530*x^84*y^32+22334*x^82*y^32+22487*x^80*y^32+33660*x^78*y^32+35557*x^76*y^32+14741*x^74*y^32+13387*x^72*y^32+32164*x^70*y^32+18429*x^68*y^32+36119*x^66*y^32+35287*x^64*y^32+42840*x^62*y^32"
            "+25360*x^60*y^32+4209*x^58*y^32+36431*x^56*y^32+20892*x^54*y^32+22762*x^52*y^32+37186*x^50*y^32+40139*x^48*y^32+33507*x^46*y^32+39864*x^44*y^32+17099*x^42*y^32+24078*x^40*y^32+13217*x^38*y^32+26178*x^36*y^32"
            "+27115*x^34*y^32+5264*x^32*y^32+20163*x^30*y^32+21928*x^28*y^32+9182*x^26*y^32+7375*x^24*y^32+36810*x^22*y^32+30855*x^20*y^32+27868*x^18*y^32+17233*x^16*y^32+6041*x^14*y^32+35985*x^12*y^32+23924*x^10*y^32"
            "+16483*x^8*y^32+6214*x^6*y^32+911*x^4*y^32+39978*x^2*y^32+31930*y^32+13259*x^98*y^30+1649*x^96*y^30+42343*x^94*y^30+40042*x^92*y^30+17446*x^90*y^30+12726*x^88*y^30+37524*x^86*y^30+36817*x^84*y^30"
            "+24541*x^82*y^30+27597*x^80*y^30+36919*x^78*y^30+9540*x^76*y^30+26864*x^74*y^30+5141*x^72*y^30+38779*x^70*y^30+3244*x^68*y^30+23123*x^66*y^30+40038*x^64*y^30+40294*x^62*y^30+33946*x^60*y^30+22834*x^58*y^30"
            "+5815*x^56*y^30+14495*x^54*y^30+3736*x^52*y^30+25571*x^50*y^30+30148*x^48*y^30+42213*x^46*y^30+1907*x^44*y^30+7740*x^42*y^30+33694*x^40*y^30+29242*x^38*y^30+37577*x^36*y^30+34653*x^34*y^30+13100*x^32*y^30"
            "+22401*x^30*y^30+16864*x^28*y^30+17561*x^26*y^30+35158*x^24*y^30+30240*x^22*y^30+1402*x^20*y^30+13055*x^18*y^30+18530*x^16*y^30+1017*x^14*y^30+7572*x^12*y^30+34197*x^10*y^30+12809*x^8*y^30+9511*x^6*y^30"
            "+11985*x^4*y^30+31220*x^2*y^30+9145*y^30+40351*x^100*y^28+20622*x^98*y^28+22519*x^96*y^28+8261*x^94*y^28+21997*x^92*y^28+30223*x^90*y^28+19856*x^88*y^28+33089*x^86*y^28+11154*x^84*y^28+4030*x^82*y^28"
            "+38869*x^80*y^28+4924*x^78*y^28+40791*x^76*y^28+40472*x^74*y^28+9788*x^72*y^28+2117*x^70*y^28+7080*x^68*y^28+16187*x^66*y^28+13884*x^64*y^28+15498*x^62*y^28+10730*x^60*y^28+17576*x^58*y^28+28463*x^56*y^28"
            "+29228*x^54*y^28+13000*x^52*y^28+15277*x^50*y^28+25205*x^48*y^28+1737*x^46*y^28+16335*x^44*y^28+34544*x^42*y^28+26159*x^40*y^28+1170*x^38*y^28+37992*x^36*y^28+23194*x^34*y^28+22400*x^32*y^28+33934*x^30*y^28"
            "+33699*x^28*y^28+36330*x^26*y^28+38636*x^24*y^28+42919*x^22*y^28+175*x^20*y^28+23148*x^18*y^28+18512*x^16*y^28+5208*x^14*y^28+39725*x^12*y^28+18623*x^10*y^28+5052*x^8*y^28+5972*x^6*y^28+31164*x^4*y^28"
            "+16078*x^2*y^28+13045*y^28+11763*x^102*y^26+21004*x^100*y^26+35020*x^98*y^26+16898*x^96*y^26+16872*x^94*y^26+20002*x^92*y^26+2124*x^90*y^26+10098*x^88*y^26+8120*x^86*y^26+1522*x^84*y^26+8732*x^82*y^26"
            "+4742*x^80*y^26+3280*x^78*y^26+13553*x^76*y^26+31868*x^74*y^26+20962*x^72*y^26+10907*x^70*y^26+39769*x^68*y^26+15994*x^66*y^26+5412*x^64*y^26+29519*x^62*y^26+11859*x^60*y^26+40331*x^58*y^26+22206*x^56*y^26"
            "+8824*x^54*y^26+17297*x^52*y^26+29074*x^50*y^26+18621*x^48*y^26+6710*x^46*y^26+9842*x^44*y^26+34232*x^42*y^26+318*x^40*y^26+7817*x^38*y^26+4536*x^36*y^26+32775*x^34*y^26+13246*x^32*y^26+23924*x^30*y^26"
            "+7712*x^28*y^26+556*x^26*y^26+16362*x^24*y^26+10096*x^22*y^26+7565*x^20*y^26+4982*x^18*y^26+9802*x^16*y^26+25388*x^14*y^26+25321*x^12*y^26+8307*x^10*y^26+27764*x^8*y^26+24583*x^6*y^26+27578*x^4*y^26"
            "+21206*x^2*y^26+37859*y^26+28377*x^104*y^24+8962*x^102*y^24+11376*x^100*y^24+14525*x^98*y^24+42979*x^96*y^24+4682*x^94*y^24+9041*x^92*y^24+8583*x^90*y^24+37014*x^88*y^24+29291*x^86*y^24+12310*x^84*y^24"
            "+31482*x^82*y^24+16114*x^80*y^24+30997*x^78*y^24+20887*x^76*y^24+17150*x^74*y^24+33006*x^72*y^24+20600*x^70*y^24+5047*x^68*y^24+17114*x^66*y^24+609*x^64*y^24+7507*x^62*y^24+25411*x^60*y^24+40113*x^58*y^24"
            "+29618*x^56*y^24+19746*x^54*y^24+35582*x^52*y^24+4122*x^50*y^24+3359*x^48*y^24+30716*x^46*y^24+24840*x^44*y^24+1799*x^42*y^24+7841*x^40*y^24+3141*x^38*y^24+634*x^36*y^24+25616*x^34*y^24+32660*x^32*y^24"
            "+41753*x^30*y^24+23043*x^28*y^24+42804*x^26*y^24+34500*x^24*y^24+28351*x^22*y^24+26411*x^20*y^24+14351*x^18*y^24+40745*x^16*y^24+19739*x^14*y^24+28593*x^12*y^24+37098*x^10*y^24+20578*x^8*y^24+31054*x^6*y^24"
            "+37188*x^4*y^24+4924*x^2*y^24+4891*y^24+22744*x^106*y^22+27515*x^104*y^22+5117*x^102*y^22+32814*x^100*y^22+13114*x^98*y^22+37879*x^96*y^22+16011*x^94*y^22+24858*x^92*y^22+23518*x^90*y^22+26351*x^88*y^22"
            "+21153*x^86*y^22+36728*x^84*y^22+22112*x^82*y^22+22345*x^80*y^22+6502*x^78*y^22+23010*x^76*y^22+41233*x^74*y^22+10499*x^72*y^22+19707*x^70*y^22+13145*x^68*y^22+38094*x^66*y^22+38179*x^64*y^22+38218*x^62*y^22"
            "+11010*x^60*y^22+30153*x^58*y^22+10249*x^56*y^22+33039*x^54*y^22+33936*x^52*y^22+9793*x^50*y^22+18374*x^48*y^22+27264*x^46*y^22+9677*x^44*y^22+3607*x^42*y^22+25575*x^40*y^22+18776*x^38*y^22+37784*x^36*y^22"
            "+6192*x^34*y^22+27424*x^32*y^22+31379*x^30*y^22+42991*x^28*y^22+35781*x^26*y^22+35142*x^24*y^22+27961*x^22*y^22+40257*x^20*y^22+21985*x^18*y^22+42941*x^16*y^22+9119*x^14*y^22+15735*x^12*y^22+24902*x^10*y^22"
            "+21706*x^8*y^22+25112*x^6*y^22+6725*x^4*y^22+24107*x^2*y^22+15931*y^22+33321*x^108*y^20+10563*x^106*y^20+3363*x^104*y^20+29142*x^102*y^20+10993*x^100*y^20+6844*x^98*y^20+6045*x^96*y^20+16946*x^94*y^20"
            "+40994*x^92*y^20+18802*x^90*y^20+5729*x^88*y^20+40880*x^86*y^20+4686*x^84*y^20+38678*x^82*y^20+23143*x^80*y^20+4598*x^78*y^20+1292*x^76*y^20+32374*x^74*y^20+1789*x^72*y^20+4012*x^70*y^20+6061*x^68*y^20"
            "+23981*x^66*y^20+26920*x^64*y^20+12102*x^62*y^20+40871*x^60*y^20+38508*x^58*y^20+8448*x^56*y^20+3318*x^54*y^20+25796*x^52*y^20+3765*x^50*y^20+13953*x^48*y^20+39638*x^46*y^20+7641*x^44*y^20+34281*x^42*y^20"
            "+25188*x^40*y^20+6245*x^38*y^20+7963*x^36*y^20+10665*x^34*y^20+19712*x^32*y^20+21808*x^30*y^20+36183*x^28*y^20+34582*x^26*y^20+24967*x^24*y^20+262*x^22*y^20+42403*x^20*y^20+28224*x^18*y^20+10950*x^16*y^20"
            "+29651*x^14*y^20+23728*x^12*y^20+17933*x^10*y^20+3664*x^8*y^20+26875*x^6*y^20+34708*x^4*y^20+2642*x^2*y^20+33896*y^20+27775*x^110*y^18+41310*x^108*y^18+30173*x^106*y^18+21709*x^104*y^18+37047*x^102*y^18"
            "+36285*x^100*y^18+14457*x^98*y^18+31018*x^96*y^18+40470*x^94*y^18+18310*x^92*y^18+10704*x^90*y^18+13274*x^88*y^18+42113*x^86*y^18+22749*x^84*y^18+14881*x^82*y^18+39762*x^80*y^18+19974*x^78*y^18"
            "+27029*x^76*y^18+30579*x^74*y^18+41704*x^72*y^18+27726*x^70*y^18+16723*x^68*y^18+38490*x^66*y^18+36600*x^64*y^18+36024*x^62*y^18+38352*x^60*y^18+41432*x^58*y^18+23731*x^56*y^18+17018*x^54*y^18+24536*x^52*y^18"
            "+32150*x^50*y^18+6730*x^48*y^18+22602*x^46*y^18+5261*x^44*y^18+5536*x^42*y^18+18176*x^40*y^18+12507*x^38*y^18+12327*x^36*y^18+5200*x^34*y^18+3951*x^32*y^18+1139*x^30*y^18+28799*x^28*y^18+42298*x^26*y^18"
            "+15858*x^24*y^18+18482*x^22*y^18+42595*x^20*y^18+41085*x^18*y^18+36609*x^16*y^18+30801*x^14*y^18+42662*x^12*y^18+33381*x^10*y^18+14186*x^8*y^18+2973*x^6*y^18+14058*x^4*y^18+9942*x^2*y^18+22053*y^18"
            "+16123*x^112*y^16+9727*x^110*y^16+39670*x^108*y^16+19731*x^106*y^16+37428*x^104*y^16+5815*x^102*y^16+15514*x^100*y^16+25870*x^98*y^16+29954*x^96*y^16+4840*x^94*y^16+33251*x^92*y^16+3840*x^90*y^16"
            "+21369*x^88*y^16+1379*x^86*y^16+37462*x^84*y^16+40905*x^82*y^16+27393*x^80*y^16+10898*x^78*y^16+546*x^76*y^16+35102*x^74*y^16+24746*x^72*y^16+35841*x^70*y^16+39231*x^68*y^16+38679*x^66*y^16+34683*x^64*y^16"
            "+12743*x^62*y^16+5524*x^60*y^16+16862*x^58*y^16+33161*x^56*y^16+25286*x^54*y^16+30665*x^52*y^16+37040*x^50*y^16+6538*x^48*y^16+10239*x^46*y^16+15301*x^44*y^16+23289*x^42*y^16+22443*x^40*y^16+35604*x^38*y^16"
            "+15837*x^36*y^16+5135*x^34*y^16+33046*x^32*y^16+11347*x^30*y^16+16223*x^28*y^16+7607*x^26*y^16+12076*x^24*y^16+15038*x^22*y^16+37709*x^20*y^16+41144*x^18*y^16+9187*x^16*y^16+32064*x^14*y^16+8593*x^12*y^16"
            "+10499*x^10*y^16+7764*x^8*y^16+36493*x^6*y^16+17176*x^4*y^16+32635*x^2*y^16+19758*y^16+25384*x^114*y^14+42375*x^112*y^14+9096*x^110*y^14+689*x^108*y^14+26996*x^106*y^14+27150*x^104*y^14+12256*x^102*y^14"
            "+27479*x^100*y^14+28589*x^98*y^14+23303*x^96*y^14+7653*x^94*y^14+32175*x^92*y^14+17866*x^90*y^14+1954*x^88*y^14+38326*x^86*y^14+24090*x^84*y^14+15817*x^82*y^14+1302*x^80*y^14+18067*x^78*y^14+39447*x^76*y^14"
            "+24658*x^74*y^14+21765*x^72*y^14+41654*x^70*y^14+18524*x^68*y^14+32640*x^66*y^14+28450*x^64*y^14+7676*x^62*y^14+14977*x^60*y^14+7476*x^58*y^14+30438*x^56*y^14+27107*x^54*y^14+23548*x^52*y^14+37117*x^50*y^14"
            "+19375*x^48*y^14+3066*x^46*y^14+1255*x^44*y^14+12791*x^42*y^14+24241*x^40*y^14+20990*x^38*y^14+21084*x^36*y^14+22010*x^34*y^14+37408*x^32*y^14+32849*x^30*y^14+35358*x^28*y^14+23592*x^26*y^14+15623*x^24*y^14"
            "+38007*x^22*y^14+11885*x^20*y^14+21078*x^18*y^14+41118*x^16*y^14+19656*x^14*y^14+12726*x^12*y^14+33941*x^10*y^14+42913*x^8*y^14+27271*x^6*y^14+25971*x^4*y^14+18223*x^2*y^14+29918*y^14+1067*x^116*y^12"
            "+21936*x^114*y^12+4889*x^112*y^12+4369*x^110*y^12+23938*x^108*y^12+27567*x^106*y^12+11840*x^104*y^12+30365*x^102*y^12+30369*x^100*y^12+11970*x^98*y^12+15389*x^96*y^12+37830*x^94*y^12+17303*x^92*y^12"
            "+19292*x^90*y^12+15776*x^88*y^12+249*x^86*y^12+342*x^84*y^12+3535*x^82*y^12+36936*x^80*y^12+3147*x^78*y^12+20237*x^76*y^12+25742*x^74*y^12+354*x^72*y^12+40560*x^70*y^12+15310*x^68*y^12+35580*x^66*y^12"
            "+19979*x^64*y^12+27972*x^62*y^12+6767*x^60*y^12+37514*x^58*y^12+8011*x^56*y^12+6231*x^54*y^12+24350*x^52*y^12+25423*x^50*y^12+38459*x^48*y^12+40086*x^46*y^12+15813*x^44*y^12+17851*x^42*y^12+37129*x^40*y^12"
            "+3496*x^38*y^12+33837*x^36*y^12+5447*x^34*y^12+6383*x^32*y^12+9474*x^30*y^12+24724*x^28*y^12+6481*x^26*y^12+15411*x^24*y^12+13093*x^22*y^12+40731*x^20*y^12+6468*x^18*y^12+40848*x^16*y^12+21875*x^14*y^12"
            "+1108*x^10*y^12+16118*x^8*y^12+34812*x^6*y^12+9296*x^4*y^12+21946*x^2*y^12+22856*y^12+35518*x^118*y^10+25608*x^116*y^10+14746*x^114*y^10+10454*x^112*y^10+19292*x^110*y^10+11001*x^108*y^10+36431*x^106*y^10"
            "+36762*x^104*y^10+3397*x^102*y^10+33909*x^100*y^10+28242*x^98*y^10+32528*x^96*y^10+22459*x^94*y^10+23104*x^92*y^10+17464*x^90*y^10+13727*x^88*y^10+13076*x^86*y^10+11231*x^84*y^10+23416*x^82*y^10"
            "+40218*x^80*y^10+18282*x^78*y^10+42255*x^76*y^10+10871*x^74*y^10+6648*x^72*y^10+20228*x^70*y^10+29206*x^68*y^10+1569*x^66*y^10+19627*x^64*y^10+24770*x^62*y^10+33794*x^60*y^10+30866*x^58*y^10+29169*x^56*y^10"
            "+28939*x^54*y^10+30376*x^52*y^10+42151*x^50*y^10+19761*x^48*y^10+37827*x^46*y^10+3359*x^44*y^10+9546*x^42*y^10+25614*x^40*y^10+31995*x^38*y^10+21234*x^36*y^10+33334*x^34*y^10+25114*x^32*y^10+38854*x^30*y^10"
            "+389*x^28*y^10+20600*x^26*y^10+22432*x^24*y^10+15923*x^22*y^10+30038*x^20*y^10+29312*x^18*y^10+14323*x^16*y^10+34932*x^14*y^10+36115*x^12*y^10+5416*x^10*y^10+41426*x^8*y^10+33099*x^6*y^10+12083*x^4*y^10"
            "+17452*x^2*y^10+42701*y^10+42314*x^120*y^8+21544*x^118*y^8+37500*x^116*y^8+13985*x^114*y^8+730*x^112*y^8+33930*x^110*y^8+27030*x^108*y^8+12788*x^106*y^8+3851*x^104*y^8+8245*x^102*y^8+25684*x^100*y^8"
            "+16418*x^98*y^8+30443*x^96*y^8+35812*x^94*y^8+39533*x^92*y^8+19972*x^90*y^8+23609*x^88*y^8+8468*x^86*y^8+32765*x^84*y^8+36027*x^82*y^8+29048*x^80*y^8+20962*x^78*y^8+11490*x^76*y^8+11794*x^74*y^8+9219*x^72*y^8"
            "+6885*x^70*y^8+38853*x^68*y^8+11940*x^66*y^8+11316*x^64*y^8+16865*x^62*y^8+17924*x^60*y^8+32336*x^58*y^8+16885*x^56*y^8+25698*x^54*y^8+1379*x^52*y^8+20310*x^50*y^8+41692*x^48*y^8+38184*x^46*y^8+29771*x^44*y^8"
            "+34485*x^42*y^8+11760*x^40*y^8+7575*x^38*y^8+35492*x^36*y^8+28155*x^34*y^8+23719*x^32*y^8+18253*x^30*y^8+3509*x^28*y^8+20494*x^26*y^8+23353*x^24*y^8+15905*x^22*y^8+24318*x^20*y^8+20535*x^18*y^8+20057*x^16*y^8"
            "+15786*x^14*y^8+38197*x^12*y^8+21260*x^10*y^8+41792*x^8*y^8+6679*x^6*y^8+42019*x^4*y^8+22441*x^2*y^8+12577*y^8+13606*x^122*y^6+31259*x^120*y^6+3199*x^118*y^6+21030*x^116*y^6+37581*x^114*y^6+14359*x^112*y^6"
            "+17525*x^110*y^6+15288*x^108*y^6+22520*x^106*y^6+32196*x^104*y^6+30644*x^102*y^6+33321*x^100*y^6+9241*x^98*y^6+17384*x^96*y^6+16640*x^94*y^6+35673*x^92*y^6+6638*x^90*y^6+20454*x^88*y^6+5794*x^86*y^6"
            "+22843*x^84*y^6+30716*x^82*y^6+10880*x^80*y^6+37932*x^78*y^6+31310*x^76*y^6+36341*x^74*y^6+731*x^72*y^6+8364*x^70*y^6+28596*x^68*y^6+33290*x^66*y^6+29033*x^64*y^6+28890*x^62*y^6+37977*x^60*y^6+27613*x^58*y^6"
            "+24192*x^56*y^6+9950*x^54*y^6+29308*x^52*y^6+11359*x^50*y^6+41909*x^48*y^6+203*x^46*y^6+38196*x^44*y^6+7704*x^42*y^6+10320*x^40*y^6+30688*x^38*y^6+35354*x^36*y^6+6957*x^34*y^6+20292*x^32*y^6+38218*x^30*y^6"
            "+24611*x^28*y^6+20466*x^26*y^6+15209*x^24*y^6+19270*x^22*y^6+27721*x^20*y^6+6845*x^18*y^6+29018*x^16*y^6+23952*x^14*y^6+27870*x^12*y^6+40985*x^10*y^6+2602*x^8*y^6+20584*x^6*y^6+32011*x^4*y^6+38749*x^2*y^6"
            "+40627*y^6+9994*x^124*y^4+34119*x^122*y^4+9024*x^120*y^4+37204*x^118*y^4+33887*x^116*y^4+9177*x^114*y^4+15179*x^112*y^4+36258*x^110*y^4+30374*x^108*y^4+19268*x^106*y^4+3566*x^104*y^4+33516*x^102*y^4"
            "+22410*x^100*y^4+1492*x^98*y^4+30548*x^96*y^4+40127*x^94*y^4+10646*x^92*y^4+1666*x^90*y^4+29859*x^88*y^4+26565*x^86*y^4+5696*x^84*y^4+10268*x^82*y^4+20378*x^80*y^4+733*x^78*y^4+18248*x^76*y^4+2908*x^74*y^4"
            "+41169*x^72*y^4+16635*x^70*y^4+4059*x^68*y^4+39873*x^66*y^4+16195*x^64*y^4+33356*x^62*y^4+8160*x^60*y^4+1093*x^58*y^4+16063*x^56*y^4+37764*x^54*y^4+12619*x^52*y^4+5780*x^50*y^4+16383*x^48*y^4+15494*x^46*y^4"
            "+26499*x^44*y^4+32670*x^42*y^4+29531*x^40*y^4+11272*x^38*y^4+6931*x^36*y^4+41099*x^34*y^4+1210*x^32*y^4+8810*x^30*y^4+27165*x^28*y^4+37630*x^26*y^4+7329*x^24*y^4+31246*x^22*y^4+17787*x^20*y^4+33572*x^18*y^4"
            "+16678*x^16*y^4+38212*x^14*y^4+30056*x^12*y^4+35986*x^10*y^4+12193*x^8*y^4+9064*x^6*y^4+24665*x^4*y^4+400*x^2*y^4+21210*y^4+42603*x^126*y^2+36901*x^124*y^2+33142*x^122*y^2+19026*x^120*y^2+26257*x^118*y^2"
            "+31668*x^116*y^2+21902*x^114*y^2+41439*x^112*y^2+36772*x^110*y^2+16520*x^108*y^2+16801*x^106*y^2+13208*x^104*y^2+22892*x^102*y^2+36494*x^100*y^2+42852*x^98*y^2+35447*x^96*y^2+29449*x^94*y^2+11369*x^92*y^2"
            "+23905*x^90*y^2+6904*x^88*y^2+23311*x^86*y^2+16416*x^84*y^2+17023*x^82*y^2+6396*x^80*y^2+29010*x^78*y^2+28121*x^76*y^2+4439*x^74*y^2+41866*x^72*y^2+2443*x^70*y^2+1787*x^68*y^2+23637*x^66*y^2+35450*x^64*y^2"
            "+25101*x^62*y^2+38210*x^60*y^2+25920*x^58*y^2+977*x^56*y^2+37499*x^54*y^2+43019*x^52*y^2+17431*x^50*y^2+42286*x^48*y^2+26977*x^46*y^2+42471*x^44*y^2+42637*x^42*y^2+16109*x^40*y^2+1806*x^38*y^2+21331*x^36*y^2"
            "+43042*x^34*y^2+21708*x^32*y^2+2999*x^30*y^2+21064*x^28*y^2+32594*x^26*y^2+7587*x^24*y^2+2077*x^22*y^2+27117*x^20*y^2+21429*x^18*y^2+24873*x^16*y^2+40822*x^14*y^2+22856*x^12*y^2+39615*x^10*y^2+32121*x^8*y^2"
            "+8136*x^6*y^2+40332*x^4*y^2+12301*x^2*y^2+32848*y^2+x^128+41259*x^126+32543*x^124+26938*x^122+25117*x^120+3458*x^118+28947*x^116+19659*x^114+20639*x^112+41369*x^110+6547*x^108+19680*x^106+31688*x^104"
            "+10952*x^102+1407*x^100+1278*x^98+13056*x^96+32272*x^94+19351*x^92+11254*x^90+18780*x^88+41868*x^86+13034*x^84+38190*x^82+34930*x^80+23678*x^78+7426*x^76+13585*x^74+41317*x^72+40154*x^70+10739*x^68+22876*x^66"
            "+19733*x^64+3127*x^62+33675*x^60+30109*x^58+20180*x^56+13924*x^54+4045*x^52+37752*x^50+30118*x^48+1615*x^46+4510*x^44+16577*x^42+7536*x^40+34837*x^38+241*x^36+9705*x^34+2196*x^32+33198*x^30+21393*x^28"
            "+32583*x^26+33148*x^24+12208*x^22+31138*x^20+15252*x^18+5157*x^16+15920*x^14+761*x^12+5651*x^10+42802*x^8+14396*x^6+34190*x^4+31143*x^2+39447)"
            "*(y^128+42603*x^2*y^126+41259*y^126+9994*x^4*y^124+36901*x^2*y^124+32543*y^124+13606*x^6*y^122+34119*x^4*y^122+33142*x^2*y^122+26938*y^122+42314*x^8*y^120+31259*x^6*y^120+9024*x^4*y^120+19026*x^2*y^120"
            "+25117*y^120+35518*x^10*y^118+21544*x^8*y^118+3199*x^6*y^118+37204*x^4*y^118+26257*x^2*y^118+3458*y^118+1067*x^12*y^116+25608*x^10*y^116+37500*x^8*y^116+21030*x^6*y^116+33887*x^4*y^116+31668*x^2*y^116"
            "+28947*y^116+25384*x^14*y^114+21936*x^12*y^114+14746*x^10*y^114+13985*x^8*y^114+37581*x^6*y^114+9177*x^4*y^114+21902*x^2*y^114+19659*y^114+16123*x^16*y^112+42375*x^14*y^112+4889*x^12*y^112+10454*x^10*y^112"
            "+730*x^8*y^112+14359*x^6*y^112+15179*x^4*y^112+41439*x^2*y^112+20639*y^112+27775*x^18*y^110+9727*x^16*y^110+9096*x^14*y^110+4369*x^12*y^110+19292*x^10*y^110+33930*x^8*y^110+17525*x^6*y^110+36258*x^4*y^110"
            "+36772*x^2*y^110+41369*y^110+33321*x^20*y^108+41310*x^18*y^108+39670*x^16*y^108+689*x^14*y^108+23938*x^12*y^108+11001*x^10*y^108+27030*x^8*y^108+15288*x^6*y^108+30374*x^4*y^108+16520*x^2*y^108+6547*y^108"
            "+22744*x^22*y^106+10563*x^20*y^106+30173*x^18*y^106+19731*x^16*y^106+26996*x^14*y^106+27567*x^12*y^106+36431*x^10*y^106+12788*x^8*y^106+22520*x^6*y^106+19268*x^4*y^106+16801*x^2*y^106+19680*y^106"
            "+28377*x^24*y^104+27515*x^22*y^104+3363*x^20*y^104+21709*x^18*y^104+37428*x^16*y^104+27150*x^14*y^104+11840*x^12*y^104+36762*x^10*y^104+3851*x^8*y^104+32196*x^6*y^104+3566*x^4*y^104+13208*x^2*y^104"
            "+31688*y^104+11763*x^26*y^102+8962*x^24*y^102+5117*x^22*y^102+29142*x^20*y^102+37047*x^18*y^102+5815*x^16*y^102+12256*x^14*y^102+30365*x^12*y^102+3397*x^10*y^102+8245*x^8*y^102+30644*x^6*y^102+33516*x^4*y^102"
            "+22892*x^2*y^102+10952*y^102+40351*x^28*y^100+21004*x^26*y^100+11376*x^24*y^100+32814*x^22*y^100+10993*x^20*y^100+36285*x^18*y^100+15514*x^16*y^100+27479*x^14*y^100+30369*x^12*y^100+33909*x^10*y^100"
            "+25684*x^8*y^100+33321*x^6*y^100+22410*x^4*y^100+36494*x^2*y^100+1407*y^100+13259*x^30*y^98+20622*x^28*y^98+35020*x^26*y^98+14525*x^24*y^98+13114*x^22*y^98+6844*x^20*y^98+14457*x^18*y^98+25870*x^16*y^98"
            "+28589*x^14*y^98+11970*x^12*y^98+28242*x^10*y^98+16418*x^8*y^98+9241*x^6*y^98+1492*x^4*y^98+42852*x^2*y^98+1278*y^98+30296*x^32*y^96+1649*x^30*y^96+22519*x^28*y^96+16898*x^26*y^96+42979*x^24*y^96"
            "+37879*x^22*y^96+6045*x^20*y^96+31018*x^18*y^96+29954*x^16*y^96+23303*x^14*y^96+15389*x^12*y^96+32528*x^10*y^96+30443*x^8*y^96+17384*x^6*y^96+30548*x^4*y^96+35447*x^2*y^96+13056*y^96+32303*x^34*y^94"
            "+1003*x^32*y^94+42343*x^30*y^94+8261*x^28*y^94+16872*x^26*y^94+4682*x^24*y^94+16011*x^22*y^94+16946*x^20*y^94+40470*x^18*y^94+4840*x^16*y^94+7653*x^14*y^94+37830*x^12*y^94+22459*x^10*y^94+35812*x^8*y^94"
            "+16640*x^6*y^94+40127*x^4*y^94+29449*x^2*y^94+32272*y^94+12442*x^36*y^92+34804*x^34*y^92+30294*x^32*y^92+40042*x^30*y^92+21997*x^28*y^92+20002*x^26*y^92+9041*x^24*y^92+24858*x^22*y^92+40994*x^20*y^92"
            "+18310*x^18*y^92+33251*x^16*y^92+32175*x^14*y^92+17303*x^12*y^92+23104*x^10*y^92+39533*x^8*y^92+35673*x^6*y^92+10646*x^4*y^92+11369*x^2*y^92+19351*y^92+29849*x^38*y^90+29872*x^36*y^90+12595*x^34*y^90"
            "+21119*x^32*y^90+17446*x^30*y^90+30223*x^28*y^90+2124*x^26*y^90+8583*x^24*y^90+23518*x^22*y^90+18802*x^20*y^90+10704*x^18*y^90+3840*x^16*y^90+17866*x^14*y^90+19292*x^12*y^90+17464*x^10*y^90+19972*x^8*y^90"
            "+6638*x^6*y^90+1666*x^4*y^90+23905*x^2*y^90+11254*y^90+31188*x^40*y^88+41133*x^38*y^88+38715*x^36*y^88+466*x^34*y^88+1811*x^32*y^88+12726*x^30*y^88+19856*x^28*y^88+10098*x^26*y^88+37014*x^24*y^88"
            "+26351*x^22*y^88+5729*x^20*y^88+13274*x^18*y^88+21369*x^16*y^88+1954*x^14*y^88+15776*x^12*y^88+13727*x^10*y^88+23609*x^8*y^88+20454*x^6*y^88+29859*x^4*y^88+6904*x^2*y^88+18780*y^88+10504*x^42*y^86"
            "+21316*x^40*y^86+27716*x^38*y^86+5344*x^36*y^86+25693*x^34*y^86+39110*x^32*y^86+37524*x^30*y^86+33089*x^28*y^86+8120*x^26*y^86+29291*x^24*y^86+21153*x^22*y^86+40880*x^20*y^86+42113*x^18*y^86+1379*x^16*y^86"
            "+38326*x^14*y^86+249*x^12*y^86+13076*x^10*y^86+8468*x^8*y^86+5794*x^6*y^86+26565*x^4*y^86+23311*x^2*y^86+41868*y^86+18042*x^44*y^84+37860*x^42*y^84+27934*x^40*y^84+138*x^38*y^84+40687*x^36*y^84"
            "+30639*x^34*y^84+1530*x^32*y^84+36817*x^30*y^84+11154*x^28*y^84+1522*x^26*y^84+12310*x^24*y^84+36728*x^22*y^84+4686*x^20*y^84+22749*x^18*y^84+37462*x^16*y^84+24090*x^14*y^84+342*x^12*y^84+11231*x^10*y^84"
            "+32765*x^8*y^84+22843*x^6*y^84+5696*x^4*y^84+16416*x^2*y^84+13034*y^84+28277*x^46*y^82+18424*x^44*y^82+32406*x^42*y^82+31960*x^40*y^82+32679*x^38*y^82+15441*x^36*y^82+39699*x^34*y^82+22334*x^32*y^82"
            "+24541*x^30*y^82+4030*x^28*y^82+8732*x^26*y^82+31482*x^24*y^82+22112*x^22*y^82+38678*x^20*y^82+14881*x^18*y^82+40905*x^16*y^82+15817*x^14*y^82+3535*x^12*y^82+23416*x^10*y^82+36027*x^8*y^82+30716*x^6*y^82"
            "+10268*x^4*y^82+17023*x^2*y^82+38190*y^82+41480*x^48*y^80+21388*x^46*y^80+30517*x^44*y^80+14292*x^42*y^80+778*x^40*y^80+2747*x^38*y^80+942*x^36*y^80+38419*x^34*y^80+22487*x^32*y^80+27597*x^30*y^80"
            "+38869*x^28*y^80+4742*x^26*y^80+16114*x^24*y^80+22345*x^22*y^80+23143*x^20*y^80+39762*x^18*y^80+27393*x^16*y^80+1302*x^14*y^80+36936*x^12*y^80+40218*x^10*y^80+29048*x^8*y^80+10880*x^6*y^80+20378*x^4*y^80"
            "+6396*x^2*y^80+34930*y^80+36780*x^50*y^78+18665*x^48*y^78+29022*x^46*y^78+27229*x^44*y^78+12380*x^42*y^78+21473*x^40*y^78+33487*x^38*y^78+34058*x^36*y^78+6841*x^34*y^78+33660*x^32*y^78+36919*x^30*y^78"
            "+4924*x^28*y^78+3280*x^26*y^78+30997*x^24*y^78+6502*x^22*y^78+4598*x^20*y^78+19974*x^18*y^78+10898*x^16*y^78+18067*x^14*y^78+3147*x^12*y^78+18282*x^10*y^78+20962*x^8*y^78+37932*x^6*y^78+733*x^4*y^78"
            "+29010*x^2*y^78+23678*y^78+38813*x^52*y^76+32809*x^50*y^76+22404*x^48*y^76+10823*x^46*y^76+10937*x^44*y^76+15078*x^42*y^76+20012*x^40*y^76+13823*x^38*y^76+12928*x^36*y^76+39044*x^34*y^76+35557*x^32*y^76"
            "+9540*x^30*y^76+40791*x^28*y^76+13553*x^26*y^76+20887*x^24*y^76+23010*x^22*y^76+1292*x^20*y^76+27029*x^18*y^76+546*x^16*y^76+39447*x^14*y^76+20237*x^12*y^76+42255*x^10*y^76+11490*x^8*y^76+31310*x^6*y^76"
            "+18248*x^4*y^76+28121*x^2*y^76+7426*y^76+25063*x^54*y^74+37642*x^52*y^74+35028*x^50*y^74+39684*x^48*y^74+16670*x^46*y^74+18269*x^44*y^74+9167*x^42*y^74+28203*x^40*y^74+39475*x^38*y^74+9450*x^36*y^74"
            "+31469*x^34*y^74+14741*x^32*y^74+26864*x^30*y^74+40472*x^28*y^74+31868*x^26*y^74+17150*x^24*y^74+41233*x^22*y^74+32374*x^20*y^74+30579*x^18*y^74+35102*x^16*y^74+24658*x^14*y^74+25742*x^12*y^74+10871*x^10*y^74"
            "+11794*x^8*y^74+36341*x^6*y^74+2908*x^4*y^74+4439*x^2*y^74+13585*y^74+1992*x^56*y^72+7849*x^54*y^72+997*x^52*y^72+34763*x^50*y^72+42481*x^48*y^72+18291*x^46*y^72+12745*x^44*y^72+38278*x^42*y^72"
            "+31068*x^40*y^72+3497*x^38*y^72+10515*x^36*y^72+14907*x^34*y^72+13387*x^32*y^72+5141*x^30*y^72+9788*x^28*y^72+20962*x^26*y^72+33006*x^24*y^72+10499*x^22*y^72+1789*x^20*y^72+41704*x^18*y^72+24746*x^16*y^72"
            "+21765*x^14*y^72+354*x^12*y^72+6648*x^10*y^72+9219*x^8*y^72+731*x^6*y^72+41169*x^4*y^72+41866*x^2*y^72+41317*y^72+35081*x^58*y^70+22602*x^56*y^70+3993*x^54*y^70+8437*x^52*y^70+2179*x^50*y^70+40986*x^48*y^70"
            "+30573*x^46*y^70+23122*x^44*y^70+3597*x^42*y^70+19311*x^40*y^70+41243*x^38*y^70+9330*x^36*y^70+18833*x^34*y^70+32164*x^32*y^70+38779*x^30*y^70+2117*x^28*y^70+10907*x^26*y^70+20600*x^24*y^70+19707*x^22*y^70"
            "+4012*x^20*y^70+27726*x^18*y^70+35841*x^16*y^70+41654*x^14*y^70+40560*x^12*y^70+20228*x^10*y^70+6885*x^8*y^70+8364*x^6*y^70+16635*x^4*y^70+2443*x^2*y^70+40154*y^70+41341*x^60*y^68+10055*x^58*y^68"
            "+43019*x^56*y^68+33525*x^54*y^68+28275*x^52*y^68+21534*x^50*y^68+9440*x^48*y^68+27194*x^46*y^68+15015*x^44*y^68+38756*x^42*y^68+33559*x^40*y^68+39880*x^38*y^68+15511*x^36*y^68+6608*x^34*y^68+18429*x^32*y^68"
            "+3244*x^30*y^68+7080*x^28*y^68+39769*x^26*y^68+5047*x^24*y^68+13145*x^22*y^68+6061*x^20*y^68+16723*x^18*y^68+39231*x^16*y^68+18524*x^14*y^68+15310*x^12*y^68+29206*x^10*y^68+38853*x^8*y^68+28596*x^6*y^68"
            "+4059*x^4*y^68+1787*x^2*y^68+10739*y^68+12276*x^62*y^66+15439*x^60*y^66+31883*x^58*y^66+38997*x^56*y^66+35796*x^54*y^66+37920*x^52*y^66+8824*x^50*y^66+5311*x^48*y^66+36056*x^46*y^66+22397*x^44*y^66"
            "+36343*x^42*y^66+19931*x^40*y^66+12411*x^38*y^66+31114*x^36*y^66+17902*x^34*y^66+36119*x^32*y^66+23123*x^30*y^66+16187*x^28*y^66+15994*x^26*y^66+17114*x^24*y^66+38094*x^22*y^66+23981*x^20*y^66+38490*x^18*y^66"
            "+38679*x^16*y^66+32640*x^14*y^66+35580*x^12*y^66+1569*x^10*y^66+11940*x^8*y^66+33290*x^6*y^66+39873*x^4*y^66+23637*x^2*y^66+22876*y^66+16668*x^64*y^64+24005*x^62*y^64+38824*x^60*y^64+23638*x^58*y^64"
            "+11795*x^56*y^64+27258*x^54*y^64+39022*x^52*y^64+15959*x^50*y^64+38442*x^48*y^64+8701*x^46*y^64+33999*x^44*y^64+24110*x^42*y^64+16403*x^40*y^64+34154*x^38*y^64+42745*x^36*y^64+8702*x^34*y^64+35287*x^32*y^64"
            "+40038*x^30*y^64+13884*x^28*y^64+5412*x^26*y^64+609*x^24*y^64+38179*x^22*y^64+26920*x^20*y^64+36600*x^18*y^64+34683*x^16*y^64+28450*x^14*y^64+19979*x^12*y^64+19627*x^10*y^64+11316*x^8*y^64+29033*x^6*y^64"
            "+16195*x^4*y^64+35450*x^2*y^64+19733*y^64+19698*x^66*y^62+17076*x^64*y^62+1458*x^62*y^62+29213*x^60*y^62+111*x^58*y^62+6799*x^56*y^62+32803*x^54*y^62+21600*x^52*y^62+40301*x^50*y^62+26692*x^48*y^62"
            "+21702*x^46*y^62+38038*x^44*y^62+19712*x^42*y^62+3283*x^40*y^62+5074*x^38*y^62+14424*x^36*y^62+15290*x^34*y^62+42840*x^32*y^62+40294*x^30*y^62+15498*x^28*y^62+29519*x^26*y^62+7507*x^24*y^62+38218*x^22*y^62"
            "+12102*x^20*y^62+36024*x^18*y^62+12743*x^16*y^62+7676*x^14*y^62+27972*x^12*y^62+24770*x^10*y^62+16865*x^8*y^62+28890*x^6*y^62+33356*x^4*y^62+25101*x^2*y^62+3127*y^62+11524*x^68*y^60+17428*x^66*y^60"
            "+23108*x^64*y^60+32140*x^62*y^60+22553*x^60*y^60+36014*x^58*y^60+21300*x^56*y^60+21301*x^54*y^60+28446*x^52*y^60+13376*x^50*y^60+7739*x^48*y^60+1608*x^46*y^60+28303*x^44*y^60+17022*x^42*y^60+11978*x^40*y^60"
            "+22161*x^38*y^60+7069*x^36*y^60+24784*x^34*y^60+25360*x^32*y^60+33946*x^30*y^60+10730*x^28*y^60+11859*x^26*y^60+25411*x^24*y^60+11010*x^22*y^60+40871*x^20*y^60+38352*x^18*y^60+5524*x^16*y^60+14977*x^14*y^60"
            "+6767*x^12*y^60+33794*x^10*y^60+17924*x^8*y^60+37977*x^6*y^60+8160*x^4*y^60+38210*x^2*y^60+33675*y^60+1146*x^70*y^58+31287*x^68*y^58+18656*x^66*y^58+16076*x^64*y^58+30310*x^62*y^58+35030*x^60*y^58"
            "+14815*x^58*y^58+4250*x^56*y^58+16149*x^54*y^58+12594*x^52*y^58+30565*x^50*y^58+36291*x^48*y^58+1153*x^46*y^58+10625*x^44*y^58+18779*x^42*y^58+35948*x^40*y^58+39319*x^38*y^58+1769*x^36*y^58+11200*x^34*y^58"
            "+4209*x^32*y^58+22834*x^30*y^58+17576*x^28*y^58+40331*x^26*y^58+40113*x^24*y^58+30153*x^22*y^58+38508*x^20*y^58+41432*x^18*y^58+16862*x^16*y^58+7476*x^14*y^58+37514*x^12*y^58+30866*x^10*y^58+32336*x^8*y^58"
            "+27613*x^6*y^58+1093*x^4*y^58+25920*x^2*y^58+30109*y^58+21994*x^72*y^56+24413*x^70*y^56+36708*x^68*y^56+12098*x^66*y^56+21715*x^64*y^56+6565*x^62*y^56+5673*x^60*y^56+32603*x^58*y^56+1391*x^56*y^56"
            "+14072*x^54*y^56+30021*x^52*y^56+31475*x^50*y^56+11040*x^48*y^56+32066*x^46*y^56+18795*x^44*y^56+18561*x^42*y^56+17011*x^40*y^56+25549*x^38*y^56+13742*x^36*y^56+4515*x^34*y^56+36431*x^32*y^56+5815*x^30*y^56"
            "+28463*x^28*y^56+22206*x^26*y^56+29618*x^24*y^56+10249*x^22*y^56+8448*x^20*y^56+23731*x^18*y^56+33161*x^16*y^56+30438*x^14*y^56+8011*x^12*y^56+29169*x^10*y^56+16885*x^8*y^56+24192*x^6*y^56+16063*x^4*y^56"
            "+977*x^2*y^56+20180*y^56+35354*x^74*y^54+23221*x^72*y^54+12873*x^70*y^54+8833*x^68*y^54+33898*x^66*y^54+42365*x^64*y^54+42796*x^62*y^54+3965*x^60*y^54+14793*x^58*y^54+35236*x^56*y^54+32225*x^54*y^54"
            "+963*x^52*y^54+30495*x^50*y^54+15778*x^48*y^54+5275*x^46*y^54+39147*x^44*y^54+32615*x^42*y^54+16383*x^40*y^54+15968*x^38*y^54+303*x^36*y^54+18866*x^34*y^54+20892*x^32*y^54+14495*x^30*y^54+29228*x^28*y^54"
            "+8824*x^26*y^54+19746*x^24*y^54+33039*x^22*y^54+3318*x^20*y^54+17018*x^18*y^54+25286*x^16*y^54+27107*x^14*y^54+6231*x^12*y^54+28939*x^10*y^54+25698*x^8*y^54+9950*x^6*y^54+37764*x^4*y^54+37499*x^2*y^54"
            "+13924*y^54+24936*x^76*y^52+1784*x^74*y^52+4777*x^72*y^52+25581*x^70*y^52+32388*x^68*y^52+36282*x^66*y^52+14679*x^64*y^52+30875*x^62*y^52+23647*x^60*y^52+15900*x^58*y^52+13580*x^56*y^52+42102*x^54*y^52"
            "+27416*x^52*y^52+16418*x^50*y^52+3419*x^48*y^52+26654*x^46*y^52+14803*x^44*y^52+14108*x^42*y^52+27811*x^40*y^52+8305*x^38*y^52+24852*x^36*y^52+39522*x^34*y^52+22762*x^32*y^52+3736*x^30*y^52+13000*x^28*y^52"
            "+17297*x^26*y^52+35582*x^24*y^52+33936*x^22*y^52+25796*x^20*y^52+24536*x^18*y^52+30665*x^16*y^52+23548*x^14*y^52+24350*x^12*y^52+30376*x^10*y^52+1379*x^8*y^52+29308*x^6*y^52+12619*x^4*y^52+43019*x^2*y^52"
            "+4045*y^52+40821*x^78*y^50+39579*x^76*y^50+39386*x^74*y^50+21109*x^72*y^50+20753*x^70*y^50+34835*x^68*y^50+3982*x^66*y^50+2435*x^64*y^50+9859*x^62*y^50+15672*x^60*y^50+8066*x^58*y^50+32097*x^56*y^50"
            "+21056*x^54*y^50+22971*x^52*y^50+5053*x^50*y^50+7692*x^48*y^50+40862*x^46*y^50+34315*x^44*y^50+31567*x^42*y^50+2656*x^40*y^50+41426*x^38*y^50+38114*x^36*y^50+11216*x^34*y^50+37186*x^32*y^50+25571*x^30*y^50"
            "+15277*x^28*y^50+29074*x^26*y^50+4122*x^24*y^50+9793*x^22*y^50+3765*x^20*y^50+32150*x^18*y^50+37040*x^16*y^50+37117*x^14*y^50+25423*x^12*y^50+42151*x^10*y^50+20310*x^8*y^50+11359*x^6*y^50+5780*x^4*y^50"
            "+17431*x^2*y^50+37752*y^50+1261*x^80*y^48+29556*x^78*y^48+18109*x^76*y^48+24457*x^74*y^48+29308*x^72*y^48+24596*x^70*y^48+24759*x^68*y^48+34954*x^66*y^48+4323*x^64*y^48+5671*x^62*y^48+23949*x^60*y^48"
            "+9231*x^58*y^48+9259*x^56*y^48+21135*x^54*y^48+461*x^52*y^48+26234*x^50*y^48+22548*x^48*y^48+11931*x^46*y^48+27419*x^44*y^48+9235*x^42*y^48+31268*x^40*y^48+35134*x^38*y^48+42462*x^36*y^48+30197*x^34*y^48"
            "+40139*x^32*y^48+30148*x^30*y^48+25205*x^28*y^48+18621*x^26*y^48+3359*x^24*y^48+18374*x^22*y^48+13953*x^20*y^48+6730*x^18*y^48+6538*x^16*y^48+19375*x^14*y^48+38459*x^12*y^48+19761*x^10*y^48+41692*x^8*y^48"
            "+41909*x^6*y^48+16383*x^4*y^48+42286*x^2*y^48+30118*y^48+14689*x^82*y^46+41191*x^80*y^46+21181*x^78*y^46+9200*x^76*y^46+7039*x^74*y^46+37585*x^72*y^46+38206*x^70*y^46+19158*x^68*y^46+25276*x^66*y^46"
            "+24661*x^64*y^46+41619*x^62*y^46+37541*x^60*y^46+1433*x^58*y^46+5648*x^56*y^46+21510*x^54*y^46+11039*x^52*y^46+12901*x^50*y^46+3458*x^48*y^46+40390*x^46*y^46+15314*x^44*y^46+8428*x^42*y^46+33075*x^40*y^46"
            "+36521*x^38*y^46+16431*x^36*y^46+38665*x^34*y^46+33507*x^32*y^46+42213*x^30*y^46+1737*x^28*y^46+6710*x^26*y^46+30716*x^24*y^46+27264*x^22*y^46+39638*x^20*y^46+22602*x^18*y^46+10239*x^16*y^46+3066*x^14*y^46"
            "+40086*x^12*y^46+37827*x^10*y^46+38184*x^8*y^46+203*x^6*y^46+15494*x^4*y^46+26977*x^2*y^46+1615*y^46+28351*x^84*y^44+27358*x^82*y^44+23857*x^80*y^44+433*x^78*y^44+21917*x^76*y^44+5638*x^74*y^44"
            "+35642*x^72*y^44+35617*x^70*y^44+37727*x^68*y^44+7124*x^66*y^44+5920*x^64*y^44+16726*x^62*y^44+34126*x^60*y^44+8801*x^58*y^44+25912*x^56*y^44+21638*x^54*y^44+17409*x^52*y^44+29730*x^50*y^44+22218*x^48*y^44"
            "+39493*x^46*y^44+36436*x^44*y^44+17151*x^42*y^44+24069*x^40*y^44+14182*x^38*y^44+22962*x^36*y^44+24942*x^34*y^44+39864*x^32*y^44+1907*x^30*y^44+16335*x^28*y^44+9842*x^26*y^44+24840*x^24*y^44+9677*x^22*y^44"
            "+7641*x^20*y^44+5261*x^18*y^44+15301*x^16*y^44+1255*x^14*y^44+15813*x^12*y^44+3359*x^10*y^44+29771*x^8*y^44+38196*x^6*y^44+26499*x^4*y^44+42471*x^2*y^44+4510*y^44+13720*x^86*y^42+35086*x^84*y^42"
            "+39746*x^82*y^42+24554*x^80*y^42+13497*x^78*y^42+21512*x^76*y^42+28496*x^74*y^42+33195*x^72*y^42+24374*x^70*y^42+17225*x^68*y^42+2548*x^66*y^42+30586*x^64*y^42+14558*x^62*y^42+35653*x^60*y^42+34927*x^58*y^42"
            "+26802*x^56*y^42+1767*x^54*y^42+6722*x^52*y^42+40763*x^50*y^42+8138*x^48*y^42+13443*x^46*y^42+3411*x^44*y^42+14630*x^42*y^42+24003*x^40*y^42+30536*x^38*y^42+14290*x^36*y^42+2599*x^34*y^42+17099*x^32*y^42"
            "+7740*x^30*y^42+34544*x^28*y^42+34232*x^26*y^42+1799*x^24*y^42+3607*x^22*y^42+34281*x^20*y^42+5536*x^18*y^42+23289*x^16*y^42+12791*x^14*y^42+17851*x^12*y^42+9546*x^10*y^42+34485*x^8*y^42+7704*x^6*y^42"
            "+32670*x^4*y^42+42637*x^2*y^42+16577*y^42+28643*x^88*y^40+4201*x^86*y^40+21750*x^84*y^40+41624*x^82*y^40+249*x^80*y^40+16557*x^78*y^40+3494*x^76*y^40+11840*x^74*y^40+16961*x^72*y^40+15768*x^70*y^40"
            "+22448*x^68*y^40+34243*x^66*y^40+29527*x^64*y^40+38081*x^62*y^40+30103*x^60*y^40+12648*x^58*y^40+3076*x^56*y^40+40617*x^54*y^40+12652*x^52*y^40+42343*x^50*y^40+29129*x^48*y^40+40146*x^46*y^40+28836*x^44*y^40"
            "+31408*x^42*y^40+9713*x^40*y^40+8423*x^38*y^40+12347*x^36*y^40+28470*x^34*y^40+24078*x^32*y^40+33694*x^30*y^40+26159*x^28*y^40+318*x^26*y^40+7841*x^24*y^40+25575*x^22*y^40+25188*x^20*y^40+18176*x^18*y^40"
            "+22443*x^16*y^40+24241*x^14*y^40+37129*x^12*y^40+25614*x^10*y^40+11760*x^8*y^40+10320*x^6*y^40+29531*x^4*y^40+16109*x^2*y^40+7536*y^40+2062*x^90*y^38+26752*x^88*y^38+29883*x^86*y^38+29763*x^84*y^38"
            "+9843*x^82*y^38+9784*x^80*y^38+3344*x^78*y^38+30316*x^76*y^38+8367*x^74*y^38+5696*x^72*y^38+1180*x^70*y^38+29257*x^68*y^38+33396*x^66*y^38+9929*x^64*y^38+39519*x^62*y^38+3398*x^60*y^38+17794*x^58*y^38"
            "+1568*x^56*y^38+12037*x^54*y^38+24340*x^52*y^38+42509*x^50*y^38+37396*x^48*y^38+33850*x^46*y^38+12758*x^44*y^38+15386*x^42*y^38+17333*x^40*y^38+5474*x^38*y^38+17425*x^36*y^38+39133*x^34*y^38+13217*x^32*y^38"
            "+29242*x^30*y^38+1170*x^28*y^38+7817*x^26*y^38+3141*x^24*y^38+18776*x^22*y^38+6245*x^20*y^38+12507*x^18*y^38+35604*x^16*y^38+20990*x^14*y^38+3496*x^12*y^38+31995*x^10*y^38+7575*x^8*y^38+30688*x^6*y^38"
            "+11272*x^4*y^38+1806*x^2*y^38+34837*y^38+4301*x^92*y^36+16466*x^90*y^36+27117*x^88*y^36+39011*x^86*y^36+23105*x^84*y^36+37032*x^82*y^36+30001*x^80*y^36+31070*x^78*y^36+27578*x^76*y^36+14337*x^74*y^36"
            "+10891*x^72*y^36+18543*x^70*y^36+3050*x^68*y^36+1802*x^66*y^36+40914*x^64*y^36+23184*x^62*y^36+21058*x^60*y^36+4717*x^58*y^36+14097*x^56*y^36+7724*x^54*y^36+26213*x^52*y^36+35211*x^50*y^36+41999*x^48*y^36"
            "+11769*x^46*y^36+31308*x^44*y^36+26901*x^42*y^36+19700*x^40*y^36+31565*x^38*y^36+11366*x^36*y^36+7643*x^34*y^36+26178*x^32*y^36+37577*x^30*y^36+37992*x^28*y^36+4536*x^26*y^36+634*x^24*y^36+37784*x^22*y^36"
            "+7963*x^20*y^36+12327*x^18*y^36+15837*x^16*y^36+21084*x^14*y^36+33837*x^12*y^36+21234*x^10*y^36+35492*x^8*y^36+35354*x^6*y^36+6931*x^4*y^36+21331*x^2*y^36+241*y^36+23258*x^94*y^34+24353*x^92*y^34"
            "+33854*x^90*y^34+19904*x^88*y^34+16134*x^86*y^34+5133*x^84*y^34+34955*x^82*y^34+5892*x^80*y^34+7191*x^78*y^34+16055*x^76*y^34+41239*x^74*y^34+23260*x^72*y^34+31526*x^70*y^34+26991*x^68*y^34+20924*x^66*y^34"
            "+10823*x^64*y^34+7717*x^62*y^34+4178*x^60*y^34+22750*x^58*y^34+6182*x^56*y^34+13452*x^54*y^34+18368*x^52*y^34+41383*x^50*y^34+22450*x^48*y^34+16720*x^46*y^34+34208*x^44*y^34+19188*x^42*y^34+24090*x^40*y^34"
            "+4711*x^38*y^34+27289*x^36*y^34+41850*x^34*y^34+27115*x^32*y^34+34653*x^30*y^34+23194*x^28*y^34+32775*x^26*y^34+25616*x^24*y^34+6192*x^22*y^34+10665*x^20*y^34+5200*x^18*y^34+5135*x^16*y^34+22010*x^14*y^34"
            "+5447*x^12*y^34+33334*x^10*y^34+28155*x^8*y^34+6957*x^6*y^34+41099*x^4*y^34+43042*x^2*y^34+9705*y^34+28554*x^96*y^32+14891*x^94*y^32+38244*x^92*y^32+18164*x^90*y^32+31532*x^88*y^32+2827*x^86*y^32"
            "+25378*x^84*y^32+41680*x^82*y^32+3102*x^80*y^32+34074*x^78*y^32+17444*x^76*y^32+31896*x^74*y^32+21439*x^72*y^32+18439*x^70*y^32+42260*x^68*y^32+22330*x^66*y^32+28152*x^64*y^32+36359*x^62*y^32+36146*x^60*y^32"
            "+39069*x^58*y^32+42121*x^56*y^32+17124*x^54*y^32+15457*x^52*y^32+4838*x^50*y^32+35776*x^48*y^32+38288*x^46*y^32+12621*x^44*y^32+35660*x^42*y^32+37365*x^40*y^32+43038*x^38*y^32+5298*x^36*y^32+37617*x^34*y^32"
            "+5264*x^32*y^32+13100*x^30*y^32+22400*x^28*y^32+13246*x^26*y^32+32660*x^24*y^32+27424*x^22*y^32+19712*x^20*y^32+3951*x^18*y^32+33046*x^16*y^32+37408*x^14*y^32+6383*x^12*y^32+25114*x^10*y^32+23719*x^8*y^32"
            "+20292*x^6*y^32+1210*x^4*y^32+21708*x^2*y^32+2196*y^32+41751*x^98*y^30+3506*x^96*y^30+22624*x^94*y^30+40102*x^92*y^30+41008*x^90*y^30+36366*x^88*y^30+34105*x^86*y^30+7204*x^84*y^30+38649*x^82*y^30"
            "+21979*x^80*y^30+11389*x^78*y^30+30974*x^76*y^30+16415*x^74*y^30+15920*x^72*y^30+5449*x^70*y^30+28008*x^68*y^30+37722*x^66*y^30+33222*x^64*y^30+13753*x^62*y^30+7207*x^60*y^30+32550*x^58*y^30+33024*x^56*y^30"
            "+10586*x^54*y^30+42555*x^52*y^30+17793*x^50*y^30+16663*x^48*y^30+31694*x^46*y^30+1299*x^44*y^30+18112*x^42*y^30+2711*x^40*y^30+38258*x^38*y^30+28347*x^36*y^30+34835*x^34*y^30+20163*x^32*y^30+22401*x^30*y^30"
            "+33934*x^28*y^30+23924*x^26*y^30+41753*x^24*y^30+31379*x^22*y^30+21808*x^20*y^30+1139*x^18*y^30+11347*x^16*y^30+32849*x^14*y^30+9474*x^12*y^30+38854*x^10*y^30+18253*x^8*y^30+38218*x^6*y^30+8810*x^4*y^30"
            "+2999*x^2*y^30+33198*y^30+20087*x^100*y^28+13657*x^98*y^28+14744*x^96*y^28+19054*x^94*y^28+10779*x^92*y^28+8560*x^90*y^28+17029*x^88*y^28+18083*x^86*y^28+599*x^84*y^28+873*x^82*y^28+12432*x^80*y^28"
            "+34808*x^78*y^28+29826*x^76*y^28+4437*x^74*y^28+11650*x^72*y^28+31649*x^70*y^28+19951*x^68*y^28+27253*x^66*y^28+33030*x^64*y^28+15058*x^62*y^28+12375*x^60*y^28+12566*x^58*y^28+31955*x^56*y^28+12990*x^54*y^28"
            "+13509*x^52*y^28+16851*x^50*y^28+30596*x^48*y^28+6762*x^46*y^28+39496*x^44*y^28+9989*x^42*y^28+29136*x^40*y^28+42573*x^38*y^28+23967*x^36*y^28+36766*x^34*y^28+21928*x^32*y^28+16864*x^30*y^28+33699*x^28*y^28"
            "+7712*x^26*y^28+23043*x^24*y^28+42991*x^22*y^28+36183*x^20*y^28+28799*x^18*y^28+16223*x^16*y^28+35358*x^14*y^28+24724*x^12*y^28+389*x^10*y^28+3509*x^8*y^28+24611*x^6*y^28+27165*x^4*y^28+21064*x^2*y^28"
            "+21393*y^28+39670*x^102*y^26+42143*x^100*y^26+38882*x^98*y^26+12003*x^96*y^26+24085*x^94*y^26+2406*x^92*y^26+6013*x^90*y^26+12013*x^88*y^26+1128*x^86*y^26+10893*x^84*y^26+19101*x^82*y^26+5754*x^80*y^26"
            "+40597*x^78*y^26+20885*x^76*y^26+37810*x^74*y^26+17637*x^72*y^26+6921*x^70*y^26+10241*x^68*y^26+20318*x^66*y^26+21067*x^64*y^26+17577*x^62*y^26+2010*x^60*y^26+28916*x^58*y^26+6931*x^56*y^26+14528*x^54*y^26"
            "+5437*x^52*y^26+2502*x^50*y^26+5572*x^48*y^26+20344*x^46*y^26+35319*x^44*y^26+42510*x^42*y^26+16831*x^40*y^26+8200*x^38*y^26+20786*x^36*y^26+27850*x^34*y^26+9182*x^32*y^26+17561*x^30*y^26+36330*x^28*y^26"
            "+556*x^26*y^26+42804*x^24*y^26+35781*x^22*y^26+34582*x^20*y^26+42298*x^18*y^26+7607*x^16*y^26+23592*x^14*y^26+6481*x^12*y^26+20600*x^10*y^26+20494*x^8*y^26+20466*x^6*y^26+37630*x^4*y^26+32594*x^2*y^26"
            "+32583*y^26+34806*x^104*y^24+7080*x^102*y^24+35489*x^100*y^24+11472*x^98*y^24+5792*x^96*y^24+1173*x^94*y^24+29521*x^92*y^24+28230*x^90*y^24+392*x^88*y^24+42071*x^86*y^24+20630*x^84*y^24+5510*x^82*y^24"
            "+35541*x^80*y^24+18785*x^78*y^24+6088*x^76*y^24+21840*x^74*y^24+14778*x^72*y^24+40896*x^70*y^24+4701*x^68*y^24+6915*x^66*y^24+29242*x^64*y^24+7552*x^62*y^24+1614*x^60*y^24+3222*x^58*y^24+7828*x^56*y^24"
            "+40503*x^54*y^24+432*x^52*y^24+27544*x^50*y^24+11292*x^48*y^24+39827*x^46*y^24+16066*x^44*y^24+4515*x^42*y^24+10347*x^40*y^24+11618*x^38*y^24+1208*x^36*y^24+30279*x^34*y^24+7375*x^32*y^24+35158*x^30*y^24"
            "+38636*x^28*y^24+16362*x^26*y^24+34500*x^24*y^24+35142*x^22*y^24+24967*x^20*y^24+15858*x^18*y^24+12076*x^16*y^24+15623*x^14*y^24+15411*x^12*y^24+22432*x^10*y^24+23353*x^8*y^24+15209*x^6*y^24+7329*x^4*y^24"
            "+7587*x^2*y^24+33148*y^24+3379*x^106*y^22+27532*x^104*y^22+38717*x^102*y^22+3622*x^100*y^22+15202*x^98*y^22+5472*x^96*y^22+1725*x^94*y^22+35520*x^92*y^22+30742*x^90*y^22+28586*x^88*y^22+8728*x^86*y^22"
            "+38664*x^84*y^22+42854*x^82*y^22+6519*x^80*y^22+29152*x^78*y^22+39051*x^76*y^22+30056*x^74*y^22+22650*x^72*y^22+17814*x^70*y^22+42846*x^68*y^22+29259*x^66*y^22+24720*x^64*y^22+40018*x^62*y^22+6990*x^60*y^22"
            "+18972*x^58*y^22+15697*x^56*y^22+17442*x^54*y^22+25772*x^52*y^22+25660*x^50*y^22+22640*x^48*y^22+7122*x^46*y^22+1968*x^44*y^22+16339*x^42*y^22+22176*x^40*y^22+18823*x^38*y^22+19388*x^36*y^22+38347*x^34*y^22"
            "+36810*x^32*y^22+30240*x^30*y^22+42919*x^28*y^22+10096*x^26*y^22+28351*x^24*y^22+27961*x^22*y^22+262*x^20*y^22+18482*x^18*y^22+15038*x^16*y^22+38007*x^14*y^22+13093*x^12*y^22+15923*x^10*y^22+15905*x^8*y^22"
            "+19270*x^6*y^22+31246*x^4*y^22+2077*x^2*y^22+12208*y^22+15389*x^108*y^20+9097*x^106*y^20+15481*x^104*y^20+22559*x^102*y^20+12450*x^100*y^20+22179*x^98*y^20+9862*x^96*y^20+38084*x^94*y^20+24174*x^92*y^20"
            "+766*x^90*y^20+10892*x^88*y^20+32551*x^86*y^20+3311*x^84*y^20+29836*x^82*y^20+27304*x^80*y^20+13545*x^78*y^20+42106*x^76*y^20+20859*x^74*y^20+40164*x^72*y^20+38174*x^70*y^20+32709*x^68*y^20+40765*x^66*y^20"
            "+33707*x^64*y^20+18031*x^62*y^20+39480*x^60*y^20+16638*x^58*y^20+40832*x^56*y^20+30693*x^54*y^20+26984*x^52*y^20+30680*x^50*y^20+3895*x^48*y^20+24426*x^46*y^20+17711*x^44*y^20+24628*x^42*y^20+29371*x^40*y^20"
            "+10215*x^38*y^20+1056*x^36*y^20+31850*x^34*y^20+30855*x^32*y^20+1402*x^30*y^20+175*x^28*y^20+7565*x^26*y^20+26411*x^24*y^20+40257*x^22*y^20+42403*x^20*y^20+42595*x^18*y^20+37709*x^16*y^20+11885*x^14*y^20"
            "+40731*x^12*y^20+30038*x^10*y^20+24318*x^8*y^20+27721*x^6*y^20+17787*x^4*y^20+27117*x^2*y^20+31138*y^20+8742*x^110*y^18+28996*x^108*y^18+8875*x^106*y^18+42694*x^104*y^18+17582*x^102*y^18+11884*x^100*y^18"
            "+26651*x^98*y^18+16472*x^96*y^18+9415*x^94*y^18+31341*x^92*y^18+4589*x^90*y^18+27171*x^88*y^18+6477*x^86*y^18+39756*x^84*y^18+5166*x^82*y^18+15388*x^80*y^18+15043*x^78*y^18+633*x^76*y^18+15006*x^74*y^18"
            "+13178*x^72*y^18+4280*x^70*y^18+24142*x^68*y^18+40850*x^66*y^18+21129*x^64*y^18+1277*x^62*y^18+5586*x^60*y^18+10451*x^58*y^18+18829*x^56*y^18+29523*x^54*y^18+26485*x^52*y^18+2403*x^50*y^18+23764*x^48*y^18"
            "+22137*x^46*y^18+24536*x^44*y^18+41504*x^42*y^18+36081*x^40*y^18+36187*x^38*y^18+21187*x^36*y^18+36012*x^34*y^18+27868*x^32*y^18+13055*x^30*y^18+23148*x^28*y^18+4982*x^26*y^18+14351*x^24*y^18+21985*x^22*y^18"
            "+28224*x^20*y^18+41085*x^18*y^18+41144*x^16*y^18+21078*x^14*y^18+6468*x^12*y^18+29312*x^10*y^18+20535*x^8*y^18+6845*x^6*y^18+33572*x^4*y^18+21429*x^2*y^18+15252*y^18+7189*x^112*y^16+17449*x^110*y^16"
            "+33417*x^108*y^16+23815*x^106*y^16+9716*x^104*y^16+10978*x^102*y^16+3686*x^100*y^16+18181*x^98*y^16+23916*x^96*y^16+27090*x^94*y^16+31252*x^92*y^16+4721*x^90*y^16+751*x^88*y^16+35594*x^86*y^16+29977*x^84*y^16"
            "+35976*x^82*y^16+18735*x^80*y^16+35323*x^78*y^16+40532*x^76*y^16+12615*x^74*y^16+31708*x^72*y^16+10545*x^70*y^16+17844*x^68*y^16+6350*x^66*y^16+6904*x^64*y^16+19526*x^62*y^16+24150*x^60*y^16+33459*x^58*y^16"
            "+22533*x^56*y^16+5902*x^54*y^16+12322*x^52*y^16+2899*x^50*y^16+5658*x^48*y^16+20010*x^46*y^16+24805*x^44*y^16+42523*x^42*y^16+5113*x^40*y^16+42142*x^38*y^16+10954*x^36*y^16+1895*x^34*y^16+17233*x^32*y^16"
            "+18530*x^30*y^16+18512*x^28*y^16+9802*x^26*y^16+40745*x^24*y^16+42941*x^22*y^16+10950*x^20*y^16+36609*x^18*y^16+9187*x^16*y^16+41118*x^14*y^16+40848*x^12*y^16+14323*x^10*y^16+20057*x^8*y^16+29018*x^6*y^16"
            "+16678*x^4*y^16+24873*x^2*y^16+5157*y^16+12066*x^114*y^14+38835*x^112*y^14+17848*x^110*y^14+9037*x^108*y^14+26019*x^106*y^14+1534*x^104*y^14+17609*x^102*y^14+8795*x^100*y^14+37427*x^98*y^14+37473*x^96*y^14"
            "+20682*x^94*y^14+36897*x^92*y^14+23669*x^90*y^14+23150*x^88*y^14+3430*x^86*y^14+12988*x^84*y^14+41226*x^82*y^14+9247*x^80*y^14+6749*x^78*y^14+9605*x^76*y^14+7097*x^74*y^14+8834*x^72*y^14+14409*x^70*y^14"
            "+33978*x^68*y^14+14602*x^66*y^14+25010*x^64*y^14+14797*x^62*y^14+20509*x^60*y^14+20790*x^58*y^14+40039*x^56*y^14+41843*x^54*y^14+16373*x^52*y^14+3400*x^50*y^14+1770*x^48*y^14+6158*x^46*y^14+563*x^44*y^14"
            "+18954*x^42*y^14+32126*x^40*y^14+37681*x^38*y^14+5538*x^36*y^14+17845*x^34*y^14+6041*x^32*y^14+1017*x^30*y^14+5208*x^28*y^14+25388*x^26*y^14+19739*x^24*y^14+9119*x^22*y^14+29651*x^20*y^14+30801*x^18*y^14"
            "+32064*x^16*y^14+19656*x^14*y^14+21875*x^12*y^14+34932*x^10*y^14+15786*x^8*y^14+23952*x^6*y^14+38212*x^4*y^14+40822*x^2*y^14+15920*y^14+33741*x^116*y^12+35681*x^114*y^12+2669*x^112*y^12+11408*x^110*y^12"
            "+37403*x^108*y^12+12197*x^106*y^12+3364*x^104*y^12+23514*x^102*y^12+16044*x^100*y^12+39266*x^98*y^12+1570*x^96*y^12+35221*x^94*y^12+38817*x^92*y^12+42632*x^90*y^12+8943*x^88*y^12+34152*x^86*y^12"
            "+32133*x^84*y^12+15451*x^82*y^12+38058*x^80*y^12+10723*x^78*y^12+12151*x^76*y^12+39697*x^74*y^12+2983*x^72*y^12+9825*x^70*y^12+24093*x^68*y^12+16563*x^66*y^12+18601*x^64*y^12+17118*x^62*y^12+23325*x^60*y^12"
            "+25238*x^58*y^12+20856*x^56*y^12+27627*x^54*y^12+27311*x^52*y^12+11532*x^50*y^12+14594*x^48*y^12+42761*x^46*y^12+22542*x^44*y^12+18903*x^42*y^12+2028*x^40*y^12+42931*x^38*y^12+28512*x^36*y^12+23484*x^34*y^12"
            "+35985*x^32*y^12+7572*x^30*y^12+39725*x^28*y^12+25321*x^26*y^12+28593*x^24*y^12+15735*x^22*y^12+23728*x^20*y^12+42662*x^18*y^12+8593*x^16*y^12+12726*x^14*y^12+36115*x^10*y^12+38197*x^8*y^12+27870*x^6*y^12"
            "+30056*x^4*y^12+22856*x^2*y^12+761*y^12+38458*x^118*y^10+35378*x^116*y^10+38852*x^114*y^10+13844*x^112*y^10+15204*x^110*y^10+14179*x^108*y^10+33524*x^106*y^10+26521*x^104*y^10+7142*x^102*y^10+18886*x^100*y^10"
            "+41153*x^98*y^10+31816*x^96*y^10+27233*x^94*y^10+11280*x^92*y^10+38975*x^90*y^10+16890*x^88*y^10+34763*x^86*y^10+7480*x^84*y^10+25699*x^82*y^10+30223*x^80*y^10+6215*x^78*y^10+19166*x^76*y^10+15179*x^74*y^10"
            "+39918*x^72*y^10+11439*x^70*y^10+7345*x^68*y^10+28333*x^66*y^10+15843*x^64*y^10+22507*x^62*y^10+37591*x^60*y^10+21942*x^58*y^10+7369*x^56*y^10+16147*x^54*y^10+1147*x^52*y^10+22755*x^50*y^10+35696*x^48*y^10"
            "+34882*x^46*y^10+6718*x^44*y^10+36965*x^42*y^10+30741*x^40*y^10+38124*x^38*y^10+13434*x^36*y^10+40104*x^34*y^10+23924*x^32*y^10+34197*x^30*y^10+18623*x^28*y^10+8307*x^26*y^10+37098*x^24*y^10+24902*x^22*y^10"
            "+17933*x^20*y^10+33381*x^18*y^10+10499*x^16*y^10+33941*x^14*y^10+1108*x^12*y^10+5416*x^10*y^10+21260*x^8*y^10+40985*x^6*y^10+35986*x^4*y^10+39615*x^2*y^10+5651*y^10+30041*x^120*y^8+20323*x^118*y^8"
            "+13296*x^116*y^8+32818*x^114*y^8+27827*x^112*y^8+38475*x^110*y^8+22500*x^108*y^8+26645*x^106*y^8+8018*x^104*y^8+18372*x^102*y^8+22394*x^100*y^8+589*x^98*y^8+1632*x^96*y^8+21631*x^94*y^8+11765*x^92*y^8"
            "+4290*x^90*y^8+13828*x^88*y^8+23526*x^86*y^8+38030*x^84*y^8+36727*x^82*y^8+23680*x^80*y^8+13219*x^78*y^8+32793*x^76*y^8+9091*x^74*y^8+34849*x^72*y^8+696*x^70*y^8+24056*x^68*y^8+37615*x^66*y^8+4486*x^64*y^8"
            "+32407*x^62*y^8+18376*x^60*y^8+25742*x^58*y^8+38738*x^56*y^8+13582*x^54*y^8+7118*x^52*y^8+9158*x^50*y^8+28740*x^48*y^8+21773*x^46*y^8+16265*x^44*y^8+9151*x^42*y^8+33024*x^40*y^8+17536*x^38*y^8+31202*x^36*y^8"
            "+13404*x^34*y^8+16483*x^32*y^8+12809*x^30*y^8+5052*x^28*y^8+27764*x^26*y^8+20578*x^24*y^8+21706*x^22*y^8+3664*x^20*y^8+14186*x^18*y^8+7764*x^16*y^8+42913*x^14*y^8+16118*x^12*y^8+41426*x^10*y^8+41792*x^8*y^8"
            "+2602*x^6*y^8+12193*x^4*y^8+32121*x^2*y^8+42802*y^8+26172*x^122*y^6+14420*x^120*y^6+8330*x^118*y^6+33509*x^116*y^6+34141*x^114*y^6+4543*x^112*y^6+22477*x^110*y^6+30910*x^108*y^6+12525*x^106*y^6"
            "+21019*x^104*y^6+24156*x^102*y^6+29781*x^100*y^6+41441*x^98*y^6+18727*x^96*y^6+36526*x^94*y^6+11426*x^92*y^6+39169*x^90*y^6+13590*x^88*y^6+36433*x^86*y^6+24896*x^84*y^6+41747*x^82*y^6+10792*x^80*y^6"
            "+2403*x^78*y^6+26505*x^76*y^6+24597*x^74*y^6+23260*x^72*y^6+9148*x^70*y^6+1137*x^68*y^6+34203*x^66*y^6+30374*x^64*y^6+4565*x^62*y^6+135*x^60*y^6+38731*x^58*y^6+16962*x^56*y^6+27791*x^54*y^6+24230*x^52*y^6"
            "+34876*x^50*y^6+39807*x^48*y^6+4728*x^46*y^6+2024*x^44*y^6+38003*x^42*y^6+10398*x^40*y^6+35925*x^38*y^6+36133*x^36*y^6+40238*x^34*y^6+6214*x^32*y^6+9511*x^30*y^6+5972*x^28*y^6+24583*x^26*y^6+31054*x^24*y^6"
            "+25112*x^22*y^6+26875*x^20*y^6+2973*x^18*y^6+36493*x^16*y^6+27271*x^14*y^6+34812*x^12*y^6+33099*x^10*y^6+6679*x^8*y^6+20584*x^6*y^6+9064*x^4*y^6+8136*x^2*y^6+14396*y^6+22127*x^124*y^4+20019*x^122*y^4"
            "+18592*x^120*y^4+30874*x^118*y^4+15132*x^116*y^4+29281*x^114*y^4+22980*x^112*y^4+10864*x^110*y^4+6007*x^108*y^4+4936*x^106*y^4+23939*x^104*y^4+31836*x^102*y^4+12108*x^100*y^4+11572*x^98*y^4+42747*x^96*y^4"
            "+30049*x^94*y^4+17193*x^92*y^4+42415*x^90*y^4+30307*x^88*y^4+40790*x^86*y^4+23322*x^84*y^4+15122*x^82*y^4+2037*x^80*y^4+27159*x^78*y^4+17150*x^76*y^4+33813*x^74*y^4+40205*x^72*y^4+9556*x^70*y^4+24621*x^68*y^4"
            "+15083*x^66*y^4+22083*x^64*y^4+3746*x^62*y^4+38976*x^60*y^4+6279*x^58*y^4+6555*x^56*y^4+27196*x^54*y^4+29138*x^52*y^4+18182*x^50*y^4+21201*x^48*y^4+23276*x^46*y^4+24659*x^44*y^4+40263*x^42*y^4+27455*x^40*y^4"
            "+8998*x^38*y^4+10807*x^36*y^4+6260*x^34*y^4+911*x^32*y^4+11985*x^30*y^4+31164*x^28*y^4+27578*x^26*y^4+37188*x^24*y^4+6725*x^22*y^4+34708*x^20*y^4+14058*x^18*y^4+17176*x^16*y^4+25971*x^14*y^4+9296*x^12*y^4"
            "+12083*x^10*y^4+42019*x^8*y^4+32011*x^6*y^4+24665*x^4*y^4+40332*x^2*y^4+34190*y^4+9940*x^126*y^2+7922*x^124*y^2+710*x^122*y^2+28677*x^120*y^2+29200*x^118*y^2+16915*x^116*y^2+17106*x^114*y^2+14802*x^112*y^2"
            "+7567*x^110*y^2+3533*x^108*y^2+16621*x^106*y^2+37877*x^104*y^2+7956*x^102*y^2+16344*x^100*y^2+36978*x^98*y^2+39545*x^96*y^2+37434*x^94*y^2+5249*x^92*y^2+39842*x^90*y^2+23580*x^88*y^2+40902*x^86*y^2"
            "+527*x^84*y^2+33256*x^82*y^2+31430*x^80*y^2+6739*x^78*y^2+42179*x^76*y^2+16431*x^74*y^2+6976*x^72*y^2+37894*x^70*y^2+20000*x^68*y^2+29973*x^66*y^2+31695*x^64*y^2+41973*x^62*y^2+2479*x^60*y^2+33448*x^58*y^2"
            "+6155*x^56*y^2+24489*x^54*y^2+7764*x^52*y^2+6935*x^50*y^2+32788*x^48*y^2+13858*x^46*y^2+19696*x^44*y^2+24391*x^42*y^2+34902*x^40*y^2+27564*x^38*y^2+24627*x^36*y^2+41873*x^34*y^2+39978*x^32*y^2+31220*x^30*y^2"
            "+16078*x^28*y^2+21206*x^26*y^2+4924*x^24*y^2+24107*x^22*y^2+2642*x^20*y^2+9942*x^18*y^2+32635*x^16*y^2+18223*x^14*y^2+21946*x^12*y^2+17452*x^10*y^2+22441*x^8*y^2+38749*x^6*y^2+400*x^4*y^2+12301*x^2*y^2"
            "+31143*y^2+13269*x^128+38886*x^126+25434*x^124+39701*x^122+1803*x^120+29220*x^118+36779*x^116+3974*x^114+7911*x^112+2699*x^110+13784*x^108+6518*x^106+1982*x^104+5676*x^102+40582*x^100+34026*x^98+20847*x^96"
            "+9317*x^94+3786*x^92+6303*x^90+35840*x^88+1260*x^86+7194*x^84+11865*x^82+37487*x^80+39482*x^78+3463*x^76+39442*x^74+2094*x^72+28384*x^70+8368*x^68+11182*x^66+22493*x^64+38408*x^62+5132*x^60+40665*x^58"
            "+17337*x^56+16314*x^54+12887*x^52+30723*x^50+467*x^48+19550*x^46+4535*x^44+14740*x^42+30512*x^40+16004*x^38+18242*x^36+21135*x^34+31930*x^32+9145*x^30+13045*x^28+37859*x^26+4891*x^24+15931*x^22+33896*x^20"
            "+22053*x^18+19758*x^16+29918*x^14+22856*x^12+42701*x^10+12577*x^8+40627*x^6+21210*x^4+32848*x^2+39447)", vars, ctx);
        time = check_omega(2, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf(" #2: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(39739*z^9*x^10*y^7*s^9+33115*z^9*x^11*y^6*s^9+33115*z^10*x^10*y^6*s^9+33115*z^9*x^12*y^5*s^9+23179*z^10*x^11*y^5*s^9+33115*z^11*x^10*y^5*s^9+39739*z^9*x^13*y^4*s^9+33115*z^10*x^12*y^4*s^9+33115*z^11*x^11*y^4*s^9+39739*z^12*x^10*y^4*s^9+41743*x*y^13*s^8+39127*x^2*y^12*s^8+39127*z*x*y^12*s^8+39127*x^3*y^11*s^8+35203*z*x^2*y^11*s^8+39127*z^2*x*y^11*s^8+41743*x^4*y^10*s^8+39127*z*x^3*y^10*s^8+39127*z^2*x^2*y^10*s^8+41743*z^3*x*y^10*s^8+21629*z^6*x*y^9*s^7+21836*z^6*x^2*y^8*s^7+21836*z^7*x*y^8*s^7+21836*z^6*x^3*y^7*s^7+621*z^7*x^2*y^7*s^7+21836*z^8*x*y^7*s^7+21629*z^6*x^4*y^6*s^7+21836*z^7*x^3*y^6*s^7+21836*z^8*x^2*y^6*s^7+21629*z^9*x*y^6*s^7+16873*z^8*x^3*y^12*s^6+7568*z^8*x^4*y^11*s^6+7568*z^9*x^3*y^11*s^6+7568*z^8*x^5*y^10*s^6+15136*z^9*x^4*y^10*s^6+7568*z^10*x^3*y^10*s^6+16873*z^8*x^6*y^9*s^6+7568*z^9*x^5*y^9*s^6+7568*z^10*x^4*y^9*s^6+16873*z^11*x^3*y^9*s^6+24116*z^6*x*y^11*s^5+29297*z^6*x^2*y^10*s^5+15295*z^7*x*y^10*s^5+29297*z^6*x^3*y^9*s^5+16588*z^7*x^2*y^9*s^5+30342*z^8*x*y^9*s^5+24116*z^6*x^4*y^8*s^5+30342*z^7*x^3*y^8*s^5+31387*z^8*x^2*y^8*s^5+25161*z^9*x*y^8*s^5+29049*z^7*x^4*y^7*s^5+1045*z^8*x^3*y^7*s^5+1045*z^9*x^2*y^7*s^5+29049*z^10*x*y^7*s^5+13845*z^6*x^2*y^13*s^3+41535*z^6*x^3*y^12*s^3+41535*z^7*x^2*y^12*s^3+41535*z^6*x^4*y^11*s^3+40019*z^7*x^3*y^11*s^3+41535*z^8*x^2*y^11*s^3+13845*z^6*x^5*y^10*s^3+41535*z^7*x^4*y^10*s^3+41535*z^8*x^3*y^10*s^3+13845*z^9*x^2*y^10*s^3+8506*z*x^10*y^9*s+25518*z*x^11*y^8*s+25518*z^2*x^10*y^8*s+25518*z*x^12*y^7*s+7985*z^2*x^11*y^7*s+25518*z^3*x^10*y^7*s+8506*z*x^13*y^6*s+25518*z^2*x^12*y^6*s+25518*z^3*x^11*y^6*s+8506*z^4*x^10*y^6*s+19583*z^8*x*y^6+15698*z^8*x^2*y^5+15698*z^9*x*y^5+15698*z^8*x^3*y^4+31396*z^9*x^2*y^4+15698*z^10*x*y^4+19583*z^8*x^4*y^3+15698*z^9*x^3*y^3+15698*z^10*x^2*y^3+19583*z^11*x*y^3)*(22750*z^4*x^4*y^4*s^4+41200*z^4*x*y^3*s^3+14342*x^4*y^2*s^3+27324*z^4*x^3*y^3*s+42152*z^2*y^3*s+14244*z^3*y^2*s+40575*z*x^2*y*s+4017*z^2*y^4+26755*z^4*x^3)", vars, ctx);
        time = check_omega(2, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf(" #3: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(16873*z^2*x^8*y^7*s^8+7568*z^2*x^9*y^6*s^8+7568*z^3*x^8*y^6*s^8+7568*z^2*x^10*y^5*s^8+15136*z^3*x^9*y^5*s^8+7568*z^4*x^8*y^5*s^8+16873*z^2*x^11*y^4*s^8+7568*z^3*x^10*y^4*s^8+7568*z^4*x^9*y^4*s^8+16873*z^5*x^8*y^4*s^8+41743*z^9*x^7*y^8*s^5+39127*z^9*x^8*y^7*s^5+39127*z^10*x^7*y^7*s^5+39127*z^9*x^9*y^6*s^5+35203*z^10*x^8*y^6*s^5+39127*z^11*x^7*y^6*s^5+41743*z^9*x^10*y^5*s^5+39127*z^10*x^9*y^5*s^5+39127*z^11*x^8*y^5*s^5+41743*z^12*x^7*y^5*s^5+19583*z^4*x^6*y^7*s^4+13845*z^6*x^9*y^6*s^4+15698*z^4*x^7*y^6*s^4+15698*z^5*x^6*y^6*s^4+41535*z^6*x^10*y^5*s^4+41535*z^7*x^9*y^5*s^4+15698*z^4*x^8*y^5*s^4+31396*z^5*x^7*y^5*s^4+15698*z^6*x^6*y^5*s^4+41535*z^6*x^11*y^4*s^4+40019*z^7*x^10*y^4*s^4+41535*z^8*x^9*y^4*s^4+19583*z^4*x^9*y^4*s^4+15698*z^5*x^8*y^4*s^4+15698*z^6*x^7*y^4*s^4+19583*z^7*x^6*y^4*s^4+13845*z^6*x^12*y^3*s^4+41535*z^7*x^11*y^3*s^4+41535*z^8*x^10*y^3*s^4+13845*z^9*x^9*y^3*s^4+39739*z^9*x^6*y^13*s^2+33115*z^9*x^7*y^12*s^2+33115*z^10*x^6*y^12*s^2+33115*z^9*x^8*y^11*s^2+23179*z^10*x^7*y^11*s^2+33115*z^11*x^6*y^11*s^2+39739*z^9*x^9*y^10*s^2+33115*z^10*x^8*y^10*s^2+33115*z^11*x^7*y^10*s^2+39739*z^12*x^6*y^10*s^2+8506*z^8*x^6*y^10*s^2+25518*z^8*x^7*y^9*s^2+25518*z^9*x^6*y^9*s^2+25518*z^8*x^8*y^8*s^2+7985*z^9*x^7*y^8*s^2+25518*z^10*x^6*y^8*s^2+21629*z^6*x^6*y^8*s^2+8506*z^8*x^9*y^7*s^2+25518*z^9*x^8*y^7*s^2+25518*z^10*x^7*y^7*s^2+21836*z^6*x^7*y^7*s^2+8506*z^11*x^6*y^7*s^2+21836*z^7*x^6*y^7*s^2+21836*z^6*x^8*y^6*s^2+621*z^7*x^7*y^6*s^2+21836*z^8*x^6*y^6*s^2+21629*z^6*x^9*y^5*s^2+21836*z^7*x^8*y^5*s^2+21836*z^8*x^7*y^5*s^2+21629*z^9*x^6*y^5*s^2+24116*z^6*x^7*y^4*s+29297*z^6*x^8*y^3*s+29297*z^7*x^7*y^3*s+29297*z^6*x^9*y^2*s+15543*z^7*x^8*y^2*s+29297*z^8*x^7*y^2*s+24116*z^6*x^10*y*s+29297*z^7*x^9*y*s+29297*z^8*x^8*y*s+24116*z^9*x^7*y*s+29049*z^7*x^6*y^6+1045*z^7*x^7*y^5+1045*z^8*x^6*y^5+1045*z^7*x^8*y^4+2090*z^8*x^7*y^4+1045*z^9*x^6*y^4+29049*z^7*x^9*y^3+1045*z^8*x^8*y^3+1045*z^9*x^7*y^3+29049*z^10*x^6*y^3)*(41200*x*y^4*s^3+40575*z*x^3*y^3*s^3+14342*x*y^3*s^3+4017*z*x^4*y^4*s^2+22750*z*x^3*y^3*s^2+26425*x^3*y^3*s^2+26755*z*x^2*s^2+14244*z^3*x^4*y*s)", vars, ctx);
        time = check_omega(2, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf(" #4: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(41743*z^7*x^5*y^6*s^8+39127*z^7*x^6*y^5*s^8+39127*z^8*x^5*y^5*s^8+39127*z^7*x^7*y^4*s^8+35203*z^8*x^6*y^4*s^8+39127*z^9*x^5*y^4*s^8+41743*z^7*x^8*y^3*s^8+39127*z^8*x^7*y^3*s^8+39127*z^9*x^6*y^3*s^8+41743*z^10*x^5*y^3*s^8+29049*x*y^5*s^7+1045*x^2*y^4*s^7+1045*z*x*y^4*s^7+1045*x^3*y^3*s^7+2090*z*x^2*y^3*s^7+1045*z^2*x*y^3*s^7+29049*x^4*y^2*s^7+1045*z*x^3*y^2*s^7+1045*z^2*x^2*y^2*s^7+29049*z^3*x*y^2*s^7+39739*z^3*x^8*y^11*s^5+33115*z^3*x^9*y^10*s^5+33115*z^4*x^8*y^10*s^5+33115*z^3*x^10*y^9*s^5+23179*z^4*x^9*y^9*s^5+33115*z^5*x^8*y^9*s^5+39739*z^3*x^11*y^8*s^5+33115*z^4*x^10*y^8*s^5+33115*z^5*x^9*y^8*s^5+39739*z^6*x^8*y^8*s^5+13845*x*y^4*s^4+41535*x^2*y^3*s^4+41535*z*x*y^3*s^4+41535*x^3*y^2*s^4+40019*z*x^2*y^2*s^4+41535*z^2*x*y^2*s^4+13845*x^4*y*s^4+41535*z*x^3*y*s^4+41535*z^2*x^2*y*s^4+13845*z^3*x*y*s^4+16873*z^3*x^10*y^11*s+7568*z^3*x^11*y^10*s+7568*z^4*x^10*y^10*s+21629*z^9*x^6*y^10*s+7568*z^3*x^12*y^9*s+15136*z^4*x^11*y^9*s+7568*z^5*x^10*y^9*s+21836*z^9*x^7*y^9*s+21836*z^10*x^6*y^9*s+16873*z^3*x^13*y^8*s+7568*z^4*x^12*y^8*s+7568*z^5*x^11*y^8*s+16873*z^6*x^10*y^8*s+21836*z^9*x^8*y^8*s+621*z^10*x^7*y^8*s+21836*z^11*x^6*y^8*s+21629*z^9*x^9*y^7*s+21836*z^10*x^8*y^7*s+21836*z^11*x^7*y^7*s+21629*z^12*x^6*y^7*s+19583*z^4*x^5*y^4*s+8506*z^4*x^3*y^4*s+15698*z^4*x^6*y^3*s+15698*z^5*x^5*y^3*s+25518*z^4*x^4*y^3*s+25518*z^5*x^3*y^3*s+15698*z^4*x^7*y^2*s+31396*z^5*x^6*y^2*s+15698*z^6*x^5*y^2*s+25518*z^4*x^5*y^2*s+7985*z^5*x^4*y^2*s+25518*z^6*x^3*y^2*s+19583*z^4*x^8*y*s+15698*z^5*x^7*y*s+15698*z^6*x^6*y*s+8506*z^4*x^6*y*s+19583*z^7*x^5*y*s+25518*z^5*x^5*y*s+25518*z^6*x^4*y*s+8506*z^7*x^3*y*s+24116*z^5*x^9*y^7+29297*z^5*x^10*y^6+29297*z^6*x^9*y^6+29297*z^5*x^11*y^5+15543*z^6*x^10*y^5+29297*z^7*x^9*y^5+24116*z^5*x^12*y^4+29297*z^6*x^11*y^4+29297*z^7*x^10*y^4+24116*z^8*x^9*y^4)*(41200*z^4*x^4*y^3*s^4+14244*z^3*x^4*y*s^4+14342*x^3*y*s^4+39676*z*x^2*y^2*s^3+27324*z^3*x*y^2*s^3+22750*z^4*y*s^3+26755*x*y*s^2+4017*x^3*y^3)", vars, ctx);
        time = check_omega(2, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf(" #5: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(13845*x^2*y^8*s^9+41535*x^3*y^7*s^9+41535*z*x^2*y^7*s^9+41535*x^4*y^6*s^9+40019*z*x^3*y^6*s^9+41535*z^2*x^2*y^6*s^9+13845*x^5*y^5*s^9+41535*z*x^4*y^5*s^9+41535*z^2*x^3*y^5*s^9+13845*z^3*x^2*y^5*s^9+41743*z^2*x^7*y^11*s^7+16873*x*y^11*s^7+39127*z^2*x^8*y^10*s^7+39127*z^3*x^7*y^10*s^7+7568*x^2*y^10*s^7+7568*z*x*y^10*s^7+39127*z^2*x^9*y^9*s^7+35203*z^3*x^8*y^9*s^7+39127*z^4*x^7*y^9*s^7+7568*x^3*y^9*s^7+15136*z*x^2*y^9*s^7+7568*z^2*x*y^9*s^7+41743*z^2*x^10*y^8*s^7+39127*z^3*x^9*y^8*s^7+39127*z^4*x^8*y^8*s^7+41743*z^5*x^7*y^8*s^7+16873*x^4*y^8*s^7+27151*z*x^3*y^8*s^7+7568*z^2*x^2*y^8*s^7+16873*z^3*x*y^8*s^7+8506*z^7*x^10*y^7*s^7+15698*z*x^4*y^7*s^7+15698*z^2*x^3*y^7*s^7+25518*z^7*x^11*y^6*s^7+25518*z^8*x^10*y^6*s^7+15698*z*x^5*y^6*s^7+31396*z^2*x^4*y^6*s^7+15698*z^3*x^3*y^6*s^7+25518*z^7*x^12*y^5*s^7+7985*z^8*x^11*y^5*s^7+25518*z^9*x^10*y^5*s^7+19583*z*x^6*y^5*s^7+15698*z^2*x^5*y^5*s^7+15698*z^3*x^4*y^5*s^7+19583*z^4*x^3*y^5*s^7+8506*z^7*x^13*y^4*s^7+25518*z^8*x^12*y^4*s^7+25518*z^9*x^11*y^4*s^7+8506*z^10*x^10*y^4*s^7+29049*z^8*x^2*y^13*s^6+1045*z^8*x^3*y^12*s^6+1045*z^9*x^2*y^12*s^6+1045*z^8*x^4*y^11*s^6+2090*z^9*x^3*y^11*s^6+1045*z^10*x^2*y^11*s^6+29049*z^8*x^5*y^10*s^6+1045*z^9*x^4*y^10*s^6+1045*z^10*x^3*y^10*s^6+29049*z^11*x^2*y^10*s^6+39739*z^3*x^5*y^4*s^4+33115*z^3*x^6*y^3*s^4+33115*z^4*x^5*y^3*s^4+33115*z^3*x^7*y^2*s^4+23179*z^4*x^6*y^2*s^4+33115*z^5*x^5*y^2*s^4+39739*z^3*x^8*y*s^4+33115*z^4*x^7*y*s^4+33115*z^5*x^6*y*s^4+39739*z^6*x^5*y*s^4+21629*z^4*x^5*y^9*s^3+21836*z^4*x^6*y^8*s^3+21836*z^5*x^5*y^8*s^3+21836*z^4*x^7*y^7*s^3+621*z^5*x^6*y^7*s^3+21836*z^6*x^5*y^7*s^3+21629*z^4*x^8*y^6*s^3+21836*z^5*x^7*y^6*s^3+21836*z^6*x^6*y^6*s^3+21629*z^7*x^5*y^6*s^3+24116*z^9*x^2*y^5+29297*z^9*x^3*y^4+29297*z^10*x^2*y^4+29297*z^9*x^4*y^3+15543*z^10*x^3*y^3+29297*z^11*x^2*y^3+24116*z^9*x^5*y^2+29297*z^10*x^4*y^2+29297*z^11*x^3*y^2+24116*z^12*x^2*y^2)*(4017*z*x^4*y^2*s^4+27324*z*x^3*s^4+26755*z^4*x^2*y^3*s^3+40575*x*s^3+14244*x*y^4*s^2+41200*z^4*x^2*s+42152*z*x*s+22750*s+14342*x^4*y)", vars, ctx);
        time = check_omega(2, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf(" #6: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(39739*z^7*x^10*y^7*s^7+33115*z^7*x^11*y^6*s^7+33115*z^8*x^10*y^6*s^7+33115*z^7*x^12*y^5*s^7+23179*z^8*x^11*y^5*s^7+33115*z^9*x^10*y^5*s^7+39739*z^7*x^13*y^4*s^7+33115*z^8*x^12*y^4*s^7+33115*z^9*x^11*y^4*s^7+39739*z^10*x^10*y^4*s^7+8506*z^3*x^10*y^4*s^7+25518*z^3*x^11*y^3*s^7+25518*z^4*x^10*y^3*s^7+25518*z^3*x^12*y^2*s^7+7985*z^4*x^11*y^2*s^7+25518*z^5*x^10*y^2*s^7+8506*z^3*x^13*y*s^7+25518*z^4*x^12*y*s^7+25518*z^5*x^11*y*s^7+8506*z^6*x^10*y*s^7+41743*x^5*y^13*s^6+39127*x^6*y^12*s^6+39127*z*x^5*y^12*s^6+39127*x^7*y^11*s^6+35203*z*x^6*y^11*s^6+39127*z^2*x^5*y^11*s^6+41743*x^8*y^10*s^6+39127*z*x^7*y^10*s^6+39127*z^2*x^6*y^10*s^6+41743*z^3*x^5*y^10*s^6+16873*z^6*x^4*y^7*s^5+7568*z^6*x^5*y^6*s^5+7568*z^7*x^4*y^6*s^5+7568*z^6*x^6*y^5*s^5+15136*z^7*x^5*y^5*s^5+7568*z^8*x^4*y^5*s^5+16873*z^6*x^7*y^4*s^5+7568*z^7*x^6*y^4*s^5+7568*z^8*x^5*y^4*s^5+16873*z^9*x^4*y^4*s^5+13845*z^7*x^6*y^13*s^4+41535*z^7*x^7*y^12*s^4+41535*z^8*x^6*y^12*s^4+41535*z^7*x^8*y^11*s^4+40019*z^8*x^7*y^11*s^4+41535*z^9*x^6*y^11*s^4+13845*z^7*x^9*y^10*s^4+41535*z^8*x^8*y^10*s^4+41535*z^9*x^7*y^10*s^4+13845*z^10*x^6*y^10*s^4+19583*z^7*x^7*y^6*s^4+15698*z^7*x^8*y^5*s^4+15698*z^8*x^7*y^5*s^4+15698*z^7*x^9*y^4*s^4+31396*z^8*x^8*y^4*s^4+15698*z^9*x^7*y^4*s^4+19583*z^7*x^10*y^3*s^4+15698*z^8*x^9*y^3*s^4+15698*z^9*x^8*y^3*s^4+19583*z^10*x^7*y^3*s^4+21629*z*x^9*y^8*s^2+24116*z^3*x^5*y^8*s^2+21836*z*x^10*y^7*s^2+21836*z^2*x^9*y^7*s^2+29297*z^3*x^6*y^7*s^2+29297*z^4*x^5*y^7*s^2+21836*z*x^11*y^6*s^2+621*z^2*x^10*y^6*s^2+21836*z^3*x^9*y^6*s^2+29297*z^3*x^7*y^6*s^2+15543*z^4*x^6*y^6*s^2+29297*z^5*x^5*y^6*s^2+21629*z*x^12*y^5*s^2+21836*z^2*x^11*y^5*s^2+21836*z^3*x^10*y^5*s^2+21629*z^4*x^9*y^5*s^2+24116*z^3*x^8*y^5*s^2+29297*z^4*x^7*y^5*s^2+29297*z^5*x^6*y^5*s^2+24116*z^6*x^5*y^5*s^2+29049*z^2*x^2*y^10+1045*z^2*x^3*y^9+1045*z^3*x^2*y^9+1045*z^2*x^4*y^8+2090*z^3*x^3*y^8+1045*z^4*x^2*y^8+29049*z^2*x^5*y^7+1045*z^3*x^4*y^7+1045*z^4*x^3*y^7+29049*z^5*x^2*y^7)*(41200*z^4*x^4*y^3*s^3+22750*x^3*s^3+26755*z^2*x^3*y^4*s^2+4017*z^4*y^2*s^2+40575*z^2*x^4*y^2*s+42152*z^2*x*s+27324*x^2*y^2+14244*z^2*x^4*y+14342*y)", vars, ctx);
        time = check_omega(2, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf(" #7: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(29049*z^5*x^8*y^11*s^9+22674*z^5*x^9*y^10*s^9+1045*z^6*x^8*y^10*s^9+22881*z^5*x^10*y^9*s^9+23926*z^6*x^9*y^9*s^9+1045*z^7*x^8*y^9*s^9+7834*z^5*x^11*y^8*s^9+1666*z^6*x^10*y^8*s^9+22881*z^7*x^9*y^8*s^9+29049*z^8*x^8*y^8*s^9+21629*z^5*x^12*y^7*s^9+21836*z^6*x^11*y^7*s^9+21836*z^7*x^10*y^7*s^9+21629*z^8*x^9*y^7*s^9+41743*z^6*x^6*y^8*s^7+39127*z^6*x^7*y^7*s^7+39127*z^7*x^6*y^7*s^7+39127*z^6*x^8*y^6*s^7+35203*z^7*x^7*y^6*s^7+39127*z^8*x^6*y^6*s^7+19583*z^4*x*y^6*s^7+41743*z^6*x^9*y^5*s^7+39127*z^7*x^8*y^5*s^7+39127*z^8*x^7*y^5*s^7+41743*z^9*x^6*y^5*s^7+15698*z^4*x^2*y^5*s^7+15698*z^5*x*y^5*s^7+15698*z^4*x^3*y^4*s^7+31396*z^5*x^2*y^4*s^7+15698*z^6*x*y^4*s^7+19583*z^4*x^4*y^3*s^7+15698*z^5*x^3*y^3*s^7+15698*z^6*x^2*y^3*s^7+19583*z^7*x*y^3*s^7+39739*z^3*x^7*y^13*s^5+33115*z^3*x^8*y^12*s^5+33115*z^4*x^7*y^12*s^5+33115*z^3*x^9*y^11*s^5+23179*z^4*x^8*y^11*s^5+33115*z^5*x^7*y^11*s^5+39739*z^3*x^10*y^10*s^5+33115*z^4*x^9*y^10*s^5+33115*z^5*x^8*y^10*s^5+39739*z^6*x^7*y^10*s^5+16873*z^9*x^10*y^5*s^5+7568*z^9*x^11*y^4*s^5+7568*z^10*x^10*y^4*s^5+7568*z^9*x^12*y^3*s^5+15136*z^10*x^11*y^3*s^5+7568*z^11*x^10*y^3*s^5+16873*z^9*x^13*y^2*s^5+7568*z^10*x^12*y^2*s^5+7568*z^11*x^11*y^2*s^5+16873*z^12*x^10*y^2*s^5+24116*z^7*x^7*y^4*s^4+29297*z^7*x^8*y^3*s^4+29297*z^8*x^7*y^3*s^4+29297*z^7*x^9*y^2*s^4+15543*z^8*x^8*y^2*s^4+29297*z^9*x^7*y^2*s^4+24116*z^7*x^10*y*s^4+29297*z^8*x^9*y*s^4+29297*z^9*x^8*y*s^4+24116*z^10*x^7*y*s^4+8506*z^6*x^3*y^5*s^3+25518*z^6*x^4*y^4*s^3+25518*z^7*x^3*y^4*s^3+25518*z^6*x^5*y^3*s^3+7985*z^7*x^4*y^3*s^3+25518*z^8*x^3*y^3*s^3+8506*z^6*x^6*y^2*s^3+25518*z^7*x^5*y^2*s^3+25518*z^8*x^4*y^2*s^3+8506*z^9*x^3*y^2*s^3+13845*z^7*x^10*y^7+41535*z^7*x^11*y^6+41535*z^8*x^10*y^6+41535*z^7*x^12*y^5+40019*z^8*x^11*y^5+41535*z^9*x^10*y^5+13845*z^7*x^13*y^4+41535*z^8*x^12*y^4+41535*z^9*x^11*y^4+13845*z^10*x^10*y^4)*(40575*z^4*x^4*s^4+22750*z^4*x^2*y^2*s^2+26755*z^3*y^2*s^2+27324*z^4*x^4*y*s^2+14342*z^4*x^3*y^2*s+14244*x^2*y*s+41200*x^3*s+4017*z^4*y^2+42152*x)", vars, ctx);
        time = check_omega(2, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf(" #8: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(39739*z^9*x^7*y^8*s^7+33115*z^9*x^8*y^7*s^7+33115*z^10*x^7*y^7*s^7+33115*z^9*x^9*y^6*s^7+23179*z^10*x^8*y^6*s^7+33115*z^11*x^7*y^6*s^7+39739*z^9*x^10*y^5*s^7+33115*z^10*x^9*y^5*s^7+33115*z^11*x^8*y^5*s^7+39739*z^12*x^7*y^5*s^7+24116*z^8*x*y^10*s^6+29297*z^8*x^2*y^9*s^6+29297*z^9*x*y^9*s^6+29297*z^8*x^3*y^8*s^6+15543*z^9*x^2*y^8*s^6+29297*z^10*x*y^8*s^6+24116*z^8*x^4*y^7*s^6+29297*z^9*x^3*y^7*s^6+29297*z^10*x^2*y^7*s^6+24116*z^11*x*y^7*s^6+21629*z^6*x^7*y^6*s^5+21836*z^6*x^8*y^5*s^5+21836*z^7*x^7*y^5*s^5+21836*z^6*x^9*y^4*s^5+621*z^7*x^8*y^4*s^5+21836*z^8*x^7*y^4*s^5+21629*z^6*x^10*y^3*s^5+21836*z^7*x^9*y^3*s^5+21836*z^8*x^8*y^3*s^5+21629*z^9*x^7*y^3*s^5+19583*z^6*x^9*y^9*s^3+15698*z^6*x^10*y^8*s^3+15698*z^7*x^9*y^8*s^3+15698*z^6*x^11*y^7*s^3+31396*z^7*x^10*y^7*s^3+15698*z^8*x^9*y^7*s^3+29049*z^7*x^8*y^7*s^3+19583*z^6*x^12*y^6*s^3+15698*z^7*x^11*y^6*s^3+15698*z^8*x^10*y^6*s^3+19583*z^9*x^9*y^6*s^3+1045*z^7*x^9*y^6*s^3+1045*z^8*x^8*y^6*s^3+1045*z^7*x^10*y^5*s^3+2090*z^8*x^9*y^5*s^3+1045*z^9*x^8*y^5*s^3+29049*z^7*x^11*y^4*s^3+1045*z^8*x^10*y^4*s^3+1045*z^9*x^9*y^4*s^3+29049*z^10*x^8*y^4*s^3+41743*z^2*x^3*y^11*s^2+39127*z^2*x^4*y^10*s^2+39127*z^3*x^3*y^10*s^2+39127*z^2*x^5*y^9*s^2+35203*z^3*x^4*y^9*s^2+39127*z^4*x^3*y^9*s^2+41743*z^2*x^6*y^8*s^2+39127*z^3*x^5*y^8*s^2+39127*z^4*x^4*y^8*s^2+41743*z^5*x^3*y^8*s^2+13845*z^7*x^7*y^4*s+41535*z^7*x^8*y^3*s+41535*z^8*x^7*y^3*s+41535*z^7*x^9*y^2*s+40019*z^8*x^8*y^2*s+41535*z^9*x^7*y^2*s+13845*z^7*x^10*y*s+41535*z^8*x^9*y*s+41535*z^9*x^8*y*s+13845*z^10*x^7*y*s+8506*z^2*x^5*y^10+16873*z^8*x^4*y^10+25518*z^2*x^6*y^9+7568*z^8*x^5*y^9+25518*z^3*x^5*y^9+7568*z^9*x^4*y^9+25518*z^2*x^7*y^8+7568*z^8*x^6*y^8+7985*z^3*x^6*y^8+15136*z^9*x^5*y^8+25518*z^4*x^5*y^8+7568*z^10*x^4*y^8+8506*z^2*x^8*y^7+16873*z^8*x^7*y^7+25518*z^3*x^7*y^7+7568*z^9*x^6*y^7+25518*z^4*x^6*y^7+7568*z^10*x^5*y^7+8506*z^5*x^5*y^7+16873*z^11*x^4*y^7)*(26755*z^4*x^4*y*s^3+11768*x^3*s^2+27324*z^2*x^2*y^2*s+4017*z*x^3*y*s+42152*z^2*x*s+41200*z*x^4*y+14342*z^4*x^2*y+22750*z^2*x^2)", vars, ctx);
        time = check_omega(2, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf(" #9: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(29049*z^4*x*y^8*s^9+1045*z^4*x^2*y^7*s^9+1045*z^5*x*y^7*s^9+1045*z^4*x^3*y^6*s^9+2090*z^5*x^2*y^6*s^9+1045*z^6*x*y^6*s^9+29049*z^4*x^4*y^5*s^9+1045*z^5*x^3*y^5*s^9+1045*z^6*x^2*y^5*s^9+29049*z^7*x*y^5*s^9+16873*z^2*x^5*y^13*s^8+7568*z^2*x^6*y^12*s^8+7568*z^3*x^5*y^12*s^8+7568*z^2*x^7*y^11*s^8+15136*z^3*x^6*y^11*s^8+7568*z^4*x^5*y^11*s^8+16873*z^2*x^8*y^10*s^8+7568*z^3*x^7*y^10*s^8+7568*z^4*x^6*y^10*s^8+16873*z^5*x^5*y^10*s^8+19583*z^6*x^2*y^13*s^6+15698*z^6*x^3*y^12*s^6+15698*z^7*x^2*y^12*s^6+15698*z^6*x^4*y^11*s^6+31396*z^7*x^3*y^11*s^6+15698*z^8*x^2*y^11*s^6+19583*z^6*x^5*y^10*s^6+15698*z^7*x^4*y^10*s^6+15698*z^8*x^3*y^10*s^6+19583*z^9*x^2*y^10*s^6+41743*z^6*x^3*y^8*s^6+39127*z^6*x^4*y^7*s^6+39127*z^7*x^3*y^7*s^6+39127*z^6*x^5*y^6*s^6+35203*z^7*x^4*y^6*s^6+39127*z^8*x^3*y^6*s^6+41743*z^6*x^6*y^5*s^6+39127*z^7*x^5*y^5*s^6+39127*z^8*x^4*y^5*s^6+41743*z^9*x^3*y^5*s^6+24116*x^3*y^12*s^4+29297*x^4*y^11*s^4+29297*z*x^3*y^11*s^4+29297*x^5*y^10*s^4+15543*z*x^4*y^10*s^4+29297*z^2*x^3*y^10*s^4+24116*x^6*y^9*s^4+29297*z*x^5*y^9*s^4+29297*z^2*x^4*y^9*s^4+24116*z^3*x^3*y^9*s^4+13845*z^8*x^9*y^7*s^3+41535*z^8*x^10*y^6*s^3+41535*z^9*x^9*y^6*s^3+41535*z^8*x^11*y^5*s^3+40019*z^9*x^10*y^5*s^3+41535*z^10*x^9*y^5*s^3+13845*z^8*x^12*y^4*s^3+41535*z^9*x^11*y^4*s^3+41535*z^10*x^10*y^4*s^3+13845*z^11*x^9*y^4*s^3+8506*z^7*x^5*y^4*s^3+25518*z^7*x^6*y^3*s^3+25518*z^8*x^5*y^3*s^3+25518*z^7*x^7*y^2*s^3+7985*z^8*x^6*y^2*s^3+25518*z^9*x^5*y^2*s^3+8506*z^7*x^8*y*s^3+25518*z^8*x^7*y*s^3+25518*z^9*x^6*y*s^3+8506*z^10*x^5*y*s^3+39739*z^4*x^8*y^10*s^2+33115*z^4*x^9*y^9*s^2+33115*z^5*x^8*y^9*s^2+33115*z^4*x^10*y^8*s^2+23179*z^5*x^9*y^8*s^2+33115*z^6*x^8*y^8*s^2+21629*z^5*x^7*y^8*s^2+39739*z^4*x^11*y^7*s^2+33115*z^5*x^10*y^7*s^2+33115*z^6*x^9*y^7*s^2+39739*z^7*x^8*y^7*s^2+21836*z^5*x^8*y^7*s^2+21836*z^6*x^7*y^7*s^2+21836*z^5*x^9*y^6*s^2+621*z^6*x^8*y^6*s^2+21836*z^7*x^7*y^6*s^2+21629*z^5*x^10*y^5*s^2+21836*z^6*x^9*y^5*s^2+21836*z^7*x^8*y^5*s^2+21629*z^8*x^7*y^5*s^2)*(14342*z*x*y^4*s^3+41200*z^2*x^3*y^2*s^3+26755*z*x^3*y^2*s^3+4017*z^4*y*s^3+22750*x^3*y^3*s^2+42152*z^2*x^2*s^2+27324*z*y^2*s+40575*x^4*y*s+14244*z^3*x^2)", vars, ctx);
        time = check_omega(2, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#10: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(24116*z^3*x^5*y^4*s^3+29297*z^3*x^6*y^3*s^3+29297*z^4*x^5*y^3*s^3+29297*z^3*x^7*y^2*s^3+15543*z^4*x^6*y^2*s^3+29297*z^5*x^5*y^2*s^3+24116*z^3*x^8*y*s^3+29297*z^4*x^7*y*s^3+29297*z^5*x^6*y*s^3+24116*z^6*x^5*y*s^3+41743*z^2*x^5*y^8*s^2+39127*z^2*x^6*y^7*s^2+39127*z^3*x^5*y^7*s^2+39127*z^2*x^7*y^6*s^2+35203*z^3*x^6*y^6*s^2+39127*z^4*x^5*y^6*s^2+41743*z^2*x^8*y^5*s^2+39127*z^3*x^7*y^5*s^2+39127*z^4*x^6*y^5*s^2+41743*z^5*x^5*y^5*s^2+8506*x^5*y^5*s^2+25518*x^6*y^4*s^2+25518*z*x^5*y^4*s^2+19583*z^4*x^4*y^4*s^2+25518*x^7*y^3*s^2+7985*z*x^6*y^3*s^2+15698*z^4*x^5*y^3*s^2+25518*z^2*x^5*y^3*s^2+15698*z^5*x^4*y^3*s^2+8506*x^8*y^2*s^2+25518*z*x^7*y^2*s^2+15698*z^4*x^6*y^2*s^2+25518*z^2*x^6*y^2*s^2+31396*z^5*x^5*y^2*s^2+8506*z^3*x^5*y^2*s^2+15698*z^6*x^4*y^2*s^2+19583*z^4*x^7*y*s^2+15698*z^5*x^6*y*s^2+15698*z^6*x^5*y*s^2+19583*z^7*x^4*y*s^2)*(22750*x*y+14244*y+18359*z^2)", vars, ctx);
        time = check_omega(2, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#11: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(26755*z^6*x^5*y^3*s^7+12889*z^5*x^4*y^4*s^6+40575*y^2*s^6+22750*z^3*s^6+14244*z*x^3*y*s^4+41200*z^7*y^2*s^3+38991*z^7*x^7*y^4*s^2+42152*z^7*y^3*s^2+27324*z^6*y^2*s^2+33099*z^3*x^2*s^2+14342*z^4*x^5*y^3*s+9788*z*x^6*y^2*s+4620*z^6*x^3*y^6+4017*z^2*x^3*y^4)*(6157*z^14*x^5*y^15*s^14+18471*z^14*x^6*y^14*s^14+18471*z^15*x^5*y^14*s^14+18471*z^14*x^7*y^13*s^14+36942*z^15*x^6*y^13*s^14+18471*z^16*x^5*y^13*s^14+6157*z^14*x^8*y^12*s^14+18471*z^15*x^7*y^12*s^14+18471*z^16*x^6*y^12*s^14+6157*z^17*x^5*y^12*s^14+24116*z^3*x^3*y^6*s^13+29297*z^3*x^4*y^5*s^13+29297*z^4*x^3*y^5*s^13+29297*z^3*x^5*y^4*s^13+15543*z^4*x^4*y^4*s^13+29297*z^5*x^3*y^4*s^13+24116*z^3*x^6*y^3*s^13+29297*z^4*x^5*y^3*s^13+29297*z^5*x^4*y^3*s^13+24116*z^6*x^3*y^3*s^13+39739*z^6*x^15*y^8*s^12+33115*z^6*x^16*y^7*s^12+33115*z^7*x^15*y^7*s^12+33115*z^6*x^17*y^6*s^12+23179*z^7*x^16*y^6*s^12+33115*z^8*x^15*y^6*s^12+39739*z^6*x^18*y^5*s^12+33115*z^7*x^17*y^5*s^12+33115*z^8*x^16*y^5*s^12+39739*z^9*x^15*y^5*s^12+29049*z^12*x^8*y^4*s^10+1045*z^12*x^9*y^3*s^10+1045*z^13*x^8*y^3*s^10+1045*z^12*x^10*y^2*s^10+2090*z^13*x^9*y^2*s^10+1045*z^14*x^8*y^2*s^10+29049*z^12*x^11*y*s^10+1045*z^13*x^10*y*s^10+1045*z^14*x^9*y*s^10+29049*z^15*x^8*y*s^10+19583*z^10*x*y^16*s^9+15698*z^10*x^2*y^15*s^9+15698*z^11*x*y^15*s^9+15698*z^10*x^3*y^14*s^9+31396*z^11*x^2*y^14*s^9+15698*z^12*x*y^14*s^9+19583*z^10*x^4*y^13*s^9+15698*z^11*x^3*y^13*s^9+15698*z^12*x^2*y^13*s^9+19583*z^13*x*y^13*s^9+40031*z^6*x^4*y^14*s^8+33991*z^6*x^5*y^13*s^8+33991*z^7*x^4*y^13*s^8+33991*z^6*x^6*y^12*s^8+24931*z^7*x^5*y^12*s^8+33991*z^8*x^4*y^12*s^8+40031*z^6*x^7*y^11*s^8+33991*z^7*x^6*y^11*s^8+33991*z^8*x^5*y^11*s^8+40031*z^9*x^4*y^11*s^8+8506*z^6*x*y^15*s^7+25518*z^6*x^2*y^14*s^7+25518*z^7*x*y^14*s^7+25518*z^6*x^3*y^13*s^7+7985*z^7*x^2*y^13*s^7+25518*z^8*x*y^13*s^7+8506*z^6*x^4*y^12*s^7+25518*z^7*x^3*y^12*s^7+25518*z^8*x^2*y^12*s^7+8506*z^9*x*y^12*s^7+41743*z^12*x^2*y^10*s^7+39127*z^12*x^3*y^9*s^7+39127*z^13*x^2*y^9*s^7+39127*z^12*x^4*y^8*s^7+35203*z^13*x^3*y^8*s^7+39127*z^14*x^2*y^8*s^7+41743*z^12*x^5*y^7*s^7+39127*z^13*x^4*y^7*s^7+39127*z^14*x^3*y^7*s^7+41743*z^15*x^2*y^7*s^7+21629*z^10*x^8*y^8*s^6+9048*z^10*x^5*y^8*s^6+21836*z^10*x^9*y^7*s^6+21836*z^11*x^8*y^7*s^6+27144*z^10*x^6*y^7*s^6+27144*z^11*x^5*y^7*s^6+21836*z^10*x^10*y^6*s^6+621*z^11*x^9*y^6*s^6+21836*z^12*x^8*y^6*s^6+27144*z^10*x^7*y^6*s^6+11237*z^11*x^6*y^6*s^6+27144*z^12*x^5*y^6*s^6+21629*z^10*x^11*y^5*s^6+21836*z^11*x^10*y^5*s^6+21836*z^12*x^9*y^5*s^6+21629*z^13*x^8*y^5*s^6+9048*z^10*x^8*y^5*s^6+27144*z^11*x^7*y^5*s^6+27144*z^12*x^6*y^5*s^6+9048*z^13*x^5*y^5*s^6+16680*z^8*x^5*y^18*s^5+6989*z^8*x^6*y^17*s^5+6989*z^9*x^5*y^17*s^5+6989*z^8*x^7*y^16*s^5+13978*z^9*x^6*y^16*s^5+6989*z^10*x^5*y^16*s^5+16680*z^8*x^8*y^15*s^5+6989*z^9*x^7*y^15*s^5+6989*z^10*x^6*y^15*s^5+16680*z^11*x^5*y^15*s^5+16873*z^10*x^15*y^9*s^4+7568*z^10*x^16*y^8*s^4+7568*z^11*x^15*y^8*s^4+7568*z^10*x^17*y^7*s^4+15136*z^11*x^16*y^7*s^4+7568*z^12*x^15*y^7*s^4+16873*z^10*x^18*y^6*s^4+7568*z^11*x^17*y^6*s^4+7568*z^12*x^16*y^6*s^4+16873*z^13*x^15*y^6*s^4+13845*x^7*y^11*s+41535*x^8*y^10*s+41535*z*x^7*y^10*s+41535*x^9*y^9*s+40019*z*x^8*y^9*s+41535*z^2*x^7*y^9*s+13845*x^10*y^8*s+41535*z*x^9*y^8*s+41535*z^2*x^8*y^8*s+13845*z^3*x^7*y^8*s+37652*x*y^12+26854*x^2*y^11+26854*z*x*y^11+26854*x^3*y^10+10657*z*x^2*y^10+26854*z^2*x*y^10+37652*x^4*y^9+26854*z*x^3*y^9+26854*z^2*x^2*y^9+37652*z^3*x*y^9)", vars, ctx);
        time = check_omega(2, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#12: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(27324*x^2*y^4*s^4+26755*z^4*x^2*y*s^3+41200*x^2*s^3+14244*z^2*x^3*y^4*s^2+22750*z^4*y^4*s^2+4017*z^2*x^3*y^2*s^2+40575*z^4*x^2*y^2*s^2+14342*z^3*x^2*y^4*s+42152*z^2*x^3*y^2*s)*(21629*z^3*x^9*y^6*s^11+21836*z^3*x^10*y^5*s^11+21836*z^4*x^9*y^5*s^11+21836*z^3*x^11*y^4*s^11+621*z^4*x^10*y^4*s^11+21836*z^5*x^9*y^4*s^11+21629*z^3*x^12*y^3*s^11+21836*z^4*x^11*y^3*s^11+21836*z^5*x^10*y^3*s^11+21629*z^6*x^9*y^3*s^11+21629*z^2*x^9*y^8*s^10+21629*z^4*x^10*y^7*s^10+21836*z^2*x^10*y^7*s^10+21836*z^3*x^9*y^7*s^10+39739*z^7*x^8*y^7*s^10+21836*z^4*x^11*y^6*s^10+21836*z^2*x^11*y^6*s^10+21836*z^5*x^10*y^6*s^10+621*z^3*x^10*y^6*s^10+33115*z^7*x^9*y^6*s^10+21836*z^4*x^9*y^6*s^10+33115*z^8*x^8*y^6*s^10+21836*z^4*x^12*y^5*s^10+21629*z^2*x^12*y^5*s^10+621*z^5*x^11*y^5*s^10+21836*z^3*x^11*y^5*s^10+33115*z^7*x^10*y^5*s^10+21836*z^6*x^10*y^5*s^10+21836*z^4*x^10*y^5*s^10+23179*z^8*x^9*y^5*s^10+21629*z^5*x^9*y^5*s^10+33115*z^9*x^8*y^5*s^10+21629*z^4*x^13*y^4*s^10+21836*z^5*x^12*y^4*s^10+39739*z^7*x^11*y^4*s^10+21836*z^6*x^11*y^4*s^10+33115*z^8*x^10*y^4*s^10+21629*z^7*x^10*y^4*s^10+33115*z^9*x^9*y^4*s^10+39739*z^10*x^8*y^4*s^10+39739*z^6*x^8*y^9*s^9+39739*z^8*x^9*y^8*s^9+33115*z^6*x^9*y^8*s^9+33115*z^7*x^8*y^8*s^9+21629*z^2*x^11*y^7*s^9+33115*z^8*x^10*y^7*s^9+33115*z^6*x^10*y^7*s^9+33115*z^9*x^9*y^7*s^9+23179*z^7*x^9*y^7*s^9+33115*z^8*x^8*y^7*s^9+21836*z^2*x^12*y^6*s^9+33115*z^8*x^11*y^6*s^9+39739*z^6*x^11*y^6*s^9+21836*z^3*x^11*y^6*s^9+23179*z^9*x^10*y^6*s^9+33115*z^7*x^10*y^6*s^9+21629*z^4*x^10*y^6*s^9+33115*z^10*x^9*y^6*s^9+33115*z^8*x^9*y^6*s^9+21629*z^2*x^9*y^6*s^9+39739*z^9*x^8*y^6*s^9+21836*z^2*x^13*y^5*s^9+39739*z^8*x^12*y^5*s^9+621*z^3*x^12*y^5*s^9+33115*z^9*x^11*y^5*s^9+621*z^4*x^11*y^5*s^9+33115*z^10*x^10*y^5*s^9+21836*z^5*x^10*y^5*s^9+21836*z^2*x^10*y^5*s^9+39739*z^11*x^9*y^5*s^9+21836*z^3*x^9*y^5*s^9+21629*z^2*x^14*y^4*s^9+21836*z^3*x^13*y^4*s^9+621*z^4*x^12*y^4*s^9+22250*z^5*x^11*y^4*s^9+21836*z^2*x^11*y^4*s^9+21836*z^6*x^10*y^4*s^9+621*z^3*x^10*y^4*s^9+21836*z^4*x^9*y^4*s^9+21629*z^4*x^13*y^3*s^9+21836*z^5*x^12*y^3*s^9+21629*z^2*x^12*y^3*s^9+21836*z^6*x^11*y^3*s^9+21836*z^3*x^11*y^3*s^9+21629*z^7*x^10*y^3*s^9+21836*z^4*x^10*y^3*s^9+21629*z^5*x^9*y^3*s^9+29049*z^3*x*y^9*s^8+39739*z^6*x^10*y^8*s^8+1045*z^3*x^2*y^8*s^8+1045*z^4*x*y^8*s^8+33115*z^6*x^11*y^7*s^8+33115*z^7*x^10*y^7*s^8+39739*z^8*x^9*y^7*s^8+39739*z^6*x^8*y^7*s^8+1045*z^3*x^3*y^7*s^8+2090*z^4*x^2*y^7*s^8+1045*z^5*x*y^7*s^8+33115*z^6*x^12*y^6*s^8+23179*z^7*x^11*y^6*s^8+23179*z^8*x^10*y^6*s^8+33115*z^9*x^9*y^6*s^8+33115*z^6*x^9*y^6*s^8+33115*z^7*x^8*y^6*s^8+29049*z^3*x^4*y^6*s^8+1045*z^4*x^3*y^6*s^8+1045*z^5*x^2*y^6*s^8+29049*z^6*x*y^6*s^8+39739*z^6*x^13*y^5*s^8+33115*z^7*x^12*y^5*s^8+23179*z^8*x^11*y^5*s^8+19867*z^9*x^10*y^5*s^8+33115*z^6*x^10*y^5*s^8+33115*z^10*x^9*y^5*s^8+23179*z^7*x^9*y^5*s^8+33115*z^8*x^8*y^5*s^8+39739*z^8*x^12*y^4*s^8+33115*z^9*x^11*y^4*s^8+39739*z^6*x^11*y^4*s^8+33115*z^10*x^10*y^4*s^8+33115*z^7*x^10*y^4*s^8+39739*z^11*x^9*y^4*s^8+33115*z^8*x^9*y^4*s^8+39739*z^9*x^8*y^4*s^8+29049*z^2*x*y^11*s^7+29049*z^4*x^2*y^10*s^7+1045*z^2*x^2*y^10*s^7+1045*z^3*x*y^10*s^7+1045*z^4*x^3*y^9*s^7+1045*z^2*x^3*y^9*s^7+1045*z^5*x^2*y^9*s^7+2090*z^3*x^2*y^9*s^7+1045*z^4*x*y^9*s^7+1045*z^4*x^4*y^8*s^7+29049*z^2*x^4*y^8*s^7+2090*z^5*x^3*y^8*s^7+1045*z^3*x^3*y^8*s^7+1045*z^6*x^2*y^8*s^7+1045*z^4*x^2*y^8*s^7+29049*z^5*x*y^8*s^7+29049*z^4*x^5*y^7*s^7+1045*z^5*x^4*y^7*s^7+1045*z^6*x^3*y^7*s^7+29049*z^7*x^2*y^7*s^7+19583*z^6*x^4*y^4*s^7+15698*z^6*x^5*y^3*s^7+15698*z^7*x^4*y^3*s^7+15698*z^6*x^6*y^2*s^7+31396*z^7*x^5*y^2*s^7+15698*z^8*x^4*y^2*s^7+19583*z^6*x^7*y*s^7+15698*z^7*x^6*y*s^7+15698*z^8*x^5*y*s^7+19583*z^9*x^4*y*s^7+13845*z^5*x^8*y^12*s^6+41535*z^5*x^9*y^11*s^6+41535*z^6*x^8*y^11*s^6+41535*z^5*x^10*y^10*s^6+40019*z^6*x^9*y^10*s^6+41535*z^7*x^8*y^10*s^6+29049*z^2*x^3*y^10*s^6+13845*z^5*x^11*y^9*s^6+41535*z^6*x^10*y^9*s^6+41535*z^7*x^9*y^9*s^6+13845*z^8*x^8*y^9*s^6+1045*z^2*x^4*y^9*s^6+1045*z^3*x^3*y^9*s^6+29049*z^4*x^2*y^9*s^6+29049*z^2*x*y^9*s^6+1045*z^2*x^5*y^8*s^6+2090*z^3*x^4*y^8*s^6+2090*z^4*x^3*y^8*s^6+1045*z^5*x^2*y^8*s^6+1045*z^2*x^2*y^8*s^6+1045*z^3*x*y^8*s^6+29049*z^2*x^6*y^7*s^6+1045*z^3*x^5*y^7*s^6+2090*z^4*x^4*y^7*s^6+31139*z^5*x^3*y^7*s^6+1045*z^2*x^3*y^7*s^6+1045*z^6*x^2*y^7*s^6+2090*z^3*x^2*y^7*s^6+1045*z^4*x*y^7*s^6+29049*z^4*x^5*y^6*s^6+20628*z^5*x^4*y^6*s^6+29049*z^2*x^4*y^6*s^6+1045*z^6*x^3*y^6*s^6+1045*z^3*x^3*y^6*s^6+29049*z^7*x^2*y^6*s^6+1045*z^4*x^2*y^6*s^6+29049*z^5*x*y^6*s^6+19583*z^7*x^5*y^5*s^6+15698*z^5*x^5*y^5*s^6+15698*z^6*x^4*y^5*s^6+15698*z^7*x^6*y^4*s^6+15698*z^5*x^6*y^4*s^6+15698*z^8*x^5*y^4*s^6+31396*z^6*x^5*y^4*s^6+15698*z^7*x^4*y^4*s^6+15698*z^7*x^7*y^3*s^6+19583*z^5*x^7*y^3*s^6+31396*z^8*x^6*y^3*s^6+15698*z^6*x^6*y^3*s^6+15698*z^9*x^5*y^3*s^6+15698*z^7*x^5*y^3*s^6+19583*z^8*x^4*y^3*s^6+19583*z^7*x^8*y^2*s^6+15698*z^8*x^7*y^2*s^6+15698*z^9*x^6*y^2*s^6+19583*z^10*x^5*y^2*s^6+13845*z^4*x^8*y^14*s^5+13845*z^6*x^9*y^13*s^5+41535*z^4*x^9*y^13*s^5+41535*z^5*x^8*y^13*s^5+41535*z^6*x^10*y^12*s^5+41535*z^4*x^10*y^12*s^5+41535*z^7*x^9*y^12*s^5+40019*z^5*x^9*y^12*s^5+41535*z^6*x^8*y^12*s^5+41535*z^6*x^11*y^11*s^5+13845*z^4*x^11*y^11*s^5+40019*z^7*x^10*y^11*s^5+41535*z^5*x^10*y^11*s^5+41535*z^8*x^9*y^11*s^5+41535*z^6*x^9*y^11*s^5+13845*z^7*x^8*y^11*s^5+13845*z^6*x^12*y^10*s^5+41535*z^7*x^11*y^10*s^5+41535*z^8*x^10*y^10*s^5+13845*z^9*x^9*y^10*s^5+19583*z^5*x^6*y^5*s^5+15698*z^5*x^7*y^4*s^5+15698*z^6*x^6*y^4*s^5+19583*z^7*x^5*y^4*s^5+19583*z^5*x^4*y^4*s^5+15698*z^5*x^8*y^3*s^5+31396*z^6*x^7*y^3*s^5+31396*z^7*x^6*y^3*s^5+15698*z^8*x^5*y^3*s^5+15698*z^5*x^5*y^3*s^5+15698*z^6*x^4*y^3*s^5+19583*z^5*x^9*y^2*s^5+15698*z^6*x^8*y^2*s^5+31396*z^7*x^7*y^2*s^5+7928*z^8*x^6*y^2*s^5+15698*z^5*x^6*y^2*s^5+15698*z^9*x^5*y^2*s^5+31396*z^6*x^5*y^2*s^5+15698*z^7*x^4*y^2*s^5+19583*z^7*x^8*y*s^5+15698*z^8*x^7*y*s^5+19583*z^5*x^7*y*s^5+15698*z^9*x^6*y*s^5+15698*z^6*x^6*y*s^5+19583*z^10*x^5*y*s^5+15698*z^7*x^5*y*s^5+19583*z^8*x^4*y*s^5+13845*z^4*x^10*y^13*s^4+41535*z^4*x^11*y^12*s^4+41535*z^5*x^10*y^12*s^4+13845*z^6*x^9*y^12*s^4+13845*z^4*x^8*y^12*s^4+41535*z^4*x^12*y^11*s^4+40019*z^5*x^11*y^11*s^4+40019*z^6*x^10*y^11*s^4+41535*z^7*x^9*y^11*s^4+41535*z^4*x^9*y^11*s^4+41535*z^5*x^8*y^11*s^4+13845*z^4*x^13*y^10*s^4+41535*z^5*x^12*y^10*s^4+40019*z^6*x^11*y^10*s^4+10813*z^7*x^10*y^10*s^4+41535*z^4*x^10*y^10*s^4+41535*z^8*x^9*y^10*s^4+40019*z^5*x^9*y^10*s^4+41535*z^6*x^8*y^10*s^4+13845*z^6*x^12*y^9*s^4+41535*z^7*x^11*y^9*s^4+13845*z^4*x^11*y^9*s^4+41535*z^8*x^10*y^9*s^4+41535*z^5*x^10*y^9*s^4+13845*z^9*x^9*y^9*s^4+41535*z^6*x^9*y^9*s^4+13845*z^7*x^8*y^9*s^4+16873*z^2*x^9*y^7*s^4+7568*z^2*x^10*y^6*s^4+7568*z^3*x^9*y^6*s^4+7568*z^2*x^11*y^5*s^4+15136*z^3*x^10*y^5*s^4+7568*z^4*x^9*y^5*s^4+41743*z^2*x^4*y^5*s^4+16873*z^2*x^12*y^4*s^4+7568*z^3*x^11*y^4*s^4+7568*z^4*x^10*y^4*s^4+16873*z^5*x^9*y^4*s^4+8506*z^5*x^5*y^4*s^4+39127*z^2*x^5*y^4*s^4+39127*z^3*x^4*y^4*s^4+25518*z^5*x^6*y^3*s^4+39127*z^2*x^6*y^3*s^4+25518*z^6*x^5*y^3*s^4+35203*z^3*x^5*y^3*s^4+39127*z^4*x^4*y^3*s^4+25518*z^5*x^7*y^2*s^4+41743*z^2*x^7*y^2*s^4+7985*z^6*x^6*y^2*s^4+39127*z^3*x^6*y^2*s^4+25518*z^7*x^5*y^2*s^4+39127*z^4*x^5*y^2*s^4+41743*z^5*x^4*y^2*s^4+8506*z^5*x^8*y*s^4+25518*z^6*x^7*y*s^4+25518*z^7*x^6*y*s^4+8506*z^8*x^5*y*s^4+16873*z*x^9*y^9*s^3+16873*z^3*x^10*y^8*s^3+7568*z*x^10*y^8*s^3+7568*z^2*x^9*y^8*s^3+7568*z^3*x^11*y^7*s^3+7568*z*x^11*y^7*s^3+7568*z^4*x^10*y^7*s^3+15136*z^2*x^10*y^7*s^3+7568*z^3*x^9*y^7*s^3+41743*z*x^4*y^7*s^3+7568*z^3*x^12*y^6*s^3+16873*z*x^12*y^6*s^3+15136*z^4*x^11*y^6*s^3+7568*z^2*x^11*y^6*s^3+7568*z^5*x^10*y^6*s^3+7568*z^3*x^10*y^6*s^3+16873*z^4*x^9*y^6*s^3+8506*z^4*x^5*y^6*s^3+41743*z^3*x^5*y^6*s^3+39127*z*x^5*y^6*s^3+39127*z^2*x^4*y^6*s^3+16873*z^3*x^13*y^5*s^3+7568*z^4*x^12*y^5*s^3+7568*z^5*x^11*y^5*s^3+16873*z^6*x^10*y^5*s^3+8506*z^6*x^6*y^5*s^3+25518*z^4*x^6*y^5*s^3+39127*z^3*x^6*y^5*s^3+39127*z*x^6*y^5*s^3+25518*z^5*x^5*y^5*s^3+39127*z^4*x^5*y^5*s^3+35203*z^2*x^5*y^5*s^3+39127*z^3*x^4*y^5*s^3+25518*z^6*x^7*y^4*s^3+25518*z^4*x^7*y^4*s^3+39127*z^3*x^7*y^4*s^3+41743*z*x^7*y^4*s^3+25518*z^7*x^6*y^4*s^3+7985*z^5*x^6*y^4*s^3+35203*z^4*x^6*y^4*s^3+39127*z^2*x^6*y^4*s^3+25518*z^6*x^5*y^4*s^3+39127*z^5*x^5*y^4*s^3+39127*z^3*x^5*y^4*s^3+41743*z^4*x^4*y^4*s^3+25518*z^6*x^8*y^3*s^3+8506*z^4*x^8*y^3*s^3+41743*z^3*x^8*y^3*s^3+7985*z^7*x^7*y^3*s^3+25518*z^5*x^7*y^3*s^3+39127*z^4*x^7*y^3*s^3+25518*z^8*x^6*y^3*s^3+25518*z^6*x^6*y^3*s^3+39127*z^5*x^6*y^3*s^3+8506*z^7*x^5*y^3*s^3+41743*z^6*x^5*y^3*s^3+8506*z^6*x^9*y^2*s^3+25518*z^7*x^8*y^2*s^3+25518*z^8*x^7*y^2*s^3+8506*z^9*x^6*y^2*s^3+24116*z^10*x^6*y^11*s^2+29297*z^10*x^7*y^10*s^2+29297*z^11*x^6*y^10*s^2+29297*z^10*x^8*y^9*s^2+15543*z^11*x^7*y^9*s^2+29297*z^12*x^6*y^9*s^2+16873*z*x^11*y^8*s^2+24116*z^10*x^9*y^8*s^2+29297*z^11*x^8*y^8*s^2+29297*z^12*x^7*y^8*s^2+24116*z^13*x^6*y^8*s^2+7568*z*x^12*y^7*s^2+7568*z^2*x^11*y^7*s^2+16873*z^3*x^10*y^7*s^2+16873*z*x^9*y^7*s^2+7568*z*x^13*y^6*s^2+15136*z^2*x^12*y^6*s^2+15136*z^3*x^11*y^6*s^2+7568*z^4*x^10*y^6*s^2+7568*z*x^10*y^6*s^2+7568*z^2*x^9*y^6*s^2+41743*z*x^6*y^6*s^2+16873*z*x^14*y^5*s^2+7568*z^2*x^13*y^5*s^2+15136*z^3*x^12*y^5*s^2+32009*z^4*x^11*y^5*s^2+7568*z*x^11*y^5*s^2+7568*z^5*x^10*y^5*s^2+15136*z^2*x^10*y^5*s^2+7568*z^3*x^9*y^5*s^2+8506*z^4*x^7*y^5*s^2+39127*z*x^7*y^5*s^2+39127*z^2*x^6*y^5*s^2+41743*z^3*x^5*y^5*s^2+41743*z*x^4*y^5*s^2+16873*z^3*x^13*y^4*s^2+7568*z^4*x^12*y^4*s^2+16873*z*x^12*y^4*s^2+7568*z^5*x^11*y^4*s^2+7568*z^2*x^11*y^4*s^2+16873*z^6*x^10*y^4*s^2+7568*z^3*x^10*y^4*s^2+16873*z^4*x^9*y^4*s^2+25518*z^4*x^8*y^4*s^2+39127*z*x^8*y^4*s^2+25518*z^5*x^7*y^4*s^2+35203*z^2*x^7*y^4*s^2+8506*z^6*x^6*y^4*s^2+35203*z^3*x^6*y^4*s^2+4582*z^4*x^5*y^4*s^2+39127*z*x^5*y^4*s^2+39127*z^2*x^4*y^4*s^2+25518*z^4*x^9*y^3*s^2+41743*z*x^9*y^3*s^2+7985*z^5*x^8*y^3*s^2+39127*z^2*x^8*y^3*s^2+7985*z^6*x^7*y^3*s^2+35203*z^3*x^7*y^3*s^2+25518*z^7*x^6*y^3*s^2+16362*z^4*x^6*y^3*s^2+39127*z*x^6*y^3*s^2+21594*z^5*x^5*y^3*s^2+35203*z^2*x^5*y^3*s^2+39127*z^3*x^4*y^3*s^2+8506*z^4*x^10*y^2*s^2+25518*z^5*x^9*y^2*s^2+7985*z^6*x^8*y^2*s^2+41743*z^3*x^8*y^2*s^2+16491*z^7*x^7*y^2*s^2+21594*z^4*x^7*y^2*s^2+41743*z*x^7*y^2*s^2+25518*z^8*x^6*y^2*s^2+4061*z^5*x^6*y^2*s^2+39127*z^2*x^6*y^2*s^2+24210*z^6*x^5*y^2*s^2+39127*z^3*x^5*y^2*s^2+41743*z^4*x^4*y^2*s^2+8506*z^6*x^9*y*s^2+25518*z^7*x^8*y*s^2+8506*z^4*x^8*y*s^2+25518*z^8*x^7*y*s^2+25518*z^5*x^7*y*s^2+8506*z^9*x^6*y*s^2+25518*z^6*x^6*y*s^2+8506*z^7*x^5*y*s^2+24116*z^9*x^6*y^13*s+24116*z^11*x^7*y^12*s+29297*z^9*x^7*y^12*s+29297*z^10*x^6*y^12*s+29297*z^11*x^8*y^11*s+29297*z^9*x^8*y^11*s+29297*z^12*x^7*y^11*s+15543*z^10*x^7*y^11*s+29297*z^11*x^6*y^11*s+29297*z^11*x^9*y^10*s+24116*z^9*x^9*y^10*s+15543*z^12*x^8*y^10*s+29297*z^10*x^8*y^10*s+29297*z^13*x^7*y^10*s+29297*z^11*x^7*y^10*s+24116*z^12*x^6*y^10*s+24116*z^11*x^10*y^9*s+29297*z^12*x^9*y^9*s+29297*z^13*x^8*y^9*s+24116*z^14*x^7*y^9*s+24116*z^9*x^8*y^12+29297*z^9*x^9*y^11+29297*z^10*x^8*y^11+24116*z^11*x^7*y^11+24116*z^9*x^6*y^11+29297*z^9*x^10*y^10+15543*z^10*x^9*y^10+15543*z^11*x^8*y^10+29297*z^12*x^7*y^10+29297*z^9*x^7*y^10+29297*z^10*x^6*y^10+24116*z^9*x^11*y^9+29297*z^10*x^10*y^9+15543*z^11*x^9*y^9+39659*z^12*x^8*y^9+29297*z^9*x^8*y^9+29297*z^13*x^7*y^9+15543*z^10*x^7*y^9+29297*z^11*x^6*y^9+24116*z^11*x^10*y^8+29297*z^12*x^9*y^8+24116*z^9*x^9*y^8+29297*z^13*x^8*y^8+29297*z^10*x^8*y^8+24116*z^14*x^7*y^8+29297*z^11*x^7*y^8+24116*z^12*x^6*y^8)", vars, ctx);
        time = check_omega(2, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#13: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(4017*z^3*x^3*y^4*s^4+27324*x^2*y^3*s^4+41200*x^2*y^2*s^2+26755*x*y*s^2+22750*z^2*x^4*y^4*s+42152*z^2*x^3*y^2*s+14244*z^2*x^3*y^2+40575*z^2*x^4*y+14342*z)*(41743*z^2*x^4*y^11*s^11+39127*z^2*x^5*y^10*s^11+39127*z^3*x^4*y^10*s^11+39127*z^2*x^6*y^9*s^11+35203*z^3*x^5*y^9*s^11+39127*z^4*x^4*y^9*s^11+41743*z^2*x^7*y^8*s^11+39127*z^3*x^6*y^8*s^11+39127*z^4*x^5*y^8*s^11+39739*z^10*x^4*y^8*s^11+41743*z^5*x^4*y^8*s^11+33115*z^10*x^5*y^7*s^11+33115*z^11*x^4*y^7*s^11+33115*z^10*x^6*y^6*s^11+23179*z^11*x^5*y^6*s^11+33115*z^12*x^4*y^6*s^11+39739*z^10*x^7*y^5*s^11+33115*z^11*x^6*y^5*s^11+33115*z^12*x^5*y^5*s^11+39739*z^13*x^4*y^5*s^11+19583*z^9*x^5*y^13*s^10+41743*z*x^4*y^13*s^10+15698*z^9*x^6*y^12*s^10+15698*z^10*x^5*y^12*s^10+41743*z^3*x^5*y^12*s^10+39127*z*x^5*y^12*s^10+39127*z^2*x^4*y^12*s^10+15698*z^9*x^7*y^11*s^10+31396*z^10*x^6*y^11*s^10+39127*z^3*x^6*y^11*s^10+39127*z*x^6*y^11*s^10+15698*z^11*x^5*y^11*s^10+39127*z^4*x^5*y^11*s^10+35203*z^2*x^5*y^11*s^10+39127*z^3*x^4*y^11*s^10+19583*z^9*x^8*y^10*s^10+15698*z^10*x^7*y^10*s^10+39127*z^3*x^7*y^10*s^10+41743*z*x^7*y^10*s^10+15698*z^11*x^6*y^10*s^10+35203*z^4*x^6*y^10*s^10+39127*z^2*x^6*y^10*s^10+19583*z^12*x^5*y^10*s^10+39127*z^5*x^5*y^10*s^10+39127*z^3*x^5*y^10*s^10+39739*z^9*x^4*y^10*s^10+41743*z^4*x^4*y^10*s^10+41743*z^3*x^8*y^9*s^10+39127*z^4*x^7*y^9*s^10+39127*z^5*x^6*y^9*s^10+39739*z^11*x^5*y^9*s^10+33115*z^9*x^5*y^9*s^10+41743*z^6*x^5*y^9*s^10+33115*z^10*x^4*y^9*s^10+33115*z^11*x^6*y^8*s^10+33115*z^9*x^6*y^8*s^10+33115*z^12*x^5*y^8*s^10+23179*z^10*x^5*y^8*s^10+33115*z^11*x^4*y^8*s^10+33115*z^11*x^7*y^7*s^10+39739*z^9*x^7*y^7*s^10+23179*z^12*x^6*y^7*s^10+33115*z^10*x^6*y^7*s^10+33115*z^13*x^5*y^7*s^10+33115*z^11*x^5*y^7*s^10+39739*z^12*x^4*y^7*s^10+39739*z^11*x^8*y^6*s^10+33115*z^12*x^7*y^6*s^10+33115*z^13*x^6*y^6*s^10+39739*z^14*x^5*y^6*s^10+19583*z^8*x^5*y^15*s^9+19583*z^10*x^6*y^14*s^9+15698*z^8*x^6*y^14*s^9+15698*z^9*x^5*y^14*s^9+15698*z^10*x^7*y^13*s^9+15698*z^8*x^7*y^13*s^9+15698*z^11*x^6*y^13*s^9+31396*z^9*x^6*y^13*s^9+15698*z^10*x^5*y^13*s^9+15698*z^10*x^8*y^12*s^9+19583*z^8*x^8*y^12*s^9+31396*z^11*x^7*y^12*s^9+15698*z^9*x^7*y^12*s^9+15698*z^12*x^6*y^12*s^9+15698*z^10*x^6*y^12*s^9+41743*z*x^6*y^12*s^9+19583*z^11*x^5*y^12*s^9+19583*z^10*x^9*y^11*s^9+15698*z^11*x^8*y^11*s^9+15698*z^12*x^7*y^11*s^9+39127*z*x^7*y^11*s^9+19583*z^13*x^6*y^11*s^9+39127*z^2*x^6*y^11*s^9+41743*z^3*x^5*y^11*s^9+41743*z*x^4*y^11*s^9+39127*z*x^8*y^10*s^9+35203*z^2*x^7*y^10*s^9+35203*z^3*x^6*y^10*s^9+39127*z^4*x^5*y^10*s^9+39127*z*x^5*y^10*s^9+39127*z^2*x^4*y^10*s^9+41743*z*x^9*y^9*s^9+39127*z^2*x^8*y^9*s^9+35203*z^3*x^7*y^9*s^9+39739*z^9*x^6*y^9*s^9+33895*z^4*x^6*y^9*s^9+39127*z*x^6*y^9*s^9+39127*z^5*x^5*y^9*s^9+35203*z^2*x^5*y^9*s^9+39127*z^3*x^4*y^9*s^9+41743*z^3*x^8*y^8*s^9+33115*z^9*x^7*y^8*s^9+39127*z^4*x^7*y^8*s^9+41743*z*x^7*y^8*s^9+33115*z^10*x^6*y^8*s^9+39127*z^5*x^6*y^8*s^9+39127*z^2*x^6*y^8*s^9+39739*z^11*x^5*y^8*s^9+41743*z^6*x^5*y^8*s^9+39127*z^3*x^5*y^8*s^9+39739*z^9*x^4*y^8*s^9+41743*z^4*x^4*y^8*s^9+33115*z^9*x^8*y^7*s^9+23179*z^10*x^7*y^7*s^9+23179*z^11*x^6*y^7*s^9+33115*z^12*x^5*y^7*s^9+33115*z^9*x^5*y^7*s^9+33115*z^10*x^4*y^7*s^9+39739*z^9*x^9*y^6*s^9+33115*z^10*x^8*y^6*s^9+23179*z^11*x^7*y^6*s^9+19867*z^12*x^6*y^6*s^9+33115*z^9*x^6*y^6*s^9+33115*z^13*x^5*y^6*s^9+23179*z^10*x^5*y^6*s^9+33115*z^11*x^4*y^6*s^9+39739*z^11*x^8*y^5*s^9+33115*z^12*x^7*y^5*s^9+39739*z^9*x^7*y^5*s^9+33115*z^13*x^6*y^5*s^9+33115*z^10*x^6*y^5*s^9+39739*z^14*x^5*y^5*s^9+33115*z^11*x^5*y^5*s^9+39739*z^12*x^4*y^5*s^9+19583*z^8*x^7*y^14*s^8+15698*z^8*x^8*y^13*s^8+15698*z^9*x^7*y^13*s^8+19583*z^10*x^6*y^13*s^8+19583*z^8*x^5*y^13*s^8+15698*z^8*x^9*y^12*s^8+31396*z^9*x^8*y^12*s^8+31396*z^10*x^7*y^12*s^8+15698*z^11*x^6*y^12*s^8+15698*z^8*x^6*y^12*s^8+15698*z^9*x^5*y^12*s^8+19583*z^8*x^10*y^11*s^8+15698*z^9*x^9*y^11*s^8+31396*z^10*x^8*y^11*s^8+7928*z^11*x^7*y^11*s^8+15698*z^8*x^7*y^11*s^8+15698*z^12*x^6*y^11*s^8+31396*z^9*x^6*y^11*s^8+15698*z^10*x^5*y^11*s^8+19583*z^10*x^9*y^10*s^8+15698*z^11*x^8*y^10*s^8+19583*z^8*x^8*y^10*s^8+15698*z^12*x^7*y^10*s^8+15698*z^9*x^7*y^10*s^8+19583*z^13*x^6*y^10*s^8+15698*z^10*x^6*y^10*s^8+19583*z^11*x^5*y^10*s^8+16873*z^3*x^3*y^5*s^8+7568*z^3*x^4*y^4*s^8+7568*z^4*x^3*y^4*s^8+7568*z^3*x^5*y^3*s^8+15136*z^4*x^4*y^3*s^8+7568*z^5*x^3*y^3*s^8+16873*z^3*x^6*y^2*s^8+7568*z^4*x^5*y^2*s^8+7568*z^5*x^4*y^2*s^8+16873*z^6*x^3*y^2*s^8+16873*z^2*x^3*y^7*s^7+16873*z^4*x^4*y^6*s^7+7568*z^2*x^4*y^6*s^7+7568*z^3*x^3*y^6*s^7+7568*z^4*x^5*y^5*s^7+7568*z^2*x^5*y^5*s^7+7568*z^5*x^4*y^5*s^7+15136*z^3*x^4*y^5*s^7+7568*z^4*x^3*y^5*s^7+7568*z^4*x^6*y^4*s^7+16873*z^2*x^6*y^4*s^7+15136*z^5*x^5*y^4*s^7+7568*z^3*x^5*y^4*s^7+7568*z^6*x^4*y^4*s^7+7568*z^4*x^4*y^4*s^7+16873*z^5*x^3*y^4*s^7+16873*z^4*x^7*y^3*s^7+7568*z^5*x^6*y^3*s^7+7568*z^6*x^5*y^3*s^7+16873*z^7*x^4*y^3*s^7+16873*z^2*x^5*y^6*s^6+7568*z^2*x^6*y^5*s^6+7568*z^3*x^5*y^5*s^6+16873*z^4*x^4*y^5*s^6+16873*z^2*x^3*y^5*s^6+7568*z^2*x^7*y^4*s^6+15136*z^3*x^6*y^4*s^6+15136*z^4*x^5*y^4*s^6+7568*z^5*x^4*y^4*s^6+7568*z^2*x^4*y^4*s^6+7568*z^3*x^3*y^4*s^6+16873*z^2*x^8*y^3*s^6+7568*z^3*x^7*y^3*s^6+15136*z^4*x^6*y^3*s^6+32009*z^5*x^5*y^3*s^6+7568*z^2*x^5*y^3*s^6+7568*z^6*x^4*y^3*s^6+15136*z^3*x^4*y^3*s^6+7568*z^4*x^3*y^3*s^6+16873*z^4*x^7*y^2*s^6+7568*z^5*x^6*y^2*s^6+16873*z^2*x^6*y^2*s^6+7568*z^6*x^5*y^2*s^6+7568*z^3*x^5*y^2*s^6+16873*z^7*x^4*y^2*s^6+7568*z^4*x^4*y^2*s^6+16873*z^5*x^3*y^2*s^6+21629*z*x*y^13*s^5+21836*z*x^2*y^12*s^5+21836*z^2*x*y^12*s^5+21836*z*x^3*y^11*s^5+621*z^2*x^2*y^11*s^5+21836*z^3*x*y^11*s^5+21629*z*x^4*y^10*s^5+21836*z^2*x^3*y^10*s^5+21836*z^3*x^2*y^10*s^5+21629*z^4*x*y^10*s^5+24116*z^3*x^5*y^6*s^5+29297*z^3*x^6*y^5*s^5+29297*z^4*x^5*y^5*s^5+13845*z^7*x^4*y^5*s^5+29297*z^3*x^7*y^4*s^5+15543*z^4*x^6*y^4*s^5+41535*z^7*x^5*y^4*s^5+29297*z^5*x^5*y^4*s^5+41535*z^8*x^4*y^4*s^5+24116*z^3*x^8*y^3*s^5+29297*z^4*x^7*y^3*s^5+41535*z^7*x^6*y^3*s^5+29297*z^5*x^6*y^3*s^5+40019*z^8*x^5*y^3*s^5+24116*z^6*x^5*y^3*s^5+41535*z^9*x^4*y^3*s^5+13845*z^7*x^7*y^2*s^5+41535*z^8*x^6*y^2*s^5+41535*z^9*x^5*y^2*s^5+13845*z^10*x^4*y^2*s^5+21629*x*y^15*s^4+21629*z^2*x^2*y^14*s^4+21836*x^2*y^14*s^4+21836*z*x*y^14*s^4+21836*z^2*x^3*y^13*s^4+21836*x^3*y^13*s^4+21836*z^3*x^2*y^13*s^4+621*z*x^2*y^13*s^4+21836*z^2*x*y^13*s^4+8506*z^2*x^8*y^12*s^4+21836*z^2*x^4*y^12*s^4+21629*x^4*y^12*s^4+621*z^3*x^3*y^12*s^4+21836*z*x^3*y^12*s^4+21836*z^4*x^2*y^12*s^4+21836*z^2*x^2*y^12*s^4+21629*z^3*x*y^12*s^4+25518*z^2*x^9*y^11*s^4+25518*z^3*x^8*y^11*s^4+21629*z^2*x^5*y^11*s^4+21836*z^3*x^4*y^11*s^4+21836*z^4*x^3*y^11*s^4+21629*z^5*x^2*y^11*s^4+25518*z^2*x^10*y^10*s^4+7985*z^3*x^9*y^10*s^4+25518*z^4*x^8*y^10*s^4+8506*z^2*x^11*y^9*s^4+25518*z^3*x^10*y^9*s^4+25518*z^4*x^9*y^9*s^4+8506*z^5*x^8*y^9*s^4+24116*z^2*x^5*y^8*s^4+24116*z^4*x^6*y^7*s^4+29297*z^2*x^6*y^7*s^4+29297*z^3*x^5*y^7*s^4+13845*z^6*x^4*y^7*s^4+29297*z^4*x^7*y^6*s^4+29297*z^2*x^7*y^6*s^4+29297*z^5*x^6*y^6*s^4+15543*z^3*x^6*y^6*s^4+13845*z^8*x^5*y^6*s^4+41535*z^6*x^5*y^6*s^4+29297*z^4*x^5*y^6*s^4+41535*z^7*x^4*y^6*s^4+29297*z^4*x^8*y^5*s^4+24116*z^2*x^8*y^5*s^4+15543*z^5*x^7*y^5*s^4+29297*z^3*x^7*y^5*s^4+41535*z^8*x^6*y^5*s^4+27781*z^6*x^6*y^5*s^4+29297*z^4*x^6*y^5*s^4+41535*z^9*x^5*y^5*s^4+40019*z^7*x^5*y^5*s^4+24116*z^5*x^5*y^5*s^4+41535*z^8*x^4*y^5*s^4+24116*z^4*x^9*y^4*s^4+29297*z^5*x^8*y^4*s^4+41535*z^8*x^7*y^4*s^4+91*z^6*x^7*y^4*s^4+40019*z^9*x^6*y^4*s^4+22600*z^7*x^6*y^4*s^4+41535*z^10*x^5*y^4*s^4+41535*z^8*x^5*y^4*s^4+13845*z^9*x^4*y^4*s^4+13845*z^8*x^8*y^3*s^4+41535*z^9*x^7*y^3*s^4+41535*z^10*x^6*y^3*s^4+13845*z^11*x^5*y^3*s^4+8506*z*x^8*y^14*s^3+21629*x^3*y^14*s^3+8506*z^3*x^9*y^13*s^3+25518*z*x^9*y^13*s^3+25518*z^2*x^8*y^13*s^3+21836*x^4*y^13*s^3+21836*z*x^3*y^13*s^3+21629*z^2*x^2*y^13*s^3+21629*x*y^13*s^3+25518*z^3*x^10*y^12*s^3+25518*z*x^10*y^12*s^3+25518*z^4*x^9*y^12*s^3+7985*z^2*x^9*y^12*s^3+25518*z^3*x^8*y^12*s^3+21836*x^5*y^12*s^3+621*z*x^4*y^12*s^3+621*z^2*x^3*y^12*s^3+21836*z^3*x^2*y^12*s^3+21836*x^2*y^12*s^3+21836*z*x*y^12*s^3+25518*z^3*x^11*y^11*s^3+8506*z*x^11*y^11*s^3+7985*z^4*x^10*y^11*s^3+25518*z^2*x^10*y^11*s^3+25518*z^5*x^9*y^11*s^3+25518*z^3*x^9*y^11*s^3+8506*z^4*x^8*y^11*s^3+21629*x^6*y^11*s^3+21836*z*x^5*y^11*s^3+621*z^2*x^4*y^11*s^3+22250*z^3*x^3*y^11*s^3+21836*x^3*y^11*s^3+21836*z^4*x^2*y^11*s^3+621*z*x^2*y^11*s^3+21836*z^2*x*y^11*s^3+8506*z^3*x^12*y^10*s^3+25518*z^4*x^11*y^10*s^3+25518*z^5*x^10*y^10*s^3+8506*z^6*x^9*y^10*s^3+21629*z^2*x^5*y^10*s^3+21836*z^3*x^4*y^10*s^3+21629*x^4*y^10*s^3+21836*z^4*x^3*y^10*s^3+21836*z*x^3*y^10*s^3+21629*z^5*x^2*y^10*s^3+21836*z^2*x^2*y^10*s^3+21629*z^3*x*y^10*s^3+29049*z^7*x^9*y^7*s^3+24116*z^2*x^7*y^7*s^3+1045*z^7*x^10*y^6*s^3+1045*z^8*x^9*y^6*s^3+29297*z^2*x^8*y^6*s^3+29297*z^3*x^7*y^6*s^3+13845*z^6*x^6*y^6*s^3+24116*z^4*x^6*y^6*s^3+24116*z^2*x^5*y^6*s^3+1045*z^7*x^11*y^5*s^3+2090*z^8*x^10*y^5*s^3+1045*z^9*x^9*y^5*s^3+29297*z^2*x^9*y^5*s^3+15543*z^3*x^8*y^5*s^3+41535*z^6*x^7*y^5*s^3+15543*z^4*x^7*y^5*s^3+41535*z^7*x^6*y^5*s^3+29297*z^5*x^6*y^5*s^3+29297*z^2*x^6*y^5*s^3+13845*z^8*x^5*y^5*s^3+29297*z^3*x^5*y^5*s^3+13845*z^6*x^4*y^5*s^3+29049*z^7*x^12*y^4*s^3+1045*z^8*x^11*y^4*s^3+1045*z^9*x^10*y^4*s^3+24116*z^2*x^10*y^4*s^3+29049*z^10*x^9*y^4*s^3+29297*z^3*x^9*y^4*s^3+41535*z^6*x^8*y^4*s^3+15543*z^4*x^8*y^4*s^3+40019*z^7*x^7*y^4*s^3+39659*z^5*x^7*y^4*s^3+29297*z^2*x^7*y^4*s^3+40019*z^8*x^6*y^4*s^3+29297*z^6*x^6*y^4*s^3+15543*z^3*x^6*y^4*s^3+41535*z^9*x^5*y^4*s^3+41535*z^6*x^5*y^4*s^3+29297*z^4*x^5*y^4*s^3+41535*z^7*x^4*y^4*s^3+13845*z^6*x^9*y^3*s^3+24116*z^4*x^9*y^3*s^3+41535*z^7*x^8*y^3*s^3+29297*z^5*x^8*y^3*s^3+24116*z^2*x^8*y^3*s^3+40019*z^8*x^7*y^3*s^3+29297*z^6*x^7*y^3*s^3+29297*z^3*x^7*y^3*s^3+10813*z^9*x^6*y^3*s^3+24116*z^7*x^6*y^3*s^3+41535*z^6*x^6*y^3*s^3+29297*z^4*x^6*y^3*s^3+41535*z^10*x^5*y^3*s^3+40019*z^7*x^5*y^3*s^3+24116*z^5*x^5*y^3*s^3+41535*z^8*x^4*y^3*s^3+13845*z^8*x^8*y^2*s^3+41535*z^9*x^7*y^2*s^3+13845*z^6*x^7*y^2*s^3+41535*z^10*x^6*y^2*s^3+41535*z^7*x^6*y^2*s^3+13845*z^11*x^5*y^2*s^3+41535*z^8*x^5*y^2*s^3+13845*z^9*x^4*y^2*s^3+8506*z*x^10*y^13*s^2+25518*z*x^11*y^12*s^2+25518*z^2*x^10*y^12*s^2+8506*z^3*x^9*y^12*s^2+8506*z*x^8*y^12*s^2+25518*z*x^12*y^11*s^2+7985*z^2*x^11*y^11*s^2+7985*z^3*x^10*y^11*s^2+25518*z^4*x^9*y^11*s^2+25518*z*x^9*y^11*s^2+25518*z^2*x^8*y^11*s^2+8506*z*x^13*y^10*s^2+25518*z^2*x^12*y^10*s^2+7985*z^3*x^11*y^10*s^2+16491*z^4*x^10*y^10*s^2+25518*z*x^10*y^10*s^2+25518*z^5*x^9*y^10*s^2+7985*z^2*x^9*y^10*s^2+25518*z^3*x^8*y^10*s^2+8506*z^3*x^12*y^9*s^2+25518*z^4*x^11*y^9*s^2+8506*z*x^11*y^9*s^2+25518*z^5*x^10*y^9*s^2+25518*z^2*x^10*y^9*s^2+37555*z^6*x^9*y^9*s^2+25518*z^3*x^9*y^9*s^2+8506*z^4*x^8*y^9*s^2+29049*z^8*x^10*y^8*s^2+1045*z^6*x^10*y^8*s^2+1045*z^7*x^9*y^8*s^2+1045*z^8*x^11*y^7*s^2+1045*z^6*x^11*y^7*s^2+1045*z^9*x^10*y^7*s^2+2090*z^7*x^10*y^7*s^2+1045*z^8*x^9*y^7*s^2+1045*z^8*x^12*y^6*s^2+29049*z^6*x^12*y^6*s^2+2090*z^9*x^11*y^6*s^2+1045*z^7*x^11*y^6*s^2+1045*z^10*x^10*y^6*s^2+1045*z^8*x^10*y^6*s^2+29049*z^9*x^9*y^6*s^2+29049*z^8*x^13*y^5*s^2+1045*z^9*x^12*y^5*s^2+1045*z^10*x^11*y^5*s^2+29049*z^11*x^10*y^5*s^2+29049*z^6*x^11*y^8*s+1045*z^6*x^12*y^7*s+1045*z^7*x^11*y^7*s+29049*z^8*x^10*y^7*s+29049*z^6*x^9*y^7*s+1045*z^6*x^13*y^6*s+2090*z^7*x^12*y^6*s+2090*z^8*x^11*y^6*s+1045*z^9*x^10*y^6*s+1045*z^6*x^10*y^6*s+1045*z^7*x^9*y^6*s+29049*z^6*x^14*y^5*s+1045*z^7*x^13*y^5*s+2090*z^8*x^12*y^5*s+31139*z^9*x^11*y^5*s+1045*z^6*x^11*y^5*s+1045*z^10*x^10*y^5*s+2090*z^7*x^10*y^5*s+1045*z^8*x^9*y^5*s+29049*z^8*x^13*y^4*s+1045*z^9*x^12*y^4*s+29049*z^6*x^12*y^4*s+1045*z^10*x^11*y^4*s+1045*z^7*x^11*y^4*s+29049*z^11*x^10*y^4*s+1045*z^8*x^10*y^4*s+29049*z^9*x^9*y^4*s)", vars, ctx);
        time = check_omega(2, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#14: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(16873*z^4*x^6*y^7*s^11+7568*z^4*x^7*y^6*s^11+7568*z^5*x^6*y^6*s^11+7568*z^4*x^8*y^5*s^11+15136*z^5*x^7*y^5*s^11+7568*z^6*x^6*y^5*s^11+16873*z^4*x^9*y^4*s^11+7568*z^5*x^8*y^4*s^11+7568*z^6*x^7*y^4*s^11+16873*z^7*x^6*y^4*s^11+16873*z^3*x^6*y^9*s^10+16873*z^5*x^7*y^8*s^10+7568*z^3*x^7*y^8*s^10+7568*z^4*x^6*y^8*s^10+7568*z^5*x^8*y^7*s^10+7568*z^3*x^8*y^7*s^10+7568*z^6*x^7*y^7*s^10+15136*z^4*x^7*y^7*s^10+7568*z^5*x^6*y^7*s^10+7568*z^5*x^9*y^6*s^10+16873*z^3*x^9*y^6*s^10+15136*z^6*x^8*y^6*s^10+7568*z^4*x^8*y^6*s^10+7568*z^7*x^7*y^6*s^10+7568*z^5*x^7*y^6*s^10+16873*z^6*x^6*y^6*s^10+16873*z^5*x^10*y^5*s^10+7568*z^6*x^9*y^5*s^10+7568*z^7*x^8*y^5*s^10+16873*z^8*x^7*y^5*s^10+41743*z^8*x^8*y^13*s^9+39127*z^8*x^9*y^12*s^9+39127*z^9*x^8*y^12*s^9+39127*z^8*x^10*y^11*s^9+35203*z^9*x^9*y^11*s^9+19583*z^3*x^9*y^11*s^9+39127*z^10*x^8*y^11*s^9+41743*z^8*x^11*y^10*s^9+39127*z^9*x^10*y^10*s^9+15698*z^3*x^10*y^10*s^9+39127*z^10*x^9*y^10*s^9+15698*z^4*x^9*y^10*s^9+41743*z^11*x^8*y^10*s^9+15698*z^3*x^11*y^9*s^9+31396*z^4*x^10*y^9*s^9+15698*z^5*x^9*y^9*s^9+19583*z^3*x^12*y^8*s^9+15698*z^4*x^11*y^8*s^9+15698*z^5*x^10*y^8*s^9+19583*z^6*x^9*y^8*s^9+16873*z^3*x^8*y^8*s^9+7568*z^3*x^9*y^7*s^9+7568*z^4*x^8*y^7*s^9+16873*z^5*x^7*y^7*s^9+16873*z^3*x^6*y^7*s^9+7568*z^3*x^10*y^6*s^9+15136*z^4*x^9*y^6*s^9+15136*z^5*x^8*y^6*s^9+7568*z^6*x^7*y^6*s^9+7568*z^3*x^7*y^6*s^9+7568*z^4*x^6*y^6*s^9+16873*z^3*x^11*y^5*s^9+7568*z^4*x^10*y^5*s^9+15136*z^5*x^9*y^5*s^9+32009*z^6*x^8*y^5*s^9+7568*z^3*x^8*y^5*s^9+7568*z^7*x^7*y^5*s^9+15136*z^4*x^7*y^5*s^9+7568*z^5*x^6*y^5*s^9+16873*z^5*x^10*y^4*s^9+7568*z^6*x^9*y^4*s^9+16873*z^3*x^9*y^4*s^9+7568*z^7*x^8*y^4*s^9+7568*z^4*x^8*y^4*s^9+16873*z^8*x^7*y^4*s^9+7568*z^5*x^7*y^4*s^9+16873*z^6*x^6*y^4*s^9+41743*z^7*x^8*y^15*s^8+41743*z^9*x^9*y^14*s^8+39127*z^7*x^9*y^14*s^8+39127*z^8*x^8*y^14*s^8+39127*z^9*x^10*y^13*s^8+39127*z^7*x^10*y^13*s^8+39127*z^10*x^9*y^13*s^8+35203*z^8*x^9*y^13*s^8+19583*z^2*x^9*y^13*s^8+39127*z^9*x^8*y^13*s^8+39127*z^9*x^11*y^12*s^8+41743*z^7*x^11*y^12*s^8+35203*z^10*x^10*y^12*s^8+39127*z^8*x^10*y^12*s^8+19583*z^4*x^10*y^12*s^8+15698*z^2*x^10*y^12*s^8+39127*z^11*x^9*y^12*s^8+39127*z^9*x^9*y^12*s^8+15698*z^3*x^9*y^12*s^8+41743*z^10*x^8*y^12*s^8+41743*z^9*x^12*y^11*s^8+39127*z^10*x^11*y^11*s^8+15698*z^4*x^11*y^11*s^8+15698*z^2*x^11*y^11*s^8+39127*z^11*x^10*y^11*s^8+15698*z^5*x^10*y^11*s^8+31396*z^3*x^10*y^11*s^8+41743*z^12*x^9*y^11*s^8+15698*z^4*x^9*y^11*s^8+15698*z^4*x^12*y^10*s^8+19583*z^2*x^12*y^10*s^8+31396*z^5*x^11*y^10*s^8+15698*z^3*x^11*y^10*s^8+15698*z^6*x^10*y^10*s^8+15698*z^4*x^10*y^10*s^8+19583*z^5*x^9*y^10*s^8+21629*z^8*x*y^10*s^8+19583*z^4*x^13*y^9*s^8+15698*z^5*x^12*y^9*s^8+15698*z^6*x^11*y^9*s^8+19583*z^7*x^10*y^9*s^8+21836*z^8*x^2*y^9*s^8+21836*z^9*x*y^9*s^8+21836*z^8*x^3*y^8*s^8+621*z^9*x^2*y^8*s^8+21836*z^10*x*y^8*s^8+21629*z^8*x^4*y^7*s^8+21836*z^9*x^3*y^7*s^8+21836*z^10*x^2*y^7*s^8+21629*z^11*x*y^7*s^8+41743*z^7*x^10*y^14*s^7+39127*z^7*x^11*y^13*s^7+39127*z^8*x^10*y^13*s^7+41743*z^9*x^9*y^13*s^7+41743*z^7*x^8*y^13*s^7+29049*z^10*x^7*y^13*s^7+39127*z^7*x^12*y^12*s^7+35203*z^8*x^11*y^12*s^7+19583*z^2*x^11*y^12*s^7+35203*z^9*x^10*y^12*s^7+39127*z^10*x^9*y^12*s^7+39127*z^7*x^9*y^12*s^7+1045*z^10*x^8*y^12*s^7+39127*z^8*x^8*y^12*s^7+1045*z^11*x^7*y^12*s^7+21629*z^7*x*y^12*s^7+41743*z^7*x^13*y^11*s^7+39127*z^8*x^12*y^11*s^7+15698*z^2*x^12*y^11*s^7+35203*z^9*x^11*y^11*s^7+15698*z^3*x^11*y^11*s^7+33895*z^10*x^10*y^11*s^7+39127*z^7*x^10*y^11*s^7+19583*z^4*x^10*y^11*s^7+39127*z^11*x^9*y^11*s^7+1045*z^10*x^9*y^11*s^7+35203*z^8*x^9*y^11*s^7+19583*z^2*x^9*y^11*s^7+2090*z^11*x^8*y^11*s^7+39127*z^9*x^8*y^11*s^7+1045*z^12*x^7*y^11*s^7+21629*z^9*x^2*y^11*s^7+21836*z^7*x^2*y^11*s^7+21836*z^8*x*y^11*s^7+15698*z^2*x^13*y^10*s^7+41743*z^9*x^12*y^10*s^7+31396*z^3*x^12*y^10*s^7+39127*z^10*x^11*y^10*s^7+41743*z^7*x^11*y^10*s^7+31396*z^4*x^11*y^10*s^7+39127*z^11*x^10*y^10*s^7+29049*z^10*x^10*y^10*s^7+39127*z^8*x^10*y^10*s^7+15698*z^5*x^10*y^10*s^7+15698*z^2*x^10*y^10*s^7+41743*z^12*x^9*y^10*s^7+1045*z^11*x^9*y^10*s^7+39127*z^9*x^9*y^10*s^7+15698*z^3*x^9*y^10*s^7+1045*z^12*x^8*y^10*s^7+41743*z^10*x^8*y^10*s^7+29049*z^13*x^7*y^10*s^7+21836*z^9*x^3*y^10*s^7+21836*z^7*x^3*y^10*s^7+21836*z^10*x^2*y^10*s^7+621*z^8*x^2*y^10*s^7+21836*z^9*x*y^10*s^7+19583*z^2*x^14*y^9*s^7+15698*z^3*x^13*y^9*s^7+31396*z^4*x^12*y^9*s^7+7928*z^5*x^11*y^9*s^7+15698*z^2*x^11*y^9*s^7+15698*z^6*x^10*y^9*s^7+31396*z^3*x^10*y^9*s^7+15698*z^4*x^9*y^9*s^7+21836*z^9*x^4*y^9*s^7+21629*z^7*x^4*y^9*s^7+621*z^10*x^3*y^9*s^7+21836*z^8*x^3*y^9*s^7+21836*z^11*x^2*y^9*s^7+21836*z^9*x^2*y^9*s^7+21629*z^10*x*y^9*s^7+19583*z^4*x^13*y^8*s^7+15698*z^5*x^12*y^8*s^7+19583*z^2*x^12*y^8*s^7+15698*z^6*x^11*y^8*s^7+15698*z^3*x^11*y^8*s^7+19583*z^7*x^10*y^8*s^7+15698*z^4*x^10*y^8*s^7+39739*z^6*x^9*y^8*s^7+19583*z^5*x^9*y^8*s^7+21629*z^9*x^5*y^8*s^7+21836*z^10*x^4*y^8*s^7+21836*z^11*x^3*y^8*s^7+21629*z^12*x^2*y^8*s^7+33115*z^6*x^10*y^7*s^7+33115*z^7*x^9*y^7*s^7+33115*z^6*x^11*y^6*s^7+23179*z^7*x^10*y^6*s^7+33115*z^8*x^9*y^6*s^7+39739*z^6*x^12*y^5*s^7+33115*z^7*x^11*y^5*s^7+33115*z^8*x^10*y^5*s^7+39739*z^9*x^9*y^5*s^7+29049*z^9*x^7*y^15*s^6+29049*z^11*x^8*y^14*s^6+1045*z^9*x^8*y^14*s^6+1045*z^10*x^7*y^14*s^6+1045*z^11*x^9*y^13*s^6+1045*z^9*x^9*y^13*s^6+1045*z^12*x^8*y^13*s^6+2090*z^10*x^8*y^13*s^6+1045*z^11*x^7*y^13*s^6+8506*z^7*x^6*y^13*s^6+1045*z^11*x^10*y^12*s^6+29049*z^9*x^10*y^12*s^6+2090*z^12*x^9*y^12*s^6+1045*z^10*x^9*y^12*s^6+1045*z^13*x^8*y^12*s^6+1045*z^11*x^8*y^12*s^6+29049*z^12*x^7*y^12*s^6+25518*z^7*x^7*y^12*s^6+25518*z^8*x^6*y^12*s^6+29049*z^11*x^11*y^11*s^6+1045*z^12*x^10*y^11*s^6+1045*z^13*x^9*y^11*s^6+29049*z^14*x^8*y^11*s^6+25518*z^7*x^8*y^11*s^6+7985*z^8*x^7*y^11*s^6+25518*z^9*x^6*y^11*s^6+21629*z^7*x^3*y^11*s^6+8506*z^7*x^9*y^10*s^6+39739*z^5*x^9*y^10*s^6+25518*z^8*x^8*y^10*s^6+25518*z^9*x^7*y^10*s^6+8506*z^10*x^6*y^10*s^6+21836*z^7*x^4*y^10*s^6+21836*z^8*x^3*y^10*s^6+21629*z^9*x^2*y^10*s^6+21629*z^7*x*y^10*s^6+39739*z^7*x^10*y^9*s^6+33115*z^5*x^10*y^9*s^6+33115*z^6*x^9*y^9*s^6+21836*z^7*x^5*y^9*s^6+621*z^8*x^4*y^9*s^6+621*z^9*x^3*y^9*s^6+21836*z^10*x^2*y^9*s^6+21836*z^7*x^2*y^9*s^6+21836*z^8*x*y^9*s^6+33115*z^7*x^11*y^8*s^6+33115*z^5*x^11*y^8*s^6+33115*z^8*x^10*y^8*s^6+23179*z^6*x^10*y^8*s^6+33115*z^7*x^9*y^8*s^6+21629*z^7*x^6*y^8*s^6+21836*z^8*x^5*y^8*s^6+621*z^9*x^4*y^8*s^6+22250*z^10*x^3*y^8*s^6+21836*z^7*x^3*y^8*s^6+21836*z^11*x^2*y^8*s^6+621*z^8*x^2*y^8*s^6+21836*z^9*x*y^8*s^6+33115*z^7*x^12*y^7*s^6+39739*z^5*x^12*y^7*s^6+23179*z^8*x^11*y^7*s^6+33115*z^6*x^11*y^7*s^6+33115*z^9*x^10*y^7*s^6+33115*z^7*x^10*y^7*s^6+39739*z^8*x^9*y^7*s^6+21629*z^9*x^5*y^7*s^6+21836*z^10*x^4*y^7*s^6+21629*z^7*x^4*y^7*s^6+21836*z^11*x^3*y^7*s^6+21836*z^8*x^3*y^7*s^6+21629*z^12*x^2*y^7*s^6+21836*z^9*x^2*y^7*s^6+21629*z^10*x*y^7*s^6+39739*z^7*x^13*y^6*s^6+33115*z^8*x^12*y^6*s^6+33115*z^9*x^11*y^6*s^6+39739*z^10*x^10*y^6*s^6+8506*z^6*x^6*y^15*s^5+29049*z^9*x^9*y^14*s^5+8506*z^8*x^7*y^14*s^5+25518*z^6*x^7*y^14*s^5+25518*z^7*x^6*y^14*s^5+1045*z^9*x^10*y^13*s^5+1045*z^10*x^9*y^13*s^5+29049*z^11*x^8*y^13*s^5+25518*z^8*x^8*y^13*s^5+25518*z^6*x^8*y^13*s^5+11516*z^9*x^7*y^13*s^5+7985*z^7*x^7*y^13*s^5+25518*z^8*x^6*y^13*s^5+1045*z^9*x^11*y^12*s^5+2090*z^10*x^10*y^12*s^5+2090*z^11*x^9*y^12*s^5+25518*z^8*x^9*y^12*s^5+8506*z^6*x^9*y^12*s^5+1045*z^12*x^8*y^12*s^5+9030*z^9*x^8*y^12*s^5+25518*z^7*x^8*y^12*s^5+26563*z^10*x^7*y^12*s^5+25518*z^8*x^7*y^12*s^5+8506*z^9*x^6*y^12*s^5+29049*z^9*x^12*y^11*s^5+1045*z^10*x^11*y^11*s^5+2090*z^11*x^10*y^11*s^5+8506*z^8*x^10*y^11*s^5+31139*z^12*x^9*y^11*s^5+26563*z^9*x^9*y^11*s^5+1045*z^13*x^8*y^11*s^5+27608*z^10*x^8*y^11*s^5+9551*z^11*x^7*y^11*s^5+29049*z^11*x^11*y^10*s^5+1045*z^12*x^10*y^10*s^5+29049*z^9*x^10*y^10*s^5+1045*z^13*x^9*y^10*s^5+1045*z^10*x^9*y^10*s^5+29049*z^14*x^8*y^10*s^5+1045*z^11*x^8*y^10*s^5+29049*z^12*x^7*y^10*s^5+39739*z^5*x^11*y^9*s^5+33115*z^5*x^12*y^8*s^5+33115*z^6*x^11*y^8*s^5+39739*z^7*x^10*y^8*s^5+39739*z^5*x^9*y^8*s^5+33115*z^5*x^13*y^7*s^5+23179*z^6*x^12*y^7*s^5+23179*z^7*x^11*y^7*s^5+33115*z^8*x^10*y^7*s^5+33115*z^5*x^10*y^7*s^5+33115*z^6*x^9*y^7*s^5+39739*z^5*x^14*y^6*s^5+33115*z^6*x^13*y^6*s^5+23179*z^7*x^12*y^6*s^5+19867*z^8*x^11*y^6*s^5+33115*z^5*x^11*y^6*s^5+33115*z^9*x^10*y^6*s^5+23179*z^6*x^10*y^6*s^5+33115*z^7*x^9*y^6*s^5+39739*z^7*x^13*y^5*s^5+33115*z^8*x^12*y^5*s^5+39739*z^5*x^12*y^5*s^5+33115*z^9*x^11*y^5*s^5+33115*z^6*x^11*y^5*s^5+39739*z^10*x^10*y^5*s^5+33115*z^7*x^10*y^5*s^5+39739*z^8*x^9*y^5*s^5+13845*z^4*x^4*y^5*s^5+41535*z^4*x^5*y^4*s^5+41535*z^5*x^4*y^4*s^5+41535*z^4*x^6*y^3*s^5+40019*z^5*x^5*y^3*s^5+41535*z^6*x^4*y^3*s^5+13845*z^4*x^7*y^2*s^5+41535*z^5*x^6*y^2*s^5+41535*z^6*x^5*y^2*s^5+13845*z^7*x^4*y^2*s^5+8506*z^6*x^8*y^14*s^4+25518*z^6*x^9*y^13*s^4+25518*z^7*x^8*y^13*s^4+8506*z^8*x^7*y^13*s^4+8506*z^6*x^6*y^13*s^4+25518*z^6*x^10*y^12*s^4+7985*z^7*x^9*y^12*s^4+7985*z^8*x^8*y^12*s^4+25518*z^9*x^7*y^12*s^4+25518*z^6*x^7*y^12*s^4+25518*z^7*x^6*y^12*s^4+8506*z^6*x^11*y^11*s^4+25518*z^7*x^10*y^11*s^4+7985*z^8*x^9*y^11*s^4+16491*z^9*x^8*y^11*s^4+25518*z^6*x^8*y^11*s^4+25518*z^10*x^7*y^11*s^4+7985*z^7*x^7*y^11*s^4+25518*z^8*x^6*y^11*s^4+8506*z^8*x^10*y^10*s^4+25518*z^9*x^9*y^10*s^4+8506*z^6*x^9*y^10*s^4+25518*z^10*x^8*y^10*s^4+25518*z^7*x^8*y^10*s^4+8506*z^11*x^7*y^10*s^4+25518*z^8*x^7*y^10*s^4+8506*z^9*x^6*y^10*s^4+24116*z^5*x^10*y^9*s^4+29297*z^5*x^11*y^8*s^4+29297*z^6*x^10*y^8*s^4+29297*z^5*x^12*y^7*s^4+15543*z^6*x^11*y^7*s^4+29297*z^7*x^10*y^7*s^4+13845*z^3*x^4*y^7*s^4+24116*z^5*x^13*y^6*s^4+29297*z^6*x^12*y^6*s^4+29297*z^7*x^11*y^6*s^4+24116*z^8*x^10*y^6*s^4+13845*z^5*x^5*y^6*s^4+41535*z^3*x^5*y^6*s^4+41535*z^4*x^4*y^6*s^4+41535*z^5*x^6*y^5*s^4+41535*z^3*x^6*y^5*s^4+41535*z^6*x^5*y^5*s^4+40019*z^4*x^5*y^5*s^4+41535*z^5*x^4*y^5*s^4+41535*z^5*x^7*y^4*s^4+13845*z^3*x^7*y^4*s^4+40019*z^6*x^6*y^4*s^4+41535*z^4*x^6*y^4*s^4+41535*z^7*x^5*y^4*s^4+41535*z^5*x^5*y^4*s^4+13845*z^6*x^4*y^4*s^4+13845*z^5*x^8*y^3*s^4+41535*z^6*x^7*y^3*s^4+41535*z^7*x^6*y^3*s^4+13845*z^8*x^5*y^3*s^4+24116*z^4*x^10*y^11*s^3+24116*z^6*x^11*y^10*s^3+29297*z^4*x^11*y^10*s^3+29297*z^5*x^10*y^10*s^3+29297*z^6*x^12*y^9*s^3+29297*z^4*x^12*y^9*s^3+29297*z^7*x^11*y^9*s^3+15543*z^5*x^11*y^9*s^3+29297*z^6*x^10*y^9*s^3+29297*z^6*x^13*y^8*s^3+24116*z^4*x^13*y^8*s^3+15543*z^7*x^12*y^8*s^3+29297*z^5*x^12*y^8*s^3+29297*z^8*x^11*y^8*s^3+29297*z^6*x^11*y^8*s^3+24116*z^7*x^10*y^8*s^3+24116*z^6*x^14*y^7*s^3+29297*z^7*x^13*y^7*s^3+29297*z^8*x^12*y^7*s^3+24116*z^9*x^11*y^7*s^3+13845*z^3*x^6*y^6*s^3+41535*z^3*x^7*y^5*s^3+41535*z^4*x^6*y^5*s^3+13845*z^5*x^5*y^5*s^3+13845*z^3*x^4*y^5*s^3+41535*z^3*x^8*y^4*s^3+40019*z^4*x^7*y^4*s^3+40019*z^5*x^6*y^4*s^3+41535*z^6*x^5*y^4*s^3+41535*z^3*x^5*y^4*s^3+41535*z^4*x^4*y^4*s^3+13845*z^3*x^9*y^3*s^3+41535*z^4*x^8*y^3*s^3+40019*z^5*x^7*y^3*s^3+10813*z^6*x^6*y^3*s^3+41535*z^3*x^6*y^3*s^3+41535*z^7*x^5*y^3*s^3+40019*z^4*x^5*y^3*s^3+41535*z^5*x^4*y^3*s^3+13845*z^5*x^8*y^2*s^3+41535*z^6*x^7*y^2*s^3+13845*z^3*x^7*y^2*s^3+41535*z^7*x^6*y^2*s^3+41535*z^4*x^6*y^2*s^3+13845*z^8*x^5*y^2*s^3+41535*z^5*x^5*y^2*s^3+13845*z^6*x^4*y^2*s^3+24116*z^4*x^12*y^10*s^2+29297*z^4*x^13*y^9*s^2+29297*z^5*x^12*y^9*s^2+24116*z^6*x^11*y^9*s^2+24116*z^4*x^10*y^9*s^2+29297*z^4*x^14*y^8*s^2+15543*z^5*x^13*y^8*s^2+15543*z^6*x^12*y^8*s^2+29297*z^7*x^11*y^8*s^2+29297*z^4*x^11*y^8*s^2+29297*z^5*x^10*y^8*s^2+24116*z^4*x^15*y^7*s^2+29297*z^5*x^14*y^7*s^2+15543*z^6*x^13*y^7*s^2+39659*z^7*x^12*y^7*s^2+29297*z^4*x^12*y^7*s^2+29297*z^8*x^11*y^7*s^2+15543*z^5*x^11*y^7*s^2+29297*z^6*x^10*y^7*s^2+24116*z^6*x^14*y^6*s^2+29297*z^7*x^13*y^6*s^2+24116*z^4*x^13*y^6*s^2+29297*z^8*x^12*y^6*s^2+29297*z^5*x^12*y^6*s^2+24116*z^9*x^11*y^6*s^2+29297*z^6*x^11*y^6*s^2+24116*z^7*x^10*y^6*s^2)*(22750*x^4*y^4*s^3+41200*z^2*x*y^3*s^3+4017*x*y^3*s^3+26755*z*x^4*y*s^3+42152*s+27324*x^4*y^3+40575*x^2*y^3+14244*z^3*x^2*y^2+14342*z*x^2*y)", vars, ctx);
        time = check_omega(2, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#15: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(22750*x^2*y^4*s^4+14244*z^3*y^4*s^4+27324*z^2*x^3*y^4*s^3+4017*y^4*s^2+14342*z^3*x*s^2+40575*x^3*y^4*s+26755*z^2*x*y^4*s+41200*z^3*y*s+42152*z^2*x^4*y^3)*(29049*z^6*x*y^11*s^11+1045*z^6*x^2*y^10*s^11+1045*z^7*x*y^10*s^11+1045*z^6*x^3*y^9*s^11+21629*z^4*x^3*y^9*s^11+2090*z^7*x^2*y^9*s^11+1045*z^8*x*y^9*s^11+29049*z^6*x^4*y^8*s^11+21836*z^4*x^4*y^8*s^11+1045*z^7*x^3*y^8*s^11+21836*z^5*x^3*y^8*s^11+1045*z^8*x^2*y^8*s^11+29049*z^9*x*y^8*s^11+21836*z^4*x^5*y^7*s^11+621*z^5*x^4*y^7*s^11+21836*z^6*x^3*y^7*s^11+21629*z^4*x^6*y^6*s^11+21836*z^5*x^5*y^6*s^11+21836*z^6*x^4*y^6*s^11+21629*z^7*x^3*y^6*s^11+29049*z^5*x*y^13*s^10+29049*z^7*x^2*y^12*s^10+1045*z^5*x^2*y^12*s^10+1045*z^6*x*y^12*s^10+1045*z^7*x^3*y^11*s^10+1045*z^5*x^3*y^11*s^10+21629*z^3*x^3*y^11*s^10+1045*z^8*x^2*y^11*s^10+2090*z^6*x^2*y^11*s^10+1045*z^7*x*y^11*s^10+1045*z^7*x^4*y^10*s^10+7627*z^5*x^4*y^10*s^10+21836*z^3*x^4*y^10*s^10+2090*z^8*x^3*y^10*s^10+1045*z^6*x^3*y^10*s^10+21836*z^4*x^3*y^10*s^10+1045*z^9*x^2*y^10*s^10+1045*z^7*x^2*y^10*s^10+29049*z^8*x*y^10*s^10+29049*z^7*x^5*y^9*s^10+21836*z^5*x^5*y^9*s^10+21836*z^3*x^5*y^9*s^10+1045*z^8*x^4*y^9*s^10+21836*z^6*x^4*y^9*s^10+621*z^4*x^4*y^9*s^10+1045*z^9*x^3*y^9*s^10+21836*z^5*x^3*y^9*s^10+29049*z^10*x^2*y^9*s^10+21836*z^5*x^6*y^8*s^10+21629*z^3*x^6*y^8*s^10+621*z^6*x^5*y^8*s^10+21836*z^4*x^5*y^8*s^10+21836*z^7*x^4*y^8*s^10+21836*z^5*x^4*y^8*s^10+21629*z^6*x^3*y^8*s^10+21629*z^5*x^7*y^7*s^10+21836*z^6*x^6*y^7*s^10+21836*z^7*x^5*y^7*s^10+21629*z^8*x^4*y^7*s^10+16873*z^8*x^8*y^6*s^10+7568*z^8*x^9*y^5*s^10+7568*z^9*x^8*y^5*s^10+7568*z^8*x^10*y^4*s^10+15136*z^9*x^9*y^4*s^10+7568*z^10*x^8*y^4*s^10+16873*z^8*x^11*y^3*s^10+7568*z^9*x^10*y^3*s^10+7568*z^10*x^9*y^3*s^10+16873*z^11*x^8*y^3*s^10+29049*z^5*x^3*y^12*s^9+1045*z^5*x^4*y^11*s^9+1045*z^6*x^3*y^11*s^9+29049*z^7*x^2*y^11*s^9+29049*z^5*x*y^11*s^9+1045*z^5*x^5*y^10*s^9+21629*z^3*x^5*y^10*s^9+2090*z^6*x^4*y^10*s^9+2090*z^7*x^3*y^10*s^9+1045*z^8*x^2*y^10*s^9+1045*z^5*x^2*y^10*s^9+1045*z^6*x*y^10*s^9+29049*z^5*x^6*y^9*s^9+21836*z^3*x^6*y^9*s^9+1045*z^6*x^5*y^9*s^9+21836*z^4*x^5*y^9*s^9+2090*z^7*x^4*y^9*s^9+21629*z^5*x^4*y^9*s^9+31139*z^8*x^3*y^9*s^9+1045*z^5*x^3*y^9*s^9+21629*z^3*x^3*y^9*s^9+1045*z^9*x^2*y^9*s^9+2090*z^6*x^2*y^9*s^9+1045*z^7*x*y^9*s^9+16873*z^7*x^8*y^8*s^9+21836*z^3*x^7*y^8*s^9+621*z^4*x^6*y^8*s^9+29049*z^7*x^5*y^8*s^9+621*z^5*x^5*y^8*s^9+1045*z^8*x^4*y^8*s^9+21836*z^6*x^4*y^8*s^9+29049*z^5*x^4*y^8*s^9+21836*z^3*x^4*y^8*s^9+1045*z^9*x^3*y^8*s^9+1045*z^6*x^3*y^8*s^9+21836*z^4*x^3*y^8*s^9+29049*z^10*x^2*y^8*s^9+1045*z^7*x^2*y^8*s^9+29049*z^8*x*y^8*s^9+16873*z^9*x^9*y^7*s^9+7568*z^7*x^9*y^7*s^9+7568*z^8*x^8*y^7*s^9+21629*z^3*x^8*y^7*s^9+21836*z^4*x^7*y^7*s^9+621*z^5*x^6*y^7*s^9+22250*z^6*x^5*y^7*s^9+21836*z^3*x^5*y^7*s^9+21836*z^7*x^4*y^7*s^9+621*z^4*x^4*y^7*s^9+21836*z^5*x^3*y^7*s^9+7568*z^9*x^10*y^6*s^9+7568*z^7*x^10*y^6*s^9+7568*z^10*x^9*y^6*s^9+15136*z^8*x^9*y^6*s^9+7568*z^9*x^8*y^6*s^9+21629*z^5*x^7*y^6*s^9+21836*z^6*x^6*y^6*s^9+21629*z^3*x^6*y^6*s^9+21836*z^7*x^5*y^6*s^9+21836*z^4*x^5*y^6*s^9+21629*z^8*x^4*y^6*s^9+21836*z^5*x^4*y^6*s^9+21629*z^6*x^3*y^6*s^9+7568*z^9*x^11*y^5*s^9+16873*z^7*x^11*y^5*s^9+15136*z^10*x^10*y^5*s^9+7568*z^8*x^10*y^5*s^9+7568*z^11*x^9*y^5*s^9+7568*z^9*x^9*y^5*s^9+16873*z^10*x^8*y^5*s^9+16873*z^9*x^12*y^4*s^9+7568*z^10*x^11*y^4*s^9+7568*z^11*x^10*y^4*s^9+16873*z^12*x^9*y^4*s^9+41743*z^5*x^2*y^10*s^8+39127*z^5*x^3*y^9*s^8+39127*z^6*x^2*y^9*s^8+39127*z^5*x^4*y^8*s^8+35203*z^6*x^3*y^8*s^8+39127*z^7*x^2*y^8*s^8+16873*z^7*x^10*y^7*s^8+8506*z^5*x^10*y^7*s^8+41743*z^5*x^5*y^7*s^8+39127*z^6*x^4*y^7*s^8+39127*z^7*x^3*y^7*s^8+41743*z^8*x^2*y^7*s^8+7568*z^7*x^11*y^6*s^8+25518*z^5*x^11*y^6*s^8+7568*z^8*x^10*y^6*s^8+25518*z^6*x^10*y^6*s^8+16873*z^9*x^9*y^6*s^8+16873*z^7*x^8*y^6*s^8+7568*z^7*x^12*y^5*s^8+25518*z^5*x^12*y^5*s^8+15136*z^8*x^11*y^5*s^8+7985*z^6*x^11*y^5*s^8+15136*z^9*x^10*y^5*s^8+25518*z^7*x^10*y^5*s^8+7568*z^10*x^9*y^5*s^8+7568*z^7*x^9*y^5*s^8+7568*z^8*x^8*y^5*s^8+16873*z^7*x^13*y^4*s^8+8506*z^5*x^13*y^4*s^8+7568*z^8*x^12*y^4*s^8+25518*z^6*x^12*y^4*s^8+15136*z^9*x^11*y^4*s^8+25518*z^7*x^11*y^4*s^8+32009*z^10*x^10*y^4*s^8+8506*z^8*x^10*y^4*s^8+7568*z^7*x^10*y^4*s^8+7568*z^11*x^9*y^4*s^8+15136*z^8*x^9*y^4*s^8+7568*z^9*x^8*y^4*s^8+16873*z^9*x^12*y^3*s^8+7568*z^10*x^11*y^3*s^8+16873*z^7*x^11*y^3*s^8+7568*z^11*x^10*y^3*s^8+7568*z^8*x^10*y^3*s^8+16873*z^12*x^9*y^3*s^8+7568*z^9*x^9*y^3*s^8+16873*z^10*x^8*y^3*s^8+41743*z^4*x^2*y^12*s^7+41743*z^6*x^3*y^11*s^7+39127*z^4*x^3*y^11*s^7+39127*z^5*x^2*y^11*s^7+39127*z^6*x^4*y^10*s^7+39127*z^4*x^4*y^10*s^7+39127*z^7*x^3*y^10*s^7+35203*z^5*x^3*y^10*s^7+39127*z^6*x^2*y^10*s^7+8506*z^4*x^10*y^9*s^7+39127*z^6*x^5*y^9*s^7+41743*z^4*x^5*y^9*s^7+35203*z^7*x^4*y^9*s^7+39127*z^5*x^4*y^9*s^7+39127*z^8*x^3*y^9*s^7+39127*z^6*x^3*y^9*s^7+13845*z^3*x^3*y^9*s^7+41743*z^7*x^2*y^9*s^7+8506*z^6*x^11*y^8*s^7+25518*z^4*x^11*y^8*s^7+25518*z^5*x^10*y^8*s^7+41743*z^6*x^6*y^8*s^7+39127*z^7*x^5*y^8*s^7+39127*z^8*x^4*y^8*s^7+41535*z^3*x^4*y^8*s^7+41743*z^9*x^3*y^8*s^7+41535*z^4*x^3*y^8*s^7+25518*z^6*x^12*y^7*s^7+25518*z^4*x^12*y^7*s^7+25518*z^7*x^11*y^7*s^7+7985*z^5*x^11*y^7*s^7+25518*z^6*x^10*y^7*s^7+41535*z^3*x^5*y^7*s^7+40019*z^4*x^4*y^7*s^7+41535*z^5*x^3*y^7*s^7+25518*z^6*x^13*y^6*s^7+8506*z^4*x^13*y^6*s^7+7985*z^7*x^12*y^6*s^7+25518*z^5*x^12*y^6*s^7+25518*z^8*x^11*y^6*s^7+25518*z^6*x^11*y^6*s^7+8506*z^7*x^10*y^6*s^7+13845*z^3*x^6*y^6*s^7+41535*z^4*x^5*y^6*s^7+41535*z^5*x^4*y^6*s^7+13845*z^6*x^3*y^6*s^7+8506*z^6*x^14*y^5*s^7+25518*z^7*x^13*y^5*s^7+25518*z^8*x^12*y^5*s^7+8506*z^9*x^11*y^5*s^7+24116*z^10*x^10*y^13*s^6+29297*z^10*x^11*y^12*s^6+29297*z^11*x^10*y^12*s^6+39739*z^3*x^2*y^12*s^6+29297*z^10*x^12*y^11*s^6+15543*z^11*x^11*y^11*s^6+29297*z^12*x^10*y^11*s^6+41743*z^4*x^4*y^11*s^6+33115*z^3*x^3*y^11*s^6+13845*z^2*x^3*y^11*s^6+33115*z^4*x^2*y^11*s^6+24116*z^10*x^13*y^10*s^6+29297*z^11*x^12*y^10*s^6+29297*z^12*x^11*y^10*s^6+24116*z^13*x^10*y^10*s^6+39127*z^4*x^5*y^10*s^6+39127*z^5*x^4*y^10*s^6+13845*z^4*x^4*y^10*s^6+33115*z^3*x^4*y^10*s^6+41535*z^2*x^4*y^10*s^6+41743*z^6*x^3*y^10*s^6+23179*z^4*x^3*y^10*s^6+41535*z^3*x^3*y^10*s^6+33115*z^5*x^2*y^10*s^6+41743*z^4*x^2*y^10*s^6+39127*z^4*x^6*y^9*s^6+35203*z^5*x^5*y^9*s^6+41535*z^4*x^5*y^9*s^6+39739*z^3*x^5*y^9*s^6+41535*z^2*x^5*y^9*s^6+35203*z^6*x^4*y^9*s^6+41535*z^5*x^4*y^9*s^6+33115*z^4*x^4*y^9*s^6+40019*z^3*x^4*y^9*s^6+39127*z^7*x^3*y^9*s^6+33115*z^5*x^3*y^9*s^6+37611*z^4*x^3*y^9*s^6+39739*z^6*x^2*y^9*s^6+39127*z^5*x^2*y^9*s^6+8506*z^4*x^12*y^8*s^6+41743*z^4*x^7*y^8*s^6+39127*z^5*x^6*y^8*s^6+41535*z^4*x^6*y^8*s^6+13845*z^2*x^6*y^8*s^6+35203*z^6*x^5*y^8*s^6+40019*z^5*x^5*y^8*s^6+41535*z^3*x^5*y^8*s^6+33895*z^7*x^4*y^8*s^6+41535*z^6*x^4*y^8*s^6+37611*z^4*x^4*y^8*s^6+39127*z^8*x^3*y^8*s^6+5997*z^5*x^3*y^8*s^6+39127*z^6*x^2*y^8*s^6+25518*z^4*x^13*y^7*s^6+25518*z^5*x^12*y^7*s^6+8506*z^6*x^11*y^7*s^6+8506*z^4*x^10*y^7*s^6+13845*z^4*x^7*y^7*s^6+41743*z^6*x^6*y^7*s^6+41535*z^5*x^6*y^7*s^6+39127*z^7*x^5*y^7*s^6+41535*z^6*x^5*y^7*s^6+41743*z^4*x^5*y^7*s^6+39127*z^8*x^4*y^7*s^6+13845*z^7*x^4*y^7*s^6+39127*z^5*x^4*y^7*s^6+41743*z^9*x^3*y^7*s^6+39127*z^6*x^3*y^7*s^6+41743*z^7*x^2*y^7*s^6+25518*z^4*x^14*y^6*s^6+7985*z^5*x^13*y^6*s^6+7985*z^6*x^12*y^6*s^6+25518*z^7*x^11*y^6*s^6+25518*z^4*x^11*y^6*s^6+25518*z^5*x^10*y^6*s^6+8506*z^4*x^15*y^5*s^6+25518*z^5*x^14*y^5*s^6+7985*z^6*x^13*y^5*s^6+16491*z^7*x^12*y^5*s^6+25518*z^4*x^12*y^5*s^6+25518*z^8*x^11*y^5*s^6+7985*z^5*x^11*y^5*s^6+25518*z^6*x^10*y^5*s^6+8506*z^6*x^14*y^4*s^6+25518*z^7*x^13*y^4*s^6+8506*z^4*x^13*y^4*s^6+25518*z^8*x^12*y^4*s^6+25518*z^5*x^12*y^4*s^6+8506*z^9*x^11*y^4*s^6+25518*z^6*x^11*y^4*s^6+8506*z^7*x^10*y^4*s^6+24116*z^9*x^10*y^15*s^5+24116*z^11*x^11*y^14*s^5+29297*z^9*x^11*y^14*s^5+29297*z^10*x^10*y^14*s^5+39739*z^2*x^2*y^14*s^5+29297*z^11*x^12*y^13*s^5+29297*z^9*x^12*y^13*s^5+29297*z^12*x^11*y^13*s^5+15543*z^10*x^11*y^13*s^5+29297*z^11*x^10*y^13*s^5+39739*z^4*x^3*y^13*s^5+33115*z^2*x^3*y^13*s^5+33115*z^3*x^2*y^13*s^5+29297*z^11*x^13*y^12*s^5+24116*z^9*x^13*y^12*s^5+15543*z^12*x^12*y^12*s^5+29297*z^10*x^12*y^12*s^5+29297*z^13*x^11*y^12*s^5+29297*z^11*x^11*y^12*s^5+24116*z^12*x^10*y^12*s^5+33115*z^4*x^4*y^12*s^5+33115*z^2*x^4*y^12*s^5+33115*z^5*x^3*y^12*s^5+23179*z^3*x^3*y^12*s^5+33115*z^4*x^2*y^12*s^5+24116*z^11*x^14*y^11*s^5+29297*z^12*x^13*y^11*s^5+29297*z^13*x^12*y^11*s^5+24116*z^14*x^11*y^11*s^5+33115*z^4*x^5*y^11*s^5+39739*z^2*x^5*y^11*s^5+23179*z^5*x^4*y^11*s^5+33115*z^3*x^4*y^11*s^5+33115*z^6*x^3*y^11*s^5+33115*z^4*x^3*y^11*s^5+39739*z^5*x^2*y^11*s^5+39739*z^4*x^6*y^10*s^5+33115*z^5*x^5*y^10*s^5+13845*z^2*x^5*y^10*s^5+33115*z^6*x^4*y^10*s^5+39739*z^7*x^3*y^10*s^5+41535*z^2*x^6*y^9*s^5+41535*z^3*x^5*y^9*s^5+13845*z^4*x^4*y^9*s^5+13845*z^2*x^3*y^9*s^5+41535*z^2*x^7*y^8*s^5+40019*z^3*x^6*y^8*s^5+40019*z^4*x^5*y^8*s^5+41535*z^5*x^4*y^8*s^5+41535*z^2*x^4*y^8*s^5+41535*z^3*x^3*y^8*s^5+13845*z^2*x^8*y^7*s^5+41535*z^3*x^7*y^7*s^5+40019*z^4*x^6*y^7*s^5+10813*z^5*x^5*y^7*s^5+41535*z^2*x^5*y^7*s^5+41535*z^6*x^4*y^7*s^5+40019*z^3*x^4*y^7*s^5+41535*z^4*x^3*y^7*s^5+13845*z^4*x^7*y^6*s^5+41535*z^5*x^6*y^6*s^5+13845*z^2*x^6*y^6*s^5+41535*z^6*x^5*y^6*s^5+41535*z^3*x^5*y^6*s^5+13845*z^7*x^4*y^6*s^5+41535*z^4*x^4*y^6*s^5+13845*z^5*x^3*y^6*s^5+24116*z^9*x^12*y^14*s^4+29297*z^9*x^13*y^13*s^4+29297*z^10*x^12*y^13*s^4+24116*z^11*x^11*y^13*s^4+24116*z^9*x^10*y^13*s^4+39739*z^2*x^4*y^13*s^4+29297*z^9*x^14*y^12*s^4+15543*z^10*x^13*y^12*s^4+15543*z^11*x^12*y^12*s^4+29297*z^12*x^11*y^12*s^4+29297*z^9*x^11*y^12*s^4+29297*z^10*x^10*y^12*s^4+33115*z^2*x^5*y^12*s^4+33115*z^3*x^4*y^12*s^4+39739*z^4*x^3*y^12*s^4+39739*z^2*x^2*y^12*s^4+24116*z^9*x^15*y^11*s^4+29297*z^10*x^14*y^11*s^4+15543*z^11*x^13*y^11*s^4+39659*z^12*x^12*y^11*s^4+29297*z^9*x^12*y^11*s^4+29297*z^13*x^11*y^11*s^4+15543*z^10*x^11*y^11*s^4+29297*z^11*x^10*y^11*s^4+33115*z^2*x^6*y^11*s^4+23179*z^3*x^5*y^11*s^4+23179*z^4*x^4*y^11*s^4+33115*z^5*x^3*y^11*s^4+33115*z^2*x^3*y^11*s^4+33115*z^3*x^2*y^11*s^4+24116*z^11*x^14*y^10*s^4+29297*z^12*x^13*y^10*s^4+24116*z^9*x^13*y^10*s^4+29297*z^13*x^12*y^10*s^4+29297*z^10*x^12*y^10*s^4+24116*z^14*x^11*y^10*s^4+29297*z^11*x^11*y^10*s^4+24116*z^12*x^10*y^10*s^4+39739*z^2*x^7*y^10*s^4+33115*z^3*x^6*y^10*s^4+23179*z^4*x^5*y^10*s^4+19867*z^5*x^4*y^10*s^4+33115*z^2*x^4*y^10*s^4+33115*z^6*x^3*y^10*s^4+23179*z^3*x^3*y^10*s^4+33115*z^4*x^2*y^10*s^4+39739*z^4*x^6*y^9*s^4+33115*z^5*x^5*y^9*s^4+39739*z^2*x^5*y^9*s^4+33115*z^6*x^4*y^9*s^4+33115*z^3*x^4*y^9*s^4+39739*z^7*x^3*y^9*s^4+33115*z^4*x^3*y^9*s^4+39739*z^5*x^2*y^9*s^4+19583*z^6*x^2*y^13*s^3+15698*z^6*x^3*y^12*s^3+15698*z^7*x^2*y^12*s^3+15698*z^6*x^4*y^11*s^3+31396*z^7*x^3*y^11*s^3+15698*z^8*x^2*y^11*s^3+19583*z^6*x^5*y^10*s^3+15698*z^7*x^4*y^10*s^3+15698*z^8*x^3*y^10*s^3+19583*z^9*x^2*y^10*s^3+19583*z^5*x^2*y^15*s^2+19583*z^7*x^3*y^14*s^2+15698*z^5*x^3*y^14*s^2+15698*z^6*x^2*y^14*s^2+15698*z^7*x^4*y^13*s^2+15698*z^5*x^4*y^13*s^2+15698*z^8*x^3*y^13*s^2+31396*z^6*x^3*y^13*s^2+15698*z^7*x^2*y^13*s^2+15698*z^7*x^5*y^12*s^2+19583*z^5*x^5*y^12*s^2+31396*z^8*x^4*y^12*s^2+15698*z^6*x^4*y^12*s^2+15698*z^9*x^3*y^12*s^2+15698*z^7*x^3*y^12*s^2+19583*z^8*x^2*y^12*s^2+19583*z^7*x^6*y^11*s^2+15698*z^8*x^5*y^11*s^2+15698*z^9*x^4*y^11*s^2+19583*z^10*x^3*y^11*s^2+19583*z^5*x^4*y^14*s+15698*z^5*x^5*y^13*s+15698*z^6*x^4*y^13*s+19583*z^7*x^3*y^13*s+19583*z^5*x^2*y^13*s+15698*z^5*x^6*y^12*s+31396*z^6*x^5*y^12*s+31396*z^7*x^4*y^12*s+15698*z^8*x^3*y^12*s+15698*z^5*x^3*y^12*s+15698*z^6*x^2*y^12*s+19583*z^5*x^7*y^11*s+15698*z^6*x^6*y^11*s+31396*z^7*x^5*y^11*s+7928*z^8*x^4*y^11*s+15698*z^5*x^4*y^11*s+15698*z^9*x^3*y^11*s+31396*z^6*x^3*y^11*s+15698*z^7*x^2*y^11*s+19583*z^7*x^6*y^10*s+15698*z^8*x^5*y^10*s+19583*z^5*x^5*y^10*s+15698*z^9*x^4*y^10*s+15698*z^6*x^4*y^10*s+19583*z^10*x^3*y^10*s+15698*z^7*x^3*y^10*s+19583*z^8*x^2*y^10*s)", vars, ctx);
        time = check_omega(2, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#16: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(8506*z^12*x^11*y^10*t^12+25518*z^12*x^12*y^9*t^12+25518*z^13*x^11*y^9*t^12+25518*z^12*x^13*y^8*t^12+7985*z^13*x^12*y^8*t^12+25518*z^14*x^11*y^8*t^12+8506*z^12*x^14*y^7*t^12+25518*z^13*x^13*y^7*t^12+25518*z^14*x^12*y^7*t^12+8506*z^15*x^11*y^7*t^12+6157*z^4*x^6*y^10*s^9*t^10+18471*z^4*x^7*y^9*s^9*t^10+18471*z^5*x^6*y^9*s^9*t^10+18471*z^4*x^8*y^8*s^9*t^10+36942*z^5*x^7*y^8*s^9*t^10+18471*z^6*x^6*y^8*s^9*t^10+6157*z^4*x^9*y^7*s^9*t^10+18471*z^5*x^8*y^7*s^9*t^10+18471*z^6*x^7*y^7*s^9*t^10+6157*z^7*x^6*y^7*s^9*t^10+16680*z^12*x^4*y^16*s^6*t^10+6989*z^12*x^5*y^15*s^6*t^10+6989*z^13*x^4*y^15*s^6*t^10+6989*z^12*x^6*y^14*s^6*t^10+13978*z^13*x^5*y^14*s^6*t^10+6989*z^14*x^4*y^14*s^6*t^10+16680*z^12*x^7*y^13*s^6*t^10+6989*z^13*x^6*y^13*s^6*t^10+6989*z^14*x^5*y^13*s^6*t^10+16680*z^15*x^4*y^13*s^6*t^10+41743*z*x^6*y^10*s^6*t^10+39127*z*x^7*y^9*s^6*t^10+39127*z^2*x^6*y^9*s^6*t^10+39127*z*x^8*y^8*s^6*t^10+35203*z^2*x^7*y^8*s^6*t^10+39127*z^3*x^6*y^8*s^6*t^10+41743*z*x^9*y^7*s^6*t^10+39127*z^2*x^8*y^7*s^6*t^10+39127*z^3*x^7*y^7*s^6*t^10+41743*z^4*x^6*y^7*s^6*t^10+16873*z^11*x^3*y^5*s^10*t^6+7568*z^11*x^4*y^4*s^10*t^6+7568*z^12*x^3*y^4*s^10*t^6+7568*z^11*x^5*y^3*s^10*t^6+15136*z^12*x^4*y^3*s^10*t^6+7568*z^13*x^3*y^3*s^10*t^6+16873*z^11*x^6*y^2*s^10*t^6+7568*z^12*x^5*y^2*s^10*t^6+7568*z^13*x^4*y^2*s^10*t^6+16873*z^14*x^3*y^2*s^10*t^6+21629*z^10*x^7*y^8*t^5+21836*z^10*x^8*y^7*t^5+21836*z^11*x^7*y^7*t^5+21836*z^10*x^9*y^6*t^5+621*z^11*x^8*y^6*t^5+21836*z^12*x^7*y^6*t^5+21629*z^10*x^10*y^5*t^5+21836*z^11*x^9*y^5*t^5+21836*z^12*x^8*y^5*t^5+21629*z^13*x^7*y^5*t^5+13845*z^10*x*y^13*s^4*t^4+41535*z^10*x^2*y^12*s^4*t^4+41535*z^11*x*y^12*s^4*t^4+41535*z^10*x^3*y^11*s^4*t^4+40019*z^11*x^2*y^11*s^4*t^4+41535*z^12*x*y^11*s^4*t^4+13845*z^10*x^4*y^10*s^4*t^4+41535*z^11*x^3*y^10*s^4*t^4+41535*z^12*x^2*y^10*s^4*t^4+13845*z^13*x*y^10*s^4*t^4+19583*z^8*x*y^12*s^11*t^3+15698*z^8*x^2*y^11*s^11*t^3+15698*z^9*x*y^11*s^11*t^3+15698*z^8*x^3*y^10*s^11*t^3+31396*z^9*x^2*y^10*s^11*t^3+15698*z^10*x*y^10*s^11*t^3+19583*z^8*x^4*y^9*s^11*t^3+15698*z^9*x^3*y^9*s^11*t^3+15698*z^10*x^2*y^9*s^11*t^3+19583*z^11*x*y^9*s^11*t^3+37652*z*x^12*y^16*s^7*t+26854*z*x^13*y^15*s^7*t+26854*z^2*x^12*y^15*s^7*t+26854*z*x^14*y^14*s^7*t+10657*z^2*x^13*y^14*s^7*t+26854*z^3*x^12*y^14*s^7*t+37652*z*x^15*y^13*s^7*t+26854*z^2*x^14*y^13*s^7*t+26854*z^3*x^13*y^13*s^7*t+37652*z^4*x^12*y^13*s^7*t+24116*z^3*x*y^4*s^9+29297*z^3*x^2*y^3*s^9+29297*z^4*x*y^3*s^9+29297*z^3*x^3*y^2*s^9+15543*z^4*x^2*y^2*s^9+29297*z^5*x*y^2*s^9+24116*z^3*x^4*y*s^9+29297*z^4*x^3*y*s^9+29297*z^5*x^2*y*s^9+24116*z^6*x*y*s^9+29049*z^4*x^12*y^10*s^6+1045*z^4*x^13*y^9*s^6+1045*z^5*x^12*y^9*s^6+1045*z^4*x^14*y^8*s^6+2090*z^5*x^13*y^8*s^6+1045*z^6*x^12*y^8*s^6+29049*z^4*x^15*y^7*s^6+1045*z^5*x^14*y^7*s^6+1045*z^6*x^13*y^7*s^6+29049*z^7*x^12*y^7*s^6+39739*z^3*x^7*y^9*s+33115*z^3*x^8*y^8*s+33115*z^4*x^7*y^8*s+33115*z^3*x^9*y^7*s+23179*z^4*x^8*y^7*s+33115*z^5*x^7*y^7*s+39739*z^3*x^10*y^6*s+33115*z^4*x^9*y^6*s+33115*z^5*x^8*y^6*s+39739*z^6*x^7*y^6*s)*(4017*x^3*y^2*s^6*u^5+42152*z*x^2*y*s^6*u^5+26755*z^5*x^5*y*s^5*u^5+41200*z^5*x^4*y^5*s^2*u^5+14342*z*x^3*s^2*u^5+14244*z^4*x*y^3*u^5+12889*z^4*x^4*y^6*u^3+27324*x*y^6*s^6*u^2+40575*z^3*x^2*y^5*s^2*u+33099*x*s*u+22750*z^5*x^4*y^2*s+38991*z^2*x^3)*(31261*z*x*y^3+26706*z^2*x*y^2+2238*z*x*y^2+6828*z*y^2+20135*z^2*x^2*y+41530*z*x^2*y+7431*z^2*x*y+27*z^2*y+14865*x^2+12602*z^3*x+10797*z*x)", vars, ctx);
        time = check_omega(2, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#17: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(19583*z^5*x^14*y^22*s^5*t^17+15698*z^5*x^15*y^21*s^5*t^17+15698*z^6*x^14*y^21*s^5*t^17+15698*z^5*x^16*y^20*s^5*t^17+31396*z^6*x^15*y^20*s^5*t^17+15698*z^7*x^14*y^20*s^5*t^17+19583*z^5*x^17*y^19*s^5*t^17+15698*z^6*x^16*y^19*s^5*t^17+15698*z^7*x^15*y^19*s^5*t^17+19583*z^8*x^14*y^19*s^5*t^17+13845*x^9*y^10*s^2*t^17+41535*x^10*y^9*s^2*t^17+41535*z*x^9*y^9*s^2*t^17+41535*x^11*y^8*s^2*t^17+40019*z*x^10*y^8*s^2*t^17+41535*z^2*x^9*y^8*s^2*t^17+13845*x^12*y^7*s^2*t^17+41535*z*x^11*y^7*s^2*t^17+41535*z^2*x^10*y^7*s^2*t^17+13845*z^3*x^9*y^7*s^2*t^17+16873*z*x^17*y^5*s^16*t^16+7568*z*x^18*y^4*s^16*t^16+7568*z^2*x^17*y^4*s^16*t^16+7568*z*x^19*y^3*s^16*t^16+15136*z^2*x^18*y^3*s^16*t^16+7568*z^3*x^17*y^3*s^16*t^16+16873*z*x^20*y^2*s^16*t^16+7568*z^2*x^19*y^2*s^16*t^16+7568*z^3*x^18*y^2*s^16*t^16+16873*z^4*x^17*y^2*s^16*t^16+16680*x^14*y^6*s^9*t^16+6989*x^15*y^5*s^9*t^16+6989*z*x^14*y^5*s^9*t^16+6989*x^16*y^4*s^9*t^16+13978*z*x^15*y^4*s^9*t^16+6989*z^2*x^14*y^4*s^9*t^16+16680*x^17*y^3*s^9*t^16+6989*z*x^16*y^3*s^9*t^16+6989*z^2*x^15*y^3*s^9*t^16+16680*z^3*x^14*y^3*s^9*t^16+15656*z^9*x^12*y^9*s^12*t^15+3917*z^9*x^13*y^8*s^12*t^15+3917*z^10*x^12*y^8*s^12*t^15+3917*z^9*x^14*y^7*s^12*t^15+7834*z^10*x^13*y^7*s^12*t^15+3917*z^11*x^12*y^7*s^12*t^15+15656*z^9*x^15*y^6*s^12*t^15+3917*z^10*x^14*y^6*s^12*t^15+3917*z^11*x^13*y^6*s^12*t^15+15656*z^12*x^12*y^6*s^12*t^15+41743*z*x^17*y^9*s^13*t^13+39127*z*x^18*y^8*s^13*t^13+39127*z^2*x^17*y^8*s^13*t^13+39127*z*x^19*y^7*s^13*t^13+35203*z^2*x^18*y^7*s^13*t^13+39127*z^3*x^17*y^7*s^13*t^13+41743*z*x^20*y^6*s^13*t^13+39127*z^2*x^19*y^6*s^13*t^13+39127*z^3*x^18*y^6*s^13*t^13+41743*z^4*x^17*y^6*s^13*t^13+40031*x^16*y^4*t^10+33991*x^17*y^3*t^10+33991*z*x^16*y^3*t^10+33991*x^18*y^2*t^10+24931*z*x^17*y^2*t^10+33991*z^2*x^16*y^2*t^10+40031*x^19*y*t^10+33991*z*x^18*y*t^10+33991*z^2*x^17*y*t^10+40031*z^3*x^16*y*t^10+24116*z*x^12*y^22*s^16*t^9+29297*z*x^13*y^21*s^16*t^9+29297*z^2*x^12*y^21*s^16*t^9+29297*z*x^14*y^20*s^16*t^9+15543*z^2*x^13*y^20*s^16*t^9+29297*z^3*x^12*y^20*s^16*t^9+24116*z*x^15*y^19*s^16*t^9+29297*z^2*x^14*y^19*s^16*t^9+29297*z^3*x^13*y^19*s^16*t^9+24116*z^4*x^12*y^19*s^16*t^9+37652*z^12*x^7*y^14*s^4*t^9+26854*z^12*x^8*y^13*s^4*t^9+26854*z^13*x^7*y^13*s^4*t^9+26854*z^12*x^9*y^12*s^4*t^9+10657*z^13*x^8*y^12*s^4*t^9+26854*z^14*x^7*y^12*s^4*t^9+37652*z^12*x^10*y^11*s^4*t^9+26854*z^13*x^9*y^11*s^4*t^9+26854*z^14*x^8*y^11*s^4*t^9+37652*z^15*x^7*y^11*s^4*t^9+39739*z^3*x*y^12*s^12*t^7+33115*z^3*x^2*y^11*s^12*t^7+33115*z^4*x*y^11*s^12*t^7+33115*z^3*x^3*y^10*s^12*t^7+23179*z^4*x^2*y^10*s^12*t^7+33115*z^5*x*y^10*s^12*t^7+39739*z^3*x^4*y^9*s^12*t^7+33115*z^4*x^3*y^9*s^12*t^7+33115*z^5*x^2*y^9*s^12*t^7+39739*z^6*x*y^9*s^12*t^7+29049*z^9*x^10*y^16*s^9*t^7+1045*z^9*x^11*y^15*s^9*t^7+1045*z^10*x^10*y^15*s^9*t^7+1045*z^9*x^12*y^14*s^9*t^7+2090*z^10*x^11*y^14*s^9*t^7+1045*z^11*x^10*y^14*s^9*t^7+29049*z^9*x^13*y^13*s^9*t^7+1045*z^10*x^12*y^13*s^9*t^7+1045*z^11*x^11*y^13*s^9*t^7+29049*z^12*x^10*y^13*s^9*t^7+9048*z^17*x*y^22*s^2*t^7+27144*z^17*x^2*y^21*s^2*t^7+27144*z^18*x*y^21*s^2*t^7+27144*z^17*x^3*y^20*s^2*t^7+11237*z^18*x^2*y^20*s^2*t^7+27144*z^19*x*y^20*s^2*t^7+9048*z^17*x^4*y^19*s^2*t^7+27144*z^18*x^3*y^19*s^2*t^7+27144*z^19*x^2*y^19*s^2*t^7+9048*z^20*x*y^19*s^2*t^7+27198*z^18*x^9*y^12*s^8*t^6+38543*z^18*x^10*y^11*s^8*t^6+38543*z^19*x^9*y^11*s^8*t^6+38543*z^18*x^11*y^10*s^8*t^6+34035*z^19*x^10*y^10*s^8*t^6+38543*z^20*x^9*y^10*s^8*t^6+27198*z^18*x^12*y^9*s^8*t^6+38543*z^19*x^11*y^9*s^8*t^6+38543*z^20*x^10*y^9*s^8*t^6+27198*z^21*x^9*y^9*s^8*t^6+40487*z^14*x^19*y^4*s^6*t^6+35359*z^14*x^20*y^3*s^6*t^6+35359*z^15*x^19*y^3*s^6*t^6+35359*z^14*x^21*y^2*s^6*t^6+27667*z^15*x^20*y^2*s^6*t^6+35359*z^16*x^19*y^2*s^6*t^6+40487*z^14*x^22*y*s^6*t^6+35359*z^15*x^21*y*s^6*t^6+35359*z^16*x^20*y*s^6*t^6+40487*z^17*x^19*y*s^6*t^6+6157*z^8*x^14*y^10*s^17*t^4+18471*z^8*x^15*y^9*s^17*t^4+18471*z^9*x^14*y^9*s^17*t^4+18471*z^8*x^16*y^8*s^17*t^4+36942*z^9*x^15*y^8*s^17*t^4+18471*z^10*x^14*y^8*s^17*t^4+6157*z^8*x^17*y^7*s^17*t^4+18471*z^9*x^16*y^7*s^17*t^4+18471*z^10*x^15*y^7*s^17*t^4+6157*z^11*x^14*y^7*s^17*t^4+8506*z^14*x^5*y^14*s^2*t^4+25518*z^14*x^6*y^13*s^2*t^4+25518*z^15*x^5*y^13*s^2*t^4+25518*z^14*x^7*y^12*s^2*t^4+7985*z^15*x^6*y^12*s^2*t^4+25518*z^16*x^5*y^12*s^2*t^4+8506*z^14*x^8*y^11*s^2*t^4+25518*z^15*x^7*y^11*s^2*t^4+25518*z^16*x^6*y^11*s^2*t^4+8506*z^17*x^5*y^11*s^2*t^4+12608*z^5*x^11*y^5*s^2*t^4+37824*z^5*x^12*y^4*s^2*t^4+37824*z^6*x^11*y^4*s^2*t^4+37824*z^5*x^13*y^3*s^2*t^4+32597*z^6*x^12*y^3*s^2*t^4+37824*z^7*x^11*y^3*s^2*t^4+12608*z^5*x^14*y^2*s^2*t^4+37824*z^6*x^13*y^2*s^2*t^4+37824*z^7*x^12*y^2*s^2*t^4+12608*z^8*x^11*y^2*s^2*t^4+21629*z^17*x^4*y^18*s^13*t^3+21836*z^17*x^5*y^17*s^13*t^3+21836*z^18*x^4*y^17*s^13*t^3+21836*z^17*x^6*y^16*s^13*t^3+621*z^18*x^5*y^16*s^13*t^3+21836*z^19*x^4*y^16*s^13*t^3+21629*z^17*x^7*y^15*s^13*t^3+21836*z^18*x^6*y^15*s^13*t^3+21836*z^19*x^5*y^15*s^13*t^3+21629*z^20*x^4*y^15*s^13*t^3)*(14244*z^3*x^6*s^7*u^8+33099*z^6*x^6*s^2*u^8+40832*z^3*x*y^6*s^2*u^7+27324*z*x^8*y^7*s*u^7+12889*z^2*x^2*y^3*s*u^7+26775*z^7*x^3*y^7*s^8*u^6+14342*z^5*x^2*y^5*s^7*u^6+24361*z^3*y^4*s^2*u^4+38991*z^6*x^3*y^4*s^8*u^3+41200*z^3*x^3*y^2*s^4*u^3+9788*z^4*x^4*y^7*s*u^3+26755*z^6*x^5*y^7*u^3+4017*z*x^9*y^5*u^3+31443*z^7*x^6*y*u^3+4620*x^4*y^4*s^8*u^2+42152*z^4*x^6*y^6*s^7*u^2+40575*z^6*x^6*y^6*s^3*u^2+22750*z^3*x^8*y^5*s^5*u)*(23880*z^4*x^3*y^4+15611*z^2*x^2*y^4+24361*z*x^2*y^4+12602*z^4*y^3+20135*z^2*y^3+12135*z^2*x^4*y^2+31261*z^4*x^3*y^2+2238*z^2*x^3*y^2+26706*z^3*x^3*y+7431*z^3*x^2*y+10797*z^3*x*y+41530*z*x*y+14865*x*y+5596*z*y+27*y+31987*z^2*x^3+6828*z^3)", vars, ctx);
        time = check_omega(2, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#18: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(13845*z^10*x^11*y^13*s^11+41535*z^10*x^12*y^12*s^11+41535*z^11*x^11*y^12*s^11+41535*z^10*x^13*y^11*s^11+40019*z^11*x^12*y^11*s^11+41535*z^12*x^11*y^11*s^11+13845*z^10*x^14*y^10*s^11+41535*z^11*x^13*y^10*s^11+41535*z^12*x^12*y^10*s^11+13845*z^13*x^11*y^10*s^11+41743*z^8*x^8*y^11*s^9+39127*z^8*x^9*y^10*s^9+39127*z^9*x^8*y^10*s^9+39127*z^8*x^10*y^9*s^9+35203*z^9*x^9*y^9*s^9+39127*z^10*x^8*y^9*s^9+41743*z^8*x^11*y^8*s^9+39127*z^9*x^10*y^8*s^9+39127*z^10*x^9*y^8*s^9+41743*z^11*x^8*y^8*s^9+16873*z^4*x^7*y^5*s^9+7568*z^4*x^8*y^4*s^9+7568*z^5*x^7*y^4*s^9+7568*z^4*x^9*y^3*s^9+15136*z^5*x^8*y^3*s^9+7568*z^6*x^7*y^3*s^9+16873*z^4*x^10*y^2*s^9+7568*z^5*x^9*y^2*s^9+7568*z^6*x^8*y^2*s^9+16873*z^7*x^7*y^2*s^9+29049*z^5*x*y^9*s^8+1045*z^5*x^2*y^8*s^8+1045*z^6*x*y^8*s^8+19583*z*x*y^8*s^8+1045*z^5*x^3*y^7*s^8+2090*z^6*x^2*y^7*s^8+15698*z*x^2*y^7*s^8+1045*z^7*x*y^7*s^8+15698*z^2*x*y^7*s^8+29049*z^5*x^4*y^6*s^8+1045*z^6*x^3*y^6*s^8+15698*z*x^3*y^6*s^8+1045*z^7*x^2*y^6*s^8+31396*z^2*x^2*y^6*s^8+29049*z^8*x*y^6*s^8+15698*z^3*x*y^6*s^8+19583*z*x^4*y^5*s^8+15698*z^2*x^3*y^5*s^8+15698*z^3*x^2*y^5*s^8+19583*z^4*x*y^5*s^8+16680*z^5*x^2*y^10*s^2+24116*z^4*x^9*y^9*s^2+6989*z^5*x^3*y^9*s^2+6989*z^6*x^2*y^9*s^2+8506*x^2*y^9*s^2+29297*z^4*x^10*y^8*s^2+29297*z^5*x^9*y^8*s^2+6989*z^5*x^4*y^8*s^2+13978*z^6*x^3*y^8*s^2+25518*x^3*y^8*s^2+6989*z^7*x^2*y^8*s^2+25518*z*x^2*y^8*s^2+29297*z^4*x^11*y^7*s^2+15543*z^5*x^10*y^7*s^2+29297*z^6*x^9*y^7*s^2+16680*z^5*x^5*y^7*s^2+6989*z^6*x^4*y^7*s^2+25518*x^4*y^7*s^2+6989*z^7*x^3*y^7*s^2+7985*z*x^3*y^7*s^2+16680*z^8*x^2*y^7*s^2+25518*z^2*x^2*y^7*s^2+24116*z^4*x^12*y^6*s^2+29297*z^5*x^11*y^6*s^2+29297*z^6*x^10*y^6*s^2+24116*z^7*x^9*y^6*s^2+8506*x^5*y^6*s^2+25518*z*x^4*y^6*s^2+25518*z^2*x^3*y^6*s^2+8506*z^3*x^2*y^6*s^2+39739*z^8*x^5*y^5*s^2+33115*z^8*x^6*y^4*s^2+33115*z^9*x^5*y^4*s^2+33115*z^8*x^7*y^3*s^2+23179*z^9*x^6*y^3*s^2+33115*z^10*x^5*y^3*s^2+39739*z^8*x^8*y^2*s^2+33115*z^9*x^7*y^2*s^2+33115*z^10*x^6*y^2*s^2+39739*z^11*x^5*y^2*s^2+37652*z*x^4*y^10*s+26854*z*x^5*y^9*s+26854*z^2*x^4*y^9*s+26854*z*x^6*y^8*s+10657*z^2*x^5*y^8*s+26854*z^3*x^4*y^8*s+37652*z*x^7*y^7*s+26854*z^2*x^6*y^7*s+26854*z^3*x^5*y^7*s+37652*z^4*x^4*y^7*s+21629*z^7*x^7*y^5+21836*z^7*x^8*y^4+21836*z^8*x^7*y^4+21836*z^7*x^9*y^3+621*z^8*x^8*y^3+21836*z^9*x^7*y^3+21629*z^7*x^10*y^2+21836*z^8*x^9*y^2+21836*z^9*x^8*y^2+21629*z^10*x^7*y^2)*(40575*z*y^4*s^5+14244*z^2*x^2*y^2*s^5+26755*z*x^5*y^4*s^4+41200*z^3*x^5*y*s^4+42152*x*y^4*s^3+22750*z^5*x^5*y^4*s^2+14342*z^4*y^4*s+33099*z^2*x^2*y^5+27324*z^5*x^2*y^4+12889*x^4*y^3+4017*z^2*x^5)*(30008*z^2*x^2*y^2+13070*x^2*y^2+41530*z*x*y^2+12602*x*y^2+31261*z^2*y^2+5951*z^2*x*y+2238*z*y+10797*z*x^2+6828*z^2)", vars, ctx);
        time = check_omega(2, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#19: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "2036*(x)^2*(y)^2*(y^55+x^27)*("
            "x^26*y^27+22502*x^25*y^27+26129*x^24*y^27+29216*x^23*y^27+34834*x^22*y^27+16100*"
            "x^21*y^27+31667*x^20*y^27+1906*x^19*y^27+22185*x^18*y^27+16677*x^17*y^27+41095*x"
            "^16*y^27+15123*x^15*y^27+14491*x^14*y^27+5571*x^13*y^27+11913*x^12*y^27+21343*x^"
            "11*y^27+32816*x^10*y^27+29154*x^9*y^27+22111*x^8*y^27+30147*x^7*y^27+1673*x^6*y^"
            "27+30810*x^5*y^27+16497*x^4*y^27+39798*x^3*y^27+4394*x^2*y^27+31502*x*y^27+17487"
            "*y^27+3083*x^26*y^26+7510*x^25*y^26+3246*x^24*y^26+11984*x^23*y^26+25636*x^22*y^"
            "26+15923*x^21*y^26+542*x^20*y^26+29538*x^19*y^26+26552*x^18*y^26+24474*x^17*y^26"
            "+4843*x^16*y^26+6664*x^15*y^26+30375*x^14*y^26+25269*x^13*y^26+21676*x^12*y^26+"
            "33967*x^11*y^26+16441*x^10*y^26+16777*x^9*y^26+32168*x^8*y^26+31612*x^7*y^26+"
            "30808*x^6*y^26+11273*x^5*y^26+19942*x^4*y^26+6182*x^3*y^26+40996*x^2*y^26+31624*"
            "x*y^26+63*y^26+20355*x^25*y^25+31973*x^24*y^25+22971*x^23*y^25+27166*x^22*y^25+"
            "36472*x^21*y^25+35782*x^20*y^25+24678*x^19*y^25+30016*x^18*y^25+4607*x^17*y^25+"
            "41947*x^16*y^25+31198*x^15*y^25+16004*x^14*y^25+10632*x^13*y^25+18864*x^12*y^25+"
            "17038*x^11*y^25+25214*x^10*y^25+38756*x^9*y^25+7046*x^8*y^25+19978*x^7*y^25+1376"
            "*x^6*y^25+41415*x^5*y^25+10386*x^4*y^25+19065*x^3*y^25+17113*x^2*y^25+16281*x*y^"
            "25+13858*y^25+5491*x^24*y^24+38472*x^23*y^24+35769*x^22*y^24+15694*x^21*y^24+"
            "33392*x^20*y^24+1146*x^19*y^24+18050*x^18*y^24+12125*x^17*y^24+29102*x^16*y^24+"
            "38748*x^15*y^24+42753*x^14*y^24+6698*x^13*y^24+28160*x^12*y^24+21070*x^11*y^24+"
            "41637*x^10*y^24+19616*x^9*y^24+14406*x^8*y^24+23476*x^7*y^24+21817*x^6*y^24+"
            "25840*x^5*y^24+25086*x^4*y^24+23775*x^3*y^24+21227*x^2*y^24+27888*x*y^24+23515*y"
            "^24+19459*x^23*y^23+36602*x^22*y^23+208*x^21*y^23+12857*x^20*y^23+12486*x^19*y^"
            "23+22368*x^18*y^23+2551*x^17*y^23+34235*x^16*y^23+14390*x^15*y^23+5882*x^14*y^23"
            "+33538*x^13*y^23+42285*x^12*y^23+8690*x^11*y^23+30406*x^10*y^23+27320*x^9*y^23+"
            "16879*x^8*y^23+3731*x^7*y^23+38196*x^6*y^23+42159*x^5*y^23+8240*x^4*y^23+31121*x"
            "^3*y^23+35093*x^2*y^23+14738*x*y^23+30083*y^23+7123*x^22*y^22+6572*x^21*y^22+"
            "8982*x^20*y^22+39575*x^19*y^22+28765*x^18*y^22+20952*x^17*y^22+8540*x^16*y^22+"
            "2594*x^15*y^22+30737*x^14*y^22+18260*x^13*y^22+19933*x^12*y^22+8099*x^11*y^22+"
            "29843*x^10*y^22+37536*x^9*y^22+13943*x^8*y^22+1823*x^7*y^22+4331*x^6*y^22+11682*"
            "x^5*y^22+28595*x^4*y^22+26253*x^3*y^22+21766*x^2*y^22+33032*x*y^22+18506*y^22+"
            "19732*x^21*y^21+29287*x^20*y^21+33671*x^19*y^21+20636*x^18*y^21+9129*x^17*y^21+"
            "668*x^16*y^21+12392*x^15*y^21+18209*x^14*y^21+42027*x^13*y^21+10723*x^12*y^21+"
            "40606*x^11*y^21+21152*x^10*y^21+28922*x^9*y^21+42902*x^8*y^21+23603*x^7*y^21+"
            "10788*x^6*y^21+40011*x^5*y^21+25015*x^4*y^21+16784*x^3*y^21+7690*x^2*y^21+14090*"
            "x*y^21+29510*y^21+36673*x^20*y^20+18226*x^19*y^20+9351*x^18*y^20+38118*x^17*y^20"
            "+17449*x^16*y^20+30651*x^15*y^20+6652*x^14*y^20+4443*x^13*y^20+28114*x^12*y^20+"
            "27713*x^11*y^20+14382*x^10*y^20+35435*x^9*y^20+15403*x^8*y^20+27528*x^7*y^20+"
            "35417*x^6*y^20+21835*x^5*y^20+25685*x^4*y^20+3166*x^3*y^20+2542*x^2*y^20+30766*x"
            "*y^20+31657*y^20+26361*x^19*y^19+1812*x^18*y^19+3983*x^17*y^19+36023*x^16*y^19+"
            "22928*x^15*y^19+6855*x^14*y^19+27546*x^13*y^19+24902*x^12*y^19+36654*x^11*y^19+"
            "5191*x^10*y^19+28174*x^9*y^19+10932*x^8*y^19+33381*x^7*y^19+18283*x^6*y^19+10138"
            "*x^5*y^19+3525*x^4*y^19+31818*x^3*y^19+26751*x^2*y^19+8791*x*y^19+35017*y^19+"
            "8068*x^18*y^18+11779*x^17*y^18+6526*x^16*y^18+37302*x^15*y^18+11164*x^14*y^18+"
            "13769*x^13*y^18+39382*x^12*y^18+40572*x^11*y^18+13765*x^10*y^18+24417*x^9*y^18+"
            "30189*x^8*y^18+21349*x^7*y^18+39167*x^6*y^18+20606*x^5*y^18+14305*x^4*y^18+29683"
            "*x^3*y^18+1266*x^2*y^18+24840*x*y^18+21656*y^18+16312*x^17*y^17+5217*x^16*y^17+"
            "41816*x^15*y^17+8280*x^14*y^17+9410*x^13*y^17+9008*x^12*y^17+30427*x^11*y^17+"
            "2633*x^10*y^17+14363*x^9*y^17+967*x^8*y^17+1177*x^7*y^17+1020*x^6*y^17+29630*x^5"
            "*y^17+15743*x^4*y^17+20952*x^3*y^17+31064*x^2*y^17+3319*x*y^17+8408*y^17+12350*x"
            "^16*y^16+14714*x^15*y^16+38478*x^14*y^16+18118*x^13*y^16+38544*x^12*y^16+17870*x"
            "^11*y^16+39355*x^10*y^16+5319*x^9*y^16+27260*x^8*y^16+30534*x^7*y^16+14619*x^6*y"
            "^16+29338*x^5*y^16+18577*x^4*y^16+5699*x^3*y^16+25077*x^2*y^16+24985*x*y^16+"
            "41762*y^16+14161*x^15*y^15+24818*x^14*y^15+2538*x^13*y^15+5694*x^12*y^15+36797*x"
            "^11*y^15+11305*x^10*y^15+266*x^9*y^15+34388*x^8*y^15+36472*x^7*y^15+35287*x^6*y^"
            "15+36112*x^5*y^15+39364*x^4*y^15+31042*x^3*y^15+24571*x^2*y^15+22543*x*y^15+"
            "33036*y^15+3046*x^14*y^14+42428*x^13*y^14+14398*x^12*y^14+8792*x^11*y^14+16394*x"
            "^10*y^14+24593*x^9*y^14+29104*x^8*y^14+29172*x^7*y^14+4617*x^6*y^14+16803*x^5*y^"
            "14+38129*x^4*y^14+25100*x^3*y^14+14495*x^2*y^14+16736*x*y^14+5132*y^14+23113*x^"
            "13*y^13+40441*x^12*y^13+38512*x^11*y^13+8600*x^10*y^13+24449*x^9*y^13+11553*x^8*"
            "y^13+20503*x^7*y^13+22462*x^6*y^13+28782*x^5*y^13+32654*x^4*y^13+8182*x^3*y^13+"
            "18208*x^2*y^13+31145*x*y^13+9877*y^13+33439*x^12*y^12+37259*x^11*y^12+3230*x^10*"
            "y^12+7924*x^9*y^12+21484*x^8*y^12+5249*x^7*y^12+28816*x^6*y^12+36832*x^5*y^12+"
            "31614*x^4*y^12+30336*x^3*y^12+28977*x^2*y^12+33842*x*y^12+27712*y^12+11480*x^11*"
            "y^11+42024*x^10*y^11+40159*x^9*y^11+4656*x^8*y^11+13905*x^7*y^11+32675*x^6*y^11+"
            "15182*x^5*y^11+33276*x^4*y^11+40271*x^3*y^11+33097*x^2*y^11+5450*x*y^11+16545*y^"
            "11+4444*x^10*y^10+29813*x^9*y^10+30598*x^8*y^10+19265*x^7*y^10+27733*x^6*y^10+"
            "12332*x^5*y^10+4041*x^4*y^10+17994*x^3*y^10+37626*x^2*y^10+18017*x*y^10+41985*y^"
            "10+18161*x^9*y^9+37064*x^8*y^9+38753*x^7*y^9+30823*x^6*y^9+12976*x^5*y^9+27156*x"
            "^4*y^9+22754*x^3*y^9+5457*x^2*y^9+37393*x*y^9+18558*y^9+5969*x^8*y^8+8372*x^7*y^"
            "8+5034*x^6*y^8+39726*x^5*y^8+23781*x^4*y^8+18050*x^3*y^8+38810*x^2*y^8+22187*x*y"
            "^8+33578*y^8+6247*x^7*y^7+31686*x^6*y^7+17254*x^5*y^7+13842*x^4*y^7+5880*x^3*y^7"
            "+7517*x^2*y^7+29126*x*y^7+25434*y^7+42719*x^6*y^6+37537*x^5*y^6+7149*x^4*y^6+"
            "6926*x^3*y^6+30890*x^2*y^6+33056*x*y^6+35681*y^6+3125*x^5*y^5+33376*x^4*y^5+"
            "24387*x^3*y^5+23868*x^2*y^5+35142*x*y^5+5143*y^5+40391*x^4*y^4+657*x^3*y^4+30346"
            "*x^2*y^4+30607*x*y^4+9432*y^4+31094*x^3*y^3+25871*x^2*y^3+29619*x*y^3+40511*y^3+"
            "747*x^2*y^2+23254*x*y^2+39636*y^2+17316*x*y+3609*y+8994)*("
            "x^53*y^53+18147*x^52*y^53+43022*x^51*y^53+40085*x^50*y^53+28237*x^49*y^53+30609*"
            "x^48*y^53+24338*x^47*y^53+34438*x^46*y^53+17995*x^45*y^53+11285*x^44*y^53+23365*"
            "x^43*y^53+34068*x^42*y^53+36580*x^41*y^53+18333*x^40*y^53+28343*x^39*y^53+10219*"
            "x^38*y^53+41928*x^37*y^53+14530*x^36*y^53+8169*x^35*y^53+10664*x^34*y^53+28456*x"
            "^33*y^53+40045*x^32*y^53+36063*x^31*y^53+40317*x^30*y^53+3892*x^29*y^53+18610*x^"
            "28*y^53+42334*x^27*y^53+11181*x^26*y^53+14187*x^25*y^53+19731*x^24*y^53+15756*x^"
            "23*y^53+9842*x^22*y^53+12573*x^21*y^53+3119*x^20*y^53+21843*x^19*y^53+2270*x^18*"
            "y^53+25305*x^17*y^53+38419*x^16*y^53+23222*x^15*y^53+29693*x^14*y^53+37556*x^13*"
            "y^53+16413*x^12*y^53+4411*x^11*y^53+27575*x^10*y^53+15150*x^9*y^53+6260*x^8*y^53"
            "+27551*x^7*y^53+25107*x^6*y^53+32932*x^5*y^53+38353*x^4*y^53+9*x^3*y^53+5495*x^2"
            "*y^53+10261*x*y^53+31369*y^53+5547*x^52*y^52+30032*x^51*y^52+284*x^50*y^52+14830"
            "*x^49*y^52+22659*x^48*y^52+10626*x^47*y^52+22237*x^46*y^52+38950*x^45*y^52+40954"
            "*x^44*y^52+9413*x^43*y^52+21276*x^42*y^52+31506*x^41*y^52+9404*x^40*y^52+10655*x"
            "^39*y^52+10743*x^38*y^52+4401*x^37*y^52+32372*x^36*y^52+15524*x^35*y^52+28427*x^"
            "34*y^52+26855*x^33*y^52+530*x^32*y^52+28907*x^31*y^52+4066*x^30*y^52+20491*x^29*"
            "y^52+2863*x^28*y^52+12779*x^27*y^52+40264*x^26*y^52+27587*x^25*y^52+2553*x^24*y^"
            "52+6016*x^23*y^52+2097*x^22*y^52+39570*x^21*y^52+34283*x^20*y^52+17879*x^19*y^52"
            "+17020*x^18*y^52+40538*x^17*y^52+25557*x^16*y^52+39049*x^15*y^52+12516*x^14*y^52"
            "+20550*x^13*y^52+37236*x^12*y^52+27582*x^11*y^52+40355*x^10*y^52+9244*x^9*y^52+"
            "12348*x^8*y^52+3487*x^7*y^52+34984*x^6*y^52+6073*x^5*y^52+39538*x^4*y^52+28840*x"
            "^3*y^52+6672*x^2*y^52+35673*x*y^52+21090*y^52+26879*x^51*y^51+19080*x^50*y^51+"
            "20236*x^49*y^51+40752*x^48*y^51+4404*x^47*y^51+28484*x^46*y^51+19369*x^45*y^51+"
            "21123*x^44*y^51+42333*x^43*y^51+3104*x^42*y^51+41974*x^41*y^51+1561*x^40*y^51+"
            "11103*x^39*y^51+10769*x^38*y^51+27387*x^37*y^51+21812*x^36*y^51+1578*x^35*y^51+"
            "39358*x^34*y^51+22262*x^33*y^51+28872*x^32*y^51+29393*x^31*y^51+37195*x^30*y^51+"
            "9259*x^29*y^51+12198*x^28*y^51+33283*x^27*y^51+6171*x^26*y^51+1678*x^25*y^51+"
            "19116*x^24*y^51+34014*x^23*y^51+42437*x^22*y^51+17526*x^21*y^51+30549*x^20*y^51+"
            "29170*x^19*y^51+16093*x^18*y^51+4549*x^17*y^51+42768*x^16*y^51+40904*x^15*y^51+"
            "3467*x^14*y^51+36407*x^13*y^51+18943*x^12*y^51+22205*x^11*y^51+7868*x^10*y^51+"
            "15795*x^9*y^51+36617*x^8*y^51+17510*x^7*y^51+15678*x^6*y^51+16520*x^5*y^51+40409"
            "*x^4*y^51+27740*x^3*y^51+35836*x^2*y^51+30552*x*y^51+20808*y^51+42970*x^50*y^50+"
            "5860*x^49*y^50+23145*x^48*y^50+9906*x^47*y^50+34850*x^46*y^50+38599*x^45*y^50+"
            "40329*x^44*y^50+15967*x^43*y^50+7261*x^42*y^50+41272*x^41*y^50+4284*x^40*y^50+"
            "37022*x^39*y^50+27603*x^38*y^50+8694*x^37*y^50+42972*x^36*y^50+26061*x^35*y^50+"
            "18941*x^34*y^50+4252*x^33*y^50+34789*x^32*y^50+34843*x^31*y^50+13722*x^30*y^50+"
            "20217*x^29*y^50+16944*x^28*y^50+7910*x^27*y^50+30072*x^26*y^50+9131*x^25*y^50+"
            "17095*x^24*y^50+38859*x^23*y^50+11391*x^22*y^50+8284*x^21*y^50+40113*x^20*y^50+"
            "16017*x^19*y^50+30360*x^18*y^50+28241*x^17*y^50+42497*x^16*y^50+12082*x^15*y^50+"
            "1284*x^14*y^50+16288*x^13*y^50+24860*x^12*y^50+13411*x^11*y^50+10975*x^10*y^50+"
            "33044*x^9*y^50+8235*x^8*y^50+22317*x^7*y^50+31218*x^6*y^50+20511*x^5*y^50+24255*"
            "x^4*y^50+26118*x^3*y^50+31607*x^2*y^50+12349*x*y^50+39022*y^50+13425*x^49*y^49+"
            "39757*x^48*y^49+27451*x^47*y^49+194*x^46*y^49+24526*x^45*y^49+9875*x^44*y^49+"
            "31606*x^43*y^49+38444*x^42*y^49+18652*x^41*y^49+29818*x^40*y^49+15631*x^39*y^49+"
            "21247*x^38*y^49+10326*x^37*y^49+30737*x^36*y^49+28353*x^35*y^49+32283*x^34*y^49+"
            "40894*x^33*y^49+19141*x^32*y^49+11212*x^31*y^49+4674*x^30*y^49+28404*x^29*y^49+"
            "33933*x^28*y^49+6387*x^27*y^49+32050*x^26*y^49+14228*x^25*y^49+29664*x^24*y^49+"
            "35576*x^23*y^49+40915*x^22*y^49+40400*x^21*y^49+1818*x^20*y^49+31280*x^19*y^49+"
            "6559*x^18*y^49+33932*x^17*y^49+37524*x^16*y^49+27603*x^15*y^49+10707*x^14*y^49+"
            "32695*x^13*y^49+26450*x^12*y^49+25719*x^11*y^49+38755*x^10*y^49+6656*x^9*y^49+"
            "21403*x^8*y^49+5632*x^7*y^49+888*x^6*y^49+42735*x^5*y^49+13602*x^4*y^49+24940*x^"
            "3*y^49+38405*x^2*y^49+8054*x*y^49+20049*y^49+27372*x^48*y^48+30233*x^47*y^48+657"
            "*x^46*y^48+29614*x^45*y^48+20701*x^44*y^48+23794*x^43*y^48+3737*x^42*y^48+6648*x"
            "^41*y^48+20102*x^40*y^48+22356*x^39*y^48+32369*x^38*y^48+12432*x^37*y^48+26613*x"
            "^36*y^48+21249*x^35*y^48+5150*x^34*y^48+2672*x^33*y^48+15097*x^32*y^48+37938*x^"
            "31*y^48+14805*x^30*y^48+22802*x^29*y^48+39201*x^28*y^48+30407*x^27*y^48+20653*x^"
            "26*y^48+33705*x^25*y^48+23079*x^24*y^48+2198*x^23*y^48+38433*x^22*y^48+7227*x^21"
            "*y^48+23891*x^20*y^48+3967*x^19*y^48+37476*x^18*y^48+36934*x^17*y^48+12206*x^16*"
            "y^48+22197*x^15*y^48+10245*x^14*y^48+22127*x^13*y^48+41418*x^12*y^48+814*x^11*y^"
            "48+31931*x^10*y^48+38041*x^9*y^48+6354*x^8*y^48+22194*x^7*y^48+22787*x^6*y^48+"
            "30170*x^5*y^48+5917*x^4*y^48+36122*x^3*y^48+19222*x^2*y^48+32886*x*y^48+22622*y^"
            "48+3812*x^47*y^47+25985*x^46*y^47+36819*x^45*y^47+13268*x^44*y^47+35718*x^43*y^"
            "47+24087*x^42*y^47+33429*x^41*y^47+38428*x^40*y^47+42427*x^39*y^47+18508*x^38*y^"
            "47+807*x^37*y^47+28580*x^36*y^47+12219*x^35*y^47+38548*x^34*y^47+41213*x^33*y^47"
            "+13374*x^32*y^47+240*x^31*y^47+18855*x^30*y^47+3163*x^29*y^47+39802*x^28*y^47+"
            "4065*x^27*y^47+21549*x^26*y^47+35680*x^25*y^47+30407*x^24*y^47+9345*x^23*y^47+"
            "37516*x^22*y^47+31854*x^21*y^47+10318*x^20*y^47+2740*x^19*y^47+11306*x^18*y^47+"
            "5988*x^17*y^47+16224*x^16*y^47+22259*x^15*y^47+19555*x^14*y^47+20347*x^13*y^47+"
            "6069*x^12*y^47+20053*x^11*y^47+9241*x^10*y^47+29531*x^9*y^47+12594*x^8*y^47+"
            "25231*x^7*y^47+2145*x^6*y^47+36468*x^5*y^47+24431*x^4*y^47+38896*x^3*y^47+13456*"
            "x^2*y^47+40531*x*y^47+9268*y^47+33917*x^46*y^46+7802*x^45*y^46+43037*x^44*y^46+"
            "20656*x^43*y^46+37129*x^42*y^46+11642*x^41*y^46+19623*x^40*y^46+796*x^39*y^46+"
            "16361*x^38*y^46+41230*x^37*y^46+29857*x^36*y^46+24087*x^35*y^46+5566*x^34*y^46+"
            "6421*x^33*y^46+40958*x^32*y^46+37371*x^31*y^46+18049*x^30*y^46+19808*x^29*y^46+"
            "5885*x^28*y^46+38032*x^27*y^46+5553*x^26*y^46+42145*x^25*y^46+34158*x^24*y^46+"
            "38717*x^23*y^46+5514*x^22*y^46+39714*x^21*y^46+17252*x^20*y^46+13415*x^19*y^46+"
            "9576*x^18*y^46+35622*x^17*y^46+34212*x^16*y^46+28019*x^15*y^46+42386*x^14*y^46+"
            "27403*x^13*y^46+38945*x^12*y^46+12139*x^11*y^46+29342*x^10*y^46+13347*x^9*y^46+"
            "37592*x^8*y^46+13920*x^7*y^46+30889*x^6*y^46+13930*x^5*y^46+21793*x^4*y^46+9482*"
            "x^3*y^46+42650*x^2*y^46+2703*x*y^46+19032*y^46+40355*x^45*y^45+34673*x^44*y^45+"
            "16318*x^43*y^45+38013*x^42*y^45+30018*x^41*y^45+7755*x^40*y^45+16854*x^39*y^45+"
            "25824*x^38*y^45+30004*x^37*y^45+22057*x^36*y^45+30486*x^35*y^45+18166*x^34*y^45+"
            "30268*x^33*y^45+42335*x^32*y^45+9703*x^31*y^45+8876*x^30*y^45+31479*x^29*y^45+"
            "4458*x^28*y^45+1013*x^27*y^45+26568*x^26*y^45+26890*x^25*y^45+37461*x^24*y^45+"
            "16554*x^23*y^45+7323*x^22*y^45+8189*x^21*y^45+6451*x^20*y^45+33300*x^19*y^45+"
            "26185*x^18*y^45+24508*x^17*y^45+23070*x^16*y^45+5113*x^15*y^45+29902*x^14*y^45+"
            "23508*x^13*y^45+29179*x^12*y^45+24325*x^11*y^45+691*x^10*y^45+9957*x^9*y^45+"
            "23116*x^8*y^45+31858*x^7*y^45+40746*x^6*y^45+23740*x^5*y^45+16587*x^4*y^45+18456"
            "*x^3*y^45+22891*x^2*y^45+24232*x*y^45+24901*y^45+1581*x^44*y^44+8067*x^43*y^44+"
            "40636*x^42*y^44+590*x^41*y^44+28238*x^40*y^44+18484*x^39*y^44+13782*x^38*y^44+"
            "26591*x^37*y^44+38088*x^36*y^44+7380*x^35*y^44+35239*x^34*y^44+1000*x^33*y^44+"
            "20981*x^32*y^44+4647*x^31*y^44+37942*x^30*y^44+8411*x^29*y^44+1844*x^28*y^44+"
            "42980*x^27*y^44+1636*x^26*y^44+19108*x^25*y^44+10146*x^24*y^44+20133*x^23*y^44+"
            "31891*x^22*y^44+36956*x^21*y^44+4778*x^20*y^44+28052*x^19*y^44+7656*x^18*y^44+"
            "23804*x^17*y^44+25604*x^16*y^44+34159*x^15*y^44+39731*x^14*y^44+40121*x^13*y^44+"
            "33040*x^12*y^44+19546*x^11*y^44+4051*x^10*y^44+22053*x^9*y^44+8357*x^8*y^44+"
            "13907*x^7*y^44+25422*x^6*y^44+16353*x^5*y^44+17091*x^4*y^44+10820*x^3*y^44+10885"
            "*x^2*y^44+26791*x*y^44+26814*y^44+30035*x^43*y^43+19261*x^42*y^43+5960*x^41*y^43"
            "+25009*x^40*y^43+31459*x^39*y^43+5593*x^38*y^43+11908*x^37*y^43+40759*x^36*y^43+"
            "25678*x^35*y^43+317*x^34*y^43+2911*x^33*y^43+21930*x^32*y^43+10338*x^31*y^43+"
            "18544*x^30*y^43+21395*x^29*y^43+32184*x^28*y^43+2508*x^27*y^43+30020*x^26*y^43+"
            "36827*x^25*y^43+32729*x^24*y^43+37356*x^23*y^43+37941*x^22*y^43+38560*x^21*y^43+"
            "25442*x^20*y^43+15046*x^19*y^43+25010*x^18*y^43+29960*x^17*y^43+317*x^16*y^43+"
            "19367*x^15*y^43+12431*x^14*y^43+22477*x^13*y^43+7299*x^12*y^43+29600*x^11*y^43+"
            "18368*x^10*y^43+30660*x^9*y^43+31625*x^8*y^43+34063*x^7*y^43+39332*x^6*y^43+"
            "42283*x^5*y^43+24888*x^4*y^43+19738*x^3*y^43+2405*x^2*y^43+5956*x*y^43+4332*y^43"
            "+39260*x^42*y^42+19385*x^41*y^42+31659*x^40*y^42+34965*x^39*y^42+34569*x^38*y^42"
            "+20394*x^37*y^42+29989*x^36*y^42+4513*x^35*y^42+8431*x^34*y^42+40216*x^33*y^42+"
            "42045*x^32*y^42+17569*x^31*y^42+17859*x^30*y^42+7805*x^29*y^42+39762*x^28*y^42+"
            "7624*x^27*y^42+12453*x^26*y^42+13594*x^25*y^42+42421*x^24*y^42+36751*x^23*y^42+"
            "21549*x^22*y^42+1117*x^21*y^42+27687*x^20*y^42+17486*x^19*y^42+37758*x^18*y^42+"
            "41570*x^17*y^42+4711*x^16*y^42+21777*x^15*y^42+9007*x^14*y^42+14510*x^13*y^42+"
            "495*x^12*y^42+27546*x^11*y^42+15369*x^10*y^42+28778*x^9*y^42+1890*x^8*y^42+38622"
            "*x^7*y^42+14667*x^6*y^42+24823*x^5*y^42+26179*x^4*y^42+37676*x^3*y^42+19558*x^2*"
            "y^42+20984*x*y^42+33240*y^42+36690*x^41*y^41+17023*x^40*y^41+17026*x^39*y^41+"
            "6811*x^38*y^41+16785*x^37*y^41+12085*x^36*y^41+29521*x^35*y^41+34282*x^34*y^41+"
            "8706*x^33*y^41+37090*x^32*y^41+30501*x^31*y^41+18192*x^30*y^41+35111*x^29*y^41+"
            "39271*x^28*y^41+5042*x^27*y^41+25069*x^26*y^41+34380*x^25*y^41+13858*x^24*y^41+"
            "9698*x^23*y^41+12876*x^22*y^41+18196*x^21*y^41+31276*x^20*y^41+38122*x^19*y^41+"
            "12008*x^18*y^41+24932*x^17*y^41+33107*x^16*y^41+20016*x^15*y^41+11013*x^14*y^41+"
            "39534*x^13*y^41+28790*x^12*y^41+9663*x^11*y^41+34756*x^10*y^41+41666*x^9*y^41+"
            "33575*x^8*y^41+31797*x^7*y^41+17028*x^6*y^41+35080*x^5*y^41+23794*x^4*y^41+31927"
            "*x^3*y^41+42101*x^2*y^41+3694*x*y^41+17898*y^41+5030*x^40*y^40+20705*x^39*y^40+"
            "40138*x^38*y^40+20010*x^37*y^40+41623*x^36*y^40+3326*x^35*y^40+21134*x^34*y^40+"
            "36214*x^33*y^40+42448*x^32*y^40+9904*x^31*y^40+3791*x^30*y^40+38996*x^29*y^40+"
            "7031*x^28*y^40+21395*x^27*y^40+16931*x^26*y^40+23563*x^25*y^40+5604*x^24*y^40+"
            "18577*x^23*y^40+17832*x^22*y^40+38462*x^21*y^40+17939*x^20*y^40+13578*x^19*y^40+"
            "5091*x^18*y^40+11389*x^17*y^40+41259*x^16*y^40+6438*x^15*y^40+35827*x^14*y^40+"
            "38132*x^13*y^40+14738*x^12*y^40+35596*x^11*y^40+22288*x^10*y^40+38612*x^9*y^40+"
            "30582*x^8*y^40+9216*x^7*y^40+22498*x^6*y^40+41447*x^5*y^40+21423*x^4*y^40+18859*"
            "x^3*y^40+2943*x^2*y^40+42*x*y^40+26122*y^40+21917*x^39*y^39+16248*x^38*y^39+"
            "12815*x^37*y^39+1065*x^36*y^39+23742*x^35*y^39+3529*x^34*y^39+6933*x^33*y^39+"
            "6197*x^32*y^39+18254*x^31*y^39+5200*x^30*y^39+14892*x^29*y^39+32191*x^28*y^39+"
            "27194*x^27*y^39+12220*x^26*y^39+15416*x^25*y^39+8038*x^24*y^39+23518*x^23*y^39+"
            "21732*x^22*y^39+40262*x^21*y^39+3864*x^20*y^39+18543*x^19*y^39+32141*x^18*y^39+"
            "6335*x^17*y^39+16625*x^16*y^39+22604*x^15*y^39+25647*x^14*y^39+14043*x^13*y^39+"
            "37489*x^12*y^39+32245*x^11*y^39+24467*x^10*y^39+124*x^9*y^39+27492*x^8*y^39+72*x"
            "^7*y^39+7216*x^6*y^39+8002*x^5*y^39+18089*x^4*y^39+11437*x^3*y^39+35176*x^2*y^39"
            "+18285*x*y^39+28670*y^39+4532*x^38*y^38+37470*x^37*y^38+19891*x^36*y^38+41542*x^"
            "35*y^38+9718*x^34*y^38+42839*x^33*y^38+8238*x^32*y^38+22952*x^31*y^38+39648*x^30"
            "*y^38+28131*x^29*y^38+22317*x^28*y^38+25299*x^27*y^38+23511*x^26*y^38+9894*x^25*"
            "y^38+13716*x^24*y^38+35578*x^23*y^38+31881*x^22*y^38+13171*x^21*y^38+31957*x^20*"
            "y^38+7557*x^19*y^38+9080*x^18*y^38+33605*x^17*y^38+28061*x^16*y^38+26224*x^15*y^"
            "38+38115*x^14*y^38+12579*x^13*y^38+9476*x^12*y^38+18826*x^11*y^38+24558*x^10*y^"
            "38+32988*x^9*y^38+4475*x^8*y^38+1586*x^7*y^38+37795*x^6*y^38+21563*x^5*y^38+"
            "41768*x^4*y^38+25716*x^3*y^38+5730*x^2*y^38+6989*x*y^38+2665*y^38+18474*x^37*y^"
            "37+17873*x^36*y^37+37340*x^35*y^37+39770*x^34*y^37+23012*x^33*y^37+12488*x^32*y^"
            "37+35457*x^31*y^37+24228*x^30*y^37+38299*x^29*y^37+31351*x^28*y^37+41876*x^27*y^"
            "37+15132*x^26*y^37+29672*x^25*y^37+19441*x^24*y^37+27592*x^23*y^37+1004*x^22*y^"
            "37+23100*x^21*y^37+24305*x^20*y^37+35140*x^19*y^37+29466*x^18*y^37+34872*x^17*y^"
            "37+31689*x^16*y^37+41226*x^15*y^37+6209*x^14*y^37+18431*x^13*y^37+15545*x^12*y^"
            "37+4217*x^11*y^37+12267*x^10*y^37+23055*x^9*y^37+27958*x^8*y^37+24946*x^7*y^37+"
            "18221*x^6*y^37+3834*x^5*y^37+35345*x^4*y^37+25813*x^3*y^37+20611*x^2*y^37+16263*"
            "x*y^37+20199*y^37+14465*x^36*y^36+3944*x^35*y^36+10818*x^34*y^36+33496*x^33*y^36"
            "+40148*x^32*y^36+40913*x^31*y^36+32228*x^30*y^36+25515*x^29*y^36+27597*x^28*y^36"
            "+40498*x^27*y^36+14302*x^26*y^36+14742*x^25*y^36+18012*x^24*y^36+20790*x^23*y^36"
            "+1636*x^22*y^36+40195*x^21*y^36+40702*x^20*y^36+22371*x^19*y^36+734*x^18*y^36+"
            "23769*x^17*y^36+38645*x^16*y^36+9831*x^15*y^36+5784*x^14*y^36+14661*x^13*y^36+"
            "33984*x^12*y^36+24153*x^11*y^36+2695*x^10*y^36+34150*x^9*y^36+1475*x^8*y^36+"
            "38333*x^7*y^36+860*x^6*y^36+15027*x^5*y^36+19228*x^4*y^36+41952*x^3*y^36+8466*x^"
            "2*y^36+13330*x*y^36+31877*y^36+22137*x^35*y^35+26911*x^34*y^35+3921*x^33*y^35+"
            "30649*x^32*y^35+36112*x^31*y^35+26462*x^30*y^35+18707*x^29*y^35+24132*x^28*y^35+"
            "21788*x^27*y^35+5384*x^26*y^35+31009*x^25*y^35+7078*x^24*y^35+32871*x^23*y^35+"
            "40177*x^22*y^35+34384*x^21*y^35+34362*x^20*y^35+14431*x^19*y^35+17033*x^18*y^35+"
            "37687*x^17*y^35+15153*x^16*y^35+4993*x^15*y^35+7366*x^14*y^35+32202*x^13*y^35+"
            "4531*x^12*y^35+21959*x^11*y^35+10107*x^10*y^35+6830*x^9*y^35+37430*x^8*y^35+"
            "26724*x^7*y^35+29060*x^6*y^35+3962*x^5*y^35+27723*x^4*y^35+42152*x^3*y^35+39841*"
            "x^2*y^35+2484*x*y^35+13587*y^35+466*x^34*y^34+31954*x^33*y^34+30801*x^32*y^34+"
            "8936*x^31*y^34+5640*x^30*y^34+2119*x^29*y^34+38507*x^28*y^34+2916*x^27*y^34+"
            "34260*x^26*y^34+9889*x^25*y^34+17037*x^24*y^34+41972*x^23*y^34+24788*x^22*y^34+"
            "42342*x^21*y^34+12153*x^20*y^34+31900*x^19*y^34+32150*x^18*y^34+353*x^17*y^34+"
            "13211*x^16*y^34+26356*x^15*y^34+22476*x^14*y^34+37191*x^13*y^34+7778*x^12*y^34+"
            "22556*x^11*y^34+5996*x^10*y^34+29641*x^9*y^34+41363*x^8*y^34+30009*x^7*y^34+"
            "22934*x^6*y^34+17759*x^5*y^34+25741*x^4*y^34+6401*x^3*y^34+36544*x^2*y^34+171*x*"
            "y^34+37979*y^34+15313*x^33*y^33+31019*x^32*y^33+17942*x^31*y^33+38375*x^30*y^33+"
            "41049*x^29*y^33+35567*x^28*y^33+17522*x^27*y^33+11884*x^26*y^33+38114*x^25*y^33+"
            "40939*x^24*y^33+37389*x^23*y^33+24852*x^22*y^33+18117*x^21*y^33+10444*x^20*y^33+"
            "4656*x^19*y^33+12501*x^18*y^33+14323*x^17*y^33+12415*x^16*y^33+27799*x^15*y^33+"
            "18562*x^14*y^33+2589*x^13*y^33+27567*x^12*y^33+16865*x^11*y^33+13922*x^10*y^33+"
            "33463*x^9*y^33+17112*x^8*y^33+11243*x^7*y^33+2891*x^6*y^33+32742*x^5*y^33+42456*"
            "x^4*y^33+2235*x^3*y^33+4800*x^2*y^33+40847*x*y^33+5399*y^33+14186*x^32*y^32+"
            "14885*x^31*y^32+41339*x^30*y^32+40829*x^29*y^32+3576*x^28*y^32+38477*x^27*y^32+"
            "8225*x^26*y^32+35327*x^25*y^32+27970*x^24*y^32+14944*x^23*y^32+15098*x^22*y^32+"
            "41185*x^21*y^32+38742*x^20*y^32+22816*x^19*y^32+18799*x^18*y^32+3939*x^17*y^32+"
            "5267*x^16*y^32+3223*x^15*y^32+18224*x^14*y^32+24474*x^13*y^32+12038*x^12*y^32+"
            "21401*x^11*y^32+7939*x^10*y^32+5051*x^9*y^32+25616*x^8*y^32+32954*x^7*y^32+11947"
            "*x^6*y^32+31059*x^5*y^32+9947*x^4*y^32+16462*x^3*y^32+5660*x^2*y^32+4926*x*y^32+"
            "31371*y^32+24980*x^31*y^31+10429*x^30*y^31+29583*x^29*y^31+25670*x^28*y^31+30325"
            "*x^27*y^31+32707*x^26*y^31+4554*x^25*y^31+4824*x^24*y^31+14157*x^23*y^31+2480*x^"
            "22*y^31+26739*x^21*y^31+33788*x^20*y^31+33490*x^19*y^31+13645*x^18*y^31+31574*x^"
            "17*y^31+24871*x^16*y^31+23525*x^15*y^31+25346*x^14*y^31+11150*x^13*y^31+5930*x^"
            "12*y^31+17831*x^11*y^31+25316*x^10*y^31+11055*x^9*y^31+40769*x^8*y^31+39049*x^7*"
            "y^31+6889*x^6*y^31+29402*x^5*y^31+17396*x^4*y^31+26425*x^3*y^31+29573*x^2*y^31+"
            "3048*x*y^31+2061*y^31+25688*x^30*y^30+10729*x^29*y^30+10281*x^28*y^30+2873*x^27*"
            "y^30+7957*x^26*y^30+6862*x^25*y^30+12421*x^24*y^30+22267*x^23*y^30+18530*x^22*y^"
            "30+31964*x^21*y^30+2708*x^20*y^30+29289*x^19*y^30+22534*x^18*y^30+16556*x^17*y^"
            "30+16343*x^16*y^30+24490*x^15*y^30+51*x^14*y^30+13138*x^13*y^30+37878*x^12*y^30+"
            "32475*x^11*y^30+9118*x^10*y^30+23428*x^9*y^30+36826*x^8*y^30+17404*x^7*y^30+6387"
            "*x^6*y^30+39313*x^5*y^30+11115*x^4*y^30+11112*x^3*y^30+39059*x^2*y^30+36567*x*y^"
            "30+3244*y^30+10546*x^29*y^29+40279*x^28*y^29+8192*x^27*y^29+31945*x^26*y^29+8036"
            "*x^25*y^29+7530*x^24*y^29+15828*x^23*y^29+39078*x^22*y^29+22083*x^21*y^29+546*x^"
            "20*y^29+32237*x^19*y^29+15428*x^18*y^29+26710*x^17*y^29+36279*x^16*y^29+493*x^15"
            "*y^29+41644*x^14*y^29+17588*x^13*y^29+37437*x^12*y^29+8873*x^11*y^29+15706*x^10*"
            "y^29+12130*x^9*y^29+10474*x^8*y^29+8157*x^7*y^29+21743*x^6*y^29+5867*x^5*y^29+"
            "36624*x^4*y^29+29051*x^3*y^29+39496*x^2*y^29+11651*x*y^29+21746*y^29+30819*x^28*"
            "y^28+38218*x^27*y^28+33643*x^26*y^28+19933*x^25*y^28+870*x^24*y^28+36669*x^23*y^"
            "28+7497*x^22*y^28+25876*x^21*y^28+1074*x^20*y^28+4707*x^19*y^28+41663*x^18*y^28+"
            "14372*x^17*y^28+31723*x^16*y^28+26094*x^15*y^28+31950*x^14*y^28+7385*x^13*y^28+"
            "34842*x^12*y^28+39265*x^11*y^28+32953*x^10*y^28+3198*x^9*y^28+37808*x^8*y^28+"
            "31909*x^7*y^28+21768*x^6*y^28+15479*x^5*y^28+1803*x^4*y^28+26958*x^3*y^28+21801*"
            "x^2*y^28+24033*x*y^28+22473*y^28+7589*x^27*y^27+4367*x^26*y^27+20572*x^25*y^27+"
            "39710*x^24*y^27+8092*x^23*y^27+25234*x^22*y^27+13211*x^21*y^27+11362*x^20*y^27+"
            "18325*x^19*y^27+12979*x^18*y^27+41351*x^17*y^27+37022*x^16*y^27+34113*x^15*y^27+"
            "10288*x^14*y^27+36321*x^13*y^27+14218*x^12*y^27+37248*x^11*y^27+40101*x^10*y^27+"
            "28540*x^9*y^27+34804*x^8*y^27+9105*x^7*y^27+6660*x^6*y^27+11207*x^5*y^27+22855*x"
            "^4*y^27+40243*x^3*y^27+41892*x^2*y^27+24857*x*y^27+28411*y^27+14847*x^26*y^26+"
            "32215*x^25*y^26+5485*x^24*y^26+4576*x^23*y^26+39245*x^22*y^26+24182*x^21*y^26+"
            "35561*x^20*y^26+5451*x^19*y^26+7418*x^18*y^26+20800*x^17*y^26+17607*x^16*y^26+"
            "12464*x^15*y^26+31957*x^14*y^26+40498*x^13*y^26+13706*x^12*y^26+14134*x^11*y^26+"
            "11623*x^10*y^26+31906*x^9*y^26+9172*x^8*y^26+9678*x^7*y^26+42073*x^6*y^26+21816*"
            "x^5*y^26+9905*x^4*y^26+21512*x^3*y^26+9958*x^2*y^26+37130*x*y^26+18774*y^26+"
            "17837*x^25*y^25+10493*x^24*y^25+27604*x^23*y^25+24967*x^22*y^25+26075*x^21*y^25+"
            "2779*x^20*y^25+14140*x^19*y^25+23181*x^18*y^25+38206*x^17*y^25+12289*x^16*y^25+"
            "17597*x^15*y^25+21453*x^14*y^25+2796*x^13*y^25+10492*x^12*y^25+35608*x^11*y^25+"
            "11080*x^10*y^25+39047*x^9*y^25+11198*x^8*y^25+40664*x^7*y^25+8717*x^6*y^25+2104*"
            "x^5*y^25+6462*x^4*y^25+945*x^3*y^25+4611*x^2*y^25+39716*x*y^25+20421*y^25+25068*"
            "x^24*y^24+37094*x^23*y^24+35791*x^22*y^24+28613*x^21*y^24+16768*x^20*y^24+17224*"
            "x^19*y^24+27360*x^18*y^24+40916*x^17*y^24+2844*x^16*y^24+35191*x^15*y^24+34962*x"
            "^14*y^24+10333*x^13*y^24+34628*x^12*y^24+28608*x^11*y^24+10876*x^10*y^24+23038*x"
            "^9*y^24+33994*x^8*y^24+30860*x^7*y^24+34744*x^6*y^24+14153*x^5*y^24+41239*x^4*y^"
            "24+25076*x^3*y^24+777*x^2*y^24+13338*x*y^24+32637*y^24+38843*x^23*y^23+26338*x^"
            "22*y^23+27918*x^21*y^23+24043*x^20*y^23+1599*x^19*y^23+19731*x^18*y^23+20154*x^"
            "17*y^23+6166*x^16*y^23+20721*x^15*y^23+21112*x^14*y^23+8898*x^13*y^23+38361*x^12"
            "*y^23+37265*x^11*y^23+10091*x^10*y^23+12686*x^9*y^23+13789*x^8*y^23+28215*x^7*y^"
            "23+27651*x^6*y^23+40100*x^5*y^23+28948*x^4*y^23+5650*x^3*y^23+39475*x^2*y^23+"
            "17889*x*y^23+12555*y^23+13632*x^22*y^22+5483*x^21*y^22+10809*x^20*y^22+5154*x^19"
            "*y^22+41705*x^18*y^22+36857*x^17*y^22+23493*x^16*y^22+9585*x^15*y^22+13620*x^14*"
            "y^22+39469*x^13*y^22+37948*x^12*y^22+27976*x^11*y^22+32606*x^10*y^22+10965*x^9*y"
            "^22+36314*x^8*y^22+11877*x^7*y^22+32677*x^6*y^22+30277*x^5*y^22+36275*x^4*y^22+"
            "20185*x^3*y^22+40116*x^2*y^22+7415*x*y^22+30632*y^22+24620*x^21*y^21+21412*x^20*"
            "y^21+830*x^19*y^21+10356*x^18*y^21+872*x^17*y^21+20256*x^16*y^21+15659*x^15*y^21"
            "+19693*x^14*y^21+15424*x^13*y^21+19893*x^12*y^21+5829*x^11*y^21+7319*x^10*y^21+"
            "24476*x^9*y^21+20773*x^8*y^21+289*x^7*y^21+33710*x^6*y^21+574*x^5*y^21+38775*x^4"
            "*y^21+9839*x^3*y^21+15268*x^2*y^21+27786*x*y^21+28450*y^21+20398*x^20*y^20+9431*"
            "x^19*y^20+1699*x^18*y^20+33355*x^17*y^20+2995*x^16*y^20+31114*x^15*y^20+9506*x^"
            "14*y^20+9026*x^13*y^20+30973*x^12*y^20+4232*x^11*y^20+16928*x^10*y^20+10179*x^9*"
            "y^20+4292*x^8*y^20+10780*x^7*y^20+40092*x^6*y^20+5546*x^5*y^20+32176*x^4*y^20+"
            "14031*x^3*y^20+13990*x^2*y^20+26324*x*y^20+40218*y^20+11114*x^19*y^19+30557*x^18"
            "*y^19+9052*x^17*y^19+25115*x^16*y^19+35097*x^15*y^19+8092*x^14*y^19+15130*x^13*y"
            "^19+8004*x^12*y^19+1495*x^11*y^19+27644*x^10*y^19+24610*x^9*y^19+38821*x^8*y^19+"
            "19648*x^7*y^19+26314*x^6*y^19+23318*x^5*y^19+38165*x^4*y^19+33175*x^3*y^19+3381*"
            "x^2*y^19+28440*x*y^19+207*y^19+21053*x^18*y^18+35057*x^17*y^18+8238*x^16*y^18+"
            "33300*x^15*y^18+12824*x^14*y^18+37945*x^13*y^18+33900*x^12*y^18+13601*x^11*y^18+"
            "29959*x^10*y^18+12862*x^9*y^18+21879*x^8*y^18+31948*x^7*y^18+3099*x^6*y^18+15507"
            "*x^5*y^18+5575*x^4*y^18+26045*x^3*y^18+29592*x^2*y^18+10936*x*y^18+15772*y^18+"
            "713*x^17*y^17+32307*x^16*y^17+41726*x^15*y^17+20941*x^14*y^17+28478*x^13*y^17+"
            "18581*x^12*y^17+22577*x^11*y^17+30966*x^10*y^17+8170*x^9*y^17+17869*x^8*y^17+"
            "6744*x^7*y^17+27888*x^6*y^17+32051*x^5*y^17+30810*x^4*y^17+14707*x^3*y^17+38126*"
            "x^2*y^17+33384*x*y^17+23882*y^17+6803*x^16*y^16+30672*x^15*y^16+26280*x^14*y^16+"
            "14477*x^13*y^16+24391*x^12*y^16+11327*x^11*y^16+2705*x^10*y^16+36645*x^9*y^16+"
            "11888*x^8*y^16+32246*x^7*y^16+13623*x^6*y^16+37322*x^5*y^16+38519*x^4*y^16+31179"
            "*x^3*y^16+34600*x^2*y^16+29376*x*y^16+2154*y^16+26152*x^15*y^15+13168*x^14*y^15+"
            "13683*x^13*y^15+4036*x^12*y^15+20263*x^11*y^15+31133*x^10*y^15+26193*x^9*y^15+"
            "6801*x^8*y^15+11653*x^7*y^15+9040*x^6*y^15+36600*x^5*y^15+946*x^4*y^15+4003*x^3*"
            "y^15+9589*x^2*y^15+412*x*y^15+2699*y^15+5480*x^14*y^14+11828*x^13*y^14+10150*x^"
            "12*y^14+26355*x^11*y^14+10819*x^10*y^14+35113*x^9*y^14+15186*x^8*y^14+20558*x^7*"
            "y^14+19722*x^6*y^14+40367*x^5*y^14+5224*x^4*y^14+1701*x^3*y^14+38742*x^2*y^14+"
            "13350*x*y^14+265*y^14+34787*x^13*y^13+23544*x^12*y^13+31722*x^11*y^13+14630*x^10"
            "*y^13+30193*x^9*y^13+34518*x^8*y^13+37652*x^7*y^13+8454*x^6*y^13+23095*x^5*y^13+"
            "40459*x^4*y^13+19841*x^3*y^13+20398*x^2*y^13+33334*x*y^13+41152*y^13+29967*x^12*"
            "y^12+9540*x^11*y^12+36229*x^10*y^12+1670*x^9*y^12+21738*x^8*y^12+744*x^7*y^12+"
            "40820*x^6*y^12+6388*x^5*y^12+14649*x^4*y^12+27675*x^3*y^12+23094*x^2*y^12+15691*"
            "x*y^12+4379*y^12+6842*x^11*y^11+42560*x^10*y^11+31197*x^9*y^11+34010*x^8*y^11+"
            "25953*x^7*y^11+32134*x^6*y^11+33154*x^5*y^11+27110*x^4*y^11+17143*x^3*y^11+20511"
            "*x^2*y^11+38503*x*y^11+17850*y^11+4599*x^10*y^10+22849*x^9*y^10+25896*x^8*y^10+"
            "25981*x^7*y^10+13940*x^6*y^10+34610*x^5*y^10+20111*x^4*y^10+37706*x^3*y^10+39941"
            "*x^2*y^10+22020*x*y^10+30878*y^10+42108*x^9*y^9+39357*x^8*y^9+5802*x^7*y^9+11442"
            "*x^6*y^9+22956*x^5*y^9+13392*x^4*y^9+10615*x^3*y^9+9505*x^2*y^9+2863*x*y^9+33798"
            "*y^9+5939*x^8*y^8+36618*x^7*y^8+13200*x^6*y^8+8689*x^5*y^8+518*x^4*y^8+4026*x^3*"
            "y^8+16640*x^2*y^8+39860*x*y^8+32154*y^8+11496*x^7*y^7+3970*x^6*y^7+16147*x^5*y^7"
            "+42885*x^4*y^7+22064*x^3*y^7+32153*x^2*y^7+39290*x*y^7+37882*y^7+20742*x^6*y^6+"
            "16687*x^5*y^6+1890*x^4*y^6+38545*x^3*y^6+35135*x^2*y^6+16921*x*y^6+42898*y^6+"
            "40861*x^5*y^5+16687*x^4*y^5+22548*x^3*y^5+9752*x^2*y^5+9003*x*y^5+14580*y^5+"
            "18136*x^4*y^4+21446*x^3*y^4+12391*x^2*y^4+29220*x*y^4+13920*y^4+4960*x^3*y^3+"
            "2851*x^2*y^3+1789*x*y^3+33655*y^3+28567*x^2*y^2+24794*x*y^2+19511*y^2+25676*x*y+"
            "8828*y+18760)*(y^55+x*y+x^55)*(y+11888*x)*(y+31162*x)*(y^2+11695*x*y+43050*x^2)*"
            "(y^2+x^2)*(y^2+31356*x*y+43050*x^2)*(y^27+x^55)*(y+x)^2", vars, ctx);
        time = check_omega(16, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        nmod_mpoly_clear(a, ctx);
        nmod_mpoly_ctx_clear(ctx);
    }
    flint_printf("total_time: %wd\n", total_time);


    flint_printf("\n------ characteristic 2 ------\n");
    total_time = 0;
    {
        nmod_mpoly_ctx_t ctx;
        nmod_mpoly_t a;
        const char * vars[] = {"x","y","z","u","v","w","s","t"};

        nmod_mpoly_ctx_init(ctx, 8, ORD_LEX, 2);
        nmod_mpoly_init(a, ctx);

        flint_printf(" #1: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^2*x^2*y^12*s^8+z^2*x^3*y^11*s^8+z^3*x^2*y^11*s^8+z^2*x^4*y^10*s^8+z^4*x^2*y^10*s^8+z^2*x^5*y^9*s^8+z^3*x^4*y^9*s^8+z^4*x^3*y^9*s^8+z^5*x^2*y^9*s^8+z^8*x^11*y^7*s^2+z^8*x^8*y^7*s^2+z^8*x^12*y^6*s^2+z^9*x^11*y^6*s^2+z^8*x^9*y^6*s^2+z^9*x^8*y^6*s^2+z^8*x^13*y^5*s^2+z^10*x^11*y^5*s^2+z^8*x^10*y^5*s^2+z^10*x^8*y^5*s^2+z^8*x^14*y^4*s^2+z^9*x^13*y^4*s^2+z^10*x^12*y^4*s^2+z^11*x^11*y^4*s^2+z^8*x^11*y^4*s^2+z^9*x^10*y^4*s^2+z^10*x^9*y^4*s^2+z^11*x^8*y^4*s^2+z^8*x^3*y^6+z^8*x^4*y^5+z^9*x^3*y^5+z^8*x^5*y^4+z^10*x^3*y^4+z^8*x^6*y^3+z^9*x^5*y^3+z^10*x^4*y^3+z^11*x^3*y^3)*(z^3*x^3*y^4*s^5+z^3*x*y^3*s^5+z^5*x^2*y^3*s^4+z^3*x^4*s^2+z^2*x^3*y^4*s+y)", vars, ctx);
        time = check_omega(12, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf(" #2: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^9*x^7*y^14*s^11+z^9*x^8*y^13*s^11+z^10*x^7*y^13*s^11+z^9*x^9*y^12*s^11+z^11*x^7*y^12*s^11+z^9*x^10*y^11*s^11+z^10*x^9*y^11*s^11+z^11*x^8*y^11*s^11+z^12*x^7*y^11*s^11+z^3*x^10*y^4*s^11+z^3*x^11*y^3*s^11+z^4*x^10*y^3*s^11+z^3*x^12*y^2*s^11+z^5*x^10*y^2*s^11+z^3*x^13*y*s^11+z^4*x^12*y*s^11+z^5*x^11*y*s^11+z^6*x^10*y*s^11+z^6*x^5*y^7*s^9+z^6*x^6*y^6*s^9+z^7*x^5*y^6*s^9+z^6*x^7*y^5*s^9+z^8*x^5*y^5*s^9+z^6*x^8*y^4*s^9+z^7*x^7*y^4*s^9+z^8*x^6*y^4*s^9+z^9*x^5*y^4*s^9+z^11*x^13*y^4*s^3+z^11*x^14*y^3*s^3+z^12*x^13*y^3*s^3+z^11*x^15*y^2*s^3+z^13*x^13*y^2*s^3+z^11*x^16*y*s^3+z^12*x^15*y*s^3+z^13*x^14*y*s^3+z^14*x^13*y*s^3+z^5*x^6*y^15*s^2+z^5*x^7*y^14*s^2+z^6*x^6*y^14*s^2+z^5*x^8*y^13*s^2+z^7*x^6*y^13*s^2+z^5*x^9*y^12*s^2+z^6*x^8*y^12*s^2+z^7*x^7*y^12*s^2+z^8*x^6*y^12*s^2)*(z^3*y^5*s^6+z*x^5*y*s^5+z^3*y^6*s^4+z^2*x*y*s^4+z^5*x^2*s^3+z*y^5*s^2+z^3*x^3*y^2*s^2)", vars, ctx);
        time = check_omega(19, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf(" #3: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z*x^3*y^5*s^5+z^4*x^4*y^5*s^4+y*s^4+z*x^4*y^3*s^3+z^3*x*y^3*s+z^3*y*s)*(z^3*x^4*y^12*s^11+z^3*x^5*y^11*s^11+z^4*x^4*y^11*s^11+z^3*x^6*y^10*s^11+z^5*x^4*y^10*s^11+z^3*x^7*y^9*s^11+z^4*x^6*y^9*s^11+z^5*x^5*y^9*s^11+z^6*x^4*y^9*s^11+z^7*x^5*y^13*s^9+z^7*x^6*y^12*s^9+z^8*x^5*y^12*s^9+z^7*x^7*y^11*s^9+z^9*x^5*y^11*s^9+z^7*x^8*y^10*s^9+z^8*x^7*y^10*s^9+z^9*x^6*y^10*s^9+z^10*x^5*y^10*s^9+z^4*x^9*y^12+z^2*x^8*y^12+z^4*x^10*y^11+z^5*x^9*y^11+z^2*x^9*y^11+z^3*x^8*y^11+z^4*x^11*y^10+z^2*x^10*y^10+z^6*x^9*y^10+z^4*x^8*y^10+z^4*x^12*y^9+z^5*x^11*y^9+z^2*x^11*y^9+z^6*x^10*y^9+z^3*x^10*y^9+z^7*x^9*y^9+z^4*x^9*y^9+z^5*x^8*y^9)", vars, ctx);
        time = check_omega(22, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf(" #4: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^9*x^9*y^17*s^16+z^9*x^10*y^16*s^16+z^10*x^9*y^16*s^16+z^9*x^11*y^15*s^16+z^11*x^9*y^15*s^16+z^9*x^12*y^14*s^16+z^10*x^11*y^14*s^16+z^11*x^10*y^14*s^16+z^12*x^9*y^14*s^16+z^15*x^16*y^7*s^15+z^15*x^17*y^6*s^15+z^16*x^16*y^6*s^15+z^15*x^18*y^5*s^15+z^17*x^16*y^5*s^15+z^15*x^19*y^4*s^15+z^16*x^18*y^4*s^15+z^17*x^17*y^4*s^15+z^18*x^16*y^4*s^15+z^14*x^12*y^12*s^12+z^14*x^13*y^11*s^12+z^15*x^12*y^11*s^12+z^14*x^14*y^10*s^12+z^16*x^12*y^10*s^12+z^14*x^15*y^9*s^12+z^15*x^14*y^9*s^12+z^16*x^13*y^9*s^12+z^17*x^12*y^9*s^12+z^16*x^12*y^20*s^11+z^16*x^13*y^19*s^11+z^17*x^12*y^19*s^11+z^16*x^14*y^18*s^11+z^18*x^12*y^18*s^11+z^16*x^15*y^17*s^11+z^17*x^14*y^17*s^11+z^18*x^13*y^17*s^11+z^19*x^12*y^17*s^11+z^7*x^7*y^18*s^6+z^7*x^8*y^17*s^6+z^8*x^7*y^17*s^6+z^7*x^9*y^16*s^6+z^9*x^7*y^16*s^6+z^7*x^10*y^15*s^6+z^8*x^9*y^15*s^6+z^9*x^8*y^15*s^6+z^10*x^7*y^15*s^6+x^8*y^18*s^2+x^9*y^17*s^2+z*x^8*y^17*s^2+x^10*y^16*s^2+z^2*x^8*y^16*s^2+x^11*y^15*s^2+z*x^10*y^15*s^2+z^2*x^9*y^15*s^2+z^3*x^8*y^15*s^2)*(z^4*x^8*y^5*s^7+z^7*x^5*s^7+z^4*x^3*y^4*s^4+z^6*x*y^5*s^3+z^7*y^8*s+x^5*y^5*s+z^7*x^6*y^2*s)", vars, ctx);
        time = check_omega(19, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf(" #5: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z*x^4*y*s^4+z^3*x^3*y*s^3+z*x^3*y*s^2+z*y*s+z^3*y^3)*(z^9*x^9*y^10*s^7+z^9*x^10*y^9*s^7+z^10*x^9*y^9*s^7+z^9*x^11*y^8*s^7+z^11*x^9*y^8*s^7+z^9*x^12*y^7*s^7+z^10*x^11*y^7*s^7+z^11*x^10*y^7*s^7+z^12*x^9*y^7*s^7+z*x^6*y^4*s^6+z*x^7*y^3*s^6+z^2*x^6*y^3*s^6+z*x^8*y^2*s^6+z^3*x^6*y^2*s^6+z*x^9*y*s^6+z^2*x^8*y*s^6+z^3*x^7*y*s^6+z^4*x^6*y*s^6+z^5*x^9*y^9*s^5+z^5*x^10*y^8*s^5+z^6*x^9*y^8*s^5+z^5*x^11*y^7*s^5+z^7*x^9*y^7*s^5+z^5*x^12*y^6*s^5+z^6*x^11*y^6*s^5+z^7*x^10*y^6*s^5+z^8*x^9*y^6*s^5)", vars, ctx);
        time = check_omega(20, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf(" #6: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "z^3*x^6*y^7*s^5+z^3*x^7*y^6*s^5+z^4*x^6*y^6*s^5+z^3*x^8*y^5*s^5+z^5*x^6*y^5*s^5+z^5*x^5*y^5*s^5+z^3*x^9*y^4*s^5+z^4*x^8*y^4*s^5+z^5*x^7*y^4*s^5+z^6*x^6*y^4*s^5+z^5*x^6*y^4*s^5+z^6*x^5*y^4*s^5+z^5*x^7*y^3*s^5+z^7*x^5*y^3*s^5+z^5*x^8*y^2*s^5+z^6*x^7*y^2*s^5+z^7*x^6*y^2*s^5+z^8*x^5*y^2*s^5+x^4*y^9*s^4+x^5*y^8*s^4+z*x^4*y^8*s^4+x^6*y^7*s^4+z^5*x^4*y^7*s^4+z^2*x^4*y^7*s^4+z^2*x^3*y^7*s^4+x^7*y^6*s^4+z*x^6*y^6*s^4+z^5*x^5*y^6*s^4+z^2*x^5*y^6*s^4+z^6*x^4*y^6*s^4+z^3*x^4*y^6*s^4+z^2*x^4*y^6*s^4+z^3*x^3*y^6*s^4+z^5*x^6*y^5*s^4+z^2*x^5*y^5*s^4+z^7*x^4*y^5*s^4+z^4*x^3*y^5*s^4+z^5*x^7*y^4*s^4+z^6*x^6*y^4*s^4+z^2*x^6*y^4*s^4+z^7*x^5*y^4*s^4+z^3*x^5*y^4*s^4+z^8*x^4*y^4*s^4+z^4*x^4*y^4*s^4+z^5*x^3*y^4*s^4+z^2*x^2*y^9*s^3+z^2*x^3*y^8*s^3+z^3*x^2*y^8*s^3+z^2*x^4*y^7*s^3+z^4*x^2*y^7*s^3+z^2*x^5*y^6*s^3+z^3*x^4*y^6*s^3+z^4*x^3*y^6*s^3+z^5*x^2*y^6*s^3", vars, ctx);
        time = check_omega(12, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf(" #7: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^5*x^3*y^6*s^8+z^4*y^8*s^6+z^3*x^8*y^5*s^6+z^5*x^4*y^7*s^4+z^2*x^4*y^8*s^3+z^3*x^3*y^6*s^3+z^7*y^7*s)*(z^10*x^7*y^9*s^16+z^10*x^8*y^8*s^16+z^11*x^7*y^8*s^16+z^10*x^9*y^7*s^16+z^12*x^7*y^7*s^16+z^10*x^10*y^6*s^16+z^11*x^9*y^6*s^16+z^12*x^8*y^6*s^16+z^13*x^7*y^6*s^16+z*x^15*y^6*s^11+z*x^16*y^5*s^11+z^2*x^15*y^5*s^11+z*x^17*y^4*s^11+z^3*x^15*y^4*s^11+z*x^18*y^3*s^11+z^2*x^17*y^3*s^11+z^3*x^16*y^3*s^11+z^4*x^15*y^3*s^11+z^5*x^2*y^11*s^9+z^5*x^3*y^10*s^9+z^6*x^2*y^10*s^9+z^5*x^4*y^9*s^9+z^7*x^2*y^9*s^9+z^5*x^5*y^8*s^9+z^6*x^4*y^8*s^9+z^7*x^3*y^8*s^9+z^8*x^2*y^8*s^9+z^4*x^15*y^15*s^8+z^15*x^6*y^15*s^8+z^4*x^16*y^14*s^8+z^5*x^15*y^14*s^8+z^15*x^7*y^14*s^8+z^16*x^6*y^14*s^8+z^4*x^17*y^13*s^8+z^6*x^15*y^13*s^8+z^15*x^8*y^13*s^8+z^17*x^6*y^13*s^8+z^4*x^18*y^12*s^8+z^5*x^17*y^12*s^8+z^6*x^16*y^12*s^8+z^7*x^15*y^12*s^8+z^15*x^9*y^12*s^8+z^16*x^8*y^12*s^8+z^17*x^7*y^12*s^8+z^18*x^6*y^12*s^8+z^9*x^17*y^5*s^7+z^9*x^18*y^4*s^7+z^10*x^17*y^4*s^7+z^11*x^14*y^4*s^7+z^9*x^19*y^3*s^7+z^11*x^17*y^3*s^7+z^11*x^15*y^3*s^7+z^12*x^14*y^3*s^7+z^9*x^20*y^2*s^7+z^10*x^19*y^2*s^7+z^11*x^18*y^2*s^7+z^12*x^17*y^2*s^7+z^11*x^16*y^2*s^7+z^13*x^14*y^2*s^7+z^11*x^17*y*s^7+z^12*x^16*y*s^7+z^13*x^15*y*s^7+z^14*x^14*y*s^7)", vars, ctx);
        time = check_omega(24, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf(" #8: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "z^5*x*y^8*s^5+z^5*x^2*y^7*s^5+z^6*x*y^7*s^5+z^5*x^3*y^6*s^5+z^7*x*y^6*s^5+z^5*x^4*y^5*s^5+z^6*x^3*y^5*s^5+z^7*x^2*y^5*s^5+z^8*x*y^5*s^5+z^4*x^2*y^6*s^4+z^4*x^3*y^5*s^4+z^5*x^2*y^5*s^4+z^4*x^4*y^4*s^4+z^6*x^2*y^4*s^4+z^4*x^5*y^3*s^4+z^5*x^4*y^3*s^4+z^6*x^3*y^3*s^4+z^7*x^2*y^3*s^4+z^4*x^3*y^6*s^3+z^4*x^4*y^5*s^3+z^5*x^3*y^5*s^3+z^4*x^5*y^4*s^3+z^6*x^3*y^4*s^3+z^4*x^6*y^3*s^3+z^5*x^5*y^3*s^3+z^6*x^4*y^3*s^3+z^7*x^3*y^3*s^3+z^2*x^4*y^9*s^2+z^2*x^5*y^8*s^2+z^3*x^4*y^8*s^2+z^2*x^6*y^7*s^2+z^4*x^4*y^7*s^2+z^2*x^7*y^6*s^2+z^3*x^6*y^6*s^2+z^4*x^5*y^6*s^2+z^5*x^4*y^6*s^2+z*x^5*y^7*s+z*x^6*y^6*s+z^2*x^5*y^6*s+z*x^7*y^5*s+z^3*x^5*y^5*s+z*x^8*y^4*s+z^2*x^7*y^4*s+z^3*x^6*y^4*s+z^4*x^5*y^4*s+z*x^6*y^7+z*x^7*y^6+z^2*x^6*y^6+z*x^8*y^5+z^3*x^6*y^5+z*x^9*y^4+z^2*x^8*y^4+z^3*x^7*y^4+z^4*x^6*y^4", vars, ctx);
        time = check_omega(10, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf(" #9: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^5*x*y^10*s^9+z^5*x^2*y^9*s^9+z^6*x*y^9*s^9+z^5*x^3*y^8*s^9+z^7*x*y^8*s^9+z^5*x^4*y^7*s^9+z^6*x^3*y^7*s^9+z^7*x^2*y^7*s^9+z^8*x*y^7*s^9+z^9*x^8*y^15*s^7+z^9*x^9*y^14*s^7+z^10*x^8*y^14*s^7+z^9*x^10*y^13*s^7+z^11*x^8*y^13*s^7+z^9*x^11*y^12*s^7+z^10*x^10*y^12*s^7+z^11*x^9*y^12*s^7+z^12*x^8*y^12*s^7+x^6*y^6*s^3+x^7*y^5*s^3+z*x^6*y^5*s^3+x^8*y^4*s^3+z^2*x^6*y^4*s^3+x^9*y^3*s^3+z*x^8*y^3*s^3+z^2*x^7*y^3*s^3+z^3*x^6*y^3*s^3+z^4*x^10*y^14+z^4*x^11*y^13+z^5*x^10*y^13+z^4*x^12*y^12+z^6*x^10*y^12+z^4*x^13*y^11+z^5*x^12*y^11+z^6*x^11*y^11+z^7*x^10*y^11)*(z^3*x^3*y^4*s^5+z^5*x^4*y*s^4+z^4*x^2*y*s^4+z^5*x^5*y^2*s+z^4*x^3*s+x*y^2)", vars, ctx);
        time = check_omega(10, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#10: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "z^7*x^2*y^10*s^6+z^7*x^3*y^9*s^6+z^8*x^2*y^9*s^6+z^7*x^4*y^8*s^6+z^9*x^2*y^8*s^6+z^7*x^5*y^7*s^6+z^8*x^4*y^7*s^6+z^9*x^3*y^7*s^6+z^10*x^2*y^7*s^6+z^5*x^3*y^10*s^4+z^7*x^2*y^10*s^4+z^5*x^4*y^9*s^4+z^7*x^3*y^9*s^4+z^6*x^3*y^9*s^4+z^8*x^2*y^9*s^4+z^5*x^5*y^8*s^4+z^7*x^4*y^8*s^4+z^7*x^3*y^8*s^4+z^9*x^2*y^8*s^4+z^5*x^6*y^7*s^4+z^7*x^5*y^7*s^4+z^6*x^5*y^7*s^4+z^8*x^4*y^7*s^4+z^7*x^4*y^7*s^4+z^9*x^3*y^7*s^4+z^8*x^3*y^7*s^4+z^10*x^2*y^7*s^4+z^3*x^5*y^10*s^2+z^3*x^6*y^9*s^2+z^4*x^5*y^9*s^2+z^3*x^7*y^8*s^2+z^5*x^5*y^8*s^2+z^3*x^8*y^7*s^2+z^4*x^7*y^7*s^2+z^5*x^6*y^7*s^2+z^6*x^5*y^7*s^2+z*x^6*y^10+z^3*x^5*y^10+z*x^7*y^9+z^3*x^6*y^9+z^2*x^6*y^9+z^4*x^5*y^9+z*x^8*y^8+z^3*x^7*y^8+z^3*x^6*y^8+z^5*x^5*y^8+z*x^9*y^7+z^3*x^8*y^7+z^2*x^8*y^7+z^4*x^7*y^7+z^3*x^7*y^7+z^5*x^6*y^7+z^4*x^6*y^7+z^6*x^5*y^7", vars, ctx);
        time = check_omega(15, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#11: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^2*x^4*y^13*s^10+z^2*x^5*y^12*s^10+z^3*x^4*y^12*s^10+z^2*x^6*y^11*s^10+z^4*x^4*y^11*s^10+z^2*x^7*y^10*s^10+z^3*x^6*y^10*s^10+z^4*x^5*y^10*s^10+z^5*x^4*y^10*s^10+z^2*x^7*y^6*s^10+z^2*x^8*y^5*s^10+z^3*x^7*y^5*s^10+z^3*x^10*y^4*s^10+z^2*x^9*y^4*s^10+z^4*x^7*y^4*s^10+z^3*x^11*y^3*s^10+z^4*x^10*y^3*s^10+z^2*x^10*y^3*s^10+z^3*x^9*y^3*s^10+z^4*x^8*y^3*s^10+z^5*x^7*y^3*s^10+z^3*x^12*y^2*s^10+z^5*x^10*y^2*s^10+z^3*x^13*y*s^10+z^4*x^12*y*s^10+z^5*x^11*y*s^10+z^6*x^10*y*s^10)*(z^5*y^2*s^4+x^5*y^3*s^3+x^2*s^2+z^4*x*y^3+z^5*x^5*y^2)", vars, ctx);
        time = check_omega(22, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#12: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "z^5*x*y^5*s^3+z^5*x^2*y^4*s^3+z^6*x*y^4*s^3+z^5*x^3*y^3*s^3+z^7*x*y^3*s^3+z^5*x^4*y^2*s^3+z^6*x^3*y^2*s^3+z^7*x^2*y^2*s^3+z^8*x*y^2*s^3+z^7*x^4*y^10*s^2+z^7*x^5*y^9*s^2+z^8*x^4*y^9*s^2+z^7*x^6*y^8*s^2+z^9*x^4*y^8*s^2+z^7*x^7*y^7*s^2+z^8*x^6*y^7*s^2+z^9*x^5*y^7*s^2+z^10*x^4*y^7*s^2+z^3*x^3*y^6*s^2+z^4*x^2*y^6*s^2+z^3*x^4*y^5*s^2+z^5*x^2*y^5*s^2+z^3*x^5*y^4*s^2+z^4*x^4*y^4*s^2+z^5*x^3*y^4*s^2+z^6*x^2*y^4*s^2+z^3*x^6*y^3*s^2+z^7*x^2*y^3*s^2+z^5*x^6*y^11*s+z^6*x^5*y^11*s+z^5*x^7*y^10*s+z^7*x^5*y^10*s+z^5*x^8*y^9*s+z^6*x^7*y^9*s+z^7*x^6*y^9*s+z^8*x^5*y^9*s+z^5*x^9*y^8*s+z^9*x^5*y^8*s", vars, ctx);
        time = check_omega(12, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#13: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "z^5*x^7*y^7*s^5+z^5*x^8*y^6*s^5+z^6*x^7*y^6*s^5+z^5*x^9*y^5*s^5+z^7*x^7*y^5*s^5+z^4*x^6*y^5*s^5+z^5*x^10*y^4*s^5+z^6*x^9*y^4*s^5+z^7*x^8*y^4*s^5+z^8*x^7*y^4*s^5+z^4*x^7*y^4*s^5+z^5*x^6*y^4*s^5+z^4*x^8*y^3*s^5+z^6*x^6*y^3*s^5+z^4*x^9*y^2*s^5+z^5*x^8*y^2*s^5+z^6*x^7*y^2*s^5+z^7*x^6*y^2*s^5+z^4*x^7*y^7*s^3+z^4*x^8*y^6*s^3+z^5*x^7*y^6*s^3+z^4*x^9*y^5*s^3+z^6*x^7*y^5*s^3+z^4*x^10*y^4*s^3+z^5*x^9*y^4*s^3+z^6*x^8*y^4*s^3+z^7*x^7*y^4*s^3+z^3*x^3*y^11*s^2+z^3*x^4*y^10*s^2+z^4*x^3*y^10*s^2+z^3*x^5*y^9*s^2+z^5*x^3*y^9*s^2+z^2*x^2*y^9*s^2+z^3*x^6*y^8*s^2+z^4*x^5*y^8*s^2+z^5*x^4*y^8*s^2+z^6*x^3*y^8*s^2+z^2*x^3*y^8*s^2+z^3*x^2*y^8*s^2+z^2*x^4*y^7*s^2+z^4*x^2*y^7*s^2+z^2*x^5*y^6*s^2+z^3*x^4*y^6*s^2+z^4*x^3*y^6*s^2+z^5*x^2*y^6*s^2+z^2*x^3*y^11+z^2*x^4*y^10+z^3*x^3*y^10+z^2*x^5*y^9+z^4*x^3*y^9+z^2*x^6*y^8+z^3*x^5*y^8+z^4*x^4*y^8+z^5*x^3*y^8", vars, ctx);
        time = check_omega(11, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#14: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(x^9*y^4*s^8+z^9*x^6*y^4*s^8+x^10*y^3*s^8+z*x^9*y^3*s^8+z^9*x^7*y^3*s^8+z^10*x^6*y^3*s^8+x^11*y^2*s^8+z^2*x^9*y^2*s^8+z^9*x^8*y^2*s^8+z^11*x^6*y^2*s^8+x^12*y*s^8+z*x^11*y*s^8+z^2*x^10*y*s^8+z^9*x^9*y*s^8+z^3*x^9*y*s^8+z^10*x^8*y*s^8+z^11*x^7*y*s^8+z^12*x^6*y*s^8+z*x^2*y^13*s+z*x^3*y^12*s+z^2*x^2*y^12*s+z*x^4*y^11*s+z^6*x^3*y^11*s+z^3*x^2*y^11*s+z*x^5*y^10*s+z^6*x^4*y^10*s+z^2*x^4*y^10*s+z^7*x^3*y^10*s+z^3*x^3*y^10*s+z^4*x^2*y^10*s+z^6*x^5*y^9*s+z^8*x^3*y^9*s+z^6*x^6*y^8*s+z^7*x^5*y^8*s+z^8*x^4*y^8*s+z^9*x^3*y^8*s)*(z^4*x^5*y*s^5+z^2*y*s^5+z^5*x^4*s^5+y^2*s^3+x^2*y^4*s^2+z^4*x^5*y^5)", vars, ctx);
        time = check_omega(9, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#15: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "z^7*x^8*y^7*s^3+z^7*x^9*y^6*s^3+z^8*x^8*y^6*s^3+z^9*x^7*y^6*s^3+z^7*x^10*y^5*s^3+z^10*x^7*y^5*s^3+z^7*x^11*y^4*s^3+z^8*x^10*y^4*s^3+z^10*x^8*y^4*s^3+z^11*x^7*y^4*s^3+z^9*x^10*y^3*s^3+z^10*x^9*y^3*s^3+z^11*x^8*y^3*s^3+z^12*x^7*y^3*s^3+z^8*x^8*y^8*s+z^8*x^9*y^7*s+z^9*x^8*y^7*s+z^10*x^7*y^7*s+z^6*x^11*y^6*s+z^8*x^10*y^6*s+z^11*x^7*y^6*s+z^6*x^12*y^5*s+z^8*x^11*y^5*s+z^7*x^11*y^5*s+z^9*x^10*y^5*s+z^8*x^10*y^5*s+z^11*x^8*y^5*s+z^12*x^7*y^5*s+z^6*x^13*y^4*s+z^10*x^10*y^4*s+z^9*x^10*y^4*s+z^11*x^9*y^4*s+z^12*x^8*y^4*s+z^13*x^7*y^4*s+z^6*x^14*y^3*s+z^7*x^13*y^3*s+z^9*x^11*y^3*s+z^10*x^10*y^3*s+z^8*x^13*y^2*s+z^9*x^12*y^2*s+z^10*x^11*y^2*s+z^11*x^10*y^2*s", vars, ctx);
        time = check_omega(21, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#16: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^5*x^2*y^6*s^6+z^4*x^2*y*s^5+z^6*x^3*y^6*s^4+z*y^5*s^4+z*x^5*y^4*s^3+x^4*y*s^2+z^6*x^3*y^6*s)*(z^13*x^5*y^14*s^9+z^13*x^6*y^13*s^9+z^14*x^5*y^13*s^9+z^13*x^7*y^12*s^9+z^15*x^5*y^12*s^9+z^13*x^8*y^11*s^9+z^14*x^7*y^11*s^9+z^15*x^6*y^11*s^9+z^16*x^5*y^11*s^9+z^9*x^10*y^7*s^7+z^9*x^11*y^6*s^7+z^10*x^10*y^6*s^7+z^9*x^12*y^5*s^7+z^11*x^10*y^5*s^7+z^9*x^13*y^4*s^7+z^10*x^12*y^4*s^7+z^11*x^11*y^4*s^7+z^12*x^10*y^4*s^7+x^5*y^8*s^6+x^6*y^7*s^6+z*x^5*y^7*s^6+x^7*y^6*s^6+z^2*x^5*y^6*s^6+x^8*y^5*s^6+z*x^7*y^5*s^6+z^2*x^6*y^5*s^6+z^3*x^5*y^5*s^6+z^7*x^13*y^16*s^2+z^7*x^14*y^15*s^2+z^8*x^13*y^15*s^2+z^7*x^15*y^14*s^2+z^9*x^13*y^14*s^2+z^7*x^16*y^13*s^2+z^8*x^15*y^13*s^2+z^9*x^14*y^13*s^2+z^10*x^13*y^13*s^2+z^6*x^13*y^10+z^6*x^14*y^9+z^7*x^13*y^9+z^6*x^15*y^8+z^8*x^13*y^8+z^6*x^16*y^7+z^7*x^15*y^7+z^8*x^14*y^7+z^9*x^13*y^7)", vars, ctx);
        time = check_omega(16, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#17: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^14*x^5*y^18*s^13+z^14*x^6*y^17*s^13+z^15*x^5*y^17*s^13+z^14*x^7*y^16*s^13+z^16*x^5*y^16*s^13+z^14*x^8*y^15*s^13+z^15*x^7*y^15*s^13+z^16*x^6*y^15*s^13+z^17*x^5*y^15*s^13+x*y^18*s^12+x^2*y^17*s^12+z*x*y^17*s^12+x^3*y^16*s^12+z^2*x*y^16*s^12+x^4*y^15*s^12+z*x^3*y^15*s^12+z^2*x^2*y^15*s^12+z^3*x*y^15*s^12+z*x^10*y^14*s^12+z*x^11*y^13*s^12+z^2*x^10*y^13*s^12+z*x^12*y^12*s^12+z^3*x^10*y^12*s^12+z*x^13*y^11*s^12+z^2*x^12*y^11*s^12+z^3*x^11*y^11*s^12+z^4*x^10*y^11*s^12+z^10*x^4*y^13*s^11+z^10*x^5*y^12*s^11+z^11*x^4*y^12*s^11+z^10*x^6*y^11*s^11+z^12*x^4*y^11*s^11+z^10*x^7*y^10*s^11+z^11*x^6*y^10*s^11+z^12*x^5*y^10*s^11+z^13*x^4*y^10*s^11+z^7*x^2*y^4*s^11+z^7*x^3*y^3*s^11+z^8*x^2*y^3*s^11+z^7*x^4*y^2*s^11+z^9*x^2*y^2*s^11+z^7*x^5*y*s^11+z^8*x^4*y*s^11+z^9*x^3*y*s^11+z^10*x^2*y*s^11+z^4*x^2*y^7*s^6+z^4*x^3*y^6*s^6+z^5*x^2*y^6*s^6+z^4*x^4*y^5*s^6+z^6*x^2*y^5*s^6+z^4*x^5*y^4*s^6+z^5*x^4*y^4*s^6+z^6*x^3*y^4*s^6+z^7*x^2*y^4*s^6)*(z^4*x^2*y^6*s^7+z^5*x^5*y^5*s^7+z^6*x^4*y^3*s^5+z^5*x^6*y^5*s^4+z^7*x^4*s^3+z^4*x*y^2*s+z^2*x^4)", vars, ctx);
        time = check_omega(16, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#18: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^4*x^2*y^10*s^9+z^4*x^3*y^9*s^9+1*x^3*y^9*s^9+z^5*x^2*y^9*s^9+z^4*x^4*y^8*s^9+1*x^4*y^8*s^9+z*x^3*y^8*s^9+z^6*x^2*y^8*s^9+z^4*x^5*y^7*s^9+1*x^5*y^7*s^9+z^5*x^4*y^7*s^9+z^6*x^3*y^7*s^9+z^2*x^3*y^7*s^9+z^7*x^2*y^7*s^9+x^6*y^6*s^9+z*x^5*y^6*s^9+z^2*x^4*y^6*s^9+z^3*x^3*y^6*s^9+z^8*x^7*y^5*s^8+z^8*x^8*y^4*s^8+z^9*x^7*y^4*s^8+z^8*x^9*y^3*s^8+z^10*x^7*y^3*s^8+z^8*x^10*y^2*s^8+z^9*x^9*y^2*s^8+z^10*x^8*y^2*s^8+z^11*x^7*y^2*s^8+z^7*x^3*y^7*s^3+z^7*x^4*y^6*s^3+z^8*x^3*y^6*s^3+z^7*x^5*y^5*s^3+z^9*x^3*y^5*s^3+z^7*x^6*y^4*s^3+z^8*x^5*y^4*s^3+z^9*x^4*y^4*s^3+z^10*x^3*y^4*s^3)*(x^5*y^5*s^4+z*y^4*s^4+x*y^2*s^4+z^2*x^4*s^4+z^3*x^2*y^2*s+z^5*y)", vars, ctx);
        time = check_omega(12, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#19: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^7*x^13*y^17*s^16+z^7*x^14*y^16*s^16+z^8*x^13*y^16*s^16+z^7*x^15*y^15*s^16+z^9*x^13*y^15*s^16+z^7*x^16*y^14*s^16+z^8*x^15*y^14*s^16+z^9*x^14*y^14*s^16+z^10*x^13*y^14*s^16+z^3*x^4*y^17*s^14+z^3*x^5*y^16*s^14+z^4*x^4*y^16*s^14+z^3*x^6*y^15*s^14+z^5*x^4*y^15*s^14+z^3*x^7*y^14*s^14+z^4*x^6*y^14*s^14+z^5*x^5*y^14*s^14+z^6*x^4*y^14*s^14+z^8*x^5*y^6*s^13+z^8*x^6*y^5*s^13+z^9*x^5*y^5*s^13+z^8*x^7*y^4*s^13+z^10*x^5*y^4*s^13+z^8*x^8*y^3*s^13+z^9*x^7*y^3*s^13+z^10*x^6*y^3*s^13+z^11*x^5*y^3*s^13+z^6*x^2*y^12*s^8+z^6*x^3*y^11*s^8+z^7*x^2*y^11*s^8+z^6*x^4*y^10*s^8+z^8*x^2*y^10*s^8+z^6*x^5*y^9*s^8+z^7*x^4*y^9*s^8+z^8*x^3*y^9*s^8+z^9*x^2*y^9*s^8+z^5*x*y^5*s^5+z^5*x^2*y^4*s^5+z^6*x*y^4*s^5+z^5*x^3*y^3*s^5+z^7*x*y^3*s^5+z^5*x^4*y^2*s^5+z^6*x^3*y^2*s^5+z^7*x^2*y^2*s^5+z^8*x*y^2*s^5+z^15*x^5*y^8*s+z^15*x^6*y^7*s+z^16*x^5*y^7*s+z^15*x^7*y^6*s+z^17*x^5*y^6*s+z^15*x^8*y^5*s+z^16*x^7*y^5*s+z^17*x^6*y^5*s+z^18*x^5*y^5*s)*(z^3*x^2*y^3*s^7+z^6*x*y^2*s^6+z^7*x^3*y*s^5+z^6*x^5*s^3+z^5*x^3*s^3+z^2*x^4*y^7*s^2+z^3*y^3*s^2)*(z^3*x^3*y^3+z^2*x^3*y^3+z^4*x^2*y^3+z*y^3+z^3*x*y^2+z*x*y^2+z^4*y+x)", vars, ctx);
        time = check_omega(17, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#20: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^7*x^11*y^17*s^12+z^7*x^12*y^16*s^12+z^8*x^11*y^16*s^12+z^7*x^13*y^15*s^12+z^9*x^11*y^15*s^12+z^7*x^14*y^14*s^12+z^8*x^13*y^14*s^12+z^9*x^12*y^14*s^12+z^10*x^11*y^14*s^12+z^6*x^6*y^17*s^6+z^6*x^7*y^16*s^6+z^7*x^6*y^16*s^6+z^6*x^8*y^15*s^6+z^8*x^6*y^15*s^6+z^6*x^9*y^14*s^6+z^7*x^8*y^14*s^6+z^8*x^7*y^14*s^6+z^9*x^6*y^14*s^6+z^9*x^10*y^10*s^6+z^9*x^11*y^9*s^6+z^10*x^10*y^9*s^6+z^9*x^12*y^8*s^6+z^11*x^10*y^8*s^6+z^9*x^13*y^7*s^6+z^10*x^12*y^7*s^6+z^11*x^11*y^7*s^6+z^12*x^10*y^7*s^6+z^4*x*y^8*s^2+z^4*x^2*y^7*s^2+z^5*x*y^7*s^2+z^4*x^3*y^6*s^2+z^6*x*y^6*s^2+z^4*x^4*y^5*s^2+z^5*x^3*y^5*s^2+z^6*x^2*y^5*s^2+z^7*x*y^5*s^2+z^4*x^9*y^5*s+z^4*x^10*y^4*s+z^5*x^9*y^4*s+z^4*x^11*y^3*s+z^6*x^9*y^3*s+z^4*x^12*y^2*s+z^5*x^11*y^2*s+z^6*x^10*y^2*s+z^7*x^9*y^2*s)*(y^2*s^5+x^3*y^6*s^4+z^6*x^6*y^2*s^4+z^6*x^6*y^6*s^3+z^5*x^2*s^3+z^3*x^6*y^3*s^2+z^6*x^6*y^6*s)*(z^2*x*y^3+z^2*x^3*y+x*y+z^2*y+z^3*x^3+z*x)", vars, ctx);
        time = check_omega(15, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#21: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^4*x*y^13*s^4*t^9+z^4*x^2*y^12*s^4*t^9+z^5*x*y^12*s^4*t^9+z^4*x^3*y^11*s^4*t^9+z^6*x*y^11*s^4*t^9+z^4*x^4*y^10*s^4*t^9+z^5*x^3*y^10*s^4*t^9+z^6*x^2*y^10*s^4*t^9+z^7*x*y^10*s^4*t^9+x^6*y^13*s^2*t^8+x^7*y^12*s^2*t^8+z*x^6*y^12*s^2*t^8+x^8*y^11*s^2*t^8+z^2*x^6*y^11*s^2*t^8+x^9*y^10*s^2*t^8+z*x^8*y^10*s^2*t^8+z^2*x^7*y^10*s^2*t^8+z^3*x^6*y^10*s^2*t^8+z^2*x^6*y^6*s^2*t^2+z^2*x^7*y^5*s^2*t^2+z^3*x^6*y^5*s^2*t^2+z^2*x^8*y^4*s^2*t^2+z^4*x^6*y^4*s^2*t^2+z^2*x^9*y^3*s^2*t^2+z^3*x^8*y^3*s^2*t^2+z^4*x^7*y^3*s^2*t^2+z^5*x^6*y^3*s^2*t^2)*(x^3*y^2*s^3*u^4+z^4*x*y*s^2*u^4+z*x^4*y*u^2+z*y*u^2+z^4*x^3*y*s^3)*(z^2*x*y^2+y+x+z^2+1)", vars, ctx);
        time = check_omega(15, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#22: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^2*x^13*y^18*s^10*t^11+z^2*x^14*y^17*s^10*t^11+z^3*x^13*y^17*s^10*t^11+z^2*x^15*y^16*s^10*t^11+z^4*x^13*y^16*s^10*t^11+z^2*x^16*y^15*s^10*t^11+z^3*x^15*y^15*s^10*t^11+z^4*x^14*y^15*s^10*t^11+z^5*x^13*y^15*s^10*t^11+z*x^17*y^5*s^7*t^9+z*x^18*y^4*s^7*t^9+z^2*x^17*y^4*s^7*t^9+z*x^19*y^3*s^7*t^9+z^3*x^17*y^3*s^7*t^9+z*x^20*y^2*s^7*t^9+z^2*x^19*y^2*s^7*t^9+z^3*x^18*y^2*s^7*t^9+z^4*x^17*y^2*s^7*t^9+z^11*x^2*y^16*s^2*t^8+z^11*x^3*y^15*s^2*t^8+z^12*x^2*y^15*s^2*t^8+z^11*x^4*y^14*s^2*t^8+z^13*x^2*y^14*s^2*t^8+z^11*x^5*y^13*s^2*t^8+z^12*x^4*y^13*s^2*t^8+z^13*x^3*y^13*s^2*t^8+z^14*x^2*y^13*s^2*t^8+z^6*x^8*y^15*s^10*t^6+z^6*x^9*y^14*s^10*t^6+z^7*x^8*y^14*s^10*t^6+z^6*x^10*y^13*s^10*t^6+z^8*x^8*y^13*s^10*t^6+z^6*x^11*y^12*s^10*t^6+z^7*x^10*y^12*s^10*t^6+z^8*x^9*y^12*s^10*t^6+z^9*x^8*y^12*s^10*t^6+z^10*x^7*y^14*s*t^4+z^10*x^8*y^13*s*t^4+z^11*x^7*y^13*s*t^4+z^10*x^9*y^12*s*t^4+z^12*x^7*y^12*s*t^4+z^10*x^10*y^11*s*t^4+z^11*x^9*y^11*s*t^4+z^12*x^8*y^11*s*t^4+z^13*x^7*y^11*s*t^4+z^6*x^7*y^4*t+z^6*x^8*y^3*t+z^7*x^7*y^3*t+z^6*x^9*y^2*t+z^8*x^7*y^2*t+z^6*x^10*y*t+z^7*x^9*y*t+z^8*x^8*y*t+z^9*x^7*y*t)*(x^5*y^3*s^7*u^5+z^5*x^2*y*s*u^4+x^4*y^4*s^8*u^3+x*s^6*u+x^6*y^6*s^2*u+z^4*x^3*y^6*s^2*u+z*x*y^7*u)*(z^3*x^3*y^2+z^2*x^2*y^2+z*y^2+z^3*x^3*y+z*y+z^2*x^4+z^3*x^2+1)", vars, ctx);
        time = check_omega(13, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#23: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^8*x^9*y^17*s^14*t^14+z^8*x^10*y^16*s^14*t^14+z^9*x^9*y^16*s^14*t^14+z^8*x^11*y^15*s^14*t^14+z^10*x^9*y^15*s^14*t^14+z^8*x^12*y^14*s^14*t^14+z^9*x^11*y^14*s^14*t^14+z^10*x^10*y^14*s^14*t^14+z^11*x^9*y^14*s^14*t^14+x^3*y^10*t^10+x^4*y^9*t^10+z*x^3*y^9*t^10+x^5*y^8*t^10+z^2*x^3*y^8*t^10+x^6*y^7*t^10+z*x^5*y^7*t^10+z^2*x^4*y^7*t^10+z^3*x^3*y^7*t^10+z^7*x^10*y^4*s^4*t^9+z^7*x^11*y^3*s^4*t^9+z^8*x^10*y^3*s^4*t^9+z^7*x^12*y^2*s^4*t^9+z^9*x^10*y^2*s^4*t^9+z^7*x^13*y*s^4*t^9+z^8*x^12*y*s^4*t^9+z^9*x^11*y*s^4*t^9+z^10*x^10*y*s^4*t^9+z^14*x^14*y^12*s^11*t^8+z^14*x^15*y^11*s^11*t^8+z^15*x^14*y^11*s^11*t^8+z^14*x^16*y^10*s^11*t^8+z^16*x^14*y^10*s^11*t^8+z^14*x^17*y^9*s^11*t^8+z^15*x^16*y^9*s^11*t^8+z^16*x^15*y^9*s^11*t^8+z^17*x^14*y^9*s^11*t^8+z^6*x^13*y^13*s^13*t^7+z^6*x^14*y^12*s^13*t^7+z^7*x^13*y^12*s^13*t^7+z^6*x^15*y^11*s^13*t^7+z^8*x^13*y^11*s^13*t^7+z^6*x^16*y^10*s^13*t^7+z^7*x^15*y^10*s^13*t^7+z^8*x^14*y^10*s^13*t^7+z^9*x^13*y^10*s^13*t^7+z^15*x^9*y^4*s^3+z^15*x^10*y^3*s^3+z^16*x^9*y^3*s^3+z^15*x^11*y^2*s^3+z^17*x^9*y^2*s^3+z^15*x^12*y*s^3+z^16*x^11*y*s^3+z^17*x^10*y*s^3+z^18*x^9*y*s^3)*(z^6*y^5*s^4*u^7+z^7*x^7*y^3*s^3*u^3+z^7*x^7*y^3*s^7*u^2+z^4*x^3*y^6*s^3*u+z^2*x^4*y^3*s^2*u+z^3*x^7*y*s*u+z^5*x*y^4*u)*(z*x^3*y^2+z^2*y^2+z^2*x*y+z*x*y+1*y+z*x^2+z^2)", vars, ctx);
        time = check_omega(14, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#24: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^7*x^5*y^14*s^4*t^9+z^7*x^6*y^13*s^4*t^9+z^8*x^5*y^13*s^4*t^9+z^7*x^7*y^12*s^4*t^9+z^9*x^5*y^12*s^4*t^9+z^7*x^8*y^11*s^4*t^9+z^8*x^7*y^11*s^4*t^9+z^9*x^6*y^11*s^4*t^9+z^10*x^5*y^11*s^4*t^9+z^7*x^4*y^14*s^4*t^7+z^7*x^5*y^13*s^4*t^7+z^8*x^4*y^13*s^4*t^7+z^7*x^6*y^12*s^4*t^7+z^9*x^4*y^12*s^4*t^7+z^7*x^7*y^11*s^4*t^7+z^8*x^6*y^11*s^4*t^7+z^9*x^5*y^11*s^4*t^7+z^10*x^4*y^11*s^4*t^7+z^10*x^13*y^10*s^3*t^7+z^10*x^14*y^9*s^3*t^7+z^11*x^13*y^9*s^3*t^7+z^10*x^15*y^8*s^3*t^7+z^12*x^13*y^8*s^3*t^7+z^10*x^16*y^7*s^3*t^7+z^11*x^15*y^7*s^3*t^7+z^12*x^14*y^7*s^3*t^7+z^13*x^13*y^7*s^3*t^7+z^12*x^5*y^13*s^4*t^4+z^12*x^6*y^12*s^4*t^4+z^13*x^5*y^12*s^4*t^4+z^12*x^7*y^11*s^4*t^4+z^14*x^5*y^11*s^4*t^4+z^12*x^8*y^10*s^4*t^4+z^13*x^7*y^10*s^4*t^4+z^14*x^6*y^10*s^4*t^4+z^15*x^5*y^10*s^4*t^4+z^6*x^13*y^12*s^3*t^2+z^6*x^14*y^11*s^3*t^2+z^7*x^13*y^11*s^3*t^2+z^6*x^15*y^10*s^3*t^2+z^8*x^13*y^10*s^3*t^2+z^6*x^16*y^9*s^3*t^2+z^7*x^15*y^9*s^3*t^2+z^8*x^14*y^9*s^3*t^2+z^9*x^13*y^9*s^3*t^2)*(x*y^3*s^5*u^6+z^3*x^2*y^3*s^3*u^4+z^4*x^6*s^5*u^3+z^2*x^6*s^3*u^3+z^6*x^2*y^4*s^6*u^2+z^2*y^5*s^4*u^2+z*y^5*s^6)*(z^2*x^2*y^3+z^3*y^3+x^2*y^2+z^3*x*y^2+z^3*x^2*y+z^3)", vars, ctx);
        time = check_omega(31, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#25: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^11*x^7*y^9*s^7*t^11+z^11*x^8*y^8*s^7*t^11+z^12*x^7*y^8*s^7*t^11+z^11*x^9*y^7*s^7*t^11+z^13*x^7*y^7*s^7*t^11+z^11*x^10*y^6*s^7*t^11+z^12*x^9*y^6*s^7*t^11+z^13*x^8*y^6*s^7*t^11+z^14*x^7*y^6*s^7*t^11+z^10*x^8*y^12*s^8*t^9+z^10*x^9*y^11*s^8*t^9+z^11*x^8*y^11*s^8*t^9+z^10*x^10*y^10*s^8*t^9+z^12*x^8*y^10*s^8*t^9+z^10*x^11*y^9*s^8*t^9+z^11*x^10*y^9*s^8*t^9+z^12*x^9*y^9*s^8*t^9+z^13*x^8*y^9*s^8*t^9+z^5*x^2*y^7*s^3*t^7+z^5*x^3*y^6*s^3*t^7+z^6*x^2*y^6*s^3*t^7+z^5*x^4*y^5*s^3*t^7+z^7*x^2*y^5*s^3*t^7+z^5*x^5*y^4*s^3*t^7+z^6*x^4*y^4*s^3*t^7+z^7*x^3*y^4*s^3*t^7+z^8*x^2*y^4*s^3*t^7+x^3*y^12*t^5+x^4*y^11*t^5+z*x^3*y^11*t^5+x^5*y^10*t^5+z^2*x^3*y^10*t^5+x^6*y^9*t^5+z*x^5*y^9*t^5+z^2*x^4*y^9*t^5+z^3*x^3*y^9*t^5+z^10*x^4*y^7*s^8*t^4+z^10*x^5*y^6*s^8*t^4+z^11*x^4*y^6*s^8*t^4+z^10*x^6*y^5*s^8*t^4+z^12*x^4*y^5*s^8*t^4+z^10*x^7*y^4*s^8*t^4+z^11*x^6*y^4*s^8*t^4+z^12*x^5*y^4*s^8*t^4+z^13*x^4*y^4*s^8*t^4)*(z^3*y^5*s^3*u^6+z^4*x^2*y^2*s^4*u^5+z^2*x*s^4*u^5+z*y^3*s^2*u^3+z^6*y^5*u^2+x^2*s^3+x^2*y*s^2)*(z^2*y^2+z*x^2*y+z^2*x+1)", vars, ctx);
        time = check_omega(16, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#26: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^8*x^4*y^6*s^7*t^10+z^8*x^5*y^5*s^7*t^10+z^9*x^4*y^5*s^7*t^10+z^8*x^6*y^4*s^7*t^10+z^10*x^4*y^4*s^7*t^10+z^8*x^7*y^3*s^7*t^10+z^9*x^6*y^3*s^7*t^10+z^10*x^5*y^3*s^7*t^10+z^11*x^4*y^3*s^7*t^10+z^2*x^8*y^14*s^5*t^10+z^2*x^9*y^13*s^5*t^10+z^3*x^8*y^13*s^5*t^10+z^2*x^10*y^12*s^5*t^10+z^4*x^8*y^12*s^5*t^10+z^2*x^11*y^11*s^5*t^10+z^3*x^10*y^11*s^5*t^10+z^4*x^9*y^11*s^5*t^10+z^5*x^8*y^11*s^5*t^10+z^8*x^2*y^13*s^6*t^6+z^8*x^3*y^12*s^6*t^6+z^9*x^2*y^12*s^6*t^6+z^8*x^4*y^11*s^6*t^6+z^10*x^2*y^11*s^6*t^6+z^8*x^5*y^10*s^6*t^6+z^9*x^4*y^10*s^6*t^6+z^10*x^3*y^10*s^6*t^6+z^11*x^2*y^10*s^6*t^6+z^8*x^6*y^15*s^6+z^8*x^7*y^14*s^6+z^9*x^6*y^14*s^6+z^8*x^8*y^13*s^6+z^10*x^6*y^13*s^6+z^8*x^9*y^12*s^6+z^9*x^8*y^12*s^6+z^10*x^7*y^12*s^6+z^11*x^6*y^12*s^6)*(z*x^2*y^3*s^4*u^4+x^3*y^2*s^4*u^4+z^2*x^2*s^4*u^2+z^4*x^3*u^2+z^5*x^2*y^2*s^5*u+z^2*x^5*y^2)*(x^2*y^2+z^2*y^2+z*x*y+1*x*y)", vars, ctx);
        time = check_omega(21, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#27: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^2*x^2*y^10*s^8*t^3+z^2*x^3*y^9*s^8*t^3+z^3*x^2*y^9*s^8*t^3+z^2*x^4*y^8*s^8*t^3+z^4*x^2*y^8*s^8*t^3+z^2*x^5*y^7*s^8*t^3+z^3*x^4*y^7*s^8*t^3+z^4*x^3*y^7*s^8*t^3+z^5*x^2*y^7*s^8*t^3+z*x^11*y^9*t^3+z*x^12*y^8*t^3+z^2*x^11*y^8*t^3+z*x^13*y^7*t^3+z^3*x^11*y^7*t^3+z*x^14*y^6*t^3+z^2*x^13*y^6*t^3+z^3*x^12*y^6*t^3+z^4*x^11*y^6*t^3+x^11*y^9*s^2*t+x^12*y^8*s^2*t+z*x^11*y^8*s^2*t+x^13*y^7*s^2*t+z^2*x^11*y^7*s^2*t+x^14*y^6*s^2*t+z*x^13*y^6*s^2*t+z^2*x^12*y^6*s^2*t+z^3*x^11*y^6*s^2*t)*(z^4*x^2*y^2*s^3*u^3+z^2*x^3*y^2*s^3*u^2+z^3*x^3*y^2*s^2*u+z*x^3*y^4*s*u+z^4*x^4*u)*(z^2*y^2+x^2*y+z*x*y+z^2*y+1*y+x)", vars, ctx);
        time = check_omega(19, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#28: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^7*x^7*y^8*s^3*t^7+z^7*x^8*y^7*s^3*t^7+z^8*x^7*y^7*s^3*t^7+z^7*x^9*y^6*s^3*t^7+z^9*x^7*y^6*s^3*t^7+z^7*x^10*y^5*s^3*t^7+z^8*x^9*y^5*s^3*t^7+z^9*x^8*y^5*s^3*t^7+z^10*x^7*y^5*s^3*t^7+z^9*x^8*y^10*s^5*t^3+z^9*x^9*y^9*s^5*t^3+z^10*x^8*y^9*s^5*t^3+z^9*x^10*y^8*s^5*t^3+z^11*x^8*y^8*s^5*t^3+z^9*x^11*y^7*s^5*t^3+z^10*x^10*y^7*s^5*t^3+z^11*x^9*y^7*s^5*t^3+z^12*x^8*y^7*s^5*t^3+z^6*x^8*y^13*s^4+z^6*x^9*y^12*s^4+z^7*x^8*y^12*s^4+z^6*x^10*y^11*s^4+z^8*x^8*y^11*s^4+z^6*x^11*y^10*s^4+z^7*x^10*y^10*s^4+z^8*x^9*y^10*s^4+z^9*x^8*y^10*s^4)*(z^4*x^2*s^3*u^4+z*x^2*y^3*s^3*u^3+z^2*x^4*y^2*s*u^2+z^4*x^3*y^4*s^3*u+z^3*x^2*y^3*s^4)*(y+z*x^2+x+z+1)", vars, ctx);
        time = check_omega(31, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#29: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(x^10*y^13*s^3*t^13+x^11*y^12*s^3*t^13+z*x^10*y^12*s^3*t^13+x^12*y^11*s^3*t^13+z^2*x^10*y^11*s^3*t^13+x^13*y^10*s^3*t^13+z*x^12*y^10*s^3*t^13+z^2*x^11*y^10*s^3*t^13+z^3*x^10*y^10*s^3*t^13+z^13*x^7*y^17*s^2*t^5+z^13*x^8*y^16*s^2*t^5+z^14*x^7*y^16*s^2*t^5+z^13*x^9*y^15*s^2*t^5+z^15*x^7*y^15*s^2*t^5+z^13*x^10*y^14*s^2*t^5+z^14*x^9*y^14*s^2*t^5+z^15*x^8*y^14*s^2*t^5+z^16*x^7*y^14*s^2*t^5+z^2*x^14*y^4*s^12*t^3+z^2*x^15*y^3*s^12*t^3+z^3*x^14*y^3*s^12*t^3+z^2*x^16*y^2*s^12*t^3+z^4*x^14*y^2*s^12*t^3+z^2*x^17*y*s^12*t^3+z^3*x^16*y*s^12*t^3+z^4*x^15*y*s^12*t^3+z^5*x^14*y*s^12*t^3+z^4*x^14*y^15*s^3*t^2+z^4*x^15*y^14*s^3*t^2+z^5*x^14*y^14*s^3*t^2+z^4*x^16*y^13*s^3*t^2+z^6*x^14*y^13*s^3*t^2+z^4*x^17*y^12*s^3*t^2+z^5*x^16*y^12*s^3*t^2+z^6*x^15*y^12*s^3*t^2+z^7*x^14*y^12*s^3*t^2+z^5*x^11*y^15*s^3+z^5*x^12*y^14*s^3+z^6*x^11*y^14*s^3+z^5*x^13*y^13*s^3+z^7*x^11*y^13*s^3+z^5*x^14*y^12*s^3+z^6*x^13*y^12*s^3+z^7*x^12*y^12*s^3+z^8*x^11*y^12*s^3)*(z*x^4*y^3*u^6+z^3*x^5*y^4*s^5*u^4+z*x^5*y^4*s^5*u^4+z^4*x^6*y^4*s*u^4+z*x*y^5*s^4*u^2+z^5*y^6*s^2*u^2+z^5*x^4*y^5)*(z*x^2*y^3+1*y^3+z^2*x^2*y^2+z^2*x*y^2+z*y^2+z^2*x*y)", vars, ctx);
        time = check_omega(21, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#30: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^2*x^4*y^6+z^2*x^5*y^5+z^3*x^4*y^5+z^2*x^6*y^4+z^4*x^4*y^4+z^2*x^7*y^3+z^3*x^6*y^3+z^4*x^5*y^3+z^5*x^4*y^3)*(z*x*y*s*u+y*s)", vars, ctx);
        time = check_omega(15, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#31: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^7*x^14*y^14*s^13*t^11+z^7*x^15*y^13*s^13*t^11+z^8*x^14*y^13*s^13*t^11+z^7*x^16*y^12*s^13*t^11+z^9*x^14*y^12*s^13*t^11+z^7*x^17*y^11*s^13*t^11+z^8*x^16*y^11*s^13*t^11+z^9*x^15*y^11*s^13*t^11+z^10*x^14*y^11*s^13*t^11+z^12*x^13*y^16*s^15*t^9+z^12*x^14*y^15*s^15*t^9+z^13*x^13*y^15*s^15*t^9+z^12*x^15*y^14*s^15*t^9+z^14*x^13*y^14*s^15*t^9+z^12*x^16*y^13*s^15*t^9+z^13*x^15*y^13*s^15*t^9+z^14*x^14*y^13*s^15*t^9+z^15*x^13*y^13*s^15*t^9+z^14*x^15*y^9*s^4*t^8+z^14*x^16*y^8*s^4*t^8+z^15*x^15*y^8*s^4*t^8+z^14*x^17*y^7*s^4*t^8+z^16*x^15*y^7*s^4*t^8+z^14*x^18*y^6*s^4*t^8+z^15*x^17*y^6*s^4*t^8+z^16*x^16*y^6*s^4*t^8+z^17*x^15*y^6*s^4*t^8+z^8*x^16*y^18*s^14*t^7+z^8*x^17*y^17*s^14*t^7+z^9*x^16*y^17*s^14*t^7+z^8*x^18*y^16*s^14*t^7+z^10*x^16*y^16*s^14*t^7+z^8*x^19*y^15*s^14*t^7+z^9*x^18*y^15*s^14*t^7+z^10*x^17*y^15*s^14*t^7+z^11*x^16*y^15*s^14*t^7+z^3*x^10*y^4*s*t^5+z^3*x^11*y^3*s*t^5+z^4*x^10*y^3*s*t^5+z^3*x^12*y^2*s*t^5+z^5*x^10*y^2*s*t^5+z^3*x^13*y*s*t^5+z^4*x^12*y*s*t^5+z^5*x^11*y*s*t^5+z^6*x^10*y*s*t^5+z^2*x^6*y^15*s^15*t^3+z^2*x^7*y^14*s^15*t^3+z^3*x^6*y^14*s^15*t^3+z^2*x^8*y^13*s^15*t^3+z^4*x^6*y^13*s^15*t^3+z^2*x^9*y^12*s^15*t^3+z^3*x^8*y^12*s^15*t^3+z^4*x^7*y^12*s^15*t^3+z^5*x^6*y^12*s^15*t^3)*(z^6*x^7*y^2*s^5*u^6+x^2*s^8*u^5+z^7*x^5*y^6*s^2*u^5+z^2*x^7*u^5+z^5*x^7*y^6*s*u^4+z^7*x^8*y^6*s^2*u+z^6*x^4*y^7)*(z*x^3*y^3+x^2*y^3+z^2*x*y^3+x^3*y^2+z^3*x*y^2+x^2*y+z^3*x^2+z^2*x)", vars, ctx);
        time = check_omega(22, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#32: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^8*x^3*y^5*s^3*t^14+z^8*x^4*y^4*s^3*t^14+z^9*x^3*y^4*s^3*t^14+z^8*x^5*y^3*s^3*t^14+z^10*x^3*y^3*s^3*t^14+z^8*x^6*y^2*s^3*t^14+z^9*x^5*y^2*s^3*t^14+z^10*x^4*y^2*s^3*t^14+z^11*x^3*y^2*s^3*t^14+z^4*x^11*y^17*s^12*t^12+z^4*x^12*y^16*s^12*t^12+z^5*x^11*y^16*s^12*t^12+z^4*x^13*y^15*s^12*t^12+z^6*x^11*y^15*s^12*t^12+z^4*x^14*y^14*s^12*t^12+z^5*x^13*y^14*s^12*t^12+z^6*x^12*y^14*s^12*t^12+z^7*x^11*y^14*s^12*t^12+z^3*x*y^10*s^14*t^4+z^3*x^2*y^9*s^14*t^4+z^4*x*y^9*s^14*t^4+z^3*x^3*y^8*s^14*t^4+z^5*x*y^8*s^14*t^4+z^3*x^4*y^7*s^14*t^4+z^4*x^3*y^7*s^14*t^4+z^5*x^2*y^7*s^14*t^4+z^6*x*y^7*s^14*t^4+z^7*x^4*y^14*s^9*t^3+z^7*x^5*y^13*s^9*t^3+z^8*x^4*y^13*s^9*t^3+z^7*x^6*y^12*s^9*t^3+z^9*x^4*y^12*s^9*t^3+z^7*x^7*y^11*s^9*t^3+z^8*x^6*y^11*s^9*t^3+z^9*x^5*y^11*s^9*t^3+z^10*x^4*y^11*s^9*t^3+x^11*y^16*s^10*t^2+x^12*y^15*s^10*t^2+z*x^11*y^15*s^10*t^2+x^13*y^14*s^10*t^2+z^2*x^11*y^14*s^10*t^2+x^14*y^13*s^10*t^2+z*x^13*y^13*s^10*t^2+z^2*x^12*y^13*s^10*t^2+z^3*x^11*y^13*s^10*t^2+z^11*x^5*y^11*s^2+z^11*x^6*y^10*s^2+z^12*x^5*y^10*s^2+z^11*x^7*y^9*s^2+z^13*x^5*y^9*s^2+z^11*x^8*y^8*s^2+z^12*x^7*y^8*s^2+z^13*x^6*y^8*s^2+z^14*x^5*y^8*s^2)*(z^3*x^5*y^5*s^5*u^7+z^2*x^4*y^3*s^3*u^5+z^3*y^2*s^2*u^5+x^6*y^3*s^4*u^4+z*x*y^3*u^3+z^3*x^5*y^3*s^5*u^2+z*x^7*y^7*s^5*u)*(y^2+z*y+1*y+x^2+z^3*x)", vars, ctx);
        time = check_omega(14, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#33: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^6*x^6*y^4*s^8*t^12+z^6*x^7*y^3*s^8*t^12+z^7*x^6*y^3*s^8*t^12+z^6*x^8*y^2*s^8*t^12+z^8*x^6*y^2*s^8*t^12+z^6*x^9*y*s^8*t^12+z^7*x^8*y*s^8*t^12+z^8*x^7*y*s^8*t^12+z^9*x^6*y*s^8*t^12+x*y^7*s^7*t^8+x^2*y^6*s^7*t^8+z*x*y^6*s^7*t^8+x^3*y^5*s^7*t^8+z^2*x*y^5*s^7*t^8+x^4*y^4*s^7*t^8+z*x^3*y^4*s^7*t^8+z^2*x^2*y^4*s^7*t^8+z^3*x*y^4*s^7*t^8+x^2*y^9*s^8*t^7+x^3*y^8*s^8*t^7+z*x^2*y^8*s^8*t^7+x^4*y^7*s^8*t^7+z^2*x^2*y^7*s^8*t^7+x^5*y^6*s^8*t^7+z*x^4*y^6*s^8*t^7+z^2*x^3*y^6*s^8*t^7+z^3*x^2*y^6*s^8*t^7+z^4*x^8*y^5*s^5*t^7+z^4*x^9*y^4*s^5*t^7+z^5*x^8*y^4*s^5*t^7+z^4*x^10*y^3*s^5*t^7+z^6*x^8*y^3*s^5*t^7+z^4*x^11*y^2*s^5*t^7+z^5*x^10*y^2*s^5*t^7+z^6*x^9*y^2*s^5*t^7+z^7*x^8*y^2*s^5*t^7+z^9*x^10*y^6*s^9*t+z^9*x^11*y^5*s^9*t+z^10*x^10*y^5*s^9*t+z^9*x^12*y^4*s^9*t+z^11*x^10*y^4*s^9*t+z^9*x^13*y^3*s^9*t+z^10*x^12*y^3*s^9*t+z^11*x^11*y^3*s^9*t+z^12*x^10*y^3*s^9*t)*(z^3*x^3*y^5*s^5*u^5+z^3*x^3*y^3*s^6*u^2+z^2*x^2*y^2*s^4*u+z^4*x^3*y^4*s^2*u+z^3*x^2*y^5*s^2+z^3*x^3*y^2*s^2+z^4*x^5*y^4*s)*(z^2*x*y^2+z*x*y^2+z^2*y+1)", vars, ctx);
        time = check_omega(21, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#34: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z*x^9*y^6*s^2*t^5+z*x^10*y^5*s^2*t^5+z^2*x^9*y^5*s^2*t^5+z*x^11*y^4*s^2*t^5+z^3*x^9*y^4*s^2*t^5+z*x^12*y^3*s^2*t^5+z^2*x^11*y^3*s^2*t^5+z^3*x^10*y^3*s^2*t^5+z^4*x^9*y^3*s^2*t^5+z^4*x^7*y^6*s^4+z^4*x^8*y^5*s^4+z^5*x^7*y^5*s^4+z^4*x^9*y^4*s^4+z^6*x^7*y^4*s^4+z^4*x^10*y^3*s^4+z^5*x^9*y^3*s^4+z^6*x^8*y^3*s^4+z^7*x^7*y^3*s^4)*(z*x^4*s^3*u^3+z*x^3*y*u^3+z*x^4*s^4+x^2*y^2)*(z*x*y)", vars, ctx);
        time = check_omega(23, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#35: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^3*x*y^20*s^5*t^17+z^3*x^2*y^19*s^5*t^17+z^4*x*y^19*s^5*t^17+z^3*x^3*y^18*s^5*t^17+z^5*x*y^18*s^5*t^17+z^3*x^4*y^17*s^5*t^17+z^4*x^3*y^17*s^5*t^17+z^5*x^2*y^17*s^5*t^17+z^6*x*y^17*s^5*t^17+z^17*x^15*y^16*s^16*t^16+z^17*x^16*y^15*s^16*t^16+z^18*x^15*y^15*s^16*t^16+z^17*x^17*y^14*s^16*t^16+z^19*x^15*y^14*s^16*t^16+z^17*x^18*y^13*s^16*t^16+z^18*x^17*y^13*s^16*t^16+z^19*x^16*y^13*s^16*t^16+z^20*x^15*y^13*s^16*t^16+z^2*x^16*y^4*s^13*t^13+z^2*x^17*y^3*s^13*t^13+z^3*x^16*y^3*s^13*t^13+z^2*x^18*y^2*s^13*t^13+z^4*x^16*y^2*s^13*t^13+z^2*x^19*y*s^13*t^13+z^3*x^18*y*s^13*t^13+z^4*x^17*y*s^13*t^13+z^5*x^16*y*s^13*t^13+z^11*x^2*y^4*s^8*t^11+z^11*x^3*y^3*s^8*t^11+z^12*x^2*y^3*s^8*t^11+z^11*x^4*y^2*s^8*t^11+z^13*x^2*y^2*s^8*t^11+z^11*x^5*y*s^8*t^11+z^12*x^4*y*s^8*t^11+z^13*x^3*y*s^8*t^11+z^14*x^2*y*s^8*t^11+z^8*x^12*y^14*s*t^11+z^8*x^13*y^13*s*t^11+z^9*x^12*y^13*s*t^11+z^8*x^14*y^12*s*t^11+z^10*x^12*y^12*s*t^11+z^8*x^15*y^11*s*t^11+z^9*x^14*y^11*s*t^11+z^10*x^13*y^11*s*t^11+z^11*x^12*y^11*s*t^11+z^14*x^11*y^12*s^14*t^5+z^14*x^12*y^11*s^14*t^5+z^15*x^11*y^11*s^14*t^5+z^14*x^13*y^10*s^14*t^5+z^16*x^11*y^10*s^14*t^5+z^14*x^14*y^9*s^14*t^5+z^15*x^13*y^9*s^14*t^5+z^16*x^12*y^9*s^14*t^5+z^17*x^11*y^9*s^14*t^5+z^7*x^12*y^15*s^10+z^7*x^13*y^14*s^10+z^8*x^12*y^14*s^10+z^7*x^14*y^13*s^10+z^9*x^12*y^13*s^10+z^7*x^15*y^12*s^10+z^8*x^14*y^12*s^10+z^9*x^13*y^12*s^10+z^10*x^12*y^12*s^10)*(z^6*x^7*y^5*s^6*u^8+z^4*x^6*y^5*s^2*u^8+z^4*x^4*y*s^7*u^7+z^7*x^8*y^7*s^4*u^7+z^4*x^4*y^6*s^3*u^5+z^4*y^5*s^2*u^4+z^8*s^4)*(z*x^3*y^4+x^3*y^3+z^2*x^2*y^3+z*y^3+z*x^3*y^2+z^3*x^3*y+z^2*x^3*y+z*x*y+z^2*x^3)", vars, ctx);
        time = check_omega(17, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#36: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^13*x^11*y^16*s^13*t^13+z^13*x^12*y^15*s^13*t^13+z^14*x^11*y^15*s^13*t^13+z^13*x^13*y^14*s^13*t^13+z^15*x^11*y^14*s^13*t^13+z^13*x^14*y^13*s^13*t^13+z^14*x^13*y^13*s^13*t^13+z^15*x^12*y^13*s^13*t^13+z^16*x^11*y^13*s^13*t^13+x^8*y^13*s^2*t^12+x^9*y^12*s^2*t^12+z*x^8*y^12*s^2*t^12+x^10*y^11*s^2*t^12+z^2*x^8*y^11*s^2*t^12+x^11*y^10*s^2*t^12+z*x^10*y^10*s^2*t^12+z^2*x^9*y^10*s^2*t^12+z^3*x^8*y^10*s^2*t^12+z^11*x^6*y^16*s^6*t^8+z^11*x^7*y^15*s^6*t^8+z^12*x^6*y^15*s^6*t^8+z^11*x^8*y^14*s^6*t^8+z^13*x^6*y^14*s^6*t^8+z^11*x^9*y^13*s^6*t^8+z^12*x^8*y^13*s^6*t^8+z^13*x^7*y^13*s^6*t^8+z^14*x^6*y^13*s^6*t^8+z^5*x^2*y^9*s^11*t^6+z^5*x^3*y^8*s^11*t^6+z^6*x^2*y^8*s^11*t^6+z^5*x^4*y^7*s^11*t^6+z^7*x^2*y^7*s^11*t^6+z^5*x^5*y^6*s^11*t^6+z^6*x^4*y^6*s^11*t^6+z^7*x^3*y^6*s^11*t^6+z^8*x^2*y^6*s^11*t^6+z^14*x^5*y^9*s^14*t+z^14*x^6*y^8*s^14*t+z^15*x^5*y^8*s^14*t+z^14*x^7*y^7*s^14*t+z^16*x^5*y^7*s^14*t+z^14*x^8*y^6*s^14*t+z^15*x^7*y^6*s^14*t+z^16*x^6*y^6*s^14*t+z^17*x^5*y^6*s^14*t+z^12*x^9*y^17*s^14+z^12*x^10*y^16*s^14+z^13*x^9*y^16*s^14+z^12*x^11*y^15*s^14+z^14*x^9*y^15*s^14+z^12*x^12*y^14*s^14+z^13*x^11*y^14*s^14+z^14*x^10*y^14*s^14+z^15*x^9*y^14*s^14)*(z^6*x^4*s^4*u^7+z^3*y*u^6+z^4*x^4*y*s^5*u^5+z^4*x^4*y^3*s^4*u^5+z^3*x^5*y^3*s^7*u^4+z^7*y*s^5*u^3+z^4*y^7*s^5)*(z^3*y^3+z*x^2*y^2+z^3*x^3*y+z*x^3*y+z^3*x*y+x^2+z^3)", vars, ctx);
        time = check_omega(19, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#37: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^5*x^13*y^15*s^8*t^13+z^5*x^14*y^14*s^8*t^13+z^6*x^13*y^14*s^8*t^13+z^5*x^15*y^13*s^8*t^13+z^7*x^13*y^13*s^8*t^13+z^5*x^16*y^12*s^8*t^13+z^6*x^15*y^12*s^8*t^13+z^7*x^14*y^12*s^8*t^13+z^8*x^13*y^12*s^8*t^13+z^3*x*y^6*t^11+z^3*x^2*y^5*t^11+z^4*x*y^5*t^11+z^3*x^3*y^4*t^11+z^5*x*y^4*t^11+z^3*x^4*y^3*t^11+z^4*x^3*y^3*t^11+z^5*x^2*y^3*t^11+z^6*x*y^3*t^11+z^5*x^6*y^16*s^5*t^9+z^5*x^7*y^15*s^5*t^9+z^6*x^6*y^15*s^5*t^9+z^5*x^8*y^14*s^5*t^9+z^7*x^6*y^14*s^5*t^9+z^5*x^9*y^13*s^5*t^9+z^6*x^8*y^13*s^5*t^9+z^7*x^7*y^13*s^5*t^9+z^8*x^6*y^13*s^5*t^9+z^6*x^11*y^6*s^11*t^7+z^6*x^12*y^5*s^11*t^7+z^7*x^11*y^5*s^11*t^7+z^6*x^13*y^4*s^11*t^7+z^8*x^11*y^4*s^11*t^7+z^6*x^14*y^3*s^11*t^7+z^7*x^13*y^3*s^11*t^7+z^8*x^12*y^3*s^11*t^7+z^9*x^11*y^3*s^11*t^7+z^3*x^14*y^16*s^2*t^3+z^3*x^15*y^15*s^2*t^3+z^4*x^14*y^15*s^2*t^3+z^3*x^16*y^14*s^2*t^3+z^5*x^14*y^14*s^2*t^3+z^3*x^17*y^13*s^2*t^3+z^4*x^16*y^13*s^2*t^3+z^5*x^15*y^13*s^2*t^3+z^6*x^14*y^13*s^2*t^3)*(x^4*y*s^6*u^3+x*y^2*s^4*u^3+x^3*y*s^4*u+x*y^4*u+x^5*y^4*s^4+z^6*y^4*s^3+x*y^6)*(z*y^2+z*x^3*y+z^3*y+1*y+z*x+z^2)", vars, ctx);
        time = check_omega(17, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#38: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^5*x^14*y^17*s^6*t^12+z^5*x^15*y^16*s^6*t^12+z^6*x^14*y^16*s^6*t^12+z^5*x^16*y^15*s^6*t^12+z^7*x^14*y^15*s^6*t^12+z^5*x^17*y^14*s^6*t^12+z^6*x^16*y^14*s^6*t^12+z^7*x^15*y^14*s^6*t^12+z^8*x^14*y^14*s^6*t^12+z^12*x^4*y^17*s^6*t^10+z^12*x^5*y^16*s^6*t^10+z^13*x^4*y^16*s^6*t^10+z^12*x^6*y^15*s^6*t^10+z^14*x^4*y^15*s^6*t^10+z^12*x^7*y^14*s^6*t^10+z^13*x^6*y^14*s^6*t^10+z^14*x^5*y^14*s^6*t^10+z^15*x^4*y^14*s^6*t^10+z^6*x^5*y^6*s^9*t^9+z^6*x^6*y^5*s^9*t^9+z^7*x^5*y^5*s^9*t^9+z^6*x^7*y^4*s^9*t^9+z^8*x^5*y^4*s^9*t^9+z^6*x^8*y^3*s^9*t^9+z^7*x^7*y^3*s^9*t^9+z^8*x^6*y^3*s^9*t^9+z^9*x^5*y^3*s^9*t^9+z^12*x^12*y^14*s^2*t^6+z^12*x^13*y^13*s^2*t^6+z^13*x^12*y^13*s^2*t^6+z^12*x^14*y^12*s^2*t^6+z^14*x^12*y^12*s^2*t^6+z^12*x^15*y^11*s^2*t^6+z^13*x^14*y^11*s^2*t^6+z^14*x^13*y^11*s^2*t^6+z^15*x^12*y^11*s^2*t^6+z^3*x^12*y^18*s*t^6+z^3*x^13*y^17*s*t^6+z^4*x^12*y^17*s*t^6+z^3*x^14*y^16*s*t^6+z^5*x^12*y^16*s*t^6+z^3*x^15*y^15*s*t^6+z^4*x^14*y^15*s*t^6+z^5*x^13*y^15*s*t^6+z^6*x^12*y^15*s*t^6+z^6*x^14*y^13*s^2*t+z^6*x^15*y^12*s^2*t+z^7*x^14*y^12*s^2*t+z^6*x^16*y^11*s^2*t+z^8*x^14*y^11*s^2*t+z^6*x^17*y^10*s^2*t+z^7*x^16*y^10*s^2*t+z^8*x^15*y^10*s^2*t+z^9*x^14*y^10*s^2*t)*(z^4*x^6*y^7*s*u^7+z^6*x^4*y^2*u^6+z*x*s^3*u^5+z^3*x*y*s^7*u^2+z^2*y^2*s^5*u^2+z^3*x*y^4*s^7*u+z^5*x*y*s*u)*(x^3*y^3+z*x^2*y^3+z^2*x*y^3+z^2*x^2*y^2+z*x^2*y^2)", vars, ctx);
        time = check_omega(23, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#39: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^5*x^10*y^13*s^5*t^11+z^5*x^11*y^12*s^5*t^11+z^6*x^10*y^12*s^5*t^11+z^5*x^12*y^11*s^5*t^11+z^7*x^10*y^11*s^5*t^11+z^5*x^13*y^10*s^5*t^11+z^6*x^12*y^10*s^5*t^11+z^7*x^11*y^10*s^5*t^11+z^8*x^10*y^10*s^5*t^11+z^8*x^2*y^15*s^3*t^11+z^8*x^3*y^14*s^3*t^11+z^9*x^2*y^14*s^3*t^11+z^8*x^4*y^13*s^3*t^11+z^10*x^2*y^13*s^3*t^11+z^8*x^5*y^12*s^3*t^11+z^9*x^4*y^12*s^3*t^11+z^10*x^3*y^12*s^3*t^11+z^11*x^2*y^12*s^3*t^11+z^14*x^14*y^5*s^14*t^10+z^14*x^15*y^4*s^14*t^10+z^15*x^14*y^4*s^14*t^10+z^14*x^16*y^3*s^14*t^10+z^16*x^14*y^3*s^14*t^10+z^14*x^17*y^2*s^14*t^10+z^15*x^16*y^2*s^14*t^10+z^16*x^15*y^2*s^14*t^10+z^17*x^14*y^2*s^14*t^10+z*x^12*y^11*s*t^7+z*x^13*y^10*s*t^7+z^2*x^12*y^10*s*t^7+z*x^14*y^9*s*t^7+z^3*x^12*y^9*s*t^7+z*x^15*y^8*s*t^7+z^2*x^14*y^8*s*t^7+z^3*x^13*y^8*s*t^7+z^4*x^12*y^8*s*t^7+z^11*x^16*y^13*s^10*t^3+z^11*x^17*y^12*s^10*t^3+z^12*x^16*y^12*s^10*t^3+z^11*x^18*y^11*s^10*t^3+z^13*x^16*y^11*s^10*t^3+z^11*x^19*y^10*s^10*t^3+z^12*x^18*y^10*s^10*t^3+z^13*x^17*y^10*s^10*t^3+z^14*x^16*y^10*s^10*t^3+z^5*x^4*y^6*s^2*t^3+z^5*x^5*y^5*s^2*t^3+z^6*x^4*y^5*s^2*t^3+z^5*x^6*y^4*s^2*t^3+z^7*x^4*y^4*s^2*t^3+z^5*x^7*y^3*s^2*t^3+z^6*x^6*y^3*s^2*t^3+z^7*x^5*y^3*s^2*t^3+z^8*x^4*y^3*s^2*t^3)*(z^4*x^2*y^3*s^5*u^7+z^3*x^6*y^2*s^7*u^6+z^2*x*y*s*u^4+y*s^4*u^3+z^3*x^4*y^3*s^4*u^2+z^7*x*s^6*u+z^5*x^6*y^7*s)*(z*x^2*y^3+z^3*x*y^3+y^2+z*x*y+z^2*y)", vars, ctx);
        time = check_omega(17, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#40: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^8*x^14+z^8*x^12+1*x^12+z^8*x^10+z^6*x^10+z^6*x^8+1*x^8+z^8*x^6+z^8*x^4+1*x^4+z^8*x^2+z^6*x^2+z*x^2+z^10+z^6+z+1)", vars, ctx);
        time = check_omega(1, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#41: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "(z^6*x^10*y^13*s^9*t^12+z^6*x^11*y^12*s^9*t^12+z^7*x^10*y^12*s^9*t^12+z^6*x^12*y^11*s^9*t^12+z^8*x^10*y^11*s^9*t^12+z^6*x^13*y^10*s^9*t^12+z^7*x^12*y^10*s^9*t^12+z^8*x^11*y^10*s^9*t^12+z^9*x^10*y^10*s^9*t^12+z^7*x^2*y^11*s*t^9+z^7*x^3*y^10*s*t^9+z^8*x^2*y^10*s*t^9+z^7*x^4*y^9*s*t^9+z^9*x^2*y^9*s*t^9+z^7*x^5*y^8*s*t^9+z^8*x^4*y^8*s*t^9+z^9*x^3*y^8*s*t^9+z^10*x^2*y^8*s*t^9+z^11*x^10*y^9*s^11*t^7+z^11*x^11*y^8*s^11*t^7+z^12*x^10*y^8*s^11*t^7+z^11*x^12*y^7*s^11*t^7+z^13*x^10*y^7*s^11*t^7+z^11*x^13*y^6*s^11*t^7+z^12*x^12*y^6*s^11*t^7+z^13*x^11*y^6*s^11*t^7+z^14*x^10*y^6*s^11*t^7+z^7*x^2*y^6*s*t^7+z^7*x^3*y^5*s*t^7+z^8*x^2*y^5*s*t^7+z^7*x^4*y^4*s*t^7+z^9*x^2*y^4*s*t^7+z^7*x^5*y^3*s*t^7+z^8*x^4*y^3*s*t^7+z^9*x^3*y^3*s*t^7+z^10*x^2*y^3*s*t^7+z^2*x^13*y^11*t^3+z^2*x^14*y^10*t^3+z^3*x^13*y^10*t^3+z^2*x^15*y^9*t^3+z^4*x^13*y^9*t^3+z^2*x^16*y^8*t^3+z^3*x^15*y^8*t^3+z^4*x^14*y^8*t^3+z^5*x^13*y^8*t^3)*(z^4*y^2*s*u^6+x^4*y^5*u^4+z^5*y^4*s^5*u^3+z^6*x^5*y^6*s^4*u^2+z^6*x^5*y*s^6*u+z^6*x^3*y^6*s^4*u+z^4*x^2*y^2*u)*(z*x^3*y^3+z^3*x^2*y^2+z^2*x^2*y^2+z^3*y^2+z*y+z^3*x)", vars, ctx);
        time = check_omega(19, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        flint_printf("#42: "); fflush(stdout);
        nmod_mpoly_set_str_pretty(a, "x^4120 + x^4118*y^2 + x^3708*y^400 + x^3706*y^402 + x^2781*y^1300 + x^2779*y^1302 + x^1339*y^2700 + x^927*y^3100 + y^4000 + x^7172*y^4167 + x^8349*y^4432 + x^8347*y^4434 + x^6760*y^4567 + x^5833*y^5467 + x^5568*y^7132 + x^11401*y^8599", vars, ctx);
        time = check_omega(2, a, vars, ctx);
        flint_printf("%wd\n", time);
        total_time += time;

        nmod_mpoly_clear(a, ctx);
        nmod_mpoly_ctx_clear(ctx);
    }
    flint_printf("total_time: %wd\n", total_time);

    for (k = 0; k <= 4; k++)
    {
        mp_limb_t ps[] = {2, 3, 11, 257, 43051};

        flint_printf("\n------ 4 variables, characteristic %wu ------\n", ps[k]);
        total_time = 0;
        {
            nmod_mpoly_ctx_t ctx;
            nmod_mpoly_t a, b, c;
            const char * vars[] = {"x","y","z","t"};

            nmod_mpoly_ctx_init(ctx, 4, ORD_LEX, ps[k]);
            nmod_mpoly_init(a, ctx);
            nmod_mpoly_init(b, ctx);
            nmod_mpoly_init(c, ctx);

            for (i = 1; i <= 20; i++)
            {
                nmod_mpoly_set_str_pretty(a, "1+x+y+z+t", vars, ctx);
                nmod_mpoly_pow_ui(c, a, i, ctx);
                nmod_mpoly_set_str_pretty(a, "x", vars, ctx);
                nmod_mpoly_add(a, a, c, ctx);
                nmod_mpoly_set_str_pretty(b, "y", vars, ctx);
                nmod_mpoly_add(b, b, c, ctx);
                nmod_mpoly_mul(c, a, b, ctx);
                flint_printf("#%wd: ", i); fflush(stdout);
                time = check_omega(2, c, vars, ctx);
                flint_printf("%wd\n", time);
                total_time += time;
            }

            nmod_mpoly_clear(a, ctx);
            nmod_mpoly_clear(b, ctx);
            nmod_mpoly_clear(c, ctx);
            nmod_mpoly_ctx_clear(ctx);
        }
        flint_printf("total_time: %wd\n", total_time);
    }

    flint_cleanup_master();
    return 0;
}
