/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file profiles.c
 * \brief Profiles related functions
 */

#include <ffgtk.h>

/** global profile list */
static GList *psProfileList = NULL;
/** active profile */
static struct sProfile *psActiveProfile = NULL;
/** profile directory */
const gchar *pnProfilesDir = NULL;
/** plugin settings directory */
const gchar *pnPluginSettingsDir = NULL;

static GtkTreeModel *psListStore = NULL;
static GtkListStore *psProfileStore = NULL;
static GtkWidget *psProfileTreeView = NULL;

/**
 * \brief Get profile directory
 * \return profile directory
 */
const gchar *getProfilesDir( void ) {
	if ( pnProfilesDir == NULL ) {
		pnProfilesDir = g_build_filename( getHomeDir(), FFGTK_USER_DIR, "profiles", NULL );
	}

	return pnProfilesDir;
}

/**
 * \brief Get plugin settings directory
 * \return plugin settings directory
 */
const gchar *getPluginSettingsDir( void ) {
	if ( pnPluginSettingsDir == NULL ) {
		pnPluginSettingsDir = g_build_filename( getHomeDir(), FFGTK_USER_DIR, "plugins", NULL );
	}

	return pnPluginSettingsDir;
}

/**
 * \brief Create new profile by name and file
 * \param pnName profile name
 * \param pnFile new profile file, if exists
 * \return new profile structure
 */
struct sProfile *createProfile( const gchar *pnName, const gchar *pnFile ) {
	struct sProfile *psProfile;

	psProfile = g_malloc0( sizeof( struct sProfile ) );

	if ( psProfile != NULL ) {
		psProfile -> pnName = g_strdup( pnName );
		if ( pnFile == NULL ) {
			gchar *pnTmp = g_strdup_printf( "%s/%s", getProfilesDir(), pnName );
			g_mkdir( pnTmp, 0755 );
			pnFile = g_build_filename( getProfilesDir(), pnName, "profile.xml", NULL );
			g_free( pnTmp );
		}
		psProfile -> pnFile = g_strdup( pnFile );
		LoadPreferences( psProfile );
		Debug( KERN_DEBUG, "Adding '%s'\n", pnName );
		psProfileList = g_list_prepend( psProfileList, psProfile );
	} else {
		Debug( KERN_WARNING, "Could not add '%s'\n", pnName );
	}

	return psProfile;
}

/**
 * \brief Add profile via xmlnode
 * \param psNode node containing profile information
 */
void addProfile( xmlnode *psNode ) {
	gchar *pnName;
	gchar *pnCompleteFile;

	pnName = xmlnode_get_data( psNode );
	if ( pnName != NULL ) {
		pnCompleteFile = g_build_filename( getProfilesDir(), pnName, "profile.xml", NULL );
		if ( pnCompleteFile != NULL ) {
			createProfile( pnName, pnCompleteFile );
			g_free( pnCompleteFile );
		}
		g_free( pnName );
	}
}

/**
 * \brief Get profile list
 * \return global profile list
 */
GList *getProfiles( void ) {
	return psProfileList;
}

/**
 * \brief Find profile structure by name
 * \param pnName profile name
 * \return profile structure or NULL
 */
struct sProfile *findProfile( const gchar *pnName ) {
	GList *psList = getProfiles();
	struct sProfile *psProfile = NULL;

	while ( psList != NULL ) {
		psProfile = psList -> data;
		if ( psProfile != NULL ) {
			if ( !strcmp( psProfile -> pnName, pnName ) ) {
				return psProfile;
			}
		}

		psList = psList -> next;
	}

	return NULL;
}

/**
 * \brief Update call list wrapper function
 * \param pUserData user data
 * \return error code
 */
gpointer updateCallList( gpointer pUserData ) {
	gint nError = -1;

	parseCsvFile( getCallerList( getActiveProfile() ), getActiveProfile() -> pnName );

	while ( 1 ) {
		nError = routerGetCallList( getActiveProfile() );
		if ( nError == 0 ) {
			break;
		} else if ( nError < 0 ) {
			RouterLoginError( TRUE, -nError );
		}
		g_usleep( 5 * G_USEC_PER_SEC );
	}

	return NULL;
}

/**
 * \brief Set active profile
 * \param psProfile profile structure
 */
void setActiveProfile( struct sProfile *psProfile ) {
	struct sFax *psFax = NULL;
	int nOldController = -1;
	int nOldFaxDebugLevel = -1;
	int nRet;

	if ( getActiveProfile() != NULL ) {
		nOldController = faxGetController( getActiveProfile() );
		nOldFaxDebugLevel = debugGetFaxLevel( getActiveProfile() );
	}

	psActiveProfile = psProfile;

	if ( psProfile == NULL ) {
		return;
	}

	Debug( KERN_DEBUG, "Setting '%s' as active plugin\n", psProfile -> pnName );
	routerInit( psProfile );

	Debug( KERN_DEBUG, "setDefaultPlugin: PASSWORD\n" );
	setDefaultPlugin( psProfile, PLUGIN_TYPE_PASSWORD, prefsGetString( psProfile, "/ffgtk/plugins/password" ) )	;
	Debug( KERN_DEBUG, "setDefaultPlugin: LOOKUP\n" );
	setDefaultPlugin( psProfile, PLUGIN_TYPE_LOOKUP, prefsGetString( psProfile, "/ffgtk/plugins/lookup" ) );
	Debug( KERN_DEBUG, "setDefaultPlugin: BOOK\n" );
	setDefaultPlugin( psProfile, PLUGIN_TYPE_BOOK, prefsGetString( psProfile, "/ffgtk/plugins/book" ) );
	Debug( KERN_DEBUG, "setDefaultPlugin: FAX\n" );
	setDefaultPlugin( psProfile, PLUGIN_TYPE_FAX, prefsGetString( psProfile, "/ffgtk/plugins/fax" ) );
	Debug( KERN_DEBUG, "setDefaultPlugin: AUDIO\n" );
	setDefaultPlugin( psProfile, PLUGIN_TYPE_AUDIO, prefsGetString( psProfile, "/ffgtk/plugins/audio" ) );

	if ( nOldController != faxGetController( psProfile ) || nOldFaxDebugLevel != debugGetFaxLevel( psProfile ) ) {
		psFax = getDefaultFaxPlugin( psProfile );
		if ( psFax != NULL ) {
			psFax -> Close();
			nRet = psFax -> Init( faxGetController( psProfile ), debugGetFaxLevel( psProfile ) );
			if ( nRet != 0 ) {
				/* Error: Could not init fax */
				Debug( KERN_DEBUG, "setDefaultPlugin\n" );
				setDefaultPlugin( psProfile, PLUGIN_TYPE_FAX, NULL );
			}
		}
	}

	/* Free caller list */
	freeCallerList();

	if ( callMonitorGetLoadOnStartup( psProfile ) ) {
		CREATE_THREAD( "callist", updateCallList, NULL );
		//g_idle_add_full( G_PRIORITY_HIGH, ( GSourceFunc ) updateCallList, NULL, NULL );
	}
}

/**
 * \brief Get active profile structure
 * \return active profile
 */
struct sProfile *getActiveProfile( void ) {
	return psActiveProfile;
}

/**
 * \brief Convert profile to xmlnode
 * \param psProfile structure
 * \return new xmlnode
 */
static xmlnode *profileToXmlnode( struct sProfile *psProfile ) {
	xmlnode *node;

	node = xmlnode_new( "profile" );
	xmlnode_insert_data( node, psProfile -> pnName, -1 );

	return node;
}

/**
 * \brief Convert profiles to xmlnode
 * \return xmlnode structure
 */
static xmlnode *profilesToXmlnode( void ) {
	xmlnode *node, *child;
	GList *cur;

	node = xmlnode_new( "profile" );
	xmlnode_set_attrib( node, "version", "1.0" );

	for ( cur = getProfiles(); cur != NULL; cur = cur -> next ) {
		child = profileToXmlnode( cur -> data );
		xmlnode_insert_child( node, child );
	}

	if ( psActiveProfile != NULL ) {
		child = xmlnode_new_child( node, "active" );
		xmlnode_insert_data( child, psActiveProfile -> pnName, -1 );
	}

	return node;
}

/**
 * \brief Save all profiles to file
 */
void saveProfiles( void ) {
	xmlnode *node;
	char *pnData;

	node = profilesToXmlnode();

	pnData = xmlnode_to_formatted_str( node, NULL );
	if ( pnData != NULL ) {
		gchar *pnFile = g_build_filename( getUserDir(), "profiles.xml", NULL );

		saveData( pnFile, pnData, -1 );
		g_free( pnFile );
		g_free(pnData);
	}
	xmlnode_free(node);
}

/**
 * \brief Remove profile from list
 * \param pnName profile name
 */
void removeProfile( const gchar *pnName ) {
	struct sProfile *psProfile = findProfile( pnName );

	if ( psProfile != NULL ) {
		unlink( psProfile -> pnFile );
		psProfileList = g_list_remove( psProfileList, psProfile );

		routerRemovePassword( psProfile );

		if ( psProfile == getActiveProfile() ) {
			if ( psProfileList != NULL && g_list_length( psProfileList ) > 0 ) {
				setActiveProfile( psProfileList -> data );
			} else {
				setActiveProfile( NULL );
			}
		}
	}
}

/**
 * \brief Load profiles from file
 */
void ProfilesLoad( void ) {
	xmlnode *psNode = NULL, *psChild;
	struct sProfile *psProfile = NULL;
	DIR *psDir = NULL;

	psDir = opendir( getUserDir() );
	if ( psDir == NULL ) {
		g_mkdir( getUserDir(), 0755 );
	} else {
		closedir( psDir );
	}

	psDir = opendir( getProfilesDir() );
	if ( psDir == NULL ) {
		g_mkdir( getProfilesDir(), 0755 );
	} else {
		closedir( psDir );
	}

	psNode = readXmlFromFile( "profiles.xml", _( "profiles" ) );
	if ( psNode == NULL ) {
		Debug( KERN_DEBUG, "Could not read profiles.xml\n" );
		goto end;
	}

	for ( psChild = xmlnode_get_child( psNode, "profile" ); psChild != NULL; psChild = xmlnode_get_next_twin( psChild ) ) {
		addProfile( psChild );
	}

	psChild = xmlnode_get_child( psNode, "active" );
	if ( psChild == NULL ) {
		Debug( KERN_DEBUG, "Could not find active profile\n" );
	} else {
		gchar *pnTmp = xmlnode_get_data( psChild );
		Debug( KERN_DEBUG, "Active profile '%s'\n", pnTmp );
		psProfile = findProfile( pnTmp );
		setActiveProfile( psProfile );
		g_free( pnTmp );
	}

	psProfileList = g_list_reverse( psProfileList );

	xmlnode_free( psNode );

end:
	if ( psProfileList == NULL || g_list_length( psProfileList ) == 0 ) {
		assistant();
	}

#ifdef HAVE_APPINDICATOR
	Debug( KERN_DEBUG, "Recreate menu\n" );
	recreateAppMenu();
#endif
}

/**
 * \brief Update profile treeview and fields
 */
static void updateProfileList( void ) {
	GList *psList = getProfiles();
	struct sProfile *psProfile;
	GtkTreeIter sIter;

	if ( psProfileStore == NULL ) {
		return;
	}

	gtk_list_store_clear( psProfileStore );

	while ( psList != NULL ) {
		psProfile = psList -> data;
		if ( psProfile != NULL ) {
			gtk_list_store_append( psProfileStore, &sIter );
			gtk_list_store_set( psProfileStore, &sIter, 0, g_locale_to_utf8( psProfile -> pnName, -1, 0, 0, 0 ), -1 );
		}

		psList = psList -> next;
	}
}

/**
 * \brief Delete profile
 * \param psButton Delete profile button
 * \param pUserData tree view
 */
static void delProfile( GtkButton *psButton, gpointer pUserData ) {
	GtkTreeIter sSelectedIter;
	GtkTreeModel *psModel;
	GtkTreeSelection *psSelection = gtk_tree_view_get_selection( pUserData );

	if ( gtk_tree_selection_get_selected( psSelection, &psModel, &sSelectedIter ) ) {
		GValue sName = { 0 };

		gtk_tree_model_get_value( psModel, &sSelectedIter, 0, &sName );

		GtkWidget *psDialog = gtk_message_dialog_new( NULL, GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_QUESTION,
			GTK_BUTTONS_OK_CANCEL, _( "Do you want to delete the profile '%s'?" ), g_value_get_string( &sName ) );
		gtk_window_set_title( GTK_WINDOW( psDialog ), _( "Delete profile" ) );
		gint nResult = gtk_dialog_run( GTK_DIALOG( psDialog ) );
		gtk_widget_destroy( psDialog );

		if ( nResult != GTK_RESPONSE_OK ) {
			return;
		}

		removeProfile( g_value_get_string( &sName ) );

		g_value_unset( &sName );

		saveProfiles();
	}
}

/**
 * \brief Profiles manage response callback
 * \param psDialog dialog window
 * \param nResult response id
 * \param pData file name
 */
static void profilesManageResponse( GtkDialog *psDialog, gint nResult, gpointer pData ) {
	switch ( nResult ) {
		case 0:
			/* Add */
			assistant();
			updateProfileList();
			break;
		case 1:
			/* Delete */
			delProfile( NULL, psProfileTreeView );
			updateProfileList();
			break;
		default:
			gtk_widget_destroy( GTK_WIDGET( psDialog ) );
			break;
	}
}

/**
 * \brief Profiles manage dialog
 */
void ProfilesManage( void ) {
	GtkBuilder *psBuilder = NULL;
	GError *psError = NULL;
	GtkWidget *psDialog = NULL;
	gchar *pnUiFile;

	psBuilder = gtk_builder_new();
	pnUiFile = getUiFile( "profiles.ui" );
	if ( gtk_builder_add_from_file( psBuilder, pnUiFile, &psError ) == 0 ) {
	    Debug( KERN_WARNING, "Error: %s\n", psError -> message );
	    g_error_free( psError );
		g_free( pnUiFile );
	    return;
	}
	g_free( pnUiFile );

	psDialog = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psProfilesDialog" ) );
	psProfileTreeView = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psProfileTreeView" ) );
	GtkTreeViewColumn *psCol = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( psCol, "Profile" );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psProfileTreeView ), psCol );

	GtkCellRenderer *psRenderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start( psCol, psRenderer, TRUE );
	gtk_tree_view_column_add_attribute( psCol, psRenderer, "text", 0 );	

	psProfileStore = gtk_list_store_new( 1, G_TYPE_STRING );
	psListStore = GTK_TREE_MODEL( psProfileStore );
	gtk_tree_view_set_model( GTK_TREE_VIEW( psProfileTreeView ), psListStore );

	gtk_builder_connect_signals( psBuilder, NULL );

	g_object_unref( G_OBJECT( psBuilder ) );

	g_signal_connect( G_OBJECT( psDialog ), "close", G_CALLBACK( gtk_widget_destroy ), psDialog );
	g_signal_connect( G_OBJECT( psDialog ), "response", G_CALLBACK( profilesManageResponse ), NULL );

	updateProfileList();

	gtk_widget_show_all( psDialog );
}
