/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file print.c
 * \brief Print caller list function
 */

#include <ffgtk.h>
#include <cairo-pdf.h>

#define FONT	"cairo:monospace 10"

/** Structure holds information about the page and start positions of columns */
typedef struct {
	gdouble fFontWidth;
	gdouble fLineHeight;
	gdouble fCharWidth;
	PangoLayout *psLayout;

	GtkTreeView *psView;
	gint nLinesPerPage;
	gint nNumLines;
	gint nNumPages;

	gint nLogoPos;
	gint nDateTimePos;
	gint nNamePos;
	gint nNumberPos;
	gint nLocalNamePos;
	gint nLocalNumberPos;
	gint nDurationPos;
} PrintData;

static gboolean check_monospace( PangoContext *psPc, PangoFontDescription *psDesc ) {
	PangoFontFamily **ppsFamilies;
	gint nNumFamilies, nI;
	const gchar *pnFont;
	gboolean bRet = TRUE;

	pnFont = pango_font_description_get_family( psDesc );
	pango_context_list_families( psPc, &ppsFamilies, &nNumFamilies );

	for ( nI = 0; nI < nNumFamilies; nI++ ) {
		const gchar *pnCheck = pango_font_family_get_name( ppsFamilies[ nI ] );

		if ( pango_font_family_is_monospace( ppsFamilies[ nI ] ) ) {
			Debug( KERN_DEBUG, "%s is monospace\n", pnCheck );
		}

		if ( pnFont && pnCheck && strcasecmp( pnFont, pnCheck ) == 0 ) {
			if ( !pango_font_family_is_monospace( ppsFamilies[ nI ] ) ) {
				bRet = FALSE;
			}
		}
	}

	g_free( ppsFamilies );

	return bRet;
}

static gint get_font_width( GtkPrintContext *psContext, PangoFontDescription *psDesc ) {
	PangoContext *psPc;
	PangoFontMetrics *psMetrics;
	gint nWidth;

	psPc = gtk_print_context_create_pango_context( psContext );
	if ( !check_monospace( psPc, psDesc ) ) {
		Debug( KERN_DEBUG, "The request font is not a monospace font!\n" );
	}

	psMetrics = pango_context_get_metrics( psPc, psDesc, pango_context_get_language( psPc ) );
	nWidth = pango_font_metrics_get_approximate_digit_width( psMetrics ) / PANGO_SCALE;
	if ( nWidth == 0 ) {
		nWidth = pango_font_metrics_get_approximate_char_width( psMetrics ) / PANGO_SCALE;
		if ( nWidth == 0 ) {
			nWidth = pango_font_description_get_size( psDesc ) / PANGO_SCALE;
		}
	}

	pango_font_metrics_unref( psMetrics );
	g_object_unref( psPc );

	return nWidth;
}

static int get_page_count( GtkPrintContext *psContext, PrintData *psData ) {
	gdouble fWidth, fHeight;
	gint nLayoutH;
	gint nLayoutV;

	if ( psData == NULL ) {
		return -1;
	}

	fWidth = gtk_print_context_get_width( psContext );
	fHeight = gtk_print_context_get_height( psContext );

	pango_layout_set_width( psData -> psLayout, fWidth * PANGO_SCALE );

	pango_layout_set_text( psData -> psLayout, "Z", -1 );
	pango_layout_get_size( psData -> psLayout, &nLayoutV, &nLayoutH );
	if ( nLayoutH <= 0 ) {
		Debug( KERN_DEBUG, "Invalid layout h (%d). Falling back to default height (%d)\n", nLayoutH, 100 * PANGO_SCALE );
		nLayoutH = 100 * PANGO_SCALE;
	}
	if ( nLayoutV <= 0 ) {
		Debug( KERN_DEBUG, "Invalid layout v (%d). Falling back to default width (%d)\n", nLayoutV, 100 * PANGO_SCALE );
		nLayoutV = 100 * PANGO_SCALE;
	}

	psData -> fLineHeight = ( gdouble ) nLayoutH / PANGO_SCALE + 2;
	psData -> fCharWidth = ( gdouble ) nLayoutV / PANGO_SCALE;
	psData -> nLinesPerPage = ceil(( fHeight - psData -> fLineHeight ) / psData -> fLineHeight );
	Debug( KERN_DEBUG, "nLinesPerPage: %d\n", psData -> nLinesPerPage );

	/* 3 for header, 1 for bar */
	psData -> nLinesPerPage -= 5;

	return ( psData -> nNumLines / psData -> nLinesPerPage ) + 1;
}

/**
 * \brief Begin printing, calculate values and store them in PrintData
 * \param psOperation print operation
 * \param psContext print context
 * \param pUserData user data pointer (PrintData)
 */
static void beginPrint( GtkPrintOperation *psOperation, GtkPrintContext *psContext, gpointer pUserData ) {
	PrintData *psData = ( PrintData * ) pUserData;
	GtkTreeModel *psModel;
	GtkTreeIter sIter;
	gboolean bValid;
	PangoFontDescription *psDesc = pango_font_description_from_string( FONT );

	psData -> nNumLines = 0;

	psModel = gtk_tree_view_get_model( psData -> psView );
	bValid = gtk_tree_model_get_iter_first( psModel, &sIter );
	while ( bValid ) {
		bValid = gtk_tree_model_iter_next( psModel, &sIter );
		psData -> nNumLines++;
	}

	psData -> psLayout = gtk_print_context_create_pango_layout( psContext );
	pango_layout_set_wrap( psData -> psLayout, PANGO_WRAP_WORD_CHAR );
	pango_layout_set_spacing( psData -> psLayout, 0 );
	pango_layout_set_attributes( psData -> psLayout, NULL );
	pango_layout_set_font_description( psData -> psLayout, psDesc );

	psData -> nNumPages = get_page_count( psContext, psData );
	psData -> fFontWidth = get_font_width( psContext, psDesc ) + 1;
	if ( psData -> fFontWidth == 0 ) {
		psData -> fFontWidth = psData -> fCharWidth;
	}

	Debug( KERN_DEBUG, "nNumPages: %d\n", psData -> nNumPages);

	psData -> nLogoPos = 4;
	psData -> nDateTimePos = psData -> nLogoPos + psData -> fFontWidth * 4;
	psData -> nNamePos = psData -> nDateTimePos + psData -> fFontWidth * 12;
	psData -> nNumberPos = psData -> nNamePos + psData -> fFontWidth * 19;
	psData -> nLocalNamePos = psData -> nNumberPos + psData -> fFontWidth * 14;
	psData -> nLocalNumberPos = psData -> nLocalNamePos + psData -> fFontWidth * 11;
	psData -> nDurationPos = psData -> nLocalNumberPos + psData -> fFontWidth * 14;

	if ( psData -> nNumPages >= 0 ) {
		gtk_print_operation_set_n_pages( psOperation, psData -> nNumPages );
	}

	pango_font_description_free( psDesc );
}

/**
 * \brief Short text name
 * \param psCairo cairo pointer
 * \param psLayout pango layout
 * \param pnText text string
 * \param nWidth maximum column width
 */
void showText( cairo_t *psCairo, PangoLayout *psLayout, gchar *pnText, gint nWidth ) {
	gint nTextWidth, nTextHeight;

	pango_layout_set_text( psLayout, pnText, -1 );
	pango_layout_get_pixel_size( psLayout, &nTextWidth, &nTextHeight );

	if ( nTextWidth > nWidth ) {
		pango_layout_set_width( psLayout, nWidth * PANGO_SCALE );
		pango_layout_set_ellipsize( psLayout, PANGO_ELLIPSIZE_END );
	}
	pango_cairo_show_layout( psCairo, psLayout );

	if ( nTextWidth > nWidth ) {
		pango_layout_set_width( psLayout, -1 );
	}
}

static gchar *get_date_time( const gchar *pnFormat ) {
	const struct tm *psTime;
	static gchar anDate[ 1024 ];
	gchar *pnLocaleFormat;
	gsize nLen;

	if ( !g_utf8_validate( pnFormat, -1, NULL ) ) {
		pnLocaleFormat = g_locale_from_utf8( pnFormat, -1, NULL, NULL, NULL );
		if ( pnLocaleFormat == NULL ) {
			return NULL;
		}
	} else {
		pnLocaleFormat = g_strdup( pnFormat );
	}

	time_t sTime = time( NULL );
	psTime = localtime( &sTime );

	nLen = strftime( anDate, 1024, pnLocaleFormat, psTime );
	g_free( pnLocaleFormat );

	if ( nLen == 0 ) {
		return NULL;
	}

	if ( !g_utf8_validate( anDate, nLen, NULL ) ) {
		return g_locale_to_utf8( anDate, nLen, NULL, NULL, NULL );
	}

	return g_strdup( anDate );
}

/**
 * \brief Draw one single page
 * \param psOperation print operation
 * \param psContext print context
 * \param nPageNr page number to print
 * \param pUserData user data pointer (PrintData)
 */
static void drawPage( GtkPrintOperation *psOperation, GtkPrintContext *psContext, gint nPageNr, gpointer pUserData ) {
	PrintData *psData = ( PrintData * ) pUserData;
	cairo_t *psCairo;
	gint nLine, nI = 0;
	GtkTreeModel *psModel;
	GtkTreeIter sIter;
	GdkPixbuf *psPix, *psDstPix;
	gboolean bValid;
	gchar *pnDateTime;
	gchar *pnName;
	gchar *pnNumber;
	gchar *pnLocalName;
	gchar *pnLocalNumber;
	gchar *pnDuration;
	gint nTmp;
	gint nLineHeight = 0;

	psCairo = gtk_print_context_get_cairo_context( psContext );
	gdouble fWidth = gtk_print_context_get_width( psContext );

	cairo_set_source_rgb( psCairo, 0, 0, 0 );
	cairo_move_to( psCairo, 0, 0 );
	pango_layout_set_width( psData -> psLayout, fWidth * PANGO_SCALE );
	pango_layout_set_alignment( psData -> psLayout, PANGO_ALIGN_LEFT );
	pango_layout_set_ellipsize( psData -> psLayout, FALSE );
	pango_layout_set_justify( psData -> psLayout, FALSE );

	pango_layout_set_width( psData -> psLayout, ( fWidth - 8 ) * PANGO_SCALE );

	/* Title */
	gchar *pnData = g_strdup_printf( _( "<b>ffgtk - Anruferliste</b>" ) );
	pango_layout_set_markup( psData -> psLayout, pnData, -1 );
	pango_layout_set_alignment( psData -> psLayout, PANGO_ALIGN_CENTER );
	cairo_move_to( psCairo, 3, psData -> fLineHeight * 0.5 );
	pango_cairo_show_layout( psCairo, psData -> psLayout );
	g_free( pnData );

	/* Page */
	pnData = g_strdup_printf( _( "<small>Page %d of %d</small>" ), nPageNr + 1, psData -> nNumPages );
	pango_layout_set_markup( psData -> psLayout, pnData, -1 );
	pango_layout_set_alignment( psData -> psLayout, PANGO_ALIGN_LEFT );
	cairo_move_to( psCairo, 4, psData -> fLineHeight * 1.5 );
	pango_cairo_show_layout( psCairo, psData -> psLayout );
	g_free( pnData );

	/* Date */
	gchar *pnDate = get_date_time("%d.%m.%Y %H:%M:%S %Z");
	pnData = g_strdup_printf( "<small>%s</small>", pnDate );
	pango_layout_set_markup( psData -> psLayout, pnData, -1 );
	pango_layout_set_alignment( psData -> psLayout, PANGO_ALIGN_RIGHT );
	cairo_move_to( psCairo, 2, psData -> fLineHeight * 1.5 );
	pango_cairo_show_layout( psCairo, psData -> psLayout );
	g_free( pnDate );
	g_free( pnData );

	/* Reset */
	cairo_move_to( psCairo, 0, 0 );
	pango_layout_set_width( psData -> psLayout, fWidth * PANGO_SCALE );
	pango_layout_set_ellipsize( psData -> psLayout, FALSE );
	pango_layout_set_justify( psData -> psLayout, FALSE );
	pango_layout_set_attributes( psData -> psLayout, NULL );
	pango_layout_set_alignment( psData -> psLayout, PANGO_ALIGN_LEFT );

	nLine = nPageNr * psData -> nLinesPerPage;

	psModel = gtk_tree_view_get_model( psData -> psView );
	bValid = gtk_tree_model_get_iter_first( psModel, &sIter );
	nTmp = 0;
	if ( nLine != 0 ) {
		while ( bValid ) {
			bValid = gtk_tree_model_iter_next( psModel, &sIter );
			if ( ++nTmp == nLine ) {
				break;
			}
		}
	}
	cairo_set_line_width( psCairo, 0 );

	/* Draw header */
	cairo_rectangle( psCairo, 2, psData -> fLineHeight * 3, fWidth - 2, psData -> fLineHeight );
	cairo_set_source_rgb( psCairo, 0.9, 0.9, 0.9 );
	cairo_fill( psCairo );
	cairo_stroke( psCairo );
	cairo_set_source_rgb( psCairo, 0.0, 0.0, 0.0 );

	cairo_move_to( psCairo, psData -> nLogoPos, psData -> fLineHeight * 3 + 1 );
	pango_layout_set_text( psData -> psLayout, _( "Type" ), -1 );
	pango_cairo_show_layout( psCairo, psData -> psLayout );

	cairo_move_to( psCairo, psData -> nDateTimePos, psData -> fLineHeight * 3 + 1 );
	pango_layout_set_text( psData -> psLayout, _( "Date/Time" ), -1 );
	pango_cairo_show_layout( psCairo, psData -> psLayout );

	cairo_move_to( psCairo, psData -> nNamePos, psData -> fLineHeight * 3 + 1 );
	pango_layout_set_text( psData -> psLayout, _( "Name" ), -1 );
	pango_cairo_show_layout( psCairo, psData -> psLayout );

	cairo_move_to( psCairo, psData -> nNumberPos, psData -> fLineHeight * 3 + 1 );
	pango_layout_set_text( psData -> psLayout, _( "Number" ), -1 );
	pango_cairo_show_layout( psCairo, psData -> psLayout );

	cairo_move_to( psCairo, psData -> nLocalNamePos, psData -> fLineHeight * 3 + 1 );
	pango_layout_set_text( psData -> psLayout, _( "Local Name" ), -1 );
	pango_cairo_show_layout( psCairo, psData -> psLayout );

	cairo_move_to( psCairo, psData -> nLocalNumberPos, psData -> fLineHeight * 3 + 1 );
	pango_layout_set_text( psData -> psLayout, _( "Local Number" ), -1 );
	pango_cairo_show_layout( psCairo, psData -> psLayout );

	cairo_move_to( psCairo, psData -> nDurationPos, psData -> fLineHeight * 3 + 1 );
	pango_layout_set_text( psData -> psLayout, _( "Duration" ), -1 );
	pango_cairo_show_layout( psCairo, psData -> psLayout );

	/* print caller rows */
	for ( nI = 1; bValid && nI <= psData -> nLinesPerPage && nLine < psData -> nNumLines; nI++ ) {
		cairo_rectangle( psCairo, 2, ( 3 + nI ) * psData -> fLineHeight, fWidth - 4, psData -> fLineHeight );
		if ( !( nI & 1 ) ) {
			cairo_set_source_rgb( psCairo, 0.9, 0.9, 0.9 );
			cairo_fill( psCairo );
			cairo_stroke( psCairo );
			cairo_set_source_rgb( psCairo, 0.0, 0.0, 0.0 );
		}
		cairo_stroke( psCairo );

		gtk_tree_model_get( psModel, &sIter, 0, &psPix, 1, &pnDateTime, 2, &pnName, 5, &pnNumber, 6, &pnLocalName, 7, &pnLocalNumber, 8, &pnDuration, -1 );
		psDstPix = gdk_pixbuf_scale_simple( psPix, 10, 10, GDK_INTERP_BILINEAR );

		cairo_save( psCairo );
		gdk_cairo_set_source_pixbuf( psCairo, psDstPix, psData -> nLogoPos, ( 3 + nI ) * psData -> fLineHeight + 2 );
		cairo_paint( psCairo );
		cairo_restore( psCairo );
		g_object_unref( psDstPix );

		cairo_move_to( psCairo, psData -> nDateTimePos, ( 3 + nI ) * psData -> fLineHeight + 1 );
		pango_layout_set_text( psData -> psLayout, pnDateTime, -1 );
		pango_cairo_show_layout( psCairo, psData -> psLayout );

		cairo_move_to( psCairo, psData -> nNamePos, ( 3 + nI ) * psData -> fLineHeight + 1 );
		showText( psCairo, psData -> psLayout, pnName, psData -> nNumberPos - psData -> nNamePos );

		cairo_move_to( psCairo, psData -> nNumberPos, ( 3 + nI ) * psData -> fLineHeight + 1 );
		showText( psCairo, psData -> psLayout, pnNumber, psData -> nLocalNamePos - psData -> nNumberPos );

		cairo_move_to( psCairo, psData -> nLocalNamePos, ( 3 + nI ) * psData -> fLineHeight + 1 );
		if ( pnLocalName != NULL && strlen( pnLocalName ) > 0 ) {
			showText( psCairo, psData -> psLayout, pnLocalName, psData -> nLocalNumberPos - psData -> nLocalNamePos );
		}

		cairo_move_to( psCairo, psData -> nLocalNumberPos, ( 3 + nI ) * psData -> fLineHeight + 1 );
		if ( pnLocalNumber != NULL && strlen( pnLocalNumber ) > 0 ) {
			showText( psCairo, psData -> psLayout, pnLocalNumber, psData -> nDurationPos - psData -> nLocalNumberPos );
		}

		cairo_move_to( psCairo, psData -> nDurationPos, ( 3 + nI ) * psData -> fLineHeight + 1 );
		if ( pnDuration != NULL && strlen( pnDuration ) > 0 ) {
			showText( psCairo, psData -> psLayout, pnDuration, fWidth - psData -> nDurationPos );
		}

		bValid = gtk_tree_model_iter_next( psModel, &sIter );
		g_free( pnDateTime );
		g_free( pnName );
		g_free( pnNumber );
		g_free( pnLocalName );
		g_free( pnLocalNumber );
		g_free( pnDuration );

		cairo_rel_move_to( psCairo, 2, nLineHeight );
		nLine++;
	}
}

/**
 * \brief Called on end of printing, free PrintData
 * \param psOperation print operation
 * \param psContext print context
 * \param pUserData user data pointer (PrintData)
 */
static void endPrint( GtkPrintOperation *psOperation, GtkPrintContext *psContext, gpointer pUserData ) {
	PrintData *psData = ( PrintData * ) pUserData;

	g_free( psData );
}

/**
 * \brief Print treeview callback function
 * \param psWidget button widget
 * \param pUserData user data pointer (TreeView)
 */
void PrintTreeView( GtkWidget *psWidget, gpointer pUserData ) {
	GtkTreeView *psView = GTK_TREE_VIEW( pUserData );
	GtkPrintOperation *psOperation;
	PrintData *psData;
	GError *psError = NULL;

	psOperation = gtk_print_operation_new();
	psData = g_new0( PrintData, 1 );
	psData -> psView = psView;

	gtk_print_operation_set_unit( psOperation, GTK_UNIT_POINTS );
	gtk_print_operation_set_show_progress( psOperation, TRUE );

#if GTK_CHECK_VERSION(2, 18, 0)
	gtk_print_operation_set_embed_page_setup( psOperation, TRUE );
#endif

	g_signal_connect( G_OBJECT( psOperation ), "begin-print", G_CALLBACK( beginPrint ), psData );
	g_signal_connect( G_OBJECT( psOperation ), "draw-page", G_CALLBACK( drawPage ), psData );
	g_signal_connect( G_OBJECT( psOperation ), "end-print", G_CALLBACK( endPrint ), psData );

	gtk_print_operation_run( psOperation, GTK_PRINT_OPERATION_ACTION_PRINT_DIALOG, NULL, &psError );

	g_object_unref( psOperation );

	if ( psError != NULL ) {
		GtkWidget *psDialog;

		psDialog = gtk_message_dialog_new( NULL, GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, "%s", psError -> message );
		g_error_free( psError );

		g_signal_connect( psDialog, "response", G_CALLBACK( gtk_widget_destroy ), NULL );
	
		gtk_widget_show( psDialog );
	}
}

#ifdef HAVE_FAXOPHONE
#include <faxophone/faxophone.h>
#include <faxophone/fax.h>
char *capiInfo2String( int nReason );

/**
 * \brief Create fax report based on give information
 * \param pnFile filename to store report
 * \param pnReportDir save directory
 * \param pnRemote remote number
 * \param pnLocal local number
 * \param pnStatus fax transfer status
 * \param pnError error message
 * \param nPages number of pages
 * \param nBitrate transferred bitrate
 */
void CreateFaxReport( struct sFaxStatus *psStatus, char *pnReportDir ) {
	cairo_t *psCairo;
	cairo_surface_t *psOut;
	int nWidth, nHeight;
	time_t sTime = time( NULL );
	struct tm *psTime = localtime( &sTime );
	char *pnBuffer;
	GdkPixbuf*psPixBuf;
	char *pnFile = psStatus -> anTiffFile;
	struct sPerson *psPerson = NULL;
	char *pnRemote = psStatus -> anTrgNo;
	char *pnLocal = psStatus -> anSrcNo;
	char *pnStatus = psStatus -> ePhase == FAX_OK ? _( "SUCCESS" ) : _( "FAILED" );
	char *pnError = capiInfo2String( psStatus -> nErrorCode );
	int nPages = psStatus -> nPageCurrent;
	int nBitrate = psStatus -> nBitrate;

	if ( pnFile == NULL ) {
		Debug( KERN_WARNING, "File is NULL\n" );
		return;
	}

	if ( pnReportDir == NULL ) {
		Debug( KERN_WARNING, "ReportDir is NULL\n" );
		return;
	}

	psPixBuf = gdk_pixbuf_new_from_file( pnFile, NULL );
	if ( psPixBuf == NULL ) {
		Debug( KERN_WARNING, "PixBuf is null (file '%s')\n", pnFile );
		return;
	}

	nWidth = gdk_pixbuf_get_width( psPixBuf );
	nHeight = gdk_pixbuf_get_height( psPixBuf );
	if ( nWidth != 1728 || nHeight != 2292 ) {
		nWidth = 1728;
		nHeight = 2292;
		GdkPixbuf *psScale = gdk_pixbuf_scale_simple( psPixBuf, nWidth, nHeight, GDK_INTERP_BILINEAR );

		g_object_unref( psPixBuf );
		psPixBuf = psScale;
	}

	pnBuffer = g_strdup_printf( "%s/fax-report_%s_%s-%02d_%02d_%d_%02d_%02d_%02d.pdf",
		pnReportDir, pnLocal, pnRemote,
		psTime -> tm_mday, psTime -> tm_mon + 1, psTime -> tm_year + 1900,
		psTime -> tm_hour, psTime -> tm_min, psTime -> tm_sec );
	psOut = cairo_pdf_surface_create( pnBuffer, nWidth, nHeight + 200 );
	g_free( pnBuffer );

	psCairo = cairo_create( psOut );
	gdk_cairo_set_source_pixbuf( psCairo, psPixBuf, 0, 200 );
	cairo_paint( psCairo );

	cairo_set_source_rgb( psCairo, 0, 0, 0 );
	cairo_select_font_face( psCairo, "cairo:monospace", CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_BOLD );
	cairo_set_font_size( psCairo, 20.0 );

	cairo_move_to( psCairo, 60.0, 60.0 );
	cairo_show_text( psCairo, _( "FAX-Report" ) );

	cairo_select_font_face( psCairo, "cairo:monospace", CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_NORMAL );

	/* Date/Time */
	cairo_move_to( psCairo, 60.0, 95.0 );
	cairo_show_text( psCairo, _( "Date/Time:" ) );
	cairo_move_to( psCairo, 220.0, 95.0 );
	pnBuffer = asctime( psTime );
	pnBuffer[ strlen( pnBuffer ) - 1 ] = 0;
	cairo_show_text( psCairo, pnBuffer );

	/* Status */
	cairo_move_to( psCairo, 500.0, 95.0 );
	cairo_show_text( psCairo, _( "Status:" ) );
	cairo_move_to( psCairo, 680.0, 95.0 );
	cairo_show_text( psCairo, pnStatus );

	/* Remote name */
	cairo_move_to( psCairo, 60.0, 120.0 );
	cairo_show_text( psCairo, _( "Remote Name:" ) );
	psPerson = findPersonByNumber( pnRemote );
	cairo_move_to( psCairo, 220.0, 120.0 );
	if ( strlen( psStatus -> anIdent ) > 0 ) {
		cairo_show_text( psCairo, psStatus -> anIdent );
	} else {
		cairo_show_text( psCairo, psPerson ? psPerson -> pnDisplayName : _( "Unknown" ) );
	}

	/* Remote number */
	cairo_move_to( psCairo, 500.0, 120.0 );
	cairo_show_text( psCairo, _( "Remote Number:" ) );
	cairo_move_to( psCairo, 680.0, 120.0 );
	cairo_show_text( psCairo, pnRemote );

	/* Local name */
	cairo_move_to( psCairo, 60.0, 145.0 );
	cairo_show_text( psCairo, _( "Local Name:" ) );
	psPerson = findPersonByNumber( pnLocal );
	cairo_move_to( psCairo, 220.0, 145.0 );
	cairo_show_text( psCairo, psPerson ? psPerson -> pnDisplayName : _( "Unknown" ) );
	
	/* Local number */
	cairo_move_to( psCairo, 500.0, 145.0 );
	cairo_show_text( psCairo, _( "Local Number:" ) );
	cairo_move_to( psCairo, 680.0, 145.0 );
	cairo_show_text( psCairo, pnLocal );

	/* Pages */
	cairo_move_to( psCairo, 60.0, 170.0 );
	cairo_show_text( psCairo, _( "Pages:" ) );
	cairo_move_to( psCairo, 220.0, 170.0 );
	pnBuffer = g_strdup_printf( "%d", nPages );
	cairo_show_text( psCairo, pnBuffer );
	g_free( pnBuffer );

	/* Bitrate */
	cairo_move_to( psCairo, 500.0, 170.0 );
	cairo_show_text( psCairo, _( "Bitrate:" ) );
	cairo_move_to( psCairo, 680.0, 170.0 );
	pnBuffer = g_strdup_printf( "%d", nBitrate );
	cairo_show_text( psCairo, pnBuffer );
	g_free( pnBuffer );

	/* Error message */
	cairo_move_to( psCairo, 60.0, 195.0 );
	cairo_show_text( psCairo, _( "Message:" ) );
	cairo_move_to( psCairo, 220.0, 195.0 );
	cairo_show_text( psCairo, pnError );

	/* line */
	cairo_set_line_width( psCairo, 0.5 );
	cairo_move_to( psCairo, 0, 200 );
	cairo_rel_line_to( psCairo, nWidth, 0 );
	cairo_stroke( psCairo );

	cairo_show_page( psCairo );

	cairo_destroy( psCairo );

	cairo_surface_flush( psOut );
	cairo_surface_destroy( psOut );

	g_object_unref( psPixBuf );
}
#endif
