/*******************************************************************************

  Intel Data Center Bridging (DCB) Software
  Copyright(c) 2007-2009 Intel Corporation.

  This program is free software; you can redistribute it and/or modify it
  under the terms and conditions of the GNU General Public License,
  version 2, as published by the Free Software Foundation.

  This program is distributed in the hope it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.

  The full GNU General Public License is included in this distribution in
  the file called "COPYING".

  Contact Information:
  e1000-eedc Mailing List <e1000-eedc@lists.sourceforge.net>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497

*******************************************************************************/

#include "dcb_osdep.h"
#include "states.h"
#include "tlv.h"
#include "ports.h"
#include "l2_packet.h"
#include "dcb_types.h"

struct port *porthead = NULL; /* Head pointer */
struct port *portcurrent = NULL; /* Working  pointer loaded from ports or
				  * port->next */

extern u8 gdcbx_subtype;

void agent_receive(void *, const u8 *, const u8 *, size_t);

void set_lldp_port_enable_state(const char *ifname, int enable)
{
	struct port *port = NULL;

	port = porthead;
	while (port != NULL) {
		if (!strncmp(ifname, port->ifname,
			MAX_DEVICE_NAME_LEN))  /* Localization OK */
			break;
		port = port->next;
	}

	if (port == NULL) {
		return;
	}

	port->portEnabled = (u8)enable;
	if (enable) {
		port->adminStatus = enabledRxTx;
	} else {
		port->adminStatus = disabled;
		port->rx.rxInfoAge = FALSE;
	}
	run_tx_sm(port, FALSE);
	run_rx_sm(port, FALSE);
}

int set_port_hw_resetting(const char *ifname, int resetting)
{
	struct port *port = NULL;

	port = porthead;
	while (port != NULL) {
		if (!strncmp(ifname, port->ifname,
			MAX_DEVICE_NAME_LEN)) {  /* Localization OK */
			break;
		}
		port = port->next;
	}

	if (port == NULL) {
		return -1;
	}

	port->hw_resetting = (u8)resetting;

	return port->hw_resetting;
}

int get_port_hw_resetting(const char *ifname)
{
	struct port *port = NULL;

	port = porthead;
	while (port != NULL) {
		if (!strncmp(ifname, port->ifname,
			MAX_DEVICE_NAME_LEN)) {  /* Localization OK */
			break;
		}
		port = port->next;
	}

	if (port)
		return port->hw_resetting;
	else
		return 0;
}


int add_port(const char *ifname)
{
	struct port *newport;
	boolean_t   success;

	newport = porthead;
	while (newport != NULL) {
		if (!strncmp(ifname, newport->ifname,
			MAX_DEVICE_NAME_LEN)) {
			return 0;
		}
		newport = newport->next;
	}

	newport  = (struct port *)malloc(sizeof(struct port));
	if (newport == NULL) {
		printf("new port malloc failed\n");
		goto fail;
	}
	memset(newport,0,sizeof(struct port));
	newport->next = NULL;
	newport->ifname = strdup(ifname);
	if (newport->ifname == NULL) {
		printf("new port name malloc failed\n");
		goto fail;
	}

	/* Initialize relevant port variables */
	newport->tx.state  = TX_LLDP_INITIALIZE;
	newport->rx.state = LLDP_WAIT_PORT_OPERATIONAL;
	newport->hw_resetting = FALSE;
	newport->portEnabled = FALSE;
	newport->adminStatus = enabledRxTx;
	newport->tx.txTTL = 0;
	newport->msap.length1 = 0;
	newport->msap.msap1 = NULL;
	newport->msap.length2 = 0;
	newport->msap.msap2 = NULL;
	newport->lldpdu = FALSE;
	newport->dcbdu = FALSE;

	/* init & enable RX path */
	rxInitializeLLDP(newport);
	newport->l2 = l2_packet_init(newport->ifname, NULL, ETH_P_LLDP,
		rxReceiveFrame, newport, 1);
	if (newport->l2 == NULL) {
		printf("Failed to open register layer 2 access to "
			"ETH_P_LLDP\n");
		goto fail;
	}
	newport->dcbx_st = gdcbx_subtype;

	/* init TX path */
	txInitializeLLDP(newport);

	/* Add TLVs */
	newport->tlvs.chassis = bld_chassis_tlv(newport);
	if (newport->tlvs.chassis == NULL) {
		printf("add_port:  bld_chassis_tlv failed\n");
		goto fail_add;
	}

	newport->tlvs.portid = bld_portid_tlv(newport);
	if (newport->tlvs.portid == NULL) {
		printf("add_port:  bld_portid_tlv failed\n");
		goto fail_add;
	}

	newport->tlvs.ttl = bld_ttl_tlv(newport);
	if (newport->tlvs.ttl == NULL) {
		printf("add_port:  bld_ttl_tlv failed\n");
		goto fail_add;
	}

	newport->tlvs.control = bld_dcbx_ctrl_tlv(newport);
	if (newport->tlvs.control == NULL) {
		printf("add_port:  bld_dcbx_ctrl_tlv failed\n");
		goto fail_add;
	}

	if (newport->dcbx_st == dcbx_subtype2) {
		newport->tlvs.pg2 = bld_dcbx2_pg_tlv(newport, &success);
		if (!success) {
			printf("bld_dcbx2_pg_tlv: failed\n");
			goto fail_add;
		}
	} else {
		newport->tlvs.pg1 = bld_dcbx1_pg_tlv(newport, &success);
		if (!success) {
			printf("bld_dcbx1_pg_tlv: failed\n");
			goto fail_add;
		}
	}

	if (newport->dcbx_st == dcbx_subtype2) {
		newport->tlvs.pfc2 = bld_dcbx2_pfc_tlv(newport, &success);
		if (!success) {
			printf("bld_dcbx2_pfc_tlv: failed\n");
			goto fail_add;
		}
	} else {
		newport->tlvs.pfc1 = bld_dcbx1_pfc_tlv(newport, &success);
		if (!success) {
			printf("bld_dcbx1_pfc_tlv: failed\n");
			goto fail_add;
		}
	}

	if (newport->dcbx_st == dcbx_subtype2) {
		newport->tlvs.app2 = bld_dcbx2_app_tlv(newport,
			APP_FCOE_STYPE, &success);
		if (!success) {
			printf("bld_dcbx2_app_tlv: failed\n");
			goto fail_add;
		}
	} else {
		newport->tlvs.app1 = bld_dcbx1_app_tlv(newport,
				APP_FCOE_STYPE, &success);
		if (!success) {
			printf("bld_dcbx1_app_tlv: failed\n");
			goto fail_add;
		}
	}

	newport->tlvs.llink = bld_dcbx_llink_tlv(newport, LLINK_FCOE_STYPE,
						&success);
	if (!success) {
		printf("bld_dcbx_llink_tlv: failed\n");
		goto fail_add;
	}

	if (newport->dcbx_st == dcbx_subtype2) {
		newport->tlvs.dcbx2 = bld_dcbx2_tlv(newport);
		if (newport->tlvs.dcbx2 == NULL) {
			printf("add_port:  bld_dcbx2_tlv failed\n");
			goto fail_add;
		}
	} else {
		newport->tlvs.dcbx1 = bld_dcbx1_tlv(newport);
		if (newport->tlvs.dcbx1 == NULL) {
			printf("add_port:  bld_dcbx1_tlv failed\n");
			goto fail_add;
		}
	}
	newport->tlvs.last_peer = NULL;
	newport->tlvs.cur_peer = NULL;

	/* enable TX path */
	if (porthead) {
		newport->next = porthead;
	}
	porthead = newport;
	return 0;

fail_add:
	newport->tlvs.last_peer = NULL;
	newport->tlvs.cur_peer = NULL;

	if (porthead) {
		newport->next = porthead;
	}
	porthead = newport;
	return -1;

fail:
	if(newport) {
		if(newport->ifname)
			free(newport->ifname);
		free(newport);
	}
	return -1;
}

int remove_port(const char *ifname)
{
	struct port *port = NULL;    /* Pointer to port to remove */
	struct port *parent = NULL;  /* Pointer to previous on port stack */

	port = porthead;
	while (port != NULL) {
		if (!strncmp(ifname, port->ifname,
			MAX_DEVICE_NAME_LEN)) {  /* Localization OK */
			printf("In remove_port: Found port %s\n",port->ifname);
			break;
		}
		parent = port;
		port = port->next;
	}

	if (port == NULL) {
		printf("remove_port: port not present\n");
		return -1;
	}
	/* Close down the socket */
	l2_packet_deinit(port->l2);

	/* Re-initialize relevant port variables */
	port->tx.state = TX_LLDP_INITIALIZE;
	port->rx.state = LLDP_WAIT_PORT_OPERATIONAL;
	port->portEnabled  = FALSE;
	port->adminStatus  = disabled;
	port->tx.txTTL = 0;

	/* Take this port out of the chain */
	if (parent == NULL) {
		porthead = port->next;
	} else if (parent->next == port) { /* Sanity check */
		parent->next = port->next;
	} else {
		return -1;
	}

	/* Remove the tlvs */
	if (port->tlvs.chassis != NULL) {
		port->tlvs.chassis = free_unpkd_tlv(port->tlvs.chassis);
	}

	if (port->tlvs.portid != NULL) {
		port->tlvs.portid = free_unpkd_tlv(port->tlvs.portid);
	}

	if (port->tlvs.ttl != NULL) {
		port->tlvs.ttl = free_unpkd_tlv(port->tlvs.ttl);
	}

	if (port->tlvs.control != NULL) {
		port->tlvs.control = free_unpkd_tlv(port->tlvs.control);
	}

	if (port->tlvs.pg1 != NULL) {
		port->tlvs.pg1 = free_unpkd_tlv(port->tlvs.pg1);
	}

	if (port->tlvs.pg2 != NULL) {
		port->tlvs.pg2 = free_unpkd_tlv(port->tlvs.pg2);
	}

	if (port->tlvs.pfc1 != NULL) {
		port->tlvs.pfc1 = free_unpkd_tlv(port->tlvs.pfc1);
	}

	if (port->tlvs.pfc2 != NULL) {
		port->tlvs.pfc2 = free_unpkd_tlv(port->tlvs.pfc2);
	}

	if (port->tlvs.app1 != NULL) {
		port->tlvs.app1 = free_unpkd_tlv(port->tlvs.app1);
	}

	if (port->tlvs.app2 != NULL) {
		port->tlvs.app2 = free_unpkd_tlv(port->tlvs.app2);
	}
	
	if (port->tlvs.llink != NULL) {
		port->tlvs.llink = free_unpkd_tlv(port->tlvs.llink);
	}


	if (port->tlvs.dcbx1 != NULL) {
		port->tlvs.dcbx1 = free_unpkd_tlv(port->tlvs.dcbx1);
	}

	if (port->tlvs.dcbx2 != NULL) {
		port->tlvs.dcbx2 = free_unpkd_tlv(port->tlvs.dcbx2);
	}

	if (port->tlvs.cur_peer != NULL) {
		port->tlvs.cur_peer = free_unpkd_tlv(port->tlvs.cur_peer);
	}

	if (port->tlvs.last_peer != NULL) {
		port->tlvs.last_peer =
			free_unpkd_tlv(port->tlvs.last_peer);
	}

	if (port->msap.msap1) {
		free(port->msap.msap1);
		port->msap.msap1 = NULL;
	}

	if (port->msap.msap2) {
		free(port->msap.msap2);
		port->msap.msap2 = NULL;
	}

	if (port->rx.framein)
		free(port->rx.framein);

	if (port->tx.frameout)
		free(port->tx.frameout);

	if (port->ifname)
		free(port->ifname);

	free(port);
	return 0;
}
