/*
  DB Playlist
  ===========
  Description: 
    a independant playlist program used to cue songs on multiple xmms sessions.

	Copyright (c) 1999, 2000, 2001 Robert Michael S Dean

	Author: Robert Michael S Dean
	Version: 1.0

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public Licensse as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/shm.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <limits.h>
#include <fcntl.h>
#include <signal.h>
#include <gtk/gtk.h>

#include <dbdebug.h>
#include <dbchannel.h>
#include <dbaudiolib.h>

#include "dbplaylist.h"

int debug_level;
GtkWidget* window;

int full_cue_flag = 0;

int shmid, sysshmid;
dbfsd_data * sysdata;
local_channel * local_channels;

int channel_index;
int msg_q_id;

GtkWidget * songlistbox;
GtkWidget * file_selector;
GtkWidget * player1_label, * player2_label;

char * songlist_titles[] = {SONG_TITLE,PLAYER_TITLE};

gint drag_flag = 0;
gint drag_row_src, drag_row_dest, drag_motion_last_row;
gint cue_index = 0;
gint selected_row = -1;

GdkColor white,red,blue,green,black,orange,pink;
GdkColormap * colormap;

int shmid, sysshmid;
dbfsd_data * sysdata;
local_channel * local_channels;

GtkTargetEntry drop_types[] =
{
	{"text/plain",			0,	DROP_PLAINTEXT},
	{"text/uri-list",		0,	DROP_URLENCODED},
	{"STRING",			0,	DROP_STRING}
};

char * playlist_filename_extensions[] = {
	".m3u",
	".M3U",
	NULL
};


player_id  player1_id, player2_id, * last_player = NULL;

int update_ok_flag = 1;
unsigned int cue_must_be = 0;

/***************************************************************/
/***************  Utility functions         ********************/
/***************************************************************/


void print_data(gpointer data)
{
	song_data * sd;
	
	sd = (song_data *) data;

    Debug("Song Data");
	Debug("===================");
	Debug("name: %s",sd->name);
	Debug("path: %s",sd->path);
	Debug("xmms: %d",sd->xmms_session_id);
}


static gint cleanup(GtkWidget* window, GdkEventAny* e, gpointer data)
{
	Debug("Got delete_event... calling cleanup()...");

	/* free shared memory */
	/* detach channel memory segment */
	if(shmdt(local_channels) == -1)
	{
		Error("DBMixer: could not detach channel memory segment.");
/* 		return FAILURE; */
	}

	/* detach system data memory segment */
	if(shmdt(sysdata) == -1)
	{
		Error("DBMixer: could not detach system data memory segment.");
/* 		return FAILURE; */
	}


	gtk_widget_destroy(window);
	window = NULL;

	gtk_main_quit();

	return TRUE;
}


/*
  cleanup function for when a row is removed from the list
*/
void destroy_data (gpointer data)
{
	song_data * sd;
	
	if (data == NULL) return;

	sd = (song_data *) data;

	g_free(sd->name);
	g_free(sd->path);
	g_free(sd);
}


/*
  when dragging within the list, set_drag_highlight changes the 
  backgournd color of the row the mouse is currently over, this 
  way you can optically determine which row will recieve the drop
*/
void set_drag_highlight(GtkWidget * w, gint row)
{
	GtkCList * list;

	list = (GtkCList *) w;

	gtk_clist_set_background(list,row,&green);
	gtk_clist_set_foreground(list,row,&black);		
}

/*
  set_cue_highlight sets the colors for the row associted with the 
  next song to be cued/played. 
 */
void set_cue_highlight(GtkWidget * w)
{
	GtkCList * list;

	list = (GtkCList *) w;

	gtk_clist_set_background(list,cue_index,&pink);
	gtk_clist_set_foreground(list,cue_index,&black);		
}


/*
  clear_highlight restores a row to its original color scheme
 */
void clear_highlight(GtkWidget *w, gint row)
{
	GtkCList * list;
	song_data * sd;

	list = (GtkCList *) w;

	gtk_clist_set_background(list,row,&white);

	sd = (song_data *) gtk_clist_get_row_data(list,row);

	if (sd == NULL) 
	{
		return;
	}

	if (sd->played)
	{
		gtk_clist_set_foreground(list,row,&red);
	}
	else
	{
		gtk_clist_set_foreground(list,row,&black);	
	}
}


/*
  swap_players - maintains alternating player sequnence from row
 */
void swap_players(GtkCList * list, int row)
{
	gint player_toggle;
	song_data * sd, * sd2;

	FREEZE_LIST(list);

	/* change player for rest of list */
	if (row == (list->rows - 1))
	{
		if (row != 0)
		{
			sd = (song_data *) gtk_clist_get_row_data(list,row-1);
			sd2 = (song_data *) gtk_clist_get_row_data(list,row);
			
			if (sd->player)
			{
				sd2->player = 0;
				gtk_clist_set_text(list,row,1,PLAYER1_STR);
			}
			else
			{
				sd2->player = 1;
				gtk_clist_set_text(list,row,1,PLAYER2_STR);
			}
		}
	}
	else
	{
		if (row < 0) row = 0;

		sd2 = (song_data *) gtk_clist_get_row_data(list,row);

		if (sd2->player) 
		{
			player_toggle = 0;
		}
		else
		{
			player_toggle = 1;
		}

		while (row < (list->rows -1))
		{				
  			row++;
			
			sd = (song_data *) gtk_clist_get_row_data(list,row);
			
			/* determine which player to use */
			if (player_toggle)
			{
				gtk_clist_set_text(list,row,1,PLAYER2_STR);
				sd->player = 1;
				player_toggle = 0;
				
			}
			else
			{
				gtk_clist_set_text(list,row,1,PLAYER1_STR);
				sd->player = 0;
				player_toggle = 1;
			}			
		}
	}

	THAW_LIST(list);
}


/*
  set_cue_player - makes sure that the next cue player always 
                   stays the same
 */
void set_cue_player()
{
	GtkCList * list;
	song_data * sd;

	list = (GtkCList *) songlistbox;

	if (list->rows == 0) return;

	sd = (song_data *) gtk_clist_get_row_data(list,cue_index);

	sd->player = cue_must_be;

	if (cue_must_be)
	{
		gtk_clist_set_text(list,cue_index,1,PLAYER2_STR);
	}
	else
	{
		gtk_clist_set_text(list,cue_index,1,PLAYER1_STR);
	}

	swap_players(list,cue_index);
}



/***************************************************************/
/***************    callback functions      ********************/
/***************************************************************/

/* called when gtk detects the begining of a drag */
void drag_data_begin(GtkWidget * widget, GdkDragContext *dc, gpointer data)
{
	GtkCList * list;

	list = (GtkCList *) widget;

	/* if there are no rows, there's no reason to drag */
	if (list->rows == 0) return;

	/* set drag_flag so that other functions will know that we are dragging */
	drag_flag = 1;
	
	/* remember row where the drag started */
	drag_row_src = list->click_cell.row;
	drag_motion_last_row = drag_row_src -1;

	set_drag_highlight(widget,drag_motion_last_row);
}

/* called when the drag is over */
void drag_data_end(GtkWidget * widget, GdkDragContext *dc, gpointer data)
{
	GtkCList * list;

	list = (GtkCList *) widget;

	if (list->rows == 0) return;

	/* clear the drag highlight bar since it is no longer needed */
	if (drag_motion_last_row != -1)
	{
		clear_highlight(widget,drag_motion_last_row);
	}
}


/* drag_data_motion - the mouse moved during a drag */
void drag_data_motion(GtkWidget *widget, GdkDragContext *dc, gint x, gint y, guint t, gpointer data)
{
	GtkCList * list;
	gint row, column;

	/* if we are dragging */
	if (drag_flag)
	{
		list = (GtkCList *) widget;

		if (list->rows == 0) return;
		
		/* freeze list to prevent flickering while clearing/setting highlights */
		FREEZE_LIST(list);

		/* find row the mouse is currently over */
		gtk_clist_get_selection_info(GTK_CLIST(songlistbox),x,y,&row,&column);	
		row--;

		/* make sure the row is within range */
		if (row == -1) row = 0;
		if ((row < 0) || (row > list->rows)) row = list->rows-1;
		
		/* remove last drag highlight */
		clear_highlight(widget,drag_motion_last_row);
		
		/* restore the cue highlight if it was overdrawn by the drag bar */
		if (drag_motion_last_row == cue_index) set_cue_highlight(widget);
		
		/* set location of new drag row */
		set_drag_highlight(widget,row);
		
		/* remember this row */
		drag_motion_last_row = row;
		
		THAW_LIST(list);
	}
}


/*
  drag_data_drop - a drag that began within the list widget has ended. 
                   As opposed to a drag from another application.
 */
gboolean drag_data_drop (GtkWidget *widget, GdkDragContext *dc, gint x, gint y, guint t, gpointer data)
{
	gint row, column;
	GtkCList * list;

	/* if we are dragging */
	if (drag_flag)
	{
		list = (GtkCList *) widget;

		/* if the aren't any rows in the list there's nothing to do */
		if (list->rows == 0) return TRUE;

		/* freeze list as many list operations are going to occur */
		FREEZE_LIST(list);

		drag_flag = 0;

		/* find out row where drop occured */
		gtk_clist_get_selection_info(GTK_CLIST(songlistbox),x,y,&row,&column);

		clear_highlight(widget,drag_motion_last_row);
		clear_highlight(widget,cue_index);

		drag_motion_last_row = -1;

		/* make sure row is within range */
		if (row > list->rows)
		{
			row = list->rows;
		}

		if (row > 0)
		{
			row--;
		}

		/* dragging is over, clear drag bar */
		clear_highlight(widget,row);

/* 		print_data(gtk_clist_get_row_data(list,drag_row_src)); */
/* 		print_data(gtk_clist_get_row_data(list,row)); */

		/* move start row to drop location */
		gtk_clist_row_move(GTK_CLIST(songlistbox),drag_row_src,row);

		/* if a song is moved from below the cue location to
           above it, move cue_index down one so that it's 
		   player does not change */
		if ((row < cue_index) && (drag_row_src > cue_index) 
			&& (cue_index < (list->rows-1)))
		{
			cue_index++;
		}

		/* if a song is moved from above the cue location to 
           below it, move the cue up one to keep it on the same song */
		if ((row > cue_index) && (cue_index) 
			&& (drag_row_src < cue_index)) 
		{
			cue_index--;
		}

		/* maintain cue order based on the cue_index */
	    set_cue_player();

		set_cue_highlight(widget);

		THAW_LIST(list);

/* 		return FALSE; */
	}
	else
	{
/* 		printf("drag is from OTHER window\n"); */
	}

	return TRUE;
}


/*
  drag_stat_received - a drag that started in another application has
                       been dropped in this application.
 */
void drag_data_received(GtkWidget * widget,
				       GdkDragContext * context,
				       gint x,
				       gint y,
				       GtkSelectionData * selection_data,
				       guint info,
				       guint time,
				       gpointer user_data)
{
	char * srcstr, * songname;
	gint   row,column,append_flag,player_toggle;
	char * list_labels[2], * tempstr;
	GtkCList * list;
	song_data * sd;
	static gchar linkstr[1024];
	static gchar templink[1024];
	gint   count;

	row = 0;
	column = 0;
	append_flag = 0;

	list = (GtkCList *) widget;

	gtk_clist_get_selection_info(GTK_CLIST(list),x,y,&row,&column);

	/* if a drag was in process within this application, exit */
	if (drag_flag)
	{
/* 		printf("**** same window\n"); */
		drag_flag = 0;
		return;
	}

	FREEZE_LIST(list);

	clear_highlight(songlistbox,cue_index);

	/* clear drag highlight */	
	clear_highlight(widget,drag_motion_last_row);
	drag_motion_last_row = -1;

	if (row == 0) 
	{
		append_flag = 1;
	}
	else
	{
		append_flag = 0;
		row--;
	}

	/* if the drop contained data, process it */
	if (selection_data->data)
	{
		srcstr = strtok(selection_data->data,"\n\r");
			
		/* step over "file" URL if it exists */
		if (strncmp(srcstr,"file:",5) == 0) srcstr+=5;

		Debug("%s",srcstr);

		/* determine if file is a link */
		if ((count = readlink(srcstr,linkstr,1024)) == -1)
		{
			/* file is not a link */
			
			if (errno != EINVAL)
			{
				perror("error code on readlink: ");
			}

			/* get songname from link */
			songname = srcstr + strlen(srcstr);
			
			while ((*songname != '/') && (songname  > srcstr))
			{
				songname--;
			}
			
			songname++;
		}
		else
		{
			/* process first file in drop list */

			linkstr[count] = '\0';
			tempstr = linkstr;

			/* find song name */

		    songname = linkstr + strlen(linkstr);

			while ((*songname != '/') && (songname > linkstr))
			{
				songname--;
			}

			/* find path to link file */

			tempstr = srcstr + strlen(srcstr);

			while ((*tempstr != '/') && (tempstr > linkstr))
			{
				tempstr--;
			}

			/* is the path absolute? */
			if (linkstr[0] != '/')
			{		
				gchar c;
						
				/* remember value in tempstr */
				c = *tempstr;
				*tempstr = '\0';
				
				/* absolute path is path to link file + the path
                   returned by readlink */
				sprintf(templink,"%s/%s",srcstr,linkstr);

				/* replace stored byte */
				srcstr = templink;
				*tempstr = c;
			}
		   
			/* increment songname if necessary */
			if (*songname == '/')
			{
				songname++;
			}
		}

		list_labels[0] = songname;

		/* create song data structure */
		sd = (song_data *) g_malloc0(sizeof(song_data));
		sd->name = g_strdup(songname);
		sd->path = g_strdup(srcstr);
		sd->xmms_session_id = -1;

		/* is this the first drop in an empty list? */
		if (row == 0)
		{
			list_labels[1] = PLAYER1_STR;
			player_toggle = 1;
			sd->player = 0;
		}
		else
		{
			/* list has data, determine state based on drop row */
			gtk_clist_get_text((GtkCList *)list,row-1,1,&tempstr);
			
			if (strcmp(tempstr,PLAYER1_STR) == 0)
			{
				list_labels[1] = PLAYER2_STR;
				player_toggle = 0;
				sd->player = 1;
			}
			else
			{
				list_labels[1] = PLAYER1_STR;
				player_toggle = 1;
				sd->player = 0;
			}
		}

		/* clear cue highlight if the drop occurred there */
		if (row == cue_index) clear_highlight(widget,cue_index);

		/* if the drop is at the end of the list, append it */
		if (append_flag)
		{
			row = gtk_clist_append((GtkCList *)list,list_labels);

			if ((cue_index == 0) && (list->rows == 1))
			{
				set_cue_highlight(widget);
			}
		}
		else
		{
			/* drop is not at end of list, insert it */
			gtk_clist_insert((GtkCList *)list,row,list_labels);

			if (row < cue_index)
			{
				cue_index++;
			}
		}

		gtk_clist_set_row_data_full((GtkCList *)list,row,sd,destroy_data);

		/* loop through remainder of list and add songs to playlist */
		while ((srcstr = strtok(NULL,"\r\n")) != NULL)
		{
			row++;

			/* extract the songname from the full path */
			songname = srcstr + strlen(srcstr);

			while ((*songname != '/') && (songname  > srcstr))
			{
				songname--;
			}

			songname++;

			/* step over "file" URL */
			if (strncmp(srcstr,"file:",5) == 0) srcstr+=5;
			
			/* allocate song data strucutre */
			sd = (song_data *) g_malloc0(sizeof(song_data));
			sd->name = g_strdup(songname);
			sd->path = g_strdup(srcstr);
			sd->xmms_session_id = -1;

			list_labels[0] = songname;

			/* determine which player to use */			
			if (player_toggle)
			{
				list_labels[1] = PLAYER2_STR;
				sd->player = 1;
				player_toggle = 0;
			}
			else
			{
				list_labels[1] = PLAYER1_STR;
				sd->player = 0;
				player_toggle = 1;
			}

			if (row == cue_index) clear_highlight(widget,cue_index);

			if (row < cue_index)
			{
				cue_index++;
			}

			/* add song to list */
			if (append_flag)
			{
				row = gtk_clist_append((GtkCList *)list,list_labels);
			}
			else
			{
				gtk_clist_insert((GtkCList *)list,row,list_labels);
			}

			gtk_clist_set_row_data_full((GtkCList *)list,row,sd,destroy_data);
		} /* end while */
	
		/* reorder players */
		set_cue_player();
	}

	set_cue_highlight(widget);

	THAW_LIST(list);
}


void select_row_callback(GtkWidget *widget,
                         gint row,gint column,
						 gpointer data)
{
	Debug("select row callback");
}


/*
  user selected the cue next song button
*/
void cue_next_song_clicked (GtkWidget *w, gchar *data)
{
	setup_next_song();
}


/*
  user selected a row, remember it
 */
void selection_made( GtkWidget      *list,
                     gint            row,
                     gint            column,
					 GdkEventButton *event,
                     gpointer        data )
{	
	selected_row = row;

    return;
}


/* selection callback from selection menu for player 1 */
void player1_select(GtkWidget *w, gpointer * data)
{
	player1_id.index = (gint) data;
}

/* callback from selection menu for player 2 */
void player2_select(GtkWidget *w, gpointer * data)
{
	player2_id.index = (gint) data;
}

/* user wishes to use full cue mode */
void full_cue_button_clicked(GtkWidget *w, gchar * data)
{
	full_cue_flag = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));
}

/* delete button was clicked, remove selected row from list */
void delete_button_clicked(GtkWidget *w, gchar * data)
{
	if (selected_row == -1) return;

	clear_highlight(songlistbox,cue_index);

	if (selected_row < cue_index)
	{
		cue_index--;
	}

	if (selected_row >= cue_index) set_cue_player();

	if (cue_index < 0) cue_index = 0;

	gtk_clist_remove((GtkCList *)songlistbox,selected_row);
	
	set_cue_player();

	set_cue_highlight(songlistbox);
}


/* swaps the players of all songs beneath the currently selected row */
void cue_swap_button_clicked(GtkWidget *w, gchar * data)
{
	gint row, player_toggle;
	song_data * sd;
	GtkCList * list;

	/* verify that a selection occured */
	if (selected_row == -1) return;

	list = (GtkCList *) songlistbox;

	row = selected_row;

	sd = (song_data *) gtk_clist_get_row_data(list,row);

	/* determine what the new player should be */
	if (sd->player)
	{
		player_toggle = 0;
	}
	else
	{
		player_toggle = 1;
	}

	if (row == cue_index) cue_must_be = player_toggle;

	/* step through list and swap players */
	while (row < (list->rows))
	{
		sd = (song_data *) gtk_clist_get_row_data(list,row);
		
		/* determine which player to use */
		if (player_toggle)
		{
			gtk_clist_set_text(list,row,1,PLAYER2_STR);
			sd->player = 1;
			player_toggle = 0;
			
		}
		else
		{
			gtk_clist_set_text(list,row,1,PLAYER1_STR);
			sd->player = 0;
			player_toggle = 1;
		}

		row++;
	}

	set_cue_player();
}


/*
  moves the cue hightlight bar up one row
 */
void cue_up_button_clicked(GtkWidget *w, gchar * data)
{
	if (cue_index == 0) return;
 
	clear_highlight(songlistbox,cue_index);
	cue_index--;
	
	set_cue_player();

	set_cue_highlight(songlistbox);
}


/* 
   moves the cue highlight bar down one row
*/
void cue_down_button_clicked(GtkWidget *w, gchar * data)
{
	GtkCList * list;

	list = (GtkCList *) songlistbox;

	if ((list->rows -1) > cue_index)
	{
		clear_highlight(songlistbox,cue_index);
		cue_index++;
		set_cue_player();

		set_cue_highlight(songlistbox);
	}
}


/*
  load a playlist from a file
 */
void load_playlist(GtkFileSelection *selector, gpointer user_data) 
{
	char * path;
	int    fid;
	int    i;
	int    playlist_flag;
	char   filename[1024];
	char * songname, * temppntr, * temppntr2;
	int    loop_flag;
	int    read_string_flag;
	int    error;
	int    len;
	int    row;
	song_data * sd;
	GtkCList * list;
	char * list_labels[2];
	gint   player_toggle;


	loop_flag = 1;
	row = 0;
	fid = 0;
	list = (GtkCList *) songlistbox;
	playlist_flag = 0;
	
	path = gtk_file_selection_get_filename (GTK_FILE_SELECTION(file_selector));
		
	/* check filename to see if it is a playlist or a file */
	/* is it a directory? */

	i = 0;
	while ((temppntr = playlist_filename_extensions[i]) != NULL)
	{
		/* check extension */
		if ((temppntr2 = strstr(path,temppntr)))
		{
			/* verify that the extension is at the end of the path name */
			if (*((char *)(temppntr2 + strlen(temppntr))) == '\0')
			{
				Debug("found extension %s in path %s\n",temppntr,path);
				playlist_flag = 1;
				break;
			}
		}

		i++;
	}

	/* verify that the playlist file has a known extension - this is a cheap
       hack, but how else are you able to detect if the file is a playlist or not? */
	if (!playlist_flag)
	{
		printf("\nERORR: selected file does not have a recognised playlist extension:\n");

		i = 0
;
		while ((temppntr = playlist_filename_extensions[i]) != NULL)
		{
			printf("  %s\n",temppntr);
			i++;
		}

		return;
 	}

    gtk_clist_clear(list);

	row = 0;
	player_toggle = 0;

	/* open path */
	path = gtk_file_selection_get_filename (GTK_FILE_SELECTION(file_selector));
	
	fid = open(path,O_RDONLY);
	
	FREEZE_LIST(list);

	/* read in playlist */
	while (loop_flag)
	{
		/* read in a song path */
		read_string_flag = 1;

		while (read_string_flag)
		{
			temppntr = filename;

			error = read(fid,temppntr,1);
			len = 0;
			
			/* read in line */
			while ((error > 0) && (*temppntr != '\n'))
			{
				temppntr++;
				error = read(fid,temppntr,1);
			    len++;
			}

			/* exit if there is an error */
			if (error == -1)
			{
				loop_flag = 0;
				break;
			}

			filename[len] = '\0';

			/* check to see if this string is valid, if not, reread */
			if (strstr(filename,XMMS_EXT_STR) == NULL)
			{
				read_string_flag = 0;
			}
		}

		/* check for EOF */
		if (len == 0)
		{
			loop_flag = 0;
			break;
		}
		else
		{
			/* extract song name from song path */
			songname = filename + strlen(filename);

			while ((*songname != '/') && (songname  > filename))
			{
				songname--;
			}

			songname++;

			/* create song_data structure */
			sd = (song_data *) g_malloc0(sizeof(song_data));
			sd->name = g_strdup(songname);
			sd->path = g_strdup(filename);
			sd->xmms_session_id = -1;

			list_labels[0] = songname;

			/* determine proper player for song */
			if (player_toggle)
			{
				list_labels[1] = PLAYER2_STR;
				sd->player = 1;
				player_toggle = 0;
			}
			else
			{
				list_labels[1] = PLAYER1_STR;
				sd->player = 0;
				player_toggle = 1;
			}

			/* add song to list */
			gtk_clist_insert((GtkCList *)list,row,list_labels);

			/* attach song_data to the song's row */
			gtk_clist_set_row_data_full((GtkCList *)list,row,sd,destroy_data);

			row++;
		}
	}

	set_cue_highlight(songlistbox);

	THAW_LIST(list);
}


/*
  user clicked the load playlist button 
*/
void load_button_clicked(GtkWidget *w, gchar * data)
{	
	file_selector = gtk_file_selection_new("Please select a file for editing.");
	
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION(file_selector)->ok_button),
						"clicked", GTK_SIGNAL_FUNC (load_playlist), NULL);
	
	/* ensure that the dialog box is destroyed when the user clicks a button. */
	
	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION(file_selector)->ok_button),
							   "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
							   (gpointer) file_selector);
	
	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION(file_selector)->cancel_button),
							   "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
							   (gpointer) file_selector);
	
	/* display that dialog */
	gtk_widget_show (file_selector);
}


/* write the playlist to a playlist file */
void save_file(GtkFileSelection *selector, gpointer user_data) 
{
	char * path;
	int    i;
	GtkCList * list;
	song_data * sd;
	FILE * outfile;

	/* get filename */
	path = gtk_file_selection_get_filename (GTK_FILE_SELECTION(file_selector));

	/* open file */
	outfile = fopen(path,"w");
	
	/* verify that open succeeded */
	if (outfile == NULL)
	{
		perror("ERROR: could not open playlist file for writing: \n");
		return;
	}

	list = (GtkCList*) songlistbox;

	FREEZE_LIST(list);

	/* write list to file */
	for (i = 0; i < list->rows; i++)
	{
		sd = gtk_clist_get_row_data(list,i);
		fprintf(outfile,"%s\n",sd->path);
	}

	fclose(outfile);

	THAW_LIST(list);
}


void save_button_clicked(GtkWidget *w, gchar * data)
{	
	file_selector = gtk_file_selection_new("Please select a file for editing.");
	
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION(file_selector)->ok_button),
						"clicked", GTK_SIGNAL_FUNC (save_file), NULL);
	
	/* ensure that the dialog box is destroyed when the user clicks a button. */
	
	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION(file_selector)->ok_button),
							   "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
							   (gpointer) file_selector);
	
	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION(file_selector)->cancel_button),
							   "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
							   (gpointer) file_selector);
	
	/* display that dialog */
	
	gtk_widget_show (file_selector);
}


void handle_motion(GtkWidget *widget, GdkEventMotion *event, 
						 gpointer user_data)
{
	int x,y;

	x = event->x;
	y = event->y;

/* 	Debug("handle_motion  x %ld y %ld\n",x,y); */
}


/***************************************************************/
/***************    widget creation functions   ****************/
/***************************************************************/

/* update_players - check to see if the dbmix clients have changed, and
                    if so, update the player selection menus */
int update_players(gpointer * data)
{
	gint i;
	GtkWidget * menu, * item;
	
	/* only update if the menu is not currently open - this causes a 
       inconsistent state between the data shown to the user and the data
       that actually exists */
	if (!GTK_WIDGET_VISIBLE(GTK_WIDGET(gtk_option_menu_get_menu(GTK_OPTION_MENU(player1_id.opt)))))
	{
		gtk_option_menu_remove_menu(GTK_OPTION_MENU(player1_id.opt));
		
		/* 	player1_id.opt = gtk_option_menu_new(); */
		
		menu = gtk_menu_new();
		
		/* add options to channel selector */
		for(i = 0; i < sysdata->num_channels;i++)
		{
			item = gtk_menu_item_new_with_label (local_channels[i].channel_name);
			gtk_signal_connect (GTK_OBJECT (item), "activate",
								player1_select,(gpointer) i);
			
			gtk_widget_show (item);
			
			gtk_menu_append (GTK_MENU (menu), item);
		}
		
			gtk_menu_set_active(GTK_MENU(menu), player1_id.index);
			
			/* add option list to channel selector */
			gtk_option_menu_set_menu (GTK_OPTION_MENU (player1_id.opt), menu);
			
			gtk_widget_show (player1_id.opt);
	}
	
	if (!GTK_WIDGET_VISIBLE(GTK_WIDGET(gtk_option_menu_get_menu(GTK_OPTION_MENU(player2_id.opt)))))	
	{
		menu = gtk_menu_new();
		
		/* add options to channel selector */
		for(i = 0; i < sysdata->num_channels;i++)
		{
			item = gtk_menu_item_new_with_label (local_channels[i].channel_name);
			gtk_signal_connect (GTK_OBJECT (item), "activate",
								player2_select,(gpointer) i);
			
			gtk_widget_show (item);
			
			gtk_menu_append (GTK_MENU (menu), item);
		}
		
		gtk_menu_set_active(GTK_MENU(menu), player2_id.index);
		
		/* add option list to channel selector */
		gtk_option_menu_set_menu (GTK_OPTION_MENU (player2_id.opt), menu);
		
		gtk_widget_show (player2_id.opt);
	}

	return update_ok_flag;
}


/*
  gtk application init
*/
void dbplaylist_init()
{
	gint        i;
	GtkWidget  *vbox, *hbox;
	GtkWidget  *channel_box;
    GtkWidget  *scrolled_window;
	GtkWidget  *button;
	GtkWidget * menu;
	GtkWidget * item;
	GtkWidget * label;


	player1_id.index = 0;
	player2_id.index = 1;

    window=gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_widget_set_usize(GTK_WIDGET(window), 350, 400);

    gtk_window_set_title(GTK_WINDOW(window), "DBMix Playlist");
    gtk_signal_connect(GTK_OBJECT(window),
		       "destroy",
		       GTK_SIGNAL_FUNC(cleanup),
		       NULL);
    
    vbox=gtk_vbox_new(FALSE, 5);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
    gtk_widget_show(vbox);


	/* create channel selectors */
	{
		channel_box = gtk_hbox_new(FALSE,0);
		gtk_widget_show(channel_box);

		gtk_container_set_border_width(GTK_CONTAINER(channel_box),0);

		player1_label = gtk_label_new(PLAYER1_STR);

		gtk_box_pack_start(GTK_BOX(channel_box),player1_label,TRUE,TRUE,5);
		gtk_widget_show(player1_label);
		
		player1_id.opt = gtk_option_menu_new();

		menu = gtk_menu_new();
		
		/* add options to channel selector */
		for(i = 0; i < sysdata->num_channels;i++)
		{
			item = gtk_menu_item_new_with_label (local_channels[i].channel_name);
			gtk_signal_connect (GTK_OBJECT (item), "activate",
								player1_select,(gpointer) i);

			gtk_widget_show (item);

			gtk_menu_append (GTK_MENU (menu), item);
		}
		
		gtk_menu_set_active(GTK_MENU(menu), player1_id.index);
		
		/* add option list to channel selector */
		gtk_option_menu_set_menu (GTK_OPTION_MENU (player1_id.opt), menu);

		gtk_widget_show (player1_id.opt);

		gtk_box_pack_start (GTK_BOX (channel_box), player1_id.opt, TRUE, TRUE, 0);

		player2_label = gtk_label_new(PLAYER2_STR);
		gtk_box_pack_start(GTK_BOX(channel_box),player2_label,TRUE,TRUE,5);
		gtk_widget_show(player2_label);
		
		player2_id.opt = gtk_option_menu_new();

		menu = gtk_menu_new();
		
		/* add options to channel selector */
		for(i = 0; i < sysdata->num_channels;i++)
		{
			item = gtk_menu_item_new_with_label (local_channels[i].channel_name);
			gtk_signal_connect (GTK_OBJECT (item), "activate",
								player1_select,(gpointer) i);

			gtk_widget_show (item);

			gtk_menu_append (GTK_MENU (menu), item);
		}
		
		gtk_menu_set_active(GTK_MENU(menu), player2_id.index);
		
		/* add option list to channel selector */
		gtk_option_menu_set_menu (GTK_OPTION_MENU (player2_id.opt), menu);

		gtk_widget_show (player2_id.opt);

		gtk_box_pack_start (GTK_BOX (channel_box), player2_id.opt, TRUE, TRUE, 0);

		gtk_box_pack_start(GTK_BOX(vbox),channel_box,FALSE,FALSE,0);

	}

	/* create */
	{
		hbox = gtk_hbox_new(TRUE,0);

		/* cue flag button */
		button = gtk_toggle_button_new();
		label = gtk_label_new(FULL_CUE_STR);
		gtk_container_add(GTK_CONTAINER(button),label);
		gtk_widget_show(label);
		gtk_widget_show(button);
		gtk_widget_show(hbox);

		gtk_signal_connect (GTK_OBJECT(button), "clicked", 
						   GTK_SIGNAL_FUNC(full_cue_button_clicked),NULL);
		gtk_box_pack_start(GTK_BOX(hbox),GTK_WIDGET(button),TRUE,TRUE,0);

		/* load list button */
		button = gtk_button_new();
		label = gtk_label_new(LOAD_STR);
		gtk_container_add(GTK_CONTAINER(button),label);
		gtk_widget_show(label);
		gtk_widget_show(button);

		gtk_signal_connect (GTK_OBJECT(button), "clicked", 
						   GTK_SIGNAL_FUNC(load_button_clicked),NULL);
		gtk_box_pack_start(GTK_BOX(hbox),GTK_WIDGET(button),TRUE,TRUE,0);

		/* save list button */
		button = gtk_button_new();
		label = gtk_label_new(SAVE_STR);
		gtk_container_add(GTK_CONTAINER(button),label);
		gtk_widget_show(label);
		gtk_widget_show(button);

		gtk_signal_connect (GTK_OBJECT(button), "clicked", 
						   GTK_SIGNAL_FUNC(save_button_clicked),NULL);
		gtk_box_pack_start(GTK_BOX(hbox),GTK_WIDGET(button),TRUE,TRUE,0);

		gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);		
	}

	/* create list manipulation buttons */
	{
		hbox = gtk_hbox_new(TRUE,0);

		gtk_widget_show(hbox);

		/* delete button */
		button = gtk_button_new();
		label = gtk_label_new(DELETE_STR);
		gtk_container_add(GTK_CONTAINER(button),label);
		gtk_widget_show(label);
		gtk_widget_show(button);

		gtk_signal_connect (GTK_OBJECT(button), "clicked", 
						   GTK_SIGNAL_FUNC(delete_button_clicked),NULL);
		gtk_box_pack_start(GTK_BOX(hbox),GTK_WIDGET(button),TRUE,TRUE,0);

		/* cue swap button */
		button = gtk_button_new();
		label = gtk_label_new(CUE_SWAP_STR);
		gtk_container_add(GTK_CONTAINER(button),label);
		gtk_widget_show(label);
		gtk_widget_show(button);

		gtk_signal_connect (GTK_OBJECT(button), "clicked", 
						   GTK_SIGNAL_FUNC(cue_swap_button_clicked),NULL);
		gtk_box_pack_start(GTK_BOX(hbox),GTK_WIDGET(button),TRUE,TRUE,0);


		/* cue up button */
		button = gtk_button_new();
		label = gtk_label_new(CUE_UP_STR);
		gtk_container_add(GTK_CONTAINER(button),label);
		gtk_widget_show(label);
		gtk_widget_show(button);

		gtk_signal_connect (GTK_OBJECT(button), "clicked", 
						   GTK_SIGNAL_FUNC(cue_up_button_clicked),NULL);
		gtk_box_pack_start(GTK_BOX(hbox),GTK_WIDGET(button),TRUE,TRUE,0);

		/* cue down button */
		button = gtk_button_new();
		label = gtk_label_new(CUE_DOWN_STR);
		gtk_container_add(GTK_CONTAINER(button),label);
		gtk_widget_show(label);
		gtk_widget_show(button);

		gtk_signal_connect (GTK_OBJECT(button), "clicked", 
						   GTK_SIGNAL_FUNC(cue_down_button_clicked),NULL);
		gtk_box_pack_start(GTK_BOX(hbox),GTK_WIDGET(button),TRUE,TRUE,0);


		gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);	
	}


	/* add cue next song button */
	{
		button = gtk_button_new_with_label("Cue Next Song");
		
		gtk_widget_show(button);

		gtk_signal_connect(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(cue_next_song_clicked),NULL);

		gtk_box_pack_start(GTK_BOX(vbox),button,FALSE,FALSE,0);
	}

    
	/* create listbox */
	{
		scrolled_window = gtk_scrolled_window_new (NULL, NULL);
		gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
										GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
		
		gtk_box_pack_start(GTK_BOX(vbox), scrolled_window, TRUE, TRUE, 0);
		gtk_widget_show (scrolled_window);
		
		songlistbox = gtk_clist_new_with_titles(NUM_SONGLIST_COLUMNS, songlist_titles);
		
		for (i = 0; i < NUM_SONGLIST_COLUMNS; i++)
		{
			gtk_clist_set_column_justification(GTK_CLIST(songlistbox),i,GTK_JUSTIFY_LEFT);
		}

		gtk_signal_connect(GTK_OBJECT(songlistbox), "select_row",
						   GTK_SIGNAL_FUNC(selection_made),
						   NULL);
		
		gtk_clist_set_shadow_type (GTK_CLIST(songlistbox), GTK_SHADOW_OUT);
		
		gtk_clist_set_column_width (GTK_CLIST(songlistbox), 0, 250);
		/* 	gtk_clist_set_column_width(GTK_CLIST(songlistbox),1,50); */
		
		gtk_container_add(GTK_CONTAINER(scrolled_window), songlistbox);
		gtk_widget_show(songlistbox);
		
		hbox = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);
		gtk_widget_show(hbox);
		
		gtk_signal_connect(GTK_OBJECT(songlistbox),
						   "row_move",
						   GTK_SIGNAL_FUNC(select_row_callback),
						   NULL);
		
		/* set up drag from other programs */
		gtk_drag_dest_set(songlistbox, GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_DROP, 
						  drop_types, 3, GDK_ACTION_COPY | GDK_ACTION_MOVE);

		gtk_signal_connect(GTK_OBJECT(songlistbox), "drag_data_received", GTK_SIGNAL_FUNC(drag_data_received), NULL);
		
		
		/* set up drag from this program */
		gtk_drag_source_set(songlistbox, GDK_BUTTON1_MASK | GDK_BUTTON2_MASK | GDK_BUTTON3_MASK, 
							drop_types, 3, GDK_ACTION_COPY | GDK_ACTION_MOVE);
		
		gtk_signal_connect(GTK_OBJECT(songlistbox), "drag_begin", 
						   GTK_SIGNAL_FUNC(drag_data_begin), NULL);
		
		gtk_signal_connect(GTK_OBJECT(songlistbox), "drag_end", 
						   GTK_SIGNAL_FUNC(drag_data_end), NULL);
		
		gtk_signal_connect(GTK_OBJECT(songlistbox), "drag_drop", 
						   GTK_SIGNAL_FUNC(drag_data_drop), NULL);
		
		gtk_signal_connect(GTK_OBJECT(songlistbox), "drag_motion", 
						   GTK_SIGNAL_FUNC(drag_data_motion), NULL);
	}


    gtk_container_add(GTK_CONTAINER(window), vbox);

    gtk_widget_show(window);	

	/* setup timeouts */
	gtk_timeout_add(UPDATE_GUI_TIMEOUT,(GtkFunction)update_players,NULL);
}


/*
  color_init - get gdk_color structures for major colors
 */
void color_init()
{

	colormap = gtk_widget_get_colormap(songlistbox);

	gdk_color_parse("white",&white);
	gdk_colormap_alloc_color(colormap,&white,FALSE,TRUE);

	gdk_color_parse("red",&red);
	gdk_colormap_alloc_color(colormap,&red,FALSE,TRUE);

	gdk_color_parse("blue",&blue);
	gdk_colormap_alloc_color(colormap,&blue,FALSE,TRUE);

	gdk_color_parse("green",&green);
	gdk_colormap_alloc_color(colormap,&green,FALSE,TRUE);

	gdk_color_parse("black",&black);
	gdk_colormap_alloc_color(colormap,&black,FALSE,TRUE);

	gdk_color_parse("pink",&pink);
	gdk_colormap_alloc_color(colormap,&pink,FALSE,TRUE);

	gdk_color_parse("orange",&orange);
	gdk_colormap_alloc_color(colormap,&orange,FALSE,TRUE);


}


int main( int argc, char *argv[] )
{
	debug_level = 0;
	channel_index = 0;

	songlistbox = NULL;

#ifdef DBMIX_DEBUG
	debug_level = 1;
	printf("\nDebugging is on.\n");
#else
	debug_level = 0;
#endif	

	/* get system data */

	sysshmid = shmget((key_t) DB_SYSTEM_SM_KEY, sizeof(dbfsd_data), 0666);

	if(sysshmid == -1) 
	{
		Error("DBMixer ERROR: could not create shared memory for system data.");
		 
		return FAILURE; 
	}
   

	sysdata = shmat(sysshmid,(void *)0, 0);
   
	if((int)sysdata == -1)
	{
		Error("DBMixer ERROR: error attaching system data shared memory.");
      
		return FAILURE;
	}

	/* verify that there is a free channel into dbfsd */
	if(sysdata->free_channel_index == -1)
	{
/* 		ch = NULL; */

		if(shmdt(sysdata) == -1)
		{
			Error("DBAudio_Init: could not detach system data memory segment.");
			return FAILURE;
		}

		errno = ERROR_NO_FREE_CHANNELS;
		return FAILURE;
	}

	/* retrieve the channel memory id */
	shmid = shmget((key_t) DB_CHANNELS_SM_KEY, 
				   (sysdata->num_channels * sizeof(local_channel)), 0666);

	if(shmid == -1) 
	{
		Error("DBMixer ERROR: error creating shared memory.");
		return FAILURE;
	}
	else
    {
		Debug("DBMixer: shmid is: %d ",shmid);
    }
  
	/* attach the channel memory segment*/
	local_channels = (local_channel *) shmat(shmid,(void *)0, 0);

	if((int)local_channels == -1)
	{
		Error("DBMixer ERROR: error attaching shared memory.");

		return FAILURE;
	}

    gtk_init(&argc, &argv);
    
	dbplaylist_init();

	color_init();

    gtk_main();
    
    return(0);
}
