#include <errno.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <unistd.h>

#include "log.h"
#include "util.h"

#ifndef O_TMPFILE
#define __O_TMPFILE 020000000
#define O_TMPFILE (__O_TMPFILE | O_DIRECTORY)
#endif

using namespace std;

int make_tempfile(const string &contents)
{
	int fd = open("/tmp", O_RDWR | O_TMPFILE, 0600);
	if (fd == -1) {
		char filename[] = "/tmp/cubemap.XXXXXX";
		mode_t old_umask = umask(077);
		fd = mkstemp(filename);
		if (fd == -1) {
			log_perror("mkstemp");
			return -1;
		}
		umask(old_umask);

		if (unlink(filename) == -1) {
			log_perror("unlink");
			// Can still continue.
		}
	}

	const char *ptr = contents.data();
	size_t to_write = contents.size();
	while (to_write > 0) {
		ssize_t ret = write(fd, ptr, to_write);
		if (ret == -1) {
			log_perror("write");
			safe_close(fd);
			return -1;
		}

		ptr += ret;
		to_write -= ret;
	}

	return fd;
}

bool read_tempfile_and_close(int fd, string *contents)
{
	bool ok = read_tempfile(fd, contents);
	safe_close(fd);  // Implicitly deletes the file.
	return ok;
}

bool read_tempfile(int fd, string *contents)
{
	ssize_t ret, has_read;

	off_t len = lseek(fd, 0, SEEK_END);
	if (len == -1) {
		log_perror("lseek");
		return false;
	}

	contents->resize(len);

	if (lseek(fd, 0, SEEK_SET) == -1) {
		log_perror("lseek");
		return false;
	}

	has_read = 0;
	while (has_read < len) {
		ret = read(fd, &((*contents)[has_read]), len - has_read);
		if (ret == -1) {
			log_perror("read");
			return false;
		}
		if (ret == 0) {
			log(ERROR, "Unexpected EOF!");
			return false;
		}
		has_read += ret;
	}

	return true;
}

int safe_close(int fd)
{
	int ret;
	do {
		ret = close(fd);
	} while (ret == -1 && errno == EINTR);

	if (ret == -1) {
		log_perror("close()");
	}

	return ret;
}
