/*
** (c) 1996-2000 The Regents of the University of California (through
** E.O. Lawrence Berkeley National Laboratory), subject to approval by
** the U.S. Department of Energy.  Your use of this software is under
** license -- the license agreement is attached and included in the
** directory as license.txt or you may contact Berkeley Lab's Technology
** Transfer Department at TTD@lbl.gov.  NOTICE OF U.S. GOVERNMENT RIGHTS.
** The Software was developed under funding from the U.S. Government
** which consequently retains certain rights as follows: the
** U.S. Government has been granted for itself and others acting on its
** behalf a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, and perform publicly
** and display publicly.  Beginning five (5) years after the date
** permission to assert copyright is obtained from the U.S. Department of
** Energy, and subject to any subsequent five (5) year renewals, the
** U.S. Government is granted for itself and others acting on its behalf
** a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, distribute copies to
** the public, perform publicly and display publicly, and to permit
** others to do so.
*/


#ifndef _RESTRICTOR_H_
#define _RESTRICTOR_H_

#include "interface.H"
#include "boundary.H"

//
// This is the base class for all amr_restrictors.  The basic functions
// box and fill work in the interiors of patches:  Give box the fine
// grid dimensions and it will tell you what coarse patch you can fill
// from it; fill then does the work.  The interface routine is for
// restrictors like node-based full-weighting that do special stuff
// at the edges of grids.
//

class amr_restrictor
{
public:
    virtual ~amr_restrictor () = 0;
    virtual Box box (const Box&,
		     const IntVect&) const;
    virtual Box rebox (const Box&,
		       const IntVect&) const;
    virtual void fill (FArrayBox&,
		       const Box&,
		       const FArrayBox&,
		       const IntVect&) const = 0;
    virtual void fill_interface (MultiFab&, MultiFab&,
				 const level_interface&,
				 const amr_boundary*,
				 const IntVect&) const;
};

class cell_average_restrictor
    : public amr_restrictor
{
public:
    explicit cell_average_restrictor (int i) ;
    virtual void fill (FArrayBox& patch, const Box& region,
		       const FArrayBox& fgr, const IntVect& rat) const;
protected:
    int integrate;
};

inline
cell_average_restrictor::cell_average_restrictor (int i)
    :
    integrate(i)
{
}

class terrain_velocity_restrictor
    : public cell_average_restrictor
{
public:
    explicit terrain_velocity_restrictor (int i) ;
    virtual void fill (FArrayBox& patch, const Box& region,
		       const FArrayBox& fgr, const IntVect& rat) const;
};

inline
terrain_velocity_restrictor::terrain_velocity_restrictor (int i)
    : cell_average_restrictor(i)
{
    BL_ASSERT(i >= 0 && i < BL_SPACEDIM);
}

class injection_restrictor
    : public amr_restrictor
{
public:
    virtual void fill (FArrayBox& patch, const Box& region,
		       const FArrayBox& fgr, const IntVect& rat) const;
};

class default_restrictor
    : public amr_restrictor
{
public:
    virtual void fill (FArrayBox& patch, const Box& region,
		       const FArrayBox& fgr, const IntVect& rat) const;
};

//
// Vanilla bilinear (full-weighting) restrictor class.
// Restricts fine-fine interfaces, even those at the boundary.
// Does nothing at coarse-fine interfaces.
//

class bilinear_restrictor
    : public amr_restrictor
{
public:
    bilinear_restrictor (int i,
			 bool hg_terrain);
    virtual Box box (const Box& fb,
		     const IntVect& rat) const;
    virtual Box rebox (const Box& cb,
		       const IntVect& rat) const;
    virtual void fill (FArrayBox& patch,
		       const Box& region,
		       const FArrayBox& fgr,
		       const IntVect& rat) const;
    virtual void fill_interface (MultiFab&,
				 MultiFab&,
				 const level_interface&,
				 const amr_boundary*,
				 const IntVect&) const;
protected:
    int  integrate;
    bool m_hg_dense;
};

#endif /*_RESTRICTOR_H_*/
