//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Sample/ParticleLayoutItem.h
//! @brief     Defines class ParticleLayoutItem
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_MODEL_SAMPLE_PARTICLELAYOUTITEM_H
#define BORNAGAIN_GUI_MODEL_SAMPLE_PARTICLELAYOUTITEM_H

#include "GUI/Model/Descriptor/DoubleProperty.h"
#include "GUI/Model/Descriptor/SelectionProperty.h"
#include "GUI/Model/Sample/InterferenceItemCatalog.h"
#include "GUI/Model/Sample/InterferenceItems.h"
#include "GUI/Model/Sample/Item3D.h"
#include "GUI/Model/Sample/ItemWithParticlesCatalog.h"
#include <QUuid>
#include <memory>

class ItemWithParticles;
class MaterialModel;

class ParticleLayoutItem : public Item3D {
public:
    ParticleLayoutItem(const MaterialModel* materials);

    //! The density value which belonging only to the layout.
    //!
    //! This is the editable value. If an interference is present, this value may not be the one to
    //! be used for building the domain layout or to be presented. For the correct value-to-use,
    //! whether an interference is present or not, use totalDensityValue().
    DoubleProperty& ownDensity() { return m_ownDensity; }
    const DoubleProperty& ownDensity() const { return m_ownDensity; }
    void setOwnDensity(double v) { m_ownDensity.setValue(v); }

    //! The real density.
    //!
    //! Returns ownDensity() value if the interference is not influencing the density. If the
    //! interference is defining the density, this is the interference-calculated density.
    double totalDensityValue() const;

    //! The particles this layout contains.
    //!
    //! Note that this is not a hierarchical search! Only the topmost particles are returned, not
    //! their contained particles. For a hierarchical search use containedItemsWithParticles
    QVector<ItemWithParticles*> itemsWithParticles() const;
    SelectionProperty<ItemWithParticlesCatalog>&
    addItemWithParticleSelection(ItemWithParticles* particle);
    void removeItemWithParticle(ItemWithParticles* particle);

    //! Return full hierarchical contained items with particles.
    //!
    //! \sa particles()
    QVector<ItemWithParticles*> containedItemsWithParticles() const;

    SelectionProperty<InterferenceItemCatalog>& interferenceSelection() { return m_interference; }
    const SelectionProperty<InterferenceItemCatalog>& interferenceSelection() const;
    void setInterference(InterferenceItem* i) { m_interference.setCurrentItem(i); }
    void removeInterference() { m_interference.setCurrentItem(nullptr); }

    //! Returns whether total density is defined by the currently selected interference.
    //!
    //! Two dimensional interference calculates density automatically; in these cases the "own"
    //! total density value should not be edited but set by the one from the interference.
    bool totalDensityIsDefinedByInterference() const;

    void writeTo(QXmlStreamWriter* w) const;
    void readFrom(QXmlStreamReader* r);

    bool isExpandLayout() const { return m_expandParticleLayout; }
    void setExpandLayout(bool b) { m_expandParticleLayout = b; }

    bool isExpandInterference() const { return m_expandInterference; }
    void setExpandInterference(bool b) { m_expandInterference = b; }

private:
    DoubleProperty m_ownDensity;
    SelectionProperty<InterferenceItemCatalog> m_interference;
    SelectionVector<ItemWithParticlesCatalog> m_particles;
    const MaterialModel* m_materialModel;
    bool m_expandParticleLayout = true;
    bool m_expandInterference = true;
};

#endif // BORNAGAIN_GUI_MODEL_SAMPLE_PARTICLELAYOUTITEM_H
