/*
 * Copyright (c) 2021, 2023 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef ARM_COMPUTE_ENABLE_SVE

#include "arm_gemm.hpp"
#include "../../utils.hpp"

#include <cassert>

namespace arm_gemm {

void sve_hybrid_u8u32_mmla_6x4VL (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<uint8_t> A_arg,
    size_t M, size_t N, const uint8_t *B_ptr, IndirectOutputArg<uint32_t> output_arg,
    const uint32_t *, Activation, bool accumulate
)
{
    struct KernelArgs {
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const uint8_t *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
    } ka;

    unsigned long flags=0;
    void *output_ptr;
    void *input_ptr;

    if (output_arg.is_indirect) {
        output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    __asm__ __volatile__(
      "ptrue p5.b\n"
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 56f\n"
      "cmp %x[M], #0x4\n"
      "bgt 45f\n"
      "beq 34f\n"
      "cmp %x[M], #0x2\n"
      "bgt 23f\n"
      "beq 12f\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "2:"  // Height 1: Column loop
      "mov x20, #0x0\n"
      "whilelt p4.s, x20, x11\n"
      "incw x20\n"
      "whilelt p3.s, x20, x11\n"
      "incw x20\n"
      "whilelt p2.s, x20, x11\n"
      "incw x20\n"
      "whilelt p1.s, x20, x11\n"
      "tbz %x[flags], #0, 3f\n"
      "ld1w { z19.s }, p4/Z, [x9]\n"
      "ld1w { z18.s }, p3/Z, [x9, #1, MUL VL]\n"
      "zip1 z8.d, z19.d, z12.d\n"
      "zip2 z12.d, z19.d, z12.d\n"
      "ld1w { z17.s }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1w { z16.s }, p1/Z, [x9, #3, MUL VL]\n"
      "zip1 z9.d, z18.d, z13.d\n"
      "zip2 z13.d, z18.d, z13.d\n"
      "zip1 z10.d, z17.d, z14.d\n"
      "zip2 z14.d, z17.d, z14.d\n"
      "zip1 z11.d, z16.d, z15.d\n"
      "zip2 z15.d, z16.d, z15.d\n"
      "b 4f\n"
      "3:"  // Height 1: no accumulate
      "mov z8.s, #0x0\n"
      "mov z9.s, #0x0\n"
      "mov z10.s, #0x0\n"
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "mov z13.s, #0x0\n"
      "mov z14.s, #0x0\n"
      "mov z15.s, #0x0\n"
      "4:"  // Height 1: setup done
      "mov x28, #0x0\n"
      "5:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 6f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "cbnz x28, 7f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "b 7f\n"
      "6:"  // Height 1: setup direct input
      "mov x26, %x[input_ptr]\n"
      "7:"  // Height 1: input setup done
      "cmp x27, #0x10\n"
      "ble 9f\n"
      "8:"  // Height 1: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x27\n"
      "ld1rqb { z20.b }, p0/Z, [x26]\n"
      "trn1 z18.d, z20.d, z19.d\n"
      "ld1b { z17.b }, p5/Z, [x10]\n"
      "ld1b { z16.b }, p5/Z, [x10, #1, MUL VL]\n"
      ".inst 0x45d19a48  // ummla z8.s, z18.b, z17.b\n"
      ".inst 0x45d09a4c  // ummla z12.s, z18.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #3, MUL VL]\n"
      ".inst 0x45d19a49  // ummla z9.s, z18.b, z17.b\n"
      ".inst 0x45d09a4d  // ummla z13.s, z18.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #4, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #5, MUL VL]\n"
      ".inst 0x45d19a4a  // ummla z10.s, z18.b, z17.b\n"
      ".inst 0x45d09a4e  // ummla z14.s, z18.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #6, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "trn2 z20.d, z20.d, z19.d\n"
      ".inst 0x45d19a4b  // ummla z11.s, z18.b, z17.b\n"
      ".inst 0x45d09a4f  // ummla z15.s, z18.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #-8, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #-7, MUL VL]\n"
      ".inst 0x45d19a88  // ummla z8.s, z20.b, z17.b\n"
      ".inst 0x45d09a8c  // ummla z12.s, z20.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #-6, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #-5, MUL VL]\n"
      ".inst 0x45d19a89  // ummla z9.s, z20.b, z17.b\n"
      ".inst 0x45d09a8d  // ummla z13.s, z20.b, z16.b\n"
      "ld1b { z16.b }, p5/Z, [x10, #-4, MUL VL]\n"
      "ld1b { z7.b }, p5/Z, [x10, #-3, MUL VL]\n"
      ".inst 0x45d09a8a  // ummla z10.s, z20.b, z16.b\n"
      ".inst 0x45c79a8e  // ummla z14.s, z20.b, z7.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #-2, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #-1, MUL VL]\n"
      "sub x27, x27, #0x10\n"
      "cmp x27, #0x10\n"
      ".inst 0x45d19a8b  // ummla z11.s, z20.b, z17.b\n"
      ".inst 0x45d09a8f  // ummla z15.s, z20.b, z16.b\n"
      "add x26, x26, #0x10\n"
      "bgt 8b\n"
      "9:"  // Height 1: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x27\n"
      "ld1rqb { z1.b }, p0/Z, [x26]\n"
      "trn1 z18.d, z1.d, z19.d\n"
      "ld1b { z17.b }, p5/Z, [x10]\n"
      "ld1b { z16.b }, p5/Z, [x10, #1, MUL VL]\n"
      ".inst 0x45d19a48  // ummla z8.s, z18.b, z17.b\n"
      ".inst 0x45d09a4c  // ummla z12.s, z18.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #3, MUL VL]\n"
      ".inst 0x45d19a49  // ummla z9.s, z18.b, z17.b\n"
      ".inst 0x45d09a4d  // ummla z13.s, z18.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #4, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #5, MUL VL]\n"
      ".inst 0x45d19a4a  // ummla z10.s, z18.b, z17.b\n"
      ".inst 0x45d09a4e  // ummla z14.s, z18.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #6, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #7, MUL VL]\n"
      "subs x27, x27, #0x8\n"
      "trn2 z1.d, z1.d, z19.d\n"
      ".inst 0x45d19a4b  // ummla z11.s, z18.b, z17.b\n"
      ".inst 0x45d09a4f  // ummla z15.s, z18.b, z16.b\n"
      "addvl x10, x10, #8\n"
      "ble 10f\n"
      "ld1b { z17.b }, p5/Z, [x10]\n"
      "ld1b { z16.b }, p5/Z, [x10, #1, MUL VL]\n"
      ".inst 0x45d19828  // ummla z8.s, z1.b, z17.b\n"
      ".inst 0x45d0982c  // ummla z12.s, z1.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #3, MUL VL]\n"
      ".inst 0x45d19829  // ummla z9.s, z1.b, z17.b\n"
      ".inst 0x45d0982d  // ummla z13.s, z1.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #4, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #5, MUL VL]\n"
      ".inst 0x45d1982a  // ummla z10.s, z1.b, z17.b\n"
      ".inst 0x45d0982e  // ummla z14.s, z1.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #6, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #7, MUL VL]\n"
      ".inst 0x45d1982b  // ummla z11.s, z1.b, z17.b\n"
      ".inst 0x45d0982f  // ummla z15.s, z1.b, z16.b\n"
      "addvl x10, x10, #8\n"
      "10:"  // Height 1: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 5b\n"
      "uzp1 z8.d, z8.d, z12.d\n"
      "uzp1 z9.d, z9.d, z13.d\n"
      "st1w { z8.s }, p4, [x9]\n"
      "uzp1 z10.d, z10.d, z14.d\n"
      "uzp1 z11.d, z11.d, z15.d\n"
      "st1w { z9.s }, p3, [x9, #1, MUL VL]\n"
      "st1w { z10.s }, p2, [x9, #2, MUL VL]\n"
      "st1w { z11.s }, p1, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "11:"  // Height 1: Writeback done
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 2b\n"
      "b 68f\n"
      "12:"  // Height 2
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "13:"  // Height 2: Column loop
      "mov x20, #0x0\n"
      "whilelt p4.s, x20, x11\n"
      "incw x20\n"
      "whilelt p3.s, x20, x11\n"
      "incw x20\n"
      "whilelt p2.s, x20, x11\n"
      "incw x20\n"
      "whilelt p1.s, x20, x11\n"
      "tbz %x[flags], #0, 14f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x20, x9, x20, LSL #2\n"
      "ld1w { z18.s }, p4/Z, [x9]\n"
      "ld1w { z2.s }, p3/Z, [x9, #1, MUL VL]\n"
      "ld1w { z17.s }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1w { z16.s }, p1/Z, [x9, #3, MUL VL]\n"
      "ld1w { z12.s }, p4/Z, [x20]\n"
      "zip1 z8.d, z18.d, z12.d\n"
      "zip2 z12.d, z18.d, z12.d\n"
      "ld1w { z13.s }, p3/Z, [x20, #1, MUL VL]\n"
      "ld1w { z14.s }, p2/Z, [x20, #2, MUL VL]\n"
      "zip1 z9.d, z2.d, z13.d\n"
      "zip2 z13.d, z2.d, z13.d\n"
      "ld1w { z15.s }, p1/Z, [x20, #3, MUL VL]\n"
      "zip1 z10.d, z17.d, z14.d\n"
      "zip2 z14.d, z17.d, z14.d\n"
      "zip1 z11.d, z16.d, z15.d\n"
      "zip2 z15.d, z16.d, z15.d\n"
      "b 15f\n"
      "14:"  // Height 2: no accumulate
      "mov z8.s, #0x0\n"
      "mov z9.s, #0x0\n"
      "mov z10.s, #0x0\n"
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "mov z13.s, #0x0\n"
      "mov z14.s, #0x0\n"
      "mov z15.s, #0x0\n"
      "15:"  // Height 2: setup done
      "mov x28, #0x0\n"
      "16:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 17f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "cbnz x28, 18f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "b 18f\n"
      "17:"  // Height 2: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "18:"  // Height 2: input setup done
      "cmp x27, #0x10\n"
      "ble 20f\n"
      "19:"  // Height 2: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x27\n"
      "ld1rqb { z20.b }, p0/Z, [x26]\n"
      "ld1rqb { z19.b }, p0/Z, [x25]\n"
      "trn1 z18.d, z20.d, z19.d\n"
      "ld1b { z17.b }, p5/Z, [x10]\n"
      "ld1b { z16.b }, p5/Z, [x10, #1, MUL VL]\n"
      ".inst 0x45d19a48  // ummla z8.s, z18.b, z17.b\n"
      ".inst 0x45d09a4c  // ummla z12.s, z18.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #3, MUL VL]\n"
      ".inst 0x45d19a49  // ummla z9.s, z18.b, z17.b\n"
      ".inst 0x45d09a4d  // ummla z13.s, z18.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #4, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #5, MUL VL]\n"
      ".inst 0x45d19a4a  // ummla z10.s, z18.b, z17.b\n"
      ".inst 0x45d09a4e  // ummla z14.s, z18.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #6, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "trn2 z20.d, z20.d, z19.d\n"
      ".inst 0x45d19a4b  // ummla z11.s, z18.b, z17.b\n"
      ".inst 0x45d09a4f  // ummla z15.s, z18.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #-8, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #-7, MUL VL]\n"
      ".inst 0x45d19a88  // ummla z8.s, z20.b, z17.b\n"
      ".inst 0x45d09a8c  // ummla z12.s, z20.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #-6, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #-5, MUL VL]\n"
      ".inst 0x45d19a89  // ummla z9.s, z20.b, z17.b\n"
      ".inst 0x45d09a8d  // ummla z13.s, z20.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #-4, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #-3, MUL VL]\n"
      ".inst 0x45d19a8a  // ummla z10.s, z20.b, z17.b\n"
      ".inst 0x45d09a8e  // ummla z14.s, z20.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #-2, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #-1, MUL VL]\n"
      "sub x27, x27, #0x10\n"
      "cmp x27, #0x10\n"
      ".inst 0x45d19a8b  // ummla z11.s, z20.b, z17.b\n"
      ".inst 0x45d09a8f  // ummla z15.s, z20.b, z16.b\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "bgt 19b\n"
      "20:"  // Height 2: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x27\n"
      "ld1rqb { z1.b }, p0/Z, [x26]\n"
      "ld1rqb { z19.b }, p0/Z, [x25]\n"
      "trn1 z18.d, z1.d, z19.d\n"
      "ld1b { z17.b }, p5/Z, [x10]\n"
      "ld1b { z16.b }, p5/Z, [x10, #1, MUL VL]\n"
      ".inst 0x45d19a48  // ummla z8.s, z18.b, z17.b\n"
      ".inst 0x45d09a4c  // ummla z12.s, z18.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #3, MUL VL]\n"
      ".inst 0x45d19a49  // ummla z9.s, z18.b, z17.b\n"
      ".inst 0x45d09a4d  // ummla z13.s, z18.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #4, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #5, MUL VL]\n"
      ".inst 0x45d19a4a  // ummla z10.s, z18.b, z17.b\n"
      ".inst 0x45d09a4e  // ummla z14.s, z18.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #6, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #7, MUL VL]\n"
      "subs x27, x27, #0x8\n"
      "trn2 z1.d, z1.d, z19.d\n"
      ".inst 0x45d19a4b  // ummla z11.s, z18.b, z17.b\n"
      ".inst 0x45d09a4f  // ummla z15.s, z18.b, z16.b\n"
      "addvl x10, x10, #8\n"
      "ble 21f\n"
      "ld1b { z17.b }, p5/Z, [x10]\n"
      "ld1b { z16.b }, p5/Z, [x10, #1, MUL VL]\n"
      ".inst 0x45d19828  // ummla z8.s, z1.b, z17.b\n"
      ".inst 0x45d0982c  // ummla z12.s, z1.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #3, MUL VL]\n"
      ".inst 0x45d19829  // ummla z9.s, z1.b, z17.b\n"
      ".inst 0x45d0982d  // ummla z13.s, z1.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #4, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #5, MUL VL]\n"
      ".inst 0x45d1982a  // ummla z10.s, z1.b, z17.b\n"
      ".inst 0x45d0982e  // ummla z14.s, z1.b, z16.b\n"
      "ld1b { z17.b }, p5/Z, [x10, #6, MUL VL]\n"
      "ld1b { z16.b }, p5/Z, [x10, #7, MUL VL]\n"
      ".inst 0x45d1982b  // ummla z11.s, z1.b, z17.b\n"
      ".inst 0x45d0982f  // ummla z15.s, z1.b, z16.b\n"
      "addvl x10, x10, #8\n"
      "21:"  // Height 2: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 16b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x20, x9, x20, LSL #2\n"
      "uzp1 z16.d, z8.d, z12.d\n"
      "uzp2 z8.d, z8.d, z12.d\n"
      "uzp1 z17.d, z9.d, z13.d\n"
      "uzp2 z9.d, z9.d, z13.d\n"
      "st1w { z16.s }, p4, [x9]\n"
      "uzp1 z16.d, z10.d, z14.d\n"
      "uzp2 z10.d, z10.d, z14.d\n"
      "st1w { z17.s }, p3, [x9, #1, MUL VL]\n"
      "uzp1 z2.d, z11.d, z15.d\n"
      "uzp2 z11.d, z11.d, z15.d\n"
      "st1w { z16.s }, p2, [x9, #2, MUL VL]\n"
      "st1w { z2.s }, p1, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "st1w { z8.s }, p4, [x20]\n"
      "st1w { z9.s }, p3, [x20, #1, MUL VL]\n"
      "st1w { z10.s }, p2, [x20, #2, MUL VL]\n"
      "st1w { z11.s }, p1, [x20, #3, MUL VL]\n"
      "22:"  // Height 2: Writeback done
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 13b\n"
      "b 68f\n"
      "23:"  // Height 3
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "24:"  // Height 3: Column loop
      "mov x20, #0x0\n"
      "whilelt p4.s, x20, x11\n"
      "incw x20\n"
      "whilelt p3.s, x20, x11\n"
      "incw x20\n"
      "whilelt p2.s, x20, x11\n"
      "incw x20\n"
      "whilelt p1.s, x20, x11\n"
      "tbz %x[flags], #0, 25f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x21, x9, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "ld1w { z19.s }, p4/Z, [x9]\n"
      "ld1w { z17.s }, p3/Z, [x9, #1, MUL VL]\n"
      "ld1w { z18.s }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1w { z16.s }, p1/Z, [x9, #3, MUL VL]\n"
      "ld1w { z12.s }, p4/Z, [x21]\n"
      "zip1 z8.d, z19.d, z12.d\n"
      "zip2 z12.d, z19.d, z12.d\n"
      "ld1w { z13.s }, p3/Z, [x21, #1, MUL VL]\n"
      "ld1w { z14.s }, p2/Z, [x21, #2, MUL VL]\n"
      "zip1 z9.d, z17.d, z13.d\n"
      "zip2 z13.d, z17.d, z13.d\n"
      "ld1w { z15.s }, p1/Z, [x21, #3, MUL VL]\n"
      "ld1w { z17.s }, p4/Z, [x20]\n"
      "zip1 z10.d, z18.d, z14.d\n"
      "zip2 z14.d, z18.d, z14.d\n"
      "ld1w { z18.s }, p3/Z, [x20, #1, MUL VL]\n"
      "ld1w { z19.s }, p2/Z, [x20, #2, MUL VL]\n"
      "zip1 z11.d, z16.d, z15.d\n"
      "zip2 z15.d, z16.d, z15.d\n"
      "ld1w { z24.s }, p1/Z, [x20, #3, MUL VL]\n"
      "zip1 z16.d, z17.d, z20.d\n"
      "zip2 z20.d, z17.d, z20.d\n"
      "zip1 z17.d, z18.d, z21.d\n"
      "zip2 z21.d, z18.d, z21.d\n"
      "zip1 z18.d, z19.d, z22.d\n"
      "zip2 z22.d, z19.d, z22.d\n"
      "zip1 z19.d, z24.d, z23.d\n"
      "zip2 z23.d, z24.d, z23.d\n"
      "b 26f\n"
      "25:"  // Height 3: no accumulate
      "mov z8.s, #0x0\n"
      "mov z9.s, #0x0\n"
      "mov z10.s, #0x0\n"
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "mov z13.s, #0x0\n"
      "mov z14.s, #0x0\n"
      "mov z15.s, #0x0\n"
      "mov z16.s, #0x0\n"
      "mov z17.s, #0x0\n"
      "mov z18.s, #0x0\n"
      "mov z19.s, #0x0\n"
      "mov z20.s, #0x0\n"
      "mov z21.s, #0x0\n"
      "mov z22.s, #0x0\n"
      "mov z23.s, #0x0\n"
      "26:"  // Height 3: setup done
      "mov x28, #0x0\n"
      "27:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 28f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "cbnz x28, 29f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "b 29f\n"
      "28:"  // Height 3: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "29:"  // Height 3: input setup done
      "cmp x27, #0x10\n"
      "ble 31f\n"
      "30:"  // Height 3: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x27\n"
      "ld1rqb { z30.b }, p0/Z, [x26]\n"
      "ld1rqb { z24.b }, p0/Z, [x25]\n"
      "ld1rqb { z28.b }, p0/Z, [x24]\n"
      "trn1 z27.d, z30.d, z24.d\n"
      "trn2 z30.d, z30.d, z24.d\n"
      "ld1b { z25.b }, p5/Z, [x10]\n"
      "trn1 z26.d, z28.d, z29.d\n"
      "ld1b { z24.b }, p5/Z, [x10, #1, MUL VL]\n"
      ".inst 0x45d99b68  // ummla z8.s, z27.b, z25.b\n"
      ".inst 0x45d99b50  // ummla z16.s, z26.b, z25.b\n"
      ".inst 0x45d89b6c  // ummla z12.s, z27.b, z24.b\n"
      ".inst 0x45d89b54  // ummla z20.s, z26.b, z24.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1b { z24.b }, p5/Z, [x10, #3, MUL VL]\n"
      ".inst 0x45d99b69  // ummla z9.s, z27.b, z25.b\n"
      ".inst 0x45d99b51  // ummla z17.s, z26.b, z25.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #4, MUL VL]\n"
      "trn2 z28.d, z28.d, z29.d\n"
      ".inst 0x45d89b6d  // ummla z13.s, z27.b, z24.b\n"
      ".inst 0x45d89b55  // ummla z21.s, z26.b, z24.b\n"
      "ld1b { z24.b }, p5/Z, [x10, #5, MUL VL]\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x45d99b6a  // ummla z10.s, z27.b, z25.b\n"
      ".inst 0x45d99b52  // ummla z18.s, z26.b, z25.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #6, MUL VL]\n"
      "cmp x27, #0x10\n"
      ".inst 0x45d89b6e  // ummla z14.s, z27.b, z24.b\n"
      ".inst 0x45d89b56  // ummla z22.s, z26.b, z24.b\n"
      "ld1b { z24.b }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      ".inst 0x45d99b6b  // ummla z11.s, z27.b, z25.b\n"
      ".inst 0x45d99b53  // ummla z19.s, z26.b, z25.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #-8, MUL VL]\n"
      "add x26, x26, #0x10\n"
      ".inst 0x45d89b6f  // ummla z15.s, z27.b, z24.b\n"
      ".inst 0x45d89b57  // ummla z23.s, z26.b, z24.b\n"
      "ld1b { z24.b }, p5/Z, [x10, #-7, MUL VL]\n"
      "add x25, x25, #0x10\n"
      ".inst 0x45d99bc8  // ummla z8.s, z30.b, z25.b\n"
      ".inst 0x45d99b90  // ummla z16.s, z28.b, z25.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #-6, MUL VL]\n"
      "add x24, x24, #0x10\n"
      ".inst 0x45d89bcc  // ummla z12.s, z30.b, z24.b\n"
      ".inst 0x45d89b94  // ummla z20.s, z28.b, z24.b\n"
      "ld1b { z24.b }, p5/Z, [x10, #-5, MUL VL]\n"
      ".inst 0x45d99bc9  // ummla z9.s, z30.b, z25.b\n"
      ".inst 0x45d99b91  // ummla z17.s, z28.b, z25.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #-4, MUL VL]\n"
      ".inst 0x45d89bcd  // ummla z13.s, z30.b, z24.b\n"
      ".inst 0x45d89b95  // ummla z21.s, z28.b, z24.b\n"
      "ld1b { z24.b }, p5/Z, [x10, #-3, MUL VL]\n"
      ".inst 0x45d99bca  // ummla z10.s, z30.b, z25.b\n"
      ".inst 0x45d99b92  // ummla z18.s, z28.b, z25.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #-2, MUL VL]\n"
      ".inst 0x45d89bce  // ummla z14.s, z30.b, z24.b\n"
      ".inst 0x45d89b96  // ummla z22.s, z28.b, z24.b\n"
      "ld1b { z24.b }, p5/Z, [x10, #-1, MUL VL]\n"
      ".inst 0x45d99bcb  // ummla z11.s, z30.b, z25.b\n"
      ".inst 0x45d99b93  // ummla z19.s, z28.b, z25.b\n"
      ".inst 0x45d89bcf  // ummla z15.s, z30.b, z24.b\n"
      ".inst 0x45d89b97  // ummla z23.s, z28.b, z24.b\n"
      "bgt 30b\n"
      "31:"  // Height 3: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x27\n"
      "ld1rqb { z1.b }, p0/Z, [x26]\n"
      "ld1rqb { z24.b }, p0/Z, [x25]\n"
      "ld1rqb { z3.b }, p0/Z, [x24]\n"
      "trn1 z27.d, z1.d, z24.d\n"
      "trn2 z1.d, z1.d, z24.d\n"
      "ld1b { z25.b }, p5/Z, [x10]\n"
      "trn1 z26.d, z3.d, z28.d\n"
      "ld1b { z24.b }, p5/Z, [x10, #1, MUL VL]\n"
      ".inst 0x45d99b68  // ummla z8.s, z27.b, z25.b\n"
      ".inst 0x45d99b50  // ummla z16.s, z26.b, z25.b\n"
      ".inst 0x45d89b6c  // ummla z12.s, z27.b, z24.b\n"
      ".inst 0x45d89b54  // ummla z20.s, z26.b, z24.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1b { z24.b }, p5/Z, [x10, #3, MUL VL]\n"
      ".inst 0x45d99b69  // ummla z9.s, z27.b, z25.b\n"
      ".inst 0x45d99b51  // ummla z17.s, z26.b, z25.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #4, MUL VL]\n"
      "subs x27, x27, #0x8\n"
      ".inst 0x45d89b6d  // ummla z13.s, z27.b, z24.b\n"
      ".inst 0x45d89b55  // ummla z21.s, z26.b, z24.b\n"
      "ld1b { z24.b }, p5/Z, [x10, #5, MUL VL]\n"
      "trn2 z3.d, z3.d, z28.d\n"
      ".inst 0x45d99b6a  // ummla z10.s, z27.b, z25.b\n"
      ".inst 0x45d99b52  // ummla z18.s, z26.b, z25.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #6, MUL VL]\n"
      ".inst 0x45d89b6e  // ummla z14.s, z27.b, z24.b\n"
      ".inst 0x45d89b56  // ummla z22.s, z26.b, z24.b\n"
      "ld1b { z24.b }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #8\n"
      ".inst 0x45d99b6b  // ummla z11.s, z27.b, z25.b\n"
      ".inst 0x45d99b53  // ummla z19.s, z26.b, z25.b\n"
      ".inst 0x45d89b6f  // ummla z15.s, z27.b, z24.b\n"
      ".inst 0x45d89b57  // ummla z23.s, z26.b, z24.b\n"
      "ble 32f\n"
      "ld1b { z25.b }, p5/Z, [x10]\n"
      "ld1b { z24.b }, p5/Z, [x10, #1, MUL VL]\n"
      ".inst 0x45d99828  // ummla z8.s, z1.b, z25.b\n"
      ".inst 0x45d99870  // ummla z16.s, z3.b, z25.b\n"
      ".inst 0x45d8982c  // ummla z12.s, z1.b, z24.b\n"
      ".inst 0x45d89874  // ummla z20.s, z3.b, z24.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1b { z24.b }, p5/Z, [x10, #3, MUL VL]\n"
      ".inst 0x45d99829  // ummla z9.s, z1.b, z25.b\n"
      ".inst 0x45d99871  // ummla z17.s, z3.b, z25.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #4, MUL VL]\n"
      ".inst 0x45d8982d  // ummla z13.s, z1.b, z24.b\n"
      ".inst 0x45d89875  // ummla z21.s, z3.b, z24.b\n"
      "ld1b { z24.b }, p5/Z, [x10, #5, MUL VL]\n"
      ".inst 0x45d9982a  // ummla z10.s, z1.b, z25.b\n"
      ".inst 0x45d99872  // ummla z18.s, z3.b, z25.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #6, MUL VL]\n"
      ".inst 0x45d8982e  // ummla z14.s, z1.b, z24.b\n"
      ".inst 0x45d89876  // ummla z22.s, z3.b, z24.b\n"
      "ld1b { z24.b }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #8\n"
      ".inst 0x45d9982b  // ummla z11.s, z1.b, z25.b\n"
      ".inst 0x45d99873  // ummla z19.s, z3.b, z25.b\n"
      ".inst 0x45d8982f  // ummla z15.s, z1.b, z24.b\n"
      ".inst 0x45d89877  // ummla z23.s, z3.b, z24.b\n"
      "32:"  // Height 3: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 27b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x21, x9, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "uzp1 z25.d, z8.d, z12.d\n"
      "uzp2 z8.d, z8.d, z12.d\n"
      "uzp1 z24.d, z9.d, z13.d\n"
      "st1w { z25.s }, p4, [x9]\n"
      "uzp2 z9.d, z9.d, z13.d\n"
      "uzp1 z25.d, z10.d, z14.d\n"
      "st1w { z24.s }, p3, [x9, #1, MUL VL]\n"
      "uzp2 z10.d, z10.d, z14.d\n"
      "uzp1 z24.d, z11.d, z15.d\n"
      "st1w { z25.s }, p2, [x9, #2, MUL VL]\n"
      "uzp2 z11.d, z11.d, z15.d\n"
      "uzp1 z16.d, z16.d, z20.d\n"
      "st1w { z24.s }, p1, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "uzp1 z17.d, z17.d, z21.d\n"
      "uzp1 z18.d, z18.d, z22.d\n"
      "st1w { z8.s }, p4, [x21]\n"
      "uzp1 z19.d, z19.d, z23.d\n"
      "st1w { z9.s }, p3, [x21, #1, MUL VL]\n"
      "st1w { z10.s }, p2, [x21, #2, MUL VL]\n"
      "st1w { z11.s }, p1, [x21, #3, MUL VL]\n"
      "st1w { z16.s }, p4, [x20]\n"
      "st1w { z17.s }, p3, [x20, #1, MUL VL]\n"
      "st1w { z18.s }, p2, [x20, #2, MUL VL]\n"
      "st1w { z19.s }, p1, [x20, #3, MUL VL]\n"
      "33:"  // Height 3: Writeback done
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 24b\n"
      "b 68f\n"
      "34:"  // Height 4
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "35:"  // Height 4: Column loop
      "mov x20, #0x0\n"
      "whilelt p4.s, x20, x11\n"
      "incw x20\n"
      "whilelt p3.s, x20, x11\n"
      "incw x20\n"
      "whilelt p2.s, x20, x11\n"
      "incw x20\n"
      "whilelt p1.s, x20, x11\n"
      "tbz %x[flags], #0, 36f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x22, x9, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "ld1w { z19.s }, p4/Z, [x9]\n"
      "add x20, x21, x20, LSL #2\n"
      "ld1w { z17.s }, p3/Z, [x9, #1, MUL VL]\n"
      "ld1w { z18.s }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1w { z16.s }, p1/Z, [x9, #3, MUL VL]\n"
      "ld1w { z12.s }, p4/Z, [x22]\n"
      "zip1 z8.d, z19.d, z12.d\n"
      "zip2 z12.d, z19.d, z12.d\n"
      "ld1w { z13.s }, p3/Z, [x22, #1, MUL VL]\n"
      "ld1w { z14.s }, p2/Z, [x22, #2, MUL VL]\n"
      "zip1 z9.d, z17.d, z13.d\n"
      "zip2 z13.d, z17.d, z13.d\n"
      "ld1w { z15.s }, p1/Z, [x22, #3, MUL VL]\n"
      "ld1w { z17.s }, p4/Z, [x21]\n"
      "zip1 z10.d, z18.d, z14.d\n"
      "zip2 z14.d, z18.d, z14.d\n"
      "ld1w { z18.s }, p3/Z, [x21, #1, MUL VL]\n"
      "ld1w { z19.s }, p2/Z, [x21, #2, MUL VL]\n"
      "zip1 z11.d, z16.d, z15.d\n"
      "zip2 z15.d, z16.d, z15.d\n"
      "ld1w { z24.s }, p1/Z, [x21, #3, MUL VL]\n"
      "ld1w { z20.s }, p4/Z, [x20]\n"
      "zip1 z16.d, z17.d, z20.d\n"
      "zip2 z20.d, z17.d, z20.d\n"
      "ld1w { z21.s }, p3/Z, [x20, #1, MUL VL]\n"
      "ld1w { z22.s }, p2/Z, [x20, #2, MUL VL]\n"
      "zip1 z17.d, z18.d, z21.d\n"
      "zip2 z21.d, z18.d, z21.d\n"
      "ld1w { z23.s }, p1/Z, [x20, #3, MUL VL]\n"
      "zip1 z18.d, z19.d, z22.d\n"
      "zip2 z22.d, z19.d, z22.d\n"
      "zip1 z19.d, z24.d, z23.d\n"
      "zip2 z23.d, z24.d, z23.d\n"
      "b 37f\n"
      "36:"  // Height 4: no accumulate
      "mov z8.s, #0x0\n"
      "mov z9.s, #0x0\n"
      "mov z10.s, #0x0\n"
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "mov z13.s, #0x0\n"
      "mov z14.s, #0x0\n"
      "mov z15.s, #0x0\n"
      "mov z16.s, #0x0\n"
      "mov z17.s, #0x0\n"
      "mov z18.s, #0x0\n"
      "mov z19.s, #0x0\n"
      "mov z20.s, #0x0\n"
      "mov z21.s, #0x0\n"
      "mov z22.s, #0x0\n"
      "mov z23.s, #0x0\n"
      "37:"  // Height 4: setup done
      "mov x28, #0x0\n"
      "38:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 39f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "cbnz x28, 40f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "b 40f\n"
      "39:"  // Height 4: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "40:"  // Height 4: input setup done
      "cmp x27, #0x10\n"
      "ble 42f\n"
      "41:"  // Height 4: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x27\n"
      "ld1rqb { z30.b }, p0/Z, [x26]\n"
      "ld1rqb { z24.b }, p0/Z, [x25]\n"
      "trn1 z29.d, z30.d, z24.d\n"
      "ld1rqb { z28.b }, p0/Z, [x24]\n"
      "ld1rqb { z27.b }, p0/Z, [x23]\n"
      "trn2 z30.d, z30.d, z24.d\n"
      "trn1 z26.d, z28.d, z27.d\n"
      "ld1b { z25.b }, p5/Z, [x10]\n"
      "ld1b { z24.b }, p5/Z, [x10, #1, MUL VL]\n"
      ".inst 0x45d99ba8  // ummla z8.s, z29.b, z25.b\n"
      ".inst 0x45d99b50  // ummla z16.s, z26.b, z25.b\n"
      ".inst 0x45d89bac  // ummla z12.s, z29.b, z24.b\n"
      ".inst 0x45d89b54  // ummla z20.s, z26.b, z24.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1b { z24.b }, p5/Z, [x10, #3, MUL VL]\n"
      ".inst 0x45d99ba9  // ummla z9.s, z29.b, z25.b\n"
      ".inst 0x45d99b51  // ummla z17.s, z26.b, z25.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #4, MUL VL]\n"
      "trn2 z28.d, z28.d, z27.d\n"
      ".inst 0x45d89bad  // ummla z13.s, z29.b, z24.b\n"
      ".inst 0x45d89b55  // ummla z21.s, z26.b, z24.b\n"
      "ld1b { z24.b }, p5/Z, [x10, #5, MUL VL]\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x45d99baa  // ummla z10.s, z29.b, z25.b\n"
      ".inst 0x45d99b52  // ummla z18.s, z26.b, z25.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #6, MUL VL]\n"
      "cmp x27, #0x10\n"
      ".inst 0x45d89bae  // ummla z14.s, z29.b, z24.b\n"
      ".inst 0x45d89b56  // ummla z22.s, z26.b, z24.b\n"
      "ld1b { z24.b }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      ".inst 0x45d99bab  // ummla z11.s, z29.b, z25.b\n"
      ".inst 0x45d99b53  // ummla z19.s, z26.b, z25.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #-8, MUL VL]\n"
      "add x26, x26, #0x10\n"
      ".inst 0x45d89baf  // ummla z15.s, z29.b, z24.b\n"
      ".inst 0x45d89b57  // ummla z23.s, z26.b, z24.b\n"
      "ld1b { z24.b }, p5/Z, [x10, #-7, MUL VL]\n"
      "add x25, x25, #0x10\n"
      ".inst 0x45d99bc8  // ummla z8.s, z30.b, z25.b\n"
      ".inst 0x45d99b90  // ummla z16.s, z28.b, z25.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #-6, MUL VL]\n"
      "add x24, x24, #0x10\n"
      ".inst 0x45d89bcc  // ummla z12.s, z30.b, z24.b\n"
      ".inst 0x45d89b94  // ummla z20.s, z28.b, z24.b\n"
      "ld1b { z24.b }, p5/Z, [x10, #-5, MUL VL]\n"
      "add x23, x23, #0x10\n"
      ".inst 0x45d99bc9  // ummla z9.s, z30.b, z25.b\n"
      ".inst 0x45d99b91  // ummla z17.s, z28.b, z25.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #-4, MUL VL]\n"
      ".inst 0x45d89bcd  // ummla z13.s, z30.b, z24.b\n"
      ".inst 0x45d89b95  // ummla z21.s, z28.b, z24.b\n"
      "ld1b { z24.b }, p5/Z, [x10, #-3, MUL VL]\n"
      ".inst 0x45d99bca  // ummla z10.s, z30.b, z25.b\n"
      ".inst 0x45d99b92  // ummla z18.s, z28.b, z25.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #-2, MUL VL]\n"
      ".inst 0x45d89bce  // ummla z14.s, z30.b, z24.b\n"
      ".inst 0x45d89b96  // ummla z22.s, z28.b, z24.b\n"
      "ld1b { z24.b }, p5/Z, [x10, #-1, MUL VL]\n"
      ".inst 0x45d99bcb  // ummla z11.s, z30.b, z25.b\n"
      ".inst 0x45d99b93  // ummla z19.s, z28.b, z25.b\n"
      ".inst 0x45d89bcf  // ummla z15.s, z30.b, z24.b\n"
      ".inst 0x45d89b97  // ummla z23.s, z28.b, z24.b\n"
      "bgt 41b\n"
      "42:"  // Height 4: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x27\n"
      "ld1rqb { z1.b }, p0/Z, [x26]\n"
      "ld1rqb { z24.b }, p0/Z, [x25]\n"
      "trn1 z28.d, z1.d, z24.d\n"
      "ld1rqb { z3.b }, p0/Z, [x24]\n"
      "ld1rqb { z27.b }, p0/Z, [x23]\n"
      "trn2 z1.d, z1.d, z24.d\n"
      "trn1 z26.d, z3.d, z27.d\n"
      "ld1b { z25.b }, p5/Z, [x10]\n"
      "ld1b { z24.b }, p5/Z, [x10, #1, MUL VL]\n"
      ".inst 0x45d99b88  // ummla z8.s, z28.b, z25.b\n"
      ".inst 0x45d99b50  // ummla z16.s, z26.b, z25.b\n"
      ".inst 0x45d89b8c  // ummla z12.s, z28.b, z24.b\n"
      ".inst 0x45d89b54  // ummla z20.s, z26.b, z24.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1b { z24.b }, p5/Z, [x10, #3, MUL VL]\n"
      ".inst 0x45d99b89  // ummla z9.s, z28.b, z25.b\n"
      ".inst 0x45d99b51  // ummla z17.s, z26.b, z25.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #4, MUL VL]\n"
      "subs x27, x27, #0x8\n"
      ".inst 0x45d89b8d  // ummla z13.s, z28.b, z24.b\n"
      ".inst 0x45d89b55  // ummla z21.s, z26.b, z24.b\n"
      "ld1b { z24.b }, p5/Z, [x10, #5, MUL VL]\n"
      "trn2 z3.d, z3.d, z27.d\n"
      ".inst 0x45d99b8a  // ummla z10.s, z28.b, z25.b\n"
      ".inst 0x45d99b52  // ummla z18.s, z26.b, z25.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #6, MUL VL]\n"
      ".inst 0x45d89b8e  // ummla z14.s, z28.b, z24.b\n"
      ".inst 0x45d89b56  // ummla z22.s, z26.b, z24.b\n"
      "ld1b { z24.b }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #8\n"
      ".inst 0x45d99b8b  // ummla z11.s, z28.b, z25.b\n"
      ".inst 0x45d99b53  // ummla z19.s, z26.b, z25.b\n"
      ".inst 0x45d89b8f  // ummla z15.s, z28.b, z24.b\n"
      ".inst 0x45d89b57  // ummla z23.s, z26.b, z24.b\n"
      "ble 43f\n"
      "ld1b { z25.b }, p5/Z, [x10]\n"
      "ld1b { z24.b }, p5/Z, [x10, #1, MUL VL]\n"
      ".inst 0x45d99828  // ummla z8.s, z1.b, z25.b\n"
      ".inst 0x45d99870  // ummla z16.s, z3.b, z25.b\n"
      ".inst 0x45d8982c  // ummla z12.s, z1.b, z24.b\n"
      ".inst 0x45d89874  // ummla z20.s, z3.b, z24.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1b { z24.b }, p5/Z, [x10, #3, MUL VL]\n"
      ".inst 0x45d99829  // ummla z9.s, z1.b, z25.b\n"
      ".inst 0x45d99871  // ummla z17.s, z3.b, z25.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #4, MUL VL]\n"
      ".inst 0x45d8982d  // ummla z13.s, z1.b, z24.b\n"
      ".inst 0x45d89875  // ummla z21.s, z3.b, z24.b\n"
      "ld1b { z24.b }, p5/Z, [x10, #5, MUL VL]\n"
      ".inst 0x45d9982a  // ummla z10.s, z1.b, z25.b\n"
      ".inst 0x45d99872  // ummla z18.s, z3.b, z25.b\n"
      "ld1b { z25.b }, p5/Z, [x10, #6, MUL VL]\n"
      ".inst 0x45d8982e  // ummla z14.s, z1.b, z24.b\n"
      ".inst 0x45d89876  // ummla z22.s, z3.b, z24.b\n"
      "ld1b { z24.b }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #8\n"
      ".inst 0x45d9982b  // ummla z11.s, z1.b, z25.b\n"
      ".inst 0x45d99873  // ummla z19.s, z3.b, z25.b\n"
      ".inst 0x45d8982f  // ummla z15.s, z1.b, z24.b\n"
      ".inst 0x45d89877  // ummla z23.s, z3.b, z24.b\n"
      "43:"  // Height 4: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 38b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x22, x9, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "uzp1 z25.d, z8.d, z12.d\n"
      "add x20, x21, x20, LSL #2\n"
      "uzp2 z8.d, z8.d, z12.d\n"
      "uzp1 z24.d, z9.d, z13.d\n"
      "st1w { z25.s }, p4, [x9]\n"
      "uzp2 z9.d, z9.d, z13.d\n"
      "uzp1 z25.d, z10.d, z14.d\n"
      "st1w { z24.s }, p3, [x9, #1, MUL VL]\n"
      "uzp2 z10.d, z10.d, z14.d\n"
      "uzp1 z24.d, z11.d, z15.d\n"
      "st1w { z25.s }, p2, [x9, #2, MUL VL]\n"
      "uzp2 z11.d, z11.d, z15.d\n"
      "uzp1 z25.d, z16.d, z20.d\n"
      "st1w { z24.s }, p1, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "uzp2 z16.d, z16.d, z20.d\n"
      "uzp1 z24.d, z17.d, z21.d\n"
      "st1w { z8.s }, p4, [x22]\n"
      "uzp2 z17.d, z17.d, z21.d\n"
      "uzp1 z21.d, z18.d, z22.d\n"
      "st1w { z9.s }, p3, [x22, #1, MUL VL]\n"
      "uzp2 z18.d, z18.d, z22.d\n"
      "uzp1 z20.d, z19.d, z23.d\n"
      "st1w { z10.s }, p2, [x22, #2, MUL VL]\n"
      "uzp2 z19.d, z19.d, z23.d\n"
      "st1w { z11.s }, p1, [x22, #3, MUL VL]\n"
      "st1w { z25.s }, p4, [x21]\n"
      "st1w { z24.s }, p3, [x21, #1, MUL VL]\n"
      "st1w { z21.s }, p2, [x21, #2, MUL VL]\n"
      "st1w { z20.s }, p1, [x21, #3, MUL VL]\n"
      "st1w { z16.s }, p4, [x20]\n"
      "st1w { z17.s }, p3, [x20, #1, MUL VL]\n"
      "st1w { z18.s }, p2, [x20, #2, MUL VL]\n"
      "st1w { z19.s }, p1, [x20, #3, MUL VL]\n"
      "44:"  // Height 4: Writeback done
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 35b\n"
      "b 68f\n"
      "45:"  // Height 5
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "46:"  // Height 5: Column loop
      "mov x20, #0x0\n"
      "whilelt p4.s, x20, x11\n"
      "incw x20\n"
      "whilelt p3.s, x20, x11\n"
      "incw x20\n"
      "whilelt p2.s, x20, x11\n"
      "incw x20\n"
      "whilelt p1.s, x20, x11\n"
      "tbz %x[flags], #0, 47f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x23, x9, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "ld1w { z19.s }, p4/Z, [x9]\n"
      "add x21, x22, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "ld1w { z17.s }, p3/Z, [x9, #1, MUL VL]\n"
      "ld1w { z18.s }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1w { z16.s }, p1/Z, [x9, #3, MUL VL]\n"
      "ld1w { z12.s }, p4/Z, [x23]\n"
      "zip1 z8.d, z19.d, z12.d\n"
      "zip2 z12.d, z19.d, z12.d\n"
      "ld1w { z13.s }, p3/Z, [x23, #1, MUL VL]\n"
      "ld1w { z14.s }, p2/Z, [x23, #2, MUL VL]\n"
      "zip1 z9.d, z17.d, z13.d\n"
      "zip2 z13.d, z17.d, z13.d\n"
      "ld1w { z15.s }, p1/Z, [x23, #3, MUL VL]\n"
      "ld1w { z17.s }, p4/Z, [x22]\n"
      "zip1 z10.d, z18.d, z14.d\n"
      "zip2 z14.d, z18.d, z14.d\n"
      "ld1w { z18.s }, p3/Z, [x22, #1, MUL VL]\n"
      "ld1w { z19.s }, p2/Z, [x22, #2, MUL VL]\n"
      "zip1 z11.d, z16.d, z15.d\n"
      "zip2 z15.d, z16.d, z15.d\n"
      "ld1w { z24.s }, p1/Z, [x22, #3, MUL VL]\n"
      "ld1w { z20.s }, p4/Z, [x21]\n"
      "zip1 z16.d, z17.d, z20.d\n"
      "zip2 z20.d, z17.d, z20.d\n"
      "ld1w { z21.s }, p3/Z, [x21, #1, MUL VL]\n"
      "ld1w { z22.s }, p2/Z, [x21, #2, MUL VL]\n"
      "zip1 z17.d, z18.d, z21.d\n"
      "zip2 z21.d, z18.d, z21.d\n"
      "ld1w { z23.s }, p1/Z, [x21, #3, MUL VL]\n"
      "ld1w { z25.s }, p4/Z, [x20]\n"
      "zip1 z18.d, z19.d, z22.d\n"
      "zip2 z22.d, z19.d, z22.d\n"
      "ld1w { z26.s }, p3/Z, [x20, #1, MUL VL]\n"
      "ld1w { z27.s }, p2/Z, [x20, #2, MUL VL]\n"
      "zip1 z19.d, z24.d, z23.d\n"
      "zip2 z23.d, z24.d, z23.d\n"
      "ld1w { z0.s }, p1/Z, [x20, #3, MUL VL]\n"
      "zip1 z24.d, z25.d, z28.d\n"
      "zip2 z28.d, z25.d, z28.d\n"
      "zip1 z25.d, z26.d, z29.d\n"
      "zip2 z29.d, z26.d, z29.d\n"
      "zip1 z26.d, z27.d, z30.d\n"
      "zip2 z30.d, z27.d, z30.d\n"
      "zip1 z27.d, z0.d, z31.d\n"
      "zip2 z31.d, z0.d, z31.d\n"
      "b 48f\n"
      "47:"  // Height 5: no accumulate
      "mov z8.s, #0x0\n"
      "mov z9.s, #0x0\n"
      "mov z10.s, #0x0\n"
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "mov z13.s, #0x0\n"
      "mov z14.s, #0x0\n"
      "mov z15.s, #0x0\n"
      "mov z16.s, #0x0\n"
      "mov z17.s, #0x0\n"
      "mov z18.s, #0x0\n"
      "mov z19.s, #0x0\n"
      "mov z20.s, #0x0\n"
      "mov z21.s, #0x0\n"
      "mov z22.s, #0x0\n"
      "mov z23.s, #0x0\n"
      "mov z24.s, #0x0\n"
      "mov z25.s, #0x0\n"
      "mov z26.s, #0x0\n"
      "mov z27.s, #0x0\n"
      "mov z28.s, #0x0\n"
      "mov z29.s, #0x0\n"
      "mov z30.s, #0x0\n"
      "mov z31.s, #0x0\n"
      "48:"  // Height 5: setup done
      "mov x28, #0x0\n"
      "49:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 50f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "cbnz x28, 51f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "b 51f\n"
      "50:"  // Height 5: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "51:"  // Height 5: input setup done
      "cmp x27, #0x10\n"
      "ble 53f\n"
      "52:"  // Height 5: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x27\n"
      "ld1rqb { z6.b }, p0/Z, [x26]\n"
      "ld1rqb { z1.b }, p0/Z, [x25]\n"
      "ld1rqb { z7.b }, p0/Z, [x24]\n"
      "ld1rqb { z2.b }, p0/Z, [x23]\n"
      "trn1 z5.d, z6.d, z1.d\n"
      "trn2 z6.d, z6.d, z1.d\n"
      "ld1rqb { z4.b }, p0/Z, [x22]\n"
      "trn1 z3.d, z7.d, z2.d\n"
      "trn2 z7.d, z7.d, z2.d\n"
      "ld1b { z1.b }, p5/Z, [x10]\n"
      "trn1 z2.d, z4.d, z0.d\n"
      "trn2 z4.d, z4.d, z0.d\n"
      "ld1b { z0.b }, p5/Z, [x10, #1, MUL VL]\n"
      ".inst 0x45c198a8  // ummla z8.s, z5.b, z1.b\n"
      ".inst 0x45c19870  // ummla z16.s, z3.b, z1.b\n"
      ".inst 0x45c19858  // ummla z24.s, z2.b, z1.b\n"
      "ld1b { z1.b }, p5/Z, [x10, #2, MUL VL]\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x45c098ac  // ummla z12.s, z5.b, z0.b\n"
      ".inst 0x45c09874  // ummla z20.s, z3.b, z0.b\n"
      "cmp x27, #0x10\n"
      "add x26, x26, #0x10\n"
      ".inst 0x45c0985c  // ummla z28.s, z2.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #3, MUL VL]\n"
      ".inst 0x45c198a9  // ummla z9.s, z5.b, z1.b\n"
      "add x25, x25, #0x10\n"
      ".inst 0x45c19871  // ummla z17.s, z3.b, z1.b\n"
      ".inst 0x45c19859  // ummla z25.s, z2.b, z1.b\n"
      "ld1b { z1.b }, p5/Z, [x10, #4, MUL VL]\n"
      "add x24, x24, #0x10\n"
      ".inst 0x45c098ad  // ummla z13.s, z5.b, z0.b\n"
      ".inst 0x45c09875  // ummla z21.s, z3.b, z0.b\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      ".inst 0x45c0985d  // ummla z29.s, z2.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #5, MUL VL]\n"
      ".inst 0x45c198aa  // ummla z10.s, z5.b, z1.b\n"
      ".inst 0x45c19872  // ummla z18.s, z3.b, z1.b\n"
      ".inst 0x45c1985a  // ummla z26.s, z2.b, z1.b\n"
      "ld1b { z1.b }, p5/Z, [x10, #6, MUL VL]\n"
      ".inst 0x45c098ae  // ummla z14.s, z5.b, z0.b\n"
      ".inst 0x45c09876  // ummla z22.s, z3.b, z0.b\n"
      ".inst 0x45c0985e  // ummla z30.s, z2.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      ".inst 0x45c198ab  // ummla z11.s, z5.b, z1.b\n"
      ".inst 0x45c19873  // ummla z19.s, z3.b, z1.b\n"
      ".inst 0x45c1985b  // ummla z27.s, z2.b, z1.b\n"
      "ld1b { z1.b }, p5/Z, [x10, #-8, MUL VL]\n"
      ".inst 0x45c098af  // ummla z15.s, z5.b, z0.b\n"
      ".inst 0x45c09877  // ummla z23.s, z3.b, z0.b\n"
      ".inst 0x45c0985f  // ummla z31.s, z2.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #-7, MUL VL]\n"
      ".inst 0x45c198c8  // ummla z8.s, z6.b, z1.b\n"
      ".inst 0x45c198f0  // ummla z16.s, z7.b, z1.b\n"
      ".inst 0x45c19898  // ummla z24.s, z4.b, z1.b\n"
      "ld1b { z1.b }, p5/Z, [x10, #-6, MUL VL]\n"
      ".inst 0x45c098cc  // ummla z12.s, z6.b, z0.b\n"
      ".inst 0x45c098f4  // ummla z20.s, z7.b, z0.b\n"
      ".inst 0x45c0989c  // ummla z28.s, z4.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #-5, MUL VL]\n"
      ".inst 0x45c198c9  // ummla z9.s, z6.b, z1.b\n"
      ".inst 0x45c198f1  // ummla z17.s, z7.b, z1.b\n"
      ".inst 0x45c19899  // ummla z25.s, z4.b, z1.b\n"
      "ld1b { z1.b }, p5/Z, [x10, #-4, MUL VL]\n"
      ".inst 0x45c098cd  // ummla z13.s, z6.b, z0.b\n"
      ".inst 0x45c098f5  // ummla z21.s, z7.b, z0.b\n"
      ".inst 0x45c0989d  // ummla z29.s, z4.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #-3, MUL VL]\n"
      ".inst 0x45c198ca  // ummla z10.s, z6.b, z1.b\n"
      ".inst 0x45c198f2  // ummla z18.s, z7.b, z1.b\n"
      ".inst 0x45c1989a  // ummla z26.s, z4.b, z1.b\n"
      "ld1b { z1.b }, p5/Z, [x10, #-2, MUL VL]\n"
      ".inst 0x45c098ce  // ummla z14.s, z6.b, z0.b\n"
      ".inst 0x45c098f6  // ummla z22.s, z7.b, z0.b\n"
      ".inst 0x45c0989e  // ummla z30.s, z4.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #-1, MUL VL]\n"
      ".inst 0x45c198cb  // ummla z11.s, z6.b, z1.b\n"
      ".inst 0x45c198f3  // ummla z19.s, z7.b, z1.b\n"
      ".inst 0x45c1989b  // ummla z27.s, z4.b, z1.b\n"
      ".inst 0x45c098cf  // ummla z15.s, z6.b, z0.b\n"
      ".inst 0x45c098f7  // ummla z23.s, z7.b, z0.b\n"
      ".inst 0x45c0989f  // ummla z31.s, z4.b, z0.b\n"
      "bgt 52b\n"
      "53:"  // Height 5: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x27\n"
      "ld1rqb { z1.b }, p0/Z, [x26]\n"
      "ld1rqb { z4.b }, p0/Z, [x25]\n"
      "ld1rqb { z3.b }, p0/Z, [x24]\n"
      "ld1rqb { z2.b }, p0/Z, [x23]\n"
      "trn1 z7.d, z1.d, z4.d\n"
      "trn2 z1.d, z1.d, z4.d\n"
      "ld1rqb { z5.b }, p0/Z, [x22]\n"
      "trn1 z6.d, z3.d, z2.d\n"
      "trn2 z3.d, z3.d, z2.d\n"
      "ld1b { z2.b }, p5/Z, [x10]\n"
      "trn1 z4.d, z5.d, z0.d\n"
      "trn2 z5.d, z5.d, z0.d\n"
      "ld1b { z0.b }, p5/Z, [x10, #1, MUL VL]\n"
      ".inst 0x45c298e8  // ummla z8.s, z7.b, z2.b\n"
      ".inst 0x45c298d0  // ummla z16.s, z6.b, z2.b\n"
      ".inst 0x45c29898  // ummla z24.s, z4.b, z2.b\n"
      "ld1b { z2.b }, p5/Z, [x10, #2, MUL VL]\n"
      "subs x27, x27, #0x8\n"
      ".inst 0x45c098ec  // ummla z12.s, z7.b, z0.b\n"
      ".inst 0x45c098d4  // ummla z20.s, z6.b, z0.b\n"
      ".inst 0x45c0989c  // ummla z28.s, z4.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #3, MUL VL]\n"
      ".inst 0x45c298e9  // ummla z9.s, z7.b, z2.b\n"
      ".inst 0x45c298d1  // ummla z17.s, z6.b, z2.b\n"
      ".inst 0x45c29899  // ummla z25.s, z4.b, z2.b\n"
      "ld1b { z2.b }, p5/Z, [x10, #4, MUL VL]\n"
      ".inst 0x45c098ed  // ummla z13.s, z7.b, z0.b\n"
      ".inst 0x45c098d5  // ummla z21.s, z6.b, z0.b\n"
      ".inst 0x45c0989d  // ummla z29.s, z4.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #5, MUL VL]\n"
      ".inst 0x45c298ea  // ummla z10.s, z7.b, z2.b\n"
      ".inst 0x45c298d2  // ummla z18.s, z6.b, z2.b\n"
      ".inst 0x45c2989a  // ummla z26.s, z4.b, z2.b\n"
      "ld1b { z2.b }, p5/Z, [x10, #6, MUL VL]\n"
      ".inst 0x45c098ee  // ummla z14.s, z7.b, z0.b\n"
      ".inst 0x45c098d6  // ummla z22.s, z6.b, z0.b\n"
      ".inst 0x45c0989e  // ummla z30.s, z4.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #7, MUL VL]\n"
      ".inst 0x45c298eb  // ummla z11.s, z7.b, z2.b\n"
      "addvl x10, x10, #8\n"
      ".inst 0x45c298d3  // ummla z19.s, z6.b, z2.b\n"
      ".inst 0x45c2989b  // ummla z27.s, z4.b, z2.b\n"
      ".inst 0x45c098ef  // ummla z15.s, z7.b, z0.b\n"
      ".inst 0x45c098d7  // ummla z23.s, z6.b, z0.b\n"
      ".inst 0x45c0989f  // ummla z31.s, z4.b, z0.b\n"
      "ble 54f\n"
      "ld1b { z2.b }, p5/Z, [x10]\n"
      "ld1b { z0.b }, p5/Z, [x10, #1, MUL VL]\n"
      ".inst 0x45c29828  // ummla z8.s, z1.b, z2.b\n"
      ".inst 0x45c29870  // ummla z16.s, z3.b, z2.b\n"
      ".inst 0x45c298b8  // ummla z24.s, z5.b, z2.b\n"
      ".inst 0x45c0982c  // ummla z12.s, z1.b, z0.b\n"
      "ld1b { z2.b }, p5/Z, [x10, #2, MUL VL]\n"
      ".inst 0x45c09874  // ummla z20.s, z3.b, z0.b\n"
      ".inst 0x45c098bc  // ummla z28.s, z5.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #3, MUL VL]\n"
      ".inst 0x45c29829  // ummla z9.s, z1.b, z2.b\n"
      ".inst 0x45c29871  // ummla z17.s, z3.b, z2.b\n"
      ".inst 0x45c298b9  // ummla z25.s, z5.b, z2.b\n"
      ".inst 0x45c0982d  // ummla z13.s, z1.b, z0.b\n"
      "ld1b { z2.b }, p5/Z, [x10, #4, MUL VL]\n"
      ".inst 0x45c09875  // ummla z21.s, z3.b, z0.b\n"
      ".inst 0x45c098bd  // ummla z29.s, z5.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #5, MUL VL]\n"
      ".inst 0x45c2982a  // ummla z10.s, z1.b, z2.b\n"
      ".inst 0x45c29872  // ummla z18.s, z3.b, z2.b\n"
      ".inst 0x45c298ba  // ummla z26.s, z5.b, z2.b\n"
      ".inst 0x45c0982e  // ummla z14.s, z1.b, z0.b\n"
      "ld1b { z2.b }, p5/Z, [x10, #6, MUL VL]\n"
      ".inst 0x45c09876  // ummla z22.s, z3.b, z0.b\n"
      ".inst 0x45c098be  // ummla z30.s, z5.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #8\n"
      ".inst 0x45c2982b  // ummla z11.s, z1.b, z2.b\n"
      ".inst 0x45c29873  // ummla z19.s, z3.b, z2.b\n"
      ".inst 0x45c298bb  // ummla z27.s, z5.b, z2.b\n"
      ".inst 0x45c0982f  // ummla z15.s, z1.b, z0.b\n"
      ".inst 0x45c09877  // ummla z23.s, z3.b, z0.b\n"
      ".inst 0x45c098bf  // ummla z31.s, z5.b, z0.b\n"
      "54:"  // Height 5: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 49b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x23, x9, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "uzp1 z2.d, z8.d, z12.d\n"
      "add x21, x22, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "uzp2 z8.d, z8.d, z12.d\n"
      "uzp1 z1.d, z9.d, z13.d\n"
      "uzp2 z9.d, z9.d, z13.d\n"
      "uzp1 z0.d, z10.d, z14.d\n"
      "st1w { z2.s }, p4, [x9]\n"
      "uzp2 z10.d, z10.d, z14.d\n"
      "uzp1 z2.d, z11.d, z15.d\n"
      "st1w { z1.s }, p3, [x9, #1, MUL VL]\n"
      "uzp2 z11.d, z11.d, z15.d\n"
      "uzp1 z1.d, z16.d, z20.d\n"
      "st1w { z0.s }, p2, [x9, #2, MUL VL]\n"
      "uzp2 z16.d, z16.d, z20.d\n"
      "uzp1 z0.d, z17.d, z21.d\n"
      "st1w { z2.s }, p1, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "uzp2 z17.d, z17.d, z21.d\n"
      "uzp1 z21.d, z18.d, z22.d\n"
      "st1w { z8.s }, p4, [x23]\n"
      "uzp2 z18.d, z18.d, z22.d\n"
      "uzp1 z20.d, z19.d, z23.d\n"
      "st1w { z9.s }, p3, [x23, #1, MUL VL]\n"
      "uzp2 z19.d, z19.d, z23.d\n"
      "uzp1 z24.d, z24.d, z28.d\n"
      "st1w { z10.s }, p2, [x23, #2, MUL VL]\n"
      "uzp1 z25.d, z25.d, z29.d\n"
      "uzp1 z26.d, z26.d, z30.d\n"
      "st1w { z11.s }, p1, [x23, #3, MUL VL]\n"
      "uzp1 z27.d, z27.d, z31.d\n"
      "st1w { z1.s }, p4, [x22]\n"
      "st1w { z0.s }, p3, [x22, #1, MUL VL]\n"
      "st1w { z21.s }, p2, [x22, #2, MUL VL]\n"
      "st1w { z20.s }, p1, [x22, #3, MUL VL]\n"
      "st1w { z16.s }, p4, [x21]\n"
      "st1w { z17.s }, p3, [x21, #1, MUL VL]\n"
      "st1w { z18.s }, p2, [x21, #2, MUL VL]\n"
      "st1w { z19.s }, p1, [x21, #3, MUL VL]\n"
      "st1w { z24.s }, p4, [x20]\n"
      "st1w { z25.s }, p3, [x20, #1, MUL VL]\n"
      "st1w { z26.s }, p2, [x20, #2, MUL VL]\n"
      "st1w { z27.s }, p1, [x20, #3, MUL VL]\n"
      "55:"  // Height 5: Writeback done
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 46b\n"
      "b 68f\n"
      "56:"  // Height 6
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "mov x20, #0x18\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "mov x9, %x[output_ptr]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "madd %x[output_ptr], x21, x20, %x[output_ptr]\n"
      "57:"  // Height 6: Column loop
      "mov x20, #0x0\n"
      "whilelt p4.s, x20, x11\n"
      "incw x20\n"
      "whilelt p3.s, x20, x11\n"
      "incw x20\n"
      "whilelt p2.s, x20, x11\n"
      "incw x20\n"
      "whilelt p1.s, x20, x11\n"
      "tbz %x[flags], #0, 58f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "ld1w { z17.s }, p4/Z, [x9]\n"
      "add x22, x23, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "ld1w { z18.s }, p3/Z, [x9, #1, MUL VL]\n"
      "ld1w { z20.s }, p2/Z, [x9, #2, MUL VL]\n"
      "add x20, x21, x20, LSL #2\n"
      "ld1w { z16.s }, p1/Z, [x9, #3, MUL VL]\n"
      "ld1w { z12.s }, p4/Z, [x24]\n"
      "zip1 z8.d, z17.d, z12.d\n"
      "ld1w { z13.s }, p3/Z, [x24, #1, MUL VL]\n"
      "ld1w { z14.s }, p2/Z, [x24, #2, MUL VL]\n"
      "zip2 z12.d, z17.d, z12.d\n"
      "zip1 z9.d, z18.d, z13.d\n"
      "ld1w { z15.s }, p1/Z, [x24, #3, MUL VL]\n"
      "ld1w { z17.s }, p4/Z, [x23]\n"
      "zip2 z13.d, z18.d, z13.d\n"
      "zip1 z10.d, z20.d, z14.d\n"
      "ld1w { z18.s }, p3/Z, [x23, #1, MUL VL]\n"
      "ld1w { z19.s }, p2/Z, [x23, #2, MUL VL]\n"
      "zip2 z14.d, z20.d, z14.d\n"
      "zip1 z11.d, z16.d, z15.d\n"
      "ld1w { z24.s }, p1/Z, [x23, #3, MUL VL]\n"
      "ld1w { z20.s }, p4/Z, [x22]\n"
      "zip2 z15.d, z16.d, z15.d\n"
      "zip1 z16.d, z17.d, z20.d\n"
      "ld1w { z21.s }, p3/Z, [x22, #1, MUL VL]\n"
      "ld1w { z22.s }, p2/Z, [x22, #2, MUL VL]\n"
      "zip2 z20.d, z17.d, z20.d\n"
      "zip1 z17.d, z18.d, z21.d\n"
      "ld1w { z23.s }, p1/Z, [x22, #3, MUL VL]\n"
      "ld1w { z25.s }, p4/Z, [x21]\n"
      "zip2 z21.d, z18.d, z21.d\n"
      "zip1 z18.d, z19.d, z22.d\n"
      "ld1w { z26.s }, p3/Z, [x21, #1, MUL VL]\n"
      "ld1w { z27.s }, p2/Z, [x21, #2, MUL VL]\n"
      "zip2 z22.d, z19.d, z22.d\n"
      "zip1 z19.d, z24.d, z23.d\n"
      "ld1w { z0.s }, p1/Z, [x21, #3, MUL VL]\n"
      "ld1w { z28.s }, p4/Z, [x20]\n"
      "zip2 z23.d, z24.d, z23.d\n"
      "zip1 z24.d, z25.d, z28.d\n"
      "ld1w { z29.s }, p3/Z, [x20, #1, MUL VL]\n"
      "ld1w { z30.s }, p2/Z, [x20, #2, MUL VL]\n"
      "zip2 z28.d, z25.d, z28.d\n"
      "zip1 z25.d, z26.d, z29.d\n"
      "ld1w { z31.s }, p1/Z, [x20, #3, MUL VL]\n"
      "zip2 z29.d, z26.d, z29.d\n"
      "zip1 z26.d, z27.d, z30.d\n"
      "zip2 z30.d, z27.d, z30.d\n"
      "zip1 z27.d, z0.d, z31.d\n"
      "zip2 z31.d, z0.d, z31.d\n"
      "b 59f\n"
      "58:"  // Height 6: no accumulate
      "mov z8.s, #0x0\n"
      "mov z9.s, #0x0\n"
      "mov z10.s, #0x0\n"
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "mov z13.s, #0x0\n"
      "mov z14.s, #0x0\n"
      "mov z15.s, #0x0\n"
      "mov z16.s, #0x0\n"
      "mov z17.s, #0x0\n"
      "mov z18.s, #0x0\n"
      "mov z19.s, #0x0\n"
      "mov z20.s, #0x0\n"
      "mov z21.s, #0x0\n"
      "mov z22.s, #0x0\n"
      "mov z23.s, #0x0\n"
      "mov z24.s, #0x0\n"
      "mov z25.s, #0x0\n"
      "mov z26.s, #0x0\n"
      "mov z27.s, #0x0\n"
      "mov z28.s, #0x0\n"
      "mov z29.s, #0x0\n"
      "mov z30.s, #0x0\n"
      "mov z31.s, #0x0\n"
      "59:"  // Height 6: setup done
      "mov x28, #0x0\n"
      "60:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 61f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "ldr x21, [x20, #0x28]\n"
      "cbnz x28, 62f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "add x21, x21, x20\n"
      "b 62f\n"
      "61:"  // Height 6: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "add x21, x22, x21\n"
      "62:"  // Height 6: input setup done
      "cmp x27, #0x10\n"
      "ble 64f\n"
      "63:"  // Height 6: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x27\n"
      "ld1rqb { z7.b }, p0/Z, [x26]\n"
      "ld1rqb { z0.b }, p0/Z, [x25]\n"
      "trn1 z6.d, z7.d, z0.d\n"
      "ld1rqb { z5.b }, p0/Z, [x24]\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "trn2 z7.d, z7.d, z0.d\n"
      "trn1 z4.d, z5.d, z1.d\n"
      "ld1rqb { z3.b }, p0/Z, [x22]\n"
      "ld1rqb { z0.b }, p0/Z, [x21]\n"
      "trn2 z5.d, z5.d, z1.d\n"
      "trn1 z2.d, z3.d, z0.d\n"
      "trn2 z3.d, z3.d, z0.d\n"
      "ld1b { z1.b }, p5/Z, [x10]\n"
      "ld1b { z0.b }, p5/Z, [x10, #1, MUL VL]\n"
      ".inst 0x45c198c8  // ummla z8.s, z6.b, z1.b\n"
      ".inst 0x45c19890  // ummla z16.s, z4.b, z1.b\n"
      ".inst 0x45c19858  // ummla z24.s, z2.b, z1.b\n"
      "ld1b { z1.b }, p5/Z, [x10, #2, MUL VL]\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x45c098cc  // ummla z12.s, z6.b, z0.b\n"
      ".inst 0x45c09894  // ummla z20.s, z4.b, z0.b\n"
      "cmp x27, #0x10\n"
      "add x26, x26, #0x10\n"
      ".inst 0x45c0985c  // ummla z28.s, z2.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #3, MUL VL]\n"
      ".inst 0x45c198c9  // ummla z9.s, z6.b, z1.b\n"
      "add x25, x25, #0x10\n"
      ".inst 0x45c19891  // ummla z17.s, z4.b, z1.b\n"
      ".inst 0x45c19859  // ummla z25.s, z2.b, z1.b\n"
      "ld1b { z1.b }, p5/Z, [x10, #4, MUL VL]\n"
      "add x24, x24, #0x10\n"
      ".inst 0x45c098cd  // ummla z13.s, z6.b, z0.b\n"
      ".inst 0x45c09895  // ummla z21.s, z4.b, z0.b\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      ".inst 0x45c0985d  // ummla z29.s, z2.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #5, MUL VL]\n"
      ".inst 0x45c198ca  // ummla z10.s, z6.b, z1.b\n"
      "add x21, x21, #0x10\n"
      ".inst 0x45c19892  // ummla z18.s, z4.b, z1.b\n"
      ".inst 0x45c1985a  // ummla z26.s, z2.b, z1.b\n"
      "ld1b { z1.b }, p5/Z, [x10, #6, MUL VL]\n"
      ".inst 0x45c098ce  // ummla z14.s, z6.b, z0.b\n"
      ".inst 0x45c09896  // ummla z22.s, z4.b, z0.b\n"
      ".inst 0x45c0985e  // ummla z30.s, z2.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      ".inst 0x45c198cb  // ummla z11.s, z6.b, z1.b\n"
      ".inst 0x45c19893  // ummla z19.s, z4.b, z1.b\n"
      ".inst 0x45c1985b  // ummla z27.s, z2.b, z1.b\n"
      "ld1b { z1.b }, p5/Z, [x10, #-8, MUL VL]\n"
      ".inst 0x45c098cf  // ummla z15.s, z6.b, z0.b\n"
      ".inst 0x45c09897  // ummla z23.s, z4.b, z0.b\n"
      ".inst 0x45c0985f  // ummla z31.s, z2.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #-7, MUL VL]\n"
      ".inst 0x45c198e8  // ummla z8.s, z7.b, z1.b\n"
      ".inst 0x45c198b0  // ummla z16.s, z5.b, z1.b\n"
      ".inst 0x45c19878  // ummla z24.s, z3.b, z1.b\n"
      "ld1b { z1.b }, p5/Z, [x10, #-6, MUL VL]\n"
      ".inst 0x45c098ec  // ummla z12.s, z7.b, z0.b\n"
      ".inst 0x45c098b4  // ummla z20.s, z5.b, z0.b\n"
      ".inst 0x45c0987c  // ummla z28.s, z3.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #-5, MUL VL]\n"
      ".inst 0x45c198e9  // ummla z9.s, z7.b, z1.b\n"
      ".inst 0x45c198b1  // ummla z17.s, z5.b, z1.b\n"
      ".inst 0x45c19879  // ummla z25.s, z3.b, z1.b\n"
      "ld1b { z1.b }, p5/Z, [x10, #-4, MUL VL]\n"
      ".inst 0x45c098ed  // ummla z13.s, z7.b, z0.b\n"
      ".inst 0x45c098b5  // ummla z21.s, z5.b, z0.b\n"
      ".inst 0x45c0987d  // ummla z29.s, z3.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #-3, MUL VL]\n"
      ".inst 0x45c198ea  // ummla z10.s, z7.b, z1.b\n"
      ".inst 0x45c198b2  // ummla z18.s, z5.b, z1.b\n"
      ".inst 0x45c1987a  // ummla z26.s, z3.b, z1.b\n"
      "ld1b { z1.b }, p5/Z, [x10, #-2, MUL VL]\n"
      ".inst 0x45c098ee  // ummla z14.s, z7.b, z0.b\n"
      ".inst 0x45c098b6  // ummla z22.s, z5.b, z0.b\n"
      ".inst 0x45c0987e  // ummla z30.s, z3.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #-1, MUL VL]\n"
      ".inst 0x45c198eb  // ummla z11.s, z7.b, z1.b\n"
      ".inst 0x45c198b3  // ummla z19.s, z5.b, z1.b\n"
      ".inst 0x45c1987b  // ummla z27.s, z3.b, z1.b\n"
      ".inst 0x45c098ef  // ummla z15.s, z7.b, z0.b\n"
      ".inst 0x45c098b7  // ummla z23.s, z5.b, z0.b\n"
      ".inst 0x45c0987f  // ummla z31.s, z3.b, z0.b\n"
      "bgt 63b\n"
      "64:"  // Height 6: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x27\n"
      "ld1rqb { z1.b }, p0/Z, [x26]\n"
      "ld1rqb { z0.b }, p0/Z, [x25]\n"
      "trn1 z7.d, z1.d, z0.d\n"
      "ld1rqb { z3.b }, p0/Z, [x24]\n"
      "ld1rqb { z2.b }, p0/Z, [x23]\n"
      "trn2 z1.d, z1.d, z0.d\n"
      "trn1 z6.d, z3.d, z2.d\n"
      "ld1rqb { z5.b }, p0/Z, [x22]\n"
      "ld1rqb { z0.b }, p0/Z, [x21]\n"
      "trn2 z3.d, z3.d, z2.d\n"
      "trn1 z4.d, z5.d, z0.d\n"
      "trn2 z5.d, z5.d, z0.d\n"
      "ld1b { z2.b }, p5/Z, [x10]\n"
      "ld1b { z0.b }, p5/Z, [x10, #1, MUL VL]\n"
      ".inst 0x45c298e8  // ummla z8.s, z7.b, z2.b\n"
      ".inst 0x45c298d0  // ummla z16.s, z6.b, z2.b\n"
      ".inst 0x45c29898  // ummla z24.s, z4.b, z2.b\n"
      "ld1b { z2.b }, p5/Z, [x10, #2, MUL VL]\n"
      "subs x27, x27, #0x8\n"
      ".inst 0x45c098ec  // ummla z12.s, z7.b, z0.b\n"
      ".inst 0x45c098d4  // ummla z20.s, z6.b, z0.b\n"
      ".inst 0x45c0989c  // ummla z28.s, z4.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #3, MUL VL]\n"
      ".inst 0x45c298e9  // ummla z9.s, z7.b, z2.b\n"
      ".inst 0x45c298d1  // ummla z17.s, z6.b, z2.b\n"
      ".inst 0x45c29899  // ummla z25.s, z4.b, z2.b\n"
      "ld1b { z2.b }, p5/Z, [x10, #4, MUL VL]\n"
      ".inst 0x45c098ed  // ummla z13.s, z7.b, z0.b\n"
      ".inst 0x45c098d5  // ummla z21.s, z6.b, z0.b\n"
      ".inst 0x45c0989d  // ummla z29.s, z4.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #5, MUL VL]\n"
      ".inst 0x45c298ea  // ummla z10.s, z7.b, z2.b\n"
      ".inst 0x45c298d2  // ummla z18.s, z6.b, z2.b\n"
      ".inst 0x45c2989a  // ummla z26.s, z4.b, z2.b\n"
      "ld1b { z2.b }, p5/Z, [x10, #6, MUL VL]\n"
      ".inst 0x45c098ee  // ummla z14.s, z7.b, z0.b\n"
      ".inst 0x45c098d6  // ummla z22.s, z6.b, z0.b\n"
      ".inst 0x45c0989e  // ummla z30.s, z4.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #7, MUL VL]\n"
      ".inst 0x45c298eb  // ummla z11.s, z7.b, z2.b\n"
      "addvl x10, x10, #8\n"
      ".inst 0x45c298d3  // ummla z19.s, z6.b, z2.b\n"
      ".inst 0x45c2989b  // ummla z27.s, z4.b, z2.b\n"
      ".inst 0x45c098ef  // ummla z15.s, z7.b, z0.b\n"
      ".inst 0x45c098d7  // ummla z23.s, z6.b, z0.b\n"
      ".inst 0x45c0989f  // ummla z31.s, z4.b, z0.b\n"
      "ble 65f\n"
      "ld1b { z2.b }, p5/Z, [x10]\n"
      "ld1b { z0.b }, p5/Z, [x10, #1, MUL VL]\n"
      ".inst 0x45c29828  // ummla z8.s, z1.b, z2.b\n"
      ".inst 0x45c29870  // ummla z16.s, z3.b, z2.b\n"
      ".inst 0x45c298b8  // ummla z24.s, z5.b, z2.b\n"
      ".inst 0x45c0982c  // ummla z12.s, z1.b, z0.b\n"
      "ld1b { z2.b }, p5/Z, [x10, #2, MUL VL]\n"
      ".inst 0x45c09874  // ummla z20.s, z3.b, z0.b\n"
      ".inst 0x45c098bc  // ummla z28.s, z5.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #3, MUL VL]\n"
      ".inst 0x45c29829  // ummla z9.s, z1.b, z2.b\n"
      ".inst 0x45c29871  // ummla z17.s, z3.b, z2.b\n"
      ".inst 0x45c298b9  // ummla z25.s, z5.b, z2.b\n"
      ".inst 0x45c0982d  // ummla z13.s, z1.b, z0.b\n"
      "ld1b { z2.b }, p5/Z, [x10, #4, MUL VL]\n"
      ".inst 0x45c09875  // ummla z21.s, z3.b, z0.b\n"
      ".inst 0x45c098bd  // ummla z29.s, z5.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #5, MUL VL]\n"
      ".inst 0x45c2982a  // ummla z10.s, z1.b, z2.b\n"
      ".inst 0x45c29872  // ummla z18.s, z3.b, z2.b\n"
      ".inst 0x45c298ba  // ummla z26.s, z5.b, z2.b\n"
      ".inst 0x45c0982e  // ummla z14.s, z1.b, z0.b\n"
      "ld1b { z2.b }, p5/Z, [x10, #6, MUL VL]\n"
      ".inst 0x45c09876  // ummla z22.s, z3.b, z0.b\n"
      ".inst 0x45c098be  // ummla z30.s, z5.b, z0.b\n"
      "ld1b { z0.b }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #8\n"
      ".inst 0x45c2982b  // ummla z11.s, z1.b, z2.b\n"
      ".inst 0x45c29873  // ummla z19.s, z3.b, z2.b\n"
      ".inst 0x45c298bb  // ummla z27.s, z5.b, z2.b\n"
      ".inst 0x45c0982f  // ummla z15.s, z1.b, z0.b\n"
      ".inst 0x45c09877  // ummla z23.s, z3.b, z0.b\n"
      ".inst 0x45c098bf  // ummla z31.s, z5.b, z0.b\n"
      "65:"  // Height 6: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 60b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "uzp1 z0.d, z8.d, z12.d\n"
      "add x22, x23, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "uzp2 z8.d, z8.d, z12.d\n"
      "uzp1 z12.d, z9.d, z13.d\n"
      "add x20, x21, x20, LSL #2\n"
      "uzp2 z9.d, z9.d, z13.d\n"
      "uzp1 z13.d, z10.d, z14.d\n"
      "st1w { z0.s }, p4, [x9]\n"
      "uzp2 z10.d, z10.d, z14.d\n"
      "uzp1 z14.d, z11.d, z15.d\n"
      "st1w { z12.s }, p3, [x9, #1, MUL VL]\n"
      "uzp2 z11.d, z11.d, z15.d\n"
      "uzp1 z15.d, z16.d, z20.d\n"
      "st1w { z13.s }, p2, [x9, #2, MUL VL]\n"
      "uzp2 z16.d, z16.d, z20.d\n"
      "uzp1 z20.d, z17.d, z21.d\n"
      "st1w { z14.s }, p1, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "uzp2 z17.d, z17.d, z21.d\n"
      "uzp1 z21.d, z18.d, z22.d\n"
      "st1w { z8.s }, p4, [x24]\n"
      "uzp2 z18.d, z18.d, z22.d\n"
      "uzp1 z22.d, z19.d, z23.d\n"
      "st1w { z9.s }, p3, [x24, #1, MUL VL]\n"
      "uzp2 z19.d, z19.d, z23.d\n"
      "uzp1 z23.d, z24.d, z28.d\n"
      "st1w { z10.s }, p2, [x24, #2, MUL VL]\n"
      "uzp2 z24.d, z24.d, z28.d\n"
      "uzp1 z28.d, z25.d, z29.d\n"
      "st1w { z11.s }, p1, [x24, #3, MUL VL]\n"
      "uzp2 z25.d, z25.d, z29.d\n"
      "uzp1 z29.d, z26.d, z30.d\n"
      "st1w { z15.s }, p4, [x23]\n"
      "uzp2 z26.d, z26.d, z30.d\n"
      "uzp1 z30.d, z27.d, z31.d\n"
      "st1w { z20.s }, p3, [x23, #1, MUL VL]\n"
      "uzp2 z27.d, z27.d, z31.d\n"
      "st1w { z21.s }, p2, [x23, #2, MUL VL]\n"
      "st1w { z22.s }, p1, [x23, #3, MUL VL]\n"
      "st1w { z16.s }, p4, [x22]\n"
      "st1w { z17.s }, p3, [x22, #1, MUL VL]\n"
      "st1w { z18.s }, p2, [x22, #2, MUL VL]\n"
      "st1w { z19.s }, p1, [x22, #3, MUL VL]\n"
      "st1w { z23.s }, p4, [x21]\n"
      "st1w { z28.s }, p3, [x21, #1, MUL VL]\n"
      "st1w { z29.s }, p2, [x21, #2, MUL VL]\n"
      "st1w { z30.s }, p1, [x21, #3, MUL VL]\n"
      "st1w { z24.s }, p4, [x20]\n"
      "st1w { z25.s }, p3, [x20, #1, MUL VL]\n"
      "st1w { z26.s }, p2, [x20, #2, MUL VL]\n"
      "st1w { z27.s }, p1, [x20, #3, MUL VL]\n"
      "66:"  // Height 6: Writeback done
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 57b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 68f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 67f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "67:"  // Update direct input
      "mov x20, #0x6\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "68:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr), [output_ptr] "+&r" (output_ptr)
      : [args_ptr] "r" (&ka), [flags] "r" (flags), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "x9", "x10", "x11", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

} // namespace arm_gemm
#endif  // ARM_COMPUTE_ENABLE_SVE
