/*###############################################################################
# Linux Management Providers (LMP), OpenDRIM_RecordLogPackage provider package
# Copyright (C) 2007 Ilsoo Byun, ETRI <widepis@etri.re.kr, widepis@empal.com>
#
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
#
# Modified by 2008 Guillaume BOTTEX, ETRI <guillaumebottex@etri.re.kr, guillaumebottex@gmail.com>
###############################################################################*/

#include "OpenDRIM_RecordLogAccess.h"

#define SYSLOG_CONFIG_FILE "/etc/syslog.conf"

static char* OpenDRIM_RecordLog_Severity_mapping[] = {"emerg", "alert", "crit", "err", \
        "warning", "notice", "info", "debug", "*", "none"};

static char* OpenDRIM_RecordLog_Facility_mapping[] = {"kern", "user", "mail", "daemon", \
        "auth", "syslog", "lpr", "news", "uucp", "cron", "authpriv", "INVALID", \
        "INVALID", "INVALID", "INVALID", "cron", "local0", "local1", "local2", \
        "local3", "local4", "local5", "local6", "local7", "*"};

static unsigned short facilityToRFC3164(const string& syslog_conf_facility);
static unsigned char severityToCIM(vector<unsigned short>& Severities, const unsigned short& RFC3164_severity, const unsigned char& severity_selection);
static unsigned short severityToRFC3164(const string& syslog_conf_severity);

static int convertLogInstanceToString(const OpenDRIM_RecordLog& log, string& output, string& errorMessage);

int OpenDRIM_RecordLogPackage_OpenDRIM_RecordLog_load(const CMPIBroker* broker, string& errorMessage) {
	_E_;
	_L_;
	return OK;
}

int OpenDRIM_RecordLogPackage_OpenDRIM_RecordLog_unload(string& errorMessage) {
	_E_;
	_L_;
	return OK;
}

static bool isAvailableLine(string& line, vector<string>& tokens) {
	_E_;

	if (line.size() == 0 || CF_startsWith(line, "\\") || line[0] == '#')
		return false;

	CF_splitTextBySpace(tokens, line);
	// If the line contains less than 2 elements, skip and go to the next line
	if (tokens.size() != 2)
		return false;

	//If the first character is '-', remove it
	if (CF_startsWith(tokens[1], "-"))
		tokens[1].erase(0,1);

	vector<string> filepath_elements;
	CF_splitText(filepath_elements, tokens[1], '/');
	string file_container;
	for (vector<unsigned short>::size_type i = 0; i < filepath_elements.size()-1; i++) {
		if (i > 0)
			file_container += "/";
		file_container += filepath_elements[i];
	}

	if (CF_startsWith(tokens[1], "/dev/") || !CF_isExist(file_container))
		return false;
	_L_;
	return true;
}

int OpenDRIM_RecordLogPackage_OpenDRIM_RecordLog_retrieve(const CMPIBroker* broker, const CMPIContext* ctx, vector<OpenDRIM_RecordLog>& result, const char** properties, string& errorMessage, const string& discriminant) {
	_E_;
	// Read content from /etc/syslog.conf
	int line_num = 0;
	string line;
	ifstream ifs(SYSLOG_CONFIG_FILE);
	OpenDRIM_RecordLog instance;
	while (getline(ifs, line)) {
		++line_num;
		vector<string> tokens;
		if (!isAvailableLine(line, tokens))
			continue;

		instance.setInstanceID(tokens[1]);
		if (discriminant=="ei") {
			if (OpenDRIM_RecordLogPackage_OpenDRIM_RecordLog_populate(
					instance, tokens, errorMessage) != OK) {
				errorMessage="Wrong format (arround line "+CF_intToStr((unsigned long) line_num)+"): /etc/syslog.conf";
				ifs.close();
				return FAILED;
			}
		}
		result.push_back(instance);
	}
	ifs.close();
	_L_;
	return OK;
}

int OpenDRIM_RecordLogPackage_OpenDRIM_RecordLog_getInstance(const CMPIBroker* broker, const CMPIContext* ctx, OpenDRIM_RecordLog& instance, const char** properties, string& errorMessage) {
	_E_;
	int line_num = 0;
	string line;
	ifstream ifs(SYSLOG_CONFIG_FILE);
	while (getline(ifs, line)) {
		++line_num;
		vector<string> tokens;
		if (!isAvailableLine(line, tokens))
			continue;

		if (!instance.InstanceID_isNULL && instance.InstanceID == tokens[1]) {
			if (OpenDRIM_RecordLogPackage_OpenDRIM_RecordLog_populate(
					instance, tokens, errorMessage) != OK) {
				errorMessage="Wrong format (arround line "+CF_intToStr((unsigned long) line_num)+"): /etc/syslog.conf";
				ifs.close();
				return FAILED;
			}
			return OK;
		}
	}
	ifs.close();
	
	if(errorMessage.empty())
			errorMessage = "No instance";
	_L_;
	return NOT_FOUND;
}

//For writing an OpenDRIM_RecordLog instance.
static unsigned short max(vector<unsigned short>& severity) {
	_E_;
	unsigned short max_num = 0;
	for (vector<unsigned short>::iterator c = severity.begin(); c != severity.end(); ++c) {
		if ((*c) > max_num)
			max_num = *c;
	}
	_L_;
	return max_num;
}

//For writing an OpenDRIM_RecordLog instance.
static string writeFacilitySeverity(unsigned short facility, unsigned short severity) {
	string result(OpenDRIM_RecordLog_Facility_mapping[facility]);
	// modified by Frederic Desmons (2007/11/13)
	result += ".=";
	//if (severity == 7) severity = 8;
	result += OpenDRIM_RecordLog_Severity_mapping[severity];
	return result;
}

//For writing an OpenDRIM_RecordLog instance.
static int convertLogInstanceToString(const OpenDRIM_RecordLog& log, string& output, string& errorMessage) {
	_E_;
	output.clear();
	// added by Frederic Desmons (2007/11/13)
	for (vector<unsigned short>::size_type i = 0; i < log.Facility.size(); i++) {
		if (log.Facility[i] > 24) {
			errorMessage = "Facility values must be lower than 25";
			return FAILED;
		}
		if (log.Facility[i] >= 11 && log.Facility[i] <= 14) {
			errorMessage = "Facility values cannot be 11, 12, 13, 14 (INVALID)";
			return FAILED;
		}
		if (log.Severity[i] > 9) {
			errorMessage = "Severity values must be lower than 10";
			return FAILED;
		}
		output += writeFacilitySeverity(log.Facility[i], log.Severity[i]);
		if (i < log.Facility.size() - 1)
			output += ";";
	}
	// end of added
	/*
	vector<unsigned short>::size_type facility_count = 0;
	do {
		unsigned short facility = log.Facility[facility_count];
		for (; facility_count < log.Facility.size(); ++facility_count) {
			if (facility != log.Facility[facility_count])
				break;
		}

		vector<unsigned short> severities;
		for (vector<unsigned short>::size_type i = 0; i < facility_count; ++i) {
			severities.push_back(log.Severity[i]);
		}
		output += writeFacilitySeverity(facility, max(severities));
		if (facility_count < log.Facility.size())
			output += ";";
	} while (facility_count < log.Facility.size());
	*/
	output += "\t\t\t\t\t\t";
	output += log.InstanceID;
	_L_;
	return OK;
}

int OpenDRIM_RecordLogPackage_OpenDRIM_RecordLog_setInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_RecordLog& newInstance, const OpenDRIM_RecordLog& oldInstance, const char** properties, string& errorMessage) {
	_E_;
	// added by Frederic Desmons (2007/11/13)
	vector<unsigned short> Facility, Severity;
	if (newInstance.getFacility(Facility) != OK) {
		if (newInstance.getSeverity(Severity) != OK) {
			// nothing to do
			return OK;
		} else {
			errorMessage = "Facility and Severity must BOTH be defined";
			return FAILED;
		}
	} else {
		if (newInstance.getSeverity(Severity) != OK) {
			errorMessage = "Facility and Severity must BOTH be defined";
			return FAILED;
		}
	}
	if (Facility.size() != Severity.size()) {
		errorMessage = "Facility and Severity must have the same size";
		return FAILED;
	}
	// end of added
	string line;
	ifstream ifs(SYSLOG_CONFIG_FILE);
	vector<string> lines;
	while (getline(ifs, line)) {
		vector<string> tokens;
		//Modify the corresponding line.
		if (isAvailableLine(line, tokens) && tokens[1] == newInstance.InstanceID) {
			line.clear();
			CF_assert(convertLogInstanceToString(newInstance, line, errorMessage));
		}
		lines.push_back(line);
	}
	ifs.close();

	// rewriting the syslog configuation file.
	ofstream ofs(SYSLOG_CONFIG_FILE, ios::trunc);
	for (vector<string>::size_type i = 0; lines.size() > i; ++i) {
		ofs << lines[i] << endl;
	}
	ofs.close();
	_L_;
	return OK;
}

int OpenDRIM_RecordLogPackage_OpenDRIM_RecordLog_createInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_RecordLog& instance, string& errorMessage) {
	_E_;
	// added by Frederic Desmons (2007/11/13)
	vector<unsigned short> Facility, Severity;
	if (instance.getFacility(Facility) != OK) {
		if (instance.getSeverity(Severity) != OK) {
			// nothing to do
			return OK;
		} else {
			errorMessage = "Facility and Severity must BOTH be defined";
			return FAILED;
		}
	} else {
		if (instance.getSeverity(Severity) != OK) {
			errorMessage = "Facility and Severity must BOTH be defined";
			return FAILED;
		}
	}
	if (Facility.size() != Severity.size()) {
		errorMessage = "Facility and Severity must have the same size";
		return FAILED;
	}
	// end of added
	string line;
	CF_assert(convertLogInstanceToString(instance, line, errorMessage));

	//Append a line.
	ofstream ofs(SYSLOG_CONFIG_FILE, ios::app);
	ofs << line << endl;
	ofs.close();
	_L_;
	return OK;
}

int OpenDRIM_RecordLogPackage_OpenDRIM_RecordLog_deleteInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_RecordLog& instance, string& errorMessage) {
	_E_;
	string line;
	ifstream ifs(SYSLOG_CONFIG_FILE);
	vector<string> lines;
	while (getline(ifs, line)) {
		vector<string> tokens;
		//Don't write
		if (isAvailableLine(line, tokens) && tokens[1] == instance.InstanceID)
			continue;
		lines.push_back(line);
	}
	ifs.close();

	//Rewrinting the syslog configuation file.
	ofstream ofs(SYSLOG_CONFIG_FILE, ios::trunc);
	for (vector<string>::size_type i = 0; lines.size() > i; ++i) {
		ofs << lines[i] << endl;
	}
	ofs.close();

	_L_;
	return OK;
}

int OpenDRIM_RecordLogPackage_OpenDRIM_RecordLog_RequestStateChange(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_RecordLog& instance, unsigned int& returnValue, const OpenDRIM_RecordLog_RequestStateChange_In& in, OpenDRIM_RecordLog_RequestStateChange_Out& out, string& errorMessage) {
	_E_;
	_L_;
	return NOT_SUPPORTED;
}

int OpenDRIM_RecordLogPackage_OpenDRIM_RecordLog_ClearLog(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_RecordLog& instance, unsigned int& returnValue, string& errorMessage) {
	_E_;
	string logfile;
	instance.getInstanceID(logfile);
	if (!CF_isExist(logfile)) {
		returnValue = 0;
		return OK;
	}
	string stdOut;
	int errorCode = CF_runCommandFL("/bin/echo > " + logfile, stdOut, errorMessage);
	if (errorCode != OK) {
		returnValue = 4;
		return OK;
	}
	returnValue = 0;
	_L_;
	return OK;
}

int OpenDRIM_RecordLogPackage_OpenDRIM_RecordLog_populate(OpenDRIM_RecordLog& instance, vector<string>& tokens, string& errorMessage) {
	_E_;

	/*
	 * Properties to fill from profile
	 * + Mandatory:
	 * [X] InstanceID			[KEY]
	 * [X] MaxNumberOfRecords
	 * [X] LogState
	 * [X] OverwritePolicy
	 * [X] RequestedState
	 * [X] EnabledState
	 * [X] OperationalStatus
	 * [X] HealthState
	 * [X] ElementName
	 * + Others:
	 * [-] AvailableRequestedStates
	 * [X] CurrentNumberOfRecords
	 * [-] StatusDescriptions
	 * [X] TimeOfLastStateChange
	 */

	instance.setElementName(tokens[1]);

	// Extract the facilities and the severities
	vector<unsigned short> Facilities;
	vector<unsigned short> Severities;

	// Split into substrings using ';' as a delimiter
	vector<string> facility_severity;
	CF_splitText(facility_severity, tokens[0], ';');
	for (size_t j=0; j<facility_severity.size(); j++) {
		// Seperate the facility and the severity
		vector<string> facility_severity_element;
		CF_splitText(facility_severity_element, facility_severity[j], '.');
		if (facility_severity_element.size()!=2 || facility_severity_element[1].size()<1) {
			return FAILED;
		}
		// 0: This severity and higher severities
		// 1: Only this severity
		// 2: All but this severity
		unsigned char severity_selection=0;
		if (facility_severity_element[1][0]=='=') {
			severity_selection=1;
			facility_severity_element[1].erase(0,1);
		}
		else if (facility_severity_element[1][0]=='!') {
			severity_selection=2;
			facility_severity_element[1].erase(0,1);
		}
		vector<string> facility_elements;
		CF_splitText(facility_elements, facility_severity_element[0], ',');
		// Add the Severity/Facility couples
		for (size_t k=0; k<facility_elements.size(); k++) {
			unsigned char nb_severities=severityToCIM(Severities, severityToRFC3164(facility_severity_element[1]), severity_selection);
			for (unsigned char l=0; l<nb_severities; l++)
				Facilities.push_back(facilityToRFC3164(facility_elements[k]));
		}
	}
	instance.setFacility(Facilities);
	instance.setSeverity(Severities);

	string time;
	CF_assert(CF_lastModified(tokens[1], time, errorMessage));
	instance.setTimeOfLastStateChange(time);

	string record_count, stdout, stderr;
	CF_assert(CF_runCommand("cat "+instance.InstanceID+" | wc -l", stdout, stderr, errorMessage));
	instance.setCurrentNumberOfRecords(atoi(stdout.c_str()));

	instance.setMaxNumberOfRecords(0);
	instance.setOverwritePolicy(7);
	instance.setHealthState(5);
	vector<unsigned short> status;
	status.push_back(2);
	instance.setOperationalStatus(status);

	instance.setLogState(LS_NOT_APPLICABLE);
	instance.setEnabledState(ES_NOT_APPLICABLE);
	instance.setRequestedState(RS_NOT_APPLICABLE);
	_L_;
	return OK;
}

// Convert facility from syslog.conf format to RFC3164 format
unsigned short facilityToRFC3164(const string& syslog_conf_facility) {
	_E_;
	string low_case_facility=CF_toLowCase(syslog_conf_facility);
	unsigned short facility=25;
	for (unsigned int i=0; i<sizeof(OpenDRIM_RecordLog_Facility_mapping)/sizeof(char*); i++) {
		if (low_case_facility==OpenDRIM_RecordLog_Facility_mapping[i]) {
			facility=i;
			break;
		}
	}
	_L_;
	return facility;
}

// convert the Severity from syslog.conf format to RFC3164 format
unsigned short severityToRFC3164(const string& syslog_conf_severity) {
	_E_;
	string low_case_severity=CF_toLowCase(syslog_conf_severity);
	unsigned short severity=10;
	for (unsigned int i=0; i<sizeof(OpenDRIM_RecordLog_Severity_mapping)/sizeof(char*); i++) {
		if (low_case_severity==OpenDRIM_RecordLog_Severity_mapping[i]) {
			severity=i;
			break;
		}
	}
	_L_;
	return severity;
}

unsigned char severityToCIM(vector<unsigned short>& Severities, const unsigned short& RFC3164_severity, const unsigned char& severity_selection) {
	_E_;
	if (RFC3164_severity==10) {
		Severities.push_back(10);
		return 1;
	}
	if (RFC3164_severity==9) {
		Severities.push_back(9);
		return 1;
	}
	if (RFC3164_severity==8) {
		for (unsigned char i=0; i<RFC3164_severity; i++)
			Severities.push_back(i);
		return RFC3164_severity;
	}
	if (severity_selection==1) {
		Severities.push_back(RFC3164_severity);
		return 1;
	}
	if (severity_selection==0) {
		for (unsigned char i=0; i<RFC3164_severity+1; i++)
			Severities.push_back(i);
		return RFC3164_severity+1;
	}
	if (severity_selection==2) {
		for (unsigned char i=0; i<8; i++) {
			if (i!=RFC3164_severity)
				Severities.push_back(i);
		}
		return 7;
	}
	_L_;
	return 0;
}
