/*
 * Copyright (C) 2013 Google Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer
 *    in the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of Google Inc. nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "core/dom/Microtask.h"

#include "bindings/v8/V8PerIsolateData.h"
#include "wtf/Vector.h"
#include <v8.h>

namespace WebCore {

void Microtask::performCheckpoint()
{
    v8::Isolate* isolate = v8::Isolate::GetCurrent();
    V8PerIsolateData* isolateData = V8PerIsolateData::from(isolate);
    ASSERT(isolateData);
    if (isolateData->recursionLevel() || isolateData->performingMicrotaskCheckpoint())
        return;
    isolateData->setPerformingMicrotaskCheckpoint(true);

    v8::HandleScope handleScope(isolate);
    v8::Local<v8::Context> context = isolateData->ensureDomInJSContext();
    v8::Context::Scope scope(context);
    v8::V8::RunMicrotasks(isolate);

    isolateData->setPerformingMicrotaskCheckpoint(false);
}

COMPILE_ASSERT(sizeof(void*) == sizeof(MicrotaskCallback), VoidPtrAndFunctionPtrAreSameSize);

static void microtaskFunctionCallback(const v8::FunctionCallbackInfo<v8::Value>& info)
{
    MicrotaskCallback callback =
        reinterpret_cast<MicrotaskCallback>(reinterpret_cast<intptr_t>(
            info.Data().As<v8::External>()->Value()));
    (*callback)();
}

void Microtask::enqueueMicrotask(MicrotaskCallback callback)
{
    v8::Isolate* isolate = v8::Isolate::GetCurrent();
    V8PerIsolateData* isolateData = V8PerIsolateData::from(isolate);
    v8::HandleScope handleScope(isolate);
    v8::Local<v8::Context> context = isolateData->ensureDomInJSContext();
    v8::Context::Scope scope(context);
    v8::Local<v8::External> handler =
        v8::External::New(isolate,
            reinterpret_cast<void*>(reinterpret_cast<intptr_t>(callback)));
    v8::V8::EnqueueMicrotask(isolate, v8::Function::New(isolate, &microtaskFunctionCallback, handler));
}

} // namespace WebCore
