<?xml version="1.0"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:exslt="http://exslt.org/common"
                xmlns:xhtml="http://www.w3.org/1999/xhtml"
                exclude-result-prefixes="exslt xhtml" version="1.0">

<!--
     This file contains the XSL templates used during an XNav
     make command.

     Copyright © 2004-2007 Brendt Wohlberg <software@wohlberg.net>

     Please see the XNav distribution README file for license
     information.
-->



 <xsl:output method="xml" indent="yes" omit-xml-declaration = "yes"
    doctype-public="-//W3C//DTD XHTML 1.0 Strict//EN"
    doctype-system="http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd"
    encoding="utf-8"/>



   <!-- Stylesheet parameter setting base path for processing -->
   <xsl:param name="basep" select="'./'"/>
   <!-- Stylesheet parameter setting path to main CSS file -->
   <xsl:param name="stylep" select="''"/>
   <!-- Stylesheet parameter setting path to header file -->
   <xsl:param name="headp" select="''"/>
   <!-- Stylesheet parameter setting path to footer file -->
   <xsl:param name="footp" select="''"/>
   <!-- Stylesheet parameter selecting whether to append 'index.html'
        to directory paths -->
   <xsl:param name="dpindex" select="0"/>
   <!-- Stylesheet parameter selecting output verbosity -->
   <xsl:param name="verbose" select="0"/>
   <!-- Stylesheet parameter determining inclusion of head/title -->
   <xsl:param name="hdtitle" select="1"/>
   <!-- Stylesheet parameter determining inclusion of head/base -->
   <xsl:param name="hdbase" select="1"/>
   <!-- Stylesheet parameter determining inclusion of head/script -->
   <xsl:param name="hdscript" select="1"/>
   <!-- Stylesheet parameter determining inclusion of head/style -->
   <xsl:param name="hdstyle" select="1"/>
   <!-- Stylesheet parameter determining inclusion of head/meta -->
   <xsl:param name="hdmeta" select="1"/>
   <!-- Stylesheet parameter determining inclusion of head/link -->
   <xsl:param name="hdlink" select="1"/>
   <!-- Stylesheet parameter determining inclusion of head/object -->
   <xsl:param name="hdobject" select="1"/>
   <!-- Stylesheet parameter determining inclusion of body {} in head/style -->
   <xsl:param name="stbody" select="1"/>
   <!-- Stylesheet parameter requesting a body substitution div -->
   <xsl:param name="stbdiv" select="0"/>



   <!-- Top level variables with content of head and foot files -->
   <xsl:variable name="head" select="document($headp,/)"/>
   <xsl:variable name="foot" select="document($footp,/)"/>

   <!-- Variable containing the node used as the document base reference -->
   <xsl:variable name="baserefnode" select="/"/>

   <!-- Suffix to append at the end of directory paths -->
   <xsl:variable name="dirsuffix">
     <xsl:choose>
       <xsl:when test="boolean(number($dpindex))">
         <xsl:text>index.html</xsl:text>
       </xsl:when>
       <xsl:otherwise>
           <xsl:text></xsl:text>
         </xsl:otherwise>
       </xsl:choose>
     </xsl:variable>
     


   <!-- Template matching root to initiate processing on top level
        xnav file -->
   <xsl:template match="/">
     <xsl:call-template name="navtree">
       <xsl:with-param name="b" select="'./'"/>
       <xsl:with-param name="d" select="''"/>
       <xsl:with-param name="l" select="''"/>
       <xsl:with-param name="p" select="''"/>
       <xsl:with-param name="x" select="/.."/>
       <xsl:with-param name="y" select="."/>
     </xsl:call-template>
   </xsl:template>



   <!-- Main template, called recursively -->
   <xsl:template name="navtree">
     <!-- Back path -->
     <xsl:param name="b" select="'./'"/>
     <!-- Directory name for this level, from parent level -->
     <xsl:param name="d" select="''"/>
     <!-- Label for this level, from parent level -->
     <xsl:param name="l" select="''"/>
     <!-- Cumulative path to this level -->
     <xsl:param name="p" select="'./'"/>
     <!-- The set of xnav elements encountered up to this level -->
     <xsl:param name="x" select="/.."/>
     <!-- The xnav element for this level -->
     <xsl:param name="y" select="/.."/>


     <!-- Provide status messages when requested -->
     <xsl:if test="boolean(number($verbose))">
       <xsl:message>
         <xsl:text>Processing directory: </xsl:text>
         <xsl:choose>
           <xsl:when test="$p=''">
             <xsl:text>[XNav Root]</xsl:text>
           </xsl:when>
           <xsl:otherwise>
             <xsl:value-of select="$p"/>
           </xsl:otherwise>
         </xsl:choose>
         <xsl:text>&#10;</xsl:text>
       </xsl:message>
     </xsl:if>

     <!-- Construct a new collection of cumulative xnav element
          wrapped in the wrapper element -->
     <xsl:variable name="nx">
       <xsl:copy-of select="$x"/>
       <wrapper dir='{$d}' label='{$l}' path='{$p}' back='{$b}'>
         <xsl:copy-of select="$y"/>
       </wrapper>
     </xsl:variable>

     <!-- Variable containing the local topnav html -->
     <xsl:variable name="topnav">
       <span class="xnav-topnav">
         <xsl:call-template name="format-top">
           <xsl:with-param name="w" select="exslt:node-set($nx)"/>
         </xsl:call-template>
       </span>
     </xsl:variable>

     <!-- Variable containing the local sidenav html -->
     <xsl:variable name="sidenav">
       <xsl:call-template name="format-side">
         <xsl:with-param name="w" select="exslt:node-set($nx)"/>
       </xsl:call-template>
     </xsl:variable>

    <!-- Process index file at this level -->
    <xsl:variable name="idxtype">
      <xsl:choose>
        <xsl:when test="$y/xnav/@type">
          <xsl:value-of select="$y/xnav/@type"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:text>html</xsl:text>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <xsl:call-template name="process-file">
      <xsl:with-param name="type" select="$idxtype"/>
      <xsl:with-param name="path" select="$p"/>
      <xsl:with-param name="back" select="$b"/>
      <xsl:with-param name="topn" select="exslt:node-set($topnav)"/>
      <xsl:with-param name="sidn" select="exslt:node-set($sidenav)"/>
      <xsl:with-param name="src" select="concat($p, 'index.xml')"/>
      <xsl:with-param name="dst" select="concat($basep, $p, 'index.html')"/>
    </xsl:call-template>
    <!-- Process other files at this level -->
    <xsl:for-each select="$y/xnav/file">
      <!-- Determine file type -->
      <xsl:variable name="type">
        <xsl:choose>
          <xsl:when test="@type">
            <xsl:value-of select="@type"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:text>html</xsl:text>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:variable>
      <!-- Call processing template -->
      <xsl:call-template name="process-file">
        <xsl:with-param name="type" select="$type"/>
        <xsl:with-param name="file" select="."/>
        <xsl:with-param name="path" select="$p"/>
        <xsl:with-param name="back" select="$b"/>
        <xsl:with-param name="topn" select="exslt:node-set($topnav)"/>
        <xsl:with-param name="sidn" select="exslt:node-set($sidenav)"/>
        <xsl:with-param name="src" select="concat($p, @href)"/>
        <xsl:with-param name="dst" select="concat($basep, $p, 
                                              substring-before(@href, '.xml'),
                                                  '.html')"/>
      </xsl:call-template>
    </xsl:for-each>

    <!-- Descend down the tree -->
    <xsl:for-each select="$y/xnav/directory[not(@enter='no')]">
      <!-- Variable $dhref is @href with a single final '/' character -->
      <xsl:variable name="dhref" select="concat(
                                           substring(@href,1,
                                             string-length(@href)-1),
                                           translate(
                                             substring(@href,
                                               string-length(@href)),
                                             '/',''),
                                           '/'
                                         )"/>
      <xsl:call-template name="navtree">
        <xsl:with-param name="b" select="concat($b, '../')"/>
        <xsl:with-param name="d" select="$dhref"/>
        <xsl:with-param name="l" select="."/>
        <xsl:with-param name="p" select="concat($p, $dhref)"/>
        <xsl:with-param name="x" select="$nx"/>
        <xsl:with-param name="y" select="document(concat($dhref, 'xnav.xml'), 
                                                  /)"/>
     </xsl:call-template>
    </xsl:for-each>

   </xsl:template>



   <!-- Format top nav from cumulative xnav collection -->
   <xsl:template name="format-top">
     <!-- The wrapped xnav collection -->
     <xsl:param name="w" select="/.."/>

     <xsl:for-each select="$w/wrapper">
       <xsl:variable name="p" select="position()"/>
       <a href="{concat($w/wrapper[position()=last()-$p+1]/@back,$dirsuffix)}">
         <xsl:choose>
           <xsl:when test="position()=1">
             <xsl:value-of select="xnav/@label"/>
           </xsl:when>
           <xsl:otherwise>
             <xsl:value-of select="@label"/>
           </xsl:otherwise>
         </xsl:choose>
       </a>
       <xsl:if test="position()!=last()">
         <xsl:text> &#8250; </xsl:text>
       </xsl:if>
     </xsl:for-each>

   </xsl:template>




   <!-- Format side nav from cumulative xnav collection -->
   <xsl:template name="format-side">
     <!-- The wrapped xnav collection -->
     <xsl:param name="w" select="/.."/>
     <!-- The directory name at which to start processing -->
     <xsl:param name="d" select="''"/>
     <!-- The recursive descent level -->
     <xsl:param name="l" select="0"/>

     <!-- Construct back references based on level of recursion -->
     <xsl:variable name="b" select="$w/wrapper[position()=last()-$l]/@back"/>

     

     <!-- A wrapper was found for the current level -->
     <xsl:if test="$w/wrapper[$l + 1]">
       <div class="xnav-sidenav">
         <xsl:for-each select="$w/wrapper[$l + 1]/xnav/*[not(@sidenav='no')]">
           <xsl:variable name="href">
             <xsl:choose>
               <xsl:when test="name()='file'">
                 <xsl:value-of select="concat($b, 
                                              substring-before(@href,'.xml'), 
                                              '.html')"/>
               </xsl:when>
               <xsl:otherwise>
                 <xsl:value-of select="concat($b, @href, $dirsuffix)"/>
               </xsl:otherwise>
             </xsl:choose>
           </xsl:variable>
           <a href="{$href}">
             <xsl:text> &#8250; </xsl:text>
             <xsl:value-of select="."/>
           </a>
           <!-- The current node is a directory, and the next wrapper
                element has a dir attribute matching the href
                attribute of this node -->
           <xsl:if test="name(.)='directory' and ./@href =
                         $w/wrapper[$l + 2]/@dir">
             <xsl:call-template name="format-side">
               <xsl:with-param name="w" select="$w"/>
               <xsl:with-param name="d" select="current()/@href"/>
               <xsl:with-param name="l" select="$l + 1"/>
             </xsl:call-template>
           </xsl:if>
         </xsl:for-each>
       </div>
     </xsl:if>

   </xsl:template>
   


   <!-- Process an xml file of the given type, converting from src to dst -->
   <xsl:template name="process-file">
     <xsl:param name="type" select="'html'"/>
     <xsl:param name="file" select="''"/>
     <xsl:param name="path" select="''"/>
     <xsl:param name="back" select="''"/>
     <xsl:param name="topn" select="/.."/>
     <xsl:param name="sidn" select="/.."/>
     <xsl:param name="src" select="''"/>
     <xsl:param name="dst" select="''"/>

     <xsl:choose>
       <!-- File type is html -->
       <xsl:when test="$type='html'">
         <!-- Write final html -->
         <xsl:call-template name="write-final-html">
           <xsl:with-param name="file" select="$file"/>
           <xsl:with-param name="path" select="$path"/>
           <xsl:with-param name="back" select="$back"/>
           <xsl:with-param name="topn" select="$topn"/>
           <xsl:with-param name="sidn" select="$sidn"/>
           <xsl:with-param name="src" select="document($src,$baserefnode)"/>
           <xsl:with-param name="dst" select="$dst"/>
         </xsl:call-template>
       </xsl:when>
       <!-- File type is not html -->
       <xsl:otherwise>
         <!-- Wrap parameters up in a top level element with name equal to
          the selected type, allowing dynamic selection of the
          processing template -->
         <xsl:variable name="pw">
           <xsl:element name="{$type}">
             <file><xsl:value-of select="$file"/></file>
             <path><xsl:value-of select="$path"/></path>
             <back><xsl:value-of select="$back"/></back>
             <topnav><xsl:copy-of select="$topn"/></topnav>
             <sidenav><xsl:copy-of select="$sidn"/></sidenav>
             <src><xsl:copy-of select="document($src, $baserefnode)"/></src>
           </xsl:element>
         </xsl:variable>
         <!-- Construct intermediate html -->
         <xsl:variable name="inter">
           <xsl:apply-templates select="exslt:node-set($pw)" 
                                mode="dynamic-template-select"/>
         </xsl:variable>
         <!-- Write final html -->
         <xsl:call-template name="write-final-html">
           <xsl:with-param name="file" select="$file"/>
           <xsl:with-param name="path" select="$path"/>
           <xsl:with-param name="back" select="$back"/>
           <xsl:with-param name="topn" select="$topn"/>
           <xsl:with-param name="sidn" select="$sidn"/>
           <xsl:with-param name="src" select="exslt:node-set($inter)"/>
           <xsl:with-param name="dst" select="$dst"/>
         </xsl:call-template>
       </xsl:otherwise>
     </xsl:choose>

   </xsl:template>



   <!-- Handle conversion of html to final xnav-wrapped html -->
   <xsl:template name="write-final-html">
     <xsl:param name="file" select="''"/>
     <xsl:param name="path" select="''"/>
     <xsl:param name="back" select="''"/>
     <xsl:param name="topn" select="/.."/>
     <xsl:param name="sidn" select="/.."/>
     <xsl:param name="src" select="/.."/>
     <xsl:param name="dst" select="''"/>

     <xsl:document href="{$dst}" method="xml" encoding="utf-8"
       omit-xml-declaration = "yes" indent="yes"
       doctype-public="-//W3C//DTD XHTML 1.0 Strict//EN"
       doctype-system="http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">
       <html>
         <head>
           <meta http-equiv="content-type" content="application/xhtml+xml; charset=utf-8" />
           <meta name="generator" content="XNav (http://www.wohlberg.net/public/software/xml/xnav/)" />
           <link rel="stylesheet" type="text/css" 
                 href="{concat($back,$stylep)}" title="XNav Stylesheet"/>
           <xsl:apply-templates select="$src/html/head/*[
                              (boolean(number($hdtitle)) and self::title) or 
                              (boolean(number($hdbase)) and self::base) or 
                              (boolean(number($hdscript)) and self::script) or 
                              (boolean(number($hdstyle)) and self::style) or 
                              (boolean(number($hdmeta)) and self::meta) or 
                              (boolean(number($hdlink)) and self::link)  or 
                              (boolean(number($hdobject)) and self::object)] |
                             $src/xhtml:html/xhtml:head/*[
                        (boolean(number($hdtitle)) and self::xhtml:title) or 
                        (boolean(number($hdbase)) and self::xhtml:base) or 
                        (boolean(number($hdscript)) and self::xhtml:script) or 
                        (boolean(number($hdstyle)) and self::xhtml:style) or 
                        (boolean(number($hdmeta)) and self::xhtml:meta) or 
                        (boolean(number($hdlink)) and self::xhtml:link)  or 
                        (boolean(number($hdobject)) and self::xhtml:object)]"/>
         </head>
         <body>
           <xsl:copy-of select="$head"/>

           <div id="xnav-topnav">
             <xsl:copy-of select="$topn"/>
             <xsl:if test="$file != ''">
               <xsl:text> &#8250; </xsl:text>
               <xsl:value-of select="$file"/>
             </xsl:if>
           </div>

           <div id="xnav-sidenav">
             <xsl:copy-of select="$sidn"/>
           </div>

           <div id="xnav-main">
             <xsl:choose>
               <xsl:when test="boolean(number($stbdiv))">
                 <div id="xnav-body-subst">
                   <xsl:copy-of select="$src/html/body/node() | 
                                        $src/xhtml:html/xhtml:body/node()"/>
                 </div>
               </xsl:when>
               <xsl:otherwise>
                 <xsl:copy-of select="$src/html/body/node() | 
                                      $src/xhtml:html/xhtml:body/node()"/>
               </xsl:otherwise>
             </xsl:choose>
           </div>

           <xsl:copy-of select="$foot"/>
         </body>
       </html>
     </xsl:document>

   </xsl:template>



   <!-- Error handling template for case when appropriate template not
        defined -->
   <xsl:template match="*" mode="dynamic-template-select">
     <xsl:message terminate="yes">
       <xsl:text>Error: could not find relevant template for type </xsl:text>
       <xsl:value-of select="name()"/>
     </xsl:message>
   </xsl:template>


   <!-- If required by $stbody stylesheet parameter, filter body {...} 
        section from CSS in a style element. -->
   <xsl:template match="style[@type='text/css'] | 
                        xhtml:style[@type='text/css']">
     <xsl:choose>
       <xsl:when test="boolean(number($stbody)) or not(contains(., 'body'))">
         <xsl:copy>
           <xsl:apply-templates select="@*"/>
           <xsl:apply-templates select="*|text()"/>
         </xsl:copy>
       </xsl:when>
       <xsl:otherwise>
         <xsl:variable name="afterbody" 
              select="substring-after(substring-after(., 'body'), '{')"/>
         <xsl:copy>
           <xsl:apply-templates select="@*"/>
           <xsl:value-of select="substring-before(., 'body')"/>
           <xsl:if test="boolean(number($stbdiv))">
             <xsl:value-of select="concat('#xnav-body-subst {',
                                          substring-before($afterbody, '}'),
                                          '}&#10;')"/>
           </xsl:if>
           <xsl:value-of select="substring-after($afterbody, '}')"/>
         </xsl:copy>
       </xsl:otherwise>
     </xsl:choose>
   </xsl:template>



   <!-- Default identity transform template. Used for html/head
        children without more specific matching templates. -->
   <xsl:template match="@*|node()">
     <xsl:copy>
       <xsl:apply-templates select="@*|node()"/>
     </xsl:copy>
   </xsl:template>



</xsl:stylesheet>
