/*
 *  This file is part of X-File Manager XFM
 *  ----------------------------------------------------------------------
  FmAw.c
  
  (c) Simon Marlow 1990-1993
  (c) Albert Graef 1994

  modified 7-1997 by strauman@sun6hft.ee.tu-berlin.de to add
  different enhancements (see README-1.4).

  modified 2003,2005,2006 by Bernhard R. Link (see Changelog)

  Functions & data for creating & maintaining  the application window
 *  ----------------------------------------------------------------------
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <xfmconfig.h>

#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <time.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>
#include <X11/Xaw3d/Form.h>
#include <X11/Xaw3d/Box.h>
#include <X11/Xaw3d/Viewport.h>
#include <X11/Xaw3d/Toggle.h>
#include <X11/Xaw3d/Label.h>

#include "global.h"

#include "Am.h"

#define APPWIDTH 300

#ifdef ENHANCE_TRANSLATIONS
#undef XtNtranslations
#define XtNtranslations ""
#endif

/*-----------------------------------------------------------------------------
  PUBLIC DATA 
-----------------------------------------------------------------------------*/
AppWindowRec aw;
Widget *aw_button_items,
app_popup_widget, *app_popup_items, app_popup_widget1, *app_popup_items1;
int n_appst = 0;
char **appst = NULL;


/*-----------------------------------------------------------------------------
  STATIC DATA 
-----------------------------------------------------------------------------*/

static const MenuItemRec app_popup_menu[] = {
  { "install app", "Install...", appInstallAppCb },
  { "install group", "Install group...", appInstallGroupCb },
  { "line1", NULL, NULL },
  { "cut", "Cut", appCutCb },
  { "copy", "Copy", appCopyCb },
  { "paste", "Paste", appPasteCb },
  { "line2", NULL, NULL },
  { "delete", "Delete", appRemoveCb },
  { "line3", NULL, NULL },
  { "select all", "Select all", appSelectAllCb },
  { "deselect all", "Deselect all", appDeselectCb },
  { "line4", NULL, NULL },
  { "about", "About xfm...", aboutCb },
  { "line5", NULL, NULL },
  { "quit", "Quit", appCloseCb },
};

static const MenuItemRec app_popup_menu1[] = {
  { "edit", "Edit...", appEditCb },
  { "line1", NULL, NULL },
  { "cut", "Cut", appCutCb },
  { "copy", "Copy", appCopyCb },
  { "line2", NULL, NULL },
  { "delete", "Delete", appRemoveCb },
};

/*-----------------------------------------------------------------------------
  Widget Argument lists
-----------------------------------------------------------------------------*/

static Arg form_args[] = {
  { XtNdefaultDistance, 0 }
};

static Arg viewport_args[] = {
  { XtNfromVert, (XtArgVal) NULL },
  { XtNwidth, APPWIDTH },
  { XtNtop, XtChainTop },
  { XtNbottom, XtChainBottom },
  { XtNleft, XtChainLeft },
  { XtNright, XtChainRight },
  { XtNallowVert, (XtArgVal) True },
};

static Arg button_box_args[] = {
  { XtNfromVert, (XtArgVal) NULL },
  { XtNtop, XtChainBottom },
  { XtNbottom, XtChainBottom },
  { XtNleft, XtChainLeft },
  { XtNright, XtChainLeft },
};

static Arg icon_box_args[] = {
  { XtNwidth, 0 },
  { XtNtranslations, (XtArgVal) NULL }
};

static Arg icon_form_args[] = {
  { XtNdefaultDistance, 0 },
  { XtNwidth, 0 }
};

static Arg icon_toggle_args[] = {
  { XtNfromHoriz, (XtArgVal) NULL },
  { XtNfromVert, (XtArgVal) NULL },
  { XtNbitmap, (XtArgVal) NULL },
  { XtNtranslations, (XtArgVal) NULL },
  { XtNwidth, 0 },
  { XtNheight, 0 },
  { XtNforeground, (XtArgVal) 0 },
};

static Arg icon_label_args[] = {
  { XtNfromHoriz, (XtArgVal) NULL },
  { XtNfromVert, (XtArgVal) NULL },
  { XtNlabel, (XtArgVal) NULL },
  { XtNfont, (XtArgVal) NULL },
  { XtNwidth, 0 },
  { XtNinternalWidth, 0 },
  { XtNinternalHeight, 0 }
};

/*-----------------------------------------------------------------------------
  Translation tables
-----------------------------------------------------------------------------*/

#ifndef ENHANCE_TRANSLATIONS

static char app_translations[] = "\
    <Enter>             : appMaybeHighlight()\n\
    <Leave>             : unhighlight()\n\
    <Btn1Up>(2)         : runApp()\n\
    <Btn1Down>,<Btn1Up> : appSelect()\n\
    <Btn1Down>,<Leave>  : appBeginDrag(1,move)\n\
    <Btn2Down>,<Btn2Up> : appToggle()\n\
    <Btn2Down>,<Leave>  : appBeginDrag(2,copy)\n";

static char iconbox_translations[] = "\
    <Btn2Up>            : dummy()\n\
    <Btn3Up>            : dummy()\n\
    Any <Btn3Down>      : appPopup()\n\
    <Enter>             : appTrackCursor()\n\
    <Leave>             : appTrackCursor()\n";
#endif

/*-----------------------------------------------------------------------------
  Action tables
-----------------------------------------------------------------------------*/
static void CatchEntryLeave2(Widget,XtPointer,XEvent*,Boolean *ctd);

static void dummy(UNUSED(Widget w), UNUSED(XEvent *event),
                  UNUSED(String *params), UNUSED(Cardinal *num_params)) {
}

static XtActionsRec app_actions[] = {
  { "appMaybeHighlight", appMaybeHighlight },
  { "runApp", runApp },
  { "appSelect", appSelect },
  { "appToggle", appToggle },
  { "appPopup", appPopup },
  { "appBeginDrag", appBeginDrag },
  { "appTrackCursor", appTrackCursor },
  { "dummy", dummy }
};

/*-----------------------------------------------------------------------------
  Button Data
-----------------------------------------------------------------------------*/

static const ButtonRec aw_buttons[] = {
  { "back", "Back", appBackCb },
  { "main", "Main", appMainCb },
  { "reload", "Reload", appLoadCb },
  { "open", "File window", appOpenCb },
};


/*-----------------------------------------------------------------------------
  PRIVATE FUNCTIONS
-----------------------------------------------------------------------------*/

static int longestName(void)
{
  int l, longest = 0;
  Cardinal i;

  for (i=0; i<aw.n_apps; i++)
    if ((l = XTextWidth(resources.icon_font, aw.apps[i].name, 
			strlen(aw.apps[i].name))) > longest)
      longest = l;
  return longest;
}

/*---------------------------------------------------------------------------*/

static int parseAppI(FILE *fp, char **name, char **directory, char **fname,
	     char **icon, char **push_action, char **drop_action)
{
  static char s[MAXAPPSTRINGLEN];
  int l;

 start:
  if (feof(fp)||!fgets(s, MAXAPPSTRINGLEN, fp))
    return 0;
  l = strlen(s);
  /* TODO: check for overlong lines */
  if (s[l-1] == '\n')
    s[--l] = '\0';
  if (!l || *s == '#')
    goto start;
  if (!(*name = split(s, ':')))
    return -1;
  if (!(*directory = split(NULL, ':')))
    return -1;
  if (!(*fname = split(NULL, ':')))
    return -1;
  if (!(*icon = split(NULL, ':')))
    return -1;
  if (!(*push_action = split(NULL, ':')))
    return -1;
  if (!(*drop_action = split(NULL, ':')))
    return -1;
  return l;
}

/*---------------------------------------------------------------------------*/

/* determine the icon of an application */

static void readApplicationBitmap(AppRec *app, Boolean interactive) {

	app->icon_bm = None;

	if( app->icon[0] != '\0' ) {
		app->icon_bm = loadIcon(app->icon,NULL,NULL);
		if( app->icon_bm == None ) {
			if( interactive )
				error("Can't read icon for", app->name);
			else
				fprintf(stderr, 
"%s: can't read icon for application %s\n", progname, app->name);
		}
	}
	if( app->icon_bm == None ) {
		/* choose a default icon, no longer on file type but
		 * on what happens with that */
		if( strcmp(app->push_action,"LOAD") == 0 ) {
			app->icon_bm = bm_appmgr;
		} else if( strcmp(app->push_action,"OPEN") == 0 ) {
			app->icon_bm = bm[DIR_BM];
		} else
			app->icon_bm = bm[FILE_BM];
	}
}

/*---------------------------------------------------------------------------*/

static void generateDeviceApp(DevRec *dev) {
	char *name;

	name = strrchr(dev->name,'/');
	if (name == NULL)
		name = dev->name;
	else 
		name++;

	aw.apps = (AppList) XTREALLOC(aw.apps, (aw.n_apps+1)*sizeof(AppRec) );
	aw.apps[aw.n_apps].name = XtNewString(name);
	aw.apps[aw.n_apps].directory = XtNewString(dev->name);

	aw.apps[aw.n_apps].icon = XtNewString(dev->type);
	aw.apps[aw.n_apps].push_action = XtNewString("OPEN");
	aw.apps[aw.n_apps].drop_action = XtNewString("");
	aw.n_apps++;
}

/*-----------------------------------------------------------------------------
  PUBLIC FUNCTIONS
-----------------------------------------------------------------------------*/

int parseApp(FILE *fp, Boolean loadicons) {
	char *name, *directory, *fname, *icon, *push_action, *drop_action;
	char s[MAXAPPSTRINGLEN];
	int p;

	p = parseAppI(fp, &name, &directory, &fname, &icon, 
			&push_action, &drop_action);
	if( p <= 0 )
		return p;
	aw.apps = (AppList) XTREALLOC(aw.apps, (aw.n_apps+1)*sizeof(AppRec) );
	aw.apps[aw.n_apps].name = XtNewString(strparse(s, name, "\\:"));
	aw.apps[aw.n_apps].directory = XtNewString(strparse(s, directory, 
				"\\:"));
	aw.apps[aw.n_apps].icon = XtNewString(strparse(s, icon, "\\:"));
	aw.apps[aw.n_apps].push_action = XtNewString(strparse(s, push_action,
				"\\:"));
	aw.apps[aw.n_apps].drop_action = XtNewString(strparse(s, drop_action,
				"\\:"));
	/* look if it is old-format and try to fix it */
	fname = strparse(s, fname, "\\:");
	push_action = aw.apps[aw.n_apps].push_action;
	if( fname[0] != '\0' ) {
		if( strcmp(push_action,"LOAD") == 0 ) {
			char *appfile = dirConcat(
					aw.apps[aw.n_apps].directory,
					fname);
			XtFree(aw.apps[aw.n_apps].directory);
			aw.apps[aw.n_apps].directory = appfile;
		} else if( strcmp(push_action,"OPEN") == 0 ) {
			if( strcmp(fname,".") == 0 ) {
				/* everything already OK */
			} else if( aw.apps[aw.n_apps].directory[0] == '\0' ) {
				char *dir = dirConcat("~",fname);
				XtFree(aw.apps[aw.n_apps].directory);
				aw.apps[aw.n_apps].directory = dir;
			} else {
				char *dir = dirConcat(
						aw.apps[aw.n_apps].directory,
						fname);
				XtFree(aw.apps[aw.n_apps].directory);
				aw.apps[aw.n_apps].directory = dir;
			}
		} else if( push_action[0] != '\0' ) {
			aw.apps[aw.n_apps].push_action = spaceConcat(
					push_action, fname);
			XtFree(push_action);
		}
		if( drop_action[0] != '\0' ) {
			char *action = spaceConcat(
					aw.apps[aw.n_apps].drop_action,
					fname);
			XtFree(aw.apps[aw.n_apps].drop_action);
			aw.apps[aw.n_apps].drop_action = action;
		}
	}
	if( loadicons )
		readApplicationBitmap(&aw.apps[aw.n_apps],False);
	aw.n_apps++;
	return p;
}

/*-------------------------------------------------------------------------*/

void createApplicationWindow(void)
{
#ifndef ENHANCE_TRANSLATIONS
  XtTranslations t;
#endif

  /* Add new actions and parse the translation tables */
  XtAppAddActions(app_context, app_actions, XtNumber(app_actions));

#ifndef ENHANCE_TRANSLATIONS
  t = XtParseTranslationTable(app_translations);
  icon_toggle_args[3].value = (XtArgVal) t;

  t = XtParseTranslationTable(iconbox_translations);
  icon_box_args[1].value = (XtArgVal) t;
#endif

  icon_label_args[3].value = (XtArgVal) resources.icon_font;

  /* create the install popups */
  createInstallPopups();

  /* create the menus */
  app_popup_items = createFloatingMenu("app popup", 
				       app_popup_menu,
				       XtNumber(app_popup_menu),
				       4, aw.shell,
				       NULL, &app_popup_widget);
  app_popup_items1 = createFloatingMenu("app popup 1", app_popup_menu1,
		     XtNumber(app_popup_menu1), 4, aw.shell,
		     NULL, &app_popup_widget1);

  XtRegisterGrabAction(appPopup, True, ButtonPressMask | ButtonReleaseMask,
		       GrabModeAsync, GrabModeAsync);

  /* create the form */
#ifndef ENHANCE_TRANSLATIONS
  aw.form = XtCreateManagedWidget("awform", formWidgetClass, aw.shell,
#else
  aw.form = XtCreateManagedWidget("awform", formWidgetClass, aw.shell,
#endif
				  form_args, XtNumber(form_args) );
  
  /* create the viewport */
  aw.viewport = XtCreateManagedWidget("viewport", viewportWidgetClass,
    aw.form, viewport_args, XtNumber(viewport_args) );

  /* create button box */
  button_box_args[0].value = (XtArgVal) aw.viewport;
  aw.button_box = XtCreateManagedWidget("button box", boxWidgetClass,
					aw.form, button_box_args, 
					XtNumber(button_box_args) );
  aw_button_items = createButtons(aw_buttons, XtNumber(aw_buttons),
				  aw.button_box, NULL);

  aw.n_selections = 0;
}

/*---------------------------------------------------------------------------*/

void createApplicationDisplay(void)
{
  Cardinal i;
  Dimension width;

  for (i=0; i<aw.n_apps; i++)
    aw.apps[i].selected = False;
  aw.n_selections = 0;
  
  XtVaGetValues(aw.viewport, XtNwidth, &width, NULL);
  icon_box_args[0].value = (XtArgVal) width;

  aw.icon_box = XtCreateWidget("icon box", boxWidgetClass,
    aw.viewport, icon_box_args, XtNumber(icon_box_args) );

  if (aw.n_apps == 0)
    XtVaCreateManagedWidget("label", labelWidgetClass, aw.icon_box,
			    XtNlabel, "No configured applications",
			    XtNfont, resources.label_font, NULL);
  else {
    width = longestName();
    if (width < resources.app_icon_width)
      width = resources.app_icon_width;
    icon_form_args[1].value = (XtArgVal) width;
    icon_label_args[4].value = (XtArgVal) width;
    icon_toggle_args[4].value = (XtArgVal) width;
    icon_toggle_args[5].value = (XtArgVal) resources.app_icon_height;
#ifdef ENHANCE_SELECTION
    icon_toggle_args[6].value = (XtArgVal) resources.highlight_pixel;
#endif
    
    for (i=0; i < aw.n_apps; i++) {
      Pixel back;
      aw.apps[i].form = XtCreateManagedWidget(aw.apps[i].name,
					      formWidgetClass, aw.icon_box,
					      icon_form_args,
					      XtNumber(icon_form_args) );
      icon_toggle_args[2].value = aw.apps[i].icon_bm;
      aw.apps[i].toggle = XtCreateManagedWidget("icon", toggleWidgetClass,
						aw.apps[i].form,
						icon_toggle_args,
						XtNumber(icon_toggle_args) );
      XtVaGetValues(aw.apps[i].toggle, XtNbackground, &back, NULL);
      XtVaSetValues(aw.apps[i].toggle, XtNborder, (XtArgVal) back, NULL);

/*
    RBW Test stuff...
*/
XtInsertEventHandler(aw.apps[i].toggle,
                     EnterWindowMask|LeaveWindowMask,
                     False,
                     (XtEventHandler)CatchEntryLeave2,
                     (XtPointer)0,
                     XtListHead);


      icon_label_args[1].value = (XtArgVal) aw.apps[i].toggle;
      icon_label_args[2].value = (XtArgVal) aw.apps[i].name;
      aw.apps[i].label = XtCreateManagedWidget("label", labelWidgetClass,
					       aw.apps[i].form,
					       icon_label_args,
					       XtNumber(icon_label_args) );
    };
  }

  if (n_appst > 0)
    fillIn(aw_button_items[0]);
  else
    grayOut(aw_button_items[0]);
  XtManageChild(aw.icon_box);
}

/*---------------------------------------------------------------------------*/

void updateApplicationDisplay(void)
{
  XtDestroyWidget(aw.icon_box);
  createApplicationDisplay();
  setApplicationWindowName();
}

/*---------------------------------------------------------------------------*/

void setApplicationWindowName(void)
{
  char *p;
  if ((p = strrchr(aw.app_file, '/')))
    p++;
  else
    p = aw.app_file;
  XSetIconName(XtDisplay(aw.shell), XtWindow(aw.shell), p);
  XStoreName(XtDisplay(aw.shell), XtWindow(aw.shell), p);
}

/*---------------------------------------------------------------------------*/

void readApplicationData(void)
{
  FILE *fp;
  int p;
  Cardinal i;
  
  aw.n_apps = 0;
  aw.apps = NULL;

  if (strcmp(aw.app_file,"XFM-DEVICES") == 0) {

	  for (i=0; i<n_devices; i++) {
		  generateDeviceApp(devs + i);
	  }
	  for (i=0; i<aw.n_apps; i++) {
		  readApplicationBitmap(&aw.apps[i],False);
	  }
	  aw.readonly = True;
	  return;
  }
  
  if (!(fp = fopen(aw.app_file, "r"))) return;

  do { 
	  p = parseApp(fp,False);
  } while( p > 0 );

  if (p == -1)
    error("Error in applications file", "");

  if (fclose(fp))
    sysError("Error reading applications file:");

  aw.readonly = False;

  for (i=0; i<aw.n_apps; i++) {
    readApplicationBitmap(&aw.apps[i],False);
  }
}

/*---------------------------------------------------------------------------*/

int writeApplicationData(void)
{
  FILE *fp;
  Cardinal i;

  if (aw.readonly)
	  return 0;
  
  if (! (fp = fopen(aw.app_file, "w") )) {
#ifdef SYSTEMWIDE_DEFAULTS
    size_t app_dir_len = strlen(resources.app_dir);
    if ( app_dir_len > 0 &&
          strcasecmp(resources.app_dir,"none") != 0) {
      size_t len_basename;
      const char *p;char *n;

      if ((p = strrchr(aw.app_file, '/')))
	p++;
      else
        p = aw.app_file;
      len_basename = strlen(p);
      n = XtMalloc(app_dir_len+len_basename+2);
      memcpy(n, resources.app_dir, app_dir_len);
      n[app_dir_len] = '/';
      memcpy(n+app_dir_len+1, p, len_basename+1);

      if (!exists(resources.app_dir)){
        mkdir(resources.app_dir,0777);
      }
      if ((fp = fopen(n, "w") )) {
        XTFREE(aw.app_file);
	aw.app_file = n;
      } else {
	XTFREE(n);
      }
    }
#endif
  }
  if (! fp ) {
    sysError("Error writing applications file:");
    return -1;
  }

  fprintf(fp, "#XFM\n");

  for (i=0; i < aw.n_apps; i++) {
    fprintexpand(fp, aw.apps[i].name, "\\:");
    fputc(':',fp);
    fprintexpand(fp, aw.apps[i].directory, "\\:");
    fputc(':',fp);
    fputc(':',fp);
    fprintexpand(fp, aw.apps[i].icon, "\\:");
    fputc(':',fp);
    fprintexpand(fp, aw.apps[i].push_action, "\\:");
    fputc(':',fp);
    fprintexpand(fp, aw.apps[i].drop_action, "\\:");
    fputc('\n',fp);
  }
  
  if (fclose(fp)) {
    sysError("Error writing applications file:");
    return -1;
  }

  return 0;
}

/*---------------------------------------------------------------------------*/

void freeApplicationResources(AppRec *app)
{
  XTFREE(app->name);
  XTFREE(app->directory);
  XTFREE(app->icon);
  XTFREE(app->push_action);
  XTFREE(app->drop_action);
}

/*---------------------------------------------------------------------------*/

void installApplication(const char *name, const char *directory,
	const char *icon, const char *push_action,
	const char *drop_action)
{

  aw.apps = (AppList) XTREALLOC(aw.apps, (aw.n_apps+1) * sizeof(AppRec));

  aw.apps[aw.n_apps].name = XtNewString(name);
  aw.apps[aw.n_apps].directory = XtNewString(directory);
  aw.apps[aw.n_apps].icon = XtNewString(icon);
  aw.apps[aw.n_apps].push_action = XtNewString(push_action);
  aw.apps[aw.n_apps].drop_action = XtNewString(drop_action);
  aw.apps[aw.n_apps].form = aw.apps[aw.n_apps].toggle = 
	  		    aw.apps[aw.n_apps].label = NULL;
  readApplicationBitmap(&aw.apps[aw.n_apps],True);
  /* only increment here so that readApplicationBitmap can show error 
   * messages without this part already visible */
  aw.n_apps++;
}

/*---------------------------------------------------------------------------*/

void replaceApplication(AppRec *app, const char *name, const char *directory,
		const char *icon, const char *push_action,
		const char *drop_action)
{
  /* This is not the nicest way, but it works, as 
   * freeApplicationResources no longer frees or unsets app->icon_bm,
   * readApplicationBitmap can be called interactive even afterwards, 
   * as the old icon will still be visible - brl */

  freeApplicationResources(app);

  app->name = XtNewString(name);
  app->directory = XtNewString(directory);
  app->icon = XtNewString(icon);
  app->push_action = XtNewString(push_action);
  app->drop_action = XtNewString(drop_action);
  app->form = app->toggle = app->label = NULL;
  readApplicationBitmap(app,True);
}

/*---------------------------------------------------------------------------*/

static void pushApplicationsFile(const char *filename)
{
	int i = n_appst++;

	appst = (char **) XTREALLOC(appst, n_appst * sizeof(char *));
	appst[i] = aw.app_file;

	if (filename[0] != '/' && strcmp(filename,"XFM-DEVICES") != 0) {
		char *fn = dirConcat(resources.app_dir, filename);
		if (exists(fn))
			aw.app_file = fn;
		else {
			XtFree(fn);
			aw.app_file = 
				dirConcat(resources.system_app_dir, filename);
		}
	} else
		aw.app_file = XtNewString(filename);
}

/*---------------------------------------------------------------------------*/

void popApplicationsFile(void)
{
  if (n_appst <= 0) return;
  XTFREE(aw.app_file);
  aw.app_file = appst[--n_appst];
  appst = (char **) XTREALLOC(appst, n_appst * sizeof(char *));
}

/*---------------------------------------------------------------------------*/

void clearApplicationsStack(void)
{
  int i;
  for (i = 0; i < n_appst; i++)
    XTFREE(appst[i]);
  XTFREE(appst);
  appst = NULL;
  n_appst = 0;
}


/*
    RBW Test stuff...
*/


static void CatchEntryLeave2(UNUSED(Widget w), UNUSED(XtPointer cld), XEvent* ev, Boolean *ctd)
{
/*
    RBW - eat extraneous LeaveNotify/EnterNotify events caused by some window
    managers when a ButtonPress happens, so that the Xt translation mechanism
    for double-clicks doesn't get confused.
*/
if (ev->xcrossing.mode == NotifyGrab || ev->xcrossing.mode == NotifyUngrab)  {
  *ctd = False;  /* Eat this event */
  }  else  {
  *ctd = True;  /* Don't eat this event */
  }
return;
}


void AmReadUpdate(void) {
	Cardinal i;

	for(i=0; i<aw.n_apps; i++)
		freeApplicationResources(&aw.apps[i]);
	XTFREE(aw.apps);

	readApplicationData();
	updateApplicationDisplay();
}

Boolean AmPushLoadFile(const char *filename,Boolean fromFW) {

	if( !resources.appmgr && !fromFW )
		return False;

	pushApplicationsFile(filename);
	AmReadUpdate();
	return True;
}
