/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "EACWSManClient.h"
#include "UNSCommonDefs.h"

const std::string EACWSManClient::DEFAULT_USER = "$$eac";
const std::string EACWSManClient::DEFAULT_PASS = "$$eac";

EACWSManClient::EACWSManClient() : BaseWSManClient(CRED_REG_INFO::EAC, DEFAULT_USER, DEFAULT_PASS)
{
  m_pPosture = NULL;
  m_pChangeHash = NULL;
  m_pHash = NULL;

  m_nPostureLen = 0;
  m_nChangeHashLen = 0;
  m_nHashLen = 0;
  m_isInit = false;

}

EACWSManClient::~EACWSManClient()
{
  delete []m_pPosture;
  delete []m_pChangeHash;
  delete []m_pHash;
}

bool EACWSManClient::SaveData(int nNewLength, int &nOldLength,
			      unsigned char **ppDest, unsigned char *pSource)
{
  if (nNewLength != nOldLength)
    {
      //New size is different - reallocate
      delete [](*ppDest);
      *ppDest = new unsigned char[nNewLength];
      if (*ppDest == NULL)
	return false;
      nOldLength = nNewLength;
    }
  memcpy(*ppDest, pSource, nNewLength);
  return true;
}


bool EACWSManClient::GetPosture(unsigned char **ppPosture, unsigned int &nPostureLen,
				unsigned char **ppChangeHash, unsigned int &nChangeHashLen)
{
  if (!Init())
    return false;

  AMT_EndpointAccessControlService::GetPosture_INPUT request;
  request.PostureType = 0; // Use enum/typedef

  AMT_EndpointAccessControlService::GetPosture_OUTPUT response;
  printf ("\nGetting Posture... (%s).\n", endpoint.c_str());
  try {
    m_service.InvokeGetPosture( request, response, m_client);
  }
  catch (GeneralWsmanException& e)
    {
      printf("\nError: failed while calling InvokeGetPosture routine\n");
      printf("%s\n", e.what());
      return false;
    }
  catch (exception& e)
    {
      printf("\nError: failed while calling InvokeGetPosture routine\n");
      printf("%s\n", e.what());
      return false;
    }

  printf ("\nPosture received successfuly.\n");

  nPostureLen = response.SignedPosture.len;
  nChangeHashLen = response.PostureChangeHash.len;
  if (!SaveData(nPostureLen, m_nPostureLen, &m_pPosture, response.SignedPosture.data))
    return false;

  if (!SaveData(nChangeHashLen, m_nChangeHashLen, &m_pChangeHash, response.PostureChangeHash.data))
    return false;

  *ppPosture = m_pPosture;
  *ppChangeHash = m_pChangeHash;

  return true;
}

bool EACWSManClient::GetPostureHash(unsigned char **ppHash, unsigned int &nHashLen)
{
  if (!Init())
    return false;

  AMT_EndpointAccessControlService::GetPostureHash_INPUT request;
  request.PostureType = 0; // Use enum/typedef

  AMT_EndpointAccessControlService::GetPostureHash_OUTPUT response;
  try {
    m_service.InvokeGetPostureHash(request, response,m_client);
  }
  catch (GeneralWsmanException& e)
    {
      printf("\nError: failed while calling InvokeGetPosture routine\n");
      printf("%s\n", e.what());
      return false;
    }
  catch (exception& e)
    {
      printf("\nError: failed while calling InvokeGetPostureHash routine\n");
      printf("%s\n", e.what());
      return false;
    }


  nHashLen = response.PostureChangeHash.len;
  if (!SaveData(nHashLen, m_nHashLen, &m_pHash, response.PostureChangeHash.data))
    return false;

  *ppHash = m_pHash;

  return true;
}

bool EACWSManClient::IsEACEnabled(bool &bEACEnabled)
{
  if (!Init(true))
    return false;
  bEACEnabled = (m_service.EnabledState == 2); // use enum/typedef
  return true;
}

bool EACWSManClient::Init(bool forceGet)
{
  if (!forceGet && m_isInit) return true;
  for(int i = 0; i < 2; i++)
    {
      if (!m_endpoint)
	SetEndpoint(i == 0 ? false : true);

      try {
	m_service.Get(NULL,m_client);
      }
      catch (GeneralWsmanException& e)
	{
	  printf("\nError: failed while calling InvokeGetPosture routine\n");
	  printf("%s\n", e.what());
	  m_endpoint = false;
	  continue;
	}
      catch (exception& e)
	{
	  printf("\nError: failed while calling get AMT_EndpointAccessControlService\n");
	  printf("%s\n", e.what());
	  m_endpoint = false;
	  continue;
	}
      m_endpoint = true;
      m_isInit = true;
      return true;
    }
  return false;
}
