use ui;
use core:geometry;
use progvis:net;
use graphics;

SolidBrush solveBg on Render = SolidBrush(green + white * 0.2);


/**
 * A panel that can be opened from one of the rows in a table.
 */
class Panel on Render {
	Bool render(Point offset, Float width, Graphics g) : abstract;
	Float height() : abstract;
	void onClick(Point pt, Float width, Painter p) : abstract;
}


/**
 * A panel that contains zero or more other panels (stacked top-to-bottom).
 */
class MultiPanel extends Panel {
	private Panel[] contained;

	// Add a panel.
	void add(Panel p) {
		contained << p;
	}

	// Render.
	Bool render(Point offset, Float width, Graphics g) : override {
		Bool repaint = false;
		for (x in contained) {
			repaint |= x.render(offset, width, g);
			offset.y += x.height;
		}
		repaint;
	}

	// Height.
	Float height() : override {
		Float h;
		for (x in contained)
			h += x.height;
		h;
	}

	// Click.
	void onClick(Point pt, Float width, Painter p) : override {
		for (x in contained) {
			Float h = x.height;
			if (pt.y < h) {
				x.onClick(pt, width, p);
				return;
			}

			pt.y -= h;
		}
	}
}


/**
 * A simple panel that shows some additional stats, along with an optional Solve button.
 */
class StatsPanel extends Panel {
	private Int problemId;
	private Text info;
	private Text solve;
	private Float textHeight;
	private Callback callback;

	init(Int problemId, Problem? parent, Bool solvable, Callback callback) {
		Str text = if (parent) {
			"Solution to: ${parent.title}";
		} else {
			"New problem";
		};

		Str solveText = if (solvable) {
			"Try to solve!";
		} else {
			"Check again";
		};

		init {
			info(text, defaultFont);
			solve(solveText, defaultFont);
			problemId = problemId;
			callback = callback;
		}

		textHeight = max(info.size.h, solve.size.h);
	}

	Bool render(Point offset, Float width, Graphics g) : override {
		Rect bounds(offset, Size(width, height));
		g.fill(bounds, tableBg);

		Point delta(tableHSpace, tableVSpace * 0.5);
		g.draw(info, tableText, offset + delta);

		Float solveW = solve.size.w + tableHSpace * 2;
		Rect button(offset + Point(width - solveW, 0), Size(solveW, height));
		g.fill(button, solveBg);
		g.line(button.p0, button.p0 + Point(0, height), tableFg);
		g.draw(solve, tableText, button.p0 + delta);

		g.draw(bounds, tableFg);

		false;
	}

	Float height() : override {
		textHeight + tableVSpace;
	}

	void onClick(Point pt, Float width, Painter p) : override {
		if (pt.x >= width - solve.size.w - tableHSpace * 2) {
			(spawn callback.solveProblem(problemId)).detach();
		}
	}
}

/**
 * Panel containing "replay" and "show solution" as appropriate. Used as "details" for a solution.
 */
class SolutionPanel extends Panel {
	private Callback callback;
	private Solution solution;
	private Text[] buttons;
	private Brush[] colors;
	private Fn<void>[] actions;
	private Float buttonHeight;

	init(Solution solution, Callback callback) {
		init {
			solution = solution;
			callback = callback;
		}

		// TODO: How to know what we clicked?
		if (solution.solution.any) {
			buttons << Text("Show the issue", defaultFont);
			colors << SolidBrush(red + white * 0.2);
			actions << &this.showSolution();
		}
		if (id = solution.improved) {
			buttons << Text("Show fix (ID ${id})", defaultFont);
			colors << SolidBrush(green + white * 0.2);
			actions << &this.showImproved();
		}

		if (buttons.empty) {
			buttons << Text("No more information available", defaultFont);
			colors << SolidBrush(white);
		}

		for (b in buttons) {
			buttonHeight = buttonHeight.max(b.size.h);
		}
	}

	Bool render(Point offset, Float width, Graphics g) : override {
		Float w = width / buttons.count.int.float;
		for (i, button in buttons) {
			Rect r(offset + Point(i.int.float * w, 0), Size(w, height));
			g.fill(r, colors[i]);
			g.draw(button, tableText, r.center - button.size / 2);
			g.draw(r, tableFg);
		}

		false;
	}

	Float height() : override {
		buttonHeight + tableVSpace;
	}

	void onClick(Point pt, Float width, Painter p) : override {
		Float w = width / buttons.count.int.float;
		Nat id = (pt.x / w).int.nat;
		if (id < actions.count)
			actions[id].call();
	}

	private void showSolution() {
		if (s = solution.solution)
			(spawn callback.showSolution(solution.to, s)).detach();
	}

	private void showImproved() {
		if (p = solution.improved)
			(spawn callback.solveProblem(p)).detach();
	}
}

/**
 * Panel containing a table.
 */
class TablePanel extends Panel {
	private Table table;

	init(View view) {
		init {
			table(view);
		}
	}

	Bool render(Point offset, Float width, Graphics g) : override {
		g.push();
		g.transform = translate(offset);

		// TODO: We could give it better data for clipping if we had it.
		Bool repaint = table.render(Rect(0, 0, width, table.height), g);

		g.pop();

		repaint;
	}

	Float height() : override {
		table.height;
	}

	void onClick(Point pt, Float width, Painter p) : override {
		table.onClick(pt, p);
	}
}
