# ScummVM - Graphic Adventure Engine
# Copyright (C) 2020-2021 Stryzhniou Fiodar

# ScummVM is the legal property of its developers, whose names
# are too numerous to list here. Please refer to the COPYRIGHT
# file distributed with this source distribution.

# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.


from __future__ import with_statement
import os
from common_names import *

# Template args = (Package Name = filename + platform, Component name, UID, major version, minor version, build number).
pkg_template = """
;
;   Warning: autogenerated file.
;

; Language - standard language definitions.
&EN

; List of localised vendor names - one per language. At least one must be provided (English [EN]).
; List must correspond to list of languages specified elsewhere in the .pkg
%%{"ScummVM"}
; The non-localised, globally unique vendor name (mandatory)
:"ScummVM"

; UID is the app's UID.
#{"%s"},(%s),%s,%s,%s

; Supports Series 60 v 3.0.
[0x101F7961], 0, 0, 0, {"Series60ProductID"}
; Supports Series 60 v 5.0.
[0x1028315F], 0, 0, 0, {"Series60ProductID"}
"""

# Template args = (UID from 1st package, major version, minor version, build number,
# major version, minor version, build number, 1st package name).
dependency = """

; ScummVM1 dependency.
(%s), %s,%s,%s ~ %s,%s,%s, {"%s"}
"""

pkg_licenses_install = """

; Scummvm Documentation, licences.
"..\\THANKS"  - "!:\\data\\scummvm\\THANKS", FT, TC
"..\\..\\..\\..\\COPYRIGHT" - "!:\\data\\scummvm\\COPYRIGHT", FT, TC
"..\\..\\..\\..\\COPYING.BSD"   - "!:\\data\\scummvm\\COPYING.BSD"
"..\\..\\..\\..\\COPYING.ISC"   - "!:\\data\\scummvm\\COPYING.ISC"
"..\\..\\..\\..\\COPYING.LGPL"  - "!:\\data\\scummvm\\COPYING.LGPL"
"..\\..\\..\\..\\COPYING.LUA"   - "!:\\data\\scummvm\\COPYING.LUA"
"..\\..\\..\\..\\COPYING.MIT"   - "!:\\data\\scummvm\\COPYING.MIT"
"..\\..\\..\\..\\COPYING.OFL"   - "!:\\data\\scummvm\\COPYING.OFL"
"..\\..\\..\\..\\COPYING.TINYGL"   - "!:\\data\\scummvm\\COPYING.TINYGL"
"..\\..\\..\\..\\COPYING"   - "!:\\data\\scummvm\\COPYING"
"..\\..\\..\\..\\AUTHORS"   - "!:\\data\\scummvm\\AUTHORS", FT, TC
"..\\..\\..\\..\\README.md" - "!:\\data\\scummvm\\README.md"
"..\\..\\..\\..\\NEWS.md"   - "!:\\data\\scummvm\\NEWS.md"
"..\\README"  - "!:\\data\\scummvm\\SYMBIAN_README", FT, TC
"""

pkg_licenses_show = """

; Scummvm Documentation.
"..\\THANKS"  - "", FT, TC
"..\\..\\..\\..\COPYRIGHT"    - "", FT, TC
"..\\..\\..\\..\COPYING"          - "", FT, TC
"..\\README"  - "", FT, TC
"""

# Template args = (extension, extension, extension, extension,
# extension, extension, extension, extension ).
# Where extension == (ordinal or (ordinal and "_test") ).
install_data = """
; Launcher, Application, AIF & Resource file.
"$(EPOCROOT)epoc32\\release\\gcce\\urel\\ScummVM%s.exe" -		"!:\\sys\\bin\\ScummVM%s.exe"
"$(EPOCROOT)epoc32\\data\\z\\resource\\apps\\ScummVM%s.rsc" -	"!:\\resource\\apps\\ScummVM%s.rsc"
"$(EPOCROOT)epoc32\\data\\z\\resource\\apps\\ScummVM%s_loc.rsc"-	"!:\\resource\\apps\\ScummVM%s_loc.rsc"
"$(EPOCROOT)epoc32\\data\\z\\private\\10003a3f\\apps\\ScummVM%s_reg.rsc" -
							"!:\\private\\10003a3f\\import\\apps\\ScummVM%s_reg.rsc"
"""

install_eclipse_data = """
; Launcher, Application, AIF & Resource file.
"$(EPOCROOT)epoc32\\release\\$(PLATFORM)\\$(TARGET)\\ScummVM%s.exe" -		"!:\\sys\\bin\\ScummVM%s.exe"
"$(EPOCROOT)epoc32\\data\\z\\resource\\apps\\ScummVM%s.rsc" -	"!:\\resource\\apps\\ScummVM%s.rsc"
"$(EPOCROOT)epoc32\\data\\z\\resource\\apps\\ScummVM%s_loc.rsc"-	"!:\\resource\\apps\\ScummVM%s_loc.rsc"
"$(EPOCROOT)epoc32\\data\\z\\private\\10003a3f\\apps\\ScummVM%s_reg.rsc" -
							"!:\\private\\10003a3f\\import\\apps\\ScummVM%s_reg.rsc"
"""

clear_uninstall = """

; Config/log files: 'empty' will automagically be removed on uninstall.
""- "c:\data\scummvm\scummvm.ini",FILENULL
""- "c:\data\scummvm\scummvm.stdout.txt",FILENULL
""- "c:\data\scummvm\scummvm.stderr.txt",FILENULL
""- "c:\data\scummvm\sdl.ini",FILENULL
"""

mif_install = """
    \"$(EPOCROOT)epoc32\\data\\z\\resource\\apps\\ScummVM.mif\" -	\"!:\\resource\\apps\\ScummVM.mif\"
"""

package_name = "ScummVM%s"

def EPOCROOTAsPath():
   return os.path.join( os.path.splitdrive(os.getcwd())[0], os.environ['EPOCROOT'])

def ResolveEpocRoot(install_data):
   return install_data.replace("$(EPOCROOT)", EPOCROOTAsPath())

def SaveInstallData(pkg, build, ext, toResolve = True):
   if build == 'full':
      ext = str(ext) + "_test"
   install = install_eclipse_data
   if toResolve:
      install = ResolveEpocRoot(install_data)
   AppendToFile(pkg, install %(ext, ext, ext, ext, ext, ext, ext, ext) )

def SaveDependency(build, pkg):
   uid = get_UIDs(build)
   dep_name = "ScummVM1.sisx"
   if build == 'full':
      dep_name = "ScummVM1_test.sisx"
   AppendToFile(pkg, dependency %(uid[0], sis_major_version, sis_minor_version, sis_build_number, \
   sis_major_version, sis_minor_version, sis_build_number, dep_name))

def ResolvePackName(ordinal, target, build):
   tmp = package_name %(str(ordinal +1) + " " + target)
   if build == 'full':
      tmp += " test"
   return tmp

def CreateLastPkg(install_uid, build, path, ordinal, target):
   pkg         = path %(ordinal +1)
   pkg_cmdline = path %( str(ordinal +1) +  "_cmdline")
   pack_name = ResolvePackName(ordinal, target, build)

   SafeWriteFile(pkg,         pkg_template %(pack_name, install_uid, sis_major_version, sis_minor_version, sis_build_number) )
   SafeWriteFile(pkg_cmdline, pkg_template %(pack_name, install_uid, sis_major_version, sis_minor_version, sis_build_number) )
   SaveDependency(build, pkg)
   SaveDependency(build, pkg_cmdline)

   AppendToFile(pkg, pkg_licenses_show)
   AppendToFile(pkg_cmdline, pkg_licenses_show)

   SaveInstallData(pkg,         build, ordinal*2 + 1, toResolve = False)
   SaveInstallData(pkg_cmdline, build, ordinal*2 + 1)

def CreatePkg(install_uid, build, path, ordinal, target):
   pkg         = path %(ordinal +1)
   pkg_cmdline = path %( str(ordinal +1) +  "_cmdline")
   pack_name = ResolvePackName(ordinal, target, build)

   SafeWriteFile(pkg,         pkg_template %(pack_name, install_uid, sis_major_version, sis_minor_version, sis_build_number) )
   SafeWriteFile(pkg_cmdline, pkg_template %(pack_name, install_uid, sis_major_version, sis_minor_version, sis_build_number) )
   SaveDependency(build, pkg)
   SaveDependency(build, pkg_cmdline)

   AppendToFile(pkg,         pkg_licenses_show)
   AppendToFile(pkg_cmdline, pkg_licenses_show)

   SaveInstallData(pkg,         build, ordinal*2 + 1, toResolve = False)
   SaveInstallData(pkg_cmdline, build, ordinal*2 + 1)
   SaveInstallData(pkg,         build, ordinal*2 + 2, toResolve = False)
   SaveInstallData(pkg_cmdline, build, ordinal*2 + 2)

def CreateFirstPkg(install_uid, build, path, target):
   ext = 1
   cmd = str(ext) + "_cmdline"

   pkg         = path %ext
   pkg_cmdline = path %cmd
   pack_name = ResolvePackName(0, target, build)

   SafeWriteFile(pkg,         pkg_template %(pack_name, install_uid, sis_major_version, sis_minor_version, sis_build_number) )
   SafeWriteFile(pkg_cmdline, pkg_template %(pack_name, install_uid, sis_major_version, sis_minor_version, sis_build_number) )

   p_license = pkg_licenses_install
   if build == 'full':
      p_license = p_license.replace("scummvm", "scummvm\\beta")
   AppendToFile(pkg,         p_license)
   AppendToFile(pkg_cmdline, p_license)
   if build == 'release':
      AppendToFile(pkg,                         mif_install)
      AppendToFile(pkg_cmdline, ResolveEpocRoot(mif_install))

   SaveInstallData(pkg,         build, 1, toResolve = False)
   SaveInstallData(pkg_cmdline, build, 1)
   SaveInstallData(pkg,         build, 2, toResolve = False)
   SaveInstallData(pkg_cmdline, build, 2)
   AppendToFile(pkg,         clear_uninstall)
   AppendToFile(pkg_cmdline, clear_uninstall)

def create_pkgs(build, path):
   uids = get_UIDs(build)
   pairs = len(uids)/2
   pkg_name = "ScummVM%s.pkg"
   pkg_name = os.path.join(path, pkg_name)
   for i in range(pairs):
      if i == 0:
         CreateFirstPkg(uids[0], build, pkg_name, path)
         continue
      CreatePkg(uids[i*2], build, pkg_name, i, path)
   if len(uids)%2 == 1:
      CreateLastPkg(uids[-1], build, pkg_name, pairs, path)

if __name__ == "__main__":
   create_pkgs(build = 'full', path = "S60v3")
